/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CBrushWin_sizeList - ブラシウィンドウのサイズリスト部分
*/


#include "CBrushWin_sizeList.h"

#include "CBrushSizeList.h"

#include "AXLayout.h"
#include "AXScrollBar.h"
#include "AXImage.h"
#include "AXMenu.h"
#include "AXStrDialog.h"
#include "AXApp.h"
#include "AXUtilStr.h"

#include "strid.h"


//----------------------------

class CBrushSizeListArea:public AXWindow
{
protected:
    enum
    {
        ONESIZE    = 27,

        DOWNF_NONE = 0,
        DOWNF_NORMAL,
        DOWNF_DEL
    };

protected:
    AXScrollBar *m_pScr;

    AXImage     m_img;

    int     m_nXCnt,
            m_nDownNo,
            m_fDown;

protected:
    int _getListNo(int x,int y);
    void _addDat();
    void _deleteDat(int no);
    void _runMenu(int x,int y,int no);
    void _drawSel(BOOL bErase);
    void _drawBrushPrev(int x,int y,int w,int h,int radius);

public:
    CBrushSizeListArea(AXWindow *pParent);

    void setScroll(AXScrollBar *p) { m_pScr = p; }
    void setScrollInfo();
    void updateAll();
    void draw();

    virtual BOOL onPaint(AXHD_PAINT *phd);
    virtual BOOL onSize();
    virtual BOOL onButtonDown(AXHD_MOUSE *phd);
    virtual BOOL onButtonUp(AXHD_MOUSE *phd);
};

//----------------------------


//************************************
// CBrushWin_sizeList
//************************************



CBrushWin_sizeList::CBrushWin_sizeList(AXWindow *pParent,int h)
    : AXWindow(pParent, 0, LF_EXPAND_W | LF_FIX_H)
{
    AXLayout *pl;

    m_nH = h;

    //

    setLayout(pl = new AXLayoutHorz);

    pl->addItem(m_pArea = new CBrushSizeListArea(this));
    pl->addItem(m_pScrV = new AXScrollBar(this, AXScrollBar::SBS_VERT, LF_EXPAND_H));

    m_pArea->setScroll(m_pScrV);
    m_pArea->setNotify(pParent);
}

//! 通知

BOOL CBrushWin_sizeList::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    if(pwin == m_pScrV)
        m_pArea->draw();

    return TRUE;
}



//************************************
// CBrushSizeListArea
//************************************



CBrushSizeListArea::CBrushSizeListArea(AXWindow *pParent)
    : AXWindow(pParent, 0, LF_EXPAND_WH)
{
    m_fDown = DOWNF_NONE;
    m_nXCnt = 1;
}

//! スクロール情報セット

void CBrushSizeListArea::setScrollInfo()
{
    int max,page;

    max = BRUSHSIZELIST->getCnt();
    max = (max + m_nXCnt - 1) / m_nXCnt;

    page = m_nH / ONESIZE;
    if(page <= 0) page = 1;

    m_pScr->setStatus(0, max, page);
}

//! すべて更新

void CBrushSizeListArea::updateAll()
{
    setScrollInfo();
    draw();
}


//========================
// ハンドラ
//========================


//! 描画

BOOL CBrushSizeListArea::onPaint(AXHD_PAINT *phd)
{
    m_img.put(m_id, phd->x, phd->y, phd->x, phd->y, phd->w, phd->h);

    return TRUE;
}

//! サイズ変更時

BOOL CBrushSizeListArea::onSize()
{
    m_img.recreate(m_nW, m_nH, ONESIZE, ONESIZE);

    //横の表示数

    m_nXCnt = (m_nW - 1) / ONESIZE;
    if(m_nXCnt < 1) m_nXCnt = 1;

    updateAll();

    return TRUE;
}

//! ボタン押し時

BOOL CBrushSizeListArea::onButtonDown(AXHD_MOUSE *phd)
{
    int no;

    if(m_fDown) return TRUE;

    no = _getListNo(phd->x, phd->y);

    if(phd->button == BUTTON_RIGHT)
    {
        //右クリックメニュー

        _runMenu(phd->rootx, phd->rooty, no);
    }
    else if(phd->button == BUTTON_LEFT)
    {
        //左ボタン（+Shift で削除）

        if(no != -1)
        {
            m_fDown   = (phd->state & STATE_SHIFT)? DOWNF_DEL: DOWNF_NORMAL;
            m_nDownNo = no;

            //通知

            if(m_fDown == DOWNF_NORMAL)
                getNotify()->onNotify(m_pParent, 0, BRUSHSIZELIST->getVal(no));

            //

            _drawSel(FALSE);

            grabPointer();
        }
    }

    return TRUE;
}

//! ボタン離し時

BOOL CBrushSizeListArea::onButtonUp(AXHD_MOUSE *phd)
{
    if(m_fDown && phd->button == BUTTON_LEFT)
    {
        if(m_fDown == DOWNF_DEL)
            _deleteDat(m_nDownNo);
        else
            _drawSel(TRUE);

        //

        m_fDown = DOWNF_NONE;
        ungrabPointer();
    }

    return TRUE;
}


//========================
// サブ
//========================


//! カーソル位置のリスト番号取得

int CBrushSizeListArea::_getListNo(int x,int y)
{
    int no;

    x /= ONESIZE;
    y /= ONESIZE;

    if(x >= m_nXCnt) return -1;

    no = (y + m_pScr->getPos()) * m_nXCnt + x;

    if(no >= BRUSHSIZELIST->getCnt()) return -1;

    return no;
}

//! メニュー実行

void CBrushSizeListArea::_runMenu(int x,int y,int no)
{
    AXMenu *pm;
    int id;

    m_nDownNo = no;

    //選択枠描画

    _drawSel(FALSE);

    //メニュー

    _trgroup(strid::GROUP_BRUSHSIZELIST);

    pm = new AXMenu;

    pm->addTrMul(0, 2);
    if(no == -1) pm->enable(1, FALSE);

    id = pm->popup(NULL, x, y, 0);

    delete pm;

    //選択枠戻す

    _drawSel(TRUE);

    //処理

    if(id == 0)
        _addDat();
    else if(id == 1)
        _deleteDat(no);
}

//! サイズデータ追加

void CBrushSizeListArea::_addDat()
{
    AXString str;

    if(AXStrDialog::getString(m_pTopLevel, NULL, _str(100), &str))
    {
        BRUSHSIZELIST->addFromText(&str);

        updateAll();
    }
}

//! 指定位置のデータ削除

void CBrushSizeListArea::_deleteDat(int no)
{
    BRUSHSIZELIST->del(no);
    updateAll();
}

//! 描画

void CBrushSizeListArea::draw()
{
    LPWORD p;
    int x,y,ix,pos,cnt,val;
    char m[16];

    //背景

    m_img.clear(0xbbbbbb);

    //

    cnt = BRUSHSIZELIST->getCnt();
    pos = m_pScr->getPos() * m_nXCnt;
    p   = BRUSHSIZELIST->getBufPt(pos);

    //

    for(y = 0; y < m_nH; y += ONESIZE)
    {
        for(ix = 0, x = 0; ix < m_nXCnt; ix++, pos++, p++, x += ONESIZE)
        {
            if(pos >= cnt) goto END;

            val = *p;

            m_img.box(x + 1, y + 1, ONESIZE - 1, ONESIZE - 1, 0xffffff);

            _drawBrushPrev(x + 2, y + 2, ONESIZE - 3, ONESIZE - 3, (val > 120)? 120: val);

            AXIntToFloatStr(m, val, 1);
            m_img.drawNumber(x + 2, y + ONESIZE - 8, m, 0x0000ff);
        }
    }

END:
    redraw();
}

//! 選択用枠描画

void CBrushSizeListArea::_drawSel(BOOL bErase)
{
    int x,y;

    if(m_nDownNo == -1) return;

    x = (m_nDownNo % m_nXCnt) * ONESIZE;
    y = (m_nDownNo / m_nXCnt - m_pScr->getPos()) * ONESIZE;

    m_img.box(x, y, ONESIZE + 1, ONESIZE + 1, (bErase)? 0xbbbbbb: 0xff0000);

    redraw(x, y, ONESIZE + 1, ONESIZE + 1);
}

//! ブラシサイズプレビュー描画

void CBrushSizeListArea::_drawBrushPrev(int x,int y,int w,int h,int radius)
{
    int pitch,bpp;
    int ix,iy,jx,jy,cx,cy,f,rr,val;
    int xx[4],yy[4];
    LPBYTE p;

    p     = m_img.getBufPt(x, y);
    bpp   = m_img.getBytes();
    pitch = m_img.getPitch() - bpp * w;

    cx = w * 2;
    cy = h * 2;

    rr = (radius << 2) / 10;
    rr *= rr;

    //4x4 オーバーサンプリング

    for(iy = 0; iy < h; iy++)
    {
        f = (iy << 2) - cy;

        for(jy = 0; jy < 4; jy++, f++)
            yy[jy] = f * f;

        //

        for(ix = 0; ix < w; ix++, p += bpp)
        {
            f = (ix << 2) - cx;

            for(jx = 0; jx < 4; jx++, f++)
                xx[jx] = f * f;

            //4x4

            val = 0;

            for(jy = 0; jy < 4; jy++)
            {
                for(jx = 0; jx < 4; jx++)
                {
                    if(xx[jx] + yy[jy] < rr)
                        val += 255;
                }
            }

            //

            val >>= 4;

            if(iy >= ONESIZE - 11) val >>= 3;   //文字にかかる部分は色を薄くする

            val = 255 - val;

            m_img.setPixelBuf(p, val, val, val);
        }

        p += pitch;
    }
}
