/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/

#ifndef _AZPT_TILEIMAGE_H_
#define _AZPT_TILEIMAGE_H_

#include "struct.h"

class CTileImageA1;
class CProgressDlg;
class CImageRGB16;
class CImage32;
class CPolygonPaint;
class AXRectSize;
class AXRect;
class AXPoint;
class AXRand;
class AXString;
class AXFileWriteBuf;
class AXZlib;
class AXFile;
class AXImage;


class CTileImage
{
public:
    enum COLTYPE
    {
        COLTYPE_RGBA    = 0,
        COLTYPE_GRAY    = 1,
        COLTYPE_A_16BIT = 2,
        COLTYPE_A_1BIT  = 3
    };

    enum{ TILEPT_EMPTY = 1 };

    typedef struct
    {
        int nWidth,nHeight,nDPI,nTPCol;
    }IMGLOADINFO;

    typedef struct
    {
        AXRect  rcTile,rcClip;
        AXPoint ptTopPx;
        int pitch;
    }TILERECTINFO;

    static TILEIMGDRAWINFO *m_pinfo;
    static BRUSHDRAWPARAM *m_pbrush;
    static AXRand *m_prand;

protected:
    void        **m_ppTile;
    int         m_nColType,
                m_nTileSize,
                m_nArrayW, m_nArrayH,
                m_nOffX, m_nOffY;
    RGBFIX15    m_col;

    CTileImage  *m_pLink;

protected:
    virtual void _getPixelCol_tile(RGBAFIX15 *pdst,void *pTile,int x,int y) const = 0;
    virtual void *_getPixelBuf_tile(void *pTile,int x,int y) const = 0;
    virtual void _setPixel_buf(void *pbuf,int x,int y,const RGBAFIX15 &col) = 0;
    virtual BOOL _isTransparentTile(void *pTile) = 0;
    virtual void  _blendTile_RGB16(CImageRGB16 *pimgDst,void *pTile,int dx,int dy,int opacity,const AXRect &rcClip,void (*funcCol)(RGBAFIX15 *,const RGBFIX15 &)) = 0;
    virtual void _reverseHorz_tile(void *pTile) = 0;
    virtual void _reverseVert_tile(void *pTile) = 0;
    virtual void _rotateLeft_tile(void *pTile) = 0;
    virtual void _rotateRight_tile(void *pTile) = 0;
    virtual void _getBufRGBA_tile(RGBAFIX15 *pDst,const void *pTile) const = 0;
    virtual void _setFromBufRGBA_tile(void *pTile,const RGBAFIX15 *pSrc,BOOL bLum) = 0;

    void _freeOneTile(void **ppTile);
    BOOL _resizeTileArray_same(const CTileImage &src);
    void *_allocTile();
    void *_allocTileClear();
    BOOL _allocTileArray();
    void **_allocTileArrayNew(int w,int h,BOOL bClear);

    BOOL _setPixelDraw_getCol(int x,int y,RGBAFIX15 *pCol,void ***pppRetTile,LPINT pTX,LPINT pTY);
    void *_getTileFromPixel(int x,int y) const;
    void *_getPixelBuf(int x,int y) const;
    void *_getPixelBuf_create(int x,int y);

    void _drawBrush_lineCurve();
    void _drawBrush_point(double x,double y,double radius,double opacity);
    void _drawBrush_point_circle(double xpos,double ypos,double radius,double opacity,const RGBAFIX15 &colDraw);
    void _drawBrush_point_img(double xpos,double ypos,double radius,double opacity,const RGBAFIX15 &colDraw);
    void _drawBrush_point_img_rot(double xpos,double ypos,double radius,double opacity,const RGBAFIX15 &colDraw);

    void _loadimg_line(int y,int w,LPDWORD pSrc);

public:
    CTileImage();
    virtual ~CTileImage();

    BOOL isExist() const { return (m_ppTile != NULL); }
    int getColType() const { return m_nColType; }
    int getArrayW() const { return m_nArrayW; }
    int getArrayH() const { return m_nArrayH; }
    int getOffX() const { return m_nOffX; }
    int getOffY() const { return m_nOffY; }
    int getTileSize() const { return m_nTileSize; }

    void **getTileBuf() const { return m_ppTile; }
    void **getTileBufPt(int tx,int ty) const { return m_ppTile + ty * m_nArrayW + tx; }
    void *getTile(int tx,int ty) const { return *(m_ppTile + ty * m_nArrayW + tx); }

    const RGBFIX15 &getImageColor() const { return m_col; }
    void setImageColor(const RGBFIX15 &col) { m_col = col; }
    void getOffset(AXPoint *ppt) { ppt->x = m_nOffX, ppt->y = m_nOffY; }
    void moveOffset(int addx,int addy) { m_nOffX += addx, m_nOffY += addy; };
    void setLink(CTileImage *pimg) { m_pLink = pimg; }
    CTileImage *getLink() { return m_pLink; }

    //

    void free();
    void freeAllTile();
    void freeTilePos(int tx,int ty);
    void freeEmptyTileFromUndo();
    BOOL freeEmptyTile();
    void freeEmptyTileRect(int x,int y,int w,int h);

    BOOL resizeTileArray(int movx,int movy,int newW,int newH);
    BOOL resizeTileArray(const TILEIMGINFO &info);
    BOOL resizeTileArray_incImage();
    BOOL resizeTileArray_double(const CTileImage &src);

    BOOL create(int w,int h);
    BOOL create(const TILEIMGINFO &info);
    BOOL create(const AXRect &rc);
    BOOL createSame(const CTileImage &src);
    BOOL copy(const CTileImage &src);

    void getInfo(TILEIMGINFO *pinfo);
    BOOL calcPixelToTile(int *pTX,int *pTY,int px,int py) const;
    void calcTileToPixel(int *pPX,int *pPY,int tx,int ty) const;
    void calcTileRectToPixel(AXRect *prc) const;
    BOOL getTileRect_inImage(AXRect *prc,const AXRectSize &rcs);
    void getEnableDrawRectPixel(AXRect *prc);
    BOOL clipRectInEnableDraw(AXRect *prc);
    void getPointAdvColor(RGBADOUBLE *pdst,double x,double y,double radius);
    BOOL getExistImgRectPx(FLAGRECT *pDst,int *pTileCnt=NULL);
    DWORD getAllocTileCnt();
    void **getTileRect_inImageInfo(TILERECTINFO *pdst,const AXRectSize &rcs);
    BOOL isSameArraySize(const TILEIMGINFO &info);

    void **getTileBufAlloc(int px,int py);
    void *getTileAlloc(int tx,int ty);
    void *getTileAlloc(DWORD pos);
    void blendToRGB16(CImageRGB16 *pimgDst,const AXRectSize &rcs,int opacity,void (*funcCol)(RGBAFIX15 *,const RGBFIX15 &));
    BOOL createFromImg(const CImageRGB16 &src,CProgressDlg *pdlg);
    BOOL convertFromRGBA8bit(LPBYTE pSrcBuf,int w,int h,CProgressDlg *pdlg,int step);
    void getHistogram(LPDWORD pbuf);
    void drawFilterPrev(AXImage *pimgDst,const AXRectSize &rcs);

    //

    virtual void getSaveTileBuf(LPBYTE pDst,void *pTile) = 0;
    virtual void setSaveTileBuf(void *pTile,LPBYTE pSrc) = 0;

    //[pixel]

    void setPixelDraw(int x,int y,const RGBAFIX15 &col);
    void setPixelDotPen(int x,int y,const RGBAFIX15 &col);
    void setPixel_create(int x,int y,const RGBAFIX15 &col);
    void setPixel_create2(int x,int y,const RGBAFIX15 &col);
    void setPixel_combine(int x,int y,const RGBAFIX15 &col,int dstopa,void (*funcCol)(RGBAFIX15 *,const RGBFIX15 &));
    void setPixel_colfunc(int x,int y,const RGBAFIX15 &col);
    void setPixel_subdraw(int x,int y,const RGBAFIX15 &col);
    void setPixelDraw_addsub(int x,int y,int v);

    void getPixel(RGBAFIX15 *pdst,int x,int y) const;
    void getPixelUndoImg(RGBAFIX15 *pdst,int x,int y) const;
    void getPixel(RGBAFIX15 *pdst,int x,int y,BOOL bClip) const;
    void getPixelMode(RGBAFIX15 *pdst,int x,int y,int mode) const;
    BOOL isPixelTransparent(int x,int y) const;

    //

    void colfunc_normal(RGBAFIX15 *pdst,const RGBAFIX15 &src,LPVOID pParam);
    void colfunc_inverse_alpha(RGBAFIX15 *pdst,const RGBAFIX15 &src,LPVOID pParam);
    void colfunc_comp_alpha(RGBAFIX15 *pdst,const RGBAFIX15 &src,LPVOID pParam);
    void colfunc_overwrite(RGBAFIX15 *pdst,const RGBAFIX15 &src,LPVOID pParam);
    void colfunc_erase(RGBAFIX15 *pdst,const RGBAFIX15 &src,LPVOID pParam);
    void colfunc_add(RGBAFIX15 *pdst,const RGBAFIX15 &src,LPVOID pParam);
    void colfunc_dodge(RGBAFIX15 *pdst,const RGBAFIX15 &src,LPVOID pParam);
    void colfunc_burn(RGBAFIX15 *pdst,const RGBAFIX15 &src,LPVOID pParam);
    void colfunc_blur(RGBAFIX15 *pdst,const RGBAFIX15 &src,LPVOID pParam);

    //[brush]

    void startDrawBrushFree(double x,double y,double press);
    void startDrawBrushNoneFree(double x,double y);
    void startBrushWater(double x,double y);
    void drawBrush_free(double x,double y,double press);
    double drawBrush_line(double x1,double y1,double x2,double y2,double pressSt,double pressEd,double tstart);
    void drawBrush_lineHeadTail(double x1,double y1,double x2,double y2,WORD wHeadTail);
    void drawBrush_point(int x,int y,int radius,int opacity);

    //[draw]

    void drawLineB(int x1,int y1,int x2,int y2,const RGBAFIX15 &col,BOOL bNoStart);
    void drawLineF(int x1,int y1,int x2,int y2,const RGBAFIX15 &col);
    void drawBox(int x,int y,int w,int h,const RGBAFIX15 &col);
    void drawCircle(double cx,double cy,double xr,double yr,const CANVASVIEWPARAM &param,BOOL bHRev,BOOL bBrush);
    void drawEllipseDot(int x1,int y1,int x2,int y2);
    void drawBezier(DPOINT *pt,WORD wHeadTail,BOOL bBrush);
    void drawFillBox(int x,int y,int w,int h,int opacity);
    BOOL drawFillPolygon(CPolygonPaint *pPoly,int opacity,BOOL bAntiAlias);
    void drawFillCircle(double cx,double cy,double xr,double yr,const CANVASVIEWPARAM &param,BOOL bHRev,int opacity,BOOL bAntiAlias);
    BOOL drawFillPolygonNoAA(CPolygonPaint *pPoly,BOOL bOn);

    void drawGradient_line(int x1,int y1,int x2,int y2,const AXRect &rc,int opacity,LPBYTE pDat);
    void drawGradient_circle(int x1,int y1,int x2,int y2,const AXRect &rc,int opacity,LPBYTE pDat);
    void drawGradient_box(int x1,int y1,int x2,int y2,const AXRect &rc,int opacity,LPBYTE pDat);
    void drawGradient_radial(int x1,int y1,int x2,int y2,const AXRect &rc,int opacity,LPBYTE pDat);
    void drawGradient_pix(int x,int y,double pos,int opacity,LPBYTE pDat);

    //[edit]

    BOOL createConvert(const CTileImage &src,BOOL bLum,CProgressDlg *pdlg);
    BOOL combine(const CTileImage &src,const AXRect &rc,int srcopa,int dstopa,void (*funcCol)(RGBAFIX15 *,const RGBFIX15 &),CProgressDlg *pdlg,int step);
    void replaceColor_tp(const RGBAFIX15 &colDst);
    void replaceColor(const RGBAFIX15 &colSrc,const RGBAFIX15 &colDst);

    void reverseHorz();
    void reverseVert();
    void rotateLeft();
    void rotateRight();

    void copyRect(CTileImage *pimgSrc,const AXRect &rc);
    void clearRect(const AXRect &rc);
    void copyClearRect(CTileImage *pimgCopy,const AXRect &rc);

    void reverseHorzRect(const AXRectSize &rcs);
    void reverseVertRect(const AXRectSize &rcs);
    void rotateLeftRect(CTileImage *pimgSrc,const AXRectSize &rcs);
    void rotateRightRect(CTileImage *pimgSrc,const AXRectSize &rcs);

    void scaleAndRotate(CTileImage *pimgSrc,const AXRect &rc,double dScale,double dAngle,BOOL bHiQuality);
    BOOL getScaleRotRect(AXRect *prc,double dScale,double dAngle);

    void copySelectImage(CTileImage *pimgSrc,const CTileImageA1 *pimgSel);
    void copyCutSelectImage(CTileImage *pimgSrc,const CTileImageA1 *pimgSel);
    void blendSelectImage(const CTileImage *pimgSrc,const CTileImageA1 *pimgSel,BOOL bOverwrite);

    //[file]

    BOOL loadImage(const AXString &filename,IMGLOADINFO *pInfo,CProgressDlg *pdlg);
    BOOL loadBMP(const AXString &filename,IMGLOADINFO *pInfo,CProgressDlg *pdlg);
    BOOL loadPNG(const AXString &filename,IMGLOADINFO *pInfo,CProgressDlg *pdlg);
    BOOL loadJPEG(const AXString &filename,IMGLOADINFO *pInfo,CProgressDlg *pdlg);
    BOOL loadGIF(const AXString &filename,IMGLOADINFO *pInfo,CProgressDlg *pdlg);

    BOOL savePNG32bit(const AXString &filename,int nDPI,CProgressDlg *pdlg);

    BOOL saveCopyFile(const AXString &filename,DWORD col);
    BOOL loadCopyFile(AXFile *pfile);

    void saveTiles(AXFileWriteBuf *pfile,AXZlib *pzlib,LPBYTE pWorkBuf,CProgressDlg *pdlg,int step);
    BOOL loadTiles(AXFile *pfile,AXZlib *pzlib,LPBYTE pWorkBuf,CProgressDlg *pdlg,int step);

    //

    static void initCurve();
    static void resetWaterCol();
};

#endif
