/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CTileImage [pixel] - 点の描画・取得、色処理
*/

#include <string.h>

#include "CTileImage.h"

#include "CImage8.h"


//---------------------------

typedef struct
{
    int x,y;
    void *pTile;
}COLFUNCPARAM;

//---------------------------


//==================================
// ツールなどでの描画時用、色セット
//==================================


//! 色セット（描画時用。アンドゥ用処理も同時に行う）
/*!
    タイルがない場合は作成される。@n
    タイル配列外の場合は、キャンバス範囲内なら配列がリサイズされる。
*/

void CTileImage::setPixelDraw(int x,int y,const RGBAFIX15 &col)
{
    void **ppTile,**ppUndo;
    int tx,ty,bEmpty = FALSE;
    RGBAFIX15 colDst;

    //-------- 描画後の色＆タイル位置取得（色が変わらないなら処理しない）

    colDst = col;

    if(_setPixelDraw_getCol(x, y, &colDst, &ppTile, &tx, &ty))
        return;

    //-------- タイル配列範囲外

    if(!ppTile)
    {
        //キャンバス範囲外なら描画しない

        if(x < 0 || y < 0 || x >= m_pinfo->nImgW || y >= m_pinfo->nImgH)
            return;

        //配列リサイズ＆タイル位置再取得

        if(!resizeTileArray_incImage()) return;

        calcPixelToTile(&tx, &ty, x, y);

        ppTile = getTileBufPt(tx, ty);

        //UNDO用配列もリサイズ
        //※失敗したらアンドゥは機能させない

        if(!(m_pinfo->pimgUndo)->_resizeTileArray_same(*this))
        {
            m_pinfo->fUndoErr |= 2;
            (m_pinfo->pimgUndo)->free();
        }
    }

    //---------- タイルがない場合確保

    if(!(*ppTile))
    {
        *ppTile = _allocTileClear();

        if(!(*ppTile))
        {
            m_pinfo->fUndoErr |= 1;
            return;
        }

        bEmpty = TRUE;
    }

    //----------- UNDO用に元イメージコピー

    ppUndo = (m_pinfo->pimgUndo)->getTileBuf();

    if(ppUndo)
    {
        ppUndo += ty * m_nArrayW + tx;

        if(bEmpty)
            //元が空の場合1をセット
            *ppUndo = (void *)TILEPT_EMPTY;
        else if(!(*ppUndo))
        {
            //すでにタイルがある場合はコピー済み、なければ作成&コピー

            *ppUndo = _allocTile();

            if(!(*ppUndo))
                m_pinfo->fUndoErr |= 2;
            else
                ::memcpy(*ppUndo, *ppTile, m_nTileSize);
        }
    }

    //--------- 更新範囲

    m_pinfo->rcfDraw.incPoint(x, y);

    //--------- 色セット

    void *pdst = _getPixelBuf(x, y);

    _setPixel_buf(pdst, x, y, colDst);
}

//! 描画後の色を取得＆タイル位置取得
/*!
    @param pCol  塗る色を入れておく。結果が返る。
    @param pppRetTile 描画先のタイルポインタが返る。NULLで配列範囲外
    @return TRUE で色の変化なし
*/

BOOL CTileImage::_setPixelDraw_getCol(int x,int y,RGBAFIX15 *pCol,void ***pppRetTile,LPINT pTX,LPINT pTY)
{
    void **ppTile = NULL;
    RGBAFIX15 colRes,colDst,colSrc,colMask;
    int tx,ty,i,f;
    COLFUNCPARAM param;

    colSrc = *pCol;

    //選択範囲外なら処理なし

    if(m_pinfo->pimgSel)
    {
        if((m_pinfo->pimgSel)->isPixelTransparent(x, y))
            return TRUE;
    }

    //レイヤマスク値取得（透明なら処理なし）

    if(!m_pinfo->pimgMask)
        colMask.a = 0x8000;
    else
    {
        (m_pinfo->pimgMask)->getPixel(&colMask, x, y);

        if(colMask.a == 0) return TRUE;
    }

    //-------- 描画先の色取得＆タイル位置セット

    colDst.zero();

    if(!calcPixelToTile(&tx, &ty, x, y))
        *pppRetTile = NULL;
    else
    {
        ppTile = getTileBufPt(tx, ty);

        if(*ppTile)
            _getPixelCol_tile(&colDst, *ppTile, x, y);

        *pppRetTile = ppTile;
        *pTX        = tx;
        *pTY        = ty;
    }

    //------- アルファマスク（保護）

    if(m_pinfo->nAMaskType == 2 && !colDst.a)
        //透明色保護
        return TRUE;
    else if(m_pinfo->nAMaskType == 3 && colDst.a)
        //不透明色保護
        return TRUE;

    //------- 色マスク判定

    if(m_pinfo->nColMaskType == 1)
    {
        //マスク（描画先が指定色なら描画しない）

        for(i = 0; i < 6; i++)
        {
            if(m_pinfo->colMask[i].compareColMask(colDst)) return TRUE;
        }
    }
    else if(m_pinfo->nColMaskType == 2)
    {
        //逆マスク（描画先が指定色なら描画、それ以外は描画しない）

        for(i = 0, f = 1; i < 6; i++)
        {
            if(m_pinfo->colMask[i].compareColMask(colDst)) { f = 0; break; }
        }

        if(f) return TRUE;
    }

    //--------------

    //テクスチャ適用

    if(m_pinfo->pTexture)
        colSrc.a = (colSrc.a * (m_pinfo->pTexture)->getPixelTexture(x, y) + 127) / 255;

    //色処理

    colRes = colDst;

    param.x = x;
    param.y = y;
    param.pTile = (ppTile)? *ppTile: NULL;

    (this->*(m_pinfo->funcColor))(&colRes, colSrc, &param);

    //レイヤマスク値適用

    if(colRes.a > colMask.a)
        colRes.a = colMask.a;

    //アルファマスク（アルファ維持）

    if(m_pinfo->nAMaskType == 1)
        colRes.a = colDst.a;

    //描画前と描画後が同じなら色をセットしない

    if(colRes == colDst)
        return TRUE;
    else
    {
        *pCol = colRes;

        return FALSE;
    }
}


//===========================
// 色セット
//===========================


//! 色をセット（タイルを作成。A=0 の場合も含む）

void CTileImage::setPixel_create(int x,int y,const RGBAFIX15 &col)
{
    void *p = _getPixelBuf_create(x, y);

    if(p) _setPixel_buf(p, x, y, col);
}

//! 色をセット（タイルを作成。A=0 は除く）

void CTileImage::setPixel_create2(int x,int y,const RGBAFIX15 &col)
{
    if(col.a)
    {
        void *p = _getPixelBuf_create(x, y);

        if(p) _setPixel_buf(p, x, y, col);
    }
}

//! ドットペン用ピクセルセット

void CTileImage::setPixelDotPen(int x,int y,const RGBAFIX15 &col)
{
    int xx,yy,ix,iy,size;
    LPBYTE p;

    p    = m_pinfo->pDotBrush->getBuf();
    size = m_pinfo->nDotPenSize;

    xx = x - (size >> 1);
    yy = y - (size >> 1);

    for(iy = size; iy; iy--, yy++)
    {
        for(ix = size; ix; ix--, p++, xx++)
        {
            if(*p)
                setPixelDraw(xx, yy, col);
        }

        xx -= size;
    }
}

//! 結合時の色セット

void CTileImage::setPixel_combine(int x,int y,const RGBAFIX15 &col,int dstopa,
        void (*funcCol)(RGBAFIX15 *,const RGBFIX15 &))
{
    RGBAFIX15 src,dst,dstBk;
    RGBFIX15 dcol;

    getPixel(&dst, x, y);

    if(col.a || dst.a)
    {
        dstBk = dst;
        dst.a = dst.a * dstopa >> 7;

        //合成

        if(dst.a)
            dcol.set(dst);
        else
            dcol.white();

        src = col;
        (*funcCol)(&src, dcol);

        //RGBA合成

        colfunc_normal(&dst, src, NULL);

        if(dst.a || dstBk.a)
            setPixel_create(x, y, dst);
    }
}

//! 色関数などあり (配列拡張なし)

void CTileImage::setPixel_colfunc(int x,int y,const RGBAFIX15 &col)
{
    RGBAFIX15 colDst;
    void **ppTile,*p;
    int tx,ty;

    colDst = col;

    if(_setPixelDraw_getCol(x, y, &colDst, &ppTile, &tx, &ty))
        return;

    //セット

    p = _getPixelBuf_create(x, y);
    if(p) _setPixel_buf(p, x, y, colDst);
}

//! 色セット（範囲計算＆配列拡張あり）

void CTileImage::setPixel_subdraw(int x,int y,const RGBAFIX15 &col)
{
    int tx,ty;
    void **pp,*pbuf;
    RGBAFIX15 colRes,colDst;

    //セットする色と元の色比較

    getPixel(&colDst, x, y);

    colRes = colDst;

    (this->*(m_pinfo->funcColor))(&colRes, col, NULL);

    if(colRes == colDst) return;

    //タイル位置 （キャンバス範囲内なら配列拡張）

    if(!calcPixelToTile(&tx, &ty, x, y))
    {
        //キャンバス範囲外なら描画しない

        if(x < 0 || y < 0 || x >= m_pinfo->nImgW || y >= m_pinfo->nImgH)
            return;

        //配列リサイズ＆タイル位置再取得

        if(!resizeTileArray_incImage()) return;

        calcPixelToTile(&tx, &ty, x, y);
    }

    //タイル確保

    pp = getTileBufPt(tx, ty);

    if(!(*pp))
    {
        *pp = _allocTileClear();
        if(!(*pp)) return;
    }

    //セット

    pbuf = _getPixelBuf(x, y);
    _setPixel_buf(pbuf, x, y, colRes);

    //

    m_pinfo->rcfDraw.incPoint(x, y);
}

//! RGBに指定値を加算/減算(透明部分は除く)

void CTileImage::setPixelDraw_addsub(int x,int y,int v)
{
    RGBAFIX15 col;
    int i,n;

    getPixel(&col, x, y);

    if(col.a)
    {
        for(i = 0; i < 3; i++)
        {
            n = col.c[i] + v;
            if(n < 0) n = 0; else if(n > 0x8000) n = 0x8000;

            col.c[i] = n;
        }

        setPixelDraw(x, y, col);
    }
}


//===========================
// 色・バッファ位置取得
//===========================


//! 指定px位置の色取得（範囲外は透明）

void CTileImage::getPixel(RGBAFIX15 *pdst,int x,int y) const
{
    int tx,ty;
    void *p;

    if(!calcPixelToTile(&tx, &ty, x, y))
        pdst->zero();
    else
    {
        p = getTile(tx, ty);

        if(p)
            _getPixelCol_tile(pdst, p, x, y);
        else
            pdst->zero();
    }
}

//! アンドゥイメージから色取得
/*
    元イメージが透明な部分のタイルポインタは TILEPT_EMPTY 値になっているので、透過色を返す。
    タイルが確保されていないタイルは、変更されていない部分なので、R 値を 0xffff とする。
*/

void CTileImage::getPixelUndoImg(RGBAFIX15 *pdst,int x,int y) const
{
    int tx,ty;
    void *p;

    if(!calcPixelToTile(&tx, &ty, x, y))
        pdst->zero();
    else
    {
        p = getTile(tx, ty);

        if(p == (void *)TILEPT_EMPTY)
            pdst->zero();
        else if(p)
            _getPixelCol_tile(pdst, p, x, y);
        else
        {
            pdst->r = 0xffff;
            pdst->g = pdst->b = pdst->a = 0;
        }
    }
}

//! クリッピング有無指定付き色取得

void CTileImage::getPixel(RGBAFIX15 *pdst,int x,int y,BOOL bClip) const
{
    int tx,ty;
    void *p;

    if(bClip)
    {
        if(x < 0) x = 0; else if(x >= m_pinfo->nImgW) x = m_pinfo->nImgW - 1;
        if(y < 0) y = 0; else if(y >= m_pinfo->nImgH) y = m_pinfo->nImgH - 1;
    }

    if(!calcPixelToTile(&tx, &ty, x, y))
        pdst->zero();
    else
    {
        p = getTile(tx, ty);

        if(p)
            _getPixelCol_tile(pdst, p, x, y);
        else
            pdst->zero();
    }
}

//! 色取得
/*!
    @param mode [0]クリッピングなし [1]キャンバス範囲でクリッピング [2]キャンバス範囲外はループ
*/

void CTileImage::getPixelMode(RGBAFIX15 *pdst,int x,int y,int mode) const
{
    int tx,ty;
    void *p;

    if(mode == 1)
    {
        if(x < 0) x = 0; else if(x >= m_pinfo->nImgW) x = m_pinfo->nImgW - 1;
        if(y < 0) y = 0; else if(y >= m_pinfo->nImgH) y = m_pinfo->nImgH - 1;
    }
    else if(mode == 2)
    {
        while(x < 0) x += m_pinfo->nImgW;
        while(x >= m_pinfo->nImgW) x -= m_pinfo->nImgW;

        while(y < 0) y += m_pinfo->nImgH;
        while(y >= m_pinfo->nImgH) y -= m_pinfo->nImgH;
    }

    if(!calcPixelToTile(&tx, &ty, x, y))
        pdst->zero();
    else
    {
        p = getTile(tx, ty);

        if(p)
            _getPixelCol_tile(pdst, p, x, y);
        else
            pdst->zero();
    }
}

//! 指定px位置が透明かどうか

BOOL CTileImage::isPixelTransparent(int x,int y) const
{
    RGBAFIX15 col;

    getPixel(&col, x, y);

    return (col.a == 0);
}

//! 指定px位置の色バッファ位置取得
/*!
    @return 配列範囲外は NULL
*/

void *CTileImage::_getPixelBuf(int x,int y) const
{
    void *p = _getTileFromPixel(x, y);

    if(p)
        return _getPixelBuf_tile(p, x, y);
    else
        return NULL;
}

//! 指定px位置の色バッファ位置取得（＋タイル作成）
/*!
    その位置にタイルがない場合は作成する。ただし、タイル配列の拡張は行わない。
*/

void *CTileImage::_getPixelBuf_create(int x,int y)
{
    int tx,ty;
    void **pp;

    //タイル位置

    if(!calcPixelToTile(&tx, &ty, x, y)) return NULL;

    //タイルがない場合確保

    pp = getTileBufPt(tx, ty);

    if(!(*pp))
    {
        *pp = _allocTileClear();
        if(!(*pp)) return NULL;
    }

    //バッファ位置

    return _getPixelBuf_tile(*pp, x, y);
}


//=============================
// 色処理関数
//=============================
/*
    pdst(RGBA) -> src(RGBA) = pdst(RGBA)
*/


//! 通常 RGBA 合成

void CTileImage::colfunc_normal(RGBAFIX15 *pdst,const RGBAFIX15 &src,LPVOID pParam)
{
    double sa,da,na;

    if(src.a == 0x8000)
        *pdst = src;
    else
    {
        sa = (double)src.a / (1 << 15);
        da = (double)pdst->a / (1 << 15);

        //A

        na = sa + da - sa * da;

        pdst->a = (int)(na * (1 << 15) + 0.5);

        //RGB

        if(pdst->a)
        {
            da = da * (1.0 - sa);
            na = 1.0 / na;

            pdst->r = (WORD)((src.r * sa + pdst->r * da) * na + 0.5);
            pdst->g = (WORD)((src.g * sa + pdst->g * da) * na + 0.5);
            pdst->b = (WORD)((src.b * sa + pdst->b * da) * na + 0.5);
        }
    }
}

//! Aを反転

void CTileImage::colfunc_inverse_alpha(RGBAFIX15 *pdst,const RGBAFIX15 &src,LPVOID pParam)
{
    pdst->a = 0x8000 - pdst->a;
}

//! Aを比較して、値が大きければ色を上書き

void CTileImage::colfunc_comp_alpha(RGBAFIX15 *pdst,const RGBAFIX15 &src,LPVOID pParam)
{
    if(src.a > pdst->a)
        *pdst = src;
}

//! 完全上書き

void CTileImage::colfunc_overwrite(RGBAFIX15 *pdst,const RGBAFIX15 &src,LPVOID pParam)
{
    *pdst = src;
}

//! 消しゴム

void CTileImage::colfunc_erase(RGBAFIX15 *pdst,const RGBAFIX15 &src,LPVOID pParam)
{
    int a = pdst->a;

    if(a)
    {
        a -= src.a;
        if(a < 0) a = 0;

        if(a)
            pdst->a = a;
        else
            pdst->zero();
    }
}

//! 加算

void CTileImage::colfunc_add(RGBAFIX15 *pdst,const RGBAFIX15 &src,LPVOID pParam)
{
    int c;

    c = pdst->r + src.a;
    if(c > 0x8000) c = 0x8000;
    pdst->r = c;

    c = pdst->g + src.a;
    if(c > 0x8000) c = 0x8000;
    pdst->g = c;

    c = pdst->b + src.a;
    if(c > 0x8000) c = 0x8000;
    pdst->b = c;
}

//! 覆い焼き
/*!
    src.a を強さとする。src の r,g,b は使わない。pdst->a は変化しない。
*/

void CTileImage::colfunc_dodge(RGBAFIX15 *pdst,const RGBAFIX15 &src,LPVOID pParam)
{
    int div,c;

    if(src.a == 0x8000)
        pdst->r = pdst->g = pdst->b = 0x8000;
    else
    {
        div = 0x8000 - src.a;

        //R

        c = (pdst->r << 15) / div;
        if(c > 0x8000) c = 0x8000;
        pdst->r = c;

        //G

        c = (pdst->g << 15) / div;
        if(c > 0x8000) c = 0x8000;
        pdst->g = c;

        //B

        c = (pdst->b << 15) / div;
        if(c > 0x8000) c = 0x8000;
        pdst->b = c;
    }
}

//! 焼き込み
/*!
    src.a を強さとする。src の r,g,b は使わない。pdst->a は変化しない。
*/

void CTileImage::colfunc_burn(RGBAFIX15 *pdst,const RGBAFIX15 &src,LPVOID pParam)
{
    int c,div;

    if(src.a == 0x8000)
        pdst->r = pdst->g = pdst->b = 0;
    else
    {
        div = 0x8000 - src.a;

        //R

        c = 0x8000 - ((0x8000 - pdst->r) << 15) / div;
        if(c < 0) c = 0;
        pdst->r = c;

        //G

        c = 0x8000 - ((0x8000 - pdst->g) << 15) / div;
        if(c < 0) c = 0;
        pdst->g = c;

        //B

        c = 0x8000 - ((0x8000 - pdst->b) << 15) / div;
        if(c < 0) c = 0;
        pdst->b = c;
    }
}

//! ぼかし
/*!
    3x3の周囲の色の平均値をとる。
    src の色は使わない。ブラシ間隔で強さを調節する。
*/

void CTileImage::colfunc_blur(RGBAFIX15 *pdst,const RGBAFIX15 &src,LPVOID pParam)
{
    int x,y,px,py,ix,iy,r,g,b,a;
    double div;
    void *pTile;
    RGBAFIX15 col;

    x     = ((COLFUNCPARAM *)pParam)->x;
    y     = ((COLFUNCPARAM *)pParam)->y;
    pTile = ((COLFUNCPARAM *)pParam)->pTile;

    px = (x - m_nOffX) & 63;
    py = (y - m_nOffY) & 63;

    r = g = b = a = 0;

    if(pTile && px >= 1 && px <= 62 && py >= 1 && py <= 62)
    {
        //確保されているタイルかつ3x3がタイル内の場合

        for(iy = -1; iy <= 1; iy++)
        {
            for(ix = -1; ix <= 1; ix++)
            {
                _getPixelCol_tile(&col, pTile, x + ix, y + iy);

                r += (col.r * col.a) >> 15;
                g += (col.g * col.a) >> 15;
                b += (col.b * col.a) >> 15;
                a += col.a;
            }
        }
    }
    else
    {
        //タイルがまたがる場合、またはタイル未確保の場合

        for(iy = -1; iy <= 1; iy++)
        {
            for(ix = -1; ix <= 1; ix++)
            {
                getPixel(&col, x + ix, y + iy);

                r += (col.r * col.a) >> 15;
                g += (col.g * col.a) >> 15;
                b += (col.b * col.a) >> 15;
                a += col.a;
            }
        }
    }

    //

    if(a)
    {
        div = 1.0 / ((double)a / 0x8000);

        pdst->r = (WORD)(r * div + 0.5);
        pdst->g = (WORD)(g * div + 0.5);
        pdst->b = (WORD)(b * div + 0.5);
        pdst->a = (WORD)(a / 9.0 + 0.5);
    }
}
