/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CTileImage [draw] - 図形などの描画
*/

#include <stdlib.h>
#include <math.h>

#include "CTileImage.h"

#include "CPolygonPaint.h"


//! ドット線直線描画（ブレゼンハム）
/*!
    @param bNoStart 開始位置は点を打たない
*/

void CTileImage::drawLineB(int x1,int y1,int x2,int y2,const RGBAFIX15 &col,BOOL bNoStart)
{
    int sx,sy,dx,dy,a,a1,e;
    void (CTileImage::*func)(int,int,const RGBAFIX15 &) = m_pinfo->funcDrawPixel;

    if(x1 == x2 && y1 == y2)
    {
        if(!bNoStart) (this->*func)(x1, y1, col);
        return;
    }

    //

    dx = (x1 < x2)? x2 - x1: x1 - x2;
    dy = (y1 < y2)? y2 - y1: y1 - y2;
    sx = (x1 <= x2)? 1: -1;
    sy = (y1 <= y2)? 1: -1;

    if(dx >= dy)
    {
        a  = 2 * dy;
        a1 = a - 2 * dx;
        e  = a - dx;

        if(bNoStart)
        {
            if(e >= 0) y1 += sy, e += a1;
            else e += a;
            x1 += sx;
        }

        while(x1 != x2)
        {
            (this->*func)(x1, y1, col);

            if(e >= 0) y1 += sy, e += a1;
            else e += a;
            x1 += sx;
        }
    }
    else
    {
        a  = 2 * dx;
        a1 = a - 2 * dy;
        e  = a - dy;

        if(bNoStart)
        {
            if(e >= 0) x1 += sx, e += a1;
            else e += a;
            y1 += sy;
        }

        while(y1 != y2)
        {
            (this->*func)(x1, y1, col);

            if(e >= 0) x1 += sx, e += a1;
            else e += a;
            y1 += sy;
        }
    }

    (this->*func)(x1, y1, col);
}

//! ドット線直線描画（固定小数点）
/*!
    ※終点は描画しない（ドット細線用）
*/

void CTileImage::drawLineF(int x1,int y1,int x2,int y2,const RGBAFIX15 &col)
{
    int dx,dy,f,d,add = 1;
    void (CTileImage::*func)(int,int,const RGBAFIX15 &) = m_pinfo->funcDrawPixel;

    if(x1 == x2 && y1 == y2) return;

    dx = (x1 < x2)? x2 - x1: x1 - x2;
    dy = (y1 < y2)? y2 - y1: y1 - y2;

    if(dx > dy)
    {
        f = y1 << 16;
        d = ((y2 - y1) << 16) / (x2 - x1);
        if(x1 > x2) add = -1, d = -d;

        while(x1 != x2)
        {
            (this->*func)(x1, (f + 0x8000) >> 16, col);
            f  += d;
            x1 += add;
        }
    }
    else
    {
        f = x1 << 16;
        d = ((x2 - x1) << 16) / (y2 - y1);
        if(y1 > y2) add = -1, d = -d;

        while(y1 != y2)
        {
            (this->*func)((f + 0x8000) >> 16, y1, col);
            f  += d;
            y1 += add;
        }
    }
}

//! 四角枠描画

void CTileImage::drawBox(int x,int y,int w,int h,const RGBAFIX15 &col)
{
    int i;
    void (CTileImage::*func)(int,int,const RGBAFIX15 &) = m_pinfo->funcDrawPixel;

    //横線

    for(i = 0; i < w; i++)
    {
        (this->*func)(x + i, y, col);
        (this->*func)(x + i, y + h - 1, col);
    }

    //縦線

    for(i = 1; i < h - 1; i++)
    {
        (this->*func)(x, y + i, col);
        (this->*func)(x + w - 1, y + i, col);
    }
}

//! 円枠描画

void CTileImage::drawCircle(double cx,double cy,double xr,double yr,
    const CANVASVIEWPARAM &param,BOOL bHRev,BOOL bBrush)
{
    int i,div;
    double add,rr,xx,yy,x1,y1,sx,sy,nx,ny,tbrush;

    //開始位置（0度）

    x1 = xr * param.dCosRev; if(bHRev) x1 = -x1;

    sx = cx + x1;
    sy = cy + xr * param.dSinRev;
    
    //水彩
    
    if(bBrush) startDrawBrushNoneFree(sx, sy);

    //分割数

    rr = (xr > yr)? xr: yr;

    div = (int)((rr * M_PI * 2) / 4.0);
    if(div < 30) div = 30; else if(div > 400) div = 400;

    //

    add = M_PI * 2 / div;
    rr  = add;

    tbrush = 0;

    for(i = 0; i < div; i++, rr += add)
    {
        xx = xr * ::cos(rr);
        yy = yr * ::sin(rr);

        x1 = xx * param.dCosRev + yy * param.dSinRev; if(bHRev) x1 = -x1;
        y1 = xx * param.dSinRev - yy * param.dCosRev;

        nx = x1 + cx;
        ny = y1 + cy;

        if(bBrush)
            tbrush = drawBrush_line(sx, sy, nx, ny, 1, 1, tbrush);
        else
            drawLineB((int)(sx + 0.5), (int)(sy + 0.5), (int)(nx + 0.5), (int)(ny + 0.5), m_pinfo->colDraw, i);

        sx = nx, sy = ny;
    }
}

//! 楕円（ドット用）
/*!
    ドットペン・円で、キャンバス回転が0の時用
*/

void CTileImage::drawEllipseDot(int x1,int y1,int x2,int y2)
{
    LONGLONG a,b,b1,dx,dy,e,e2;
    void (CTileImage::*func)(int,int,const RGBAFIX15 &) = m_pinfo->funcDrawPixel;
    RGBAFIX15 col;

    col = m_pinfo->colDraw;

    if(x1 == x2 && y1 == y2)
    {
        (this->*func)(x1, y1, col);
        return;
    }

    //

    a = (x1 < x2)? x2 - x1: x1 - x2;
    b = (y1 < y2)? y2 - y1: y1 - y2;

    if(a >= b)
    {
        //横長

        b1 = b & 1;
        dx = 4 * (1 - a) * b * b;
        dy = 4 * (b1 + 1) * a * a;
        e  = dx + dy + b1 * a * a;

        if(x1 > x2) { x1 = x2; x2 += a; }
        if(y1 > y2) y1 = y2;

        y1 += (b + 1) / 2;
        y2 = y1 - b1;

        a *= 8 * a;
        b1 = 8 * b * b;

        do
        {
            (this->*func)(x2, y1, col);
            if(x1 != x2) (this->*func)(x1, y1, col);
            if(y1 != y2) (this->*func)(x1, y2, col);
            if(x1 != x2 && y1 != y2) (this->*func)(x2, y2, col);

            e2 = 2 * e;
            if(e2 <= dy) { y1++; y2--; e += dy += a; }
            if(e2 >= dx || 2 * e > dy) { x1++; x2--; e += dx += b1; }
        }while(x1 <= x2);
    }
    else
    {
        //縦長

        b1 = a & 1;
        dy = 4 * (1 - b) * a * a;
        dx = 4 * (b1 + 1) * b * b;
        e  = dx + dy + b1 * b * b;

        if(y1 > y2) { y1 = y2; y2 += b; }
        if(x1 > x2) x1 = x2;

        x1 += (a + 1) / 2;
        x2 = x1 - b1;

        b *= 8 * b;
        b1 = 8 * a * a;

        do
        {
            (this->*func)(x2, y1, col);
            if(x1 != x2) (this->*func)(x1, y1, col);
            if(y1 != y2) (this->*func)(x1, y2, col);
            if(x1 != x2 && y1 != y2) (this->*func)(x2, y2, col);

            e2 = 2 * e;
            if(e2 <= dx) { x1++; x2--; e += dx += b; }
            if(e2 >= dy || 2 * e > dx) { y1++; y2--; e += dy += b1; }
        }while(y1 <= y2);
    }
}

//! ベジェ曲線

void CTileImage::drawBezier(DPOINT *pt,WORD wHeadTail,BOOL bBrush)
{
    double ind,outd,nx,ny,sx,sy,pressSt,pressEd,tbrush;
    double t,tt,t1,t2,t3,t4,len,add;
    int i,head,tail,div,nT;

    //入り抜き

    if(bBrush)
    {
        head = wHeadTail >> 8;
        tail = wHeadTail & 0xff;

        if(tail > 100 - head) tail = 100 - head;

        ind  = (head)? 1.0 / ((double)head / 100.0): 0;
        outd = (tail)? 1.0 / ((double)tail / 100.0): 0;

        pressSt = (head)? 0: 1.0;
    }

    //4分割で長さ取得 -> 分割数計算

    sx = pt[0].x, sy = pt[0].y;

    for(i = 0, t = 0.25, len = 0; i < 3; i++, t += 0.25)
    {
        tt = 1 - t;

        t1 = tt * tt * tt;
        t2 = 3 * t * tt * tt;
        t3 = 3 * t * t * tt;
        t4 = t * t * t;

        nx = pt[0].x * t1 + pt[2].x * t2 + pt[3].x * t3 + pt[1].x * t4;
        ny = pt[0].y * t1 + pt[2].y * t2 + pt[3].y * t3 + pt[1].y * t4;

        len += ::sqrt((nx - sx) * (nx - sx) + (ny - sy) * (ny - sy));

        sx = nx, sy = ny;
    }

    if(len == 0) return;

    div = (int)(len / 4.0);
    if(div < 30) div = 30; else if(div > 400) div = 400;

    add = 1.0 / div;
    
    //水彩
    
    if(bBrush) startDrawBrushNoneFree(sx, sy);

    //曲線

    sx = pt[0].x, sy = pt[0].y;

    tbrush = 0;

    for(i = 0, t = add; i < div; i++, t += add)
    {
        tt = 1 - t;

        t1 = tt * tt * tt;
        t2 = 3 * t * tt * tt;
        t3 = 3 * t * t * tt;
        t4 = t * t * t;

        nx = pt[0].x * t1 + pt[2].x * t2 + pt[3].x * t3 + pt[1].x * t4;
        ny = pt[0].y * t1 + pt[2].y * t2 + pt[3].y * t3 + pt[1].y * t4;

        //描画

        if(!bBrush)
            drawLineB((int)sx, (int)sy, (int)nx, (int)ny, m_pinfo->colDraw, i);
        else
        {
            nT = (int)(t * 100);

            if(nT < head)
                pressEd = t * ind;
            else if(nT > 100 - tail)
                pressEd = (1.0 - t) * outd;
            else
                pressEd = 1.0;

            tbrush = drawBrush_line(sx, sy, nx, ny, pressSt, pressEd, tbrush);

            pressSt = pressEd;
        }

        //

        sx = nx, sy = ny;
    }
}

//! 四角形塗りつぶし

void CTileImage::drawFillBox(int x,int y,int w,int h,int opacity)
{
    RGBAFIX15 col;
    int ix,iy;
    void (CTileImage::*func)(int,int,const RGBAFIX15 &) = m_pinfo->funcDrawPixel;

    col = m_pinfo->colDraw;
    col.a = (0x8000 * opacity + 50) / 100;

    for(iy = 0; iy < h; iy++)
    {
        for(ix = 0; ix < w; ix++)
            (this->*func)(x + ix, y + iy, col);
    }
}

//! 多角形塗りつぶし

BOOL CTileImage::drawFillPolygon(CPolygonPaint *pPoly,int opacity,BOOL bAntiAlias)
{
    int miny,maxy,y,minx,xw,i,x1,srca;
    AXRect rc;
    LPBYTE pxbuf,p;
    RGBAFIX15 col;

    if(pPoly->getPosCnt() < 4) return FALSE;

    //Yの範囲

    getEnableDrawRectPixel(&rc);

    miny = pPoly->getMinY();
    maxy = pPoly->getMaxY();

    if(miny > rc.bottom || maxy < rc.top) return FALSE;

    if(miny < rc.top) miny = rc.top;
    if(maxy > rc.bottom) maxy = rc.bottom;

    //開始

    if(!pPoly->beginDraw(TRUE)) return FALSE;

    //

    pxbuf = pPoly->getXbuf();
    minx  = pPoly->getMinX();
    xw    = pPoly->getWidth();

    //

    srca = (0x8000 * opacity + 50) / 100;

    col = m_pinfo->colDraw;
    col.a = srca;

    //-------- Yの範囲でスキャン

    for(y = miny; y <= maxy; y++)
    {
        if(!pPoly->setXbuf_AA(y)) continue;

        //描画

        p  = pxbuf;
        x1 = minx;

        if(bAntiAlias)
        {
            for(i = xw; i; i--, p++, x1++)
            {
                col.a = (*p) * srca >> 6;

                if(col.a) setPixelDraw(x1, y, col);
            }
        }
        else
        {
            for(i = xw; i; i--, p++, x1++)
            {
                if(*p) setPixelDraw(x1, y, col);
            }
        }
    }

    return TRUE;
}

//! 多角形塗りつぶし（選択範囲用、非アンチエイリアス）

BOOL CTileImage::drawFillPolygonNoAA(CPolygonPaint *pPoly,BOOL bOn)
{
    int miny,maxy,y,x1,x2;
    AXRect rc;
    RGBAFIX15 col;

    if(pPoly->getPosCnt() < 4) return FALSE;

    //Yの範囲

    getEnableDrawRectPixel(&rc);

    miny = pPoly->getMinY();
    maxy = pPoly->getMaxY();

    if(miny > rc.bottom || maxy < rc.top) return FALSE;

    if(miny < rc.top) miny = rc.top;
    if(maxy > rc.bottom) maxy = rc.bottom;

    //開始

    if(!pPoly->beginDraw(FALSE)) return FALSE;

    col.zero();
    if(bOn) col.a = 0x8000;

    //描画

    for(y = miny; y <= maxy; y++)
    {
        if(!pPoly->findIntersectionInt(y)) continue;

        //描画

        while(pPoly->getNextLine(&x1, &x2))
        {
            for(; x1 <= x2; x1++)
                setPixel_subdraw(x1, y, col);
        }
    }

    return TRUE;
}

//! 円塗りつぶし描画

void CTileImage::drawFillCircle(double cx,double cy,double xr,double yr,
    const CANVASVIEWPARAM &param,BOOL bHRev,int opacity,BOOL bAntiAlias)
{
    int nx,ny,i,j,c,flag,srca;
    double xx,yy,rr,x1,y1,dx,dy,xx2,yy2,xt;
    DPOINT pt[4];
    AXRect rc;
    RGBAFIX15 col;

    srca = (0x8000 * opacity + 50) / 100;

    col = m_pinfo->colDraw;
    col.a = srca;

    //--------- 描画範囲計算 (10度単位)

    for(i = 0, rr = 0; i < 36; i++, rr += M_PI / 18)
    {
        xx = xr * ::cos(rr);
        yy = yr * ::sin(rr);

        x1 = xx * param.dCosRev + yy * param.dSinRev; if(bHRev) x1 = -x1;
        y1 = xx * param.dSinRev - yy * param.dCosRev;

        xx = x1 + cx;
        yy = y1 + cy;

        nx = (int)xx, ny = (int)yy;

        if(i == 0)
        {
            rc.left = rc.right = nx;
            rc.top = rc.bottom = ny;
        }
        else
            rc.incPoint(nx, ny);
    }

    //一応拡張

    rc.left -= 2, rc.top -= 2;
    rc.right += 2, rc.bottom += 2;

    //範囲内に調整

    if(!clipRectInEnableDraw(&rc)) return;

    //-------------

    if(xr < yr)
    {
        rr = xr * xr;
        dx = 1.0, dy = xr / yr;
    }
    else
    {
        rr = yr * yr;
        dx = yr / xr, dy = 1.0;
    }

    //--------- 描画

    yy = rc.top - cy;

    for(ny = rc.top; ny <= rc.bottom; ny++, yy += 1.0)
    {
        xx = rc.left - cx;

        for(nx = rc.left; nx <= rc.right; nx++, xx += 1.0)
        {
            if(bAntiAlias)
            {
                //-------- アンチエイリアス - 4x4

                //内外判定

                pt[0].x = pt[3].x = xx;
                pt[0].y = pt[1].y = yy;
                pt[1].x = pt[2].x = xx + 1;
                pt[2].y = pt[3].y = yy + 1;

                for(i = 0, flag = 0; i < 4; i++)
                {
                    xt = pt[i].x;
                    if(bHRev) xt = -xt;

                    x1 = xt * param.dCos - pt[i].y * param.dSin;
                    y1 = xt * param.dSin + pt[i].y * param.dCos;

                    x1 *= dx;
                    y1 *= dy;

                    if(x1 * x1 + y1 * y1 <= rr) flag |= (1 << i);
                }

                if(flag == 0) continue;

                if(flag == 15)
                {
                    col.a = srca;
                    setPixelDraw(nx, ny, col);
                    continue;
                }

                //4x4 平均

                c = 0;

                for(i = 0, yy2 = yy; i < 4; i++, yy2 += 0.25)
                {
                    for(j = 0, xx2 = xx; j < 4; j++, xx2 += 0.25)
                    {
                        xt = xx2;
                        if(bHRev) xt = -xt;

                        x1 = xt * param.dCos - yy2 * param.dSin;
                        y1 = xt * param.dSin + yy2 * param.dCos;

                        x1 *= dx;
                        y1 *= dy;

                        if(x1 * x1 + y1 * y1 < rr)
                            c++;
                    }
                }

                col.a = srca * c >> 4;
                if(col.a) setPixelDraw(nx, ny, col);
            }
            else
            {
                //--------- 非アンチ

                xt = xx;
                if(bHRev) xt = -xt;

                x1 = xt * param.dCos - yy * param.dSin;
                y1 = xt * param.dSin + yy * param.dCos;

                x1 *= dx;
                y1 *= dy;

                if(x1 * x1 + y1 * y1 < rr)
                    setPixelDraw(nx, ny, col);
            }
        }
    }
}

//===========================
// グラデーション
//===========================


//! グラデーション、点描画 (setPixelDraw)
/*
    [0] フラグ 0bit:反転 1bit:繰り返し 2bit:単色
    [1] 色の数
    [2] Aの数
    [3-] 色ポイント (WORD: 位置,R,G,B)
    [3 + 色の数 * 8] Aポイント (WORD: 位置,A)
*/

void CTileImage::drawGradient_pix(int x,int y,double pos,int opacity,LPBYTE pDat)
{
    int npos,i,pos1,pos2;
    double dtmp;
    LPWORD p,p2;
    RGBAFIX15 col;

    //--------- 位置(15bit固定小数点)

    if(pDat[0] & 2)
    {
        //繰り返し
        pos = ::modf(pos, &dtmp);
        if(pos < 0) pos = 1.0 + pos;
    }
    else
    {
        if(pos < 0) pos = 0;
        else if(pos > 1) pos = 1;
    }

    npos = (int)(pos * 0x8000 + 0.5);

    //反転

    if(pDat[0] & 1) npos = 0x8000 - npos;

    //------- 色

    p = (LPWORD)(pDat + 3);

    while(p[4] < npos) p += 4;

    if(pDat[0] & 4)
    {
        //単色

        col.r = p[1];
        col.g = p[2];
        col.b = p[3];
    }
    else
    {
        //グラデーション

        p2 = p + 4;

        pos1 = npos - *p;
        pos2 = *p2 - *p;

        p++, p2++;

        for(i = 0; i < 3; i++, p++, p2++)
            col.c[i] = (*p2 - *p) * pos1 / pos2 + *p;
    }

    //------- A

    p = (LPWORD)(pDat + 3 + (pDat[1] << 3));

    while(p[2] < npos) p += 2;

    p2 = p + 2;

    col.a = (p2[1] - p[1]) * (npos - *p) / (*p2 - *p) + p[1];
    col.a = col.a * opacity >> 8;

    //

    setPixelDraw(x, y, col);
}

//! グラデーション（線形）
/*!
    @param rc   描画する範囲(px)
    @param opacity 0-256
*/

void CTileImage::drawGradient_line(int x1,int y1,int x2,int y2,const AXRect &rc,int opacity,LPBYTE pDat)
{
    int x,y;
    double abx,aby,abab,dd,dd2,addx,addy;

    if(x1 == x2 && y1 == y2) return;

    abx  = x2 - x1;
    aby  = y2 - y1;
    abab = abx * abx + aby * aby;

    dd   = abx * (rc.left - x1) + aby * (rc.top - y1);
    dd   = dd / abab;

    addx = abx / abab;
    addy = aby / abab;

    /*--------
        abap = abx * (x - x1) + aby * (y - y1);
        pos  = abap / abab;

        x が +1 するごとに abx * 1 / abab,
        y が +1 するごとに aby * 1 / abab,
        +1 するごとの差分は一定なので、開始時点での値を求めておいて、加算していく
    --------*/

    for(y = rc.top; y <= rc.bottom; y++, dd += addy)
    {
        for(x = rc.left, dd2 = dd; x <= rc.right; x++, dd2 += addx)
            drawGradient_pix(x, y, dd2, opacity, pDat);
    }
}

//! グラデーション（円形）

void CTileImage::drawGradient_circle(int x1,int y1,int x2,int y2,const AXRect &rc,int opacity,LPBYTE pDat)
{
    int ix,iy;
    double div,x,y;

    if(x1 == x2 && y1 == y2) return;

    x = x1 - x2;
    y = y1 - y2;

    div = 1.0 / ::sqrt(x * x + y * y);

    //

    for(iy = rc.top; iy <= rc.bottom; iy++)
    {
        y = iy - y1;
        y = y * y;

        x = rc.left - x1;

        for(ix = rc.left; ix <= rc.right; ix++, x += 1.0)
            drawGradient_pix(ix, iy, ::sqrt(x * x + y) * div, opacity, pDat);
    }
}

//! グラデーション（矩形）

void CTileImage::drawGradient_box(int x1,int y1,int x2,int y2,const AXRect &rc,int opacity,LPBYTE pDat)
{
    int ix,iy,absy;
    double div,x,y,xx,yy,dx,dy;

    if(x1 == x2 && y1 == y2) return;

    x = ::abs(x2 - x1);
    y = ::abs(y2 - y1);

    div = 1.0 / ::sqrt(x * x + y * y);

    dx = (y1 == y2)? 0: x / y;
    dy = (x1 == x2)? 0: y / x;

    //

    for(iy = rc.top; iy <= rc.bottom; iy++)
    {
        absy = (y1 < iy)? iy - y1: y1 - iy;
        xx   = absy * dx;

        for(ix = rc.left; ix <= rc.right; ix++)
        {
            x = (x1 < ix)? ix - x1: x1 - ix;
            y = absy;

            if(y1 == y2)
                y = 0;
            else if(x < xx)
                x = xx;

            if(x1 == x2)
                x = 0;
            else
            {
                yy = x * dy;
                if(y < yy) y = yy;
            }

        /*--------
            x = abs(x1 - ix);
            y = abs(y1 - iy);

            if(y1 == y2)
                y = 0;
            else
            {
                xx = y * abs(x1 - x2) / abs(y1 - y2);
                if(xx > x) x = xx;
            }

            if(x1 == x2)
                x = 0;
            else
            {
                yy = x * abs(y1 - y2) / abs(x1 - x2);
                if(yy > y) y = yy;
            }
        --------*/

            drawGradient_pix(ix, iy, ::sqrt(x * x + y * y) * div, opacity, pDat);
        }
    }
}

//! グラデーション（放射線状）

void CTileImage::drawGradient_radial(int x1,int y1,int x2,int y2,const AXRect &rc,int opacity,LPBYTE pDat)
{
    int ix,iy;
    double x,y,start,pos,dtmp;

    if(x1 == x2 && y1 == y2) return;

    start = ::atan2(y2 - y1, x2 - x1);

    y = rc.top - y1;

    for(iy = rc.top; iy <= rc.bottom; iy++, y += 1.0)
    {
        x = rc.left - x1;

        for(ix = rc.left; ix <= rc.right; ix++, x += 1.0)
        {
            pos = (::atan2(y, x) - start) * 0.5 / M_PI;
            pos = ::modf(pos, &dtmp);

            if(pos < 0) pos += 1.0;

            drawGradient_pix(ix, iy, pos, opacity, pDat);
        }
    }
}
