/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CTileImage [brush] - ブラシでの描画
*/

#include <math.h>

#include "CTileImage.h"

#include "AXRand.h"

#include "CSinTable.h"
#include "CImage8.h"



//-----------------------

#define CURVE_CNT   10

DRAWPOINT g_posCurve[4];                //曲線補間用
double    g_dCurveParam[CURVE_CNT][4];  //曲線補間用、パラメータ

struct BRUSHDRAWWORK
{
    double  dT,
            dBkX, dBkY,     //前回の位置
            dBkPress;       //前回の筆圧
    int     nRotAngleRes;   //最終的な回転角度

    RGBADOUBLE colWater,    //前回の色
               colWaterFG;  //描画色
} g_brdrawwork;

//-----------------------

//! （関数）曲線初期化

void CTileImage::initCurve()
{
    int i;
    double t,tt,t1_,t2_,t3_;

    //曲線補間用パラメータセット

    tt = 1.0 / 6.0;

    for(i = 0, t = 1.0 / CURVE_CNT; i < CURVE_CNT; i++, t += 1.0 / CURVE_CNT)
    {
        t1_ = 4.0 - (t + 3.0);
        t2_ = t + 2.0;
        t3_ = t + 1.0;

        g_dCurveParam[i][0] = tt * t1_ * t1_ * t1_;
        g_dCurveParam[i][1] = tt * ( 3.0 * t2_ * t2_ * t2_ - 24.0 * t2_ * t2_ + 60.0 * t2_ - 44.0);
        g_dCurveParam[i][2] = tt * (-3.0 * t3_ * t3_ * t3_ + 12.0 * t3_ * t3_ - 12.0 * t3_ + 4.0);
        g_dCurveParam[i][3] = tt * t * t * t;
    }
}

//! フィルタ時などでのブラシ描画時、水彩を適当に初期化

void CTileImage::resetWaterCol()
{
    g_brdrawwork.colWaterFG.zero();
    g_brdrawwork.colWater.zero();
}


//==============================
// 初期化
//==============================


//! ブラシ自由線描画、開始処理

void CTileImage::startDrawBrushFree(double x,double y,double press)
{
    g_brdrawwork.dBkX = x;
    g_brdrawwork.dBkY = y;
    g_brdrawwork.dBkPress = press;
    g_brdrawwork.dT = 0;

    //曲線補間

    if(m_pbrush->dwFlags & BRUSHDF_CURVE)
    {
        for(int i = 0; i < 3; i++)
        {
            g_posCurve[i].x = x;
            g_posCurve[i].y = y;
            g_posCurve[i].press = press;
        }
    }

    //水彩

    startBrushWater(x, y);
}

//! 自由線以外のブラシ描画開始時

void CTileImage::startDrawBrushNoneFree(double x,double y)
{
    startBrushWater(x, y);
}

//! 水彩描画開始

void CTileImage::startBrushWater(double x,double y)
{
    if(m_pbrush->dwFlags & BRUSHDF_WATER)
    {
        g_brdrawwork.colWaterFG.r = m_pinfo->colDraw.r;
        g_brdrawwork.colWaterFG.g = m_pinfo->colDraw.g;
        g_brdrawwork.colWaterFG.b = m_pinfo->colDraw.b;
        g_brdrawwork.colWaterFG.a = 1;

        getPointAdvColor(&g_brdrawwork.colWater, x, y, m_pbrush->dRadius);
    }
}


//================================
// 線の描画
//================================


//! ブラシで自由線描画

void CTileImage::drawBrush_free(double x,double y,double press)
{
    //描画

    if(m_pbrush->dwFlags & BRUSHDF_CURVE)
    {
        //曲線

        g_posCurve[3].x = x;
        g_posCurve[3].y = y;
        g_posCurve[3].press = press;

        _drawBrush_lineCurve();

        for(int i = 0; i < 3; i++)
            g_posCurve[i] = g_posCurve[i + 1];
    }
    else
    {
        //直線

        g_brdrawwork.dT = drawBrush_line(g_brdrawwork.dBkX, g_brdrawwork.dBkY, x, y,
                            g_brdrawwork.dBkPress, press, g_brdrawwork.dT);

        g_brdrawwork.dBkX     = x;
        g_brdrawwork.dBkY     = y;
        g_brdrawwork.dBkPress = press;
    }
}

//! 曲線補間の線描画

void CTileImage::_drawBrush_lineCurve()
{
    int i;
    double bx,by,nx,ny,pressSt,pressEd;
    double t1,t2,t3,t4;
    DRAWPOINT *pPos = g_posCurve;

    bx = g_brdrawwork.dBkX;
    by = g_brdrawwork.dBkY;
    pressSt = g_brdrawwork.dBkPress;

    //

    for(i = 0; i < CURVE_CNT; i++)
    {
        t1 = g_dCurveParam[i][0];
        t2 = g_dCurveParam[i][1];
        t3 = g_dCurveParam[i][2];
        t4 = g_dCurveParam[i][3];

        //

        nx = pPos[0].x * t1 + pPos[1].x * t2 + pPos[2].x * t3 + pPos[3].x * t4;
        ny = pPos[0].y * t1 + pPos[1].y * t2 + pPos[2].y * t3 + pPos[3].y * t4;

        pressEd = pPos[0].press * t1 + pPos[1].press * t2 + pPos[2].press * t3 + pPos[3].press * t4;

        g_brdrawwork.dT = drawBrush_line(bx, by, nx, ny, pressSt, pressEd, g_brdrawwork.dT);

        //

        bx = nx;
        by = ny;
        pressSt = pressEd;
    }

    //

    g_brdrawwork.dBkX     = nx;
    g_brdrawwork.dBkY     = ny;
    g_brdrawwork.dBkPress = pressEd;
}


//=====================================
// 線を引く
//=====================================


//! ブラシで線を引く（線形補間）
/*!
    @return 次の t 値が返る
*/

double CTileImage::drawBrush_line(double x1,double y1,double x2,double y2,
        double pressSt,double pressEd,double tstart)
{
    double len,t,dx,dy,ttm,dtmp;
    double press_len,radius_len,opa_len;
    double press,press_r,press_o,radius,opacity,x,y;
    int nRotBasic,ntmp;

    //線の長さ

    dx = x2 - x1;
    dy = y2 - y1;

    len = ::sqrt(dx * dx + dy * dy);
    if(len == 0) return tstart;

    //間隔をt値に合わせる

    ttm = m_pbrush->dInterval / len;

    //各幅

    radius_len = 1.0 - m_pbrush->dMinSize;
    opa_len    = 1.0 - m_pbrush->dMinOpacity;
    press_len  = pressEd - pressSt;

    //ブラシ画像回転角度

    nRotBasic = m_pbrush->nRotAngle;

    if(m_pbrush->dwFlags & BRUSHDF_ROT_DIR)
        nRotBasic += (int)(::atan2(dy, dx) * 256 / M_PI);   //進行方向

    //------------------

    t = tstart / len;

    while(t < 1.0)
    {
        //--------- 筆圧 (0.0-1.0)

        //t位置の筆圧

        press = press_len * t + pressSt;

        //ガンマ補正（硬さ調整）

        press_r = (m_pbrush->dwFlags & BRUSHDF_GAMMA_SIZE)? ::pow(press, m_pbrush->dGammaSize): press;
        press_o = (m_pbrush->dwFlags & BRUSHDF_GAMMA_OPACITY)? ::pow(press, m_pbrush->dGammaOpacity): press;

        //最小補正

        press_r = press_r * radius_len + m_pbrush->dMinSize;
        press_o = press_o * opa_len    + m_pbrush->dMinOpacity;

        //--------- 半径・濃度

        if(m_pbrush->dwFlags & BRUSHDF_RND_SIZE)
            press_r = press_r * (m_prand->getDouble() * (1.0 - m_pbrush->dRandSizeMin) + m_pbrush->dRandSizeMin);

        radius  = press_r * m_pbrush->dRadius;
        opacity = press_o * m_pbrush->dOpacity;

        //---------- 位置

        x = x1 + dx * t;
        y = y1 + dy * t;

        //ランダム（円範囲）

        if(m_pbrush->dwFlags & BRUSHDF_RND_POS)
        {
            dtmp = (radius * m_pbrush->dRandPosLen) * m_prand->getDouble();
            ntmp = m_prand->getDWORD() & 511;

            x += dtmp * SINTABLE->getCos(ntmp);
            y += dtmp * SINTABLE->getSin(ntmp);
        }

        //----------- ブラシ画像回転角度

        g_brdrawwork.nRotAngleRes = nRotBasic;

        if(m_pbrush->nRotRandom)
            g_brdrawwork.nRotAngleRes += m_prand->getRange(-(m_pbrush->nRotRandom), m_pbrush->nRotRandom);

        //------------ 点描画

        _drawBrush_point(x, y, radius, opacity);

        //------------ 次へ

        dtmp = radius * ttm;
        if(dtmp < 0.0001) dtmp = 0.0001;

        t += dtmp;
    }

    //次回の開始t値（長さ[px]に変換）

    return len * (t - 1.0);
}

//! 入り抜き指定あり直線

void CTileImage::drawBrush_lineHeadTail(double x1,double y1,double x2,double y2,WORD wHeadTail)
{
    int in,out;
    double t,dx,dy,d,xx,yy,xx2,yy2;

    in  = wHeadTail >> 8;
    out = wHeadTail & 0xff;

    if(out > 100 - in) out = 100 - in;

    if(in == 0 && out == 0)
        drawBrush_line(x1, y1, x2, y2, 1, 1, 0);
    else
    {
        dx = x2 - x1;
        dy = y2 - y1;

        xx = x1, yy = y1;
        t  = 0;

        //入り

        if(in)
        {
            d  = in * 0.01;
            xx = x1 + dx * d;
            yy = y1 + dy * d;

            t = drawBrush_line(x1, y1, xx, yy, 0, 1, t);
        }

        //中間部分 (press = 1)

        if(out != 100 - in)
        {
            d   = (100 - out) * 0.01;
            xx2 = x1 + dx * d;
            yy2 = y1 + dy * d;

            t = drawBrush_line(xx, yy, xx2, yy2, 1, 1, t);

            xx = xx2, yy = yy2;
        }

        //抜き

        if(out)
            drawBrush_line(xx, yy, x2, y2, 1, 0, t);
    }
}


//=================================
// 点の描画
//=================================


//! 点の描画（フィルタから）
/*
    色は m_pinfo->colDraw にセットしておく。
*/

void CTileImage::drawBrush_point(int x,int y,int radius,int opacity)
{
    //ブラシ画像回転角度

    g_brdrawwork.nRotAngleRes = m_pbrush->nRotAngle;

    if(m_pbrush->nRotRandom)
        g_brdrawwork.nRotAngleRes += m_prand->getRange(-(m_pbrush->nRotRandom), m_pbrush->nRotRandom);

    //

    _drawBrush_point(x, y, radius, (double)opacity / 0x8000);
}

//! 水彩・混色
/*
    やまかわ氏の "gimp-painter-" (MixBrush 改造版) のソースを参考にしています。
    http://sourceforge.jp/projects/gimp-painter/releases/?package_id=6799
*/

void _compositeWaterCol(RGBADOUBLE *pdst,const RGBADOUBLE &col1,double op1,
    const RGBADOUBLE &col2,double op2,BOOL bAlpha)
{
    double a1,a2;

    a1 = col1.a * op1;
    a2 = col2.a * (1.0 - a1) * op2;

    if(a1 == 0)
    {
        *pdst = col2;
        pdst->a = a2;
    }
    else if(a2 == 0)
    {
        *pdst = col1;
        pdst->a = a1;
    }
    else
    {
        pdst->a = a1 + a2;

        a1 = a1 / (a1 + a2);
        a2 = 1.0 - a1;

        pdst->r = col1.r * a1 + col2.r * a2;
        pdst->g = col1.g * a1 + col2.g * a2;
        pdst->b = col1.b * a1 + col2.b * a2;
    }

    if(bAlpha)
    {
        op1 = ::pow(op1, 1.35);
        op2 = ::pow(op2, 1.15);
        a1  = op1 + op2;

        if(a1 == 0)
            pdst->a = 0;
        else
        {
            a2 = op1 / a1;
            pdst->a = (col1.a * a2 + col2.a * (1.0 - a2)) * (op1 + (1.0 - op1) * op2);
        }
    }
}

//! ブラシ・点の描画（各処理への分岐）

void CTileImage::_drawBrush_point(double x,double y,double radius,double opacity)
{
    RGBAFIX15 col;

    if(opacity != 0 && radius > 0.05)
    {
        col = m_pinfo->colDraw;

        //水彩・色の計算

        if(m_pbrush->dwFlags & BRUSHDF_WATER)
        {
            RGBADOUBLE colm,colc;

            //描画色+前回の色

            _compositeWaterCol(&colm, g_brdrawwork.colWaterFG, m_pbrush->dWater[0], g_brdrawwork.colWater, 1, FALSE);

            //+キャンバス色

            getPointAdvColor(&colc, x, y, radius);

            _compositeWaterCol(&g_brdrawwork.colWater, colm, m_pbrush->dWater[2], colc, m_pbrush->dWater[1], TRUE);

            //結果色

            col.r = (WORD)(g_brdrawwork.colWater.r + 0.5);
            col.g = (WORD)(g_brdrawwork.colWater.g + 0.5);
            col.b = (WORD)(g_brdrawwork.colWater.b + 0.5);

            if(col.r > 0x8000) col.r = 0x8000;
            if(col.g > 0x8000) col.g = 0x8000;
            if(col.b > 0x8000) col.b = 0x8000;

            if(opacity > g_brdrawwork.colWater.a)
            {
                opacity = g_brdrawwork.colWater.a;
                if(opacity == 0) return;
            }
        }

        //点の描画

        if(!m_pinfo->pBrush)
            _drawBrush_point_circle(x, y, radius, opacity, col);
        else
        {
            //ブラシ画像

            if(g_brdrawwork.nRotAngleRes)
                _drawBrush_point_img_rot(x, y, radius, opacity, col);
            else
                _drawBrush_point_img(x, y, radius, opacity, col);
        }
    }
}

//! ブラシ・通常円での点描画

void CTileImage::_drawBrush_point_circle(double xpos,double ypos,double radius,double opacity,const RGBAFIX15 &colDraw)
{
    int subnum,x1,y1,x2,y2,ix,iy,xx,yy;
    double rrdiv,dsubadd,dsumdiv,dtmp,dsum,dd,ytbl[11],xtbl[11];
    RGBAFIX15 col;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15 &) = m_pinfo->funcDrawPixel;

    //大きさによりサブピクセル数調整

    if(radius < 3) subnum = 11;
    else if(radius < 15) subnum = 5;
    else if(radius < 40) subnum = 3;
    else subnum = 1;

    //

    x1 = (int)floor(xpos - radius);
    y1 = (int)floor(ypos - radius);
    x2 = (int)floor(xpos + radius);
    y2 = (int)floor(ypos + radius);

    rrdiv   = 1.0 / (radius * radius);
    dsubadd = 1.0 / subnum;
    dsumdiv = 1.0 / (subnum * subnum);

    col = colDraw;

    //------------------

    for(yy = y1; yy <= y2; yy++)
    {
        //Yテーブル

        dtmp = yy - ypos;

        for(iy = 0; iy < subnum; iy++, dtmp += dsubadd)
            ytbl[iy] = dtmp * dtmp;

        //

        for(xx = x1; xx <= x2; xx++)
        {
            //粗さ

            if(m_pbrush->nRoughVal >= 2 && (m_prand->getDWORD() % m_pbrush->nRoughVal))
                continue;

            //Xテーブル

            dtmp = xx - xpos;

            for(ix = 0; ix < subnum; ix++, dtmp += dsubadd)
                xtbl[ix] = dtmp * dtmp;

            //オーバーサンプリング

            dsum = 0;

            for(iy = 0; iy < subnum; iy++)
            {
                for(ix = 0; ix < subnum; ix++)
                {
                    dd = (xtbl[ix] + ytbl[iy]) * rrdiv;

                    if(dd <= 1)
                    {
                        dd = dd + m_pbrush->dShapeHard - dd * m_pbrush->dShapeHard;

                        if(dd < 0) dd = 0; else if(dd > 1) dd = 1;

                        dsum += 1.0 - dd;
                    }
                }
            }

            //色セット

            col.a = (WORD)(opacity * (dsum * dsumdiv) * (1 << 15) + 0.5);

            if(col.a)
            {
                if(m_pbrush->dwFlags & BRUSHDF_NONEAA)
                    col.a = (WORD)(opacity * (1<<15) + 0.5);

                (this->*funcPix)(xx, yy, col);
            }
        }
    }
}

//! ブラシ・ブラシ画像による点描画（回転なし）

void CTileImage::_drawBrush_point_img(double xpos,double ypos,double radius,double opacity,const RGBAFIX15 &colDraw)
{
    int subnum,brimgsize,x1,y1,x2,y2,xx,yy,ix,iy,xtbl[11],c;
    double dsubadd,dadd,dby,dfbx,dbx,dsumdiv,dtmp;
    LPBYTE pBrushBuf,pBrushY,ytbl[11];
    RGBAFIX15 col;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15 &) = m_pinfo->funcDrawPixel;

    //サブピクセル数

    if(radius < 4) subnum = 11;
    else if(radius < 25) subnum = 9;
    else subnum = 3;

    //

    x1 = (int)floor(xpos - radius);
    y1 = (int)floor(ypos - radius);
    x2 = (int)floor(xpos + radius);
    y2 = (int)floor(ypos + radius);

    pBrushBuf = (m_pinfo->pBrush)->getBuf();
    brimgsize = (m_pinfo->pBrush)->getWidth();

    dadd    = (double)brimgsize / (radius * 2);
    dsubadd = dadd / subnum;
    dsumdiv = 1.0 / ((subnum * subnum) * 255.0);

    dtmp  = brimgsize * 0.5;
    dby   = (y1 - ypos) * dadd + dtmp;
    dfbx  = (x1 - xpos) * dadd + dtmp;

    col = colDraw;

    //--------------

    for(yy = y1; yy <= y2; yy++, dby += dadd)
    {
        //Yテーブル

        for(iy = 0, dtmp = dby; iy < subnum; iy++, dtmp += dsubadd)
        {
            c = (int)dtmp;

            if(c >= 0 && c < brimgsize)
                ytbl[iy] = pBrushBuf + c * brimgsize;
            else
                ytbl[iy] = NULL;
        }

        //

        for(xx = x1, dbx = dfbx; xx <= x2; xx++, dbx += dadd)
        {
            //粗さ

            if(m_pbrush->nRoughVal >= 2 && (m_prand->getDWORD() % m_pbrush->nRoughVal))
                continue;

            //Xテーブル

            for(ix = 0, dtmp = dbx; ix < subnum; ix++, dtmp += dsubadd)
            {
                c = (int)dtmp;

                if(c >= 0 && c < brimgsize)
                    xtbl[ix] = c;
                else
                    xtbl[ix] = -1;
            }

            //オーバーサンプリング

            c = 0;

            for(iy = 0; iy < subnum; iy++)
            {
                pBrushY = ytbl[iy];
                if(!pBrushY) continue;

                for(ix = 0; ix < subnum; ix++)
                {
                    if(xtbl[ix] != -1)
                        c += *(pBrushY + xtbl[ix]);
                }
            }

            //点セット

            col.a = (WORD)(opacity * (c * dsumdiv) * (1 << 15) + 0.5);

            if(col.a)
            {
                if(m_pbrush->dwFlags & BRUSHDF_NONEAA)
                    col.a = (WORD)(opacity * (1<<15) + 0.5);

                (this->*funcPix)(xx, yy, col);
            }
        }
    }
}

//! ブラシ・ブラシ画像による点描画（回転あり）

void CTileImage::_drawBrush_point_img_rot(double xpos,double ypos,double radius,double opacity,const RGBAFIX15 &colDraw)
{
    int subnum,x1,y1,x2,y2,xx,yy,brimgsize,c,ix,iy,nx,ny;
    double dtmp,dscale,dsumdiv,dcos,dsin,dstx,dsty,daddcos,daddsin,dsubaddcos,dsubaddsin;
    double dx,dy,dx2,dy2,dx3,dy3;
    RGBAFIX15 col;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15 &) = m_pinfo->funcDrawPixel;

    //サブピクセル数

    if(radius < 4) subnum = 11;
    else if(radius < 25) subnum = 7;
    else subnum = 3;

    //回転後の範囲を考慮し大きめに描画

    dtmp = radius * 1.44;

    x1 = (int)floor(xpos - dtmp);
    y1 = (int)floor(ypos - dtmp);
    x2 = (int)floor(xpos + dtmp);
    y2 = (int)floor(ypos + dtmp);

    brimgsize = (m_pinfo->pBrush)->getWidth();

    dscale  = (double)brimgsize / (radius * 2);
    dsumdiv = 1.0 / ((subnum * subnum) * 255.0);

    //

    dcos = SINTABLE->getCos(-(g_brdrawwork.nRotAngleRes));
    dsin = SINTABLE->getSin(-(g_brdrawwork.nRotAngleRes));

    dx = x1 - xpos;
    dy = y1 - ypos;
    dtmp = brimgsize * 0.5;

    dstx = (dx * dcos - dy * dsin) * dscale + dtmp;
    dsty = (dx * dsin + dy * dcos) * dscale + dtmp;

    //

    dtmp = 1.0 / subnum;

    daddcos    = dcos * dscale;
    daddsin    = dsin * dscale;
    dsubaddcos = daddcos * dtmp;
    dsubaddsin = daddsin * dtmp;

    col = colDraw;

    //----------------
    //xx,yy:cos, xy:sin, yx:-sin

    for(yy = y1; yy <= y2; yy++)
    {
        dx = dstx;
        dy = dsty;

        for(xx = x1; xx <= x2; xx++, dx += daddcos, dy += daddsin)
        {
            //粗さ

            if(m_pbrush->nRoughVal >= 2 && (m_prand->getDWORD() % m_pbrush->nRoughVal))
                continue;

            //

            dx2 = dx;
            dy2 = dy;

            //オーバーサンプリング

            c = 0;

            for(iy = subnum; iy; iy--)
            {
                dx3 = dx2;
                dy3 = dy2;

                for(ix = subnum; ix; ix--)
                {
                    nx = (int)dx3;
                    ny = (int)dy3;

                    if(nx >= 0 && nx < brimgsize && ny >= 0 && ny < brimgsize)
                        c += (m_pinfo->pBrush)->getPixel(nx, ny);

                    dx3 += dsubaddsin;
                    dy3 += dsubaddcos;
                }

                dx2 -= dsubaddsin;
                dy2 += dsubaddcos;
            }

            //点セット

            col.a = (WORD)(opacity * (c * dsumdiv) * (1 << 15) + 0.5);

            if(col.a)
            {
                if(m_pbrush->dwFlags & BRUSHDF_NONEAA)
                    col.a = (WORD)(opacity * (1<<15) + 0.5);

                (this->*funcPix)(xx, yy, col);
            }
        }

        dstx -= daddsin;
        dsty += daddcos;
    }
}
