/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CTileImageGray [色(16bit)+A(16bit) のグレイスケールイメージ]
*/

#include "CTileImageGray.h"

#include "CImageRGB16.h"
#include "struct.h"


//-------------------

#define _GETPIXPOS(x,y)    ((((y) - m_nOffY) & 63) << 6) + (((x) - m_nOffX) & 63)

//-------------------


CTileImageGray::CTileImageGray()
{
    m_nColType  = COLTYPE_GRAY;
    m_nTileSize = sizeof(PIXGRAY) * 64 * 64;
}

//! タイルとpx位置から色バッファ位置取得

void *CTileImageGray::_getPixelBuf_tile(void *pTile,int x,int y) const
{
    return (PIXGRAY *)pTile + _GETPIXPOS(x, y);
}

//! タイルとpx位置から色取得

void CTileImageGray::_getPixelCol_tile(RGBAFIX15 *pdst,void *pTile,int x,int y) const
{
    PIXGRAY *p = (PIXGRAY *)pTile + _GETPIXPOS(x, y);

    pdst->r = pdst->g = pdst->b = p->c;
    pdst->a = p->a;
}

//! 色バッファ位置に色をセット

void CTileImageGray::_setPixel_buf(void *pbuf,int x,int y,const RGBAFIX15 &col)
{
    ((PIXGRAY *)pbuf)->c = (col.r + col.g + col.b) / 3;
    ((PIXGRAY *)pbuf)->a = col.a;
}

//! タイルがすべて透明な状態か

BOOL CTileImageGray::_isTransparentTile(void *pTile)
{
    PIXGRAY *p = (PIXGRAY *)pTile;
    int i;

    for(i = 64 * 64; i; i--, p++)
    {
        if(p->a) return FALSE;
    }

    return TRUE;
}

//! CImageRGB16 に合成（タイル単位）
/*!
    @param opacity 0-128
*/

void CTileImageGray::_blendTile_RGB16(CImageRGB16 *pimgDst,void *pTile,int dx,int dy,int opacity,
    const AXRect &rcClip,void (*funcCol)(RGBAFIX15 *,const RGBFIX15 &))
{
    int w,h,sx,sy,pitchs,pitchd,ix,iy;
    PIXGRAY *pSrc;
    RGBFIX15 *pDst;
    RGBAFIX15 src;

    sx = sy = 0;
    w = h = 64;

    //クリッピング

    if(dx < rcClip.left) w += dx - rcClip.left, sx += rcClip.left - dx, dx = rcClip.left;
    if(dy < rcClip.top)  h += dy - rcClip.top , sy += rcClip.top  - dy, dy = rcClip.top;
    if(dx + w > rcClip.right)  w = rcClip.right  - dx;
    if(dy + h > rcClip.bottom) h = rcClip.bottom - dy;

    if(w <= 0 || h <= 0) return;

    //合成

    pSrc   = (PIXGRAY *)pTile + (sy << 6) + sx;
    pDst   = pimgDst->getBufPt(dx, dy);
    pitchs = 64 - w;
    pitchd = pimgDst->getWidth() - w;

    for(iy = h; iy > 0; iy--, pSrc += pitchs, pDst += pitchd)
    {
        for(ix = w; ix > 0; ix--, pSrc++, pDst++)
        {
            src.a = pSrc->a * opacity >> 7;

            if(src.a)
            {
                src.r = src.g = src.b = pSrc->c;

                (*funcCol)(&src, *pDst);

                pDst->r = ((src.r - pDst->r) * src.a >> 15) + pDst->r;
                pDst->g = ((src.g - pDst->g) * src.a >> 15) + pDst->g;
                pDst->b = ((src.b - pDst->b) * src.a >> 15) + pDst->b;
            }
        }
    }
}

//! タイルイメージをRGBAFIX15に変換

void CTileImageGray::_getBufRGBA_tile(RGBAFIX15 *pDst,const void *pTile) const
{
    const PIXGRAY *ps = (const PIXGRAY *)pTile;
    int i;

    for(i = 64 * 64; i; i--, ps++, pDst++)
    {
        pDst->r = pDst->g = pDst->b = ps->c;
        pDst->a = ps->a;
    }
}

//! RGBAFIX15バッファからタイルイメージセット

void CTileImageGray::_setFromBufRGBA_tile(void *pTile,const RGBAFIX15 *pSrc,BOOL bLum)
{
    PIXGRAY *pd = (PIXGRAY *)pTile;
    int i;

    for(i = 64 * 64; i; i--, pSrc++, pd++)
    {
        pd->c = (pSrc->r * 77 + pSrc->g * 150 + pSrc->b * 29) >> 8;
        pd->a = pSrc->a;
    }
}

//! ファイル出力用にタイルのデータ取得

void CTileImageGray::getSaveTileBuf(LPBYTE pDst,void *pTile)
{
    PIXGRAY *pSrc;
    int i;

    //COL

    pSrc = (PIXGRAY *)pTile;

    for(i = 64 * 64; i; i--, pDst += 2, pSrc++)
    {
        pDst[0] = (BYTE)(pSrc->c >> 8);
        pDst[1] = (BYTE)pSrc->c;
    }

    //A

    pSrc = (PIXGRAY *)pTile;

    for(i = 64 * 64; i; i--, pDst += 2, pSrc++)
    {
        pDst[0] = (BYTE)(pSrc->a >> 8);
        pDst[1] = (BYTE)pSrc->a;
    }
}

//! ファイル読み込み時のタイルデータ変換

void CTileImageGray::setSaveTileBuf(void *pTile,LPBYTE pSrc)
{
    PIXGRAY *pDst;
    int i;

    //COL

    pDst = (PIXGRAY *)pTile;

    for(i = 64 * 64; i; i--, pDst++, pSrc += 2)
        pDst->c = ((WORD)pSrc[0] << 8) | pSrc[1];

    //A

    pDst = (PIXGRAY *)pTile;

    for(i = 64 * 64; i; i--, pDst++, pSrc += 2)
        pDst->a = ((WORD)pSrc[0] << 8) | pSrc[1];
}

//! タイルを左右反転

void CTileImageGray::_reverseHorz_tile(void *pTile)
{
    PIXGRAY *p1,*p2,c;
    int ix,iy;

    p1 = (PIXGRAY *)pTile;
    p2 = p1 + 63;

    for(iy = 64; iy; iy--)
    {
        for(ix = 32; ix; ix--, p1++, p2--)
        {
            c   = *p1;
            *p1 = *p2;
            *p2 = c;
        }

        p1 += 64 - 32;
        p2 += 64 + 32;
    }
}

//! タイルを上下反転

void CTileImageGray::_reverseVert_tile(void *pTile)
{
    PIXGRAY *p1,*p2,c;
    int ix,iy;

    p1 = (PIXGRAY *)pTile;
    p2 = p1 + 63 * 64;

    for(iy = 32; iy; iy--, p2 -= 128)
    {
        for(ix = 64; ix; ix--, p1++, p2++)
        {
            c   = *p1;
            *p1 = *p2;
            *p2 = c;
        }
    }
}

//! タイルを左に90度回転

void CTileImageGray::_rotateLeft_tile(void *pTile)
{
    PIXGRAY *p1,*p2,*p3,*p4,c1,c2;
    int ix,iy;

    p1 = (PIXGRAY *)pTile;
    p2 = p1 + 63 * 64;
    p3 = p1 + 63;
    p4 = p2 + 63;

    for(iy = 32; iy; iy--)
    {
        for(ix = 32; ix; ix--, p1++, p2 -= 64, p3 += 64, p4--)
        {
            c1  = *p1;
            c2  = *p2;
            *p1 = *p3;
            *p2 = c1;
            *p3 = *p4;
            *p4 = c2;
        }

        p1 += 32;
        p2 += 32 * 64 + 1;
        p3 -= 32 * 64 + 1;
        p4 -= 32;
    }
}

//! タイルを右に90度回転

void CTileImageGray::_rotateRight_tile(void *pTile)
{
    PIXGRAY *p1,*p2,*p3,*p4,c1,c3;
    int ix,iy;

    p1 = (PIXGRAY *)pTile;
    p2 = p1 + 63 * 64;
    p3 = p1 + 63;
    p4 = p2 + 63;

    for(iy = 32; iy; iy--)
    {
        for(ix = 32; ix; ix--, p1++, p2 -= 64, p3 += 64, p4--)
        {
            c1 = *p1;
            c3 = *p3;

            *p1 = *p2;
            *p2 = *p4;
            *p3 = c1;
            *p4 = c3;
        }

        p1 += 32;
        p2 += 32 * 64 + 1;
        p3 -= 32 * 64 + 1;
        p4 -= 32;
    }
}
