/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CTileImageA1 [Alpha(1bit)で色はレイヤ色を使うイメージ]

    - 選択範囲イメージなどでも使う。
*/


#include <string.h>

#include "CTileImageA1.h"

#include "CImageRGB16.h"
#include "CXImage.h"



CTileImageA1::CTileImageA1()
{
    m_nColType  = COLTYPE_A_1BIT;
    m_nTileSize = 8 * 64;
}

//! タイルとpx位置から色バッファ位置取得

void *CTileImageA1::_getPixelBuf_tile(void *pTile,int x,int y) const
{
    x = (x - m_nOffX) & 63;
    y = (y - m_nOffY) & 63;

    return (LPBYTE)pTile + (y << 3) + (x >> 3);
}

//! タイルとpx位置から色取得

void CTileImageA1::_getPixelCol_tile(RGBAFIX15 *pdst,void *pTile,int x,int y) const
{
    LPBYTE p;

    x = (x - m_nOffX) & 63;
    y = (y - m_nOffY) & 63;

    p = (LPBYTE)pTile + (y << 3) + (x >> 3);

    pdst->r = m_col.r;
    pdst->g = m_col.g;
    pdst->b = m_col.b;
    pdst->a = ( *p & (1 << (7 - (x & 7))) )? 0x8000: 0;
}

//! 色バッファ位置に色をセット

void CTileImageA1::_setPixel_buf(void *pbuf,int x,int y,const RGBAFIX15 &col)
{
    x = (x - m_nOffX) & 63;
    x = 1 << (7 - (x & 7));

    if(col.a >= 0x4000)
        *((LPBYTE)pbuf) |= x;
    else
        *((LPBYTE)pbuf) &= ~x;
}

//! タイルがすべて透明な状態か

BOOL CTileImageA1::_isTransparentTile(void *pTile)
{
    LPDWORD p = (LPDWORD)pTile;
    int i;

    //4Byte単位で比較

    for(i = m_nTileSize >> 2; i; i--)
    {
        if(*(p++)) return FALSE;
    }

    return TRUE;
}

//! CImageRGB16 に合成（タイル単位）
/*!
    @param opacity 0-128
*/

void CTileImageA1::_blendTile_RGB16(CImageRGB16 *pimgDst,void *pTile,int dx,int dy,int opacity,
    const AXRect &rcClip,void (*funcCol)(RGBAFIX15 *,const RGBFIX15 &))
{
    int w,h,sx,sy,pitchd,ix,iy;
    BYTE stf,f;
    LPBYTE pSrc,pSrcBk;
    RGBFIX15 *pDst;
    RGBAFIX15 src,srcCol;

    sx = sy = 0;
    w = h = 64;

    //クリッピング

    if(dx < rcClip.left) w += dx - rcClip.left, sx += rcClip.left - dx, dx = rcClip.left;
    if(dy < rcClip.top)  h += dy - rcClip.top , sy += rcClip.top  - dy, dy = rcClip.top;
    if(dx + w > rcClip.right)  w = rcClip.right  - dx;
    if(dy + h > rcClip.bottom) h = rcClip.bottom - dy;

    if(w <= 0 || h <= 0) return;

    //

    pSrc   = (LPBYTE)pTile + (sy << 3) + (sx >> 3);
    pDst   = pimgDst->getBufPt(dx, dy);
    pitchd = pimgDst->getWidth() - w;

    stf = 1 << (7 - (sx & 7));

    srcCol.r = m_col.r;
    srcCol.g = m_col.g;
    srcCol.b = m_col.b;
    srcCol.a = 0x8000 * opacity >> 7;

    //

    for(iy = h; iy > 0; iy--)
    {
        pSrcBk = pSrc;
        f      = stf;

        for(ix = w; ix > 0; ix--, pDst++)
        {
            if(*pSrc & f)
            {
                src = srcCol;

                (*funcCol)(&src, *pDst);

                pDst->r = ((src.r - pDst->r) * src.a >> 15) + pDst->r;
                pDst->g = ((src.g - pDst->g) * src.a >> 15) + pDst->g;
                pDst->b = ((src.b - pDst->b) * src.a >> 15) + pDst->b;
            }

            f >>= 1;
            if(!f)
            {
                f = 0x80;
                pSrc++;
            }
        }

        pSrc = pSrcBk + 8;
        pDst += pitchd;
    }
}

//! タイルイメージをRGBAFIX15に変換

void CTileImageA1::_getBufRGBA_tile(RGBAFIX15 *pDst,const void *pTile) const
{
    const BYTE *ps = (const BYTE *)pTile;
    int i;
    BYTE f = 0x80;

    for(i = 64 * 64; i; i--, pDst++)
    {
        pDst->r = m_col.r;
        pDst->g = m_col.g;
        pDst->b = m_col.b;
        pDst->a = (*ps & f)? 0x8000: 0;

        f >>= 1;
        if(f == 0) f = 0x80, ps++;
    }
}

//! RGBAFIX15バッファからタイルイメージセット

void CTileImageA1::_setFromBufRGBA_tile(void *pTile,const RGBAFIX15 *pSrc,BOOL bLum)
{
    LPBYTE pd = (LPBYTE)pTile;
    int i,j,a;
    BYTE f,val;

    for(i = 64 * 8; i; i--)
    {
        for(j = 8, f = 0x80, val = 0; j; j--, f >>= 1, pSrc++)
        {
            if(bLum)
                a = 0x8000 - ((pSrc->r * 77 + pSrc->g * 150 + pSrc->b * 29) >> 8);
            else
                a = pSrc->a;

            if(a >= 0x4000) val |= f;
        }

        *(pd++) = val;
    }
}

//! ファイル出力用にタイルのデータ取得

void CTileImageA1::getSaveTileBuf(LPBYTE pDst,void *pTile)
{
    ::memcpy(pDst, pTile, m_nTileSize);
}

//! ファイル読み込み時のタイルデータ変換

void CTileImageA1::setSaveTileBuf(void *pTile,LPBYTE pSrc)
{
    ::memcpy(pTile, pSrc, m_nTileSize);
}

//! タイルを左右反転

void CTileImageA1::_reverseHorz_tile(void *pTile)
{
    BYTE *p1,*p2,s1,s2,d1,d2,f1,f2;
    int ix,iy,i;

    p1 = (LPBYTE)pTile;
    p2 = p1 + 7;

    for(iy = 64; iy; iy--, p1 += 4, p2 += 12)
    {
        for(ix = 4; ix; ix--)
        {
            s1 = *p1, s2 = *p2;
            d1 = d2 = 0;
            f1 = 0x80, f2 = 1;

            for(i = 8; i; i--, f1 >>= 1, f2 <<= 1)
            {
                if(s1 & f1) d2 |= f2;
                if(s2 & f1) d1 |= f2;
            }

            *(p1++) = d1;
            *(p2--) = d2;
        }
    }
}

//! タイルを上下反転

void CTileImageA1::_reverseVert_tile(void *pTile)
{
    BYTE *p1,*p2,c;
    int ix,iy;

    p1 = (LPBYTE)pTile;
    p2 = p1 + 63 * 8;

    for(iy = 32; iy; iy--, p2 -= 16)
    {
        for(ix = 8; ix; ix--, p1++, p2++)
        {
            c   = *p1;
            *p1 = *p2;
            *p2 = c;
        }
    }
}

//! タイルを左に90度回転

void CTileImageA1::_rotateLeft_tile(void *pTile)
{
    LPBYTE p1,p2,p3,p4;
    int f1,f2,f3,f4,ix,iy;
    BYTE c1,c2,c3,c4;

    p1 = (LPBYTE)pTile;
    p2 = p1 + 63 * 8;
    p3 = p1 + 7;
    p4 = p2 + 7;

    f1 = f2 = 7;
    f3 = f4 = 0;

    for(iy = 32; iy; iy--)
    {
        for(ix = 32; ix; ix--)
        {
            c1 = (*p1 >> f1) & 1;
            c2 = (*p2 >> f2) & 1;
            c3 = (*p3 >> f3) & 1;
            c4 = (*p4 >> f4) & 1;

            if(c3) *p1 |= 1 << f1; else *p1 &= ~(1 << f1);
            if(c1) *p2 |= 1 << f2; else *p2 &= ~(1 << f2);
            if(c4) *p3 |= 1 << f3; else *p3 &= ~(1 << f3);
            if(c2) *p4 |= 1 << f4; else *p4 &= ~(1 << f4);

            f1--; if(f1 < 0) f1 = 7, p1++;
            p2 -= 8;
            p3 += 8;
            f4++; if(f4 > 7) f4 = 0, p4--;
        }

        p1 += 4;
        p2 += 32 * 8, f2--; if(f2 < 0) f2 = 7, p2++;
        p3 -= 32 * 8, f3++; if(f3 > 7) f3 = 0, p3--;
        p4 -= 4;
    }
}

//! タイルを右に90度回転

void CTileImageA1::_rotateRight_tile(void *pTile)
{
    LPBYTE p1,p2,p3,p4;
    int f1,f2,f3,f4,ix,iy;
    BYTE c1,c2,c3,c4;

    p1 = (LPBYTE)pTile;
    p2 = p1 + 63 * 8;
    p3 = p1 + 7;
    p4 = p2 + 7;

    f1 = f2 = 7;
    f3 = f4 = 0;

    for(iy = 32; iy; iy--)
    {
        for(ix = 32; ix; ix--)
        {
            c1 = (*p1 >> f1) & 1;
            c2 = (*p2 >> f2) & 1;
            c3 = (*p3 >> f3) & 1;
            c4 = (*p4 >> f4) & 1;

            if(c2) *p1 |= 1 << f1; else *p1 &= ~(1 << f1);
            if(c4) *p2 |= 1 << f2; else *p2 &= ~(1 << f2);
            if(c1) *p3 |= 1 << f3; else *p3 &= ~(1 << f3);
            if(c3) *p4 |= 1 << f4; else *p4 &= ~(1 << f4);

            f1--; if(f1 < 0) f1 = 7, p1++;
            p2 -= 8;
            p3 += 8;
            f4++; if(f4 > 7) f4 = 0, p4--;
        }

        p1 += 4;
        p2 += 32 * 8, f2--; if(f2 < 0) f2 = 7, p2++;
        p3 -= 32 * 8, f3++; if(f3 > 7) f3 = 0, p3--;
        p4 -= 4;
    }
}


//=====================================
//
//=====================================


//! 指定位置が 1 かどうか

BOOL CTileImageA1::isPixelOn(int x,int y) const
{
    int tx,ty;
    LPBYTE p;

    if(!calcPixelToTile(&tx, &ty, x, y))
        return FALSE;
    else
    {
        p = (LPBYTE)getTile(tx, ty);

        if(!p)
            return FALSE;
        else
        {
            x = (x - m_nOffX) & 63;
            y = (y - m_nOffY) & 63;

            p += (y << 3) + (x >> 3);

            return (*p & (1 << (7 - (x & 7))));
        }
    }
}


//=====================================
// 1px拡張/縮小
//=====================================


//! 1px拡張/縮小

void CTileImageA1::edgeA1(BOOL bInflate)
{
    CTileImageA1 imgSrc;
    LPBYTE pbuf;
    void **ppSrc;
    int tx,ty,px,py,w,h;

    if(!m_ppTile) return;

    //判定用にコピー

    if(!imgSrc.copy(*this)) return;

    //タイルごとに処理

    ppSrc = imgSrc.getTileBuf();
    w     = imgSrc.getArrayW();
    h     = imgSrc.getArrayH();

    pbuf = new BYTE[9 * 68];

    for(ty = 0, py = imgSrc.getOffY(); ty < h; ty++, py += 64)
    {
        for(tx = 0, px = imgSrc.getOffX(); tx < w; tx++, px += 64, ppSrc++)
        {
            if(*ppSrc)
            {
                imgSrc._getExBuf(pbuf, ppSrc, tx, ty, px, py);

                if(bInflate)
                    _edgeA1_inflate(pbuf, px - 1, py - 1);
                else
                    _edgeA1_deflate(pbuf, px, py);
            }
        }
    }

    delete []pbuf;
}

//! 1px拡張・タイル処理

void CTileImageA1::_edgeA1_inflate(LPBYTE pbuf,int px,int py)
{
    LPBYTE ps;
    BYTE f;
    int ix,iy,n;
    RGBAFIX15 col;

    col.set(0,0,0,0x8000);

    ps = pbuf + 9 * 1;

    for(iy = 0; iy < 66; iy++, ps++)
    {
        f = 0x40;

        for(ix = 0; ix < 66; ix++)
        {
            if(!(*ps & f))
            {
                if((ps[-9] & f) || (ps[9] & f))
                    //上下
                    n = 1;
                else
                {
                    //左右

                    n = (f == 0x80)? ps[-1] & 1: *ps & (f << 1);

                    if(!n) n = (f == 1)? ps[1] & 0x80: *ps & (f >> 1);
                }

                if(n) setPixel_subdraw(px + ix, py + iy, col);
            }

            f >>= 1;
            if(!f) { f = 0x80; ps++; }
        }
    }
}

//! 1px縮小・タイル処理

void CTileImageA1::_edgeA1_deflate(LPBYTE pbuf,int px,int py)
{
    LPBYTE ps;
    BYTE f;
    int ix,iy,n;
    RGBAFIX15 col;

    col.zero();

    ps = pbuf + 9 * 2;

    for(iy = 0; iy < 64; iy++, ps++)
    {
        f = 0x20;

        for(ix = 0; ix < 64; ix++)
        {
            if(*ps & f)
            {
                if(!(ps[-9] & f) || !(ps[9] & f))
                    //上下
                    n = 1;
                else
                {
                    //左右

                    n = !((f == 0x80)? ps[-1] & 1: *ps & (f << 1));

                    if(!n) n = !((f == 1)? ps[1] & 0x80: *ps & (f >> 1));
                }

                if(n) setPixel_subdraw(px + ix, py + iy, col);
            }

            f >>= 1;
            if(!f) { f = 0x80; ps++; }
        }
    }
}


//=====================================
// 塗りつぶし時用
//=====================================


//! 水平線描画

BOOL CTileImageA1::drawLineH_A1(int x1,int x2,int y)
{
    void **ppTile;
    LPBYTE pd;
    int i,tx,typos;
    BYTE f;

    ppTile = getTileBufAlloc(x1, y);
    if(!ppTile) return FALSE;

    tx    = (x1 - m_nOffX) & 63;
    typos = ((y - m_nOffY) & 63) << 3;
    f     = 1 << (7 - (tx & 7));
    pd    = (LPBYTE)*ppTile + typos + (tx >> 3);

    for(i = x1; i <= x2; i++)
    {
        *pd |= f;

        //次へ

        if(i == x2) break;

        tx++;

        f >>= 1;
        if(f == 0)
        {
            f = 0x80;
            pd++;

            //次のタイル

            if(tx == 64)
            {
                ppTile++;

                if(!(*ppTile))
                {
                    *ppTile = _allocTileClear();
                    if(!(*ppTile)) return FALSE;
                }

                pd = (LPBYTE)*ppTile + typos;
                tx = 0;
            }
        }
    }

    return TRUE;
}

//! 垂直線描画

BOOL CTileImageA1::drawLineV_A1(int y1,int y2,int x)
{
    void **ppTile;
    LPBYTE pd;
    int i,tx,ty;
    BYTE f;

    ppTile = getTileBufAlloc(x, y1);
    if(!ppTile) return FALSE;

    tx    = (x - m_nOffX) & 63;
    ty    = (y1 - m_nOffY) & 63;
    f     = 1 << (7 - (tx & 7));
    pd    = (LPBYTE)*ppTile + (ty << 3) + (tx >> 3);

    for(i = y1; i <= y2; i++)
    {
        *pd |= f;

        //次へ

        if(i == y2) break;

        ty++;

        if(ty != 64)
            pd += 8;
        else
        {
            ppTile += m_nArrayW;

            if(!(*ppTile))
            {
                *ppTile = _allocTileClear();
                if(!(*ppTile)) return FALSE;
            }

            pd = (LPBYTE)*ppTile + (tx >> 3);
            ty = 0;
        }
    }

    return TRUE;
}

//! このイメージで点がある部分に pimgDst へ点を描画 (setPixelDraw)

void CTileImageA1::drawPixelEachTile(CTileImage *pimgDst,const RGBAFIX15 &col)
{
    void **pp;
    LPBYTE pSrc;
    int tx,ty,x,y,ix,iy;
    BYTE f;
    void (CTileImage::*func)(int,int,const RGBAFIX15 &) = m_pinfo->funcDrawPixel;

    pp = m_ppTile;
    y  = m_nOffY;

    for(ty = m_nArrayH; ty > 0; ty--, y += 64)
    {
        x = m_nOffX;

        for(tx = m_nArrayW; tx > 0; tx--, x += 64, pp++)
        {
            if(!(*pp)) continue;

            pSrc = (LPBYTE)*pp;
            f    = 0x80;

            for(iy = 0; iy < 64; iy++)
            {
                for(ix = 0; ix < 64; ix++)
                {
                    if(*pSrc & f)
                        (pimgDst->*func)(x + ix, y + iy, col);

                    f >>= 1;
                    if(f == 0)
                    {
                        f = 0x80;
                        pSrc++;
                    }
                }
            }
        }
    }
}

//! A1同士の結合（配列が同じであること）

void CTileImageA1::combine_A1(const CTileImage &img)
{
    void **ppDst,**ppSrc;
    LPDWORD ps,pd;
    int i,j;

    ppDst = m_ppTile;
    ppSrc = img.getTileBuf();

    for(i = m_nArrayW * m_nArrayH; i; i--, ppSrc++, ppDst++)
    {
        if(*ppSrc && *ppDst)
        {
            //4バイト単位で OR で結合

            ps = (LPDWORD)*ppSrc;
            pd = (LPDWORD)*ppDst;

            for(j = 8 * 64 / 4; j; j--)
                *(pd++) |= *(ps++);
        }
        else if(*ppSrc && !(*ppDst))
        {
            //dst を確保してコピー

            *ppDst = _allocTile();
            if(*ppDst)
                ::memcpy(*ppDst, *ppSrc, 8 * 64);
        }
    }
}


//=====================================
// ベジェ曲線（XOR描画用）
//=====================================


//! ベジェ線描画（XOR描画用）
/*!
    @param prc  描画範囲が入る（範囲調整は行なっていない）
    @param type 0 で補助線2を描画しない
*/

void CTileImageA1::drawBezierTemp(AXPoint *pPos,AXRect *prc,int type)
{
    double x[4],y[4];
    double t,tt,t1,t2,t3,t4;
    int i,nx,ny,sx,sy;
    AXRect rc;
    RGBAFIX15 col;

    col.set(0,0,0,0x8000);

    sx = pPos[0].x, sy = pPos[0].y;

    //補助線

    drawLineB(sx, sy, pPos[2].x, pPos[2].y, col, FALSE);
    drawBox(pPos[2].x - 2, pPos[2].y - 2, 5, 5, col);

    if(type)
    {
        drawLineB(pPos[3].x, pPos[3].y, pPos[1].x, pPos[1].y, col, FALSE);
        drawBox(pPos[3].x - 2, pPos[3].y - 2, 5, 5, col);
    }

    //最小・最大

    rc.left = rc.right = sx;
    rc.top = rc.bottom = sy;

    rc.incPoint(pPos[1].x, pPos[1].y);
    rc.incPoint(pPos[2].x - 2, pPos[2].y - 2);
    rc.incPoint(pPos[2].x + 2, pPos[2].y + 2);
    rc.incPoint(pPos[3].x - 2, pPos[3].y - 2);
    rc.incPoint(pPos[3].x + 2, pPos[3].y + 2);

    //曲線

    x[0] = pPos[0].x, y[0] = pPos[0].y;
    x[1] = pPos[2].x, y[1] = pPos[2].y;
    x[2] = pPos[3].x, y[2] = pPos[3].y;
    x[3] = pPos[1].x, y[3] = pPos[1].y;

    for(i = 1, t = 0.01; i <= 100; i++, t += 0.01)
    {
        tt = 1 - t;

        t1 = tt * tt * tt;
        t2 = 3 * t * tt * tt;
        t3 = 3 * t * t * tt;
        t4 = t * t * t;

        nx = (int)(x[0] * t1 + x[1] * t2 + x[2] * t3 + x[3] * t4 + 0.5);
        ny = (int)(y[0] * t1 + y[1] * t2 + y[2] * t3 + y[3] * t4 + 0.5);

        //描画

        drawLineB(sx, sy, nx, ny, col, FALSE);

        //

        rc.incPoint(nx, ny);

        sx = nx, sy = ny;
    }

    *prc = rc;
}


//=====================================
// XOR合成
//=====================================


//! CXImage に XOR で合成（ベジェ曲線時）

void CTileImageA1::blendXor(CXImage *pimgDst,const AXRectSize &rcs)
{
    TILERECTINFO info;
    int ix,iy,xx,yy;
    void **pp;

    //各情報取得

    pp = getTileRect_inImageInfo(&info, rcs);
    if(!pp) return;

    //タイルごとに合成

    yy = info.ptTopPx.y;

    for(iy = info.rcTile.bottom - info.rcTile.top + 1; iy > 0; iy--, yy += 64, pp += info.pitch)
    {
        xx = info.ptTopPx.x;

        for(ix = info.rcTile.right - info.rcTile.left + 1; ix > 0 ; ix--, xx += 64, pp++)
        {
            if(*pp)
                _blendTile_xor(pimgDst, *pp, xx, yy, info.rcClip);
        }
    }
}

//! CXImage に XOR 合成（タイル単位）

void CTileImageA1::_blendTile_xor(CXImage *pimgDst,void *pTile,int dx,int dy,const AXRect &rcClip)
{
    int w,h,sx,sy,pitchd,ix,iy,bpp;
    BYTE stf,f;
    LPBYTE pSrc,pSrcBk,pDst;

    sx = sy = 0;
    w = h = 64;

    //クリッピング

    if(dx < rcClip.left) w += dx - rcClip.left, sx += rcClip.left - dx, dx = rcClip.left;
    if(dy < rcClip.top)  h += dy - rcClip.top , sy += rcClip.top  - dy, dy = rcClip.top;
    if(dx + w > rcClip.right)  w = rcClip.right  - dx;
    if(dy + h > rcClip.bottom) h = rcClip.bottom - dy;

    if(w <= 0 || h <= 0) return;

    //

    pSrc   = (LPBYTE)pTile + (sy << 3) + (sx >> 3);
    pDst   = pimgDst->getBufPt(dx, dy);
    bpp    = pimgDst->getBytes();
    pitchd = pimgDst->getPitch() - bpp * w;

    stf = 1 << (7 - (sx & 7));

    //

    for(iy = h; iy > 0; iy--)
    {
        pSrcBk = pSrc;
        f      = stf;

        for(ix = w; ix > 0; ix--, pDst += bpp)
        {
            if(*pSrc & f)
                pimgDst->setPixelBufXor(pDst);

            f >>= 1;
            if(!f)
            {
                f = 0x80;
                pSrc++;
            }
        }

        pSrc = pSrcBk + 8;
        pDst += pitchd;
    }
}


//=====================================
// 選択範囲
//=====================================


//! キャンバスに選択範囲の輪郭を描画
/*!
    @param rcsImg info.rcsDst に相当するイメージ範囲

    描画するキャンバス範囲に相当するイメージ範囲内のピクセルをすべて判定し、描画する。
*/

void CTileImageA1::drawSelection(CXImage *pimgDst,const DRAWCANVASINFO &info,const AXRectSize &rcsImg)
{
    TILERECTINFO tr;
    int ix,iy,xx,yy;
    AXRect rcDstClip;
    void **pp;
    LPBYTE pbuf;

    if(!m_ppTile) return;

    //各情報取得

    pp = getTileRect_inImageInfo(&tr, rcsImg);
    if(!pp) return;

    rcDstClip.set(info.rcsDst);

    //タイルごとに処理

    pbuf = new BYTE[9 * 68];    //64x64 + 周辺2px分

    for(iy = tr.rcTile.top, yy = tr.ptTopPx.y; iy <= tr.rcTile.bottom; iy++, yy += 64, pp += tr.pitch)
    {
        for(ix = tr.rcTile.left, xx = tr.ptTopPx.x; ix <= tr.rcTile.right; ix++, xx += 64, pp++)
        {
            if(*pp)
            {
                _getExBuf(pbuf, pp, ix, iy, xx, yy);

                _drawSelection_tile(pimgDst, pbuf, xx - 1, yy - 1, tr.rcClip, rcDstClip, info);
            }
        }
    }

    delete []pbuf;
}

//! 指定タイルの周辺 2px を含むデータをバッファにセット
/*
    68x68 の1bit。
    (1,1)-(66,66) はタイルの境界でピクセルが有/無の場合に輪郭描画を処理するため。
    (0,0)-(67,67) は上記の輪郭判定用。
*/

void CTileImageA1::_getExBuf(LPBYTE pbuf,void **ppTile,int tx,int ty,int dx,int dy)
{
    int i,j;
    LPBYTE ps,pd;
    BYTE bt,bt2;

    //Yの上下2px分をクリア

    ::memset(pbuf, 0, 18);
    ::memset(pbuf + 66 * 9, 0, 18);

    //現在のタイルをバッファの(2,2)に64x64コピー

    ps = (LPBYTE)(*ppTile);
    pd = pbuf + 2 * 9;

    for(i = 64; i; i--)
    {
        for(j = 8, bt2 = 0; j; j--)
        {
            bt = *(ps++);

            *(pd++) = bt2 | (bt >> 2);

            bt2 = (bt & 3) << 6;
        }

        *(pd++) = bt2;
    }

    //左のタイルの右端2px（透明なら何もしない）

    if(tx && *(ppTile - 1))
    {
        ps = (LPBYTE)(*(ppTile - 1)) + 7;
        pd = pbuf + 2 * 9;

        for(i = 64; i; i--, ps += 8, pd += 9)
            *pd |= (*ps & 3) << 6;
    }

    //右のタイルの左端2px

    if(tx < m_nArrayW - 1 && *(ppTile + 1))
    {
        ps = (LPBYTE)(*(ppTile + 1));
        pd = pbuf + 2 * 9 + 8;

        for(i = 64; i; i--, ps += 8, pd += 9)
            *pd |= (*ps & 0xc0) >> 2;
    }

    //上のタイルの下端2px

    if(ty && *(ppTile - m_nArrayW))
    {
        ps = (LPBYTE)(*(ppTile - m_nArrayW)) + 62 * 8;
        pd = pbuf;

        for(i = 2; i; i--)
        {
            for(j = 8, bt2 = 0; j; j--)
            {
                bt = *(ps++);

                *(pd++) = bt2 | (bt >> 2);

                bt2 = (bt & 3) << 6;
            }

            *(pd++) = bt2;
        }
    }

    //下のタイルの上端2px

    if(ty < m_nArrayH - 1 && *(ppTile + m_nArrayW))
    {
        ps = (LPBYTE)(*(ppTile + m_nArrayW));
        pd = pbuf + 66 * 9;

        for(i = 2; i; i--)
        {
            for(j = 8, bt2 = 0; j; j--)
            {
                bt = *(ps++);

                *(pd++) = bt2 | (bt >> 2);

                bt2 = (bt & 3) << 6;
            }

            *(pd++) = bt2;
        }
    }

    //(1,1)-(66,66)の4隅

    if(!isPixelTransparent(dx - 1, dy - 1))
        pbuf[9] |= 0x40;

    if(!isPixelTransparent(dx + 64, dy - 1))
        pbuf[9 + 8] |= 0x20;

    if(!isPixelTransparent(dx - 1, dy + 64))
        pbuf[66 * 9] |= 0x40;

    if(!isPixelTransparent(dx + 64, dy + 64))
        pbuf[66 * 9 + 8] |= 0x20;
}

//! 選択範囲輪郭描画（タイル単位）
/*
    タイル(64x64)とその周辺1px分を判定する。
    pbuf は 68x68 (Y1列は 9Byte)。

    タイルの周辺1pxを含めるのは、タイルの端にピクセルがあり、かつ隣合うタイルが空の場合、
    空タイル側では判定処理が行われないので、そのピクセルの境界が描画されないため。
*/

void CTileImageA1::_drawSelection_tile(CXImage *pimgDst,LPBYTE pbuf,int dx,int dy,
    const AXRect &rcClip,const AXRect &rcDstClip,const DRAWCANVASINFO &info)
{
    int sx,sy,w,h,ix,iy,n,x,y,right,bottom,bPixel,flag;
    double ddx,ddy,ddx2,ddy2,add[4];
    BYTE f,stf;
    LPBYTE pSrc,pSrcBk;

    //クリッピング（バッファの外周1pxは含めない）

    sx = sy = 0;
    w = h = 66;

    if(dx < rcClip.left) w += dx - rcClip.left, sx += rcClip.left - dx, dx = rcClip.left;
    if(dy < rcClip.top)  h += dy - rcClip.top , sy += rcClip.top  - dy, dy = rcClip.top;
    if(dx + w > rcClip.right)  w = rcClip.right  - dx;
    if(dy + h > rcClip.bottom) h = rcClip.bottom - dy;

    if(w <= 0 || h <= 0) return;

    //

    right  = m_pinfo->nImgW - 1;
    bottom = m_pinfo->nImgH - 1;

    sx++, sy++;

    pSrc = pbuf + sy * 9 + (sx >> 3);
    stf  = 1 << (7 - (sx & 7));

    bPixel = (info.pParam->dScale <= 1);

    //

    info.imgTocanv(&ddx2, &ddy2, dx, dy);

    add[0] = info.pParam->dScale * info.pParam->dCos;
    add[1] = info.pParam->dScale * info.pParam->dSin;
    add[2] = -add[1];
    add[3] = add[0];

    if(info.bHRev) add[0] = -add[0], add[1] = -add[1];

    //

    y = dy;

    for(iy = 0; iy < h; iy++, y++)
    {
        x = dx;
        f = stf;
        pSrcBk = pSrc;

        ddx = ddx2;
        ddy = ddy2;

        for(ix = 0; ix < w; ix++, x++)
        {
            if(*pSrc & f)
            {
                //--------- イメージ端の場合

                flag = (y == 0) | ((y == bottom) << 1) | ((x == 0) << 2) | ((x == right) << 3);

                if(flag)
                {
                    if(bPixel)
                        pimgDst->setPixelSelection((int)(ddx + 0.5), (int)(ddy + 0.5), rcDstClip);
                    else
                        _drawSelection_point(pimgDst, ddx, ddy, add, flag, rcDstClip);
                }
            }
            else
            {
                //--------- 上下左右いずれかに点があれば輪郭描画

                flag = 0;

                //上

                if(pSrc[-9] & f) flag |= 1;

                //下

                if(pSrc[9] & f) flag |= 2;

                //左

                n = (f == 0x80)? pSrc[-1] & 1: *pSrc & (f << 1);

                if(n) flag |= 4;

                //右

                n = (f == 1)? pSrc[1] & 0x80: *pSrc & (f >> 1);

                if(n) flag |= 8;

                //描画

                if(flag)
                {
                    if(bPixel)
                        pimgDst->setPixelSelection((int)(ddx + 0.5), (int)(ddy + 0.5), rcDstClip);
                    else
                        _drawSelection_point(pimgDst, ddx, ddy, add, flag, rcDstClip);
                }
            }

            f >>= 1;
            if(!f) { f = 0x80; pSrc++; }

            ddx += add[0];
            ddy += add[1];
        }

        pSrc = pSrcBk + 9;

        ddx2 += add[2];
        ddy2 += add[3];
    }
}

//! 輪郭描画

void CTileImageA1::_drawSelection_point(CXImage *pimgDst,double dx,double dy,double *padd,int flag,const AXRect &rcDstClip)
{
    dx += 0.5;
    dy += 0.5;

    if(flag & 1)
        pimgDst->drawLineSelection(dx, dy, dx + padd[0], dy + padd[1], rcDstClip);

    if(flag & 2)
        pimgDst->drawLineSelection(dx + padd[2], dy + padd[3], dx + padd[0] + padd[2], dy + padd[1] + padd[3], rcDstClip);

    if(flag & 4)
        pimgDst->drawLineSelection(dx, dy, dx + padd[2], dy + padd[3], rcDstClip);

    if(flag & 8)
        pimgDst->drawLineSelection(dx + padd[0], dy + padd[1], dx + padd[0] + padd[2], dy + padd[1] + padd[3], rcDstClip);
}
