/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CTileImageg [タイルで管理するイメージの基底クラス]

    - 一つのタイルは 64x64 pixel。
    - 各色のタイプごとに継承クラスを作る。
    - 2次元配列でタイルを管理し、色が存在する部分だけ確保する。
    - 2次元配列は必要に応じて自動的にサイズが拡張される。
    - 配列のポインタが TILEPT_EMPTY(1) の場合、UNDO用で"元データが空"という意味。
*/


#include <stdlib.h>
#include <string.h>

#include "CTileImage.h"

#include "CImageRGB16.h"
#include "CProgressDlg.h"
#include "AXImage.h"


//------------------------------

TILEIMGDRAWINFO *CTileImage::m_pinfo = NULL;
BRUSHDRAWPARAM *CTileImage::m_pbrush = NULL;
AXRand *CTileImage::m_prand = NULL;

//------------------------------


CTileImage::CTileImage()
{
    m_ppTile = NULL;
    m_nArrayW = m_nArrayH = 0;

    m_col.r = m_col.g = m_col.b = 0;
}

CTileImage::~CTileImage()
{
    free();
}


//=========================
// 解放
//=========================


//! すべて（タイル＆配列）解放

void CTileImage::free()
{
    if(m_ppTile)
    {
        //タイル解放

        freeAllTile();

        //配列解放

        ::free(m_ppTile);

        m_ppTile = NULL;
        m_nArrayW = m_nArrayH = 0;
    }
}

//! タイルをすべて解放（配列はそのまま）

void CTileImage::freeAllTile()
{
    void **p = m_ppTile;
    int i;

    if(p)
    {
        for(i = m_nArrayW * m_nArrayH; i > 0; i--, p++)
            _freeOneTile(p);
    }
}

//! 一つの指定タイルを解放

void CTileImage::_freeOneTile(void **ppTile)
{
    if(*ppTile)
    {
        //値が1の場合はUNDO用で元データが空という意味なので、解放は行わない。

        if(*ppTile != (void *)TILEPT_EMPTY)
            ::free(*ppTile);

        *ppTile = NULL;
    }
}

//! 指定位置のタイルを削除

void CTileImage::freeTilePos(int tx,int ty)
{
    void **p = getTileBufPt(tx, ty);
    if(*p) _freeOneTile(p);
}

//! 空のタイルを解放（アンドゥ側でタイルが確保されている部分のみ）
/*
    アンドゥが確保されているということは、そこはイメージが変更された部分。
    イメージが変更された部分のうち、消しゴム等で透明になったタイルは解放してメモリ負担を軽減させる。
*/

void CTileImage::freeEmptyTileFromUndo()
{
    void **ppUndo,**ppDst;
    int i;

    ppUndo = (m_pinfo->pimgUndo)->getTileBuf();
    ppDst  = m_ppTile;

    for(i = m_nArrayW * m_nArrayH; i; i--, ppUndo++, ppDst++)
    {
        if(*ppUndo && *ppDst)
        {
            if(_isTransparentTile(*ppDst))
                _freeOneTile(ppDst);
        }
    }
}

//! 空のタイルを解放
/*!
    @return TRUE ですべて空になった
*/

BOOL CTileImage::freeEmptyTile()
{
    void **pp = m_ppTile;
    int i;
    BOOL bAll = TRUE;

    if(!pp) return FALSE;

    for(i = m_nArrayW * m_nArrayH; i; i--, pp++)
    {
        if(*pp)
        {
            if(_isTransparentTile(*pp))
                _freeOneTile(pp);
            else
                bAll = FALSE;
        }
    }

    return bAll;
}

//! 指定px範囲内のタイルで、透明なタイルを解放

void CTileImage::freeEmptyTileRect(int x,int y,int w,int h)
{
    int tx1,ty1,tx2,ty2,ix,iy,pitch;
    void **pp;

    if(!m_ppTile) return;

    calcPixelToTile(&tx1, &ty1, x, y);
    calcPixelToTile(&tx2, &ty2, x + w - 1, y + h - 1);

    pp = getTileBufPt(tx1, ty1);
    pitch = m_nArrayW - (tx2 - tx1 + 1);

    for(iy = ty1; iy <= ty2; iy++, pp += pitch)
    {
        for(ix = tx1; ix <= tx2; ix++, pp++)
        {
            if(*pp && _isTransparentTile(*pp))
                _freeOneTile(pp);
        }
    }
}


//==============================
// 確保
//==============================


//! タイル確保

void *CTileImage::_allocTile()
{
    return (void *)::malloc(m_nTileSize);
}

//! タイル確保＋クリア

void *CTileImage::_allocTileClear()
{
    void *p = (void *)::malloc(m_nTileSize);

    ::memset(p, 0, m_nTileSize);

    return p;
}

//! タイル配列の確保処理
/*!
    m_nArrayW,m_nArrayH はあらかじめセットしておく。
*/

BOOL CTileImage::_allocTileArray()
{
    long size;

    size = m_nArrayW * m_nArrayH * sizeof(void *);

    //確保

    m_ppTile = (void **)::malloc(size);
    if(!m_ppTile) return FALSE;

    //ゼロクリア

    ::memset(m_ppTile, 0, size);

    return TRUE;
}

//! リサイズなど用に新規タイル配列確保

void **CTileImage::_allocTileArrayNew(int w,int h,BOOL bClear)
{
    long size;
    void **ppNew;

    size = w * h * sizeof(void *);

    ppNew = (void **)::malloc(size);
    if(!ppNew) return NULL;

    if(bClear) ::memset(ppNew, 0, size);

    return ppNew;
}


//===========================
// タイル配列リサイズ
//===========================


//! タイル配列リサイズ
/*!
    拡張処理のみ。サイズの縮小処理は行わない。
    左/上方向への拡張ならば、オフセット位置をずらす。

    @param movx,movy  負の値の場合、タイル位置の移動数。0以上なら右/下方向への拡張。
    @return FALSE で失敗
*/

BOOL CTileImage::resizeTileArray(int movx,int movy,int newW,int newH)
{
    void **ppNew,**ppd;
    int ix,iy,tx,ty;

    if(movx > 0) movx = 0;
    if(movy > 0) movy = 0;

    //変更なし

    if(newW == m_nArrayW && newH == m_nArrayH)
        return TRUE;

    //新規確保

    ppNew = _allocTileArrayNew(newW, newH, TRUE);
    if(!ppNew) return FALSE;

    //元データコピー

    ppd = ppNew;

    for(iy = newH, ty = movy; iy > 0; iy--, ty++)
    {
        for(ix = newW, tx = movx; ix > 0; ix--, tx++, ppd++)
        {
            if(tx >= 0 && tx < m_nArrayW && ty >= 0 && ty < m_nArrayH)
                *ppd = *(m_ppTile + ty * m_nArrayW + tx);
        }
    }

    //入れ替え

    ::free(m_ppTile);

    m_ppTile  = ppNew;
    m_nArrayW = newW;
    m_nArrayH = newH;
    m_nOffX  += movx << 6;
    m_nOffY  += movy << 6;

    return TRUE;
}

//! 情報からタイル配列リサイズ (UNDO用。縮小処理あり)

BOOL CTileImage::resizeTileArray(const TILEIMGINFO &info)
{
    void **ppNew,**ppd;
    int ix,iy,tx,ty,mx,my;

    //配列サイズ変更なし

    if(info.nArrayW == m_nArrayW && info.nArrayH == m_nArrayH)
        return TRUE;

    //新規確保

    ppNew = _allocTileArrayNew(info.nArrayW, info.nArrayH, TRUE);
    if(!ppNew) return FALSE;

    //元データコピー

    ppd = ppNew;
    mx  = (info.nOffX - m_nOffX) >> 6;
    my  = (info.nOffY - m_nOffY) >> 6;

    for(iy = info.nArrayH, ty = my; iy > 0; iy--, ty++)
    {
        for(ix = info.nArrayW, tx = mx; ix > 0; ix--, tx++, ppd++)
        {
            if(tx >= 0 && tx < m_nArrayW && ty >= 0 && ty < m_nArrayH)
                *ppd = *(m_ppTile + ty * m_nArrayW + tx);
        }
    }

    //入れ替え

    ::free(m_ppTile);

    m_ppTile  = ppNew;
    m_nArrayW = info.nArrayW;
    m_nArrayH = info.nArrayH;
    m_nOffX   = info.nOffX;
    m_nOffY   = info.nOffY;

    return TRUE;
}

//! 画像サイズ全体に合うようタイル配列リサイズ

BOOL CTileImage::resizeTileArray_incImage()
{
    AXRect rc;
    int x1,y1,x2,y2;

    //現在描画可能な範囲(px)

    getEnableDrawRectPixel(&rc);

    //px -> タイル位置

    calcPixelToTile(&x1, &y1, rc.left, rc.top);
    calcPixelToTile(&x2, &y2, rc.right, rc.bottom);

    //リサイズ

    return resizeTileArray(x1, y1, x2 - x1 + 1, y2 - y1 + 1);
}

//! 指定イメージと同じ構成になるようにタイル配列リサイズ
/*
    src の配列リサイズ後すぐに実行すること。
    setPixelDraw() 時のアンドゥイメージのリサイズ用。
*/

BOOL CTileImage::_resizeTileArray_same(const CTileImage &src)
{
    return resizeTileArray(
		(src.getOffX() - m_nOffX) >> 6, (src.getOffY() - m_nOffY) >> 6,
		src.getArrayW(), src.getArrayH());
}

//! 2つのイメージ全体を含むようにタイル配列リサイズ

BOOL CTileImage::resizeTileArray_double(const CTileImage &src)
{
    AXRect rc,rc2;
    int x1,y1,x2,y2;

    //px範囲

    calcTileRectToPixel(&rc);
    src.calcTileRectToPixel(&rc2);

    rc.combine(rc2);

    //px を this のタイル位置に変換

    calcPixelToTile(&x1, &y1, rc.left, rc.top);
    calcPixelToTile(&x2, &y2, rc.right, rc.bottom);

    return resizeTileArray(x1, y1, x2 - x1 + 1, y2 - y1 + 1);
}


//===========================
// 作成
//===========================


//! 新規作成（イメージサイズ指定）

BOOL CTileImage::create(int w,int h)
{
    free();

    m_nArrayW = (w + 63) / 64;
    m_nArrayH = (h + 63) / 64;
    m_nOffX = m_nOffY = 0;

    return _allocTileArray();
}

//! 情報を指定して作成

BOOL CTileImage::create(const TILEIMGINFO &info)
{
    free();

    m_nArrayW = info.nArrayW;
    m_nArrayH = info.nArrayH;
    m_nOffX   = info.nOffX;
    m_nOffY   = info.nOffY;

    return _allocTileArray();
}

//! 指定px範囲を含む範囲で作成
/*!
    オフセットは左上pxに合わせる
*/

BOOL CTileImage::create(const AXRect &rc)
{
    free();

    m_nArrayW = (rc.right - rc.left + 1 + 63) / 64;
    m_nArrayH = (rc.bottom - rc.top + 1 + 63) / 64;
    m_nOffX   = rc.left;
    m_nOffY   = rc.top;

    return _allocTileArray();
}

//! 指定イメージと同じになるようタイル配列を作成

BOOL CTileImage::createSame(const CTileImage &src)
{
    BOOL flag = FALSE;

    //情報コピー

    m_nOffX = src.m_nOffX;
    m_nOffY = src.m_nOffY;

    //タイル配列作成するか

    if(!m_ppTile)
        flag = TRUE;
    else
        flag = (m_nArrayW != src.m_nArrayW || m_nArrayH != src.m_nArrayH);

    if(!flag) return TRUE;

    //再作成（確保できなかった場合はNULLとなる）

    free();

    m_nArrayW = src.m_nArrayW;
    m_nArrayH = src.m_nArrayH;

    return _allocTileArray();
}

//! コピー作成（同じカラータイプであること）

BOOL CTileImage::copy(const CTileImage &src)
{
    void **ppSrc,**ppDst;
    int i;

    free();

    //情報コピー

    m_nArrayW   = src.m_nArrayW;
    m_nArrayH   = src.m_nArrayH;
    m_nOffX     = src.m_nOffX;
    m_nOffY     = src.m_nOffY;

    //タイル配列確保

    if(!_allocTileArray()) return FALSE;

    //タイルコピー

    ppSrc = src.getTileBuf();
    ppDst = m_ppTile;

    for(i = m_nArrayW * m_nArrayH; i; i--, ppSrc++, ppDst++)
    {
        if(*ppSrc)
        {
            *ppDst = _allocTile();
            if(!(*ppDst)) return FALSE;

            ::memcpy(*ppDst, *ppSrc, m_nTileSize);
        }
    }

    return TRUE;
}


//============================
// 計算/取得
//============================


//! 画像の情報取得

void CTileImage::getInfo(TILEIMGINFO *pinfo)
{
    pinfo->nArrayW = m_nArrayW;
    pinfo->nArrayH = m_nArrayH;
    pinfo->nOffX   = m_nOffX;
    pinfo->nOffY   = m_nOffY;
}

//! px位置からタイル位置取得
/*!
    @return TRUEで範囲内、FALSEで範囲外
*/

BOOL CTileImage::calcPixelToTile(int *pTX,int *pTY,int px,int py) const
{
    int tx,ty;

    tx = (px - m_nOffX) >> 6;
    ty = (py - m_nOffY) >> 6;

    *pTX = tx;
    *pTY = ty;

    return (tx >= 0 && tx < m_nArrayW && ty >= 0 && ty < m_nArrayH);
}

//! タイル位置から左上のpx位置取得

void CTileImage::calcTileToPixel(int *pPX,int *pPY,int tx,int ty) const
{
    *pPX = (tx << 6) + m_nOffX;
    *pPY = (ty << 6) + m_nOffY;
}

//! タイル全体の四隅をpxで取得

void CTileImage::calcTileRectToPixel(AXRect *prc) const
{
    int x1,y1,x2,y2;

    calcTileToPixel(&x1, &y1, 0, 0);
    calcTileToPixel(&x2, &y2, m_nArrayW - 1, m_nArrayH - 1);

    prc->left   = x1;
    prc->top    = y1;
    prc->right  = x2 + 63;
    prc->bottom = y2 + 63;
}

//! px位置からタイル取得
/*!
    @return 配列範囲外はNULL
*/

void *CTileImage::_getTileFromPixel(int x,int y) const
{
    int tx,ty;

    if(calcPixelToTile(&tx, &ty, x, y))
        return getTile(tx, ty);
    else
        return NULL;
}

//! 指定px範囲内のタイル範囲を取得
/*!
    @param rcs イメージのpx範囲
    @return FALSE でタイル範囲外
*/

BOOL CTileImage::getTileRect_inImage(AXRect *prc,const AXRectSize &rcs)
{
    int sx,sy,ex,ey,flag;

    //四隅のタイル位置

    calcPixelToTile(&sx, &sy, rcs.x, rcs.y);
    calcPixelToTile(&ex, &ey, rcs.x + rcs.w - 1, rcs.y + rcs.h - 1);

    //範囲外かどうかのフラグ

    flag  = (sx < 0) | ((ex < 0) << 1);
    flag |= ((sx >= m_nArrayW) << 2) | ((ex >= m_nArrayW) << 3);

    flag |= ((sy < 0) << 4) | ((ey < 0) << 5);
    flag |= ((sy >= m_nArrayH) << 6) | ((ey >= m_nArrayH) << 7);

    //範囲外か

    if((flag & 0x03) == 0x03 || (flag & 0x30) == 0x30) return FALSE;
    if((flag & 0x0c) == 0x0c || (flag & 0xc0) == 0xc0) return FALSE;

    //調整

    if(flag & 1) sx = 0;
    if(flag & 8) ex = m_nArrayW - 1;

    if(flag & 0x10) sy = 0;
    if(flag & 0x80) ey = m_nArrayH - 1;

    //セット

    prc->left   = sx;
    prc->right  = ex;
    prc->top    = sy;
    prc->bottom	= ey;

    return TRUE;
}

//! 指定px範囲内を、タイルごとに処理する際の各データ取得
/*!
    @return 開始位置のタイルのポインタ(NULLで範囲外)
*/

void **CTileImage::getTileRect_inImageInfo(TILERECTINFO *pdst,const AXRectSize &rcs)
{
    //イメージの範囲内のタイルの範囲

    if(!getTileRect_inImage(&pdst->rcTile, rcs)) return NULL;

    //左上タイルのpx位置

    calcTileToPixel(&pdst->ptTopPx.x, &pdst->ptTopPx.y, pdst->rcTile.left, pdst->rcTile.top);

    //rcs を AXRect へ（クリッピング用）

    pdst->rcClip.left   = rcs.x;
    pdst->rcClip.top    = rcs.y;
    pdst->rcClip.right  = rcs.x + rcs.w;
    pdst->rcClip.bottom = rcs.y + rcs.h;

    //

    pdst->pitch = m_nArrayW - (pdst->rcTile.right - pdst->rcTile.left + 1);

    //タイル位置

    return getTileBufPt(pdst->rcTile.left, pdst->rcTile.top);
}

//! タイル全体とイメージ範囲を含むpx範囲取得（現在描画可能なpx範囲）

void CTileImage::getEnableDrawRectPixel(AXRect *prc)
{
    AXRect rc1,rc2;

    //タイルの四隅 -> px

    calcTileRectToPixel(&rc1);

    //キャンバス範囲

    rc2.left = rc2.top = 0;
    rc2.right  = m_pinfo->nImgW - 1;
    rc2.bottom = m_pinfo->nImgH - 1;

    //2つのpx範囲を合わせる

    rc1.combine(rc2);

    *prc = rc1;
}

//! AXRect 範囲(px)を描画可能な範囲内に調整
/*!
    @return FALSE で範囲外
*/

BOOL CTileImage::clipRectInEnableDraw(AXRect *prc)
{
    AXRect rc1;

    getEnableDrawRectPixel(&rc1);

    //範囲外

    if(prc->left > rc1.right || prc->top > rc1.bottom) return FALSE;
    if(prc->right < rc1.left || prc->bottom < rc1.top) return FALSE;

    //調整

    if(prc->left   < rc1.left)   prc->left = rc1.left;
    if(prc->top    < rc1.top)    prc->top = rc1.top;
    if(prc->right  > rc1.right)  prc->right = rc1.right;
    if(prc->bottom > rc1.bottom) prc->bottom = rc1.bottom;

    return TRUE;
}

//! 指定位置の範囲内の平均色取得

void CTileImage::getPointAdvColor(RGBADOUBLE *pdst,double x,double y,double radius)
{
    int xx,yy,x1,y1,x2,y2,cnt;
    double dx,dy,rr,dr,dg,db,da,a;
    RGBAFIX15 col;

    x1 = (int)(x - radius);
    y1 = (int)(y - radius);
    x2 = (int)(x + radius);
    y2 = (int)(y + radius);

    rr = radius * radius;

    dr = dg = db = da = 0;
    cnt = 0;

    for(yy = y1; yy <= y2; yy++)
    {
        dy = yy - y;
        dy = dy * dy;

        for(xx = x1; xx <= x2; xx++)
        {
            dx = xx - x;
            dx = dx * dx;

            if(dx + dy < rr)
            {
                getPixel(&col, xx, yy);

                a = (double)col.a / 0x8000;

                dr += col.r * a;
                dg += col.g * a;
                db += col.b * a;
                da += a;

                cnt++;
            }
        }
    }

    //平均色

    if(da == 0)
        pdst->zero();
    else
    {
        pdst->a = da / cnt;

        da = 1.0 / da;

        pdst->r = dr * da;
        pdst->g = dg * da;
        pdst->b = db * da;
    }
}

//! イメージがある部分のpx範囲取得（イメージ範囲外も含む）
/*!
    @param pTileCnt イメージがあるタイル数が入る。NULL でなし。
    @return FALSE で範囲なし
*/

BOOL CTileImage::getExistImgRectPx(FLAGRECT *pDst,int *pTileCnt)
{
    void **pp = m_ppTile;
    AXRect rc;
    int x,y,cnt = 0;

    //タイル最小・最大位置

    rc.set(m_nArrayW, m_nArrayH, -1, -1);

    for(y = 0; y < m_nArrayH; y++)
    {
        for(x = 0; x < m_nArrayW; x++, pp++)
        {
            if(*pp)
            {
                rc.incPoint(x, y);
                cnt++;
            }
        }
    }

    //タイル数

    if(pTileCnt) *pTileCnt = cnt;

    //タイル -> px変換

    if(rc.right == -1)
    {
        //一つもない

        pDst->zero();
        return FALSE;
    }
    else
    {
        calcTileToPixel(&x, &y, 0, 0);

        pDst->set(x + (rc.left << 6), y + (rc.top << 6),
                  x + (rc.right << 6) + 63, y + (rc.bottom << 6) + 63);

        return TRUE;
    }
}

//! 確保されているタイル数取得

DWORD CTileImage::getAllocTileCnt()
{
    void **pp = m_ppTile;
    DWORD i,cnt = 0;

    for(i = m_nArrayW * m_nArrayH; i; i--, pp++)
    {
        if(*pp) cnt++;
    }

    return cnt;
}

//! 指定情報と現在のタイル配列サイズが同じか

BOOL CTileImage::isSameArraySize(const TILEIMGINFO &info)
{
    return (m_nArrayW == info.nArrayW && m_nArrayH == info.nArrayH);
}


//=============================
// ほか
//=============================


//! 指定px位置のタイルを確保して取得（タイル配列範囲外は除く）

void **CTileImage::getTileBufAlloc(int px,int py)
{
    int tx,ty;
    void **pp;

    if(!calcPixelToTile(&tx, &ty, px, py))
        return NULL;

    pp = getTileBufPt(tx, ty);

    if(!(*pp))
    {
        *pp = _allocTileClear();
        if(!(*pp)) return NULL;
    }

    return pp;
}

//! 指定タイル位置にタイルを確保して取得

void *CTileImage::getTileAlloc(int tx,int ty)
{
    void **pp = getTileBufPt(tx, ty);

    if(!(*pp))
    {
        *pp = _allocTileClear();
        if(!(*pp)) return NULL;
    }

    return *pp;
}

void *CTileImage::getTileAlloc(DWORD pos)
{
    void **pp = m_ppTile + pos;

    if(!(*pp))
    {
        *pp = _allocTileClear();
        if(!(*pp)) return NULL;
    }

    return *pp;
}


//! CImageRGB16 に合成

void CTileImage::blendToRGB16(CImageRGB16 *pimgDst,const AXRectSize &rcs,int opacity,
    void (*funcCol)(RGBAFIX15 *,const RGBFIX15 &))
{
    TILERECTINFO info;
    int ix,iy,xx,yy;
    void **pp;

    if(!m_ppTile || opacity == 0) return;

    //各情報取得

    pp = getTileRect_inImageInfo(&info, rcs);
    if(!pp) return;

    //タイルごとに合成

    yy = info.ptTopPx.y;

    for(iy = info.rcTile.bottom - info.rcTile.top + 1; iy > 0; iy--, yy += 64, pp += info.pitch)
    {
        xx = info.ptTopPx.x;

        for(ix = info.rcTile.right - info.rcTile.left + 1; ix > 0 ; ix--, xx += 64, pp++)
        {
            if(*pp)
                _blendTile_RGB16(pimgDst, *pp, xx, yy, opacity, info.rcClip, funcCol);
        }
    }
}

//! CImageRGB16 から変換して作成
/*
    ! this は COLTYPE_RGBA である
*/

BOOL CTileImage::createFromImg(const CImageRGB16 &src,CProgressDlg *pdlg)
{
    int ix,iy;
    void **pp;

    if(!create(src.getWidth(), src.getHeight())) return FALSE;

    //

    pp = m_ppTile;

    pdlg->beginProgSub(30, m_nArrayW * m_nArrayH, TRUE);

    for(iy = 0; iy < m_nArrayH; iy++)
    {
        for(ix = 0; ix < m_nArrayW; ix++, pp++)
        {
            *pp = _allocTile();
            if(!*pp) return FALSE;

            src.getBufTile((RGBAFIX15 *)(*pp), ix << 6, iy << 6);

            pdlg->incProgSub();
        }
    }

    return TRUE;
}

//! RGBA8bit のバッファから変換

BOOL CTileImage::convertFromRGBA8bit(LPBYTE pSrcBuf,int w,int h,CProgressDlg *pdlg,int step)
{
    int tx,ty,ix,iy,dw,dh,px,py,pitch,i,bZero;
    RGBAFIX15 *pbuf,*pDst;
    LPBYTE pSrc;
    void **pp;

    pbuf = (RGBAFIX15 *)::malloc(sizeof(RGBAFIX15) * 64 * 64);
    if(!pbuf) return FALSE;

    //

    pdlg->beginProgSub(step, m_nArrayH);

    pp = m_ppTile;

    for(ty = 0, py = 0; ty < m_nArrayH; ty++, py += 64)
    {
        for(tx = 0, px = 0; tx < m_nArrayW; tx++, px += 64, pp++)
        {
            //64x64 分を RGBAFIX15 バッファに

            dw = dh = 64;

            if(px + 64 > w) dw = w - px;
            if(py + 64 > h) dh = h - py;

            pSrc  = pSrcBuf + (w << 2) * py + (px << 2);
            pDst  = pbuf;
            pitch = (w - dw) << 2;
            bZero = TRUE;

            for(iy = 0; iy < dh; iy++, pSrc += pitch, pDst += 64 - dw)
            {
                for(ix = 0; ix < dw; ix++, pDst++, pSrc += 4)
                {
                    if(pSrc[3] == 0)
                        pDst->zero();
                    else
                    {
                        for(i = 0; i < 4; i++)
                            pDst->c[i] = ((pSrc[i] << 15) + 127) / 255;

                        bZero = FALSE;
                    }
                }
            }

            //すべて透明なら処理なし

            if(bZero) continue;

            //タイル作成&セット

            *pp = _allocTileClear();
            if(!(*pp)) return FALSE;

            _setFromBufRGBA_tile(*pp, pbuf, FALSE);
        }

        pdlg->incProgSub();
    }

    ::free(pbuf);

    return TRUE;
}

//! ヒストグラム取得
/*
    輝度は 0-256 の範囲 (257個)
*/

void CTileImage::getHistogram(LPDWORD pbuf)
{
    void **pp = m_ppTile;
    LPWORD pw;
    int i,j,v,bRGBA;

    bRGBA = (m_nColType == COLTYPE_RGBA);

    for(i = m_nArrayW * m_nArrayH; i; i--, pp++)
    {
        if(!(*pp)) continue;

        pw = (LPWORD)(*pp);

        if(bRGBA)
        {
            for(j = 64 * 64; j; j--, pw += 4)
            {
                if(pw[3])
                {
                    v = (pw[0] * 77 + pw[1] * 150 + pw[2] * 29) >> (8 + 7);
                    pbuf[v]++;
                }
            }
        }
        else
        {
            for(j = 64 * 64; j; j--, pw += 2)
            {
                if(pw[1])
                {
                    v = pw[0] >> 7;
                    pbuf[v]++;
                }
            }
        }
    }
}

//! フィルタプレビュー用描画
/*
    rcs は CTileImage の範囲。pimgDst へは (1,1) を始点として描画。
*/

void CTileImage::drawFilterPrev(AXImage *pimgDst,const AXRectSize &rcs)
{
    LPBYTE pd;
    int ix,iy,bytes,pitch,fy,r,g,b,dc,colCk[2];
    RGBAFIX15 col;

    if(!m_ppTile || !pimgDst->isExist()) return;

    colCk[0] = 28784;
    colCk[1] = 24672;

    pd    = pimgDst->getBufPt(1, 1);
    bytes = pimgDst->getBytes();
    pitch = pimgDst->getPitch() - bytes * rcs.w;

    for(iy = 0; iy < rcs.h; iy++, pd += pitch)
    {
        fy = (iy >> 3) & 1;

        for(ix = 0; ix < rcs.w; ix++, pd += bytes)
        {
            getPixel(&col, rcs.x + ix, rcs.y + iy);

            dc = colCk[fy ^ ((ix >> 3) & 1)];

            if(col.a == 0)
                r = g = b = dc;
            else if(col.a == 0x8000)
            {
                r = col.r;
                g = col.g;
                b = col.b;
            }
            else
            {
                r = ((col.r - dc) * col.a >> 15) + dc;
                g = ((col.g - dc) * col.a >> 15) + dc;
                b = ((col.b - dc) * col.a >> 15) + dc;
            }

            r = r * 255 >> 15;
            g = g * 255 >> 15;
            b = b * 255 >> 15;

            pimgDst->setPixelBuf(pd, r, g, b);
        }
    }
}
