/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CImageRGB16 [RGB(15bit固定小数点)カラーのイメージ]

    - レイヤ合成先のイメージで使う。
*/


#include <stdlib.h>
#include <string.h>

#include "CImageRGB16.h"

#include "CProgressDlg.h"

#include "AXImage.h"
#include "AXBMPSave.h"
#include "AXPNGSave.h"
#include "AXJPEG.h"
#include "AXGIFSave.h"
#include "AXApp.h"
#include "AXUtil.h"

//----------------------

#define FIXF_BIT    14
#define FIXF_VAL    (1 << FIXF_BIT)

//----------------------


CImageRGB16::CImageRGB16()
{
    m_pbuf = NULL;
    m_nWidth = m_nHeight = 0;
}

CImageRGB16::~CImageRGB16()
{
    free();
}

//! 解放

void CImageRGB16::free()
{
    if(m_pbuf)
    {
        ::free(m_pbuf);

        m_pbuf = NULL;
        m_nWidth = m_nHeight = 0;
    }
}

//! 作成

BOOL CImageRGB16::create(int w,int h)
{
    free();

    m_pbuf = (RGBFIX15 *)::malloc(w * h * sizeof(RGBFIX15));
    if(!m_pbuf) return FALSE;

    m_nWidth  = w;
    m_nHeight = h;

    return TRUE;
}

//! クリア

void CImageRGB16::clear(const RGBFIX15 &col)
{
    RGBFIX15 *p = m_pbuf;
    int i,size;

    size = sizeof(RGBFIX15) * m_nWidth;

    for(i = m_nWidth; i > 0; i--)
        *(p++) = col;

    //2行目以降は1行ずつコピー

    for(i = m_nHeight - 1; i > 0; i--, p += m_nWidth)
        ::memcpy(p, p - m_nWidth, size);
}

//! 指定範囲クリア

void CImageRGB16::clear(const AXRectSize &rcs,const RGBFIX15 &col)
{
    RGBFIX15 *p = getBufPt(rcs.x, rcs.y);
    int ix,iy,pitch;

    pitch = m_nWidth - rcs.w;

    for(iy = rcs.h; iy > 0; iy--, p += pitch)
    {
        for(ix = rcs.w; ix > 0; ix--)
            *(p++) = col;
    }
}

//! チェック柄で塗りつぶし
/*!
    @param pCol 2個の配列
*/

void CImageRGB16::fillCheck(const AXRectSize &rcs,const RGBFIX15 *pCol)
{
    int ix,iy,x,y,fy,pitch;
    RGBFIX15 *p;

    p = getBufPt(rcs.x, rcs.y);
    pitch = m_nWidth - rcs.w;

    y = rcs.y;

    for(iy = rcs.h; iy; iy--, y++, p += pitch)
    {
        fy = (y >> 3) & 1;

        for(ix = rcs.w, x = rcs.x; ix; ix--, x++)
            *(p++) = pCol[((x >> 3) & 1) ^ fy];
    }
}

//! タイル分のバッファ取得

void CImageRGB16::getBufTile(RGBAFIX15 *pDst,int x,int y) const
{
    int w,h,ix,iy,pitchs;
    const RGBFIX15 *pSrc;

    w = h = 64;

    if(x + w > m_nWidth) w = m_nWidth - x;
    if(y + h > m_nHeight) h = m_nHeight - y;

    if(w != 64 || h != 64)
        ::memset(pDst, 0, sizeof(RGBAFIX15) * 64 * 64);

    //

    pSrc   = getBufPt(x, y);
    pitchs = m_nWidth - w;

    for(iy = h; iy; iy--)
    {
        for(ix = w; ix; ix--, pDst++, pSrc++)
        {
            pDst->r = pSrc->r;
            pDst->g = pSrc->g;
            pDst->b = pSrc->b;
            pDst->a = 0x8000;
        }

        pDst += 64 - w;
        pSrc += pitchs;
    }
}

//! バッファのデータを RGB8bit に変換して上書き

void CImageRGB16::toRGB8bit()
{
    RGBFIX15 *pSrc,col;
    LPBYTE pDst;
    int i;

    pSrc = m_pbuf;
    pDst = (LPBYTE)m_pbuf;

    for(i = m_nWidth * m_nHeight; i; i--, pDst += 3)
    {
        col = *(pSrc++);

        pDst[0] = (col.r * 255 + 0x4000) >> 15;
        pDst[1] = (col.g * 255 + 0x4000) >> 15;
        pDst[2] = (col.b * 255 + 0x4000) >> 15;
    }
}

//! バッファの RGB8bit データをパレット番号に変換して上書き

void CImageRGB16::_toPaletteCol(LPDWORD pPalBuf,int pcnt)
{
    LPBYTE pSrc,pDst;
    LPDWORD p;
    int i,j;
    DWORD col;

    pSrc = (LPBYTE)m_pbuf;
    pDst = pSrc;

    for(i = m_nWidth * m_nHeight; i; i--, pSrc += 3)
    {
        col = _RGB(pSrc[0], pSrc[1], pSrc[2]);

        for(j = 0, p = pPalBuf; j < pcnt; j++)
        {
            if(*(p++) == col) break;
        }

        if(j >= pcnt) j = 0;

        *(pDst++) = (BYTE)j;
    }
}



//=====================================
// 画像ファイルへ出力
//=====================================

#define SAVERET_OK          0
#define SAVERET_ERR         1
#define SAVERET_GIF_OVER    2


//! BMP保存

int CImageRGB16::saveBMP(const AXString &filename,int dpi,CProgressDlg *pdlg)
{
    AXBMPSave bmp;
    AXBMPSave::INFO info;
    int y,x,pitch;
    LPBYTE pSrc,pDst,ps,pd;

    info.nWidth     = m_nWidth;
    info.nHeight    = m_nHeight;
    info.nBits      = 24;
    info.nResoH     = AXDPItoDPM(dpi);
    info.nResoV     = info.nResoH;

    if(!bmp.openFile(filename, &info)) return SAVERET_ERR;

    //

    pdlg->beginProgSub(30, m_nHeight, TRUE);

    pitch = m_nWidth * 3;
    pSrc  = (LPBYTE)m_pbuf + pitch * (m_nHeight - 1);
    pDst  = bmp.getLineBuf();

    for(y = m_nHeight; y > 0; y--, pSrc -= pitch)
    {
        ps = pSrc;
        pd = pDst;

        for(x = m_nWidth; x > 0; x--, ps += 3, pd += 3)
        {
            pd[0] = ps[2];
            pd[1] = ps[1];
            pd[2] = ps[0];
        }

        bmp.putLine();

        pdlg->incProgSub();
    }

    bmp.close();

    return SAVERET_OK;
}

//! PNG保存

int CImageRGB16::savePNG(const AXString &filename,int nLevel,int colTP,int dpi,CProgressDlg *pProgDlg)
{
    AXPNGSave png;
    AXPNGSave::INFO info;
    int y,pcnt,pitch;
    LPBYTE ps;
    BYTE filter = 0;

    //パレット作成

    pcnt = _getPalette8bit(&info.memPalette);

    //

    info.nWidth  = m_nWidth;
    info.nHeight = m_nHeight;
    info.nBits   = (pcnt <= 0)? 24: 8;
    info.nPalCnt = (pcnt <= 0)? 0: pcnt;

    if(!png.openFile(filename, &info)) return SAVERET_ERR;

    //解像度

    y = AXDPItoDPM(dpi);
    png.put_pHYs(y, y);

    //透過色

    if(colTP != -1)
        png.put_tRNS(colTP);

    //---------- イメージ

    if(!png.startImg(nLevel)) return SAVERET_ERR;

    pProgDlg->beginProgSub(30, m_nHeight, TRUE);

    if(pcnt <= 0)
        pitch = m_nWidth * 3;
    else
    {
        _toPaletteCol(info.memPalette, pcnt);
        pitch = m_nWidth;
    }

    //

    ps = (LPBYTE)m_pbuf;

    for(y = m_nHeight; y > 0; y--, ps += pitch)
    {
        png.putImg(&filter, 1);
        png.putImg(ps, pitch);

        pProgDlg->incProgSub();
    }

    png.endImg();

    //

    png.close();

    return SAVERET_OK;
}

//! JPEG保存

int CImageRGB16::saveJPEG(const AXString &filename,int nQuality,int nSamp,int dpi,CProgressDlg *pProgDlg)
{
    AXJPEG jpeg;
    AXJPEG::SAVEINFO info;
    LPBYTE ps,pd;
    int y,pitch;

    info.nWidth     = m_nWidth;
    info.nHeight    = m_nHeight;
    info.nQuality   = nQuality;
    info.nDPI_H     = dpi;
    info.nDPI_V     = dpi;
    info.uFlags     = 0;

    if(nSamp == 444)
        info.nSamplingType = AXJPEG::SAMP_444;
    else if(nSamp == 422)
        info.nSamplingType = AXJPEG::SAMP_422;
    else
        info.nSamplingType = AXJPEG::SAMP_411;

    if(!jpeg.openFileSave(filename, &info)) return SAVERET_ERR;

    //

    pProgDlg->beginProgSub(30, m_nHeight, TRUE);

    ps = (LPBYTE)m_pbuf;
    pd = jpeg.getImgBuf();
    pitch = m_nWidth * 3;

    for(y = m_nHeight; y > 0; y--, ps += pitch)
    {
        ::memcpy(pd, ps, pitch);

        jpeg.putLine();

        pProgDlg->incProgSub();
    }

    jpeg.close();

    return SAVERET_OK;
}

//! GIF保存

int CImageRGB16::saveGIF(const AXString &filename,int colTP,CProgressDlg *pdlg)
{
    AXGIFSave gif;
    AXGIFSave::GLOBALINFO ginfo;
    AXGIFSave::IMGINFO info;
    int pcnt;

    //パレット作成

    pcnt = _getPalette8bit(&ginfo.memGlobalPal);

    if(pcnt == -1) return SAVERET_ERR;
    else if(pcnt == 0) return SAVERET_GIF_OVER;

    //

    ginfo.wWidth    = m_nWidth;
    ginfo.wHeight   = m_nHeight;
    ginfo.nBits     = gif.calcBits(pcnt);
    ginfo.nGlobalPalCnt = pcnt;
    ginfo.nBKIndex  = 0;

    if(!gif.openFile(filename, &ginfo)) return SAVERET_ERR;

    //画像情報

    gif.initIMGINFO(&info, m_nWidth, m_nHeight);

    if(colTP != -1)
        info.nTPIndex = gif.getColIndex(colTP, ginfo.memGlobalPal, pcnt);

    //イメージ

    _toPaletteCol(ginfo.memGlobalPal, pcnt);

    gif.putImg(info, (LPBYTE)m_pbuf, (info.nTPIndex != -1));

    //

    gif.close();

    return SAVERET_OK;
}

//! パレットを作成
/*!
    @return 色数。-1 でエラー、0 で 257 色以上。
*/

int CImageRGB16::_getPalette8bit(AXMem *pmem)
{
    int i,j,f,pcnt = 0;
    DWORD col;
    LPBYTE pSrc;
    LPDWORD pPal,p;

    //確保

    if(!pmem->alloc(sizeof(DWORD) * 256)) return -1;

    //

    pSrc = (LPBYTE)m_pbuf;
    pPal = *pmem;

    for(i = m_nWidth * m_nHeight; i; i--, pSrc += 3)
    {
        //既存パレットから検索

        col = _RGB(pSrc[0], pSrc[1], pSrc[2]);

        for(j = pcnt, f = 0, p = pPal; j > 0; j--)
        {
            if(col == *(p++))
            {
                f = TRUE;
                break;
            }
        }

        //色追加

        if(!f)
        {
            if(pcnt >= 256)
            {
                pmem->free();
                return 0;
            }

            pPal[pcnt++] = col;
        }
    }

    return pcnt;
}


//=====================================
// キャンバスへ描画
//=====================================


//! キャンバスへの描画（回転なし・低品質）

void CImageRGB16::drawCanvasNormal(AXImage *pimgDst,const DRAWCANVASINFO &info)
{
    int w,h,bpp,pitchd,ix,iy,n;
    long addx,addy,stx,fx,fy;
    double scalex;
    LPBYTE pDst;
    RGBFIX15 *pSrcY,*pSrc;
    DWORD pixEx;

    w = info.rcsDst.w;
    h = info.rcsDst.h;

    pDst   = pimgDst->getBufPt(info.rcsDst.x, info.rcsDst.y);
    bpp    = pimgDst->getBytes();
    pitchd = pimgDst->getPitch() - w * bpp;
    pixEx  = axapp->rgbToPix(info.dwExCol);

    scalex = info.pParam->dScaleDiv;

    addx = addy = (long)(info.pParam->dScaleDiv * FIXF_VAL);

    if(info.bHRev)
    {
        scalex = -scalex;
        addx   = -addx;
    }

    stx = (long)(((info.rcsDst.x - info.nScrollX) * scalex + info.nBaseX) * FIXF_VAL);
    fy  = (long)(((info.rcsDst.y - info.nScrollY) * info.pParam->dScaleDiv + info.nBaseY) * FIXF_VAL);

    //

    for(iy = h; iy > 0; iy--, fy += addy)
    {
        n = fy >> FIXF_BIT;

        //Yが範囲外

        if(fy < 0 || n >= m_nHeight)
        {
            pimgDst->lineHBuf(pDst, w, info.dwExCol);
            pDst += pimgDst->getPitch();
            continue;
        }

        //X

        pSrcY = m_pbuf + n * m_nWidth;

        for(ix = w, fx = stx; ix > 0; ix--, fx += addx, pDst += bpp)
        {
            n = fx >> FIXF_BIT;

            if(fx >= 0 && n < m_nWidth)
            {
                pSrc = pSrcY + n;

                pimgDst->setPixelBuf(pDst, pSrc->r * 255 >> 15, pSrc->g * 255 >> 15, pSrc->b * 255 >> 15);
            }
            else
                pimgDst->setPixelBufPx(pDst, pixEx);
        }

        pDst += pitchd;
    }
}

//! キャンバスへの描画（回転なし・縮小[4x4オーバーサンプリング]）

void CImageRGB16::drawCanvasScaleDown(AXImage *pimgDst,const DRAWCANVASINFO &info)
{
    int w,h,bpp,pitchd,ix,iy,jx,jy,srcx,srcy,r,g,b,tblX[4];
    long addx,addy,stx,fx,fy,addx2,addy2,jf;
    double scalex;
    LPBYTE pDst;
    RGBFIX15 *pSrcY[4],*pSrc;
    DWORD pixEx;

    w = info.rcsDst.w;
    h = info.rcsDst.h;

    pixEx = axapp->rgbToPix(info.dwExCol);

    pDst   = pimgDst->getBufPt(info.rcsDst.x, info.rcsDst.y);
    bpp    = pimgDst->getBytes();
    pitchd = pimgDst->getPitch() - w * bpp;

    scalex = info.pParam->dScaleDiv;
    if(info.bHRev) scalex = -scalex;

    addx = (long)(scalex * FIXF_VAL);
    addy = (long)(info.pParam->dScaleDiv * FIXF_VAL);

    addx2 = addx >> 2;
    addy2 = addy >> 2;

    stx = (long)(((info.rcsDst.x - info.nScrollX) * scalex + info.nBaseX) * FIXF_VAL);
    fy  = (long)(((info.rcsDst.y - info.nScrollY) * info.pParam->dScaleDiv + info.nBaseY) * FIXF_VAL);

    //

    for(iy = h; iy > 0; iy--, fy += addy)
    {
        srcy = fy >> FIXF_BIT;

        //Yが範囲外

        if(srcy < 0 || srcy >= m_nHeight)
        {
            pimgDst->lineHBuf(pDst, w, info.dwExCol);
            pDst += pimgDst->getPitch();
            continue;
        }

        //Y位置テーブル

        for(jy = 0, jf = fy; jy < 4; jy++, jf += addy2)
        {
            srcy = jf >> FIXF_BIT;
            if(srcy < 0) srcy = 0; else if(srcy >= m_nHeight) srcy = m_nHeight - 1;

            pSrcY[jy] = m_pbuf + srcy * m_nWidth;
        }

        //X

        for(ix = w, fx = stx; ix > 0; ix--, fx += addx, pDst += bpp)
        {
            srcx = fx >> FIXF_BIT;

            if(srcx < 0 || srcx >= m_nWidth)
                //Xが範囲外
                pimgDst->setPixelBufPx(pDst, pixEx);
            else
            {
                //X位置テーブル

                for(jx = 0, jf = fx; jx < 4; jx++, jf += addx2)
                {
                    srcx = jf >> FIXF_BIT;
                    if(srcx < 0) srcx = 0; else if(srcx >= m_nWidth) srcx = m_nWidth - 1;

                    tblX[jx] = srcx;
                }

                //4x4 平均値

                r = g = b = 0;

                for(jy = 0; jy < 4; jy++)
                {
                    for(jx = 0; jx < 4; jx++)
                    {
                        pSrc = pSrcY[jy] + tblX[jx];

                        r += pSrc->r;
                        g += pSrc->g;
                        b += pSrc->b;
                    }
                }

                //

                r = (r >> 4) * 255 >> 15;
                g = (g >> 4) * 255 >> 15;
                b = (b >> 4) * 255 >> 15;

                pimgDst->setPixelBuf(pDst, r, g, b);
            }
        }

        pDst += pitchd;
    }
}

//! キャンバスへの描画（回転あり・補間なし）

void CImageRGB16::drawCanvasRotNormal(AXImage *pimgDst,const DRAWCANVASINFO &info)
{
    int w,h,bpp,pitchd,ix,iy,sx,sy;
    long stx,sty,fx,fy,dxx,dxy,dyx,dyy;
    LPBYTE pDst;
    RGBFIX15 *pSrc;
    DWORD pixEx;
    double scalex;

    w = info.rcsDst.w;
    h = info.rcsDst.h;

    pixEx = axapp->rgbToPix(info.dwExCol);

    pDst   = pimgDst->getBufPt(info.rcsDst.x, info.rcsDst.y);
    bpp    = pimgDst->getBytes();
    pitchd = pimgDst->getPitch() - w * bpp;

    //

    fx = info.rcsDst.x - info.nScrollX;
    fy = info.rcsDst.y - info.nScrollY;

    scalex = info.pParam->dScaleDiv;
    if(info.bHRev) scalex = -scalex;

    stx = (long)(((fx * info.pParam->dCosRev - fy * info.pParam->dSinRev) * scalex + info.nBaseX) * FIXF_VAL);
    sty = (long)(((fx * info.pParam->dSinRev + fy * info.pParam->dCosRev) * info.pParam->dScaleDiv + info.nBaseY) * FIXF_VAL);

    //xx,yy:cos, xy:sin, yx:-sin

    dxx = (long)(info.pParam->dCosRev * info.pParam->dScaleDiv * FIXF_VAL);
    dxy = (long)(info.pParam->dSinRev * info.pParam->dScaleDiv * FIXF_VAL);
    dyx = -dxy;
    dyy = dxx;

    if(info.bHRev)
        dxx = -dxx, dyx = -dyx;

    //

    for(iy = h; iy > 0; iy--)
    {
        fx = stx, fy = sty;

        for(ix = w; ix > 0; ix--, pDst += bpp)
        {
            sx = fx >> FIXF_BIT;
            sy = fy >> FIXF_BIT;

            if(sx >= 0 && sx < m_nWidth && sy >= 0 && sy < m_nHeight)
            {
                pSrc = m_pbuf + sy * m_nWidth + sx;

                pimgDst->setPixelBuf(pDst, pSrc->r * 255 >> 15, pSrc->g * 255 >> 15, pSrc->b * 255 >> 15);
            }
            else
                pimgDst->setPixelBufPx(pDst, pixEx);

            fx += dxx;
            fy += dxy;
        }

        stx  += dyx;
        sty  += dyy;
        pDst += pitchd;
    }
}

//! キャンバスへの描画（回転あり・補間あり[2x2オーバーサンプリング]）

void CImageRGB16::drawCanvasRotHiQuality(AXImage *pimgDst,const DRAWCANVASINFO &info)
{
    int w,h,bpp,pitchd,ix,iy,jx,jy,sx,sy,r,g,b;
    long stx,sty,fx,fy,dxx,dxy,dyx,dyy,fxx,fyy,fxx2,fyy2,jdxx,jdxy,jdyx,jdyy;
    LPBYTE pDst;
    RGBFIX15 *pSrc;
    DWORD pixEx;
    double scalex;

    w = info.rcsDst.w;
    h = info.rcsDst.h;

    pixEx = axapp->rgbToPix(info.dwExCol);

    pDst   = pimgDst->getBufPt(info.rcsDst.x, info.rcsDst.y);
    bpp    = pimgDst->getBytes();
    pitchd = pimgDst->getPitch() - w * bpp;

    //

    fx = info.rcsDst.x - info.nScrollX;
    fy = info.rcsDst.y - info.nScrollY;

    scalex = info.pParam->dScaleDiv;
    if(info.bHRev) scalex = -scalex;

    stx = (long)(((fx * info.pParam->dCosRev - fy * info.pParam->dSinRev) * scalex + info.nBaseX) * FIXF_VAL);
    sty = (long)(((fx * info.pParam->dSinRev + fy * info.pParam->dCosRev) * info.pParam->dScaleDiv + info.nBaseY) * FIXF_VAL);

    //xx,yy:cos, xy:sin, yx:-sin

    dxx = (long)(info.pParam->dCosRev * info.pParam->dScaleDiv * FIXF_VAL);
    dxy = (long)(info.pParam->dSinRev * info.pParam->dScaleDiv * FIXF_VAL);
    dyx = -dxy;
    dyy = dxx;

    if(info.bHRev)
        dxx = -dxx, dyx = -dyx;

    //

    jdxx = dxx >> 1, jdxy = dxy >> 1;
    jdyx = dyx >> 1, jdyy = dyy >> 1;

    //

    for(iy = h; iy > 0; iy--)
    {
        fx = stx, fy = sty;

        for(ix = w; ix > 0; ix--, pDst += bpp)
        {
            sx = fx >> FIXF_BIT;
            sy = fy >> FIXF_BIT;

            if(sx >= 0 && sx < m_nWidth && sy >= 0 && sy < m_nHeight)
            {
                //範囲内

                r = g = b = 0;
                fxx = fx, fyy = fy;

                for(jy = 2; jy; jy--, fxx += jdyx, fyy += jdyy)
                {
                    fxx2 = fxx, fyy2 = fyy;

                    for(jx = 2; jx; jx--, fxx2 += jdxx, fyy2 += jdxy)
                    {
                        sx = fxx2 >> FIXF_BIT;
                        sy = fyy2 >> FIXF_BIT;

                        if(sx < 0) sx = 0; else if(sx >= m_nWidth) sx = m_nWidth - 1;
                        if(sy < 0) sy = 0; else if(sy >= m_nHeight) sy = m_nHeight - 1;

                        pSrc = m_pbuf + sy * m_nWidth + sx;

                        r += pSrc->r;
                        g += pSrc->g;
                        b += pSrc->b;
                    }
                }

                r = (r >> 2) * 255 >> 15;
                g = (g >> 2) * 255 >> 15;
                b = (b >> 2) * 255 >> 15;

                pimgDst->setPixelBuf(pDst, r, g, b);
            }
            else
                //範囲外
                pimgDst->setPixelBufPx(pDst, pixEx);

            fx += dxx;
            fy += dxy;
        }

        stx  += dyx;
        sty  += dyy;
        pDst += pitchd;
    }
}
