/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    フィルタ描画関数 - サブ
*/

#include <math.h>

#include "CTileImage.h"
#include "CProgressDlg.h"

#include "filterdraw.h"
#include "filterdrawfunc.h"

#include "drawdat.h"
#include "draw_main.h"
#include "global.h"


//-------------------------------------

void FILTERDRAW::progSetMax(int max)
{
    if(pProgDlg) pProgDlg->setProgMax(max);
}

void FILTERDRAW::progBeginOneStep(int step,int max)
{
    if(pProgDlg) pProgDlg->beginProgSub(step, max, TRUE);
}

void FILTERDRAW::progBeginSub(int step,int max)
{
    if(pProgDlg) pProgDlg->beginProgSub(step, max, FALSE);
}

void FILTERDRAW::progInc()
{
    if(pProgDlg) pProgDlg->incProg();
}

void FILTERDRAW::progIncSub()
{
    if(pProgDlg) pProgDlg->incProgSub();
}

//-------------------------------------


namespace filterdraw
{

//! 処理範囲をキャンバス内にクリッピングして prs にセット

BOOL clipInCanvas(RECTANDSIZE *prs,FILTERDRAW &info)
{
    RECTANDSIZE rs = info.rs;
    int cw,ch;

    if(info.bClipping)
    {
        cw = CTileImage::m_pinfo->nImgW;
        ch = CTileImage::m_pinfo->nImgH;

        //範囲外

        if(rs.x2 < 0 || rs.x1 >= cw || rs.y2 < 0 || rs.y1 >= ch)
           return FALSE;

        //調整

        if(rs.x1 < 0) rs.x1 = 0;
        if(rs.y1 < 0) rs.y1 = 0;
        if(rs.x2 >= cw) rs.x2 = cw - 1;
        if(rs.y2 >= ch) rs.y2 = ch - 1;

        rs.w = rs.x2 - rs.x1 + 1;
        rs.h = rs.y2 - rs.y1 + 1;
    }

    *prs = rs;

    return TRUE;
}

//! 作業用イメージを確保

CTileImage *allocTmpImage(FILTERDRAW &info,const RECTANDSIZE &rs)
{
    CTileImage *p;
    AXRect rc;

    p = draw::allocTileImage(info.pimgSrc->getColType());

    rs.toRect(&rc);

    if(!p->create(rc))
    {
        delete p;
        return FALSE;
    }

    return p;
}

//! 描画する色を取得

void getDrawColor(RGBAFIX15 *pcol,int type)
{
    switch(type)
    {
        //描画色
        case 0:
            *pcol = g_draw->colDraw;
            break;
        //背景色
        case 1:
            *pcol = g_draw->colBack;
            break;
        //黒
        case 2:
            pcol->r = pcol->g = pcol->b = 0;
            break;
        //白
        case 3:
            pcol->r = pcol->g = pcol->b = 0x8000;
            break;
    }

    pcol->a = 0x8000;
}

//! rs の範囲のイメージをコピー (src -> dst)

void copyRect(FILTERDRAW &info)
{
    AXRect rc;

    info.rs.toRect(&rc);
    info.pimgDst->copyRect(info.pimgSrc, rc);
}


//==============================
// 平均色
//==============================


//! 平均色を求める場合の色を追加 (指定色)

void addAdvColor(double *pcol,const RGBAFIX15 &col)
{
    if(col.a == 0x8000)
    {
        pcol[0] += col.r;
        pcol[1] += col.g;
        pcol[2] += col.b;
        pcol[3] += 1;
    }
    else if(col.a)
    {
        double a = (double)col.a / 0x8000;

        pcol[0] += col.r * a;
        pcol[1] += col.g * a;
        pcol[2] += col.b * a;
        pcol[3] += a;
    }
}

//! 平均色を求める場合の色を追加 (指定位置)

void addAdvColor(double *pcol,CTileImage *pimg,int x,int y,BOOL bClip)
{
    RGBAFIX15 col;
    double a;

    pimg->getPixel(&col, x, y, bClip);

    if(col.a == 0x8000)
    {
        pcol[0] += col.r;
        pcol[1] += col.g;
        pcol[2] += col.b;
        pcol[3] += 1;
    }
    else if(col.a)
    {
        a = (double)col.a / 0x8000;

        pcol[0] += col.r * a;
        pcol[1] += col.g * a;
        pcol[2] += col.b * a;
        pcol[3] += a;
    }
}

//! 平均色を求める場合の色を追加 (重みあり)

void addAdvColor(double *pcol,double weight,CTileImage *pimg,int x,int y,BOOL bClip)
{
    RGBAFIX15 col;
    double a;

    pimg->getPixel(&col, x, y, bClip);

    if(col.a == 0x8000)
    {
        pcol[0] += col.r * weight;
        pcol[1] += col.g * weight;
        pcol[2] += col.b * weight;
        pcol[3] += weight;
    }
    else if(col.a)
    {
        a = (double)col.a / 0x8000 * weight;

        pcol[0] += col.r * a;
        pcol[1] += col.g * a;
        pcol[2] += col.b * a;
        pcol[3] += a;
    }
}

//! 平均色を取得

void getAdvColor(RGBAFIX15 *pdst,double *pcol,double weightmul)
{
    double d;

    pdst->a = (WORD)(pcol[3] * weightmul * 0x8000 + 0.5);

    if(pdst->a == 0)
        pdst->zero();
    else
    {
        d = 1.0 / pcol[3];

        pdst->r = (WORD)(pcol[0] * d + 0.5);
        pdst->g = (WORD)(pcol[1] * d + 0.5);
        pdst->b = (WORD)(pcol[2] * d + 0.5);
    }
}


//============================
//
//============================


//! キャンバス範囲外時の背景を指定した線形補完色取得
/*
    bktype : [0]透明 [1]端の色 [2]そのまま
*/

void getLinerColBkgnd(RGBAFIX15 *pcol,CTileImage *pimg,double dx,double dy,int x,int y,int bktype)
{
    if(dx >= 0 && dx < CTileImage::m_pinfo->nImgW && dy >= 0 && dy < CTileImage::m_pinfo->nImgH)
        //範囲内
        getLinerCol(pcol, pimg, dx, dy, 1);
    else
    {
        //範囲外

        if(bktype == 0)
            //透明
            pcol->zero();
        else if(bktype == 2)
            //そのまま
            pimg->getPixel(pcol, x, y);
        else
            //端の色
            getLinerCol(pcol, pimg, dx, dy, 1);
    }
}

//! 線形補完色取得
/*!
    @param mode 範囲外の処理 [0:そのまま 1:範囲内に収める 2:範囲外はループ]
*/

void getLinerCol(RGBAFIX15 *pcol,CTileImage *pimg,double dx,double dy,int mode)
{
    RGBAFIX15 col[4];
    double d1,d2;
    int ix,iy,i,j,fx1,fy1,fx2,fy2,n1,n2;

    d2  = ::modf(dx, &d1); if(d2 < 0) d2 += 1;
    ix  = (int)d1;
    fx1 = (int)(d2 * 0x8000 + 0.5);
    fx2 = 0x8000 - fx1;

    d2  = ::modf(dy, &d1); if(d2 < 0) d2 += 1;
    iy  = (int)d1;
    fy1 = (int)(d2 * 0x8000 + 0.5);
    fy2 = 0x8000 - fy1;

    //4点の色取得

    pimg->getPixelMode(col, ix, iy, mode);
    pimg->getPixelMode(col + 1, ix + 1, iy, mode);
    pimg->getPixelMode(col + 2, ix, iy + 1, mode);
    pimg->getPixelMode(col + 3, ix + 1, iy + 1, mode);

    if(col[0].a + col[1].a + col[2].a + col[3].a == 0)
    {
        pcol->zero();
        return;
    }

    //アルファ値計算

    n1 = (fx2 * col[0].a >> 15) + (fx1 * col[1].a >> 15);
    n2 = (fx2 * col[2].a >> 15) + (fx1 * col[3].a >> 15);

    pcol->a = (fy2 * n1 >> 15) + (fy1 * n2 >> 15);

    //RGB

    if(pcol->a == 0)
        pcol->zero();
    else
    {
        for(i = 0; i < 3; i++)
        {
            for(j = 0; j < 4; j++)
                col[j].c[i] = col[j].c[i] * col[j].a >> 15;

            n1 = (fx2 * col[0].c[i] >> 15) + (fx1 * col[1].c[i] >> 15);
            n2 = (fx2 * col[2].c[i] >> 15) + (fx1 * col[3].c[i] >> 15);

            n1 = (fy2 * n1 >> 15) + (fy1 * n2 >> 15);
            n1 = (n1 << 15) / pcol->a;
            if(n1 > 0x8000) n1 = 0x8000;

            pcol->c[i] = n1;
        }
    }
}

//! 作業用にガウスぼかしを行う
/*
    失敗時は imgDst を削除する。
*/

BOOL tmpGaussBlur(const FILTERDRAW &info,CTileImage *pimgSrc,CTileImage *pimgDst,
        int radius,TILEIMGPIXFUNC funcPix)
{
    FILTERDRAW tmpinfo;
    TILEIMGPIXFUNC funcBk = CTileImage::m_pinfo->funcDrawPixel;
    BOOL ret;

    tmpinfo = info;
    tmpinfo.pimgSrc   = pimgSrc;
    tmpinfo.pimgDst   = pimgDst;
    tmpinfo.valbar[0] = radius;

    CTileImage::m_pinfo->funcDrawPixel = funcPix;

    ret = blur_gauss(tmpinfo);

    CTileImage::m_pinfo->funcDrawPixel = funcBk;

    if(!ret)
    {
        delete pimgDst;
        return FALSE;
    }

    return TRUE;
}


//==============================
// 色変換
//==============================


//! RGB -> YCrCb 変換

void RGBtoYCrCb(int *pcol)
{
    int c[3],i;

    c[0] = (pcol[0] * 77 + pcol[1] * 150 + pcol[2] * 29) >> 8;
    c[1] = ((pcol[0] * 128 - pcol[1] * 107 - pcol[2] * 21) >> 8) + 0x4000;
    c[2] = ((pcol[0] * -43 - pcol[1] * 85 + pcol[2] * 128) >> 8) + 0x4000;

    for(i = 0; i < 3; i++)
    {
        if(c[i] < 0) c[i] = 0;
        else if(c[i] > 0x8000) c[i] = 0x8000;

        pcol[i] = c[i];
    }
}

//! YCrCb -> RGB 変換

void YCrCbtoRGB(int *pcol)
{
    int c[3],i,cr,cb;

    cr = pcol[1] - 0x4000;
    cb = pcol[2] - 0x4000;

    c[0] = pcol[0] + (cr * 359 >> 8);
    c[1] = pcol[0] - ((cr * 183 + cb * 88) >> 8);
    c[2] = pcol[0] + (cb * 454 >> 8);

    for(i = 0; i < 3; i++)
    {
        if(c[i] < 0) c[i] = 0;
        else if(c[i] > 0x8000) c[i] = 0x8000;

        pcol[i] = c[i];
    }
}

//! RGB -> HSV

void RGBtoHSV(int *pcol)
{
    int min,max,r,g,b,d,rt,gt,bt,h;

    r = pcol[0];
    g = pcol[1];
    b = pcol[2];

    if(r >= g) max = r; else max = g; if(b > max) max = b;
    if(r <= g) min = r; else min = g; if(b < min) min = b;

    pcol[2] = max;

    d = max - min;

    if(max == 0) pcol[1] = 0;
    else pcol[1] = (d << 15) / max;

    if(pcol[1] == 0)
    {
        pcol[0] = 0;
        return;
    }

    rt = max - r * 60 / d;
    gt = max - g * 60 / d;
    bt = max - b * 60 / d;

    if(r == max) h = bt - gt;
    else if(g == max) h = 120 + rt - bt;
    else h = 240 + gt - rt;

    if(h < 0) h += 360;
    else if(h >= 360) h -= 360;

    pcol[0] = h;
}

//! HSV -> RGB

void HSVtoRGB(int *pcol)
{
    int r,g,b,ht,d,t1,t2,t3,v;

    v = pcol[2];

    if(pcol[1] == 0)
        r = g = b = v;
    else
    {
        ht = pcol[0] * 6;
        d  = ht % 360;

        t1 = v * (0x8000 - pcol[1]) >> 15;
        t2 = v * (0x8000 - pcol[1] * d / 360) >> 15;
        t3 = v * (0x8000 - pcol[1] * (360 - d) / 360) >> 15;

        switch(ht / 360)
        {
            case 0: r = v, g = t3, b = t1; break;
            case 1: r = t2, g = v, b = t1; break;
            case 2: r = t1, g = v, b = t3; break;
            case 3: r = t1, g = t2, b = v; break;
            case 4: r = t3, g = t1, b = v; break;
            default: r = v, g = t1, b = t2;
        }
    }

    pcol[0] = r;
    pcol[1] = g;
    pcol[2] = b;
}

//! HSV -> RGB (S,V = max)

void HSVtoRGB_hue(RGBAFIX15 *pcol,int hue)
{
    int ht,d,t2,t3;

    ht = hue * 6;
    d  = ht % 360;

    t2 = 0x8000 - 0x8000 * d / 360;
    t3 = 0x8000 - 0x8000 * (360 - d) / 360;

    switch(ht / 360)
    {
        case 0:
            pcol->r = 0x8000, pcol->g = t3, pcol->b = 0;
            break;
        case 1:
            pcol->r = t2, pcol->g = 0x8000, pcol->b = 0;
            break;
        case 2:
            pcol->r = 0, pcol->g = 0x8000, pcol->b = t3;
            break;
        case 3:
            pcol->r = 0, pcol->g = t2, pcol->b = 0x8000;
            break;
        case 4:
            pcol->r = t3, pcol->g = 0, pcol->b = 0x8000;
            break;
        default:
            pcol->r = 0x8000, pcol->g = 0, pcol->b = t2;
    }
}

};
