/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    フィルタ描画関数 - 描画
*/

#include <math.h>
#include <string.h>

#include "CTileImage.h"
#include "CPerlinNoise.h"

#include "AXRand.h"
#include "AXMem.h"

#include "filterdraw.h"
#include "filterdrawfunc.h"

#include "drawdat.h"
#include "global.h"
#include "draw_opfunc.h"


namespace filterdraw
{

//! 枠線

BOOL draw_frame(FILTERDRAW &info)
{
    RGBAFIX15 col;
    int i,w,h;

    getDrawColor(&col, info.valcombo[0]);

    w = CTileImage::m_pinfo->nImgW;
    h = CTileImage::m_pinfo->nImgH;

    for(i = 0; i < info.valbar[0] && w > 0 && h > 0; i++)
    {
        info.pimgDst->drawBox(i, i, w, h, col);

        w -= 2;
        h -= 2;
    }

    return TRUE;
}

//! ライン

BOOL draw_line(FILTERDRAW &info)
{
    int pos,ix,iy,w,minw,mins,rangew,ranges;
    CTileImage *pimg;
    RGBAFIX15 col;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15&) = CTileImage::m_pinfo->funcDrawPixel;

    minw   = info.valbar[0];
    mins   = info.valbar[2];
    rangew = info.valbar[1] - minw; if(rangew < 0) rangew = 0;
    ranges = info.valbar[3] - mins; if(ranges < 0) ranges = 0;

    getDrawColor(&col, info.valcombo[0]);

    pimg = info.pimgDst;

    info.progSetMax(2);

    //横線

    if(info.valcheck[0])
    {
        //初期Y位置
        pos = info.rs.y1 + CTileImage::m_prand->getRange(0, ranges);
        if(ranges == 0) pos += mins;

        while(pos <= info.rs.y2)
        {
            //太さ

            w = CTileImage::m_prand->getRange(0, rangew) + minw;
            if(pos + w > info.rs.y2) w = info.rs.y2 - pos + 1;

            //描画

            for(iy = 0; iy < w; iy++)
            {
                for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
                    (pimg->*funcPix)(ix, pos + iy, col);
            }

            //次の位置

            pos += CTileImage::m_prand->getRange(0, ranges) + mins + w;
        }
    }

    info.progInc();

    //縦線

    if(info.valcheck[1])
    {
        pos = info.rs.x1 + CTileImage::m_prand->getRange(0, ranges);
        if(ranges == 0) pos += mins;

        while(pos <= info.rs.x2)
        {
            w = CTileImage::m_prand->getRange(0, rangew) + minw;
            if(pos + w > info.rs.x2) w = info.rs.x2 - pos + 1;

            for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
            {
                for(ix = 0; ix < w; ix++)
                    (pimg->*funcPix)(pos + ix, iy, col);
            }

            pos += CTileImage::m_prand->getRange(0, ranges) + mins + w;
        }
    }

    info.progInc();

    return TRUE;
}

//! チェック柄

BOOL draw_check(FILTERDRAW &info)
{
    int ix,iy,colw,roww,colMod,rowMod,yf,xf,n;
    CTileImage *pimg;
    RGBAFIX15 col;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15&) = CTileImage::m_pinfo->funcDrawPixel;

    colw = info.valbar[0];
    roww = info.valbar[1];

    if(info.valcheck[0]) roww = colw;

    colMod = colw * 2;
    rowMod = roww * 2;

    getDrawColor(&col, info.valcombo[0]);

    pimg = info.pimgDst;

    //

    for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
    {
        n = iy;
        while(n < 0) n += CTileImage::m_pinfo->nImgH;

        yf = (n % rowMod) / roww;

        for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
        {
            n = ix;
            while(n < 0) n += CTileImage::m_pinfo->nImgW;

            xf = (n % colMod) / colw;

            if((xf ^ yf) == 0)
                (pimg->*funcPix)(ix, iy, col);
        }

        info.progIncSub();
    }

    return TRUE;
}

//! アミトーン(1) 線数・DPI 指定

BOOL draw_amitone1(FILTERDRAW &info)
{
    return draw_amitone(info, (double)info.valbar[3] / (info.valbar[0] * 0.1));
}

//! アミトーン(2) サイズ指定

BOOL draw_amitone2(FILTERDRAW &info)
{
    return draw_amitone(info, info.valbar[0] * 0.1);
}

//! アミトーン
/*
    len : ボックスの1辺の長さ
*/

BOOL draw_amitone(FILTERDRAW &info,double len)
{
    int x,y,sx,sy,c,density;
    double dsin,dcos,xx,yy,len_div,len_half,rr,cx,cy,sdx,sdy,dyy;
    CTileImage *pimg;
    RGBAFIX15 col;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15&) = CTileImage::m_pinfo->funcDrawPixel;

    pimg = info.pimgDst;

    getDrawColor(&col, info.valcombo[0]);

    density = info.valbar[1];

    rr   = -info.valbar[2] * M_PI / 180.0;
    dcos = ::cos(rr);
    dsin = ::sin(rr);

    c  = (density > 50)? 100 - density: density;        //50%以上は白黒反転
    rr = ::sqrt(len * len * (c * 0.01) * (1.0 / M_PI)); //点の半径
    rr *= rr;

    len_div  = 1.0 / len;
    len_half = len * 0.5;

    //----------------

    for(y = info.rs.y1; y <= info.rs.y2; y++)
    {
        sdx = info.rs.x1;
        sdy = y;

        xx = sdx * dcos - sdy * dsin;
        yy = sdx * dsin + sdy * dcos;

        //

        for(x = info.rs.x1; x <= info.rs.x2; x++)
        {
            //ボックスの中央位置

            cx = ::floor(xx * len_div) * len + len_half;
            cy = ::floor(yy * len_div) * len + len_half;

            //

            if(info.valcheck[0])
            {
                //アンチエイリアス (5x5)

                c = 0;

                for(sy = 0, sdy = yy - cy; sy < 5; sy++, sdy += 0.2)
                {
                    dyy = sdy * sdy;

                    for(sx = 0, sdx = xx - cx; sx < 5; sx++, sdx += 0.2)
                    {
                        if(sdx * sdx + dyy < rr) c++;
                    }
                }

                c = (c << 15) / 25;
            }
            else
            {
                //非アンチ

                sdx = xx - cx;
                sdy = yy - cy;

                if((int)(sdx * sdx + sdy * sdy + 0.5) < (int)(rr + 0.5))
                    c = 0x8000;
                else
                    c = 0;
            }

            //50%以上は白黒反転

            if(density > 50) c = 0x8000 - c;

            col.a = c;

            (pimg->*funcPix)(x, y, col);

            //

            xx += dcos;
            yy += dsin;
        }

        info.progIncSub();
    }

    return TRUE;
}

//! 縁取り（点を打つ）

BOOL draw_edge(FILTERDRAW &info)
{
    int ix,iy,type;
    DRAWPOINTFUNC func;
    DRAWPOINTDAT dat;
    CTileImage *pimgSrc;

    type = info.valcombo[0];

    getDrawPointFunc(&func, info.valcombo[1]);

    dat.pimgSrc  = info.pimgSrc;
    dat.pimgDst  = info.pimgDst;
    dat.radius   = info.valbar[0];
    dat.opacity  = (info.valbar[1] << 15) / 100;
    dat.coltype  = info.valcombo[2];
    dat.masktype = type + 1;

    //描画関数

    CTileImage::m_pinfo->funcColor = (type == 0)? &CTileImage::colfunc_comp_alpha: &CTileImage::colfunc_normal;

    if(info.isPrev())
        CTileImage::m_pinfo->funcDrawPixel = &CTileImage::setPixel_colfunc;

    //プレビュー時はコピー

    if(info.isPrev()) copyRect(info);

    //

    pimgSrc = info.pimgSrc;

    for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
    {
        for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
        {
            if(type)
            {
                //不透明部分の内側

                if(!pimgSrc->isPixelTransparent(ix, iy))
                    continue;

                if(pimgSrc->isPixelTransparent(ix - 1, iy) && pimgSrc->isPixelTransparent(ix + 1, iy) &&
                   pimgSrc->isPixelTransparent(ix, iy - 1) && pimgSrc->isPixelTransparent(ix, iy + 1))
                   continue;
            }
            else
            {
                //不透明部分の外側 (内側判定)

                if(pimgSrc->isPixelTransparent(ix, iy))
                    continue;

                if(!pimgSrc->isPixelTransparent(ix - 1, iy) && !pimgSrc->isPixelTransparent(ix + 1, iy) &&
                   !pimgSrc->isPixelTransparent(ix, iy - 1) && !pimgSrc->isPixelTransparent(ix, iy + 1))
                   continue;
            }

            (*func)(ix, iy, dat);
        }

        info.progIncSub();
    }

    return TRUE;
}

//! ランダム点

BOOL draw_rndpoint(FILTERDRAW &info)
{
    int ix,iy,n,rmin,rrange,omin,orange,omax;
    DWORD amount;
    DRAWPOINTFUNC func;
    DRAWPOINTDAT dat;

    getDrawPointFunc(&func, info.valcombo[0]);

    dat.pimgSrc  = NULL;
    dat.pimgDst  = info.pimgDst;
    dat.coltype  = info.valcombo[1];
    dat.masktype = 0;

    //

    amount = info.valbar[0];
    omax   = (info.valbar[2] << 15) / 100;
    rmin   = (info.valbar[3] << 8) / 100;
    rrange = 256 - rmin;
    omin   = (info.valbar[4] << 8) / 100;
    orange = 256 - omin;

    //描画準備

    if(info.valcombo[0] == 3)
    {
        draw::setBeforeDraw_brush(-1);
        CTileImage::resetWaterCol();
    }
    else
    {
        CTileImage::m_pinfo->funcColor = &CTileImage::colfunc_normal;

        if(info.isPrev())
            CTileImage::m_pinfo->funcDrawPixel = &CTileImage::setPixel_colfunc;
    }

    //プレビュー時はコピー

    if(info.isPrev()) copyRect(info);

    //

    for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
    {
        for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
        {
            if((CTileImage::m_prand->getDWORD() & 8191) >= amount) continue;

            //半径

            n = (CTileImage::m_prand->getRange(0, 256) * rrange >> 8) + rmin;
            dat.radius = info.valbar[1] * n >> 8;

            //不透明度

            n = (CTileImage::m_prand->getRange(0, 256) * orange >> 8) + omin;
            dat.opacity = omax * n >> 8;

            //

            if(dat.radius > 0 && dat.opacity)
                (*func)(ix, iy, dat);
        }

        info.progIncSub();
    }

    return TRUE;
}

//! 雲模様

BOOL draw_cloud(FILTERDRAW &info)
{
    CPerlinNoise perlin;
    int ix,iy,i,n,xx,yy,bAlpha;
    double resmul;
    RGBAFIX15 col,col1,col2;
    CTileImage *pimg;
    TILEIMGPIXFUNC funcPix = CTileImage::m_pinfo->funcDrawPixel;

    if(!perlin.init(CTileImage::m_prand, info.valbar[0] / 400.0, info.valbar[1] * 0.01))
        return FALSE;

    //

    resmul = (info.valbar[2] * 600 + 5000);

    //色

    bAlpha = FALSE;

    switch(info.valcombo[0])
    {
        //黒/白
        case 0:
            col1.zero();
            col2.r = col2.g = col2.b = 0x8000;
            break;
        //描画色/背景色
        case 1:
            col1 = g_draw->colDraw;
            col2 = g_draw->colBack;
            break;
        //黒+アルファ値
        case 2:
            col.r = col.g = col.b = 0;
            bAlpha = TRUE;
            break;
    }

    //

    pimg = info.pimgDst;

    col.a = 0x8000;

    for(iy = info.rs.y1, yy = 0; iy <= info.rs.y2; iy++, yy++)
    {
        for(ix = info.rs.x1, xx = 0; ix <= info.rs.x2; ix++, xx++)
        {
            n = (int)(perlin.noise(xx, yy) * resmul + 0x4000);
            if(n < 0) n = 0; else if(n > 0x8000) n = 0x8000;

            if(bAlpha)
                col.a = n;
            else
            {
                for(i = 0; i < 3; i++)
                    col.c[i] = ((col2.c[i] - col1.c[i]) * n >> 15) + col1.c[i];
            }

            (pimg->*funcPix)(ix, iy, col);
        }

        info.progIncSub();
    }

    return TRUE;
}

};
