/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    フィルタ描画関数 - ぼかし
*/

#include <math.h>

#include "CTileImage.h"

#include "AXMem.h"

#include "filterdraw.h"
#include "filterdrawfunc.h"


namespace filterdraw
{

//! ぼかし

BOOL blur_blur(FILTERDRAW &info)
{
    CTileImage *pimgTmp,*pimg;
    int ix,iy,range,cnt,i;
    double d[4],weight;
    RGBAFIX15 col;
    RECTANDSIZE rsTmp;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15&) = CTileImage::m_pinfo->funcDrawPixel;

    range  = info.valbar[0];
    cnt    = range * 2 + 1;
    weight = 1.0 / cnt;

    info.progSetMax(50);

    //----------- pimgTmp に水平方向へぼかし (range 分余分に)

    rsTmp = info.rs;
    rsTmp.inflate(range);

    pimgTmp = allocTmpImage(info, rsTmp);
    if(!pimgTmp) return FALSE;

    //

    pimg = info.pimgSrc;

    info.progBeginSub(25, rsTmp.h);

    for(iy = rsTmp.y1; iy <= rsTmp.y2; iy++)
    {
        for(ix = rsTmp.x1; ix <= rsTmp.x2; ix++)
        {
            d[0] = d[1] = d[2] = d[3] = 0;

            for(i = -range; i <= range; i++)
                addAdvColor(d, pimg, ix + i, iy, info.bClipping);

            getAdvColor(&col, d, weight);

            if(col.a) pimgTmp->setPixel_create(ix, iy, col);
        }

        info.progIncSub();
    }

    //---------- 垂直方向にぼかし(pimgTmp をソースとする)

    pimg = info.pimgDst;

    info.progBeginSub(25, info.rs.h);

    for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
    {
        for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
        {
            d[0] = d[1] = d[2] = d[3] = 0;

            for(i = -range; i <= range; i++)
                addAdvColor(d, pimgTmp, ix, iy + i, info.bClipping);

            getAdvColor(&col, d, weight);

            (pimg->*funcPix)(ix, iy, col);
        }

        info.progIncSub();
    }

    //

    delete pimgTmp;

    return TRUE;
}

//! ガウスぼかし

BOOL blur_gauss(FILTERDRAW &info)
{
    CTileImage *pimgTmp,*pimg;
    AXMem memTbl;
    int ix,iy,cnt,range,i;
    RECTANDSIZE rsTmp;
    double *pTable,dtmp,dWeight,dcol[4];
    RGBAFIX15 col;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15&) = CTileImage::m_pinfo->funcDrawPixel;

    range = info.valbar[0] * 3;
    cnt   = range * 2 + 1;

    info.progSetMax(50);

    //----------- ガウステーブル

    if(!memTbl.alloc(sizeof(double) * cnt)) return FALSE;

    pTable = (double *)memTbl.getBuf();

    dtmp    = 1.0 / (2 * info.valbar[0] * info.valbar[0]);
    dWeight = 0;

    for(i = 0, ix = -range; i < cnt; i++, ix++)
    {
        pTable[i] = ::exp(-(ix * ix) * dtmp);
        dWeight += pTable[i];
    }

    dWeight = 1.0 / dWeight;

    //----------- pimgTmp に水平方向へぼかし (range 分余分に)

    rsTmp = info.rs;
    rsTmp.inflate(range);

    pimgTmp = allocTmpImage(info, rsTmp);
    if(!pimgTmp) return FALSE;

    //

    pimg = info.pimgSrc;

    info.progBeginSub(25, rsTmp.h);

    for(iy = rsTmp.y1; iy <= rsTmp.y2; iy++)
    {
        for(ix = rsTmp.x1; ix <= rsTmp.x2; ix++)
        {
            dcol[0] = dcol[1] = dcol[2] = dcol[3] = 0;

            for(i = 0; i < cnt; i++)
                addAdvColor(dcol, pTable[i], pimg, ix + i - range, iy, info.bClipping);

            getAdvColor(&col, dcol, dWeight);

            if(col.a) pimgTmp->setPixel_create(ix, iy, col);
        }

        info.progIncSub();
    }

    //---------- 垂直方向にぼかし(pimgTmp をソースとする)

    pimg = info.pimgDst;

    info.progBeginSub(25, info.rs.h);

    for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
    {
        for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
        {
            dcol[0] = dcol[1] = dcol[2] = dcol[3] = 0;

            for(i = 0; i < cnt; i++)
                addAdvColor(dcol, pTable[i], pimgTmp, ix, iy + i - range, info.bClipping);

            getAdvColor(&col, dcol, dWeight);

            (pimg->*funcPix)(ix, iy, col);
        }

        info.progIncSub();
    }

    //

    delete pimgTmp;

    return TRUE;
}

//! モーションブラー
/*
    0.5px 間隔で指定角度+距離分の範囲をガウスぼかし
*/

BOOL blur_motion(FILTERDRAW &info)
{
    AXMem memTbl;
    CTileImage *pimg;
    int ix,iy,i,cnt,range;
    int stx,sty,fx,fy,addx,addy;
    double sind,cosd,dWeight,d,dTmp,dcol[4],*pTable;
    RGBAFIX15 col;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15&) = CTileImage::m_pinfo->funcDrawPixel;

    range = info.valbar[0] * 3;
    cnt   = (range * 2 + 1) * 2;    //0.5px間隔のため2倍

    //----------- ガウステーブル

    if(!memTbl.alloc(sizeof(double) * cnt)) return FALSE;

    pTable = (double *)memTbl.getBuf();

    d       = -range;
    dTmp    = 1.0 / (2 * info.valbar[0] * info.valbar[0]);
    dWeight = 0;

    for(i = 0; i < cnt; i++, d += 0.5)
    {
        pTable[i] = ::exp(-(d * d) * dTmp);

        dWeight += pTable[i];
    }

    dWeight = 1.0 / dWeight;

    //----------- 角度

    d    = -info.valbar[1] * M_PI / 180;
    sind = ::sin(d);
    cosd = ::cos(d);

    stx  = (int)(cosd * -range * (1 << 16));
    sty  = (int)(sind * -range * (1 << 16));
    addx = (int)(cosd * 0.5 * (1 << 16));
    addy = (int)(sind * 0.5 * (1 << 16));

    //-------------

    pimg = info.pimgDst;

    for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
    {
        for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
        {
            fx = stx, fy = sty;

            dcol[0] = dcol[1] = dcol[2] = dcol[3] = 0;

            for(i = 0; i < cnt; i++)
            {
                addAdvColor(dcol, pTable[i], info.pimgSrc, ix + (fx >> 16), iy + (fy >> 16), info.bClipping);

                fx += addx;
                fy += addy;
            }

            getAdvColor(&col, dcol, dWeight);

            (pimg->*funcPix)(ix, iy, col);
        }

        info.progIncSub();
    }

    return TRUE;
}

//! 放射状ぼかし

BOOL blur_radial(FILTERDRAW &info)
{
    CTileImage *pimgSrc,*pimgDst;
    int ix,iy,i,len,mode;
    double factor,rr,dx,dy,rd,dsin,dcos,xx,yy,dcol[4];
    RGBAFIX15 col;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15&) = CTileImage::m_pinfo->funcDrawPixel;

    factor  = info.valbar[0] * 0.01;
    mode    = (info.bClipping)? 1: 0;
    pimgSrc = info.pimgSrc;
    pimgDst = info.pimgDst;

    //

    for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
    {
        dy = iy - info.tmpy;

        for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
        {
            dx = ix - info.tmpx;

            //中央位置からの距離に強さを適用して処理長さ計算

            rr = ::sqrt(dx * dx + dy * dy);

            len = (int)(rr * factor);

            if(len < 1)
            {
                pimgSrc->getPixel(&col, ix, iy);
                (pimgDst->*funcPix)(ix, iy, col);
                continue;
            }

            //角度

            rd = ::atan2(dy, dx);
            dsin = ::sin(rd);
            dcos = ::cos(rd);

            //角度と長さの範囲から色取得

            dcol[0] = dcol[1] = dcol[2] = dcol[3] = 0;

            xx = (rr - len) * dcos + info.tmpx;
            yy = (rr - len) * dsin + info.tmpy;

            for(i = 0; i <= len; i++, xx += dcos, yy += dsin)
            {
                getLinerCol(&col, pimgSrc, xx, yy, mode);

                addAdvColor(dcol, col);
            }

            //

            getAdvColor(&col, dcol, 1.0 / (len + 1));

            (pimgDst->*funcPix)(ix, iy, col);
        }

        info.progIncSub();
    }

    return TRUE;
}

//! レンズぼかし

BOOL blur_lens(FILTERDRAW &info)
{
    CTileImage *pimgSrc,*pimgDst;
    int ix,iy,xx,yy,radius,size,cnt;
    double dcol[4],weight,brBase,brRes,a,*pTbl;
    AXMem mem,memTbl;
    LPBYTE pbuf;
    BYTE f;
    RGBAFIX15 col;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15&) = CTileImage::m_pinfo->funcDrawPixel;

    radius = info.valbar[0];
    brBase = 1.010 + info.valbar[1] * 0.001;
    brRes  = 128.0 / ::log(brBase);

    //テーブル

    if(!memTbl.alloc(sizeof(double) * 257)) return FALSE;

    pTbl = (double *)memTbl.getBuf();

    for(ix = 0; ix <= 256; ix++)
        pTbl[ix] = ::pow(brBase, ix);

    //円形データ (1bit)

    size = radius * 2 + 1;

    if(!mem.allocClear((size * size + 7) >> 3)) return FALSE;

    pbuf = mem;
    f    = 0x80;
    cnt  = 0;

    for(iy = 0; iy < size; iy++)
    {
        yy = iy - radius;
        yy *= yy;

        for(ix = 0, xx = -radius; ix < size; ix++, xx++)
        {
            if(xx * xx + yy <= radius)
            {
                *pbuf |= f;
                cnt++;
            }

            f >>= 1;
            if(f == 0) { f = 0x80; pbuf++; }
        }
    }

    weight = 1.0 / cnt;

    //------------

    pimgSrc = info.pimgSrc;
    pimgDst = info.pimgDst;

    for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
    {
        for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
        {
            //円形の平均色

            dcol[0] = dcol[1] = dcol[2] = dcol[3] = 0;

            pbuf = mem;
            f    = 0x80;

            for(yy = -radius; yy <= radius; yy++)
            {
                for(xx = -radius; xx <= radius; xx++)
                {
                    if(*pbuf & f)
                    {
                        pimgSrc->getPixel(&col, ix + xx, iy + yy, info.bClipping);

                        a = (double)col.a / 0x8000;

                        dcol[0] += pTbl[col.r >> 7] * a;
                        dcol[1] += pTbl[col.g >> 7] * a;
                        dcol[2] += pTbl[col.b >> 7] * a;
                        dcol[3] += a;
                    }

                    f >>= 1;
                    if(f == 0) { f = 0x80; pbuf++; }
                }
            }

            //

            col.a = (WORD)(dcol[3] * weight * 0x8000 + 0.5);

            if(col.a == 0)
                col.zero();
            else
            {
                a = 1.0 / dcol[3];

                for(xx = 0; xx < 3; xx++)
                {
                    yy = (int)(::log(dcol[xx] * a) * brRes);
                    if(yy < 0) yy = 0; else if(yy > 0x8000) yy = 0x8000;

                    col.c[xx] = yy;
                }
            }

            //

            (pimgDst->*funcPix)(ix, iy, col);
        }

        info.progIncSub();
    }

    return TRUE;
}

};
