/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CUndoDatBuf - アンドゥデータバッファクラス
*/


#include <stdlib.h>
#include <string.h>

#include "CUndoDatBuf.h"

#include "CConfig.h"
#include "CApp.h"
#include "global.h"

#include "AXFile.h"
#include "AXString.h"
#include "AXMem.h"
#include "AXUtilStr.h"
#include "AXUtilFile.h"


//--------------------

AXFile  g_fileWrite,
        g_fileRead;
LPBYTE  g_pbufWrite,
        g_pbufRead;
DWORD   g_dwWriteSize;
int     g_nFileNoCnt = 0;

#define FILENO_BUF  -1

//--------------------

DWORD CUndoDatBuf::m_dwAllBufSize = 0;


CUndoDatBuf::CUndoDatBuf()
{
    m_dwSize  = 0;
    m_pBuf    = NULL;
    m_nFileNo = FILENO_BUF;
}

CUndoDatBuf::~CUndoDatBuf()
{
    free();
}

//! 解放

void CUndoDatBuf::free()
{
    if(m_nFileNo == FILENO_BUF)
    {
        AXFree((void **)&m_pBuf);

        m_dwAllBufSize -= m_dwSize;
    }
    else
    {
        //ファイル削除

        AXString str;

        _getFileName(&str, m_nFileNo);
        AXDeleteFile(str);

        m_nFileNo = FILENO_BUF;
    }
}

//! データをコピー

BOOL CUndoDatBuf::copy(CUndoDatBuf &src)
{
    if(!alloc(src.m_dwSize))
        return FALSE;

    //

    if(isBuf() && src.isBuf())
        ::memcpy(m_pBuf, src.m_pBuf, m_dwSize);
    else
    {
        AXMem mem;

        if(!mem.alloc(m_dwSize)) return FALSE;

        if(!src.openRead()) return FALSE;
        if(!openWrite()) { src.closeRead(); return FALSE; }

        src.read(mem, m_dwSize);
        write(mem, m_dwSize);

        closeWrite();
        src.closeRead();
    }

    return TRUE;
}


//=============================
// 確保
//=============================


//! バッファ/ファイル確保
/*!
    @param size 0 以下でファイルへ
*/

BOOL CUndoDatBuf::alloc(int size)
{
    BOOL ret;

    m_dwSize = (size <= 0)? 0: size;

    if(size <= 0 || (m_dwAllBufSize + size > g_conf->uUndoMaxBuf))
        //ファイル
        ret = _allocFile();
    else
    {
        //バッファ優先

        ret = _allocBuf(size);

        if(ret)
            m_dwAllBufSize += size;
        else
            ret = _allocFile();
    }

    return ret;
}

//! ファイル確保

BOOL CUndoDatBuf::_allocFile()
{
    if(CAPP->getTmpDir().isEmpty()) return FALSE;

    m_nFileNo = g_nFileNoCnt;

    g_nFileNoCnt = (g_nFileNoCnt + 1) & 0xffff;

    return TRUE;
}

//! メモリ確保

BOOL CUndoDatBuf::_allocBuf(DWORD size)
{
    m_pBuf = (LPBYTE)::malloc(size);

    return (m_pBuf != NULL);
}

//! テンポラリファイル名取得

void CUndoDatBuf::_getFileName(AXString *pstr,int no)
{
    char m[16];

    AXValToHexStrDig(m, m_nFileNo, 4);

    *pstr = CAPP->getTmpDir();
    pstr->path_add(m);
}


//=============================
// データ書き込み
//=============================


//! 書き込みオープン

BOOL CUndoDatBuf::openWrite()
{
    if(m_nFileNo == FILENO_BUF)
    {
        //バッファ

        if(!m_pBuf) return FALSE;

        g_pbufWrite   = m_pBuf;
        g_dwWriteSize = 0;
    }
    else
    {
        //ファイル

        AXString str;

        _getFileName(&str, m_nFileNo);

        if(!g_fileWrite.openWrite(str)) return FALSE;
    }

    return TRUE;
}

//! 書き込み

void CUndoDatBuf::write(const void *pBuf,DWORD size)
{
    if(m_nFileNo == FILENO_BUF)
    {
        ::memcpy(g_pbufWrite, pBuf, size);

        g_pbufWrite   += size;
        g_dwWriteSize += size;
    }
    else
        g_fileWrite.write(pBuf, size);
}

//! 書き込み閉じる

void CUndoDatBuf::closeWrite()
{
    if(m_nFileNo != FILENO_BUF)
        g_fileWrite.close();
}


//=============================
// データ読み込み
//=============================


//! 読み込みオープン

BOOL CUndoDatBuf::openRead()
{
    if(m_nFileNo == FILENO_BUF)
    {
        if(!m_pBuf) return FALSE;

        g_pbufRead = m_pBuf;
    }
    else
    {
        AXString str;

        _getFileName(&str, m_nFileNo);

        if(!g_fileRead.openRead(str)) return FALSE;
    }

    return TRUE;
}

//! 読み込み

void CUndoDatBuf::read(void *pBuf,DWORD size)
{
    if(m_nFileNo == FILENO_BUF)
    {
        ::memcpy(pBuf, g_pbufRead, size);
        g_pbufRead += size;
    }
    else
        g_fileRead.read(pBuf, size);
}

//! 読みクローズ

void CUndoDatBuf::closeRead()
{
    if(m_nFileNo != FILENO_BUF)
        g_fileRead.close();
}

//! 読み込みシーク

void CUndoDatBuf::seekRead(int size)
{
    if(m_nFileNo == FILENO_BUF)
        g_pbufRead += size;
    else
        g_fileRead.seekCur(size);
}
