/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CUndoDat - アンドゥ個々のデータ、実行処理
*/

#include <string.h>

#include "CUndoDat.h"

#include "CUndo.h"
#include "CLayerList.h"
#include "CLayerItem.h"
#include "CTileImage.h"

#include "drawdat.h"
#include "struct.h"
#include "global.h"

#include "draw_calc.h"
#include "draw_main.h"


//! 作成

CUndoDat::CUndoDat(int cmd)
{
    m_nCommand = cmd;
}

//! 実行
/*
    m_rcsUpdate.x : 更新イメージ範囲
        x が -1 の場合は更新はしない（キャンバス範囲外の場合）
        UPDATE_RESIZE の場合、w,h は新しいイメージサイズ
*/

BOOL CUndoDat::run(AXUndo *pUndo,BOOL bUndo)
{
    CUndo *pundo = (CUndo *)pUndo;
    CLayerItem *p;

    pundo->m_nUpdateType = CUndo::UPDATE_ALLANDLAYER;
    pundo->m_rcsUpdate.x = -1;

    //

    switch(m_nCommand)
    {
        //アンドゥイメージ
        case CMD_UNDOIMG:
        case CMD_UNDOIMG_SMALL:
            //x == -1 で範囲外

            pundo->m_nUpdateType = CUndo::UPDATE_RECT;
            pundo->m_rcsUpdate.x = m_nVal[1];
            pundo->m_rcsUpdate.y = m_nVal[2];
            pundo->m_rcsUpdate.w = m_nVal[3];
            pundo->m_rcsUpdate.h = m_nVal[4];

            if(m_nCommand == CMD_UNDOIMG)
                return _run_undoImage(!bUndo);
            else
                return restoreImageSmall();

        //レイヤ追加
        case CMD_LAYER_NEW:
            return _run_layerNew(pundo, bUndo);
        //レイヤ複製
        case CMD_LAYER_COPY:
            return _run_layerCopy(pundo, bUndo);
        //レイヤ削除
        case CMD_LAYER_DEL:
            return _run_layerDel(pundo, bUndo);
        //レイヤクリア
        case CMD_LAYER_CLEAR:
            return _run_layerClear(pundo, bUndo);
        //レイヤカラータイプ
        case CMD_LAYER_COLTYPE:
            return _run_layerColType(pundo, bUndo);
        //レイヤ下に移す/結合
        case CMD_LAYER_COMBINE:
            return _run_layerCombine(pundo, bUndo);
        //複数レイヤ結合
        case CMD_LAYER_COMBINEMUL:
            return _run_layerCombineMul(pundo, bUndo);
        //レイヤオフセット移動
        case CMD_LAYER_OFFSET:
            return _run_layerOffset(pundo);
        //レイヤ移動
        case CMD_LAYER_MOVE:
            _run_layerMove(pundo);
            break;
        //レイヤ左右/上下反転
        case CMD_LAYER_HVREV:
            _run_layerHVRev(pundo);
            break;
        //レイヤ90度回転
        case CMD_LAYER_ROTATE90:
            _run_layerRotate90(pundo);
            break;
        //レイヤフラグ情報
        case CMD_LAYER_FLAGS:
            p = _getLayer(m_nVal[0]);
            if(p) p->m_dwFlags ^= m_nVal[1];

            pundo->m_nUpdateType = CUndo::UPDATE_LAYERLIST;
            break;
        //キャンバスサイズ変更
        case CMD_CANVAS_RESIZE:
            g_draw->player->resizeCanvas(m_nVal[0], m_nVal[1]);

            pundo->m_nUpdateType = CUndo::UPDATE_RESIZE;
            pundo->m_rcsUpdate.w = m_nVal[2];
            pundo->m_rcsUpdate.h = m_nVal[3];
            break;
        //キャンバス拡大縮小
        case CMD_CANVAS_SCALE:
            return _run_canvasScale(pundo, bUndo);
    }

    return TRUE;
}

//! 逆のデータセット

BOOL CUndoDat::setReverseDat(AXUndo *pUndo,AXUndoDat *pSrc,BOOL bUndo)
{
    CUndoDat *pSrcDat = (CUndoDat *)pSrc;

    //ソースの情報コピー

    m_nCommand = pSrcDat->m_nCommand;
    ::memcpy(m_nVal, pSrcDat->m_nVal, sizeof(int) * VAL_CNT);

    //

    switch(m_nCommand)
    {
        //アンドゥイメージ
        case CMD_UNDOIMG:
            return writeUndoImageRev(pSrcDat, bUndo);
        //アンドゥイメージ・少範囲
        case CMD_UNDOIMG_SMALL:
            return writeImageSmall(_getLayerImg(m_nVal[0]), FALSE);
        //レイヤ追加
        case CMD_LAYER_NEW:
            if(bUndo)
                return writeLayerSingle(_getLayer(m_nVal[0], m_nVal[1]));
            else
                return TRUE;
        //レイヤ削除
        case CMD_LAYER_DEL:
            if(bUndo)
                return TRUE;
            else
                return writeLayerFolder(_getLayer(m_nVal[0]));
        //レイヤクリア
        case CMD_LAYER_CLEAR:
            if(bUndo)
                return TRUE;
            else
                return writeLayerOnlyImage(_getLayer(m_nVal[0]));
        //レイヤカラータイプ
        case CMD_LAYER_COLTYPE:
            if(bUndo)
                return TRUE;
            else
                return writeLayerOnlyImage(_getLayer(m_nVal[0]));
        //レイヤ下に移す/結合
        case CMD_LAYER_COMBINE:
            if(m_nVal[0])
                return writeLayerTwoImage(_getLayer(m_nVal[1], m_nVal[2]));
            else
                return writeLayerCombine(!bUndo);
        //複数レイヤ結合
        case CMD_LAYER_COMBINEMUL:
            if(m_nVal[0] == 0)
            {
                //フォルダ
                if(bUndo)
                    return writeLayerSingle(_getLayer(m_nVal[1], m_nVal[2]));
                else
                    return writeLayerFolder(_getLayer(m_nVal[1], m_nVal[2]));
            }
            else
            {
                //すべて
                if(bUndo)
                    return writeLayerSingle(g_draw->player->getTopItem());
                else
                    return writeLayerAll();
            }
            break;
        //レイヤオフセット移動
        case CMD_LAYER_OFFSET:
            m_nVal[0] = -m_nVal[0];
            m_nVal[1] = -m_nVal[1];

            if(m_nVal[2] > 1)
                m_dat.copy(pSrcDat->m_dat);
            break;
        //レイヤ移動
        case CMD_LAYER_MOVE:
            m_nVal[0] = pSrcDat->m_nVal[2];
            m_nVal[1] = pSrcDat->m_nVal[3];
            m_nVal[2] = pSrcDat->m_nVal[0];
            m_nVal[3] = pSrcDat->m_nVal[1];
            break;
        //レイヤ90度回転
        case CMD_LAYER_ROTATE90:
            m_nVal[1] = !m_nVal[1];
            break;
        //キャンバスサイズ変更
        case CMD_CANVAS_RESIZE:
            m_nVal[0] = -m_nVal[0];
            m_nVal[1] = -m_nVal[1];
            m_nVal[2] = g_draw->nImgW;
            m_nVal[3] = g_draw->nImgH;
            break;
        //キャンバス拡大縮小
        case CMD_CANVAS_SCALE:
            m_nVal[0] = g_draw->nImgW;
            m_nVal[1] = g_draw->nImgH;
            m_nVal[2] = g_draw->nImgDPI;

            if(bUndo)
                return writeLayerSingle(g_draw->player->getTopItem());
            else
                return writeLayerAll();
    }

    return TRUE;
}

//------------

//! カレントレイヤ番号セット

void CUndoDat::setval_curlayer(int valno)
{
    m_nVal[valno] = g_draw->player->getLayerItemNo(g_draw->pcurlayer);
}

//! 指定位置にレイヤ番号セット

void CUndoDat::setval_layerno(int valno,CLayerItem *pItem)
{
    m_nVal[valno] = g_draw->player->getLayerItemNo(pItem);
}

//! レイヤの親番号と子番号セット(2つ)

void CUndoDat::setval_layerno_parent(int valno,CLayerItem *pItem)
{
    g_draw->player->getLayerItemNo_parent(pItem, m_nVal + valno);
}

//------------

//! 番号からレイヤ取得

CLayerItem *CUndoDat::_getLayer(int no)
{
    return g_draw->player->getLayerItemFromNo(no);
}

//! 親番号と子番号からレイヤ取得

CLayerItem *CUndoDat::_getLayer(int pno,int no)
{
    CLayerItem *p[2];

    g_draw->player->getLayerItemFromNo_parent(p, pno, no);

    return p[1];
}

//! 番号からレイヤイメージ取得

CTileImage *CUndoDat::_getLayerImg(int no)
{
    CLayerItem *p = g_draw->player->getLayerItemFromNo(no);

    if(p)
        return p->m_pimg;
    else
        return NULL;
}

//! FLAGRECT からアンドゥの更新範囲セット

void CUndoDat::_setUpdateRect(CUndo *pundo,const FLAGRECT &rcf)
{
    if(!draw::getImgRect(&pundo->m_rcsUpdate, rcf))
        pundo->m_rcsUpdate.x = -1;
}

//! 指定レイヤのイメージ範囲からアンドゥの更新範囲セット

void CUndoDat::_setUpdateRect(CUndo *pundo,CLayerItem *pItem)
{
    FLAGRECT rcf;

    pItem->getVisibleImgRect(&rcf);
    _setUpdateRect(pundo, rcf);
}

//! レイヤ削除
/*
    削除するレイヤがカレントレイヤまたはカレントレイヤフォルダの子の場合、
    カレントレイヤを変更する。
*/

void CUndoDat::_deleteLayer(CLayerItem *p)
{
    CLayerItem *pCur;

    if(!p) return;

    //削除後のカレントレイヤ

    pCur = g_draw->pcurlayer;

    if(p == pCur || pCur->isChild(p))
    {
        pCur = (CLayerItem *)p->nextTreeItemPass();
        if(!pCur) pCur = (CLayerItem *)p->prevTreeItem();
    }

    //削除

    g_draw->player->deleteLayer(p);

    //カレントレイヤ

    g_draw->pcurlayer = pCur;
}



//================================
// 実行処理
//================================


//! レイヤ新規追加

BOOL CUndoDat::_run_layerNew(CUndo *pundo,BOOL bUndo)
{
    FLAGRECT rcf;

    if(bUndo)
        //削除
        return _run_layerDelete(pundo, _getLayer(m_nVal[0], m_nVal[1]));
    else
    {
        //レイヤ復元

        if(!restoreLayerSingle(m_nVal[0], m_nVal[1], &rcf))
            return FALSE;

        pundo->m_nUpdateType = CUndo::UPDATE_RECTANDLAYER;
        _setUpdateRect(pundo, rcf);

        return TRUE;
    }
}

//! レイヤ複製

BOOL CUndoDat::_run_layerCopy(CUndo *pundo,BOOL bUndo)
{
    CLayerItem *p,*pSrc;

    pSrc = _getLayer(m_nVal[0]);
    if(!pSrc) return FALSE;

    if(bUndo)
        //削除
        return _run_layerDelete(pundo, pSrc);
    else
    {
        //複製

        p = g_draw->player->copyLayer(pSrc);
        if(!p) return FALSE;

        pundo->m_nUpdateType = CUndo::UPDATE_RECTANDLAYER;
        _setUpdateRect(pundo, p);
    }

    return TRUE;
}

//! レイヤ削除

BOOL CUndoDat::_run_layerDel(CUndo *pundo,BOOL bUndo)
{
    if(!bUndo)
        return _run_layerDelete(pundo, _getLayer(m_nVal[0]));
    else
    {
        FLAGRECT rcf;

        if(!restoreLayerMul(&rcf)) return FALSE;

        pundo->m_nUpdateType = CUndo::UPDATE_RECTANDLAYER;
        _setUpdateRect(pundo, rcf);
    }

    return TRUE;
}

//! レイヤクリア

BOOL CUndoDat::_run_layerClear(CUndo *pundo,BOOL bUndo)
{
    CLayerItem *p;

    p = _getLayer(m_nVal[0]);
    if(!p) return FALSE;

    pundo->m_nUpdateType = CUndo::UPDATE_RECT;

    if(bUndo)
    {
        //イメージ復元

        if(!restoreLayerOnlyImage(p)) return FALSE;

        _setUpdateRect(pundo, p);
    }
    else
    {
        //クリア

        _setUpdateRect(pundo, p);

        if(!p->m_pimg->create(g_draw->nImgW, g_draw->nImgH))
            return FALSE;
    }

    return TRUE;
}

//! レイヤカラータイプ変更

BOOL CUndoDat::_run_layerColType(CUndo *pundo,BOOL bUndo)
{
    CLayerItem *p;
    CTileImage *pimg;

    p = _getLayer(m_nVal[0]);
    if(!p) return FALSE;

    if(bUndo)
    {
        //イメージ復元

        p->m_nColType = m_nVal[1];

        if(!restoreLayerOnlyImage(p)) return FALSE;
    }
    else
    {
        //再変換

        pimg = draw::allocTileImage(m_nVal[2]);

        if(!pimg->createConvert(*(p->m_pimg), m_nVal[3], NULL))
        {
            delete pimg;
            return FALSE;
        }

        p->replaceImg(pimg);
    }

    pundo->m_nUpdateType = CUndo::UPDATE_RECTANDLAYER;
    _setUpdateRect(pundo, p);

    return TRUE;
}

//! レイヤ下に移す/結合

BOOL CUndoDat::_run_layerCombine(CUndo *pundo,BOOL bUndo)
{
    CLayerItem *p;
    FLAGRECT rcf;

    if(m_nVal[0])
    {
        //下に移す

        p = _getLayer(m_nVal[1], m_nVal[2]);
        if(!p) return FALSE;

        pundo->m_nUpdateType = CUndo::UPDATE_RECT;

        if(!bUndo) _setUpdateRect(pundo, p);

        if(!restoreLayerTwoImage(p)) return FALSE;

        if(bUndo) _setUpdateRect(pundo, p);
    }
    else
    {
        //結合

        if(!restoreLayerCombine(bUndo, &rcf)) return FALSE;

        pundo->m_nUpdateType = CUndo::UPDATE_RECTANDLAYER;
        _setUpdateRect(pundo, rcf);
    }

    return TRUE;
}

//! 複数レイヤ結合

BOOL CUndoDat::_run_layerCombineMul(CUndo *pundo,BOOL bUndo)
{
    CLayerItem *p;
    FLAGRECT rcf,rcf2;

    if(m_nVal[0] == 0)
    {
        //--------- フォルダ

        p = _getLayer(m_nVal[1], m_nVal[2]);
        if(!p) return FALSE;

        if(bUndo)
        {
            //フォルダレイヤ復元 -> 結合後レイヤ削除

            if(!restoreLayerMul(&rcf)) return FALSE;

            _deleteLayer(p);
        }
        else
        {
            //結合後レイヤ復元 -> 結合元フォルダ削除

            if(!restoreLayerSingle(m_nVal[1], m_nVal[2], &rcf))
                return FALSE;

            p->getVisibleImgRect(&rcf2);
            rcf.combine(rcf2);

            _deleteLayer(p);
        }

        //更新

        pundo->m_nUpdateType = CUndo::UPDATE_RECTANDLAYER;
        _setUpdateRect(pundo, rcf);
    }
    else
    {
        //---------- すべてのレイヤ

        if(!_run_sub_combineAll(pundo, bUndo))
            return FALSE;
    }

    return TRUE;
}

//! 単体レイヤ削除処理

BOOL CUndoDat::_run_layerDelete(CUndo *pundo,CLayerItem *p)
{
    if(!p) return FALSE;

    pundo->m_nUpdateType = CUndo::UPDATE_RECTANDLAYER;
    _setUpdateRect(pundo, p);

    _deleteLayer(p);

    return TRUE;
}

//! レイヤ移動

void CUndoDat::_run_layerMove(CUndo *pundo)
{
    CLayerItem *pSrc[2],*pDst[2];
    FLAGRECT rcf;

    //移動元

    g_draw->player->getLayerItemFromNo_parent(pSrc, m_nVal[0], m_nVal[1]);
    if(!pSrc[1]) return;

    //移動先（移動元のリンクを外した状態でアイテム取得）

    g_draw->player->remove(pSrc[1]);

    g_draw->player->getLayerItemFromNo_parent(pDst, m_nVal[2], m_nVal[3]);

    //移動（リンクを再セット）

    if(pDst[1])
        g_draw->player->insert(pDst[1], pSrc[1]);
    else
        g_draw->player->add(pDst[0], pSrc[1]);

    //更新

    pundo->m_nUpdateType = CUndo::UPDATE_RECTANDLAYER;

    pSrc[1]->getVisibleImgRect(&rcf);
    _setUpdateRect(pundo, rcf);
}

//! レイヤ上下左右反転
/*
    ロックレイヤが実行時と同じ状況でなければならないことに注意。
*/

void CUndoDat::_run_layerHVRev(CUndo *pundo)
{
    CLayerItem *p;
    FLAGRECT rcf;

    p = _getLayer(m_nVal[0]);
    if(!p) return;

    p->reverseHorzVert(m_nVal[1], &rcf);

    //更新範囲

    pundo->m_nUpdateType = CUndo::UPDATE_RECT;
    _setUpdateRect(pundo, rcf);
}

//! レイヤ90度回転

void CUndoDat::_run_layerRotate90(CUndo *pundo)
{
    CLayerItem *p;
    FLAGRECT rcf;

    p = _getLayer(m_nVal[0]);
    if(!p) return;

    p->rotate90(m_nVal[1], &rcf);

    //更新範囲

    pundo->m_nUpdateType = CUndo::UPDATE_RECT;
    _setUpdateRect(pundo, rcf);
}

//! レイヤオフセット移動

BOOL CUndoDat::_run_layerOffset(CUndo *pundo)
{
    CLayerItem *p;
    short no;
    int i;
    FLAGRECT rcf,rcf1,rcf2;

    if(m_nVal[2] == 1)
    {
        //------ レイヤ数が一つ

        p = _getLayer(m_nVal[3]);

        if(p)
        {
            p->m_pimg->getExistImgRectPx(&rcf1);

            p->m_pimg->moveOffset(m_nVal[0], m_nVal[1]);

            rcf.combineMove(rcf1, m_nVal[0], m_nVal[1]);
        }
    }
    else
    {
        //------ 複数レイヤ

        rcf.clear();

        if(!m_dat.openRead()) return FALSE;

        for(i = m_nVal[2]; i > 0; i--)
        {
            m_dat.read(&no, 2);

            p = _getLayer(no);
            if(p)
            {
                p->m_pimg->getExistImgRectPx(&rcf1);

                p->m_pimg->moveOffset(m_nVal[0], m_nVal[1]);

                rcf2.combineMove(rcf1, m_nVal[0], m_nVal[1]);
                rcf.combine(rcf2);
            }
        }

        m_dat.closeRead();
    }

    //

    pundo->m_nUpdateType = CUndo::UPDATE_RECT;
    _setUpdateRect(pundo, rcf);

    return TRUE;
}

//! キャンバス拡大縮小

BOOL CUndoDat::_run_canvasScale(CUndo *pundo,BOOL bUndo)
{
    if(!_run_sub_combineAll(pundo, bUndo)) return FALSE;

    //

    g_draw->nImgDPI = m_nVal[2];

    pundo->m_nUpdateType = CUndo::UPDATE_RESIZE;
    pundo->m_rcsUpdate.w = m_nVal[0];
    pundo->m_rcsUpdate.h = m_nVal[1];

    return TRUE;
}

//! 全レイヤ結合処理

BOOL CUndoDat::_run_sub_combineAll(CUndo *pundo,BOOL bUndo)
{
    FLAGRECT rcf;
    CLayerItem *p;

    if(bUndo)
    {
        //全レイヤ復元 -> 結合後レイヤ削除

        p = g_draw->player->getTopItem();

        if(!restoreLayerMul(&rcf)) return FALSE;

        _deleteLayer(p);
    }
    else
    {
        //すべて削除 -> 結合後レイヤ復元

        g_draw->player->clearItem();

        if(!restoreLayerSingle(-1, 0, &rcf))
            return FALSE;

        g_draw->pcurlayer = g_draw->player->getTopItem();
    }

    pundo->m_nUpdateType = CUndo::UPDATE_ALLANDLAYER;

    return TRUE;
}


