/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CPolygonPaint - 多角形描画用の位置計算クラス
*/


#include "CPolygonPaint.h"


#define SUBPIX_X        8
#define SUBPIX_X_SHIFT  3
#define SUBPIX_Y        8


CPolygonPaint::CPolygonPaint()
{
    m_nPosCnt = 0;
}

//! 解放

void CPolygonPaint::free()
{
    m_memPos.free();
    m_memEdge.free();
    m_memXbuf.free();
}

//! 位置バッファ確保

BOOL CPolygonPaint::alloc(int cnt)
{
    free();

    //確保

    if(!m_memPos.alloc(cnt * sizeof(POSDAT), sizeof(POSDAT) * 50))
        return FALSE;

    m_nPosCnt = 0;

    return TRUE;
}

//! 点追加

BOOL CPolygonPaint::add(double x,double y)
{
    POSDAT *p,dat;
    int nx,ny;

    //前の位置と同じなら追加しない

    if(m_nPosCnt)
    {
        p = (POSDAT *)m_memPos.getBuf() + m_nPosCnt - 1;

        if(p->x == x && p->y == y) return TRUE;
    }

    //追加

    dat.x = x;
    dat.y = y;

    if(!m_memPos.addDat(&dat, sizeof(POSDAT)))
        return FALSE;

    m_nPosCnt++;

    //最小・最大XY

    nx = (int)x, ny = (int)y;

    if(m_nPosCnt == 1)
    {
        m_nMinX = m_nMaxX = nx;
        m_nMinY = m_nMaxY = ny;
    }
    else
    {
        if(nx < m_nMinX) m_nMinX = nx;
        if(nx > m_nMaxX) m_nMaxX = nx;

        if(ny < m_nMinY) m_nMinY = ny;
        if(ny > m_nMaxY) m_nMaxY = ny;
    }

    return TRUE;
}

//! 点追加終了

void CPolygonPaint::endPos()
{
    POSDAT *p;

    //始点を追加

    if(m_nPosCnt)
    {
        p = (POSDAT *)m_memPos.getBuf();

        add(p->x, p->y);
    }

    //X範囲拡張

    m_nMinX--, m_nMaxX++;
}


//================================
// 描画時用
//================================


//! 描画前、バッファ確保

BOOL CPolygonPaint::beginDraw(BOOL bAntiAlias)
{
    m_nWidth = m_nMaxX - m_nMinX + 1;

    //交点バッファ

    if(!m_memEdge.alloc(sizeof(EDGEDAT) * 20, sizeof(EDGEDAT) * 20))
        return FALSE;

    m_nEdgeCnt = 0;

    //Xバッファ

    if(bAntiAlias)
    {
        if(!m_memXbuf.alloc(m_nWidth)) return FALSE;
    }

    return TRUE;
}

//! 指定Y位置におけるX範囲の色のある数セット（アンチエイリアス用）
/*
    値は、最大で [Xsubpixel x Ysubpixl]。
*/

BOOL CPolygonPaint::setXbuf_AA(int ypos)
{
    EDGEDAT *pe;
    LPBYTE pxbuf;
    int i,k,cnt,x1,x2;
    double y;

    pxbuf = m_memXbuf;

    AXMemZero(pxbuf, m_nWidth);

    y = ypos;

    for(k = 0; k < SUBPIX_Y; k++, y += 1.0 / SUBPIX_Y)
    {
        //交点リスト作成

        if(!_findIntersection(y)) return FALSE;

        //交点間

        pe  = (EDGEDAT *)m_memEdge.getBuf();
        cnt = 0;

        for(i = m_nEdgeCnt - 1; i > 0; i--, pe++)
        {
            x1 = pe[0].x;
            x2 = pe[1].x;

            //カウント

            if(pe->bUpper) cnt++; else cnt--;

            //セット

            if(cnt || x1 == x2)
            {
                for(; x1 <= x2; x1++)
                    pxbuf[x1 >> SUBPIX_X_SHIFT]++;
            }
        }
    }

    return TRUE;
}

//! 交点検索（非アンチエイリアス用）

BOOL CPolygonPaint::findIntersectionInt(int y)
{
    POSDAT *p,*p1,*p2;
    int i,bUpper,x;
    double yy;

    //交点数クリア

    m_memEdge.clearNowSize();

    m_nEdgeCnt = 0;

    m_nNowPos     = 0;
    m_nNowEdgeCnt = 0;

    //各辺から交点検索

    yy = y + 0.5;

    p = (POSDAT *)m_memPos.getBuf();

    for(i = m_nPosCnt - 1; i > 0; i--, p++)
    {
        bUpper = (p[0].y < p[1].y);

        //

        if(bUpper)
            p1 = p, p2 = p + 1;
        else
            p1 = p + 1, p2 = p;

        //辺の範囲外

        if(p1->y == p2->y || yy < p1->y || yy > p2->y) continue;

        //交点X位置計算

        x = (int)(p1->x + (double)(yy - p1->y) / (p2->y - p1->y) * (p2->x - p1->x));

        if(!_addEdge(x, bUpper)) return FALSE;
    }

    _sortEdge();

    return TRUE;
}

//! 非アンチエイリアス、次の描画する線取得

BOOL CPolygonPaint::getNextLine(int *pX1,int *pX2)
{
    EDGEDAT *p;

    if(m_nNowPos >= m_nEdgeCnt - 1) return FALSE;

    p = (EDGEDAT *)m_memEdge.getBuf() + m_nNowPos;

    for(; m_nNowPos < m_nEdgeCnt - 1; m_nNowPos++, p++)
    {
        //カウント

        if(p->bUpper) m_nNowEdgeCnt++; else m_nNowEdgeCnt--;

        //

        if(m_nNowEdgeCnt || p[0].x == p[1].x)
        {
            *pX1 = p[0].x;
            *pX2 = p[1].x;

            m_nNowPos++;

            return TRUE;
        }
    }

    return FALSE;
}

//! 指定Y位置とのX交点リスト作成（アンチエイリアス用）

BOOL CPolygonPaint::_findIntersection(double y)
{
    POSDAT *p,*p1,*p2;
    int i,bUpper,x;
    double y1,y2;

    //交点数クリア

    m_memEdge.clearNowSize();

    m_nEdgeCnt = 0;

    //各辺から交点検索

    p = (POSDAT *)m_memPos.getBuf();

    for(i = m_nPosCnt - 1; i > 0; i--, p++)
    {
        y1 = p[0].y;
        y2 = p[1].y;

        //横線

        if(y1 == y2) continue;

        //辺の範囲外

        if(!((y >= y1 && y <= y2) || (y >= y2 && y <= y1))) continue;

        //入れ替え

        bUpper = (y1 < y2);

        if(y1 < y2)
            p1 = p, p2 = p + 1;
        else
            p1 = p + 1, p2 = p;

        //交点X位置計算 (m_nMinX が基準 & サブピクセル位置）

        x = (int)((p1->x + (y - p1->y) / (p2->y - p1->y) * (p2->x - p1->x) - m_nMinX) * SUBPIX_X);

        if(!_addEdge(x, bUpper)) return FALSE;
    }

    _sortEdge();

    return TRUE;
}

//! 交点追加

BOOL CPolygonPaint::_addEdge(int x,BOOL bUpper)
{
    EDGEDAT dat;

    dat.x      = x;
    dat.bUpper = bUpper;

    if(!m_memEdge.addDat(&dat, sizeof(EDGEDAT)))
        return FALSE;

    m_nEdgeCnt++;

    return TRUE;
}

//! 交点を小さい順に並び替え

void CPolygonPaint::_sortEdge()
{
    EDGEDAT *p,*p2,*pMin,dat;
    int i,j;

    p = (EDGEDAT *)m_memEdge.getBuf();

    for(i = 0; i < m_nEdgeCnt - 1; i++, p++)
    {
        //一番小さい値取得

        pMin = p;
        p2   = p + 1;

        for(j = i + 1; j < m_nEdgeCnt; j++, p2++)
        {
            if(p2->x < pMin->x)
                pMin = p2;
        }

        //入れ替え

        if(p != pMin)
        {
            dat   = *p;
            *p    = *pMin;
            *pMin = dat;
        }
    }
}
