/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CGradList, CGradListItem - グラデーションデータ
*/


#include <stdlib.h>
#include <string.h>

#include "CGradList.h"

#include "struct.h"

#include "AXString.h"
#include "AXByteString.h"
#include "AXImage.h"
#include "AXFileWriteBuf.h"
#include "AXBuf.h"
#include "AXUtil.h"


//**************************************
// CGradList
//**************************************
/*
    データが更新された場合のみ終了時に保存。
    データを変更した場合は m_bUpdate を TRUE に。
*/


CGradList *CGradList::m_pSelf = NULL;


CGradList::CGradList()
{
    m_pSelf = this;

    m_bUpdate = FALSE;
}

//! 新規データ追加
/*!
    @param pstr  NULL 以外でテキストから貼り付け
*/

CGradListItem *CGradList::addNewGrad(const AXString *pstr)
{
    CGradListItem *p;
    BOOL ret;

    if(m_nCnt >= MAXCNT) return NULL;

    p = new CGradListItem;

    //

    if(pstr)
        ret = p->setFromTextFormat(*pstr);
    else
        ret = p->setDefault();

    if(!ret)
    {
        delete p;
        return NULL;
    }

    AXList::add(p);

    //

    m_bUpdate = TRUE;

    return p;
}

//! 複製を追加

CGradListItem *CGradList::copyGrad(CGradListItem *pItem)
{
    CGradListItem *p;

    if(m_nCnt >= MAXCNT) return NULL;

    p = new CGradListItem;

    if(!p->copy(pItem))
    {
        delete p;
        return NULL;
    }

    AXList::insert(p, pItem);

    //

    m_bUpdate = TRUE;

    //下の方のアイテムを選択する
    return pItem;
}

//! 描画用データセット

BOOL CGradList::setDrawDat(AXMem *pmem,int no,const RGBAFIX15 &colDraw,const RGBAFIX15 &colBack,BOOL bRev,BOOL bLoop)
{
    CGradListItem *p;

    p = (CGradListItem *)getItem(no);
    if(!p) return FALSE;

    return p->setDrawDat(pmem, colDraw, colBack, bRev, bLoop);
}

//! ファイルに保存 (Big Endian)

void CGradList::saveFile(const AXString &filename)
{
    AXFileWriteBuf file;
    CGradListItem *p;
    LPBYTE pSrc,ps;
    int i;

    //更新されていない場合、保存しない
    if(!m_bUpdate) return;

    if(!file.open(filename)) return;

    //ヘッダ
    file.putStr("AZPLGRAD");
    //バージョン
    file.putBYTE(0);
    //個数
    file.putBYTE(m_nCnt);

    //------- 各データ

    for(p = (CGradListItem *)m_pTop; p; p = p->next())
    {
        pSrc = p->m_memDat;

        file.put(pSrc, 3);

        //COL

        ps = pSrc + 3;

        for(i = pSrc[1]; i; i--, ps += 5)
        {
            file.putWORDBE(*((LPWORD)ps));
            file.put(ps + 2, 3);
        }

        //A

        for(i = pSrc[2]; i; i--, ps += 4)
        {
            file.putWORDBE(*((LPWORD)ps));
            file.putWORDBE(*((LPWORD)(ps + 2)));
        }
    }

    file.close();
}

//! ファイルから読み込み

void CGradList::loadFile(const AXString &filename)
{
    AXFile file;
    AXMem memAll;
    BYTE ver,cnt,i,dat[3];
    int size,j;
    CGradListItem *p;
    AXBuf buf;
    LPBYTE pDst;

    //------------ ファイルから読み込み

    if(!file.openRead(filename)) return;

    //ヘッダ

    if(!file.readCompare("AZPLGRAD")) return;

    //バージョン

    file.read(&ver, 1);
    if(ver != 0) return;

    //個数

    file.read(&cnt, 1);

    if(cnt >= MAXCNT) cnt = MAXCNT;

    //データをまとめて読み込み

    size = file.getSize() - 10;

    if(cnt == 0 || size <= 0) return;

    if(!memAll.alloc(size)) return;

    file.read(memAll, size);

    file.close();

    //---------- データ変換

    buf.init(memAll, size, AXBuf::ENDIAN_BIG);

    for(i = cnt; i; i--)
    {
        if(!buf.getDat(dat, 3)) return;

        //確保

        p = new CGradListItem;

        if(!p->m_memDat.alloc(3 + dat[1] * 5 + dat[2] * 4))
        {
            delete p;
            return;
        }

        AXList::add(p);

        //

        pDst = p->m_memDat;

        ::memcpy(pDst, dat, 3);
        pDst += 3;

        //COL

        for(j = dat[1]; j; j--, pDst += 5)
        {
            buf.getWORD(pDst);
            buf.getDat(pDst + 2, 3);
        }

        //A

        for(j = dat[2]; j; j--, pDst += 4)
        {
            buf.getWORD(pDst);
            buf.getWORD(pDst + 2);
        }
    }
}


//**************************************
// CGradListItem
//**************************************
/*
    [データ]

    BYTE : フラグ (0bit:繰り返し, 1bit:単色)
    BYTE : 色ポイントの数
    BYTE : Aポイントの数

    (色ポイント)
    WORD : 位置 (下位bit:位置(0-1024), 14-15bit:色タイプ[0/描画色 1/背景色 2/指定色])
    BYTE x 3: R,G,B (0-255)
    ...

    (Aポイント)
    WORD : 位置 (0-1024)
    WORD : A (0-1024)
    ...
*/


//! 新規用データセット

BOOL CGradListItem::setDefault()
{
    LPBYTE p;

    if(!m_memDat.allocClear(3 + 5 * 2 + 4 * 2)) return FALSE;

    p = m_memDat;

    p[1] = 2;
    p[2] = 2;

    p += 8;

    *((LPWORD)p) = 1024 | (1<<14); p += 2;
    *(p++) = 255;
    *(p++) = 255;
    *(p++) = 255;

    p += 2;

    *((LPWORD)p) = 1024; p += 2;
    *((LPWORD)p) = 1024; p += 2;
    *((LPWORD)p) = 1024;

    return TRUE;
}

//! データコピー

BOOL CGradListItem::copy(CGradListItem *pItem)
{
    if(!m_memDat.alloc(pItem->m_memDat.getSize()))
        return FALSE;
    else
    {
        m_memDat.copyFrom(pItem->m_memDat, pItem->m_memDat.getSize());

        return TRUE;
    }
}

//! クリップボードへのコピー用、データをテキストで取得

BOOL CGradListItem::getTextFormat(AXString *pstr)
{
    int srcsize,len,i;
    LPBYTE pDst,pd;
    AXMem mem,memText;

    srcsize = m_memDat.getSize();

    //-------- データをビッグエンディアンに

    if(!mem.alloc(srcsize)) return FALSE;

    mem.copyFrom(m_memDat, srcsize);

    pDst = mem;
    pd   = pDst + 3;

    for(i = pDst[1]; i; i--, pd += 5)
        AXSetWORDBE(pd, *((LPWORD)pd));

    for(i = pDst[2]; i; i--, pd += 4)
    {
        AXSetWORDBE(pd, *((LPWORD)pd));
        AXSetWORDBE(pd + 2, *((LPWORD)(pd + 2)));
    }

    //--------- テキストセット

    *pstr = "AZPTLGR:";
    *pstr += srcsize;
    *pstr += ':';

    //BASE64

    len = AXGetBase64EncSize(srcsize);

    if(!memText.alloc(len + 1)) return FALSE;

    pDst = memText;

    AXEncodeBase64(pDst, mem, srcsize);

    pDst[len] = 0;

    *pstr += (LPCSTR)pDst;

    return TRUE;
}

//! テキスト形式からデータセット

BOOL CGradListItem::setFromTextFormat(const AXString &strText)
{
    int srcsize,decsize,i;
    LPSTR pc;
    LPBYTE pDst,pd;
    AXByteString str;

    //UNICODE -> ASCII

    strText.toAscii(&str);

    pc = str;

    //ヘッダ

    if(::strncmp(pc, "AZPTLGR:", 8) != 0) return FALSE;

    pc += 8;

    //ソースサイズ

    srcsize = ::atoi(pc);
    if(srcsize > MAXSIZE) return FALSE;

    for(; *pc && *pc != ':'; pc++);

    if(!*pc) return FALSE;
    pc++;

    //データ

    if(!m_memDat.alloc(srcsize)) return FALSE;

    decsize = AXDecodeBase64(m_memDat, pc, srcsize);
    if(decsize != srcsize) return FALSE;

    //------------ ビッグエンディアンから変換

    pDst = m_memDat;
    pd   = pDst + 3;

    for(i = pDst[1]; i; i--, pd += 5)
        *((LPWORD)pd) = (pd[0] << 8) | pd[1];

    for(i = pDst[2]; i; i--, pd += 4)
    {
        *((LPWORD)pd)       = (pd[0] << 8) | pd[1];
        *((LPWORD)(pd + 2)) = (pd[2] << 8) | pd[3];
    }

    return TRUE;
}

//! 描画時用のデータ作成
/*
    BYTE : フラグ (0bit:反転, 1bit:繰り返し, 2bit:単色)
    BYTE : 色ポイントの数
    BYTE : Aポイントの数

    (色)
    WORD : 位置 (15bit固定小数点)
    WORD x 3 : R,G,B (15bit固定小数点)
    ...

    (A)
    WORD : 位置 (15bit固定小数点)
    WORD : A (15bit固定小数点)
*/

BOOL CGradListItem::setDrawDat(AXMem *pmem,const RGBAFIX15 &colDraw,const RGBAFIX15 &colBack,
    BOOL bRev,BOOL bLoop)
{
    LPBYTE pSrc,pDst,ps;
    LPWORD pd;
    WORD pos;
    BYTE flag;
    int i,j,type;

    pSrc = m_memDat;

    if(!pmem->alloc(3 + pSrc[1] * 8 + pSrc[2] * 4))
        return FALSE;

    pDst = *pmem;

    //フラグ

    flag = (bRev)? 1: 0;
    if(bLoop || (pSrc[0] & 1)) flag |= 2;
    if(pSrc[0] & 2) flag |= 4;

    pDst[0] = flag;

    //ポイントの数

    pDst[1] = pSrc[1];
    pDst[2] = pSrc[2];

    //------- 色

    pd = (LPWORD)(pDst + 3);
    ps = pSrc + 3;

    for(i = pSrc[1]; i; i--, pd += 4, ps += 5)
    {
        pos  = *((LPWORD)ps);
        type = pos >> 14;

        pd[0] = (pos & 0x3fff) << 5;   //(pos << 15) >> 10

        if(type == 0)
        {
            //描画色

            for(j = 0; j < 3; j++)
                pd[1 + j] = colDraw.c[j];
        }
        else if(type == 1)
        {
            //背景色

            for(j = 0; j < 3; j++)
                pd[1 + j] = colBack.c[j];
        }
        else
        {
            //指定色

            for(j = 0; j < 3; j++)
                pd[1 + j] = ((ps[2 + j] << 15) + 127) / 255;
        }
    }

    //-------- A

    for(i = pSrc[2]; i; i--, pd += 2, ps += 4)
    {
        pd[0] = *((LPWORD)ps) << 5;
        pd[1] = *((LPWORD)(ps + 2)) << 5;
    }

    return TRUE;
}

//! プレビューイメージ描画

void CGradListItem::drawPrev(AXImage *pimg,int x,int y,int w,int h)
{
    LPBYTE p = m_memDat;

    drawGradPrev(pimg, x, y, w, h, p + 3, p + 3 + p[1] * 5, p[0] & 2);
}

//! （関数）プレビューイメージ描画

void drawGradPrev(AXImage *pimg,int x,int y,int w,int h,LPBYTE pColBuf,LPBYTE pABuf,BOOL bSingle)
{
    LPBYTE p,pCol,pDst;
    LPWORD pA;
    int bpp,pitch,ix,iy,pos,pos1,pos2,i,col[3],a,xn,n;
    BYTE dcol[3];
    BYTE colCk[2][3] = {
        {0xff,0xe8,0xeb}, {0xff,0xc8,0xce}
    };

    pCol = pColBuf;
    pA   = (LPWORD)pABuf;

    pDst  = pimg->getBufPt(x, y);
    bpp   = pimg->getBytes();
    pitch = pimg->getPitch();

    //グラデーション描画
    /*  背景4x4グリッドのチェック柄と合成する（色は黒） */

    for(ix = 0; ix < w; ix++, pDst += bpp)
    {
        pos = (ix << 10) / (w - 1);

        //COL

        while(pos > (*((LPWORD)(pCol + 5)) & 0x3fff)) pCol += 5;

        if(bSingle)
        {
            //単色

            for(i = 0; i < 3; i++)
                col[i] = pCol[2 + i];
        }
        else
        {
            pos1 = *((LPWORD)pCol) & 0x3fff;
            pos2 = *((LPWORD)(pCol + 5)) & 0x3fff;

            pos2 = pos2 - pos1;
            pos1 = pos - pos1;

            for(i = 0; i < 3; i++)
                col[i] = (pCol[7 + i] - pCol[2 + i]) * pos1 / pos2 + pCol[2 + i];
        }

        //A

        while(pos > pA[2]) pA += 2;

        a = (pA[3] - pA[1]) * (pos - pA[0]) / (pA[2] - pA[0]) + pA[1];

        //縦に描画

        p  = pDst;
        xn = (ix & 7) >> 2;

        for(iy = 0; iy < h; iy++, p += pitch)
        {
            n = xn ^ ((iy & 7) >> 2);

            for(i = 0; i < 3; i++)
                dcol[i] = ((col[i] - colCk[n][i]) * a >> 10) + colCk[n][i];

            pimg->setPixelBuf(p, dcol[0], dcol[1], dcol[2]);
        }
    }
}
