/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CDrawPaint - 塗りつぶし処理クラス
*/

#include "CDrawPaint.h"

#include "draw_main.h"


CDrawPaint::CDrawPaint()
{
    m_pBuf = NULL;
}

CDrawPaint::~CDrawPaint()
{
    free();
}

//! 解放

void CDrawPaint::free()
{
    AXFree((void **)&m_pBuf);

    m_imgRes.free();
    m_imgTmp.free();
}

//! 実行前の初期化
/*!
    @return エラー値
*/

int CDrawPaint::init(int type,int diff,int opacity,const RGBAFIX15 &colDraw,
    CTileImage *pDst,CTileImage *pRef,const AXPoint &ptStart)
{
    AXRect rc;

    m_pimgDst = pDst;
    m_pimgRef = pRef;
    m_ptStart = ptStart;

    m_nType    = type;
    m_nOpacity = opacity;
    m_nDiff    = (0x8000 * diff + 500) / 1000;

    if(type == TYPE_ERASE)
        m_colDraw.zero();
    else
    {
        m_colDraw  = colDraw;
        m_colDraw.a = (0x8000 * opacity + 50) / 100;
    }

    _getRefPoint(&m_colBegin, ptStart.x, ptStart.y);

    //判定元の範囲取得

    switch(type)
    {
        case TYPE_ERASE:
            if(!m_pimgRef->getExistImgRectPx(&m_rcfRef)) return RET_NODRAW;
            break;
        default:
            m_rcfRef.set(0, 0, CTileImage::m_pinfo->nImgW - 1, CTileImage::m_pinfo->nImgH - 1);
            break;
    }

    //開始の点が範囲内か

    if(m_ptStart.x < m_rcfRef.x1 || m_ptStart.y < m_rcfRef.y1 ||
       m_ptStart.x > m_rcfRef.x2 || m_ptStart.y > m_rcfRef.y2)
       return RET_NODRAW;

    //色比較関数

    switch(type)
    {
        case TYPE_RGB:        m_funcComp = &CDrawPaint::_compare_rgb; break;
        case TYPE_ALPHA:      m_funcComp = &CDrawPaint::_compare_alpha; break;
        case TYPE_CANVAS:     m_funcComp = &CDrawPaint::_compare_canvas; break;
        case TYPE_ERASE:      m_funcComp = &CDrawPaint::_compare_erase; break;
        default: m_funcComp = NULL; break;
    }

    //開始の点が描画に値するか

    if(type == TYPE_ERASE && m_colBegin.a == 0)
        return RET_NODRAW;
    else if(type == TYPE_ALPHA_AUTO && m_colBegin.a)
        return RET_NODRAW;

    //バッファ確保

    m_pBuf = (PAINTBUF *)AXMalloc(sizeof(PAINTBUF) * BUFSIZE);
    if(!m_pBuf) return RET_ERR_MEMORY;

    //作業用イメージ

    m_rcfRef.toRect(&rc);

    if(!m_imgRes.create(rc)) return RET_ERR_MEMORY;

    if(type == TYPE_ALPHA_AUTO)
    {
        if(!m_imgTmp.create(rc)) return RET_ERR_MEMORY;
    }

    return RET_SUCCESS;
}

//! 実行

void CDrawPaint::run()
{
    if(m_nType == TYPE_ALPHA_AUTO)
    {
        _runAlphaAuto_horz();   //m_imgRes に
        _runAlphaAuto_vert();   //m_imgTmp に

        m_imgRes.combine_A1(m_imgTmp);
    }
    else
        _runNormal();

    //m_imgRes で点がある部分を描画

    m_imgRes.drawPixelEachTile(m_pimgDst, m_colDraw);

    free();
}


//==========================
// サブ
//==========================


//! 判定元の色取得

void CDrawPaint::_getRefPoint(RGBAFIX15 *pdst,int x,int y)
{
    if(m_nType != TYPE_CANVAS)
        m_pimgRef->getPixel(pdst, x, y);
    else
    {
        RGBFIX15 col;

        draw::getPixelBlendColor(&col, x, y);

        pdst->r = col.r;
        pdst->g = col.g;
        pdst->b = col.b;
        pdst->a = 0x8000;
    }
}

//! 判定元のアルファ値取得

int CDrawPaint::_getRefAlpha(int x,int y)
{
    RGBAFIX15 col;

    m_pimgRef->getPixel(&col, x, y);
    return col.a;
}


//==========================
// 通常塗りつぶし処理
//==========================


//! 実行

void CDrawPaint::_runNormal()
{
    int lx,rx,ly,oy,_lx,_rx;
    PAINTBUF *pst,*ped;

    pst = m_pBuf;
    ped = m_pBuf + 1;

    pst->lx = pst->rx = m_ptStart.x;
    pst->y  = pst->oy = m_ptStart.y;

    //

    do
    {
        lx = pst->lx;
        rx = pst->rx;
        ly = pst->y;
        oy = pst->oy;

        _lx = lx - 1;
        _rx = rx + 1;

        if(++pst == m_pBuf + BUFSIZE) pst = m_pBuf;

        //現在の点が非対象か

        if(m_imgRes.isPixelOn(lx, ly) || !(this->*m_funcComp)(lx, ly)) continue;

        //右方向の境界探す

        for(; rx < m_rcfRef.x2; rx++)
        {
            if(m_imgRes.isPixelOn(rx + 1, ly) || !(this->*m_funcComp)(rx + 1, ly)) break;
        }

        //左方向の境界探す

        for(; lx > m_rcfRef.x1; lx--)
        {
            if(m_imgRes.isPixelOn(lx - 1, ly) || !(this->*m_funcComp)(lx - 1, ly)) break;
        }

        //lx-rx の水平線描画
        //※描画できなければエラー

        if(!m_imgRes.drawLineH_A1(lx, rx, ly)) return;

        //真上の走査

        if(ly - 1 >= m_rcfRef.y1)
        {
            if(ly - 1 == oy)
            {
                _run_scan(lx, _lx, ly - 1, ly, &ped);
                _run_scan(_rx, rx, ly - 1, ly, &ped);
            }
            else
                _run_scan(lx, rx, ly - 1, ly, &ped);
        }

        //真下の走査

        if(ly + 1 <= m_rcfRef.y2)
        {
            if(ly + 1 == oy)
            {
                _run_scan(lx, _lx, ly + 1, ly, &ped);
                _run_scan(_rx, rx, ly + 1, ly, &ped);
            }
            else
                _run_scan(lx, rx, ly + 1, ly, &ped);
        }

    }while(pst != ped);
}

//! スキャン

void CDrawPaint::_run_scan(int lx,int rx,int y,int oy,PAINTBUF **pped)
{
    while(lx <= rx)
    {
        //開始点

        for(; lx < rx; lx++)
        {
            if(!m_imgRes.isPixelOn(lx, y) && (this->*m_funcComp)(lx, y)) break;
        }

        if(m_imgRes.isPixelOn(lx, y) || !(this->*m_funcComp)(lx, y)) break;

        (*pped)->lx = lx;

        //終了点

        for(; lx <= rx; lx++)
        {
            if(m_imgRes.isPixelOn(lx, y) || !(this->*m_funcComp)(lx, y)) break;
        }

        (*pped)->rx = lx - 1;
        (*pped)->y  = y;
        (*pped)->oy = oy;

        if(++(*pped) == m_pBuf + BUFSIZE)
            (*pped) = m_pBuf;
    }
}


//===============================
// アンチエイリアス自動判定
//===============================
/*
    境界を検索する時に最大アルファ値を記憶しておき、アルファ値が下がった点を境界とする。
    これを水平方向、垂直方向の両方で実行する。
*/


//! 水平走査（m_imgRes に描画）

void CDrawPaint::_runAlphaAuto_horz()
{
    int lx,rx,ly,oy,_lx,_rx,a,max,lx2,rx2,flag;
    PAINTBUF *pst,*ped;

    pst = m_pBuf;
    ped = m_pBuf + 1;

    pst->lx = pst->rx = m_ptStart.x;
    pst->y  = pst->oy = m_ptStart.y;

    //

    do
    {
        lx = pst->lx;
        rx = pst->rx;
        ly = pst->y;
        oy = pst->oy;

        _lx = lx - 1;
        _rx = rx + 1;

        if(++pst == m_pBuf + BUFSIZE) pst = m_pBuf;

        //対象外か

        if(m_imgRes.isPixelOn(lx, ly) || _getRefAlpha(lx, ly)) continue;

        //右方向の境界探す

        rx2 = rx;

        for(max = 0, flag = 1; rx < m_rcfRef.x2; rx++)
        {
            if(m_imgRes.isPixelOn(rx + 1, ly)) break;

            a = _getRefAlpha(rx + 1, ly);

            if(a) flag = 0;
            if(a >= m_colDraw.a || a < max) break;

            max = a;
            if(flag) rx2++;
        }

        //左方向の境界探す

        lx2 = lx;

        for(max = 0, flag = 1; lx > m_rcfRef.x1; lx--)
        {
            if(m_imgRes.isPixelOn(lx - 1, ly)) break;

            a = _getRefAlpha(lx - 1, ly);

            if(a) flag = 0;
            if(a >= m_colDraw.a || a < max) break;

            max = a;
            if(flag) lx2--;
        }

        //lx-rx の水平線描画

        if(!m_imgRes.drawLineH_A1(lx, rx, ly)) return;

        //真上の走査

        if(ly - 1 >= m_rcfRef.y1)
        {
            if(ly - 1 == oy)
            {
                _auto_scan_h(lx2, _lx, ly - 1, ly, &ped);
                _auto_scan_h(_rx, rx2, ly - 1, ly, &ped);
            }
            else
                _auto_scan_h(lx2, rx2, ly - 1, ly, &ped);
        }

        //真下の走査

        if(ly + 1 <= m_rcfRef.y2)
        {
            if(ly + 1 == oy)
            {
                _auto_scan_h(lx2, _lx, ly + 1, ly, &ped);
                _auto_scan_h(_rx, rx2, ly + 1, ly, &ped);
            }
            else
                _auto_scan_h(lx2, rx2, ly + 1, ly, &ped);
        }

    }while(pst != ped);
}

//! 水平スキャン

void CDrawPaint::_auto_scan_h(int lx,int rx,int y,int oy,PAINTBUF **pped)
{
    while(lx <= rx)
    {
        //開始点

        for(; lx < rx; lx++)
        {
            if(!m_imgRes.isPixelOn(lx, y) && !_getRefAlpha(lx, y)) break;
        }

        if(m_imgRes.isPixelOn(lx, y) || _getRefAlpha(lx, y)) break;

        (*pped)->lx = lx;

        //終了点

        for(; lx <= rx; lx++)
        {
            if(m_imgRes.isPixelOn(lx, y) || _getRefAlpha(lx, y)) break;
        }

        (*pped)->rx = lx - 1;
        (*pped)->y  = y;
        (*pped)->oy = oy;

        if(++(*pped) == m_pBuf + BUFSIZE)
            (*pped) = m_pBuf;
    }
}

//! 垂直走査（m_imgTmp に描画）

void CDrawPaint::_runAlphaAuto_vert()
{
    int ly,ry,xx,ox,_ly,_ry,a,max,ly2,ry2,flag;
    PAINTBUF *pst,*ped;

    pst = m_pBuf;
    ped = m_pBuf + 1;

    pst->lx = pst->rx = m_ptStart.y;
    pst->y  = pst->oy = m_ptStart.x;

    //

    do
    {
        ly = pst->lx;
        ry = pst->rx;
        xx = pst->y;
        ox = pst->oy;

        _ly = ly - 1;
        _ry = ry + 1;

        if(++pst == m_pBuf + BUFSIZE) pst = m_pBuf;

        //対象外か

        if(m_imgTmp.isPixelOn(xx, ly) || _getRefAlpha(xx, ly)) continue;

        //下方向の境界探す

        ry2 = ry;

        for(max = 0, flag = 1; ry < m_rcfRef.y2; ry++)
        {
            if(m_imgTmp.isPixelOn(xx, ry + 1)) break;

            a = _getRefAlpha(xx, ry + 1);

            if(a) flag = 0;
            if(a >= m_colDraw.a || a < max) break;

            max = a;
            if(flag) ry2++;
        }

        //上方向の境界探す

        ly2 = ly;

        for(max = 0, flag = 1; ly > m_rcfRef.y1; ly--)
        {
            if(m_imgTmp.isPixelOn(xx, ly - 1)) break;

            a = _getRefAlpha(xx, ly - 1);

            if(a) flag = 0;
            if(a >= m_colDraw.a || a < max) break;

            max = a;
            if(flag) ly2--;
        }

        //ly-ry の垂直線描画

        if(!m_imgTmp.drawLineV_A1(ly, ry, xx)) return;

        //左の走査

        if(xx - 1 >= m_rcfRef.x1)
        {
            if(xx - 1 == ox)
            {
                _auto_scan_v(ly2, _ly, xx - 1, xx, &ped);
                _auto_scan_v(_ry, ry2, xx - 1, xx, &ped);
            }
            else
                _auto_scan_v(ly2, ry2, xx - 1, xx, &ped);
        }

        //右の走査

        if(xx + 1 <= m_rcfRef.x2)
        {
            if(xx + 1 == ox)
            {
                _auto_scan_v(ly2, _ly, xx + 1, xx, &ped);
                _auto_scan_v(_ry, ry2, xx + 1, xx, &ped);
            }
            else
                _auto_scan_v(ly2, ry2, xx + 1, xx, &ped);
        }

    }while(pst != ped);
}

//! 垂直スキャン

void CDrawPaint::_auto_scan_v(int ly,int ry,int x,int ox,PAINTBUF **pped)
{
    while(ly <= ry)
    {
        //開始点

        for(; ly < ry; ly++)
        {
            if(!m_imgTmp.isPixelOn(x, ly) && !_getRefAlpha(x, ly)) break;
        }

        if(m_imgTmp.isPixelOn(x, ly) || _getRefAlpha(x, ly)) break;

        (*pped)->lx = ly;

        //終了点

        for(; ly <= ry; ly++)
        {
            if(m_imgTmp.isPixelOn(x, ly) || _getRefAlpha(x, ly)) break;
        }

        (*pped)->rx = ly - 1;
        (*pped)->y  = x;
        (*pped)->oy = ox;

        if(++(*pped) == m_pBuf + BUFSIZE)
            (*pped) = m_pBuf;
    }
}


//================================
// 色比較関数
//================================


//! RGB

BOOL CDrawPaint::_compare_rgb(int x,int y)
{
    RGBAFIX15 col;

    m_pimgRef->getPixel(&col, x, y);

    if(!m_colBegin.a)
        //開始点が透明だった場合、A=0 の同色範囲
        return !col.a;
    else if(!col.a)
        return FALSE;
    else
    {
        return (col.r >= m_colBegin.r - m_nDiff && col.r <= m_colBegin.r + m_nDiff &&
                col.g >= m_colBegin.g - m_nDiff && col.g <= m_colBegin.g + m_nDiff &&
                col.b >= m_colBegin.b - m_nDiff && col.b <= m_colBegin.b + m_nDiff);
    }
}

//! アルファ値

BOOL CDrawPaint::_compare_alpha(int x,int y)
{
    RGBAFIX15 col;

    m_pimgRef->getPixel(&col, x, y);

    return (col.a >= m_colBegin.a - m_nDiff && col.a <= m_colBegin.a + m_nDiff);
}

//! キャンバス

BOOL CDrawPaint::_compare_canvas(int x,int y)
{
    RGBFIX15 col;

    draw::getPixelBlendColor(&col, x, y);

    return (col.r >= m_colBegin.r - m_nDiff && col.r <= m_colBegin.r + m_nDiff &&
            col.g >= m_colBegin.g - m_nDiff && col.g <= m_colBegin.g + m_nDiff &&
            col.b >= m_colBegin.b - m_nDiff && col.b <= m_colBegin.b + m_nDiff);
}

//! 不透明範囲

BOOL CDrawPaint::_compare_erase(int x,int y)
{
    RGBAFIX15 col;

    m_pimgRef->getPixel(&col, x, y);

    return (col.a != 0);
}
