/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/

#include <math.h>
#include <stdlib.h>

#include "drawdat.h"
#include "drawOpDef.h"

#include "global.h"

#include "draw_opxor.h"
#include "draw_opfunc.h"
#include "draw_opsub.h"
#include "draw_calc.h"

#include "CCanvasWin.h"
#include "CStatusBar.h"
#include "CXImage.h"
#include "CTileImageA1.h"
#include "CPolygonPaint.h"


namespace draw
{


//=============================
// XOR直線
//=============================
/*
    ptTmp[0] : 始点
    ptTmp[1] : 終点（現在の位置）
*/


//! 押し

BOOL op_xorline_down(int after)
{
    setOpInfo(OPNO_XORLINE, op_xorline_move, op_xorline_up, after);

    getNowPoint(&g_draw->work.ptTmp[0]);

    g_draw->work.ptTmp[1] = g_draw->work.ptTmp[0];

    drawXorLine();

    return TRUE;
}

//! 移動（※XorLine2共通）

void op_xorline_move(UINT uModKey)
{
    AXPoint pt;

    drawXorLine();

    getNowPoint(&pt);
    if(uModKey & MOD_SHIFT) getPtLine45(&pt, g_draw->work.ptTmp[0]);

    g_draw->work.ptTmp[1] = pt;

    drawXorLine();

    //ステータスバーに情報表示

    if(g_draw->work.nNowOpNo == OPNO_XORLINE)
        setStatusBar_line(g_draw->work.ptTmp[0], g_draw->work.ptTmp[1]);
}

//! 離し
/*!
    @return グラブ終了するか
*/

BOOL op_xorline_up()
{
    drawXorLine();

    CANVASAREA->clearTimerUpdate(CCanvasWinArea::TIMERID_UPDATE);

    STATUSBAR->setLabel_help();

    switch(g_draw->work.nOpAfter)
    {
        //直線描画
        case OPAFTER_DRAWLINE:
            draw_line();
            break;
        //ベジェ曲線
        case OPAFTER_BEZIER:
            return op_bezier_up_first();
        //グラデーション
        case OPAFTER_GRAD:
            draw_grad();
            break;
        //定規、平行線/格子線
        case OPAFTER_RULE_LINE:
        case OPAFTER_RULE_GRID:
            double x1,y1,x2,y2,d;

            g_draw->view.winToimg(&x1, &y1, g_draw->work.ptTmp[0].x, g_draw->work.ptTmp[0].y);
            g_draw->view.winToimg(&x2, &y2, g_draw->work.ptTmp[1].x, g_draw->work.ptTmp[1].y);

            d = ::atan2(y2 - y1, x2 - x1);

            if(g_draw->work.nOpAfter == OPAFTER_RULE_LINE)
                g_draw->rule.dAngleLine = d;
            else
                g_draw->rule.dAngleGrid = d;
            break;
    }

	return TRUE;
}


//=================================
// XOR直線2（連続直線/集中線/多角形）
//=================================
/*
    ptTmp[0] : 始点
    ptTmp[1] : 終点（マウス位置）
    ptTmp[2] : 一番最初の点（保存）
*/


//! 最初の押し

BOOL op_xorline2_down(int after)
{
    getNowPoint(&g_draw->work.ptTmp[0]);

    //各初期化

    switch(after)
    {
        //連続直線・集中線
        case OPAFTER_SUCCLINE:
        case OPAFTER_CONCLINE:
            setBeforeDraw(g_draw->work.nOpToolNo);
            beginDrawUndo();
            break;
        //多角形
        case OPAFTER_POLYGON:
            if(!op_polygon_down()) return FALSE;
            break;
    }

    /* 多角形が初期化に失敗する場合があるので、ここで行う */

    setOpInfo(OPNO_XORLINE2, op_xorline_move, op_temp_up_norelease, after);

    g_draw->work.ptTmp[1] = g_draw->work.ptTmp[0];
    g_draw->work.ptTmp[2] = g_draw->work.ptTmp[0];

    drawXorLine();

    return TRUE;
}

//! 2度目以降の押し時

void op_xorline2_down2()
{
    switch(g_draw->work.nOpAfter)
    {
        //連続直線・集中線
        case OPAFTER_SUCCLINE:
        case OPAFTER_CONCLINE:
            drawXorLine();

            draw_lineSuccConc();

            if(g_draw->work.nOpAfter == OPAFTER_SUCCLINE)
                g_draw->work.ptTmp[0] = g_draw->work.ptTmp[1];

            drawXorLine();
            break;
        //多角形
        case OPAFTER_POLYGON:
            op_polygon_down2();
            break;
    }
}

//! キャンセルor終了処理

BOOL opCancel_xorline2()
{
    switch(g_draw->work.nOpAfter)
    {
        //連続直線・集中線
        case OPAFTER_SUCCLINE:
        case OPAFTER_CONCLINE:
            opCancel_lineSuccConc(FALSE);
            break;
        //多角形
        case OPAFTER_POLYGON:
            opCancel_polygon(TRUE);
            break;
    }

    return TRUE;
}


//=====================================
// XOR四角枠（キャンバスウィンドウに対する）
//=====================================
/*
    ptTmp[0] : 左上（ウィンドウ）
    ptTmp[1] : 右下（ウィンドウ）
    ptTmp[2] : 開始点
*/


//! 押し

BOOL op_xorboxwin_down(int after)
{
    setOpInfo(OPNO_XORBOXWIN, op_xorboxwin_move, op_xorboxwin_up, after);

    getNowPoint(&g_draw->work.ptTmp[2]);

    g_draw->work.ptTmp[0] = g_draw->work.ptTmp[1] = g_draw->work.ptTmp[2];

    drawXorBoxWin();

    return TRUE;
}

//! 移動時

void op_xorboxwin_move(UINT uModKey)
{
    AXPoint pt;
    int w,h;

    drawXorBoxWin();

    getNowPoint(&pt);

    //左上、左下

    if(pt.x < g_draw->work.ptTmp[2].x)
    {
        g_draw->work.ptTmp[0].x = pt.x;
        g_draw->work.ptTmp[1].x = g_draw->work.ptTmp[2].x;
    }
    else
    {
        g_draw->work.ptTmp[0].x = g_draw->work.ptTmp[2].x;
        g_draw->work.ptTmp[1].x = pt.x;
    }

    if(pt.y < g_draw->work.ptTmp[2].y)
    {
        g_draw->work.ptTmp[0].y = pt.y;
        g_draw->work.ptTmp[1].y = g_draw->work.ptTmp[2].y;
    }
    else
    {
        g_draw->work.ptTmp[0].y = g_draw->work.ptTmp[2].y;
        g_draw->work.ptTmp[1].y = pt.y;
    }

    //正方形調整

    if(uModKey & MOD_SHIFT)
    {
        w = g_draw->work.ptTmp[1].x - g_draw->work.ptTmp[0].x;
        h = g_draw->work.ptTmp[1].y - g_draw->work.ptTmp[0].y;

        if(w < h)
        {
            if(pt.x < g_draw->work.ptTmp[2].x)
                g_draw->work.ptTmp[0].x = g_draw->work.ptTmp[1].x - h;
            else
                g_draw->work.ptTmp[1].x = g_draw->work.ptTmp[0].x + h;
        }
        else
        {
            if(pt.y < g_draw->work.ptTmp[2].y)
                g_draw->work.ptTmp[0].y = g_draw->work.ptTmp[1].y - w;
            else
                g_draw->work.ptTmp[1].y = g_draw->work.ptTmp[0].y + w;
        }
    }

    //

    drawXorBoxWin();
}

//! 離し時

BOOL op_xorboxwin_up()
{
    drawXorBoxWin();
    CANVASAREA->clearTimerUpdate(CCanvasWinArea::TIMERID_UPDATE);

    switch(g_draw->work.nOpAfter)
    {
        //四角枠描画
        case OPAFTER_DRAWBOX:
            draw_box();
            break;
        //四角塗りつぶし描画
        case OPAFTER_DRAWFILLBOX:
            draw_fillBox();
            break;
        //選択範囲
        case OPAFTER_SELBOX:
            draw_sel_box();
            break;
    }

    return TRUE;
}


//====================================
// XOR四角枠 (イメージに対する)
//====================================
/*
    ptTmp[0] : 開始点（イメージ位置）
    ptTmp[1] : 左上位置（イメージ位置）
    ptTmp[2] : 右下位置（イメージ位置）
*/


//! 押し

BOOL op_xorboximg_down(int after)
{
    setOpInfo(OPNO_XORBOXIMG, op_xorboximg_move, op_xorboximg_up, after);

    getNowImgPoint(&g_draw->work.ptTmp[0]);

    g_draw->work.ptTmp[1] = g_draw->work.ptTmp[2] = g_draw->work.ptTmp[0];

    drawXorBoxImg();

    return TRUE;
}

//! 移動時

void op_xorboximg_move(UINT uModKey)
{
    AXPoint pt;
    int w,h,x1,y1,x2,y2;

    getNowImgPoint(&pt);

    //消す

    drawXorBoxImg();

    //左上、左下

    if(pt.x < g_draw->work.ptTmp[0].x)
        x1 = pt.x, x2 = g_draw->work.ptTmp[0].x;
    else
        x1 = g_draw->work.ptTmp[0].x, x2 = pt.x;

    if(pt.y < g_draw->work.ptTmp[0].y)
        y1 = pt.y, y2 = g_draw->work.ptTmp[0].y;
    else
        y1 = g_draw->work.ptTmp[0].y, y2 = pt.y;

    //正方形調整

    if(uModKey & MOD_SHIFT)
    {
        w = x2 - x1;
        h = y2 - y1;

        if(w < h)
        {
            if(pt.x < g_draw->work.ptTmp[0].x)
                x1 = x2 - h;
            else
                x2 = x1 + h;
        }
        else
        {
            if(pt.y < g_draw->work.ptTmp[0].y)
                y1 = y2 - w;
            else
                y2 = y1 + w;
        }
    }

    //

    g_draw->work.ptTmp[1].x = x1, g_draw->work.ptTmp[1].y = y1;
    g_draw->work.ptTmp[2].x = x2, g_draw->work.ptTmp[2].y = y2;

    //

    drawXorBoxImg();
}

//! 離し時

BOOL op_xorboximg_up()
{
    drawXorBoxImg();
    CANVASAREA->clearTimerUpdate(CCanvasWinArea::TIMERID_UPDATE);

    //範囲をイメージ内に調整 -> rcsTmp[0] (範囲外なら処理しない)

    if(!adjustImgRect(&g_draw->work.rcsTmp[0],
            g_draw->work.ptTmp[1].x, g_draw->work.ptTmp[1].y,
            g_draw->work.ptTmp[2].x, g_draw->work.ptTmp[2].y))
        return TRUE;

    //

    switch(g_draw->work.nOpAfter)
    {
        //左右反転
        case 0:
            draw_boxedit_revHV(TRUE);
            break;
        //上下反転
        case 1:
            draw_boxedit_revHV(FALSE);
            break;
        //左に90度回転
        case 2:
            draw_boxedit_rotate90(TRUE);
            break;
        //右に90度回転
        case 3:
            draw_boxedit_rotate90(FALSE);
            break;
        //拡大縮小＆回転
        case 4:
            //後で実行
            CANVASWIN->sendCommand(CCanvasWin::CMDID_SCALEROTDLG, 0, 0);
            break;
    }

    return TRUE;
}


//====================================
// XOR楕円 （キャンバスウィンドウに対する）
//====================================
/*
    ptTmp[0] : 位置
    ptTmp[1] : 半径
*/


//! 押し

BOOL op_xorcircle_down(int after)
{
    setOpInfo(OPNO_XORCIRCLE, op_xorcircle_move, op_xorcircle_up, after);

    getNowPoint(&g_draw->work.ptTmp[0]);

    g_draw->work.ptTmp[1].zero();

    drawXorCircle();

    return TRUE;
}

//! 移動

void op_xorcircle_move(UINT uModKey)
{
    AXPoint pt;
    int xr,yr;

    drawXorCircle();

    getNowPoint(&pt);

    xr = ::abs(pt.x - g_draw->work.ptTmp[0].x);
    yr = ::abs(pt.y - g_draw->work.ptTmp[0].y);

    if(uModKey & MOD_SHIFT)
    {
        if(xr > yr) yr = xr;
        else xr = yr;
    }

    g_draw->work.ptTmp[1].x = xr;
    g_draw->work.ptTmp[1].y = yr;

    drawXorCircle();
}

//! 離し

BOOL op_xorcircle_up()
{
    drawXorCircle();

    CANVASAREA->clearTimerUpdate(CCanvasWinArea::TIMERID_UPDATE);

    switch(g_draw->work.nOpAfter)
    {
        //円枠描画
        case OPAFTER_DRAWCIRCLE:
            draw_circle();
            break;
        //円塗りつぶし描画
        case OPAFTER_DRAWFILLCIRCLE:
            draw_fillCircle();
            break;
        //定規、楕円
        case OPAFTER_RULE_ELLIPSE:
            {
            RULEDAT *p = &g_draw->rule;
            double xx,yy;

            g_draw->view.winToimg(&p->ptCtEll, g_draw->work.ptTmp[0].x, g_draw->work.ptTmp[0].y);

            xx = g_draw->work.ptTmp[1].x * g_draw->view.param.dScaleDiv;
            yy = g_draw->work.ptTmp[1].y * g_draw->view.param.dScaleDiv;
            if(xx == 0) xx = 0.1;
            if(yy == 0) yy = 0.1;

            p->dEllHV[0]    = xx / yy;
            p->dEllHV[1]    = yy / xx;
            p->dEllTmp[0]   = g_draw->view.param.dCosRev;
            p->dEllTmp[1]   = g_draw->view.param.dSinRev;
            p->dEllTmp[2]   = g_draw->view.param.dCos;
            p->dEllTmp[3]   = g_draw->view.param.dSin;
            p->bEllRevH     = g_draw->view.bHRev;
            }
            break;
    }

    return TRUE;
}


//=============================
// XOR投げ縄
//=============================
/*
    ptTmp[0] : 前の位置
    ptTmp[1] : 現在の位置
*/


//! 押し時

BOOL op_xorlasso_down(int after)
{
    double x,y;

    getNowPoint(&g_draw->work.ptTmp[0]);

    //作業用

    if(!g_draw->pimgXor->create(g_draw->view.szCanvas.w, g_draw->view.szCanvas.h))
        return FALSE;

    //多角形位置

    if(!g_draw->pPolyPaint->alloc(200))
    {
        g_draw->pimgXor->free();
        return FALSE;
    }

    g_draw->view.winToimg(&x, &y, g_draw->work.ptWinNow.x, g_draw->work.ptWinNow.y);
    g_draw->pPolyPaint->add(x, y);

    //

    setOpInfo(OPNO_XORLASSO, op_xorlasso_move, op_xorlasso_up, after);

    g_draw->work.ptTmp[1] = g_draw->work.ptTmp[0];

    g_draw->drawinfo.funcDrawPixel = &CTileImage::setPixel_create;

    drawXorLasso(FALSE);

    return TRUE;
}

//! 移動時

void op_xorlasso_move(UINT uModKey)
{
    double x,y;

    g_draw->work.ptTmp[0] = g_draw->work.ptTmp[1];

    getNowPoint(&g_draw->work.ptTmp[1]);

    drawXorLasso(FALSE);

    //点追加

    g_draw->view.winToimg(&x, &y, g_draw->work.ptWinNow.x, g_draw->work.ptWinNow.y);
    g_draw->pPolyPaint->add(x, y);
}

//! 離し時

BOOL op_xorlasso_up()
{
    drawXorLasso(TRUE);
    CANVASAREA->clearTimerUpdate(CCanvasWinArea::TIMERID_UPDATE);

    g_draw->pimgXor->free();

    g_draw->pPolyPaint->endPos();

    switch(g_draw->work.nOpAfter)
    {
        //塗りつぶし
        case OPAFTER_DRAWFILLLASSO:
            draw_fillPolygon_common();
            break;
        //選択範囲
        case OPAFTER_SELLASSO:
            draw_sel_polygonCommon();
            break;
    }

    g_draw->pPolyPaint->free();

    return TRUE;
}


//=============================
// そのほか
//=============================


//! 定規[集中線・正円]の中心位置 押し時
/*
    ptTmp[0] : 位置
*/

BOOL op_ruleCenterPos_down()
{
    AXPoint pt;

    setOpInfo(OPNO_RULEPOS, op_temp_move, op_ruleCenterPos_up);

    getNowPoint(&g_draw->work.ptTmp[0]);

    //中央位置セット

    g_draw->view.winToimg(&pt, g_draw->work.ptTmp[0].x, g_draw->work.ptTmp[0].y);

    if(g_draw->rule.type == RULETYPE_CONCLINE)
        g_draw->rule.ptCtConc = pt;
    else
        g_draw->rule.ptCtCir = pt;

    //

    drawXorRulePos();

    return TRUE;
}

//! 離し時

BOOL op_ruleCenterPos_up()
{
    drawXorRulePos();
    return TRUE;
}


//=============================
// XOR描画
//=============================


//! XOR直線
/*
    ptTmp[0]が始点、ptTmp[1]が終点
*/

void drawXorLine()
{
    AXRectSize rcs;

    g_draw->pimgCanvas->line(g_draw->work.ptTmp[0].x, g_draw->work.ptTmp[0].y,
                g_draw->work.ptTmp[1].x, g_draw->work.ptTmp[1].y, AXImage::COL_XOR);

    //更新

    rcs.setFromPoint(g_draw->work.ptTmp[0].x, g_draw->work.ptTmp[0].y,
                g_draw->work.ptTmp[1].x, g_draw->work.ptTmp[1].y);

    CANVASAREA->redraw(rcs);
    CANVASAREA->setTimer_update();
}

//! XOR四角枠（ウィンドウに対する）
/*
    ptTmp[0]が左上、ptTmp[1]が右下
*/

void drawXorBoxWin()
{
    AXRectSize rcs;

    rcs.x = g_draw->work.ptTmp[0].x;
    rcs.y = g_draw->work.ptTmp[0].y;
    rcs.w = g_draw->work.ptTmp[1].x - g_draw->work.ptTmp[0].x + 1;
    rcs.h = g_draw->work.ptTmp[1].y - g_draw->work.ptTmp[0].y + 1;

    //

    g_draw->pimgCanvas->box(rcs.x, rcs.y, rcs.w, rcs.h, AXImage::COL_XOR);

    //更新

    CANVASAREA->redraw(rcs);
    CANVASAREA->setTimer_update();
}

//! XOR四角枠描画（イメージに対する）

void drawXorBoxImg()
{
    AXPoint pt[5];
    int x1,y1,x2,y2;

    x1 = g_draw->work.ptTmp[1].x, y1 = g_draw->work.ptTmp[1].y;
    x2 = g_draw->work.ptTmp[2].x, y2 = g_draw->work.ptTmp[2].y;

    if(g_draw->view.nScale != 1000) x2++, y2++;

    //イメージ -> ウィンドウ位置

    g_draw->view.imgTowin(pt    , x1, y1);
    g_draw->view.imgTowin(pt + 1, x2, y1);
    g_draw->view.imgTowin(pt + 2, x2, y2);
    g_draw->view.imgTowin(pt + 3, x1, y2);
    pt[4] = pt[0];

    //描画

    g_draw->pimgCanvas->lines(pt, 5, AXImage::COL_XOR);

    //更新

    CANVASAREA->redraw();
    CANVASAREA->setTimer_update();
}

//! XOR円
/*
    ptTmp[0]が中心、ptTmp[1]が半径
*/

void drawXorCircle()
{
    int x1,y1,x2,y2;

    x1 = g_draw->work.ptTmp[0].x - g_draw->work.ptTmp[1].x;
    y1 = g_draw->work.ptTmp[0].y - g_draw->work.ptTmp[1].y;
    x2 = g_draw->work.ptTmp[0].x + g_draw->work.ptTmp[1].x;
    y2 = g_draw->work.ptTmp[0].y + g_draw->work.ptTmp[1].y;

    g_draw->pimgCanvas->ellipse(x1, y1, x2, y2, AXImage::COL_XOR);

    //更新

    CANVASAREA->redraw(x1, y1, x2 - x1 + 1, y2 - y1 + 1);
    CANVASAREA->setTimer_update();
}

//! 定規中央位置（十字線）
/*
    ptTmp[0]が中心位置
*/

void drawXorRulePos()
{
    //描画

    g_draw->pimgCanvas->lineH(g_draw->work.ptTmp[0].x - 15, g_draw->work.ptTmp[0].y, 31, AXImage::COL_XOR);
    g_draw->pimgCanvas->lineV(g_draw->work.ptTmp[0].x, g_draw->work.ptTmp[0].y - 15, 31, AXImage::COL_XOR);

    //更新

    CANVASAREA->redraw(g_draw->work.ptTmp[0].x - 15, g_draw->work.ptTmp[0].y - 15, 31, 31);
}

//! XORベジェ曲線

void drawXorBezier(BOOL bErase)
{
    AXRect rc;

    //描画

    if(!bErase)
    {
        g_draw->pimgXor->freeAllTile();
        g_draw->pimgXor->drawBezierTemp(g_draw->work.ptTmp, &rc, g_draw->work.nOpAfter);

        //AXRect -> AXRectSize

        g_draw->view.rectInCanvas(&rc);
        g_draw->work.rcsTmp[0].set(rc);
    }

    //XOR

    g_draw->pimgXor->blendXor(g_draw->pimgCanvas, g_draw->work.rcsTmp[0]);

    CANVASAREA->redraw(g_draw->work.rcsTmp[0]);
    CANVASAREA->setTimer_update();
}

//! XOR投げ縄

void drawXorLasso(BOOL bErase)
{
    AXRect rc;
    AXRectSize rcs;
    RGBAFIX15 col;

    if(bErase)
    {
        //消去

        rcs.set(0, 0, g_draw->view.szCanvas.w, g_draw->view.szCanvas.h);

        g_draw->pimgXor->blendXor(g_draw->pimgCanvas, rcs);

        CANVASAREA->redraw();
    }
    else
    {
        //新しい線描画

        col.set(0,0,0,0x8000);

        rc.setFromPoint(g_draw->work.ptTmp[0], g_draw->work.ptTmp[1]);

        if(g_draw->view.rectInCanvas(&rcs, rc))
        {
            g_draw->pimgXor->blendXor(g_draw->pimgCanvas, rcs);

            g_draw->pimgXor->drawLineB(g_draw->work.ptTmp[0].x, g_draw->work.ptTmp[0].y,
                g_draw->work.ptTmp[1].x, g_draw->work.ptTmp[1].y, col, FALSE);

            g_draw->pimgXor->blendXor(g_draw->pimgCanvas, rcs);

            CANVASAREA->redraw(rcs);
        }
    }

    CANVASAREA->setTimer_update();
}

//! XORブラシ円描画
/*
    ptTmp[0] : 中心ウィンドウ位置
    nTmp[0]  : 半径pxサイズ
*/

void drawXorBrushCircle(BOOL bErase)
{
    AXRectSize rcs;

    if(bErase)
        rcs = g_draw->work.rcsTmp[0];
    else
    {
        rcs.x = g_draw->work.ptTmp[0].x - g_draw->work.nTmp[0];
        rcs.y = g_draw->work.ptTmp[0].y - g_draw->work.nTmp[0];
        rcs.w = rcs.h = (g_draw->work.nTmp[0] << 1) + 1;

        g_draw->work.rcsTmp[0] = rcs;
    }

    //描画

    g_draw->pimgCanvas->ellipse(rcs.x, rcs.y, rcs.x + rcs.w - 1, rcs.y + rcs.h - 1, AXImage::COL_XOR);

    //更新

    CANVASAREA->redraw(rcs);
}

};
