/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    ポインタデバイス操作メイン処理
*/

#include <math.h>

#include "drawdat.h"
#include "drawOpDef.h"

#include "draw_main.h"
#include "draw_opmain.h"
#include "draw_opsub.h"
#include "draw_opxor.h"
#include "draw_opfunc.h"

#include "CMainWin.h"
#include "CPrevWin.h"
#include "CCanvasWin.h"
#include "CStatusBar.h"

#include "CConfig.h"
#include "CKeyDat.h"
#include "CDevList.h"
#include "CTileImageA1.h"

#include "global.h"

#include "AXKey.h"


namespace draw
{

//------------------------

BOOL _onDown_keycmd(int id);
BOOL _onDown_tool(int tool,int subno);
BOOL _onDown_draw(int toolno,int drawtype,BOOL bRegBrush=FALSE,BOOL bPressMax=FALSE);
BOOL _onDown_sel(int type);
BOOL _onDown_drawPolygon(int toolno,int type);
void _onDown_rule(BOOL bDot);

//------------------------


//! ボタン押し時
/*!
    @param btt  CDevItem のボタン＋動作値
    @return グラブするかどうか
*/

BOOL onDown(const DRAWPOINT &pt,UINT btt)
{
    int cmdid;
    UINT key;

    //値初期化

    g_draw->work.ptWinDown = pt;
    g_draw->work.ptWinNow  = pt;
    g_draw->work.ptWinBk   = pt;
    g_draw->work.uBttDown  = btt;

    //装飾キー

    g_draw->work.uModDown = 0;
    if(btt & CDevItem::FLAG_CTRL) g_draw->work.uModDown |= MOD_CTRL;
    if(btt & CDevItem::FLAG_SHIFT) g_draw->work.uModDown |= MOD_SHIFT;
    if(btt & CDevItem::FLAG_ALT) g_draw->work.uModDown |= MOD_ALT;

    //

    if(g_draw->work.nNowOpNo != OPNO_NONE)
    {
        //--------- 操作中

        if((btt >> CDevItem::SHIFT_ACTION) == CDevItem::ACTION_MAIN)
        {
            //連続直線/集中線/多角形の2点目以降

            if(g_draw->work.nNowOpNo == OPNO_XORLINE2)
                op_xorline2_down2();
        }

        return FALSE;
    }
    else
    {
        //---------- 非操作時

        g_draw->work.uOpFlags = 0;

        //左ボタン＋キー処理

        if((btt & CDevItem::MASK_BUTTON) == 0)
        {
            //+スペースキー

            if(CANVASWIN->isDownKeySpace())
            {
                if((btt & CDevItem::MASK_MOD) == CDevItem::FLAG_CTRL)
                {
                    //+Ctrl : キャンバス回転

                    g_draw->work.uOpFlags |= OPFLAGS_NOMOVEMOD;
                    return op_canvasRotate_down();
                }
                else if((btt & CDevItem::MASK_MOD) == CDevItem::FLAG_SHIFT)
                    //Shift+上下ドラッグ : 表示倍率変更
                    return op_zoomDrag_down();
                else
                    //キャンバス移動
                    return op_canvasMove_down();
            }
        }

        //処理

        cmdid = btt >> CDevItem::SHIFT_ACTION;

        if(cmdid != CDevItem::ACTION_MAIN)
        {
            //[デバイス設定]
            /*
                ボタン+装飾キーで実行される動作の場合、動作側での Ctrl,Shift,Alt は無効とする。
                例えば 右ボタン+Shift でスポイトにした場合、スポイト側で +Shift が処理されてしまうため。
            */

            if(btt & CDevItem::MASK_MOD)
            {
                g_draw->work.uModDown = 0;
                g_draw->work.uOpFlags |= OPFLAGS_NOMOVEMOD;
            }

            return _onDown_keycmd(cmdid);
        }
        else
        {
            //メイン操作（[キー設定]キー＋操作判定あり）

            key = CANVASWIN->getLastDownKey();

            if(key)
            {
                cmdid = KEYDAT->getCmdDraw(key);

                if(cmdid != -1) return _onDown_keycmd(cmdid);
            }

            return _onDown_tool(g_draw->tool.toolno, -1);
        }
    }

    return FALSE;
}

//! キー＋操作時、またはデバイス設定の動作

BOOL _onDown_keycmd(int id)
{
    if(id >= 2000 && id < 3000)
    {
        if(id < 2100)
            //ツール動作 [2000-]
            return _onDown_tool(id - 2000, -1);
        else if(id < 2200)
        {
            //描画タイプ [2100-]（ブラシ/ドット時。それ以外は通常動作）

            return _onDown_tool(g_draw->tool.toolno, (g_draw->tool.isBrushOrDotPen())? id - 2100: -1);
        }
        else
        {
            //他 (2200-)

            switch(id - 2200)
            {
                case 0: return _onDown_draw(TOOL_BRUSH, 0, TRUE, FALSE); //登録ブラシで描画
                case 1: return _onDown_draw(TOOL_BRUSH, 0, TRUE, TRUE);  //登録ブラシで描画(筆圧最大)
                case 2: return op_zoomDrag_down();          //表示倍率変更
                case 3: return op_dragBrushSize_down(FALSE);  //ブラシサイズ変更
                case 4: return op_dragBrushSize_down(TRUE);   //登録ブラシのサイズ変更
                case 5: return op_midcolor_down();          //中間色作成
                case 6: return op_replaceColor_down();      //色の置換
            }
        }
    }
    else
        //実行コマンド[デバイス設定]
        MAINWIN->sendCommand(id, 0, 0);

    return FALSE;
}

//! 左ダブルクリック時
/*!
    @return TRUE でダブルクリックとして処理し、グラブ解除。FALSE で通常の押しとして処理する。
*/

BOOL onLDblClk()
{
    //ベジェは対象外

    if(g_draw->work.nNowOpNo == OPNO_BEZIER) return FALSE;

    //[連続直線/集中線/多角形] の場合、終了

    if(onCancel())
    {
        g_draw->work.nNowOpNo = OPNO_NONE;
        return TRUE;
    }

    return FALSE;
}

//! ダイアログ中に左クリックされた時

void onLDownInDlg(int x,int y)
{
    switch(g_draw->work.nNowOpNo)
    {
        //テキスト
        case OPNO_DRAWTEXT:
            op_textInDlg_down(x, y);
            break;
        //フィルタダイアログ中
        case OPNO_FILTERDLG:
            ((AXWindow *)g_draw->work.pTmp)->sendCommand(10000, (x << 16)|y, 0);
            break;
    }
}

//! ポインタ移動時

void onMove(const DRAWPOINT &pt,BOOL bCtrl,BOOL bShift)
{
    AXPoint ptn;
    UINT modkey;

    //イメージ位置

    g_draw->view.winToimg(&ptn, pt.x, pt.y);

    //ステータスバー

    STATUSBAR->setLabel_pos(ptn.x, ptn.y);

    //プレビューウィンドウ、ルーペ時

    if(g_conf->isPrevWin() && g_conf->isPrevWinLoupe())
        PREVWIN->moveCanvasPos(ptn);

    //

    if(g_draw->work.nNowOpNo == OPNO_NONE)
    {
        //------ 操作中でない

        /* カーソルがメインウィンドウ上を移動した時、
           メインウィンドウがアクティブになっていない場合は、アクティブにする */

        if(g_conf->uFlags & CConfig::FLAG_AUTOACTIVE)
        {
            if(!MAINWIN->isFocused())
                MAINWIN->setActive();
        }
    }
    else
    {
        //位置

        g_draw->work.ptWinNow = pt;

        //装飾キー

        modkey = 0;

        if(!(g_draw->work.uOpFlags & OPFLAGS_NOMOVEMOD))
        {
            if(bCtrl)  modkey |= MOD_CTRL;
            if(bShift) modkey |= MOD_SHIFT;
        }

        //処理

        (*(g_draw->work.funcMove))(modkey);

        //

        g_draw->work.ptWinBk = pt;
    }
}

//! ボタン離し時
/*!
    @param bttno ボタン番号(1-)
    @return グラブ解除するか
*/

BOOL onUp(const DRAWPOINT &pt,UINT bttno)
{
    BOOL bRelease = TRUE;

    if(g_draw->work.nNowOpNo == OPNO_NONE) return FALSE;

    //位置

    g_draw->work.ptWinNow = pt;

    //右ボタンキャンセル動作

    if(bttno == 3 && onCancel()) goto END;

    //押し時と同じボタンか

    if(bttno - 1 != (g_draw->work.uBttDown & CDevItem::MASK_BUTTON))
        return FALSE;

    //処理

    bRelease = (*(g_draw->work.funcUp))();

    //操作終了

END:
    if(bRelease)
        g_draw->work.nNowOpNo = OPNO_NONE;

    return bRelease;
}

//! 右ボタン離し/ダブルクリック 時のキャンセル処理
/*!
    @return TRUE で操作終了
*/

BOOL onCancel()
{
    switch(g_draw->work.nNowOpNo)
    {
        //連続直線/集中線/多角形
        case OPNO_XORLINE2:
            return opCancel_xorline2();
        //ベジェ曲線
        case OPNO_BEZIER:
            return opCancel_bezier();
    }

    return FALSE;
}

//! 操作中のキー操作
/*!
    @return TRUEでグラブ解放
*/

BOOL onKey_inOp(UINT uKey)
{
    int no,after;
    BOOL bRelease = FALSE;

    no    = g_draw->work.nNowOpNo;
    after = g_draw->work.nOpAfter;

    switch(uKey)
    {
        //ENTER
        case KEY_ENTER:
        case KEY_NUM_ENTER:
            //[連続直線/集中線/多角形/ベジェ] 描画またはキャンセル
            if(no == OPNO_XORLINE2 || no == OPNO_BEZIER)
                bRelease = onCancel();
            break;

        //ESC
        case KEY_ESCAPE:
            //[多角形] キャンセル
            if(no == OPNO_XORLINE2 && after == OPAFTER_POLYGON)
                bRelease = opCancel_polygon(FALSE);
            else if(no == OPNO_XORLINE2 || no == OPNO_BEZIER)
                //[連続直線/集中線/ベジェ] キャンセル
                bRelease = onCancel();
            break;

        //BACKSPACE
        case KEY_BACKSPACE:
            if(no == OPNO_XORLINE2 && after == OPAFTER_SUCCLINE)
                //[連続直線] 始点と結び終了
                bRelease = opCancel_lineSuccConc(TRUE);
            break;
    }

    if(bRelease)
    {
        g_draw->work.nNowOpNo = OPNO_NONE;
        return TRUE;
    }
    else
        return FALSE;
}


//=============================
// メイン操作
//=============================


//! 押し時（ツール動作）
/*!
    @param subno -1 で指定ツールのサブタイプ
*/

BOOL _onDown_tool(int tool,int subno)
{
    if(subno == -1)
        subno = g_draw->tool.toolSubNo[tool];

    //ブラシ/ドット描画

    if(tool == TOOL_BRUSH || tool == TOOL_DOTPEN)
        return _onDown_draw(tool, subno);

    //他ツール

    switch(tool)
    {
        //イメージ移動
        case TOOL_MOVE:
            return op_imgmove_down();
        //図形塗り・消し
        case TOOL_POLYPAINT:
        case TOOL_POLYERASE:
            return _onDown_drawPolygon(tool, subno);
        //塗りつぶし
        case TOOL_PAINT:
        case TOOL_PAINTERASE:
            g_draw->work.nOpToolNo = tool;

            setOpInfo(OPNO_TEMP, op_temp_move, op_paint_up);
            return TRUE;
        //グラデーション
        case TOOL_GRAD:
            if(checkDrawLayer())
                return FALSE;
            else
                return op_xorline_down(OPAFTER_GRAD);
        //テキスト(後で実行する)
        case TOOL_TEXT:
            if(!checkDrawLayer())
                CANVASWIN->sendCommand(CCanvasWin::CMDID_TEXTDLG, 0, 0);
            return FALSE;
        //自動選択
        case TOOL_MAGICWAND:
            setOpInfo(OPNO_TEMP, op_temp_move, op_magicwand_up);
            return TRUE;
        //選択範囲
        case TOOL_SEL:
            return _onDown_sel(subno);
        //矩形編集
        case TOOL_BOXEDIT:
            if(checkDrawLayer())
                return FALSE;
            else
                return op_xorboximg_down(subno);
        //キャンバス回転
        case TOOL_CANVROTATE:
            return op_canvasRotate_down();
        //キャンバス移動
        case TOOL_CANVMOVE:
            return op_canvasMove_down();
        //ズーム
        case TOOL_SCALE:
            return op_zoomClick_down(subno);
        //スポイト
        case TOOL_SPOIT:
            return op_spoit_down();
    }

    return FALSE;
}


//=================================
// 各ツールの押し時
//=================================


//! 押し時（ブラシ/ドットペン描画）
/*!
    @param bPressMax 筆圧を最大で固定
*/

BOOL _onDown_draw(int toolno,int drawtype,BOOL bRegBrush,BOOL bPressMax)
{
    g_draw->work.rcfDrawUp.clear();

    g_draw->work.nOpToolNo = toolno;

    //自由線以外時、描画可能かチェック

    if(drawtype != DRAWTYPE_FREE && checkDrawLayer())
        return FALSE;

    //

    switch(drawtype)
    {
        //自由線
        case DRAWTYPE_FREE:
            if(g_draw->work.uModDown & MOD_ALT)
            {
                //+Alt でスポイト

                return op_spoit_down();
            }
            else if(g_draw->rule.type && (g_draw->work.uModDown & MOD_CTRL))
            {
                //+Ctrl で定規ONの場合は定規設定

                switch(g_draw->rule.type)
                {
                    case RULETYPE_LINE:
                        return op_xorline_down(OPAFTER_RULE_LINE);
                    case RULETYPE_GRIDLINE:
                        return op_xorline_down(OPAFTER_RULE_GRID);
                    case RULETYPE_CONCLINE:
                    case RULETYPE_CIRCLE:
                        return op_ruleCenterPos_down();
                    case RULETYPE_ELLIPSE:
                        return op_xorcircle_down(OPAFTER_RULE_ELLIPSE);
                    default:
                        return FALSE;
                }
            }
            else if(toolno == TOOL_BRUSH && g_draw->work.uModDown == MOD_SHIFT)
            {
                //ブラシ時 +Shift でブラシサイズ変更

                return op_dragBrushSize_down(FALSE);
            }
            else
            {
                //描画不可

                if(checkDrawLayer()) return FALSE;

                //

                _onDown_rule((toolno == TOOL_DOTPEN));

                //

                if(toolno == TOOL_BRUSH)
                    //ブラシ
                    op_brush_free_down(bRegBrush, bPressMax);
                else
                    //ドット
                    op_dotpen_free_down();
            }
            return TRUE;
        //直線
        case DRAWTYPE_LINE:
            return op_xorline_down(OPAFTER_DRAWLINE);
        //四角枠
        case DRAWTYPE_BOX:
            return op_xorboxwin_down(OPAFTER_DRAWBOX);
        //円枠
        case DRAWTYPE_CIRCLE:
            return op_xorcircle_down(OPAFTER_DRAWCIRCLE);
        //連続直線
        case DRAWTYPE_SUCCLINE:
            return op_xorline2_down(OPAFTER_SUCCLINE);
        //集中線
        case DRAWTYPE_CONCLINE:
            return op_xorline2_down(OPAFTER_CONCLINE);
        //ベジェ曲線
        case DRAWTYPE_BEZIER:
            if(!g_draw->pimgXor->create(g_draw->view.szCanvas.w, g_draw->view.szCanvas.h))
                return FALSE;

            return op_xorline_down(OPAFTER_BEZIER);
    }

    return FALSE;
}

//! 押し時（図形塗りつぶし）

BOOL _onDown_drawPolygon(int toolno,int type)
{
    if(checkDrawLayer()) return FALSE;

    g_draw->work.nOpToolNo = toolno;

    switch(type)
    {
        //四角
        case 0:
            return op_xorboxwin_down(OPAFTER_DRAWFILLBOX);
        //円
        case 1:
            return op_xorcircle_down(OPAFTER_DRAWFILLCIRCLE);
        //多角形
        case 2:
            g_draw->work.nOpAfter2 = OPAFTER2_DRAWFILLPOLYGON;
            return op_xorline2_down(OPAFTER_POLYGON);
        //投げ縄
        default:
            return op_xorlasso_down(OPAFTER_DRAWFILLLASSO);
    }

    return FALSE;
}

//! 押し時（選択範囲）

BOOL _onDown_sel(int type)
{
    AXPoint pt;

    /* +Ctrl で範囲内が押されたら、ドラッグ移動
      (Ctrl+Shift で上書き移動） */

    if(g_draw->work.uModDown & MOD_CTRL)
    {
        getNowImgPoint(&pt);

        if(pt.x >= 0 && pt.x < g_draw->nImgW && pt.y >= 0 && pt.y < g_draw->nImgH &&
            isSelExist() && !g_draw->pimgSel->isPixelTransparent(pt.x, pt.y))
        {
            return op_selimgmove_down();
        }
    }

    //移動時の装飾キー無効

    g_draw->work.uOpFlags |= OPFLAGS_NOMOVEMOD;

    //

    switch(type)
    {
        //ブラシ
        case 0:
            return op_selbrush_down();
        //四角
        case 1:
            return op_xorboxwin_down(OPAFTER_SELBOX);
        //多角形
        case 2:
            g_draw->work.nOpAfter2 = OPAFTER2_SELPOLYGON;
            return op_xorline2_down(OPAFTER_POLYGON);
        //投げ縄
        default:
            return op_xorlasso_down(OPAFTER_SELLASSO);
    }

    return FALSE;
}

//! 定規時、押し時の計算

void _onDown_rule(BOOL bDot)
{
    RULEDAT *p = &g_draw->rule;
    DRAWPOINT pt;
    double xx,yy,x,y;

    //押し時の位置

    getDrawPoint(&pt, FALSE, FALSE);

    p->ptDown.x = pt.x;
    p->ptDown.y = pt.y;

    //ドット線の場合切り捨て

    if(bDot)
    {
        p->ptDown.x = ::floor(p->ptDown.x);
        p->ptDown.y = ::floor(p->ptDown.y);
    }

    //各タイプ別

    switch(p->type)
    {
        //格子線
        case RULETYPE_GRIDLINE:
            p->dTmp = p->dAngleGrid;
            p->nTmp = FALSE;
            break;
        //集中線 (dTmp = 角度)
        case RULETYPE_CONCLINE:
            xx = p->ptDown.x - p->ptCtConc.x;
            yy = p->ptDown.y - p->ptCtConc.y;
            p->dTmp = ::atan2(yy, xx);
            break;
        //正円 (dTmp = 半径)
        case RULETYPE_CIRCLE:
            xx = p->ptDown.x - p->ptCtCir.x;
            yy = p->ptDown.y - p->ptCtCir.y;
            p->dTmp = ::sqrt(xx * xx + yy * yy);
            break;
        //楕円 (dTmp = 半径)
        case RULETYPE_ELLIPSE:
            x = p->ptDown.x - p->ptCtEll.x;
            y = p->ptDown.y - p->ptCtEll.y;
            if(p->bEllRevH) x = -x;

            xx = (x * p->dEllTmp[2] - y * p->dEllTmp[3]) * p->dEllHV[1];
            yy =  x * p->dEllTmp[3] + y * p->dEllTmp[2];

            p->dTmp = ::sqrt(xx * xx + yy * yy);
            break;
    }
}

};
