/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    操作時のサブ処理
*/

#include <math.h>
#include <string.h>

#include "drawdat.h"
#include "global.h"

#include "drawOpDef.h"
#include "draw_main.h"
#include "draw_opfunc.h"
#include "draw_calc.h"
#include "draw_update.h"

#include "CCanvasWin.h"
#include "CStatusBar.h"

#include "CConfig.h"
#include "CTileImageA1.h"
#include "CImage8.h"
#include "CLayerList.h"
#include "CLayerItem.h"
#include "CBrushList.h"
#include "CBrushItem.h"
#include "CUndo.h"


namespace draw
{

//--------------------------------

void (CTileImage::*g_colfunc[8])(RGBAFIX15 *pdst,const RGBAFIX15 &src,LPVOID pParam) = {
    &CTileImage::colfunc_normal, &CTileImage::colfunc_comp_alpha,
    &CTileImage::colfunc_overwrite, &CTileImage::colfunc_erase,
    &CTileImage::colfunc_dodge, &CTileImage::colfunc_burn,
    &CTileImage::colfunc_add, &CTileImage::colfunc_blur
};

//--------------------------------


//! 現在のレイヤに描画可能かチェック

int checkDrawLayer()
{
    if(g_draw->pcurlayer->isFolder())
        //フォルダ
        return CHECKDRAW_FOLDER;
    else if(g_draw->pcurlayer->isLockDraw())
        //ロックチェック
        return CHECKDRAW_LOCK;
    else
        return CHECKDRAW_OK;
}

//! AXPoint に現在のウィンドウ位置取得

void getNowPoint(AXPoint *ppt)
{
    ppt->x = (int)g_draw->work.ptWinNow.x;
    ppt->y = (int)g_draw->work.ptWinNow.y;
}

//! AXPoint に現在のイメージ位置取得

void getNowImgPoint(AXPoint *ppt)
{
    g_draw->view.winToimg(ppt, g_draw->work.ptWinNow.x, g_draw->work.ptWinNow.y);
}

//! ステータスバーに2点間の情報セット

void setStatusBar_line(const AXPoint &pt1,const AXPoint &pt2)
{
    double x1,y1,x2,y2;
    int angle;

    g_draw->view.winToimg(&x1, &y1, pt1.x, pt1.y);
    g_draw->view.winToimg(&x2, &y2, pt2.x, pt2.y);

    x1 = x2 - x1;
    y1 = y2 - y1;

    //角度(1.0=100)

    angle = (int)(::atan2(y1, x1) * -18000 / M_PI);
    if(angle < 0) angle += 36000;

    STATUSBAR->setLabel_lineinfo(::sqrt(x1 * x1 + y1 * y1), angle);
}

//! 操作時の情報セット

void setOpInfo(int opno,void (*funcMove)(UINT),BOOL (*funcUp)(),int afterno)
{
    g_draw->work.nNowOpNo = opno;
    g_draw->work.nOpAfter = afterno;
    g_draw->work.funcMove = funcMove;
    g_draw->work.funcUp   = funcUp;
}

//! pimgTmp のイメージ解放

void freeTmpTileImg(int no)
{
    if(g_draw->pimgTmp[no])
    {
        delete g_draw->pimgTmp[no];
        g_draw->pimgTmp[no] = NULL;
    }
}

//! pimgTmp のイメージ確保（カレントレイヤのカラータイプで）

BOOL allocTmpTileImg(int no,const AXRect &rc)
{
    freeTmpTileImg(no);

    g_draw->pimgTmp[no] = allocTileImage(g_draw->pcurlayer->m_nColType);

    if(!g_draw->pimgTmp[no]->create(rc))
    {
        freeTmpTileImg(no);
        return FALSE;
    }
    
    //レイヤ色をセット
    
    g_draw->pimgTmp[no]->setImageColor(g_draw->pcurlayer->m_pimg->getImageColor());

    return TRUE;
}


//=============================
// アンドゥ用
//=============================


//! 描画用、アンドゥ準備

void beginDrawUndo()
{
    CTileImage *pimg;

    pimg = getCurImg();

    //

    g_draw->drawinfo.fUndoErr = 0;
    g_draw->drawinfo.rcfDraw.clear();

    //現在のイメージ情報取得(アンドゥ用)

    pimg->getInfo(&g_draw->work.infoImgUndo);

    //-------- アンドゥ用イメージ作成
    /* カレントレイヤのイメージと同じカラータイプ・タイル配列になるよう作成。
       !配列を同じに出来なかった場合はアンドゥなし */

    if(g_draw->pimgUndo->getColType() != pimg->getColType())
    {
        //カラータイプが異なる -> 再確保

        delete g_draw->pimgUndo;
        g_draw->pimgUndo = allocTileImage(pimg->getColType());
    }

    //同じ情報で作成

    g_draw->pimgUndo->createSame(*pimg);

    //セット

    g_draw->drawinfo.pimgUndo = g_draw->pimgUndo;
}

//! 描画用、アンドゥ終了
/*!
    @param rcs 更新イメージ範囲（キャンバス範囲内調整済み）
*/

void endDrawUndo(const AXRectSize &rcs)
{
    //アンドゥが確保できなかった場合はクリア

    if(g_draw->drawinfo.fUndoErr & 2)
        g_draw->pundo->deleteAllDat();

    //アンドゥ成功時

    if(g_draw->drawinfo.fUndoErr == 0)
    {
        //空タイルを解放

        getCurImg()->freeEmptyTileFromUndo();

        //UNDO追加

        g_draw->pundo->add_curLayerUndoImage(g_draw->work.infoImgUndo, rcs);
    }

    //タイル削除（配列は残す）

    g_draw->pimgUndo->freeAllTile();
}


//=============================
// アンドゥ・更新の共通処理
//=============================


//! カーソルを砂時計に＆アンドゥ開始

void beginDraw()
{
    CANVASAREA->setCursorWait();
    beginDrawUndo();
}

//! カーソルを戻す＆終了処理

void endDraw()
{
    commonAfterDraw();
    CANVASAREA->restoreCursorTool();
}

//! [自由線/連続直線/集中線] 描画時の共通処理（描画した後）
/*!
    @param bTimer TRUEでタイマー更新セット。FALSEで直接更新。
*/
/*
    drawinfo.rcfDraw : 描画された範囲
    rcfDrawUp : 全体の描画された範囲
*/

void commonFreeDraw(BOOL bTimer)
{
    AXRectSize rcsUp;

    if(g_draw->drawinfo.rcfDraw.flag)
    {
        //範囲追加

        g_draw->work.rcfDrawUp.combine(g_draw->drawinfo.rcfDraw);

        //更新（キャンバス範囲内）

        if(getImgRect(&rcsUp, g_draw->drawinfo.rcfDraw))
        {
            if(bTimer)
                CANVASAREA->setTimer_updateRect(rcsUp);
            else
                updateRect(rcsUp);
        }
    }
}

//! 自由線描画終了時など、複数描画後の共通処理
/*!
    rcfDrawUp に全体の更新範囲を入れておく。
*/

void commonDrawEnd()
{
    AXRectSize rcs;

    CANVASAREA->clearTimerUpdate(CCanvasWinArea::TIMERID_UPDATERECT);

    if(g_draw->work.rcfDrawUp.flag)
    {
        //全体の描画範囲をキャンバス範囲内に

        if(!getImgRect(&rcs, g_draw->work.rcfDrawUp))
            rcs.x = -1;

        //UNDO

        endDrawUndo(rcs);

        //更新

        updateAfterDraw(rcs);
    }
}

//! 直線描画時などの描画後共通処理（drawinfo.rcfDraw の範囲で更新）

void commonAfterDraw()
{
    AXRectSize rcs;

    if(g_draw->drawinfo.rcfDraw.flag)
    {
        //イメージ範囲内に

        if(!getImgRect(&rcs, g_draw->drawinfo.rcfDraw))
            rcs.x = -1;

        //UNDO

        endDrawUndo(rcs);

        //更新

        if(rcs.x != -1)
        {
            updateRect(rcs);
            updateAfterDraw(rcs);
        }
    }
}


//==============================
// 描画前のセット
//==============================


//! 描画前の各情報セット
/*!
    @param toolno    -1 でマスクなどのみセット
    @param brushno   ブラシ描画時は-1で選択ブラシ・それ以外で登録ブラシ、ドットペン時はデータリストの番号(-1で現在の選択)
*/

void setBeforeDraw(int toolno,int brushno)
{
    TILEIMGDRAWINFO *p = &g_draw->drawinfo;

    //描画時の点セット関数

    p->funcDrawPixel = &CTileImage::setPixelDraw;

    //デフォルトでオプションのテクスチャ

    p->pTexture = (g_draw->pimg8OptTex->isExist())? g_draw->pimg8OptTex: NULL;

    //レイヤマスク

    p->pimgMask = g_draw->player->getLayerMaskImg(g_draw->pcurlayer);

    //選択範囲イメージ

    p->pimgSel = (g_draw->work.rcfSel.flag && g_draw->pimgSel->isExist())? g_draw->pimgSel: NULL;

    //色マスク・アルファマスク

    p->nColMaskType = g_conf->btColMaskType;
    p->nAMaskType   = g_draw->pcurlayer->m_nAmaskType;

    //描画色

    p->colDraw = g_draw->colDraw;

    //----------------

    if(toolno == -1) return;

    switch(toolno)
    {
        //ブラシ
        case TOOL_BRUSH:
            setBeforeDraw_brush(brushno);
            break;
        //ドットペン
        case TOOL_DOTPEN:
            setBeforeDraw_dotpen(brushno);
            break;
        //図形塗りつぶし
        case TOOL_POLYPAINT:
            g_draw->work.optDrawTmp = g_draw->tool.optPolyPaint;

            p->funcColor = g_colfunc[(g_draw->tool.optPolyPaint >> 8) & 7];
            break;
        //図形消しゴム
        case TOOL_POLYERASE:
            g_draw->work.optDrawTmp = g_draw->tool.optPolyErase;

            p->funcColor = &CTileImage::colfunc_erase;
            break;
        //塗りつぶし
        case TOOL_PAINT:
            p->funcColor = g_colfunc[(g_draw->tool.optPaint >> 3) & 7];
            break;
        //不透明範囲消し（テクスチャ無効）
        case TOOL_PAINTERASE:
            p->funcColor = &CTileImage::colfunc_overwrite;
            p->pTexture  = NULL;
            break;
        //グラデーション
        case TOOL_GRAD:
            p->funcColor = g_colfunc[(g_draw->tool.optGrad >> 8) & 7];
            break;
        //テキスト
        case TOOL_TEXT:
            p->funcColor = &CTileImage::colfunc_normal;
            break;
        //自動選択・選択範囲
        case TOOL_MAGICWAND:
        case TOOL_SEL:
            p->funcDrawPixel = &CTileImage::setPixel_subdraw;
            p->funcColor     = &CTileImage::colfunc_overwrite;
            p->rcfDraw.clear();

            setBeforeDraw_clearMask();
            break;
        //矩形編集
        case TOOL_BOXEDIT:
            p->funcColor = &CTileImage::colfunc_overwrite;
            setBeforeDraw_clearMask();
            break;
    }
}

//! 選択・レイヤマスク・テクスチャ・色マスク・Aマスク クリア

void setBeforeDraw_clearMask()
{
    g_draw->drawinfo.pimgMask  = NULL;
    g_draw->drawinfo.pimgSel   = NULL;
    g_draw->drawinfo.pTexture  = NULL;

    g_draw->drawinfo.nColMaskType = 0;
    g_draw->drawinfo.nAMaskType   = 0;
}

//! ブラシ描画用データセット

void setBeforeDraw_brush(int brushno)
{
    CBrushItem *pitem;
    BRUSHDRAWPARAM *pparam;
    BOOL bRegBrush = (brushno != -1);

    pitem  = BRUSHLIST->getDrawBrush(bRegBrush);
    pparam = BRUSHLIST->getDrawParam(bRegBrush);

    g_draw->drawinfo.funcColor = g_colfunc[pitem->btPixType];
    g_draw->drawinfo.pBrush    = pparam->pimgBrush;

    //テクスチャ

    if(pitem->strTexImg[0] == '?' && pitem->strTexImg[1] == 0)
        g_draw->drawinfo.pTexture = (g_draw->pimg8OptTex->isExist())? g_draw->pimg8OptTex: NULL;
    else
        g_draw->drawinfo.pTexture = pparam->pimgTexture;

    //ぼかしの場合は濃度を元に間隔を調整

    if(pitem->btPixType == CBrushItem::PIXTYPE_BLUR)
        pparam->dInterval = (100 - pitem->btOpacity) * 0.02 + 0.04;
    else
        pparam->dInterval = pparam->dIntervalSrc;

    //

    CTileImage::m_pbrush = pparam;
}

//! 選択範囲：ドラッグ移動時

void setBeforeDraw_selImgMove(BOOL bOverwrite)
{
    setBeforeDraw(-1);

    g_draw->drawinfo.pimgSel   = NULL;
    g_draw->drawinfo.pTexture  = NULL;
    g_draw->drawinfo.funcColor = (bOverwrite)? &CTileImage::colfunc_overwrite: &CTileImage::colfunc_normal;
}

//! 選択範囲：ブラシ描画時

void setBeforeDraw_selBrush()
{
    BOOL bErase;
    BRUSHDRAWPARAM *p = &g_draw->work.brparam;

    setBeforeDraw_brush(-1);
    setBeforeDraw_clearMask();

    //+Shift または消しゴムブラシの場合は範囲削除

    bErase = ((BRUSHLIST->getEditItem())->btPixType == CBrushItem::PIXTYPE_ERASE);
    if(g_draw->work.uModDown == MOD_SHIFT) bErase = TRUE;

    g_draw->drawinfo.funcDrawPixel = &CTileImage::setPixel_subdraw;
    g_draw->drawinfo.funcColor     = (bErase)? &CTileImage::colfunc_erase: &CTileImage::colfunc_overwrite;
    g_draw->drawinfo.colDraw.zero();

    //work.brparam にコピー

    ::memcpy(p, CTileImage::m_pbrush, sizeof(BRUSHDRAWPARAM));

    CTileImage::m_pbrush = p;

    //

    p->dOpacity     = 1;
    p->dMinOpacity  = 1;
    p->dwFlags |= BRUSHDF_NONEAA;
    p->dwFlags &= ~(BRUSHDF_WATER);
}

//! ドットペン時のセット

void setBeforeDraw_dotpen(int listno)
{
    DWORD val;
    int size,type;

    //ペン情報取得

    if(listno == -1)
        listno = g_draw->tool.btDotPenListSel;

    val  = g_draw->tool.dwDotPenList[listno];
    size = val & 0xff;
    type = (val >> 16) & 3;

    //

    g_draw->drawinfo.funcDrawPixel = &CTileImage::setPixelDotPen;
    g_draw->drawinfo.funcColor     = (type == 2)? &CTileImage::colfunc_erase: &CTileImage::colfunc_normal;

    //ブラシサイズ

    if(size != g_draw->drawinfo.nDotPenSize)
    {
        g_draw->drawinfo.nDotPenSize = size;

        g_draw->pimg8DotBrush->drawDotBrush(size);
    }

    //細線

    g_draw->work.nTmp[0] = (type == 1);

    //描画濃度

    g_draw->drawinfo.colDraw.a = (((val >> 8) & 0xff) * 0x8000  + 50) / 100;
}

//! フィルタ用

void setBeforeDraw_filter(BOOL bPrev)
{
    g_draw->drawinfo.pimgMask     = NULL;
    g_draw->drawinfo.pTexture     = NULL;
    g_draw->drawinfo.nColMaskType = 0;
    g_draw->drawinfo.nAMaskType   = 0;

    if(bPrev)
    {
        g_draw->drawinfo.pimgSel  = NULL;
        g_draw->drawinfo.funcDrawPixel = &CTileImage::setPixel_create2;
    }
    else
    {
        g_draw->drawinfo.pimgSel       = (isSelExist())? g_draw->pimgSel: NULL;
        g_draw->drawinfo.funcDrawPixel = &CTileImage::setPixelDraw;
        g_draw->drawinfo.funcColor     = &CTileImage::colfunc_overwrite;
    }
}

//! フィルタ、漫画用:フラッシュの描画時

void setBeforeDraw_filterDrawFlash()
{
    BRUSHDRAWPARAM *p = &g_draw->work.brparam;

    g_draw->drawinfo.pBrush = NULL;

    CTileImage::m_pbrush = p;

    p->dShapeHard   = -4;
    p->dOpacity     = 1;
    p->dMinSize     = 0;
    p->dMinOpacity  = 0;
    p->dInterval    = 0.4;
    p->nRotAngle    = 0;
    p->nRotRandom   = 0;
    p->nRoughVal    = 0;
    p->dwFlags      = 0;
}


//==============================
// 描画位置取得
//==============================


//! 描画時のイメージ位置データ取得
/*!
    @param bRuleInt  定規時、x,y を int 値で計算（ドット描画時）
*/

void getDrawPoint(DRAWPOINT *pdst,BOOL bEnableRule,BOOL bRuleInt)
{
    double x,y,xx,yy,len,ad;
    int a;

    //イメージ位置に変換

    g_draw->view.winToimg(&x, &y, g_draw->work.ptWinNow.x, g_draw->work.ptWinNow.y);

    //定規時の位置補正

    if(bEnableRule && g_draw->rule.type)
    {
        RULEDAT *p = &g_draw->rule;

        if(bRuleInt)
        {
            x = ::floor(x);
            y = ::floor(y);
        }

        switch(p->type)
        {
            //======== 平行線・格子線・集中線（角度：360度 = 1024）
            case RULETYPE_LINE:
            case RULETYPE_GRIDLINE:
            case RULETYPE_CONCLINE:
                xx = x - p->ptDown.x;
                yy = y - p->ptDown.y;

                len = ::sqrt(xx * xx + yy * yy);            //押し位置からの距離
                a   = (int)(::atan2(yy, xx) * 512 / M_PI);  //押し位置からの角度(int)

                ad = (p->type == RULETYPE_LINE)? p->dAngleLine: p->dTmp;

                //[格子線]時は長さが一定になるまで、そのまま。長さが一定まで来たら、角度を計算

                if(p->type == RULETYPE_GRIDLINE && !p->nTmp)
                {
                    x = p->ptDown.x;
                    y = p->ptDown.y;

                    if(len >= 3.0)
                    {
                        //45〜135度、245〜335度の範囲は+90度角度

                        a = (a - ((int)(p->dAngleGrid * 512 / M_PI)) + 1024) & 1023;

                        if((a > 128 && a < 128 + 256) || (a > 128 + 256 * 2 && a < 1024 - 128))
                            p->dTmp += M_PI / 2;

                        p->nTmp = TRUE;
                    }

                    break;
                }

                //-90〜+90 度の範囲ならそのまま、反対側なら反転角度

                a = (a - ((int)(ad * 512 / M_PI)) + 1024) & 1023;
                if(a > 256 && a < 1024 - 256) ad += M_PI;

                x = len * ::cos(ad) + p->ptDown.x;
                y = len * ::sin(ad) + p->ptDown.y;
                break;
            //========= 正円
            case RULETYPE_CIRCLE:
                xx = x - p->ptCtCir.x;
                yy = y - p->ptCtCir.y;
                ad = ::atan2(yy, xx);

                x = p->dTmp * ::cos(ad) + p->ptCtCir.x;
                y = p->dTmp * ::sin(ad) + p->ptCtCir.y;
                break;
            //========= 楕円
            case RULETYPE_ELLIPSE:
                //角度

                xx = x - p->ptCtEll.x; if(p->bEllRevH) xx = -xx;
                yy = y - p->ptCtEll.y;

                x = (xx * p->dEllTmp[2] - yy * p->dEllTmp[3]) * p->dEllHV[1];
                y =  xx * p->dEllTmp[3] + yy * p->dEllTmp[2];

                ad = ::atan2(y, x);

                //楕円

                xx = p->dTmp * p->dEllHV[0] * ::cos(ad);
                yy = p->dTmp * ::sin(ad);

                x = xx * p->dEllTmp[0] - yy * p->dEllTmp[1];
                y = xx * p->dEllTmp[1] + yy * p->dEllTmp[0] + p->ptCtEll.y;

                if(p->bEllRevH) x = -x;
                x += p->ptCtEll.x;
                break;
        }
    }

    //セット

    pdst->x = x;
    pdst->y = y;
    pdst->press = g_draw->work.ptWinNow.press;
}

//! ドット描画用、イメージ位置取得

void getDrawPointInt(AXPoint *ppt,BOOL bEnableRule)
{
    DRAWPOINT pt;

    getDrawPoint(&pt, bEnableRule, TRUE);

    if(bEnableRule && g_draw->rule.type)
    {
        //定規時は繰り上げ

        ppt->x = (int)(pt.x + 0.5);
        ppt->y = (int)(pt.y + 0.5);
    }
    else
    {
        ppt->x = (int)pt.x;
        ppt->y = (int)pt.y;
    }
}

};
