/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    レイヤ処理関連
*/

#include <string.h>

#include "drawdat.h"

#include "global.h"

#include "CMainWin.h"
#include "CLayerWin.h"
#include "CProgressDlg.h"

#include "CLayerList.h"
#include "CLayerItem.h"
#include "CTileImage.h"
#include "CTileImageRGBA.h"
#include "CUndo.h"

#include "draw_main.h"
#include "draw_update.h"
#include "draw_opfunc.h"

#include "AXFile.h"


namespace draw
{

//----------------------------

typedef struct
{
    const AXString *pstrFname;
    CTileImage  *pimg;
    CLayerItem  *pItem;
    BOOL    bAPLD;
}THREAD_LOADIMG;

typedef struct
{
    CTileImage  *pimg,*pimgSrc;
    BOOL        bLum;
}THREAD_CHANGECOLTYPE;

typedef struct
{
    CTileImage  *pimg,*pimgSrc;
    int     srcopa,dstopa;
    AXRect  rc;
    void (*funcCol)(RGBAFIX15 *,const RGBFIX15 &);
}THREAD_COMBINE;

typedef struct
{
    CTileImage  *pimgDst;
    CLayerItem  *pFirst;
    int     type;
}THREAD_COMBINESOME;

//----------------------------


//! カレントレイヤ変更時

void layer_changeCurrent(CLayerItem *p)
{
    if(p != g_draw->pcurlayer)
    {
        CLayerItem *pBk = g_draw->pcurlayer;

        g_draw->pcurlayer = p;

        LAYERWIN->updateLayerItem(pBk);
        LAYERWIN->updateCurLayer(TRUE);
    }
}

//! レイヤ表示/非表示反転

void layer_revShow(CLayerItem *p)
{
    FLAGRECT rcf;

    if(p->isVisibleFlag())
    {
        //表示 -> 非表示

        p->getVisibleImgRect(&rcf);

        p->m_dwFlags ^= CLayerItem::FLAG_VISIBLE;

        updateRectAndPrev(rcf);
    }
    else
    {
        //非表示 -> 表示

        p->m_dwFlags ^= CLayerItem::FLAG_VISIBLE;

        updateLayerItemRect(p);
    }
}

//! フォルダの展開反転

void layer_revFolderExpand(CLayerItem *p)
{
    p->m_dwFlags ^= CLayerItem::FLAG_EXPAND;

    //表示数が変わるので再計算
    g_draw->player->calcViewItemCnt();

    LAYERWIN->updateLayerAll();
}

//! 合成モード変更

void layer_changeBlendMode(int no)
{
    g_draw->pcurlayer->m_nBlendMode = no;

    updateLayerItemRect(g_draw->pcurlayer);
    LAYERWIN->updateCurLayer(FALSE);
}

//! 不透明度変更

void layer_changeOpacity(int opacity)
{
    g_draw->pcurlayer->m_nOpacity = (int)(opacity * 128 / 100.0 + 0.5);

    updateLayerItemRect(g_draw->pcurlayer);
    LAYERWIN->updateCurLayer(FALSE);
}

//! ロックフラグ反転

void layer_revLockFlag(CLayerItem *p)
{
    p->m_dwFlags ^= CLayerItem::FLAG_LOCK;

    //undo

    g_draw->pundo->add_layerFlags(p, CLayerItem::FLAG_LOCK);

    //更新

    if(p == g_draw->pcurlayer)
        LAYERWIN->updateCurLayer(TRUE);
    else
        LAYERWIN->updateLayerItem(p);
}

//! 塗りつぶし判定元フラグ反転

void layer_revPaintFlag(CLayerItem *p)
{
    if(p->isPaint())
    {
        p->m_dwFlags ^= CLayerItem::FLAG_PAINT;
        LAYERWIN->updateLayerItem(p);
    }
    else
    {
        //判定元は一つだけ

        g_draw->player->flagOffAll(~CLayerItem::FLAG_PAINT);
        p->m_dwFlags |= CLayerItem::FLAG_PAINT;
        LAYERWIN->updateLayerAll();
    }
}

//! リンクフラグ反転

void layer_revLinkFlag(CLayerItem *p)
{
    p->m_dwFlags ^= CLayerItem::FLAG_LINK;
    LAYERWIN->updateLayerItem(p);
}

//! チェックフラグ反転

void layer_revCheckFlag(CLayerItem *p)
{
    p->m_dwFlags ^= CLayerItem::FLAG_CHECK;
    LAYERWIN->updateLayerItem(p);
}

//! レイヤマスクフラグ変更
/*!
    @param type [0]自動反転、[1]下レイヤマスクON/OFF、[2]マスクレイヤON/OFF
*/

void layer_changeLayerMask(CLayerItem *p,int type)
{
    DWORD flag_off;
    BOOL f;

    flag_off = p->m_dwFlags & ~(CLayerItem::FLAG_LAYERMASK | CLayerItem::FLAG_LAYERMASKUNDER);

    //

    if(type != 2 && (p->isLayerMaskUnder() || type == 1))
    {
        //下レイヤをマスク ON/OFF

        if(p->isLayerMaskUnder())
            p->m_dwFlags = flag_off;
        else
            p->m_dwFlags = flag_off | CLayerItem::FLAG_LAYERMASKUNDER;

        LAYERWIN->updateLayerItem(p);
    }
    else
    {
        //マスクレイヤ ON/OFF

        f = !p->isLayerMask();

        g_draw->player->flagOffAll(~CLayerItem::FLAG_LAYERMASK);

        if(f) p->m_dwFlags = flag_off | CLayerItem::FLAG_LAYERMASK;

        LAYERWIN->updateLayerAll();
    }
}

//! レイヤの線の色変更

void layer_changeCol(CLayerItem *p,DWORD col)
{
    p->setLayerCol(col);

    updateLayerItemRect(p);
    LAYERWIN->updateLayerItem(p);
}


//===================================
// コマンド
//===================================


//! 新規レイヤ

void layer_new(const AXString &strName,int nBlendMode,int nColType,DWORD dwCol)
{
    CLayerItem *p;

    //追加

    p = g_draw->player->addLayer(g_draw->pcurlayer, nColType, NULL,
            g_draw->nImgW, g_draw->nImgH);

    if(!p) return;

    p->m_strName    = strName;
    p->m_nBlendMode = nBlendMode;

    p->setLayerCol(dwCol);

    //カレントレイヤ

    g_draw->pcurlayer = p;

    //UNDO・更新

    g_draw->pundo->add_layerNew();

    LAYERWIN->updateLayerAll();
}

//! 新規フォルダ

void layer_newFolder(const AXString &strName)
{
    CLayerItem *p;

    //追加

    p = g_draw->player->addLayer(g_draw->pcurlayer);
    if(!p) return;

    p->m_strName = strName;
    p->expand();

    //カレントレイヤ

    g_draw->pcurlayer = p;

    //UNDO・更新

    g_draw->pundo->add_layerNew();

    LAYERWIN->updateLayerAll();
}

// [スレッド] 画像ファイルから新規レイヤ

void *thread_layerFromImg(void *pParam)
{
    CProgressDlg *pdlg = (CProgressDlg *)pParam;
    THREAD_LOADIMG *pdat = (THREAD_LOADIMG *)pdlg->m_pParam1;
    int ret;

    if(pdat->bAPLD)
    {
        pdat->pItem = g_draw->player->addLayer_APLD(g_draw->pcurlayer, *(pdat->pstrFname), pdlg);
        ret = (pdat->pItem != NULL);
    }
    else
        ret = pdat->pimg->loadImage(*(pdat->pstrFname), NULL, pdlg);

    pdlg->endThread(ret);

    return NULL;
}

//! 画像ファイルから新規レイヤ

BOOL layer_fromImg(const AXString &filename)
{
    CProgressDlg *pDlg;
    CTileImage *pimg = NULL;
    CLayerItem *p;
    char head[8];
    BOOL bAPLD = FALSE;
    THREAD_LOADIMG info;

    //タイプ判別

    if(AXFile::readFile(filename, head, 8))
    {
        head[7] = 0;
        if(::strcmp(head, "AZPLLYD") == 0) bAPLD = TRUE;
    }

    //読み込み

    if(!bAPLD)
        pimg = new CTileImageRGBA;

    info.pstrFname = &filename;
    info.pimg      = pimg;
    info.bAPLD     = bAPLD;

    pDlg = new CProgressDlg(MAINWIN, thread_layerFromImg, &info);
    if(!pDlg->run()) goto ERR;

    //レイヤ

    if(bAPLD)
        p = info.pItem;
    else
    {
        p = g_draw->player->addLayer(g_draw->pcurlayer);
        if(!p) goto ERR;

        p->m_strName.path_filenameNoExt(filename);

        p->replaceImg(pimg);
    }

    //カレントレイヤ

    g_draw->pcurlayer = p;

    //UNDO

    g_draw->pundo->add_layerNew();

    return TRUE;

ERR:
    if(pimg) delete pimg;
    return FALSE;
}

//! カレントレイヤを複製

void layer_copy()
{
    CLayerItem *p;

    //フォルダは不可

    if(g_draw->pcurlayer->isFolder()) return;

    //複製

    p = g_draw->player->copyLayer(g_draw->pcurlayer);
    if(!p) return;

    //カレントレイヤ

    g_draw->pcurlayer = p;

    //UNDO

    g_draw->pundo->add_layerCopy();

    //更新

    updateLayerItemRect(p);
    LAYERWIN->updateLayerAll();
}

//! カレントレイヤ削除
/*
    フォルダの場合、フォルダ下のアイテムすべて削除
*/

void layer_delete(BOOL bUpdate)
{
    FLAGRECT rcf;
    CLayerItem *pCur;

    //削除してはいけない場合
    /* 親がルートで、前後にレイヤがない場合。そのレイヤがフォルダで子がある場合も含む。 */

    pCur = g_draw->pcurlayer;

    if(!pCur->parent() && !pCur->prev() && !pCur->next())
        return;

    //UNDO・更新範囲

    if(bUpdate)
    {
        g_draw->pundo->add_layerDel();

        g_draw->pcurlayer->getVisibleImgRect(&rcf);
    }

    //削除後のカレントレイヤ

    pCur = (CLayerItem *)g_draw->pcurlayer->nextTreeItemPass();

    if(!pCur) pCur = (CLayerItem *)g_draw->pcurlayer->prevTreeItem();

    //削除

    g_draw->player->deleteLayer(g_draw->pcurlayer);

    //カレントレイヤ

    g_draw->pcurlayer = pCur;

    //更新

    if(bUpdate)
    {
        updateRectAndPrev(rcf);
        LAYERWIN->updateLayerAll();
    }
}

//! カレントレイヤをクリア

void layer_clear()
{
    FLAGRECT rcf;

    if(checkDrawLayer()) return;

    g_draw->pundo->add_layerClear();

    g_draw->pcurlayer->getVisibleImgRect(&rcf);

    g_draw->pcurlayer->m_pimg->create(g_draw->nImgW, g_draw->nImgH);

    updateRectAndPrev(rcf);
}

//

void *thread_layerChangeColType(void *pParam)
{
    CProgressDlg *pdlg = (CProgressDlg *)pParam;
    THREAD_CHANGECOLTYPE *pdat = (THREAD_CHANGECOLTYPE *)pdlg->m_pParam1;
    int ret;

    ret = pdat->pimg->createConvert(*(pdat->pimgSrc), pdat->bLum, pdlg);

    pdlg->endThread(ret);

    return NULL;
}

//! カラータイプ変更

BOOL layer_changeColType(CLayerItem *pItem,int type,BOOL bLum)
{
    CTileImage *pimg;
    FLAGRECT rcf;
    CProgressDlg *pdlg;
    THREAD_CHANGECOLTYPE dat;

    //イメージ変換

    pimg = allocTileImage(type);

    dat.pimg    = pimg;
    dat.pimgSrc = pItem->m_pimg;
    dat.bLum    = bLum;

    pdlg = new CProgressDlg(MAINWIN, thread_layerChangeColType, &dat);

    if(!pdlg->run())
    {
        delete pimg;
        return FALSE;
    }

    //UNDO

    g_draw->pundo->add_layerColType(pItem, type, bLum);

    //イメージ置換

    pItem->getVisibleImgRect(&rcf);

    pItem->replaceImg(pimg);

    //更新

    updateRectAndPrev(rcf);
    LAYERWIN->updateLayerItem(pItem);

    return TRUE;
}


//============================
// 結合
//============================


void *thread_layerCombine(void *pParam)
{
    CProgressDlg *pdlg = (CProgressDlg *)pParam;
    THREAD_COMBINE *pdat = (THREAD_COMBINE *)pdlg->m_pParam1;

    pdlg->setProgMax(30);

    pdat->pimg->combine(*(pdat->pimgSrc), pdat->rc, pdat->srcopa, pdat->dstopa, pdat->funcCol, pdlg, 30);

    pdlg->endThread(1);

    return NULL;
}

//! 下レイヤに結合/移す

void layer_combine(BOOL bDrop)
{
    CLayerItem *pcur,*punder;
    FLAGRECT rcf,rcfDst;
    THREAD_COMBINE info;
    CProgressDlg *pdlg;

    //結合元、結合先（フォルダは除く）

    pcur = g_draw->pcurlayer;
    if(pcur->isFolder()) return;

    punder = pcur->next();
    if(!punder) return;

    if(punder->isFolder()) return;

    //イメージのある範囲

    pcur->m_pimg->getExistImgRectPx(&rcf);
    punder->m_pimg->getExistImgRectPx(&rcfDst);

    //処理する範囲
    /*
        レイヤ結合の場合、結合先のレイヤ不透明度をイメージに適用させた後、
        レイヤ不透明度を 100% に変更するので、結合先も処理範囲に含める。
    */

    if(!bDrop) rcf.combine(rcfDst);

    //ドロップで元イメージがない場合
    if(bDrop && !rcf.flag) return;

    //UNDO

    g_draw->pundo->add_layerCombine(bDrop);

    //イメージ結合

    if(rcf.flag)
    {
        info.pimg    = punder->m_pimg;
        info.pimgSrc = pcur->m_pimg;
        info.srcopa  = pcur->m_nOpacity;
        info.dstopa  = (bDrop)? 128: punder->m_nOpacity;
        info.funcCol = g_draw->funcBlendCol[(bDrop)? 0: pcur->m_nBlendMode];

        rcf.toRect(&info.rc);

        pdlg = new CProgressDlg(MAINWIN, thread_layerCombine, &info);
        pdlg->run();
    }

    //

    if(bDrop)
        //カレントレイヤをクリア
        pcur->m_pimg->create(g_draw->nImgW, g_draw->nImgH);
    else
    {
        punder->m_nOpacity = 128;

        //カレントを削除
        layer_delete(FALSE);

        LAYERWIN->updateLayerAll();
    }

    //

    updateRectAndPrev(rcf);
}

//(スレッド)複数レイヤの結合

void *thread_layerCombineSome(void *pParam)
{
    CProgressDlg *pdlg = (CProgressDlg *)pParam;
    THREAD_COMBINESOME *pdat = (THREAD_COMBINESOME *)pdlg->m_pParam1;
    CLayerItem *p;
    int cnt;
    FLAGRECT rcf;
    AXRect rc;
    BOOL ret = TRUE;

    //結合数

    for(p = pdat->pFirst, cnt = 0; p; p = p->m_pLink, cnt++);

    //結合

    pdlg->setProgMax(cnt * 10);

    for(p = pdat->pFirst; p; p = p->m_pLink)
    {
        if(p->m_pimg->getExistImgRectPx(&rcf))
        {
            rcf.toRect(&rc);

            if(!pdat->pimgDst->combine(*(p->m_pimg), rc, p->m_nOpacity, 128,
                    g_draw->funcBlendCol[p->m_nBlendMode], pdlg, 10))
            {
                ret = FALSE;
                break;
            }
        }
    }

    pdlg->endThread(ret);

    return NULL;
}

//! 複数レイヤの結合

BOOL layer_combinesome(int type,BOOL bNewLayer,int coltype)
{
    CTileImage *pimg;
    CProgressDlg *pdlg;
    THREAD_COMBINESOME info;
    CLayerItem *pFirst,*pNew,*p;

    //リンクをセット（結合レイヤが一つもない場合あり）

    pFirst = g_draw->player->setLinkCombineSome(type, g_draw->pcurlayer);

    //結合用イメージ作成

    pimg = allocTileImage(coltype);

    if(!pimg->create(g_draw->nImgW, g_draw->nImgH)) goto ERR;

    //pimg にイメージ結合

    if(pFirst)
    {
        info.pimgDst = pimg;
        info.type    = type;
        info.pFirst  = pFirst;

        pdlg = new CProgressDlg(MAINWIN, thread_layerCombineSome, &info);
        if(!pdlg->run()) goto ERR;
    }

    //UNDO

    if(!bNewLayer)
        g_draw->pundo->add_layerCombineMul(type);

    //各タイプごとの処理

    switch(type)
    {
        //フォルダ内（カレントの上に）
        case 0:
            pNew = g_draw->player->addLayer(NULL);

            g_draw->player->move(pNew, g_draw->pcurlayer);

            pNew->m_strName = g_draw->pcurlayer->m_strName;

            if(!bNewLayer) layer_delete(FALSE);
            break;
        //すべて（一番上に）
        case 1:
            if(!bNewLayer) g_draw->player->clearItem();

            pNew = g_draw->player->addLayer(NULL);

            g_draw->player->move(pNew, g_draw->player->getTopItem());

            pNew->m_strName = "combine";
            break;
        //チェック（チェックの一番上のレイヤの上に）
        default:
            for(p = pFirst; p && p->m_pLink; p = p->m_pLink);

            pNew = g_draw->player->addLayer(p);

            pNew->m_strName = "combine";
            break;
    }

    //

    pNew->replaceImg(pimg);

    g_draw->pcurlayer = pNew;

    //UNDO

    if(bNewLayer)
        g_draw->pundo->add_layerNew();

    //更新

    updateAllAndLayer();

    return TRUE;

ERR:
    delete pimg;
    return FALSE;
}

//(スレッド) 画像の統合

void *thread_layerUnite(void *pParam)
{
    CProgressDlg *pdlg = (CProgressDlg *)pParam;
    int ret;

    //pimgBlend に合成してそれを元に作成

    blendImage();

    ret = ((CTileImage *)pdlg->m_pParam1)->createFromImg(*(g_draw->pimgBlend), pdlg);

    pdlg->endThread(ret);

    return NULL;
}

//! 画像の統合

BOOL layer_unite()
{
    CTileImage *pimg;
    CProgressDlg *pdlg;

    //統合イメージ作成

    pimg = allocTileImage(CLayerItem::COLTYPE_RGBA);

    pdlg = new CProgressDlg(MAINWIN, thread_layerUnite, pimg);

    if(!pdlg->run())
    {
        delete pimg;
        updateImage();  //チェック柄表示時はイメージを元に戻す

        return FALSE;
    }

    //UNDO (複数レイヤすべて結合と同じ)

    g_draw->pundo->add_layerCombineMul(1);

    //レイヤ全削除

    g_draw->player->clearItem();

    //統合後レイヤ

    g_draw->pcurlayer = g_draw->player->addLayer(NULL);

    g_draw->pcurlayer->m_strName = "layer0";
    g_draw->pcurlayer->replaceImg(pimg);

    //

    updateAllAndLayer();

    return TRUE;
}


//============================
// 編集
//============================


//! 左右/上下反転
/*
    ロックの判定は reverseHorzVert() 内で行っている。
*/

void layer_revHorzVert(BOOL bHorz)
{
    FLAGRECT rcf;

    if(g_draw->pcurlayer->reverseHorzVert(bHorz, &rcf))
    {
        g_draw->pundo->add_layerHVRev(bHorz);

        updateRectAndPrev(rcf);
    }
}

//! 90度回転

void layer_rotate90(BOOL bLeft)
{
    FLAGRECT rcf;

    if(g_draw->pcurlayer->rotate90(bLeft, &rcf))
    {
        g_draw->pundo->add_layerRotate90(bLeft);

        updateRectAndPrev(rcf);
    }
}


//============================


//! D＆Dでの位置移動
/*
    pDst : NULL で一番下
    type : 1 でフォルダ
*/

void layer_moveDND(CLayerItem *pDst,int type)
{
    CLayerItem *pcur,*pParentBk,*pPrevBk;
    int bkno[2];

    pcur      = g_draw->pcurlayer;
    pParentBk = pcur->parent();
    pPrevBk   = pcur->prev();

    g_draw->player->getLayerItemNo_parent(pcur, bkno);

    //移動

    if(!pDst)
        //一番下へ
        g_draw->player->moveLast(pcur, NULL);
    else if(type == 0)
        //pDst の上に移動
        g_draw->player->move(pcur, pDst);
    else
    {
        //----- フォルダ

        if(pDst->m_pFirst)
            //フォルダ下の先頭
            g_draw->player->move(pcur, pDst->first());
        else
            //フォルダに子がない場合
            g_draw->player->moveLast(pcur, pDst);

        //展開させる

        pDst->expand();

        g_draw->player->calcViewItemCnt();
    }

    //位置が変わっていない

    if(pcur->parent() == pParentBk && pcur->prev() == pPrevBk)
        return;

    //更新

    g_draw->pundo->add_layerMove(pcur, bkno);

    updateLayerItemRect(pcur);
    LAYERWIN->updateLayerAll();
}

//! カレントレイヤを上下に移動

void layer_moveUpDown(BOOL bUp)
{
    CLayerItem *p = g_draw->pcurlayer;
    int bkno[2];

    g_draw->player->getLayerItemNo_parent(g_draw->pcurlayer, bkno);

    //

    if(bUp)
    {
        if(!p->prev()) return;

        g_draw->player->move(p, p->prev());
    }
    else
    {
        p = p->next();
        if(!p) return;

        p = p->next();

        if(p)
            g_draw->player->move(g_draw->pcurlayer, p);
        else
            g_draw->player->moveLast(g_draw->pcurlayer, g_draw->pcurlayer->parent());
    }

    //更新

    g_draw->pundo->add_layerMove(g_draw->pcurlayer, bkno);

    updateLayerItemRect(g_draw->pcurlayer);
    LAYERWIN->updateLayerAll();
}

//! すべて表示/非表示/カレントのみ
/*!
    @param type [0]非表示 [1]表示 [2]カレントのみ
*/

void layer_showAll(int type)
{
    CLayerItem *p;

    for(p = g_draw->player->getTopItem(); p; p = (CLayerItem *)p->nextTreeItem())
    {
        if(type == 1)
            p->m_dwFlags |= CLayerItem::FLAG_VISIBLE;
        else
            p->m_dwFlags &= ~CLayerItem::FLAG_VISIBLE;
    }

    if(type == 2)
        g_draw->pcurlayer->visibleOn();

    updateAllAndLayer();
}

//! チェックレイヤの表示反転

void layer_showCheckRev()
{
    CLayerItem *p;
    FLAGRECT rcf,rcf2;

    rcf.clear();

    for(p = g_draw->player->getTopItem(); p; p = (CLayerItem *)p->nextTreeItem())
    {
        if(p->isCheck())
        {
            if(p->m_pimg->getExistImgRectPx(&rcf2))
                rcf.combine(rcf2);

            p->m_dwFlags ^= CLayerItem::FLAG_VISIBLE;
        }
    }

    //更新

    updateRectAndPrev(rcf);
    LAYERWIN->updateLayerAll();
}

//! 通常レイヤを表示反転

void layer_showToggleNormal()
{
    CLayerItem *p;
    FLAGRECT rcf,rcf2;

    rcf.clear();

    for(p = g_draw->player->getTopItem(); p; p = (CLayerItem *)p->nextTreeItem())
    {
        if(p->isNormal())
        {
            if(p->m_pimg->getExistImgRectPx(&rcf2))
                rcf.combine(rcf2);

            p->m_dwFlags ^= CLayerItem::FLAG_VISIBLE;
        }
    }

    //更新

    updateRectAndPrev(rcf);
    LAYERWIN->updateLayerAll();
}

//! レイヤの選択位置を上下に移動

void layer_curSelUpDown(BOOL bUp)
{
    CLayerItem *p = g_draw->pcurlayer;

    if(bUp)
        p = p->prevExpand();
    else
        p = p->nextExpand();

    if(p)
    {
        g_draw->pcurlayer = p;

        LAYERWIN->updateShowCur();
    }
}

};
