/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    ADW(AzDrawing) 読み込み
*/


#include "CProgressDlg.h"

#include "CImageRGB16.h"
#include "CTileImage.h"
#include "CLayerList.h"
#include "CLayerItem.h"

#include "AXFile.h"
#include "AXZlib.h"
#include "AXMem.h"
#include "AXUtil.h"

#include "draw_main.h"
#include "draw_file.h"

#include "drawdat.h"
#include "global.h"


namespace draw
{

//---------------------------------

int _loadADW_ver2(AXFile &file,CProgressDlg *pProgDlg);
int _loadADW_ver1(AXFile &file,CProgressDlg *pProgDlg);

//---------------------------------


//! ADW読み込み

int loadADW(const AXString &filename,CProgressDlg *pProgDlg)
{
    AXFile file;
    BYTE ver;
    int ret;

    //ファイル開く

    if(!file.openRead(filename)) return LOADERR_OPENFILE;

    file.setEndian(AXFile::ENDIAN_LITTLE);

    //ヘッダ

    if(!file.readCompare("AZDWDAT")) return LOADERR_FORMAT;

    //バージョン

    file.read(&ver, 1);

    //各バージョン処理

    if(ver == 0)
        ret = _loadADW_ver1(file, pProgDlg);
    else if(ver == 1)
        ret = _loadADW_ver2(file, pProgDlg);
    else
        ret = LOADERR_FORMAT;

    //

    file.close();

    return ret;
}

//! ver2 読み込み

int _loadADW_ver2(AXFile &file,CProgressDlg *pProgDlg)
{
    AXZlib zlib;
    CLayerItem *p;
    void *pTile;
    DWORD dw;
    WORD wSize,wWidth,wHeight,wLayerCnt,wLayerSel,wDPI,wLayerInfoSize,tx,ty;
    WORD wName[25];
    DWORD ti,dwTileCnt;
    BYTE bt;
    AXRect rc;
    TILEIMGINFO imginfo;
    int i,j;
    AXMem memBuf;
    LPBYTE pSrc;
    LPWORD pDst;

    //

    if(!memBuf.alloc(4096)) return LOADERR_ETC;

    if(!zlib.allocBuf(4096)) return LOADERR_ETC;
    if(!zlib.initDecBuf()) return LOADERR_ETC;

    //---------- プレビュースキップ

    file.seekCur(4);
    file.read(&dw, 4);
    file.seekCur(dw);

    //---------- 全体情報

    //データサイズ
    file.readWORD(&wSize);
    //キャンバス幅
    file.readWORD(&wWidth);
    //キャンバス高さ
    file.readWORD(&wHeight);
    //DPI
    file.readWORD(&wDPI);
    //レイヤ数
    file.readWORD(&wLayerCnt);
    //カレントレイヤNo
    file.readWORD(&wLayerSel);
    //レイヤ情報サイズ
    file.readWORD(&wLayerInfoSize);

    file.seekCur(wSize - 12);

    //---------- 新規イメージ

    if(!newImage(wWidth, wHeight, wDPI, FALSE)) return LOADERR_ETC;

    //----------- レイヤ

    p = NULL;

    pProgDlg->setProgMax(wLayerCnt * 10);

    for(i = wLayerCnt; i; i--)
    {
        //全体の範囲とタイル数

        if(!file.readDWORD(&rc.left)) return LOADERR_ETC;
        file.readDWORD(&rc.top);
        file.readDWORD(&rc.right);
        file.readDWORD(&rc.bottom);
        file.readDWORD(&dwTileCnt);

        //タイルイメージ情報

        if(rc.left >= rc.right || rc.top >= rc.bottom)
        {
            //全体が透明の状態

            imginfo.nOffX = imginfo.nOffY = 0;
            imginfo.nArrayW = imginfo.nArrayH = 1;
        }
        else
        {
            imginfo.nOffX   = rc.left;
            imginfo.nOffY   = rc.top;
            imginfo.nArrayW = (rc.right - rc.left + 63) / 64;
            imginfo.nArrayH = (rc.bottom - rc.top + 63) / 64;
        }

        //レイヤ追加 (A16)

        p = g_draw->player->addLayer(p, CLayerItem::COLTYPE_A16, &imginfo, 0, 0);
        if(!p) return LOADERR_ETC;

        //レイヤ情報

        file.read(wName, 50);

        file.read(&bt, 1);
        p->m_nOpacity = bt;

        file.readDWORD(&p->m_dwCol);
        p->setLayerCol(p->m_dwCol);

        file.read(&bt, 1);
        p->m_dwFlags = 0;
        if(bt & 1) p->m_dwFlags |= CLayerItem::FLAG_VISIBLE;

        file.seekCur(3);

        //名前

        for(j = 0; j < 25; j++)
            AXGetWORDLE((LPBYTE)(wName + j), wName + j);

        wName[24] = 0;
        p->m_strName = wName;

        //情報余分をシーク

        file.seekCur(wLayerInfoSize - 79);

        //タイル

        pProgDlg->beginProgSub(10, dwTileCnt);

        for(ti = dwTileCnt; ti; ti--)
        {
            if(!file.readWORD(&tx)) return LOADERR_ETC;
            file.readWORD(&ty);
            file.readWORD(&wSize);

            //タイル位置が範囲外 or 圧縮サイズが範囲外

            if(tx >= imginfo.nArrayW || ty >= imginfo.nArrayH || wSize == 0 || wSize > 4096)
                return LOADERR_ETC;

            //読み込み (memBuf に)

            if(wSize == 4096)
            {
                //無圧縮

                file.read(memBuf, 4096);
            }
            else
            {
                //ZIP圧縮

                file.read(zlib.getBuf(), wSize);

                if(!zlib.reset()) return LOADERR_ETC;

                if(!zlib.getDecBuf(memBuf, 4096, zlib.getBuf(), wSize))
                    return LOADERR_ETC;
            }

            //タイル作成

            pTile = p->m_pimg->getTileAlloc(tx, ty);
            if(!pTile) return LOADERR_ETC;

            //変換 (8bitA -> 16bitA)

            pSrc = memBuf;
            pDst = (LPWORD)pTile;

            for(j = 64 * 64; j; j--, pSrc++)
                *(pDst++) = _8TO16COL(*pSrc);

            //

            pProgDlg->incProg();
        }
    }

    //カレントレイヤ

    g_draw->pcurlayer = g_draw->player->getLayerItemFromNo(wLayerSel);

    return LOADERR_SUCCESS;
}

//! ADW ver1 読み込み

int _loadADW_ver1(AXFile &file,CProgressDlg *pProgDlg)
{
    CLayerItem *p;
    AXZlib zlib;
    DWORD dwSize;
    WORD wd,wLayerCnt,wLayerSel;
    BYTE bt;
    char name[32];
    int i,x,y,pitch,w,h;
    LPBYTE pSrc,pTmpBuf;
    RGBAFIX15 col;

    //

    if(!zlib.allocBuf(16 * 1024)) return LOADERR_ETC;

    //------- ヘッダデータ

    //幅・高さ

    file.readWORD(&wd);
    w = wd;

    file.readWORD(&wd);
    h = wd;

    //レイヤ数

    file.readWORD(&wLayerCnt);

    //レイヤ通算カウント

    file.seekCur(2);

    //カレントレイヤNo

    file.readWORD(&wLayerSel);

    //圧縮タイプ

    file.read(&bt, 1);
    if(bt != 0) return LOADERR_FORMAT;  //ゼロ圧縮は対象外

    //-------- プレビューイメージ

    file.readDWORD(&dwSize);
    file.seekCur(dwSize);

    //---------- 新規イメージ

    if(!newImage(w, h, -1, FALSE)) return LOADERR_ETC;

    //---------- レイヤ

    p       = NULL;
    pTmpBuf = (LPBYTE)g_draw->pimgBlend->getBuf();
    pitch   = (w + 3) & (~3);

    col.zero();

    pProgDlg->setProgMax(wLayerCnt * 10);

    for(i = wLayerCnt; i; i--)
    {
        //レイヤ追加 (A16)

        p = g_draw->player->addLayer(p, CLayerItem::COLTYPE_A16, NULL, w, h);
        if(!p) return LOADERR_ETC;

        //情報

        file.read(name, 32);
        p->m_strName.setUTF8(name, 31);

        file.read(&bt, 1);
        p->m_nOpacity = bt;

        file.read(&bt, 1);
        p->m_dwFlags = 0;
        if(bt & 1) p->m_dwFlags |= CLayerItem::FLAG_VISIBLE;

        file.readDWORD(&p->m_dwCol);

        p->setLayerCol(p->m_dwCol);

        //イメージ読み込み

        file.readDWORD(&dwSize);

        if(!zlib.initDecFile(&file, dwSize)) return LOADERR_ETC;

        pProgDlg->beginProgSub(10, h);

        for(y = 0; y < h; y++)
        {
            if(!zlib.getDecFile(pTmpBuf, pitch)) return LOADERR_ETC;

            pSrc = pTmpBuf;

            for(x = 0; x < w; x++, pSrc++)
            {
                if(*pSrc)
                {
                    col.a = _8TO16COL(*pSrc);

                    p->m_pimg->setPixel_create(x, y, col);
                }
            }

            pProgDlg->incProg();
        }

        zlib.end();
    }

    //カレントレイヤ

    g_draw->pcurlayer = g_draw->player->getLayerItemFromNo(wLayerSel);

    return LOADERR_SUCCESS;
}

};
