/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/

#include <math.h>
#include <stdlib.h>

#include "drawdat.h"
#include "draw_calc.h"

#include "CTileImage.h"
#include "CConfig.h"
#include "global.h"


namespace draw
{

//! AXRectSize、範囲追加（x == -1 で範囲なし）
/*!
    @return 範囲があるか（両方共範囲なしで FALSE）
*/

BOOL unionRectSize(AXRectSize *pdst,const AXRectSize &src)
{
    //両方とも範囲なし

    if(pdst->x < 0 && src.x < 0) return FALSE;

    //srcが範囲なし（そのまま）

    if(pdst->x >= 0 && src.x < 0) return TRUE;

    //pdstが範囲なし（srcをコピー）

    if(pdst->x < 0 && src.x >= 0)
    {
        *pdst = src;
        return TRUE;
    }

    //両方範囲あり

    pdst->combine(src);

    return TRUE;
}

//! FLAGRECT のイメージ範囲を範囲内に収めて AXRectSize へ

BOOL getImgRect(AXRectSize *prcs,const FLAGRECT &src)
{
    if(src.flag)
        return adjustImgRect(prcs, src.x1, src.y1, src.x2, src.y2);
    else
        return FALSE;
}

//! イメージ座標の範囲内に収めて AXRectSize へ
/*
    ※ x1,y1 が左上、x2,y2 が右上になっていること
*/

BOOL adjustImgRect(AXRectSize *prcs,int x1,int y1,int x2,int y2)
{
    //範囲外

    if(x2 < 0 || y2 < 0 || x1 >= g_draw->nImgW || y1 >= g_draw->nImgH) return FALSE;

    //調整

    if(x1 < 0) x1 = 0;
    if(y1 < 0) y1 = 0;
    if(x2 >= g_draw->nImgW) x2 = g_draw->nImgW - 1;
    if(y2 >= g_draw->nImgH) y2 = g_draw->nImgH - 1;

    //セット

    prcs->x	= x1;
    prcs->y = y1;
    prcs->w = x2 - x1 + 1;
    prcs->h = y2 - y1 + 1;

    return TRUE;
}

//! ウィンドウの範囲からイメージの範囲を取得
/*!
    @return FALSE でイメージ範囲外
*/

BOOL winToimgRect(AXRectSize *pdst,const AXRectSize &rcs)
{
    AXPoint pt[4],ptMin,ptMax;
    int i;

    //四隅をイメージ位置に変換

    g_draw->view.winToimg(pt    , rcs.x - 1    , rcs.y - 1);
    g_draw->view.winToimg(pt + 1, rcs.x + rcs.w, rcs.y - 1);
    g_draw->view.winToimg(pt + 2, rcs.x - 1    , rcs.y + rcs.h);
    g_draw->view.winToimg(pt + 3, rcs.x + rcs.w, rcs.y + rcs.h);

    //矩形

    ptMin = ptMax = pt[0];

    for(i = 1; i < 4; i++)
    {
        if(pt[i].x < ptMin.x) ptMin.x = pt[i].x;
        if(pt[i].y < ptMin.y) ptMin.y = pt[i].y;
        if(pt[i].x > ptMax.x) ptMax.x = pt[i].x;
        if(pt[i].y > ptMax.y) ptMax.y = pt[i].y;
    }

    //調整

    return adjustImgRect(pdst, ptMin.x - 1, ptMin.y - 1, ptMax.x + 1, ptMax.y + 1);
}


//=============================
// 描画関連
//=============================


//! 直線を45度単位に調整
/*!
    @param pdst  調整したい位置
    @param st    始点
*/

void getPtLine45(AXPoint *pdst,const AXPoint &st)
{
    int dx,dy,adx,ady,n;

    dx = pdst->x - st.x;
    dy = pdst->y - st.y;

    adx = ::abs(dx);
    ady = ::abs(dy);

    n = (adx > ady)? adx: ady;

    if(::abs(adx - ady) < n / 2)
    {
        //45度線

        if(adx < ady)
            pdst->y = st.y + ((dy < 0)? -adx: adx);
        else
            pdst->x = st.x + ((dx < 0)? -ady: ady);
    }
    else if(adx > ady)
        //水平
        pdst->y = st.y;
    else
        //垂直
        pdst->x = st.x;
}

//! [直線/連続直線/集中線] 描画位置取得
/*!
    ptTmp[0],[1] にウィンドウ位置

    @param prc  描画イメージ位置(double)
*/

void getDrawLineRect(DRECT *prc)
{
    g_draw->view.winToimg(&prc->x1, &prc->y1, g_draw->work.ptTmp[0].x, g_draw->work.ptTmp[0].y);
    g_draw->view.winToimg(&prc->x2, &prc->y2, g_draw->work.ptTmp[1].x, g_draw->work.ptTmp[1].y);
}

//! 四角枠描画の位置(4点)取得
/*
    ptTmp[0],[1] にウィンドウ位置
*/

void getDrawBoxPoint(DPOINT *pdst)
{
    g_draw->view.winToimg(&pdst[0].x, &pdst[0].y, g_draw->work.ptTmp[0].x, g_draw->work.ptTmp[0].y);
    g_draw->view.winToimg(&pdst[1].x, &pdst[1].y, g_draw->work.ptTmp[1].x, g_draw->work.ptTmp[0].y);
    g_draw->view.winToimg(&pdst[2].x, &pdst[2].y, g_draw->work.ptTmp[1].x, g_draw->work.ptTmp[1].y);
    g_draw->view.winToimg(&pdst[3].x, &pdst[3].y, g_draw->work.ptTmp[0].x, g_draw->work.ptTmp[1].y);
}

//! 四角描画の範囲取得

void getDrawBox(AXRectSize *prcs)
{
    AXPoint pt[4],ptMin,ptMax;
    int i;

    g_draw->view.winToimg(pt    , g_draw->work.ptTmp[0].x, g_draw->work.ptTmp[0].y);
    g_draw->view.winToimg(pt + 1, g_draw->work.ptTmp[1].x, g_draw->work.ptTmp[0].y);
    g_draw->view.winToimg(pt + 2, g_draw->work.ptTmp[1].x, g_draw->work.ptTmp[1].y);
    g_draw->view.winToimg(pt + 3, g_draw->work.ptTmp[0].x, g_draw->work.ptTmp[1].y);

    //矩形

    ptMin = ptMax = pt[0];

    for(i = 1; i < 4; i++)
    {
        if(pt[i].x < ptMin.x) ptMin.x = pt[i].x;
        if(pt[i].y < ptMin.y) ptMin.y = pt[i].y;
        if(pt[i].x > ptMax.x) ptMax.x = pt[i].x;
        if(pt[i].y > ptMax.y) ptMax.y = pt[i].y;
    }

    //

    prcs->x = ptMin.x;
    prcs->y = ptMin.y;
    prcs->w = ptMax.x - ptMin.x + 1;
    prcs->h = ptMax.y - ptMin.y + 1;
}

//! オフセット位置移動時の計算

BOOL calcMoveOffset(CTileImage *pimg,BOOL bHorz,BOOL bVert,AXPoint *pptMove)
{
    AXPoint pt,pt2,ptMov;
    double x,y;

    //ウィンドウ位置移動数

    x = g_draw->work.ptWinNow.x - g_draw->work.ptWinBk.x;
    y = g_draw->work.ptWinNow.y - g_draw->work.ptWinBk.y;

    if(bVert) x = 0;
    if(bHorz) y = 0;

    g_draw->work.dptTmp[0].x += x;
    g_draw->work.dptTmp[0].y += y;

    //ウィンドウ -> イメージ

    pt2.x = (int)g_draw->work.dptTmp[0].x;
    pt2.y = (int)g_draw->work.dptTmp[0].y;

    g_draw->view.getMoveDst(&pt, pt2);

    pt += g_draw->work.ptTmp[0];

    //オフセット位置、前回からの移動数

    pimg->getOffset(&ptMov);

    ptMov.x = pt.x - ptMov.x;
    ptMov.y = pt.y - ptMov.y;

    if(ptMov.x == 0 && ptMov.y == 0)
        return FALSE;
    else
    {
        g_draw->work.ptTmp[1] += ptMov;

        *pptMove = ptMov;

        return TRUE;
    }
}


//=============================
// ほか
//=============================


//! キャンバス内にイメージ全体が収まるように表示倍率取得

int getScaleWithinCanvas()
{
    AXRectSize rcs;
    int w,h,s;

    //上下左右に余白10pxつける

    if(g_draw->view.szCanvas.w < 20 || g_draw->view.szCanvas.h < 20)
        w = h = 20;
    else
    {
        w = g_draw->view.szCanvas.w - 20;
        h = g_draw->view.szCanvas.h - 20;
    }

    //

    rcs.set(0, 0, g_draw->nImgW, g_draw->nImgH);
    rcs.inBoxKeepAspect(w, h, TRUE);

    //倍率

    s = (int)((double)rcs.w / g_draw->nImgW * 1000 + 0.5);

    if(s < SCALE_MIN) s = SCALE_MIN;
    else if(s > SCALE_MAX) s = SCALE_MAX;

    return s;
}

//! キャンバスウィンドウのスクロールバーの最大値取得
/*!
    四隅をイメージ座標 -> ウィンドウ座標変換し、一番大きい半径の値＋ウィンドウサイズ
*/

void getCanvasScrollMax(AXSize *psize)
{
    AXPoint pt[4];
    int i,n,max = 1;
    double x,y;

    //イメージ座標 -> ウィンドウ座標

    pt[0].x = pt[0].y = 0;
    pt[1].x = g_draw->nImgW, pt[1].y = 0;
    pt[2].x = 0,             pt[2].y = g_draw->nImgH;
    pt[3].x = g_draw->nImgW, pt[3].y = g_draw->nImgH;

    for(i = 0; i < 4; i++)
    {
        g_draw->view.imgTowin(pt + i, pt[i].x, pt[i].y);

        x = pt[i].x - (g_draw->view.szCanvas.w >> 1);
        y = pt[i].y - (g_draw->view.szCanvas.h >> 1);

        n = (int)(::sqrt(x * x + y * y) + 0.5);

        if(n > max) max = n;
    }

    //

    psize->w = max * 2 + g_draw->view.szCanvas.w;
    psize->h = max * 2 + g_draw->view.szCanvas.h;
}

//! 1段階上げ/下げした表示倍率取得

int getScaleUpDown(BOOL bUp)
{
    int n,up,down;

    n    = g_draw->view.nScale / 10;
    up   = g_conf->nScaleUpStep;
    down = g_conf->nScaleDownStep;

    if((!bUp && n <= 100) || (bUp && n < 100))
    {
        //〜100%

        if(bUp)
        {
            n = 100 - (100 - n - 1) / down * down;
            if(n > 100) n = 100;
        }
        else
            n = 100 - (100 - n + down) / down * down;
    }
    else
    {
        //100%〜

        if(bUp)
            n = (n + up) / up * up;
        else
        {
            n = (n - 1) / up * up;
            if(n < 100) n = 100;
        }
    }

    //

    n *= 10;

    if(n < SCALE_MIN) n = SCALE_MIN;
    else if(n > SCALE_MAX) n = SCALE_MAX;

    return n;
}

};
