/*****************************************************************************
 *  Copyright (C) 2012-2015 Azel.
 *
 *  This file is part of AzXClass.
 *
 *  This library is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this library.  If not, see <http://www.gnu.org/licenses/>.
******************************************************************************/

#include "AXExpander.h"

#include "AXLayout.h"
#include "AXFont.h"
#include "AXApp.h"
#include "AXAppRes.h"


/*!
    @class AXExpander
    @brief 展開/非展開の状態があるウィジェット

    - 子ウィンドウをヘッダの下に表示する。
    - 非展開時はウィンドウサイズをヘッダ部分のみにすることで、下を隠す。
    - レイアウトは AXExpander にセットし、内部のアイテムは、この子供にする。
    - デフォルトで、AXExpander の子ウィンドウの通知先は、AXExpander の通知先と同じになる。

    @ingroup widget
*/
/*!
    @var AXExpander::EXPS_BORDER_TOP
    @brief 上に境界線を描画する
    @var AXExpander::EXPN_TOGGLE
    @brief 展開/非展開が切り替えられた
*/


AXExpander::~AXExpander()
{

}

AXExpander::AXExpander(AXWindow *pParent,UINT uStyle,UINT uLayoutFlags)
    : AXWindow(pParent, uStyle, uLayoutFlags)
{
    _createExpander();
}

AXExpander::AXExpander(AXWindow *pParent,UINT uStyle,UINT uLayoutFlags,UINT uItemID,DWORD dwPadding)
    : AXWindow(pParent, uStyle, uLayoutFlags, uItemID, dwPadding)
{
    _createExpander();
}

AXExpander::AXExpander(AXWindow *pParent,UINT uStyle,UINT uLayoutFlags,UINT uItemID,DWORD dwPadding,LPCUSTR pstr)
    : AXWindow(pParent, uStyle, uLayoutFlags, uItemID, dwPadding)
{
    _createExpander();

    m_strText = pstr;
}

AXExpander::AXExpander(AXWindow *pParent,UINT uStyle,UINT uLayoutFlags,UINT uItemID,DWORD dwPadding,WORD wStrID)
    : AXWindow(pParent, uStyle, uLayoutFlags, uItemID, dwPadding)
{
    _createExpander();

    m_strText = _str(wStrID);
}

//! 作成

void AXExpander::_createExpander()
{
    m_uFlags |= AXWindow::FLAG_CHILD_NOTIFY_PARENT;
    m_bExpand = FALSE;
}

//! ラベルセット

void AXExpander::setText(const AXString &str)
{
    m_strText = str;
}

//! 内側の余白セット

void AXExpander::setInsidePadding(int left,int top,int right,int bottom)
{
    m_rcPadding.set(left, top, right, bottom);
}

//! 展開/非展開セット

void AXExpander::expand(BOOL bExpand)
{
    m_bExpand = (bExpand != 0);

    if(isLayouted())
        _changeExpand();
}

//! 標準サイズ計算

void AXExpander::calcDefSize()
{
    int w;

    if(m_bExpand)
    {
        //レイアウト全体のサイズ

        AXWindow::calcDefSize();

        //余白

        m_nDefW += m_rcPadding.left + m_rcPadding.right;
        m_nDefH += m_rcPadding.top + m_rcPadding.bottom;
    }
    else
        m_nDefW = m_nDefH = 0;

    //上部

    m_nHeaderH = m_pFont->getHeight();
    if(m_nHeaderH < 7) m_nHeaderH = 7;

    w = m_pFont->getTextWidth(m_strText) + 14;

    m_nDefW = (w < m_nDefW)? m_nDefW: w;
    m_nDefH += m_nHeaderH;

    if(m_uStyle & EXPS_BORDER_TOP) m_nDefH += 2;
}

//! 範囲取得

void AXExpander::getClientRect(AXRect *prc)
{
    prc->left   = m_rcPadding.left;
    prc->top    = m_nHeaderH + m_rcPadding.top;
    prc->right  = m_nW - m_rcPadding.right - 1;
    prc->bottom = m_nH - m_rcPadding.bottom - 1;

    if(m_uStyle & EXPS_BORDER_TOP) prc->top += 2;
}

//! 展開変更時

void AXExpander::_changeExpand()
{
    //サイズを変更

    calcDefSize();
    resize(m_nW, m_nDefH);
    redraw();

    //

    getNotify()->sendNotify(this, EXPN_TOGGLE, 0);
}

//! ボタン押し時

BOOL AXExpander::onButtonDown(AXHD_MOUSE *phd)
{
    if(phd->button == BUTTON_LEFT && phd->y < m_nHeaderH)
    {
        m_bExpand = !m_bExpand;

        _changeExpand();
    }

    return TRUE;
}

//! 描画

BOOL AXExpander::onPaint(AXHD_PAINT *phd)
{
    int y = 0;

    //背景

    drawFillBox(0, 0, m_nW, m_nH, AXAppRes::FACE);

    //ヘッダー背景

    if(m_uStyle & EXPS_HEADER_DARK)
        drawFillBox(0, 0, m_nW, m_nHeaderH, AXAppRes::FACEDARK);

    //上の線

    if(m_uStyle & EXPS_BORDER_TOP)
    {
        drawLineH(0, 0, m_nW, AXAppRes::FRAMEOUTSIDE);
        y = 2;
    }

    //矢印

    if(m_bExpand)
        drawArrowDown7(4, y + ((m_nHeaderH - 4) >> 1), AXAppRes::TEXTNORMAL);
    else
        drawArrowRight7(2, y + (m_nHeaderH >> 1), AXAppRes::TEXTNORMAL);

    //ラベル

    if(m_strText.isNoEmpty())
    {
        //テキスト

        AXDrawText dt(m_id);

        dt.draw(*m_pFont, 7 + 5, y, m_strText);

        dt.end();
    }

    return TRUE;
}
