/*
 * Asterisk -- A telephony toolkit for Linux.
 *
 * MySQL CDR logger 
 * 
 * James Sharp <jsharp@psychoses.org>
 *
 * Modified August 2003
 * Tilghman Lesher <asterisk__cdr__cdr_mysql__200308@the-tilghman.com>
 *
 * Modified August 6, 2005
 * Joseph Benden <joe@thrallingpenguin.com>
 * Added mysql connection timeout parameter
 * Added an automatic reconnect as to not lose a cdr record
 * Cleaned up the original code to match the coding guidelines
 *
 * Modified Juli 2006
 * Martin Portmann <map@infinitum.ch>
 * Added mysql ssl support
 *
 * This program is free software, distributed under the terms of
 * the GNU General Public License.
 *
 */

/*** MODULEINFO
	<depend>mysqlclient</depend>
 ***/

#include <asterisk.h>

#include <sys/types.h>
#include <asterisk/config.h>
#include <asterisk/options.h>
#include <asterisk/channel.h>
#include <asterisk/cdr.h>
#include <asterisk/module.h>
#include <asterisk/logger.h>
#include <asterisk/cli.h>
#include <asterisk/strings.h>
#include <asterisk/linkedlists.h>

#include <stdio.h>
#include <string.h>

#include <stdlib.h>
#include <unistd.h>
#include <time.h>

#include <mysql/mysql.h>
#include <mysql/errmsg.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <errno.h>

#define DATE_FORMAT "%Y-%m-%d %T"
static char *desc = "MySQL CDR Backend";
static char *name = "mysql";
static char *config = "cdr_mysql.conf";

static struct ast_str *hostname = NULL, *dbname = NULL, *dbuser = NULL, *password = NULL, *dbsock = NULL, *dbtable = NULL, *dbcharset = NULL;

static struct ast_str *ssl_ca = NULL, *ssl_cert = NULL, *ssl_key = NULL;

static int dbport = 0;
static int connected = 0;
static time_t connect_time = 0;
static int records = 0;
static int totalrecords = 0;
static int timeout = 0;
static int calldate_compat = 0;

AST_MUTEX_DEFINE_STATIC(mysql_lock);

struct unload_string {
	AST_LIST_ENTRY(unload_string) entry;
	struct ast_str *str;
};

static AST_LIST_HEAD_STATIC(unload_strings, unload_string);

struct column {
	char *name;
	char *cdrname;
	char *type;
	AST_LIST_ENTRY(column) list;
};

/* Protected with mysql_lock */
static AST_RWLIST_HEAD_STATIC(columns, column);

static MYSQL mysql = { { NULL }, };

static char *handle_cli_cdr_mysql_status(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "cdr mysql status";
		e->usage =
			"Usage: cdr mysql status\n"
			"       Shows current connection status for cdr_mysql\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != 3)
		return CLI_SHOWUSAGE;

	if (connected) {
		char status[256], status2[100] = "";
		int ctime = time(NULL) - connect_time;
		if (dbport)
			snprintf(status, 255, "Connected to %s@%s, port %d", dbname->str, hostname->str, dbport);
		else if (dbsock)
			snprintf(status, 255, "Connected to %s on socket file %s", dbname->str, S_OR(dbsock->str, "default"));
		else
			snprintf(status, 255, "Connected to %s@%s", dbname->str, hostname->str);

		if (!ast_strlen_zero(dbuser->str))
			snprintf(status2, 99, " with username %s", dbuser->str);
		if (!ast_strlen_zero(dbtable->str))
			snprintf(status2, 99, " using table %s", dbtable->str);
		if (ctime > 31536000) {
			ast_cli(a->fd, "%s%s for %d years, %d days, %d hours, %d minutes, %d seconds.\n", status, status2, ctime / 31536000, (ctime % 31536000) / 86400, (ctime % 86400) / 3600, (ctime % 3600) / 60, ctime % 60);
		} else if (ctime > 86400) {
			ast_cli(a->fd, "%s%s for %d days, %d hours, %d minutes, %d seconds.\n", status, status2, ctime / 86400, (ctime % 86400) / 3600, (ctime % 3600) / 60, ctime % 60);
		} else if (ctime > 3600) {
			ast_cli(a->fd, "%s%s for %d hours, %d minutes, %d seconds.\n", status, status2, ctime / 3600, (ctime % 3600) / 60, ctime % 60);
		} else if (ctime > 60) {
			ast_cli(a->fd, "%s%s for %d minutes, %d seconds.\n", status, status2, ctime / 60, ctime % 60);
		} else {
			ast_cli(a->fd, "%s%s for %d seconds.\n", status, status2, ctime);
		}
		if (records == totalrecords)
			ast_cli(a->fd, "  Wrote %d records since last restart.\n", totalrecords);
		else
			ast_cli(a->fd, "  Wrote %d records since last restart and %d records since last reconnect.\n", totalrecords, records);
	} else {
		ast_cli(a->fd, "Not currently connected to a MySQL server.\n");
	}

	return CLI_SUCCESS;
}

static struct ast_cli_entry cdr_mysql_status_cli[] = {
	AST_CLI_DEFINE(handle_cli_cdr_mysql_status, "Show connection status of cdr_mysql"),
};

static int mysql_log(struct ast_cdr *cdr)
{
	char *sql1 = ast_calloc(1, 4096), *sql2 = ast_calloc(1, 2048);
	int sql1size = 4096, sql2size = 2048;
	int retries = 5;
#if MYSQL_VERSION_ID >= 50013
	my_bool my_bool_true = 1;
#endif

	if (!sql1 || !sql2) {
		ast_log(LOG_ERROR, "Memory error\n");
		return -1;
	}

	ast_mutex_lock(&mysql_lock);

db_reconnect:
	if ((!connected) && (hostname || dbsock) && dbuser && password && dbname && dbtable ) {
		/* Attempt to connect */
		mysql_init(&mysql);
		/* Add option to quickly timeout the connection */
		if (timeout && mysql_options(&mysql, MYSQL_OPT_CONNECT_TIMEOUT, (char *)&timeout) != 0) {
			ast_log(LOG_ERROR, "mysql_options returned (%d) %s\n", mysql_errno(&mysql), mysql_error(&mysql));
		}
#if MYSQL_VERSION_ID >= 50013
		/* Add option for automatic reconnection */
		if (mysql_options(&mysql, MYSQL_OPT_RECONNECT, &my_bool_true) != 0) {
			ast_log(LOG_ERROR, "mysql_options returned (%d) %s\n", mysql_errno(&mysql), mysql_error(&mysql));
		}
#endif
		if (ssl_ca || ssl_cert || ssl_key) {
			mysql_ssl_set(&mysql, ssl_key ? ssl_key->str : NULL, ssl_cert ? ssl_cert->str : NULL, ssl_ca ? ssl_ca->str : NULL, NULL, NULL);
		}
		if (mysql_real_connect(&mysql, hostname->str, dbuser->str, password->str, dbname->str, dbport, dbsock && !ast_strlen_zero(dbsock->str) ? dbsock->str : NULL, ssl_ca ? CLIENT_SSL : 0)) {
			connected = 1;
			connect_time = time(NULL);
			records = 0;
			if (dbcharset) {
				snprintf(sql1, sizeof(sql1), "SET NAMES '%s'", dbcharset->str);
				mysql_real_query(&mysql, sql1, strlen(sql1));
				ast_debug(1, "SQL command as follows: %s\n", sql1);
			}
		} else {
			ast_log(LOG_ERROR, "Cannot connect to database server %s: (%d) %s\n", hostname->str, mysql_errno(&mysql), mysql_error(&mysql));
			connected = 0;
		}
	} else {
		/* Long connection - ping the server */
		int error;
		if ((error = mysql_ping(&mysql))) {
			connected = 0;
			records = 0;
			switch (mysql_errno(&mysql)) {
				case CR_SERVER_GONE_ERROR:
				case CR_SERVER_LOST:
					ast_log(LOG_ERROR, "Server has gone away. Attempting to reconnect.\n");
					break;
				default:
					ast_log(LOG_ERROR, "Unknown connection error: (%d) %s\n", mysql_errno(&mysql), mysql_error(&mysql));
			}
			retries--;
			if (retries) {
				goto db_reconnect;
			} else {
				ast_log(LOG_ERROR, "Retried to connect five times, giving up.\n");
			}
		}
	}

	if (connected) {
		int column_count = 0;
		char *cdrname;
		char workspace[2048], *value = NULL, *ptr;
		int sql2len;
		struct column *entry;

		snprintf(sql1, sql1size, "INSERT INTO %s (", dbtable ? dbtable->str : "cdr");
		strcpy(sql2, ") VALUES ('");

		AST_RWLIST_RDLOCK(&columns);
		AST_RWLIST_TRAVERSE(&columns, entry, list) {
			if (!strcmp(entry->name, "calldate")) {
				/*!\note
				 * For some dumb reason, "calldate" used to be formulated using
				 * the datetime the record was posted, rather than the start
				 * time of the call.  If someone really wants the old compatible
				 * behavior, it's provided here.
				 */
				if (calldate_compat) {
					struct timeval tv = ast_tvnow();
					struct ast_tm tm;
					char timestr[128];
					ast_localtime(&tv, &tm, NULL);
					ast_strftime(timestr, sizeof(timestr), "%Y-%m-%d %T", &tm);
					ast_cdr_setvar(cdr, "calldate", timestr, 0);
					cdrname = "calldate";
				} else {
					cdrname = "start";
				}
			} else {
				cdrname = entry->cdrname;
			}

			/* Construct SQL */
			if (strlen(sql1) + 2 + strlen(entry->name) > sql1size) {
				char *tmp = ast_realloc(sql1, sql1size * 2);
				if (!tmp)
					goto log_exit;
				sql1size *= 2;
				sql1 = tmp;
			}

			/* Need the type and value to determine if we want the raw value or not */
			if ((!strcmp(cdrname, "start") ||
				 !strcmp(cdrname, "answer") ||
				 !strcmp(cdrname, "end") ||
				 !strcmp(cdrname, "disposition") ||
				 !strcmp(cdrname, "amaflags")) &&
				(strstr(entry->type, "int") ||
				 strstr(entry->type, "dec") ||
				 strstr(entry->type, "float") ||
				 strstr(entry->type, "double") ||
				 strstr(entry->type, "real") ||
				 strstr(entry->type, "numeric") ||
				 strstr(entry->type, "fixed")))
				ast_cdr_getvar(cdr, cdrname, &value, workspace, sizeof(workspace), 0, 1);
			else
				ast_cdr_getvar(cdr, cdrname, &value, workspace, sizeof(workspace), 0, 0);

			if (strlen(sql2) + (value ? strlen(value) * 2 : 0) + 4 > sql2size) {
				char *tmp = ast_realloc(sql2, sql2size * 2);
				if (!tmp)
					goto log_exit;
				sql2size *= 2;
				sql2 = tmp;
			}
			if (value) {
				if (column_count++) {
					strcat(sql1, ",");
					strcat(sql2, "','");
				}

				strcat(sql1, entry->name);

				/*!\note We're manually escaping here, to ensure that we know exactly
				 * how much space is used.  Since we only accept ASCII strings at this
				 * point in time, there is no danger in this simplistic escape method,
				 * but I wouldn't recommend this technique for other databases or if
				 * we convert to an internal representation of UTF-8 sometime in the
				 * future.
				 */
				sql2len = strlen(sql2);
				for (ptr = value; *ptr; ptr++) {
					if (*ptr == '\\' || *ptr == '\'')
						sql2[sql2len++] = '\\';
					sql2[sql2len++] = *ptr;
				}
				sql2[sql2len] = '\0';
			}
		}
		AST_RWLIST_UNLOCK(&columns);

		ast_debug(1, "Inserting a CDR record.\n");
		if (strlen(sql1) + 3 + strlen(sql2) > sql1size) {
			char *tmp = ast_realloc(sql1, strlen(sql1) + 3 + strlen(sql2));
			if (!tmp)
				goto log_exit;
			sql1 = tmp;
		}
		strcat(sql1, sql2);
		strcat(sql1, "')");

		ast_debug(1, "SQL command as follows: %s\n", sql1);

		if (mysql_real_query(&mysql, sql1, strlen(sql1))) {
			ast_log(LOG_ERROR, "Failed to insert into database: (%d) %s\n", mysql_errno(&mysql), mysql_error(&mysql));
			mysql_close(&mysql);
			connected = 0;
		} else {
			records++;
			totalrecords++;
		}
	}
log_exit:
	ast_free(sql1);
	ast_free(sql2);
	ast_mutex_unlock(&mysql_lock);
	return 0;
}

static int my_unload_module(int reload)
{ 
	struct unload_string *us;
	struct column *entry;

	ast_cli_unregister_multiple(cdr_mysql_status_cli, sizeof(cdr_mysql_status_cli) / sizeof(struct ast_cli_entry));

	if (connected) {
		mysql_close(&mysql);
		connected = 0;
		records = 0;
	}

	AST_LIST_LOCK(&unload_strings);
	while ((us = AST_LIST_REMOVE_HEAD(&unload_strings, entry))) {
		ast_free(us->str);
		ast_free(us);
	}
	AST_LIST_UNLOCK(&unload_strings);

	if (!reload) {
		AST_RWLIST_WRLOCK(&columns);
	}
	while ((entry = AST_RWLIST_REMOVE_HEAD(&columns, list))) {
		ast_free(entry);
	}
	if (!reload) {
		AST_RWLIST_UNLOCK(&columns);
	}

	dbport = 0;
	ast_cdr_unregister(name);
	
	return 0;
}

static int my_load_config_string(struct ast_config *cfg, const char *category, const char *variable, struct ast_str **field, const char *def)
{
	struct unload_string *us;
	const char *tmp;

	if (!(us = ast_calloc(1, sizeof(*us))))
		return -1;

	if (!(*field = ast_str_create(16))) {
		ast_free(us);
		return -1;
	}

	us->str = *field;

	AST_LIST_LOCK(&unload_strings);
	AST_LIST_INSERT_HEAD(&unload_strings, us, entry);
	AST_LIST_UNLOCK(&unload_strings);
	
	tmp = ast_variable_retrieve(cfg, category, variable);

	ast_str_set(field, 0, "%s", tmp ? tmp : def);

	return 0;
}

static int my_load_config_number(struct ast_config *cfg, const char *category, const char *variable, int *field, int def)
{
	const char *tmp;

	tmp = ast_variable_retrieve(cfg, category, variable);

	if (!tmp || sscanf(tmp, "%d", field) < 1)
		*field = def;

	return 0;
}

static int my_load_module(int reload)
{
	int res;
	struct ast_config *cfg;
	struct ast_variable *var;
	struct ast_flags config_flags = { reload ? CONFIG_FLAG_FILEUNCHANGED : 0 };
	struct column *entry;
	char *temp;
	struct ast_str *compat;
	MYSQL_ROW row;
	MYSQL_RES *result;
	char sqldesc[128];
#if MYSQL_VERSION_ID >= 50013
	my_bool my_bool_true = 1;
#endif

	cfg = ast_config_load(config, config_flags);
	if (!cfg) {
		ast_log(LOG_WARNING, "Unable to load config for mysql CDR's: %s\n", config);
		return AST_MODULE_LOAD_SUCCESS;
	} else if (cfg == CONFIG_STATUS_FILEUNCHANGED)
		return AST_MODULE_LOAD_SUCCESS;

	if (reload) {
		AST_RWLIST_WRLOCK(&columns);
		my_unload_module(1);
	}

	var = ast_variable_browse(cfg, "global");
	if (!var) {
		/* nothing configured */
		if (reload) {
			AST_RWLIST_UNLOCK(&columns);
		}
		return AST_MODULE_LOAD_SUCCESS;
	}

	res = 0;

	res |= my_load_config_string(cfg, "global", "hostname", &hostname, "localhost");
	res |= my_load_config_string(cfg, "global", "dbname", &dbname, "astriskcdrdb");
	res |= my_load_config_string(cfg, "global", "user", &dbuser, "root");
	res |= my_load_config_string(cfg, "global", "sock", &dbsock, "");
	res |= my_load_config_string(cfg, "global", "table", &dbtable, "cdr");
	res |= my_load_config_string(cfg, "global", "password", &password, "");

	res |= my_load_config_string(cfg, "global", "charset", &dbcharset, "");

	res |= my_load_config_string(cfg, "global", "ssl_ca", &ssl_ca, "");
	res |= my_load_config_string(cfg, "global", "ssl_cert", &ssl_cert, "");
	res |= my_load_config_string(cfg, "global", "ssl_key", &ssl_key, "");

	res |= my_load_config_number(cfg, "global", "port", &dbport, 0);
	res |= my_load_config_number(cfg, "global", "timeout", &timeout, 0);
	res |= my_load_config_string(cfg, "global", "compat", &compat, "no");
	if (ast_true(compat->str)) {
		calldate_compat = 1;
	} else {
		calldate_compat = 0;
	}

	if (res < 0) {
		if (reload) {
			AST_RWLIST_UNLOCK(&columns);
		}
		return AST_MODULE_LOAD_FAILURE;
	}

	/* Check for any aliases */
	if (!reload) {
		/* Lock, if not already */
		AST_RWLIST_WRLOCK(&columns);
	}
	while ((entry = AST_LIST_REMOVE_HEAD(&columns, list))) {
		ast_free(entry);
	}

	ast_debug(1, "Got hostname of %s\n", hostname->str);
	ast_debug(1, "Got port of %d\n", dbport);
	ast_debug(1, "Got a timeout of %d\n", timeout);
	if (dbsock)
		ast_debug(1, "Got sock file of %s\n", dbsock->str);
	ast_debug(1, "Got user of %s\n", dbuser->str);
	ast_debug(1, "Got dbname of %s\n", dbname->str);
	ast_debug(1, "Got password of %s\n", password->str);
	ast_debug(1, "%sunning in calldate compatibility mode\n", calldate_compat ? "R" : "Not r");

	if (dbcharset) {
		ast_debug(1, "Got DB charste of %s\n", dbcharset->str);
	}

	mysql_init(&mysql);

	if (timeout && mysql_options(&mysql, MYSQL_OPT_CONNECT_TIMEOUT, (char *)&timeout) != 0) {
		ast_log(LOG_ERROR, "cdr_mysql: mysql_options returned (%d) %s\n", mysql_errno(&mysql), mysql_error(&mysql));
	}

#if MYSQL_VERSION_ID >= 50013
	/* Add option for automatic reconnection */
	if (mysql_options(&mysql, MYSQL_OPT_RECONNECT, &my_bool_true) != 0) {
		ast_log(LOG_ERROR, "cdr_mysql: mysql_options returned (%d) %s\n", mysql_errno(&mysql), mysql_error(&mysql));
	}
#endif

	if ((ssl_ca && !ast_strlen_zero(ssl_ca->str)) || (ssl_cert && !ast_strlen_zero(ssl_cert->str)) || (ssl_key && !ast_strlen_zero(ssl_key->str))) {
		mysql_ssl_set (&mysql, ssl_key->str, ssl_cert->str, ssl_ca->str, NULL, NULL);
	}
	temp = dbsock && !ast_strlen_zero(dbsock->str) ? dbsock->str : NULL;
	if (!mysql_real_connect(&mysql, hostname->str, dbuser->str, password->str, dbname->str, dbport, temp, ssl_ca && !ast_strlen_zero(ssl_ca->str) ? CLIENT_SSL : 0)) {
		ast_log(LOG_ERROR, "Failed to connect to mysql database %s on %s.\n", dbname->str, hostname->str);
		connected = 0;
		records = 0;
	} else {
		ast_debug(1, "Successfully connected to MySQL database.\n");
		connected = 1;
		records = 0;
		connect_time = time(NULL);
		if (dbcharset) {
			snprintf(sqldesc, sizeof(sqldesc), "SET NAMES '%s'", dbcharset->str);
			mysql_real_query(&mysql, sqldesc, strlen(sqldesc));
			ast_debug(1, "SQL command as follows: %s\n", sqldesc);
		}

		/* Get table description */
		snprintf(sqldesc, sizeof(sqldesc), "DESC %s", dbtable ? dbtable->str : "cdr");
		if (mysql_query(&mysql, sqldesc)) {
			ast_log(LOG_ERROR, "Unable to query table description!!  Logging disabled.\n");
			mysql_close(&mysql);
			connected = 0;
			AST_RWLIST_UNLOCK(&columns);
			ast_config_destroy(cfg);
			return AST_MODULE_LOAD_SUCCESS;
		}

		if (!(result = mysql_store_result(&mysql))) {
			ast_log(LOG_ERROR, "Unable to query table description!!  Logging disabled.\n");
			mysql_close(&mysql);
			connected = 0;
			AST_RWLIST_UNLOCK(&columns);
			ast_config_destroy(cfg);
			return AST_MODULE_LOAD_SUCCESS;
		}

		while ((row = mysql_fetch_row(result))) {
			struct column *entry;
			int foundalias = 0;

			ast_debug(1, "Got a field '%s' of type '%s'\n", row[0], row[1]);
			/* Check for an alias */
			for (var = ast_variable_browse(cfg, "aliases"); var; var = var->next) {
				if (strcasecmp(var->value, row[0])) {
					continue;
				}

				if (!(entry = ast_calloc(1, sizeof(*entry) + strlen(var->name) + 1 + strlen(var->value) + 1 + strlen(row[1]) + 1))) {
					continue;
				}

				entry->cdrname = (char *)entry + sizeof(*entry);
				entry->name = (char *)entry + sizeof(*entry) + strlen(var->name) + 1;
				entry->type = (char *)entry + sizeof(*entry) + strlen(var->name) + 1 + strlen(var->value) + 1;
				strcpy(entry->cdrname, var->name);
				strcpy(entry->name, var->value);
				strcpy(entry->type, row[1]);

				AST_LIST_INSERT_TAIL(&columns, entry, list);
				ast_log(LOG_NOTICE, "Found an alias from CDR variable %s to DB column %s, type %s\n", entry->cdrname, entry->name, entry->type);
				foundalias = 1;
				break;
			}

			if (!foundalias && (entry = ast_calloc(1, sizeof(*entry) + strlen(row[0]) + 1 + strlen(row[1]) + 1))) {
				entry->cdrname = (char *)entry + sizeof(*entry);
				entry->name = (char *)entry + sizeof(*entry);
				entry->type = (char *)entry + sizeof(*entry) + strlen(row[0]) + 1;
				strcpy(entry->name, row[0]);
				strcpy(entry->type, row[1]);

				AST_LIST_INSERT_TAIL(&columns, entry, list);
				ast_log(LOG_NOTICE, "Found a DB column %s, type %s\n", entry->name, entry->type);
			}
		}
		mysql_free_result(result);
	}
	AST_RWLIST_UNLOCK(&columns);
	ast_config_destroy(cfg);

	res = ast_cdr_register(name, desc, mysql_log);
	if (res) {
		ast_log(LOG_ERROR, "Unable to register MySQL CDR handling\n");
	} else {
		res = ast_cli_register_multiple(cdr_mysql_status_cli, sizeof(cdr_mysql_status_cli) / sizeof(struct ast_cli_entry));
	}

	return res;
}

static int load_module(void)
{
	return my_load_module(0);
}

static int unload_module(void)
{
	return my_unload_module(0);
}

static int reload(void)
{
	int ret;

	ast_mutex_lock(&mysql_lock);
	ret = my_load_module(1);
	ast_mutex_unlock(&mysql_lock);

	return ret;
}

AST_MODULE_INFO(ASTERISK_GPL_KEY, AST_MODFLAG_DEFAULT, "MySQL CDR Backend",
	.load = load_module,
	.unload = unload_module,
	.reload = reload,
		);

