/* ANTINAT
 * =======
 * This software is Copyright (c) 2003-07 Malcolm Smith.
 * No warranty is provided, including but not limited to
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * This code is licenced subject to the GNU General
 * Public Licence (GPL).  See the COPYING file for more.
 */

#include "an_serv.h"
#include <antinat.h>
#ifdef HAVE_CRYPT_H
#include <crypt.h>
#endif
#ifdef HAVE_PWD_H
#include <pwd.h>
#endif
#ifdef HAVE_SHADOW_H
#include <shadow.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef ENABLE_OPEN_DIRECTORY

#ifdef HAVE_DIRECTORYSERVICE_DIRECTORYSERVICE_H
#include <CoreServices/CoreServices.h>
#include <DirectoryService/DirectoryService.h>
#elif defined(HAVE_DIRECTORYSERVICE_DIRSERVICES_H)
#include <DirectoryService/DirServices.h>
#include <DirectoryService/DirServicesConst.h>
#include <DirectoryService/DirServicesTypes.h>
#include <DirectoryService/DirServicesUtils.h>
#else
#error Cant build directory services support on this arch
#endif

/*
 ***** Parameters

 Set gOpenDirectoryAllowLocalUsersOnly to search only the local 
 node when authenticating.  That is, we'll only authenticate 
 users stored in on the local machine's directory and won't 
 consider users accessible via the authentication search 
 path, such as LDAP and Active Directory entries.
*/

static int gOpenDirectoryAllowLocalUsersOnly = 0;

/*
 Set TEST_BUFFER_DOUBLING to 1 to test the buffer doubling 
 functionality (see DoubleTheBufferSizeIfItsTooSmall).
*/

#define TEST_BUFFER_DOUBLING 0
#if TEST_BUFFER_DOUBLING
enum {
	kDefaultDSBufferSize = 1
};
#else
enum {
	kDefaultDSBufferSize = 1024
};
#endif

static tDirStatus
dsDataBufferAppendData (tDataBufferPtr buf,
						const void *dataPtr, size_t dataLen)
/*
    Appends a value to a data buffer.  dataPtr and dataLen describe 
    the value to append.  buf is the data buffer to which it's added.
    */
{
	tDirStatus err;

	ASSERT (buf != NULL);
	ASSERT (dataPtr != NULL);
	ASSERT (buf->fBufferLength <= buf->fBufferSize);

	if ((buf->fBufferLength + dataLen) > buf->fBufferSize) {
		err = eDSBufferTooSmall;
	} else {
		memcpy (&buf->fBufferData[buf->fBufferLength], dataPtr, dataLen);
		buf->fBufferLength += dataLen;
		err = eDSNoErr;
	}

	return err;
}

static tDirStatus
dsDataListAndHeaderDeallocate (tDirReference inDirReference,
							   tDataListPtr inDataList)
/*
    dsDataListDeallocate deallocates the list contents but /not/ the 
    list header (because the list header could be allocated on the 
    stack).  This routine is a wrapper that deallocates both.  It's 
    the logical opposite of the various list allocation routines 
    (for example, dsBuildFromPath).
    */
{
	tDirStatus err;

	ASSERT (inDirReference != 0);
	ASSERT (inDataList != NULL);

	err = dsDataListDeallocate (inDirReference, inDataList);
	if (err == eDSNoErr) {
		free (inDataList);
	}

	return err;
}

static void
DoubleTheBufferSizeIfItsTooSmall (tDirStatus * errPtr,
								  tDirNodeReference dirRef,
								  tDataBufferPtr * bufPtrPtr)
/*
    This routine is designed to handle the case where a 
    Open Directory routine returns eDSBufferTooSmall.  
    If so, it doubles the size of the buffer, allowing the 
    caller to retry the Open Directory routine with the 
    large buffer.
   
    errPtr is a pointer to a Open Directory error.  
    This routine does nothing unless that error is 
    eDSBufferTooSmall.  In that case it frees the buffer 
    referenced by *bufPtrPtr, replacing it with a buffer 
    of twice the size.  It then leaves *errPtr set to 
    eDSBufferTooSmall so that the caller retries the 
    call with the larger buffer.
    */
{
	tDirStatus err;
	tDirStatus junk;
	tDataBufferPtr tmpBuf;

	ASSERT (errPtr != NULL);
	ASSERT (dirRef != 0);
	ASSERT (bufPtrPtr != NULL);
	ASSERT (*bufPtrPtr != NULL);

	if (*errPtr == eDSBufferTooSmall) {
/*
         If the buffer size is already bigger than 16 MB, don't try to 
         double it again; something has gone horribly wrong.
	*/

		err = eDSNoErr;
		if ((*bufPtrPtr)->fBufferSize >= (16 * 1024 * 1024)) {
			err = eDSAllocationFailed;
		}

		if (err == eDSNoErr) {
			tmpBuf =
				dsDataBufferAllocate (dirRef, (*bufPtrPtr)->fBufferSize * 2);
			if (tmpBuf == NULL) {
				err = eDSAllocationFailed;
			} else {
#ifdef ENABLE_DEBUG
				DEBUG_LOG ("Open Directory Doubled buffer size.");
#endif

				junk = dsDataBufferDeAllocate (dirRef, *bufPtrPtr);
				ASSERT (junk == eDSNoErr);

				*bufPtrPtr = tmpBuf;
			}
		}

		/*
		   If err is eDSNoErr, the buffer expansion was successful 
		   so we leave *errPtr set to eDSBufferTooSmall.  If err 
		   is any other value, the expansion failed and we set 
		   *errPtr to that error.
		 */

		if (err != eDSNoErr) {
			*errPtr = err;
		}
	}
}

static tDirStatus
dsFindDirNodesQ (tDirReference inDirReference,
				 tDataBufferPtr * inOutDataBufferPtrPtr,
				 tDataListPtr inNodeNamePattern,
				 tDirPatternMatch inPatternMatchType,
				 unsigned long *outDirNodeCount,
				 tContextData * inOutContinueData)
/*
    A wrapper for dsFindDirNodes that handles two special cases:
    
    o If the routine returns eDSBufferTooSmall, it doubles the 
      size of the buffer referenced by *inOutDataBufferPtrPtr 
      and retries.
   
      Note that this change requires a change of the function 
      prototype; the second parameter is a pointer to a pointer 
      to the buffer, rather than just a pointer to the buffer. 
      This is so that I can modify the client's buffer pointer.
   
    o If the routine returns no nodes but there's valid continue data, 
      it retries.
   
    In other respects this works just like dsFindDirNodes.
    */
{
	tDirStatus err;

	/*
	   I only supply pre-conditions for the parameters that I touch.
	 */
	ASSERT (inOutDataBufferPtrPtr != NULL);
	ASSERT (*inOutDataBufferPtrPtr != NULL);
	ASSERT (outDirNodeCount != NULL);
	ASSERT (inOutContinueData != NULL);

	do {
		do {
			err = dsFindDirNodes (inDirReference,
								  *inOutDataBufferPtrPtr,
								  inNodeNamePattern,
								  inPatternMatchType,
								  outDirNodeCount, inOutContinueData);
			DoubleTheBufferSizeIfItsTooSmall (&err, inDirReference,
											  inOutDataBufferPtrPtr);
		} while (err == eDSBufferTooSmall);
	} while ((err == eDSNoErr) && (*outDirNodeCount == 0)
			 && (*inOutContinueData != NULL));

	return err;
}

static tDirStatus
dsGetRecordListQ (tDirNodeReference inDirReference,
				  tDirNodeReference inDirNodeReference,
				  tDataBufferPtr * inOutDataBufferPtr,
				  tDataListPtr inRecordNameList,
				  tDirPatternMatch inPatternMatchType,
				  tDataListPtr inRecordTypeList,
				  tDataListPtr inAttributeTypeList,
				  dsBool inAttributeInfoOnly,
				  unsigned long *inOutRecordEntryCount,
				  tContextData * inOutContinueData)
/*
     A wrapper for dsGetRecordList that handles two special cases:
    
     o If the routine returns eDSBufferTooSmall, it doubles the 
       size of the buffer referenced by *inOutDataBufferPtr 
       and retries.
    
      Note that this change requires a change of the function 
       prototype; the second parameter is a pointer to a pointer 
       to the buffer, rather than just a pointer to the buffer. 
       This is so that I can modify the client's buffer pointer.
    
     o If the routine returns no records but there's valid continue data, 
       it retries.
    
     In other respects this works just like dsGetRecordList.
     */
{
	tDirStatus err;
	unsigned long originalRecordCount;

	/*
	   I only supply pre-conditions for the parameters that I touch.
	 */
	ASSERT (inOutDataBufferPtr != NULL);
	ASSERT (*inOutDataBufferPtr != NULL);
	ASSERT (inOutRecordEntryCount != NULL);
	ASSERT (inOutContinueData != NULL);

	originalRecordCount = *inOutRecordEntryCount;

	do {
		*inOutRecordEntryCount = originalRecordCount;
		do {
			err = dsGetRecordList (inDirNodeReference,
								   *inOutDataBufferPtr,
								   inRecordNameList,
								   inPatternMatchType,
								   inRecordTypeList,
								   inAttributeTypeList,
								   inAttributeInfoOnly,
								   inOutRecordEntryCount, inOutContinueData);
			DoubleTheBufferSizeIfItsTooSmall (&err, inDirReference,
											  inOutDataBufferPtr);
		} while (err == eDSBufferTooSmall);
	} while ((err == eDSNoErr) && (*inOutRecordEntryCount == 0)
			 && (*inOutContinueData != NULL));

	return err;
}

static tDirStatus
dsDoDirNodeAuthQ (tDirNodeReference inDirReference,
				  tDirNodeReference inDirNodeReference,
				  tDataNodePtr inDirNodeAuthName,
				  dsBool inDirNodeAuthOnlyFlag,
				  tDataBufferPtr inAuthStepData,
				  tDataBufferPtr * outAuthStepDataResponsePtr,
				  tContextData * inOutContinueData)
/*
     A wrapper for dsDoDirNodeAuth that handles a special cases, 
     to wit, if dsDoDirNodeAuth returns eDSBufferTooSmall, it doubles 
     the size of the buffer referenced by *outAuthStepDataResponsePtr 
     and retries.
    
     Note that this change requires a change of the function 
     prototype; the second parameter is a pointer to a pointer 
     to the buffer, rather than just a pointer to the buffer. 
     This is so that I can modify the client's buffer pointer.
    
    In other respects this works just like dsDoDirNodeAuth.
    */
{
	tDirStatus err;

	ASSERT (outAuthStepDataResponsePtr != NULL);
	ASSERT (*outAuthStepDataResponsePtr != NULL);

	do {
		err = dsDoDirNodeAuth (inDirNodeReference,
							   inDirNodeAuthName,
							   inDirNodeAuthOnlyFlag,
							   inAuthStepData,
							   *outAuthStepDataResponsePtr,
							   inOutContinueData);
		DoubleTheBufferSizeIfItsTooSmall (&err, inDirReference,
										  outAuthStepDataResponsePtr);
	} while (err == eDSBufferTooSmall);

	return err;
}


static tDirStatus
GetSearchNodePathList (tDirReference dirRef,
					   tDataListPtr * searchNodePathListPtr)
	/*
	   Returns the path to the Open Directory search node.
	   dirRef is the connection to Open Directory.
	   On success, *searchNodePathListPtr is a data list that 
	   contains the search node's path components.
	 */
{
	tDirStatus err;
	tDirStatus junk;
	tDataBufferPtr buf;
	tDirPatternMatch patternToFind;
	unsigned long nodeCount;
	tContextData context;

	ASSERT (dirRef != 0);
	ASSERT (searchNodePathListPtr != NULL);
	ASSERT (*searchNodePathListPtr == NULL);

	if (gOpenDirectoryAllowLocalUsersOnly) {
		patternToFind = eDSLocalNodeNames;
	} else {
		patternToFind = eDSAuthenticationSearchNodeName;
	}

	context = NULL;

	/*
	   Allocate a buffer for the node find results.  We'll grow 
	   this buffer if it proves to be to small.
	 */

	buf = dsDataBufferAllocate (dirRef, kDefaultDSBufferSize);
	err = eDSNoErr;
	if (buf == NULL) {
		err = eDSAllocationFailed;
	}

	/*
	   Find the node.  Note that this is a degenerate case because 
	   we're only looking for a single node, the search node, so 
	   we don't need to loop calling dsFindDirNodes, which is the 
	   standard way of using dsFindDirNodes.
	 */

	if (err == eDSNoErr) {
		err = dsFindDirNodesQ (dirRef, &buf,	/* place results here */
							   NULL,	/* no pattern, rather... */
							   patternToFind,	/* ... hardwired search type */
							   &nodeCount, &context);
	}

	/*
	   If we didn't find any nodes, that's bad.
	 */

	if ((err == eDSNoErr) && (nodeCount < 1)) {
		err = eDSNodeNotFound;
	}

	/*
	   Grab the first node from the buffer.  Note that the inDirNodeIndex 
	   parameter to dsGetDirNodeName is one-based, so we pass in the constant 
	   1.

	   Also, if we found more than one, that's unusual, but not enough to 
	   cause us to error.
	 */

	if (err == eDSNoErr) {
#ifdef ENABLE_DEBUG
		if (nodeCount > 1) {
			DEBUG_LOG ("GetSearchNodePathList returned nodeCount>1");
		}
#endif
		err = dsGetDirNodeName (dirRef, buf, 1, searchNodePathListPtr);
	}

	/* Clean up. */

	if (context != NULL) {
		junk = dsReleaseContinueData (dirRef, context);
		ASSERT (junk == eDSNoErr);
	}
	if (buf != NULL) {
		junk = dsDataBufferDeAllocate (dirRef, buf);
		ASSERT (junk == eDSNoErr);
	}

	ASSERT ((err == eDSNoErr) == (*searchNodePathListPtr != NULL));

	return err;
}

static tDirStatus
FindUsersAuthInfo (tDirReference dirRef,
				   tDirNodeReference nodeRef,
				   const char *username,
				   tDataListPtr * pathListToAuthNodePtr,
				   char **userNameForAuthPtr)
/*
	Finds the authentication information for a given user. 
	dirRef is the connection to Open Directory.
	nodeRef is the node to use to do the searching.  Typically 
	this is the authentication search node, whose path is found 
	using GetSearchNodePathList.  username is the user whose 
	information we're looking for.
	//
	On success, *pathListToAuthNodePtr is a data list that 
	contain's the path components of the authentication node 
	for the specified user.
	On success, *userNameForAuthPtr contains a pointer to C string 
	that is the user's name for authentication.  This can be 
	different from username.  For example, if user's long name is 
	"Mr Gumby" and their short name is "mrgumby", username can be 
	either the long name or the short name, but *userNameForAuthPtr 
	will always be the short name.  The caller is responsible for 
	freeing this string using free.
*/
{
	tDirStatus err;
	tDirStatus junk;
	tDataBufferPtr buf;
	tDataListPtr recordType;
	tDataListPtr recordName;
	tDataListPtr requestedAttributes;
	unsigned long recordCount;
	tAttributeListRef foundRecAttrList;
	tContextData context;
	tRecordEntryPtr foundRecEntry;
	tDataListPtr pathListToAuthNode;
	char *userNameForAuth;

	ASSERT (dirRef != 0);
	ASSERT (nodeRef != 0);
	ASSERT (username != NULL);
	ASSERT (pathListToAuthNodePtr != NULL);
	ASSERT (*pathListToAuthNodePtr == NULL);
	ASSERT (userNameForAuthPtr != NULL);
	ASSERT (*userNameForAuthPtr == NULL);

	recordType = NULL;
	recordName = NULL;
	requestedAttributes = NULL;
	foundRecAttrList = 0;
	context = NULL;
	foundRecEntry = NULL;
	pathListToAuthNode = NULL;
	userNameForAuth = NULL;

/*
	Allocate a buffer for the record results.  We'll grow this 
	buffer if it proves to be too small.
*/

	err = eDSNoErr;
	buf = dsDataBufferAllocate (dirRef, kDefaultDSBufferSize);
	if (buf == NULL) {
		err = eDSAllocationFailed;
	}
/*
	Create the information needed for the search.  We're searching for 
	a record of type kDSStdRecordTypeUsers whose name is "username".  
	We want to get back the kDSNAttrMetaNodeLocation and kDSNAttrRecordName 
	attributes.
*/

	if (err == eDSNoErr) {
		recordType =
			dsBuildListFromStrings (dirRef, kDSStdRecordTypeUsers, NULL);
		recordName = dsBuildListFromStrings (dirRef, username, NULL);
		requestedAttributes =
			dsBuildListFromStrings (dirRef, kDSNAttrMetaNodeLocation,
									kDSNAttrRecordName, NULL);

		if ((recordType == NULL) || (recordName == NULL)
			|| (requestedAttributes == NULL)) {
			err = eDSAllocationFailed;
		}
	}
	/* Search for a matching record. */

	if (err == eDSNoErr) {
		recordCount = 1;		/* we only want one match (the first) */

		err = dsGetRecordListQ (dirRef,
								nodeRef,
								&buf,
								recordName,
								eDSExact,
								recordType,
								requestedAttributes,
								0, &recordCount, &context);
	}
	if ((err == eDSNoErr) && (recordCount < 1)) {
		err = eDSRecordNotFound;
	}
/*
	Get the first record from the search.  Then enumerate the attributes for 
	that record.  For each attribute, extract the first value (remember that 
	attributes can by multi-value).  Then see if the attribute is one that 
	we care about.  If it is, remember the value for later processing.
*/

	if (err == eDSNoErr) {
		ASSERT (recordCount == 1);	/* we only asked for one record, shouldn't get more back */

		err =
			dsGetRecordEntry (nodeRef, buf, 1, &foundRecAttrList,
							  &foundRecEntry);
	}
	if (err == eDSNoErr) {
		unsigned long attrIndex;

		/* Iterate over the attributes. */

		for (attrIndex = 1; attrIndex <= foundRecEntry->fRecordAttributeCount;
			 attrIndex++) {
			tAttributeValueListRef thisValue;
			tAttributeEntryPtr thisAttrEntry;
			tAttributeValueEntryPtr thisValueEntry;
			const char *thisAttrName;

			thisValue = 0;
			thisAttrEntry = NULL;
			thisValueEntry = NULL;

			/* Get the information for this attribute. */

			err =
				dsGetAttributeEntry (nodeRef, buf, foundRecAttrList,
									 attrIndex, &thisValue, &thisAttrEntry);

			if (err == eDSNoErr) {
				thisAttrName = thisAttrEntry->fAttributeSignature.fBufferData;

				/* We only care about attributes that have values. */

				if (thisAttrEntry->fAttributeValueCount > 0) {

/*
					Get the first value for this attribute.  This is common code for 
					the two potential attribute values listed below, so we do it first.
*/

					err =
						dsGetAttributeValue (nodeRef, buf, 1, thisValue,
											 &thisValueEntry);

					if (err == eDSNoErr) {
						const char *thisValueDataPtr;
						unsigned long thisValueDataLen;

						thisValueDataPtr =
							thisValueEntry->fAttributeValueData.fBufferData;
						thisValueDataLen =
							thisValueEntry->fAttributeValueData.fBufferLength;

						/* Handle each of the two attributes we care about; ignore any others. */

						if (strcmp (thisAttrName, kDSNAttrMetaNodeLocation) ==
							0) {
							ASSERT (pathListToAuthNode == NULL);	/* same attribute twice */

/*
							This is the kDSNAttrMetaNodeLocation attribute, which contains 
							a path to the node used for authenticating this record; convert 
							its value into a path list.
*/

							pathListToAuthNode = dsBuildFromPath (dirRef,
																  thisValueDataPtr,
																  "/");
							if (pathListToAuthNode == NULL) {
								err = eDSAllocationFailed;
							}
						} else if (strcmp (thisAttrName, kDSNAttrRecordName)
								   == 0) {
							ASSERT (userNameForAuth == NULL);	/* same attribute twice */

/*
							This is the kDSNAttrRecordName attribute, which contains the 
							user name used for authentication; remember its value in a
							freshly allocated string.
*/

							userNameForAuth =
								(char *) malloc (thisValueDataLen + 1);
							if (userNameForAuth == NULL) {
								err = eDSAllocationFailed;
							} else {
								memcpy (userNameForAuth,
										thisValueDataPtr, thisValueDataLen);
								userNameForAuth[thisValueDataLen] = 0;	/* terminating null */
							}
#ifdef ENABLE_DEBUG
						} else {
							DEBUG_LOG
								("FindUsersAuthInfo: Unexpected attribute.");
#endif
						}
					}
#ifdef ENABLE_DEBUG
				} else {
					DEBUG_LOG
						("FindUsersAuthInfo: Unexpected no-value attribute");
#endif
				}
			}
			/* Clean up. */

			if (thisValueEntry != NULL) {
				junk = dsDeallocAttributeValueEntry (dirRef, thisValueEntry);
				ASSERT (junk == eDSNoErr);
			}
			if (thisValue != 0) {
				junk = dsCloseAttributeValueList (thisValue);
				ASSERT (junk == eDSNoErr);
			}
			if (thisAttrEntry != NULL) {
				junk = dsDeallocAttributeEntry (dirRef, thisAttrEntry);
				ASSERT (junk == eDSNoErr);
			}

			if (err != eDSNoErr) {
				break;
			}
		}
	}
	/* Copy results out to caller. */

	if (err == eDSNoErr) {
		if ((pathListToAuthNode != NULL) && (userNameForAuth != NULL)) {
			/* Copy out results. */

			*pathListToAuthNodePtr = pathListToAuthNode;
			*userNameForAuthPtr = userNameForAuth;

			/* NULL out locals so that we don't dispose them. */

			pathListToAuthNode = NULL;
			userNameForAuth = NULL;
		} else {
			err = eDSAttributeNotFound;
		}
	}
	/* Clean up. */

	if (pathListToAuthNode != NULL) {
		junk = dsDataListAndHeaderDeallocate (dirRef, pathListToAuthNode);
		ASSERT (junk == eDSNoErr);
	}
	if (userNameForAuth != NULL) {
		free (userNameForAuth);
	}
	if (foundRecAttrList != 0) {
		junk = dsCloseAttributeList (foundRecAttrList);
		ASSERT (junk == eDSNoErr);
	}
	if (context != NULL) {
		junk = dsReleaseContinueData (dirRef, context);
		ASSERT (junk == eDSNoErr);
	}
	if (foundRecAttrList != 0) {
		junk = dsDeallocRecordEntry (dirRef, foundRecEntry);
		ASSERT (junk == eDSNoErr);
	}
	if (requestedAttributes != NULL) {
		junk = dsDataListAndHeaderDeallocate (dirRef, requestedAttributes);
		ASSERT (junk == eDSNoErr);
	}
	if (recordName != NULL) {
		junk = dsDataListAndHeaderDeallocate (dirRef, recordName);
		ASSERT (junk == eDSNoErr);
	}
	if (recordType != NULL) {
		junk = dsDataListAndHeaderDeallocate (dirRef, recordType);
		ASSERT (junk == eDSNoErr);
	}
	if (buf != NULL) {
		junk = dsDataBufferDeAllocate (dirRef, buf);
		ASSERT (junk == eDSNoErr);
	}

	ASSERT ((err == eDSNoErr) == ((*pathListToAuthNodePtr != NULL)
								  && (*userNameForAuthPtr != NULL)));

	return err;
}

static tDirStatus
AuthenticateWithNode (tDirReference dirRef,
					  tDataListPtr pathListToAuthNode,
					  const char *userNameForAuth, const char *password)
/*
	Authenticate a user with their authentication node.
	dirRef is the connection to Open Directory.
	pathListToAuthNode is a data list that contain's the 
	path components of the authentication node for the 
	specified user.  userNameForAuth and password are the 
	user name and password to authenticate.
*/
{
	tDirStatus err;
	tDirStatus junk;
	size_t userNameLen;
	size_t passwordLen;
	tDirNodeReference authNodeRef;
	tDataNodePtr authMethod;
	tDataBufferPtr authOutBuf;
	tDataBufferPtr authInBuf;
	unsigned long length;

	ASSERT (dirRef != 0);
	ASSERT (pathListToAuthNode != NULL);
	ASSERT (userNameForAuth != NULL);
	ASSERT (password != NULL);

	authNodeRef = 0;
	authMethod = NULL;
	authOutBuf = NULL;
	authInBuf = NULL;

	userNameLen = strlen (userNameForAuth);
	passwordLen = strlen (password);

	/* Open the authentication node. */

	err = dsOpenDirNode (dirRef, pathListToAuthNode, &authNodeRef);

/*
	Create the input parameters to dsDoDirNodeAuth and then call it.  The most 
	complex input parameter to dsDoDirNodeAuth is authentication data itself, 
	held in authInBuf.  This holds the following items:
	//
	4 byte length of user name (includes trailing null)
	user name, including trailing null
	4 byte length of password (includes trailing null)
	password, including trailing null
*/

	if (err == eDSNoErr) {
		authMethod =
			dsDataNodeAllocateString (dirRef,
									  kDSStdAuthNodeNativeClearTextOK);
		if (authMethod == NULL) {
			err = eDSAllocationFailed;
		}
	}
	if (err == eDSNoErr) {
/*
		Allocate some arbitrary amount of space for the authOutBuf.  This 
		buffer comes back containing a credential generated by the 
		authentication (apparently a kDS1AttrAuthCredential).  However, 
		we never need this information, so we basically just create the 
		buffer, pass it in to dsDoDirNodeAuth, and then throw it away. 
		Unfortunately dsDoDirNodeAuth won't let us pass in NULL.
*/

		authOutBuf = dsDataBufferAllocate (dirRef, kDefaultDSBufferSize);
		if (authOutBuf == NULL) {
			err = eDSAllocationFailed;
		}
	}
	if (err == eDSNoErr) {
		authInBuf =
			dsDataBufferAllocate (dirRef,
								  sizeof (length) + userNameLen + 1 +
								  sizeof (length) + passwordLen + 1);
		if (authInBuf == NULL) {
			err = eDSAllocationFailed;
		}
	}
	if (err == eDSNoErr) {
		length = userNameLen + 1;	/* + 1 to include trailing null */
		junk = dsDataBufferAppendData (authInBuf, &length, sizeof (length));
		ASSERT (junk == noErr);

		junk =
			dsDataBufferAppendData (authInBuf, userNameForAuth,
									userNameLen + 1);
		ASSERT (junk == noErr);

		length = passwordLen + 1;	/* + 1 to include trailing null */
		junk = dsDataBufferAppendData (authInBuf, &length, sizeof (length));
		ASSERT (junk == noErr);

		junk = dsDataBufferAppendData (authInBuf, password, passwordLen + 1);
		ASSERT (junk == noErr);

		/* Call dsDoDirNodeAuth to do the authentication. */

		err =
			dsDoDirNodeAuthQ (dirRef, authNodeRef, authMethod, 1,
							  authInBuf, &authOutBuf, NULL);
	}
	/* Clean up. */

	if (authInBuf != NULL) {
		junk = dsDataBufferDeAllocate (dirRef, authInBuf);
		ASSERT (junk == eDSNoErr);
	}
	if (authOutBuf != NULL) {
		junk = dsDataBufferDeAllocate (dirRef, authOutBuf);
		ASSERT (junk == eDSNoErr);
	}
	if (authMethod != NULL) {
		junk = dsDataNodeDeAllocate (dirRef, authMethod);
		ASSERT (junk == eDSNoErr);
	}
	if (authNodeRef != 0) {
		junk = dsCloseDirNode (authNodeRef);
		ASSERT (junk == eDSNoErr);
	}

	return err;
}

static tDirStatus
CheckPasswordUsingOpenDirectory (const char *username, const char *password)
	/* Check a user name and password using Open Directory. */
{
	tDirStatus err;
	tDirStatus junk;
	tDirReference dirRef;
	tDataListPtr pathListToSearchNode;
	tDirNodeReference searchNodeRef;
	tDataListPtr pathListToAuthNode;
	char *userNameForAuth;

	ASSERT (username != NULL);
	ASSERT (password != NULL);

	dirRef = 0;
	pathListToSearchNode = NULL;
	searchNodeRef = 0;
	pathListToAuthNode = NULL;
	userNameForAuth = NULL;

	/* Connect to Open Directory. */

	err = dsOpenDirService (&dirRef);

	/* Open the search node. */

	if (err == eDSNoErr) {
		err = GetSearchNodePathList (dirRef, &pathListToSearchNode);
	}
	if (err == eDSNoErr) {
		err = dsOpenDirNode (dirRef, pathListToSearchNode, &searchNodeRef);
	}
/*
	Search for the user's record and extract the user's authentication 
	node and authentication user name..
*/

	if (err == eDSNoErr) {
		err =
			FindUsersAuthInfo (dirRef, searchNodeRef, username,
							   &pathListToAuthNode, &userNameForAuth);
	}
	/* Open the authentication node and do the authentication. */

	if (err == eDSNoErr) {
		err =
			AuthenticateWithNode (dirRef, pathListToAuthNode, userNameForAuth,
								  password);
	}
	/* Clean up. */

	if (userNameForAuth != NULL) {
		free (userNameForAuth);
	}
	if (pathListToAuthNode != NULL) {
		junk = dsDataListAndHeaderDeallocate (dirRef, pathListToAuthNode);
		ASSERT (junk == eDSNoErr);
	}
	if (searchNodeRef != 0) {
		junk = dsCloseDirNode (searchNodeRef);
		ASSERT (junk == eDSNoErr);
	}
	if (pathListToSearchNode != NULL) {
		junk = dsDataListAndHeaderDeallocate (dirRef, pathListToSearchNode);
		ASSERT (junk == eDSNoErr);
	}
	if (dirRef != 0) {
		junk = dsCloseDirService (dirRef);
		ASSERT (junk == eDSNoErr);
	}

	return err;
}

static BOOL
AuthenticateWithOpenDirectory (const char *username, const char *password)
	/* Authenticate a user using Open Directory. */
{
	tDirStatus err;

	ASSERT ((username != NULL) || (password == NULL));

	err = CheckPasswordUsingOpenDirectory (username, password);
	switch (err) {
	case eDSNoErr:
#ifdef ENABLE_DEBUG
		DEBUG_LOG ("Authenticated via Open Directory");
#endif
		return TRUE;
		break;
#ifdef ENABLE_DEBUG
	case eDSAuthFailed:
		DEBUG_LOG ("Authentication failed via Open Directory");
		break;
	case eDSRecordNotFound:
		DEBUG_LOG
			("Authentication failed because directory record was not found.");
		break;
	default:
		{
			char szTemp[200];
			sprintf(szTemp, "Authentication failed because of error %i", err);
			DEBUG_LOG (szTemp);
			break;
		}
#endif
	}
	return FALSE;
}

#endif /* ENABLE_OPEN_DIRECTORY */

#ifdef ENABLE_PAM

#ifdef HAVE_PAM_PAM_APPL_H
#include <pam/pam_appl.h>
#endif
#ifdef HAVE_SECURITY_PAM_APPL_H
#include <security/pam_appl.h>
#endif
#ifdef HAVE_PAM_PAM_MISC_H
#include <pam/pam_misc.h>
#endif
#ifdef HAVE_SECURITY_PAM_MISC_H
#include <security/pam_misc.h>
#endif

static int
PasswordPAMConversationCallback (int num_msg,
								 const struct pam_message **msg,
								 struct pam_response **resp,
								 void *appdata_ptr)
/*
	A PAM conversation callback that passes a hardwired password to 
	PAM.  appdata_ptr is callback-specific; in this case it's a pointer 
	to our password (or NULL if no password was supplied).  The other 
	parameters are controlled by PAM.
	
	Our callback has a very simple algorithm.  If there's more than 
	one message, we fail because I didn't write the code to handle the 
	multiple messages case (which is quite difficult).  Otherwise, we 
	look to see if a) we have a password (via appdata_ptr), and b) the 
	single message is a request for non-echoed string.  If so, we 
	assume that PAM wants the password, and we pass it back.  If not, 
	we pass the request to misc_conv, which handles all of the other 
	messages in a standard way.
*/
{
	int err;
	BOOL passItOn;
	struct pam_response *reply;

	ASSERT (num_msg == 1);		/* we're just not prepared to handle anything else */
	ASSERT (msg != NULL);
	ASSERT (resp != NULL);

	if (num_msg != 1) {
		err = PAM_ABORT;
	} else {
/*
		We pass the request directly to misc_conv except when 
		a) the user supplied a password on the command line, and 
		b) PAM is asking us for hidden text (which we assume to be a 
		password request, which is somewhat bogus).
*/

		passItOn = TRUE;
		ASSERT (appdata_ptr != NULL);
		if (appdata_ptr != NULL) {
			if (msg[0]->msg_style == PAM_PROMPT_ECHO_OFF) {
				passItOn = FALSE;
			}
		}

		if (passItOn) {
			err = misc_conv (num_msg, msg, resp, appdata_ptr);
		} else {
			reply =
				(struct pam_response *) calloc (1,
												sizeof (struct pam_response));
			ASSERT (reply != NULL);

			if (reply == NULL) {
				err = PAM_CONV_ERR;
			} else {
				reply->resp = strdup ((char *) appdata_ptr);
				reply->resp_retcode = 0;

				*resp = reply;

				err = 0;
			}
		}
	}

	return err;
}

static const char *kPAMServiceName = "antinat";

static BOOL
AuthenticateWithPAM (const char *username, const char *password)
/*
	Authenticate (and authorise for that matter) a user using 
	PAM.  username or password may both be NULL, in which case this 
	prompts for the the corresponding information.
*/
{
	int err;
	int junk;
	struct pam_conv conv;
	pam_handle_t *pamH;

	ASSERT ((username != NULL) || (password == NULL));
#ifdef ENABLE_DEBUG
	DEBUG_LOG ("PAM: Attempting to authenticate.");
#endif

	pamH = NULL;

/*
	Open a connection to PAM.  These few lines reveal the problem 
	using PAM on Mac OS X.  You have to provide a service name, which 
	PAM then looks up in "/etc/pam.d/".  If it doesn't find your service 
	specifically listed there, it uses the "other" configuration file.  
	That configuration file always denies all requests.  So, PAM is only 
	truly useful if you either A) already have a service listed in 
	that directory, or B) you're prepared to add one.  Option B isn't 
	really appropriate for a standard Macintosh program, although it 
	may be appropriate if you're porting some sort of UNIX program 
	and include this configuration step as part of your installer 
	(or installation instructions).
*/

	conv.conv = PasswordPAMConversationCallback;
	conv.appdata_ptr = (void *) password;

	err = pam_start (kPAMServiceName, username, &conv, &pamH);

	/* Tell PAM to authenticate. */

	if (err == 0) {
		err = pam_authenticate (pamH, 0);
	}
	/* Clean up. */

	if (pamH != NULL) {
		junk = pam_end (pamH, err);
		ASSERT (junk == 0);
	}
	if (err == 0) {
#ifdef ENABLE_DEBUG
		DEBUG_LOG ("PAM: Authenticated successfully.");
#endif
		return TRUE;
	}
#ifdef ENABLE_DEBUG
	DEBUG_LOG ("PAM: Authentication failed.");
#endif
	return FALSE;
}

#endif

static BOOL
auth_config_getpw (config_t * conf, const char *username, char **passwd)
{
	const char *realpasswd;
	char *tmp;
	BOOL authed;
#ifdef ENABLE_DEBUG
	char szDebug[300];
#endif
	authed = FALSE;
	realpasswd = NULL;
	realpasswd = config_getUser (conf, username);
#ifdef ENABLE_DEBUG
	if (realpasswd)
		sprintf (szDebug, "Found user, has hash %s", realpasswd);
	else
		sprintf (szDebug, "Username not found - %s", username);
	DEBUG_LOG (szDebug);
#endif
	if (realpasswd) {
		tmp = (char *) malloc (strlen (realpasswd) + 1);
		if (tmp != NULL) {
			strcpy (tmp, realpasswd);
			*passwd = tmp;
			return TRUE;
		}
	}
	return FALSE;
}

static BOOL
auth_getPasswordByUsername (config_t * conf, const char *username,
							char **passwd)
{
	/* This is a placeholder so we can support more datasources. */
	return auth_config_getpw (conf, username, passwd);
}


static const char lets[] =
	"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwyxz0123456789+/";

static BOOL
auth_chap_gen_challenge (char *data, int len)
{
	int i;
	for (i = 0; i < len; i++) {
		data[i] = lets[rand () % 64];
	}
	return TRUE;
}

static BOOL
auth_chap_send_fail (conn_t * conn)
{
	char str[16];
	str[0] = 0x01;
	str[1] = 0x01;
	str[2] = 0x00;
	str[3] = 0x01;
	str[4] = 0x01;
	return conn_sendData (conn, str, 5);
}

static BOOL
auth_chap_send_ok (conn_t * conn)
{
	char str[16];
	str[0] = 0x01;
	str[1] = 0x01;
	str[2] = 0x00;
	str[3] = 0x01;
	str[4] = 0x00;
	return conn_sendData (conn, str, 5);
}

/* This is an internal client library function. */
int _an_socks5_hmacmd5_chap (const unsigned char *, int,
							 const char *, unsigned char *);


static BOOL
auth_hmacmd5_chap (const unsigned char *challenge, int challen,
				   const char *passwd, unsigned char **response, int *resplen)
{
	unsigned char *respbuf;
	respbuf = (unsigned char *) malloc (20);
	if (respbuf == NULL)
		return FALSE;
	if (_an_socks5_hmacmd5_chap (challenge, challen,
								 passwd, respbuf) == AN_ERROR_SUCCESS) {
		*response = respbuf;
		*resplen = 16;
		return TRUE;
	}
	free (respbuf);
	return FALSE;
}

BOOL
auth_chap (conn_t * conn)
{
	unsigned char version;
	unsigned char tmpch;
	unsigned char navas;
	unsigned char att[10];
	char challenge[100];
	char username[256];
	int thealgo;
	unsigned char *correctresp;
	unsigned char *gotresp;
	char *passwd;
	int resplen;
	BOOL authed;
	BOOL finished;
	BOOL sentChallenge;
	BOOL sendChallenge;
	int sendAlgo;
	int i, j;
#ifdef ENABLE_DEBUG
	char szDebug[300];
#endif
	finished = FALSE;

	authed = FALSE;
	gotresp = NULL;
	correctresp = NULL;
	passwd = NULL;
	sentChallenge = FALSE;
	thealgo = -1;
	strcpy (username, "");

	while (!finished) {
		sendChallenge = FALSE;
		sendAlgo = -1;
		if (!conn_getChar (conn, &version))
			return FALSE;
		if (version != 1)
			return FALSE;
		if (!conn_getChar (conn, &navas))
			return FALSE;
		for (i = 0; i < navas; i++) {
			if (!conn_getSlab (conn, (char *) &att, 2))
				return FALSE;
			/* Documented:
			 * 0- status
			 * 1- Text-message
			 * 2- User-identity
			 * 3- Challenge
			 * 4- Response
			 * 5- Charset
			 * 16-Identifier
			 * 17-Algorithms
			 */
#ifdef ENABLE_DEBUG
			sprintf (szDebug, "recved att %x data length %i", att[0], att[1]);
			DEBUG_LOG (szDebug);
#endif
			switch (att[0]) {
			case 2:
				if (!conn_getSlab (conn, username, att[1]))
					goto barfed;
#ifdef ENABLE_DEBUG
				sprintf (szDebug, "Set username to %s", username);
				DEBUG_LOG (szDebug);
#endif
				if (!sentChallenge)
					sendChallenge = TRUE;
				break;
			case 4:
				gotresp = (unsigned char *) malloc (att[1] + 1);
				if (gotresp == NULL) {
					auth_chap_send_fail (conn);
					goto barfed;
				}
				if (!conn_getSlab (conn, (char *) gotresp, att[1])) {
					auth_chap_send_fail (conn);
					goto barfed;
				}
				if (thealgo < 0) {
					auth_chap_send_fail (conn);
					goto barfed;
				}

				if (!auth_getPasswordByUsername
					(conn->conf, username, &passwd)) {
					auth_chap_send_fail (conn);
					goto barfed;
				}
				switch (thealgo) {
				case 0x85:		/* HMAC-MD5 */
					if (auth_hmacmd5_chap
						((unsigned char *) challenge, 64, passwd,
						 &correctresp, &resplen) == FALSE) {
						auth_chap_send_fail (conn);
						goto barfed;
					}
					break;
				default:
					auth_chap_send_fail (conn);
					goto barfed;
					break;
				}

				if (resplen != att[1]) {
					/* If the lengths differ, so do the responses */
					auth_chap_send_fail (conn);
					goto barfed;
				}
				if (memcmp (correctresp, gotresp, resplen) == 0) {
					auth_chap_send_ok (conn);
					conn_setUser (conn, username);
					conn_setPass (conn, passwd);
					conn->authsrc = AUTH_CONFIG;
					authed = TRUE;
					goto barfed;
				} else {
					auth_chap_send_fail (conn);
					goto barfed;
				}
				break;
			case 0x11:
				for (j = 0; j < att[1]; j++) {
					if (!conn_getChar (conn, &tmpch))
						goto barfed;
					if (sendAlgo < 0) {
						if (tmpch == 0x85)
							sendAlgo = tmpch;
					}

				}
				if (sendAlgo < 0)
					sendAlgo = -2;
				if (!sentChallenge)
					sendChallenge = TRUE;
				break;
			}
		}
		if (sendAlgo == -2) {
			auth_chap_send_fail (conn);
			goto barfed;
		}
		att[0] = 0x01;
		att[1] = 0x00;
		if (sendChallenge)
			att[1]++;
		if (sendAlgo >= 0)
			att[1]++;
		if (!conn_sendData (conn, (char *) att, 2))
			goto barfed;
		if (sendAlgo >= 0) {
			att[0] = 0x11;
			att[1] = 0x01;
			att[2] = sendAlgo;
			if (!conn_sendData (conn, (char *) att, 3))
				goto barfed;
			thealgo = sendAlgo;
		}
		if (sendChallenge) {
			auth_chap_gen_challenge (challenge, 64);
			att[0] = 0x03;
			att[1] = 0x40;
			if (!conn_sendData (conn, (char *) att, 2))
				goto barfed;
			if (!conn_sendData (conn, (char *) challenge, 64))
				goto barfed;
		}
	}

  barfed:
	if (gotresp != NULL)
		free (gotresp);
	if (correctresp != NULL)
		free (correctresp);
	if (passwd != NULL)
		free (passwd);
#ifdef ENABLE_DEBUG
	if (authed) {
		DEBUG_LOG ("authenticated successfully");
	} else {
		DEBUG_LOG ("failed authentication");
	}
#endif
	return authed;
}

#ifndef _WIN32_
static BOOL
auth_local_crypt (const char *username, const char *passwd)
{
	char csalt[12];
	char *realpasswd;
	char pwhash[50];
	BOOL authed;
#ifdef ENABLE_DEBUG
	char szDebug[300];
#endif
#ifdef HAVE_GETSPNAM
	struct spwd *shad_pw;
#ifdef HAVE_GETSPNAM_R
	struct spwd realspwd;
#endif
#else
#ifdef HAVE_GETPWNAM
	struct passwd *pw_pw;
#ifdef HAVE_GETPWNAM_R
	struct passwd realpwd;
#endif
#endif
#endif
#if defined(HAVE_GETSPNAM_R)||(HAVE_GETPWNAM_R)
	char buf[1024];
#endif
#if (defined(HAVE_CRYPT_CRYPT_R)||defined(HAVE_CRYPT_R))&&(!defined(BROKEN_CRYPT_R))
	struct crypt_data cd;
#endif

#ifdef HAVE_GETSPNAM
	shad_pw = NULL;
#endif
	authed = FALSE;
	realpasswd = NULL;
#ifdef HAVE_GETSPNAM			/* The system supports shadow password */
#ifndef HAVE_GETSPNAM_R			/* System has no reentrant shadow functions */
	os_mutex_lock (&getspnam_lock);
	shad_pw = getspnam (username);
#else /* System has reentrant shadow functions */
#ifdef BROKEN_GETSPNAM			/* System has Solaris-style shadow functions */
	shad_pw = getspnam_r (username, &realspwd, buf, sizeof (buf));
#else /* System has Linux-style shadow functions */
	getspnam_r (username, &realspwd, buf, sizeof (buf), &shad_pw);
#endif /* BROKEN_GETSPNAM */
#endif /* HAVE_GETSPNAM_R */
	if (shad_pw) {
		realpasswd = (char *) malloc (strlen (shad_pw->sp_pwdp) + 1);
		if (realpasswd)
			strcpy (realpasswd, shad_pw->sp_pwdp);
	}
#ifndef HAVE_GETSPNAM_R			/* No reentrant shadow functions */
	os_mutex_unlock (&getspnam_lock);
#endif
#else /* No shadow - try passwd */
#ifdef HAVE_GETPWNAM			/* Do we have passwd functions */
#ifndef HAVE_GETPWNAM_R			/* We have no reentrant passwd functions */
	os_mutex_lock (&getpwnam_lock);
	pw_pw = getpwnam (username);
#else /* We have reentrant passwd functions */
#ifdef BROKEN_GETPWNAM			/* System has Solaris-style passwd functions */
	pw_pw = getpwnam_r (username, &realpwd, buf, sizeof (buf));
#else /* System has Linux-style passwd functions */
	getpwnam_r (username, &realpwd, buf, sizeof (buf), &pw_pw);
#endif /* BROKEN_GETPWNAM */
#endif /* HAVE_GETPWNAM_R */
	if (pw_pw) {
		realpasswd = (char *) malloc (strlen (pw_pw->pw_passwd) + 1);
		if (realpasswd)
			strcpy (realpasswd, pw_pw->pw_passwd);
	}
#ifndef HAVE_GETPWNAM_R			/* No reentrant passwd functions */
	os_mutex_unlock (&getpwnam_lock);
#endif /* HAVE_GETPWNAM_R */
#endif /* HAVE_GETPWNAM */
#endif /* HAVE_GETSPNAM */
#ifdef ENABLE_DEBUG
	if (realpasswd) {
		sprintf (szDebug, "Crypt: Found user, has hash %s", realpasswd);
		DEBUG_LOG (szDebug);
	} else
		DEBUG_LOG ("Crypt: Username not found");
#endif
	if (realpasswd) {
		csalt[0] = realpasswd[0];
		csalt[1] = realpasswd[1];
		csalt[2] = '\0';
		if (strcmp (csalt, "$1") == 0) {
			strncpy (csalt, realpasswd, 11);
			csalt[12] = '\0';
		}
		/* Apparently, crypt_r can't auth against non-MD5 passwds :( */
#if (defined(HAVE_CRYPT_CRYPT_R)||defined(HAVE_CRYPT_R))&&(!defined(BROKEN_CRYPT_R))
		strcpy (pwhash, crypt_r (passwd, csalt, &cd));
		if (strcmp (pwhash, realpasswd) == 0) {
			authed = TRUE;
		} else {
#ifdef ENABLE_DEBUG
			DEBUG_LOG ("Couldn't authenticate with crypt_r.  Trying crypt.");
#endif
#endif
			os_mutex_lock (&crypt_lock);
			strcpy (pwhash, crypt (passwd, csalt));
			os_mutex_unlock (&crypt_lock);
			if (strcmp (pwhash, realpasswd) == 0) {
				authed = TRUE;
			}
#if (defined(HAVE_CRYPT_CRYPT_R)||defined(HAVE_CRYPT_R))&&(!defined(BROKEN_CRYPT_R))
		}
#endif
	}

	if (realpasswd)
		free (realpasswd);
	return authed;
}
#endif

#ifdef _WIN32_

#ifndef LOGON32_LOGON_NETWORK
#define LOGON32_LOGON_NETWORK 3
#endif

#ifndef LOGON32_PROVIDER_DEFAULT
#define LOGON32_PROVIDER_DEFAULT 0
#endif

typedef BOOL (WINAPI * LogonUserProc) (LPCSTR, LPCSTR, LPCSTR, DWORD, DWORD,
									   PHANDLE);
static BOOL
auth_local_win32 (const char *username, const char *passwd)
{
	BOOL ret;
	HANDLE tok;
	HINSTANCE hDll;
	LogonUserProc lu;
#ifdef ENABLE_DEBUG
	DWORD err;
#endif

	hDll = LoadLibrary ("advapi32.dll");
	if (hDll == NULL) {
#ifdef ENABLE_DEBUG
		DEBUG_LOG
			("Local authentication cannot be supported on this build of Windows.");
#endif
		return FALSE;
	}

	lu = (LogonUserProc) GetProcAddress (hDll, "LogonUserA");

	if (lu == NULL) {
#ifdef ENABLE_DEBUG
		DEBUG_LOG
			("Local authentication cannot be supported on this build of Windows.");
#endif
		FreeLibrary (hDll);
		return FALSE;
	}

	ret =
		lu ((LPTSTR) username, NULL, (LPTSTR) passwd, LOGON32_LOGON_NETWORK,
			LOGON32_PROVIDER_DEFAULT, &tok);
	if (ret) {
		/* It worked? */
		CloseHandle (tok);
		FreeLibrary (hDll);
		return TRUE;
	}

	FreeLibrary (hDll);

#ifdef ENABLE_DEBUG
	err = GetLastError ();
	if (err == ERROR_PRIVILEGE_NOT_HELD) {
		DEBUG_LOG
			("Cannot locally authenticate without having 'Act as part of the operating system' user right");
	}
#endif

	return FALSE;
}
#endif

static BOOL
auth_authUsernamePassword (conn_t * conn, const char *username,
						   const char *passwd)
{
	config_t *conf;
	char *correctpw;
	BOOL authed;
	conf = conn->conf;
	/* This is a placeholder for more auth mechanisms. */
	if (config_allowLocalUsers (conf)) {
#ifndef _WIN32_
#ifdef ENABLE_OPEN_DIRECTORY
		if (AuthenticateWithOpenDirectory (username, passwd)) {
			conn->authsrc = AUTH_LOCAL;
			return TRUE;
		}
#endif
#ifdef ENABLE_PAM
		if (AuthenticateWithPAM (username, passwd)) {
			conn->authsrc = AUTH_LOCAL;
			return TRUE;
		}
#endif
		if (auth_local_crypt (username, passwd)) {
			conn->authsrc = AUTH_LOCAL;
			return TRUE;
		}
#else
		if (auth_local_win32 (username, passwd)) {
			conn->authsrc = AUTH_LOCAL;
			return TRUE;
		}
#endif
	}
	authed = FALSE;
	if (auth_config_getpw (conf, username, &correctpw)) {
		if (strcmp (correctpw, passwd) == 0) {
			authed = TRUE;
			conn->authsrc = AUTH_CONFIG;
		}
		free (correctpw);
	}
	return authed;
}

BOOL
auth_unpw (conn_t * conn)
{
	unsigned char version;
	unsigned char length;
	char username[260];
	char passwd[260];
	char rep[3];
	BOOL authed;
#ifdef ENABLE_DEBUG
	char szDebug[300];
#endif

	authed = FALSE;
	if (!conn_getChar (conn, &version))
		return FALSE;
	if (version != 1)
		return FALSE;
	if (!conn_getChar (conn, &length))
		return FALSE;

	/* getSlab null terminates */
	if (!conn_getSlab (conn, username, length))
		goto barfed;
	if (!conn_getChar (conn, &length))
		goto barfed;
	if (!conn_getSlab (conn, passwd, length))
		goto barfed;

	authed = auth_authUsernamePassword (conn, username, passwd);

  barfed:
	rep[0] = 0x01;
	if (authed) {
#ifdef ENABLE_DEBUG
		sprintf (szDebug, "User %s authenticated", username);
#endif
		rep[1] = '\0';
		conn_setUser (conn, username);
		conn_setPass (conn, passwd);
	} else {
#ifdef ENABLE_DEBUG
		sprintf (szDebug, "User %s NOT authenticated", username);
#endif
		rep[1] = 0x01;
	}
#ifdef ENABLE_DEBUG
	DEBUG_LOG (szDebug);
#endif
	conn_sendData (conn, rep, 2);
	return authed;
}
