/*-----------------------------------------------------------------------
    This file is part of aaPhoto.

    aaPhoto is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    aaPhoto is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
------------------------------------------------------------------------*/


//#define __UNIX__
//#define __WIN32__
//#define __AROS__
//#define __BMP_ONLY__



// ---------------------------------------------------
// ----------- Auto Adjust Photo ---------------------
// ----------- Horváth András (c) 2006-2009 ----------
// ----------- Hungary, www.log69.com ----------------
// ---------------------------------------------------

/*

Changelog:
2009/02/22 - aaPhoto v0.30 - implementation of PNG format (RGB and Gray images read / write with alpha channel support)
                           - bugfix: reading corrupt exif info in JPEG files could get into an infinite loop
                           - bugfix: length of exif info was determined wrongly
                           - rework of the parameters and switches parsing part
                           - lots of code cleanup
                           - most of the comments in code translated into english
                           - print messages get flushed out with fflush now during process
                           - the --info switch removed
                           - the -o switch removed for saftey reasons, --overwrite still available
                           - the -h switch added for unix compatibility
                           - the -j1 and -j2 switch removed, --jpg and --jp2 still available
                           - the --png switch is now new for PNG output
                           - the -s switch changed from --speed to --silent for compatibility reasons
                           - the --mute switch changed to -s, --silent and --quiet for unix / posix compatibility
                           - the -V, --verbose switch is now new for more detailed output during image process
                           - the --test switch now turns the --autoadjust switch on by default
			   - the -d, --description and -l, --lincense switches removed
2008/02/02 - aaPhoto v0.29 - bugfix: colorspace variable was not defined during the load of BMP format
                           - bugfix: BMP format handling fixed for Jpeg conversion
                           - grayscale images can also be used as an input
                           - change: file name buffer increased (for processing files in folders)
                           - Exif metadata information is now restored during conversion in Jpeg images
2007/08/11 - aaPhoto v0.28 - new aaRGB v0.57 version update
                           - bugfix: removing extra slashes from the end of folders
2007/07/04 - aaPhoto v0.27 - new aaRGB v0.56 version update v0.56 with "Apply only on selection" function
2007/05/26 - aaPhoto v0.26 - bugfix: Win32 version crashed during JPEG-2000 conversion
2007/05/19 - aaPhoto v0.25 - expanding functions: Rotate 90, 180, 270, Flip x, Flip y
2007/05/01 - aaPhoto v0.24 - improving timing values
                           - input parameter can be folders too beside files
                           - bitmap info parameter now working with bmp too
                           - simplifying parameter input: no --autoadjust parameter needed from now, default is on
2007/04/03 - aaPhoto v0.23 - new aaRGB v0.55 version update
2007/04/01 - aaPhoto v0.22 - new aaRGB v0.54 version update
2007/03/30 - aaPhoto v0.21 - bmp_only macro created in source code for other platforms
                           - bug fixes
2007/03/29 - aaPhoto v0.20 - new aaRGB v0.53 version update
2007/02/25 - aaPhoto v0.19 - extra information output within image for testing purposes
2007/02/22 - aaPhoto v0.18 - custom code for BMP input/output (helps in testing)
2007/01/24 - aaPhoto v0.17 - JasPer coder implemented for further image formats
2007/01/04 - aaPhoto v0.16 - stable working command-line version for linux enviroment with BMP format support

*/




// global constants and variables

const int max_char = 256;
const int max_file_name_buffer = 4096 * 256;

int   file_counter;
char *file_name;

unsigned char *file_name_buffer;
long  file_name_counter;
long  file_name_buffer_pointer;

unsigned char *bitmap_buffer;
unsigned long  bitmap_width;
unsigned long  bitmap_height;

int   bitmap_format_bmp_clrspc_type;
int   bitmap_format_jpg_clrspc_type;
int   bitmap_format_jpg_file_type;
int   bitmap_format_png_clrspc_type;
int   bitmap_format_png_interlace_type;
int   bitmap_format_png_compression_type;
int   bitmap_format_png_filter_type;


char *exif_buffer;
long  exif_buffer_length;
long  exif_file_length;
int   exif_flag;

int opt_help;
int opt_version;
int opt_autoadjust;
int opt_overwrite;
int opt_jpg;
int opt_jp2;
int opt_png;
int opt_resize;
int opt_rotate90;
int opt_rotate180;
int opt_rotate270;
int opt_flipx;
int opt_flipy;
int opt_resize_percent;
int opt_speed;
int opt_quality;
int opt_verbose;
int opt_recursive;
int opt_quiet;
int opt_test;

long  tim1, tim2;
float speed_aargb;
float speed_resize;
float time_load;
float time_aargb;
float time_resize;
float time_save;
float time_all;

int char_temp_x;
int char_temp_y;



// setting system dependent filename separation character
#ifdef __WIN32__
	char slsh[] = "\\";
#else
	char slsh[] = "/";
#endif


// setting system timing values for speed check
#ifndef CLOCKS_PER_SEC
    #ifdef __UNIX__
        #define CLOCKS_PER_SEC 1000000
    #endif
    #ifdef __WIN32__
        #define CLOCKS_PER_SEC 1000
    #endif
    #ifdef __AROS__
        #define CLOCKS_PER_SEC 50
    #endif
#endif
float time_divider = CLOCKS_PER_SEC;



#ifndef __BMP_ONLY__
#include <jasper/jasper.h>
#include <png.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <ctype.h>

//#include <stddir.h>
#include <dirent.h>

#include "aaio.c"
#include "aargb.c"
#include "aaresize.c"




void PRINT_VERSION(void){
char *helpinfo[] = {
"*************************\n"
"*** Auto Adjust Photo ***\n"
"*************************\n"
"Copyright 2006-2009 Andras Horvath\n"
"E-mail: mail@log69.com - any suggestions & feedbacks are welcome\n"
"URL: www.log69.com - the official site\n"
"aaPhoto (command-line) version - v0.30\n"
"aaRGB (color-correction engine) version - v0.58\n"
"last update = 2009/02/22\n"
"\n"
"The following libraries are used by this program:\n"
"libjpeg - IJG JPEG software, http://www.ijg.org/\n"
"libjasper - JasPer software, http://www.ece.uvic.ca/~mdadams/jasper/\n"
"libz - Compression library, http://www.zlib.net/\n"
"libpng - PNG software, http://www.libpng.org/\n"
#ifdef __BMP_ONLY__
"\nBMP_ONLY version. Only BMP format is supported here.\n"
#endif
"\n\0"
}; STRING_PRINT(*helpinfo);
}



void PRINT_DESCRIPTION(void){
char *helpinfo[] = {
"[DESCRIPTION]\n"
"Auto Adjust Photo is a small command-line image manipulation tool "
"that allows the user to easily correct family photos. It tries to "
"make the picture look better. "
"The program does this by analyzing the input image and then sets the "
"most optimal contrast, gamma, color balance and saturation for it. "
"More details can be found at www.log69.com\n"
"\n\0"
}; STRING_PRINT(*helpinfo);
}



void PRINT_LICENSE(void){
char *helpinfo[] = {
"[LICENSE]\n"
"This program is free software; you can redistribute it and/or modify "
"it under the terms of the GNU General Public License as published by "
"the Free Software Foundation; either version 3 of the License, or "
"(at your option) any later version.\n"
"\n"
"This program is distributed in the hope that it will be useful, "
"but WITHOUT ANY WARRANTY; without even the implied warranty of "
"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the "
"GNU General Public License for more details.\n"
"\n"
"You should have received a copy of the GNU General Public License "
"along with this program.  If not, see <http://www.gnu.org/licenses/>.\n"
"\n\0"
}; STRING_PRINT(*helpinfo);
}



void PRINT_HELP(void){
char *helpinfo[] = {
"[HELP]\n"
"USAGE: aaphoto [options] [source files]\n"
"The following image types are supported (thanks to JasPer and libpng):\n"
"mif, pnm / pgm / ppm, bmp, ras, jp2, jpc, jpg, pgx, png\n"
"Quality settings can be applied only to jp2, jpc, jpg formats\n"
"The following options are supported:\n"
"    -h   --help          Print this help\n"
"    -v   --version       Print version information\n"
"    -a   --autoadjust    *Auto adjust the colors of the image\n"
"         --overwrite     *Overwrite mode, the original source file is replaced\n"
"         --jpg           *JPEG image output\n"
"         --jp2           *JPEG 2000 image output\n"
"         --png           *PNG image output with alpha channel support\n"
"    -r   --resize        *Resize image with given value taking the longer side in % or pixels\n"
"         --rotate90      *Rotate image with 90 degrees clockwise\n"
"         --rotate180     *Rotate image with 180 degrees\n"
"         --rotate270     *Rotate image with 90 degrees counter-clockwise\n"
"         --flipx         *Mirror image horizontally\n"
"         --flipy         *Mirror image vertically\n"
"    -q   --quality       *Set image quality from 1 to 100\n"
"    -s   --silent        Silent mode, no information printed during operation\n"
"         --quiet         ...same as above\n"
"    -V   --verbose       Print verbose information about processing\n"
//"    -R   --recursive     *Process images recursively\n"
"         --speed         *Print extra speed information\n"
"         --test          *Print detailed test information into image\n"
"\n"
"    * marked parameters need input file\n"
"\n"
"EXAMPLES:\n"
"     aaphoto image.jpg\n"
"     aaphoto -a -r600 -q85 *.jpg\n"
"     aaphoto mydir\n"
"     aaphoto -V --resize70% image.png\n"
"     aaphoto --quality60 image.jp2\n"
"\n"
"REMARKS:\n"
"- auto adjust parameter is set by default but only if no other parameters are given\n"
"- _new filename will be generated without --overwrite parameter\n"
"- on directory input every file will be processed in it but not recursively\n"
"- order of parameters does not matter\n"
"- resizing can be set in percentage too\n"
"- resize parameter should be less or equal than original\n"
"- resize uses the best (and slowest) resampling method\n"
"- default jpeg compression quality is 95%\n"
"- Exif information is restored in jpeg images\n"
"\n\0"

}; STRING_PRINT(*helpinfo);
}



/*
void PRINT_INFO(char *file_name)
{
		if (BITMAP_CHECK(file_name) != 0){
			char tt [256] = "";
			GET_FILE_NAME(file_name, tt);
			STRING_PRINT("tt"); STRING_PRINT(" - ");
			STRING_PRINTD("%d", (int)bitmap_width);
			STRING_PRINT (" x ");
			STRING_PRINTD("%d", (int)bitmap_height);
			STRING_PRINT (" pixels - ");
			unsigned long size = bitmap_width * bitmap_height * 3;
			if (size >= 1024 * 1024){ STRING_PRINTF("%.2f Mb in memory", (float)(size) / 1024 / 1024); }
			else{
				if (size >= 1024){ STRING_PRINTF("%.2f Kb in memory", (float)(size) / 1024); }
				else{
					STRINF_PRINTD("%d bytes in memory", (int)(size));
				}
			}
			STRING_PRINT("\n");
		}
		else{
			STRING_PRINT(file_name);
			STRING_PRINT("\nerror: file could not be loaded\n");
		}
}
*/



int MAIN_RESIZE(void)
{
	// if there is an Alpha channel too beside RGB (clrspc_type says)
	// then we use the alpha flag to indicate it to the resize function
	int alpha_flag = 0;
	if ((bitmap_format_png_clrspc_type == 4) || (bitmap_format_png_clrspc_type == 6)){
		alpha_flag = 1; }

	if (opt_resize){

		unsigned long opr = opt_resize;
		// calculating resize in percentage
		if (opt_resize_percent){
			if (bitmap_width > bitmap_height) { opr = bitmap_width  * opr / 100; }
			else                              { opr = bitmap_height * opr / 100; }
		}

		// if the new size is bigger or equal than the original size, then no process is done
		if (!((opr > bitmap_width) && (opr > bitmap_height))){

			unsigned long new_width, new_height;
			if (bitmap_width > bitmap_height){
				new_width = opr;
				new_height = bitmap_height * opr / bitmap_width;
			}
			else{
				new_width = bitmap_width * opr / bitmap_height;
				new_height = opr;
			}

			if (new_width < 1) new_width = 1;
			if (new_height < 1) new_height = 1;

			if (RESIZE(&bitmap_buffer, &bitmap_width, &bitmap_height, new_width, new_height, alpha_flag) == 0){
				STRING_PRINT("\nerror: memory allocation error\n"); return 0; }

			// measuring speed
			// the timing variables get defined in inner function
			// so to measure only the speed of resize without the memory allocation
			float sec = tim2 - tim1;
			sec = sec / time_divider;
			time_resize = sec;
			float speed = bitmap_width * bitmap_height;
			speed = speed / 1000000;
			speed = speed / sec;
			speed_resize = speed;
		}

//		else {
//			STRING_PRINT("\nerror: resize parameter too big, "
//				"target size should be less or equal than original\n"); return 0; }

	}

	// Rotate and Flip calls
	if (opt_rotate90) { if (ROTATE90(&bitmap_buffer, &bitmap_width, &bitmap_height, alpha_flag) == 0)
			{ STRING_PRINT("\nerror: memory allocation error\n"); return 0; } }
	if (opt_rotate180) { if (ROTATE180(&bitmap_buffer, &bitmap_width, &bitmap_height, alpha_flag) == 0)
			{ STRING_PRINT("\nerror: memory allocation error\n"); return 0; } }
	if (opt_rotate270) { if (ROTATE270(&bitmap_buffer, &bitmap_width, &bitmap_height, alpha_flag) == 0)
			{ STRING_PRINT("\nerror: memory allocation error\n"); return 0; } }
	if (opt_flipx) { if (FLIPX(&bitmap_buffer, &bitmap_width, &bitmap_height, alpha_flag) == 0)
			{ STRING_PRINT("\nerror: memory allocation error\n"); return 0; } }
	if (opt_flipy) { if (FLIPY(&bitmap_buffer, &bitmap_width, &bitmap_height, alpha_flag) == 0)
			{ STRING_PRINT("\nerror: memory allocation error\n"); return 0; } }


	return 1;
}




void MAIN_AARGB(void)
{
	long tim1, tim2;
	tim1 = clock();

//	AARGB_NORMAL(bitmap_buffer, bitmap_width, bitmap_height);
	AARGB_MAIN(bitmap_buffer, bitmap_width, bitmap_height, 0, 0, bitmap_width-1, bitmap_height-1, 0, 0, opt_test);

	tim2 = clock();
	float sec = tim2 - tim1;
	sec = sec / time_divider;
	time_aargb = sec;
	float speed = bitmap_width * bitmap_height;
	speed = speed / 1000000;
	speed = speed / sec;
	speed_aargb = speed;
}




int MAIN_RUN(char *file_name)
{
	/* print info */ STRING_PRINTV("verbose mode on\n");
	long timl1, timl2;
	timl1 = clock();

	/* print info */ if (!(opt_verbose)) STRING_PRINT(file_name);
	/* print info */ STRING_PRINTV(file_name); STRING_PRINTV("\n");
	/* print info */ STRING_PRINTV("loading\n");
	if (BITMAP_LOAD(file_name) == 0){ STRING_PRINT("\nerror: file could not be loaded\n"); return 0; }

	timl2 = clock();
	time_load = (float)(timl2 - timl1) / time_divider;

	char file_name_new [max_char];
	if (GET_FILE_NAME_NEW(file_name, file_name_new) == 0){ \
		STRING_PRINT("\nerror: file already exists with the new name: "); \
		STRING_PRINT(file_name_new); STRING_PRINT("\n"); return 0; }

	/* print info */ if (!(opt_verbose)) STRING_PRINT(" .");
	/* print info */ if (opt_autoadjust) STRING_PRINTV("adjusting colors\n");
	if (opt_autoadjust) MAIN_AARGB();

	/* print info */ if (!(opt_verbose)) STRING_PRINT(" .");
	if ((opt_resize) || (opt_rotate90) || (opt_rotate180) || (opt_rotate270) || (opt_flipx) || (opt_flipy))
		{	/* print info */ if (opt_resize) STRING_PRINTV("resizing image\n");
			if (MAIN_RESIZE() == 0) return 0; }

	long tims1, tims2;
	tims1 = clock();

	/* print info */ if (!(opt_verbose)) STRING_PRINT(" .");
	/* print info */ STRING_PRINTV("saving\n");
	if (BITMAP_SAVE(file_name_new) == 0){ STRING_PRINT("\nerror: file could not be saved\n"); return 0; }

	tims2 = clock();
	time_save = (float)(tims2 - tims1) / time_divider;

	/* print info */ if (!(opt_verbose)) STRING_PRINT(" done\n");
	/* print info */ STRING_PRINTV("done\n");

	/* print speed */
	if (opt_speed){
		if (opt_autoadjust) { STRING_PRINTF("%.2f mpixel/sec for auto adjusting\n", speed_aargb); }
		if (opt_resize) { STRING_PRINTF("%.2f mpixel/sec for resizing\n", speed_resize); }
		STRING_PRINTF("%.2f sec for loading\n", time_load);
		if (opt_autoadjust){ STRING_PRINTF("%.2f sec for auto adjusting\n", time_aargb); }
		if (opt_resize){ STRING_PRINTF("%.2f sec for resizing\n", time_resize); }
		STRING_PRINTF("%.2f sec for saving\n", time_save);
	}

	return 1;
}




int MAIN_ARGUMENTS_READ(int argc, char **argv)
{
	opt_help = 0;
	opt_version = 0;
	opt_autoadjust = 0;
	opt_overwrite = 0;
	opt_jpg = 0;
	opt_jp2 = 0;
	opt_png = 0;
	opt_resize = 0;
	opt_rotate90 = 0;
	opt_rotate180 = 0;
	opt_rotate270 = 0;
	opt_flipx = 0;
	opt_flipy = 0;
	opt_resize_percent = 0;
	opt_speed = 0;
	opt_quality = 0;
	opt_verbose = 0;
	opt_recursive = 0;
	opt_quiet = 0;
	opt_test = 0;


	int opt_wrong = 1;
	int opt_wrong_total = 0;
	int opt_counter = 0;

	if (argc >= 2) {
		argc--;

		char *myarg;
		while (argc--){
			//printf("%s\n", *argv);

			opt_wrong = 1;
			myarg = *argv++;
			myarg = *argv;

			if (myarg[0] == '-') {

				// checking switches
				if ((STRING_COMPARE(myarg, "-h\0")) || (STRING_COMPARE(myarg, "--help\0"))){
					opt_wrong = 0; opt_help = 1; }
				if ((STRING_COMPARE(myarg, "-v\0")) || (STRING_COMPARE(myarg, "--version\0"))){
					opt_wrong = 0; opt_version = 1; }
				if ((STRING_COMPARE(myarg, "-a\0")) || (STRING_COMPARE(myarg, "--autoadjust\0"))){
					opt_wrong = 0; opt_autoadjust = 1; }
				if (STRING_COMPARE(myarg, "--overwrite\0")){
					opt_wrong = 0; opt_overwrite = 1; }
				if (STRING_COMPARE(myarg, "--jpg\0")){
					opt_wrong = 0; opt_jpg = 1; }
				if (STRING_COMPARE(myarg, "--jp2\0")){
					opt_wrong = 0; opt_jp2 = 1; }
				if (STRING_COMPARE(myarg, "--png\0")){
					opt_wrong = 0; opt_png = 1; }
//				if ((STRING_COMPARE(myarg, "-R\0")) || (STRING_COMPARE(myarg, "--recursive\0"))){
//					opt_wrong = 0; opt_recursive = 1; }
				if ((STRING_COMPARE(myarg, "-s\0")) || (STRING_COMPARE(myarg, "--silent\0")) || \
					(STRING_COMPARE(myarg, "--quiet\0"))){
					opt_wrong = 0; opt_quiet = 1; }
				if ((STRING_COMPARE(myarg, "-V\0")) || (STRING_COMPARE(myarg, "--verbose\0"))){
					opt_wrong = 0; opt_verbose = 1; }
				if (STRING_COMPARE(myarg, "--speed\0")){
					opt_wrong = 0; opt_speed = 1; }
				if (STRING_COMPARE(myarg, "--test\0")){
					opt_wrong = 0; opt_test = 1; opt_autoadjust = 1; }

				// checking --quality switch
				if ((STRING_COMPARE_FIX(myarg, "-q", 2)) || (STRING_COMPARE_FIX(myarg, "--quality", 9))){
					char num [8] = "";
					int num_max = 8;
					int num_offs;
					if (STRING_COMPARE_FIX(myarg, "--quality", 9)){
						num_offs = 9;
					}
					else{
						num_offs = 2;
					}
					int i = 0;
					while ((myarg[num_offs+i] != 0) && (i < num_max)){
						num[i] = myarg[num_offs+i];
						i++;
					}
					num [i] = 0;
					if (i >= num_max){ STRING_PRINT("error: bad parameters\n"); return 0; }

					if (STRING_CONVERT_TO_INTEGER(num, &opt_quality) == 0){
						STRING_PRINT("error: bad parameters\n"); return 0; }
					if ((opt_quality >= 1) && (opt_quality <= 100)){ opt_wrong = 0; }
					else{ STRING_PRINT("error: bad parameters\n"); return 0; }
				}

				// checking --resize switch
				if ((STRING_COMPARE_FIX(myarg, "-r", 2)) || (STRING_COMPARE_FIX(myarg, "--resize", 8))){
					char num [16] = "";
					int num_max = 16;
					int num_offs;
					if (STRING_COMPARE_FIX(myarg, "--resize", 8)){
						num_offs = 8;
					}
					else{
						num_offs = 2;
					}
					int i = 0;
					while ((myarg[num_offs+i] != 0) && (myarg[num_offs+i] != '%') && (i < num_max)){
						num[i] = myarg[num_offs+i];
						i++;
					}
					num [i] = 0;
					if (i >= num_max){ STRING_PRINT("error: bad parameters\n"); return 0; }
					// is the resize value given in percentage?
					if (myarg[num_offs+i] == '%') opt_resize_percent = 1;

					if (STRING_CONVERT_TO_INTEGER(num, &opt_resize) == 0){
						STRING_PRINT("error: bad parameters\n"); return 0; }

					if (opt_resize_percent){
						if ((opt_resize >= 1) && (opt_resize <= 100)){ opt_wrong = 0; }
						else{ STRING_PRINT("error: bad parameters\n"); return 0; }
					}
					else{
						if ((opt_resize >= 1) && (opt_resize <= 100000)){ opt_wrong = 0; }
						else{ STRING_PRINT("error: bad parameters\n"); return 0; }
					}

				}

				// checking --rotate and --flip switches
				if (STRING_COMPARE(myarg, "--rotate90\0")){ opt_wrong = 0; opt_rotate90 = 1; }
				if (STRING_COMPARE(myarg, "--rotate180\0")){ opt_wrong = 0; opt_rotate180 = 1; }
				if (STRING_COMPARE(myarg, "--rotate270\0")){ opt_wrong = 0; opt_rotate270 = 1; }
				if (STRING_COMPARE(myarg, "--flipx\0")){ opt_wrong = 0; opt_flipx = 1; }
				if (STRING_COMPARE(myarg, "--flipy\0")){ opt_wrong = 0; opt_flipy = 1; }

				// we remember if there were switches but given in bad way
				if (opt_wrong){ opt_wrong_total = 1; }
				else{ opt_counter++; }
			}
			else {
				// expanding the file list that contains the files to be processed one by one
				FILE_LIST_ADD(myarg);
			}

		}

		//error message if bad parameters
		if (opt_wrong_total){ STRING_PRINT("error: bad parameters\n"); return 0; }
		else{

			// check if only 1 type of output format is specified on input
			int format_cnt = 0;
			if (opt_jpg) format_cnt++;
			if (opt_jp2) format_cnt++;
			if (opt_png) format_cnt++;
			if (format_cnt > 1){
				STRING_PRINT("error: more than one output format specified\n");
				return 0; }

			// managing info switches
			if (opt_version) { PRINT_VERSION(); PRINT_LICENSE(); }
			if (opt_help)    { PRINT_DESCRIPTION(); PRINT_HELP(); }

			// AUTOADJUST DEFAULT = 1 IF NO OTHER PARAMETER EXCEPT FILENAME
			// if no parameters given except filenames, then the --autoadjust parameter is 1 by default
			if ((opt_counter <= 0) && (file_name_counter > 0)) { opt_autoadjust = 1; opt_counter++; }

			// Main processing work of the switches (load -> process -> save)
			// ------------------------------------------------------------------
			// checking switches that need filename as input
			if ((opt_autoadjust != 0) ||
			    (opt_overwrite  != 0) ||
			    (opt_jpg        != 0) ||
			    (opt_jp2        != 0) ||
			    (opt_png        != 0) ||
			    (opt_resize     != 0) ||
			    (opt_rotate90   != 0) ||
			    (opt_rotate180  != 0) ||
			    (opt_rotate270  != 0) ||
			    (opt_flipx      != 0) ||
			    (opt_flipy      != 0) ||
			    (opt_quality    != 0) ||
			    (opt_recursive  != 0) ||
			    (opt_speed      != 0) ||
			    (opt_test       != 0)){

				// there were filenames given as input
				if (file_name_counter > 0){

					// if there are more than 1 files to read, then no error messages are print during process
					//if (file_name_counter > 1) opt_quiet = 1;

					// if the number of input files is greater than 1, then we process them one by one
					char tt[max_char];
					int c = 0;
					int d;
					int res; // we store the result value of the MAIN_RUN function
                        	                 // only it has no sense currently because it is not used any further yet
					int i;
					for (i=0; i<file_name_counter; i++){
						d = 0;
						while (file_name_buffer[c] != 0){
							tt[d] = file_name_buffer[c];
							d++;
							c++;
						}
						tt[d] = 0; d++; c++;
						// --------------------------------------------------------
							res = MAIN_RUN(tt);
						// --------------------------------------------------------
					}
				}

				// error here, because no input filenames were given
				else {
					STRING_PRINT("error: filename missing\n");
					return 0;
				}
			}
		}
	}

	// if no switches or filenames or any parameters were given,
	// -saying it other way the argument counter = 1 which is the command name itself-
	// then print something
	else {
		//PRINT_VERSION();
		//PRINT_DESCRIPTION();
		//PRINT_LICENSE();
		//PRINT_HELP();

		STRING_PRINT("No parameters given. Use -h (--help) or 'man aaphoto' for usage information\n");
	}

	return 1;
}




int MAIN_INIT(void)
{
	// number of files
	file_counter = 0;
	// number of files stored in file_name_buffer
	file_name_counter = 0;
	// pointer pointing to the end of the filenames read from the files that are stored in file_name_buffer
	// saying it other way, it is the offset pointer pointing to the next filename
	file_name_buffer_pointer = 0;
	// allocating memory
	file_name_buffer = malloc(max_file_name_buffer);
	if (file_name_buffer == 0){
		STRING_PRINT("error: memory allocation failure\n"); return 0; }

	speed_aargb = 0;
	speed_resize = 0;
	time_load = 0;
	time_save = 0;
	time_all = 0;

#ifndef __BMP_ONLY__
	bitmap_format_png_interlace_type = PNG_INTERLACE_NONE;
	bitmap_format_png_compression_type = PNG_COMPRESSION_TYPE_DEFAULT;
	bitmap_format_png_filter_type = PNG_FILTER_TYPE_DEFAULT;
#endif

	return 1;
}



int main(int argc, char **argv)
{
	long tim1_main, tim2_main;
	tim1_main = clock();

	if (MAIN_INIT() == 0) return 0;

	long result = 0;
	if (MAIN_ARGUMENTS_READ(argc, argv) == 0){ result = 1; }

	tim2_main = clock();
	time_all = (float)(tim2_main - tim1_main) / time_divider;
	if (time_all < 0.009) time_all = 0;

	/* print speed and time*/
	if ((opt_speed) && (result == 0)){
		STRING_PRINTF("%.2f sec for all operations\n", time_all);
	}

   	exit(EXIT_SUCCESS);
}
