/* -*- mode: C++; tab-width: 8; c-basic-offset: 8; indent-tabs-mode: t; -*- */

//#define DEBUGTAG "Replayer"
#define USE_BREAKPOINT_TARGET 1

#include "replayer.h"

#include <assert.h>
#include <err.h>
#include <fcntl.h>
#include <inttypes.h>
#include <limits.h>
#include <netinet/in.h>
#include <sched.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/mman.h>
#include <sys/personality.h>
#include <sys/poll.h>
#include <asm/ptrace-abi.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/user.h>
#include <sys/wait.h>
#include <unistd.h>

#include <map>
#include <sstream>
#include <string>
#include <vector>

#include "preload/syscall_buffer.h"

#include "dbg.h"
#include "debugger_gdb.h"
#include "hpc.h"
#include "replay_syscall.h"
#include "task.h"
#include "trace.h"
#include "util.h"

#include <perfmon/pfmlib_perf_event.h>

/* Why a skid region?  Interrupts generated by perf counters don't
 * fire at exactly the programmed point (as of 2013 kernel/HW);
 * there's a variable slack region, which is technically unbounded.
 * This means that an interrupt programmed for retired branch k might
 * fire at |k + 50|, for example.  To counteract the slack, we program
 * interrupts just short of our target, by the |SKID_SIZE| region
 * below, and then more slowly advance to the real target.
 *
 * How was this magic number determined?  Trial and error: we want it
 * to be as small as possible for efficiency, but not so small that
 * overshoots are observed.  If all other possible causes of overshoot
 * have been ruled out, like memory divergence, then you'll know that
 * this magic number needs to be increased if the following symptom is
 * observed during replay.  Running with DEBUGLOG enabled (see above),
 * a sequence of log messages like the following will appear
 *
 * 1. programming interrupt for [target - SKID_SIZE] rcbs
 * 2. Error: Replay diverged.  Dumping register comparison.
 * 3. Error: [list of divergent registers; arbitrary]
 * 4. Error: overshot target rcb=[target] by [i]
 *
 * The key is that no other replayer log messages occur between (1)
 * and (2).  This spew means that the replayer programmed an interrupt
 * for rcb=[target-SKID_SIZE], but the tracee was actually interrupted
 * at rcb=[target+i].  And that in turn means that the kernel/HW
 * skidded too far past the programmed target for rr to handle it.
 *
 * If that occurs, the SKID_SIZE needs to be increased by at least
 * [i].
 *
 * NB: there are probably deeper reasons for the target slack that
 * could perhaps let it be deduced instead of arrived at empirically;
 * perhaps pipeline depth and things of that nature are involved.  But
 * those reasons if they exit are currently not understood.
 */
#define SKID_SIZE 70

using namespace std;

// |parent| is the (potential) debugger client.  It waits until the
// server, |child|, creates a debug socket.  Then the client exec()s
// the debugger over itself.
static pid_t parent, child;
// The server writes debugger params to the pipe, and the clients
// reads those params out.
static int debugger_params_pipe[2];

// When we notify the debugger of process exit, it wants to be able to
// poke around at that last task.  So we store it here to allow
// processing debugger requests for it later.
static Task* last_task;
// Another little hack: technically replayer doesn't know about the
// fact that debugger_gdb hides all but one tgid from the gdb client.
// But to recognize the last_task above (another little hack), we need
// to known when an exiting thread from the target task group is the
// last.
pid_t debugged_tgid;

/* Nonzero after the first exec() has been observed during replay.
 * After this point, the first recorded binary image has been exec()d
 * over the initial rr image. */
bool validate = false;

/**
 * Restart a fresh debugging session, possibly updating our replay
 * params based on the debugger's |req|.  This results in the old
 * debugging socket being closed, and a new one created that the user
 * must manually connect to.
 */
static void restart_replay(struct dbg_context* dbg, struct dbg_request req);

static void debug_memory(Task* t)
{
	if (should_dump_memory(t, t->trace)) {
		dump_process_memory(t, t->trace.global_time, "rep");
	}
	if (validate && should_checksum(t, t->trace)) {
		/* Validate the checksum we computed during the
		 * recording phase. */
		validate_process_memory(t, t->trace.global_time);
	}
}

/**
 * Return the value of |reg| in |regs|, or set |*defined = 0| and
 * return an undefined value if |reg| isn't found.
 */
static long get_reg(const struct user_regs_struct* regs, DbgRegister reg,
		    int* defined)
{
	*defined = 1;
	switch (reg) {
	case DREG_EAX: return regs->eax;
	case DREG_ECX: return regs->ecx;
	case DREG_EDX: return regs->edx;
	case DREG_EBX: return regs->ebx;
	case DREG_ESP: return regs->esp;
	case DREG_EBP: return regs->ebp;
	case DREG_ESI: return regs->esi;
	case DREG_EDI: return regs->edi;
	case DREG_EIP: return regs->eip;
	case DREG_EFLAGS: return regs->eflags;
	case DREG_CS: return regs->xcs;
	case DREG_SS: return regs->xss;
	case DREG_DS: return regs->xds;
	case DREG_ES: return regs->xes;
	case DREG_FS: return regs->xfs;
	case DREG_GS: return regs->xgs;
	case DREG_ORIG_EAX: return regs->orig_eax;
	default:
		*defined = 0;
		return 0;
	}
}

static dbg_threadid_t get_threadid(Task* t)
{
	dbg_threadid_t thread;
	thread.pid = t->tgid();
	thread.tid = t->rec_tid;
	return thread;
}

static byte* read_mem(Task* t, void* addr, size_t len, size_t* read_len)
{
	byte* buf = (byte*)malloc(len);
	ssize_t nread = t->read_bytes_fallible(addr, len, buf);
	*read_len = max(0, nread);
	return buf;
}

/**
 * Reply to debugger requests until the debugger asks us to resume
 * execution.
 */
static struct dbg_request process_debugger_requests(struct dbg_context* dbg,
						    Task* t)
{
	if (!dbg) {
		struct dbg_request continue_all_tasks;
		memset(&continue_all_tasks, 0, sizeof(continue_all_tasks));
		continue_all_tasks.type = DREQ_CONTINUE;
		continue_all_tasks.target = DBG_ALL_THREADS;
		return continue_all_tasks;
	}
	while (1) {
		debug("%d (rec:%d): getting next debugger request ...",
		      t->tid, t->rec_tid);
		struct dbg_request req = dbg_get_request(dbg);
		Task* target = NULL;

		if (dbg_is_resume_request(&req)) {
			debug("  is resume request");
			return req;
		}

		// Debugger client requested that we restart execution
		// from the beginning.  Restart our debug session.
		if (DREQ_RESTART == req.type) {
			restart_replay(dbg, req);
			fatal("Not reached");
		}

		/* These requests don't require a target task. */
		switch (req.type) {
		case DREQ_GET_CURRENT_THREAD:
			dbg_reply_get_current_thread(dbg, get_threadid(t));
			continue;
		case DREQ_GET_OFFSETS:
			/* TODO */
			dbg_reply_get_offsets(dbg);
			continue;
		case DREQ_GET_THREAD_LIST: {
			size_t len = Task::count();
			vector<dbg_threadid_t> tids;
			for (auto it = Task::begin(); it != Task::end(); ++it) {
				Task* t = it->second;
				tids.push_back(get_threadid(t));
			}
			dbg_reply_get_thread_list(dbg, tids.data(), len);
			continue;
		}
		case DREQ_INTERRUPT:
			/* Tell the debugger we stopped and await
			 * further instructions. */
			dbg_notify_stop(dbg, get_threadid(t), 0);
			continue;
		default:
			/* fall through to next switch stmt */
			break;
		}

		target = (req.target.tid > 0) ? Task::find(req.target.tid) : t;
		/* These requests query or manipulate which task is
		 * the target, so it's OK if the task doesn't
		 * exist. */
		switch (req.type) {
		case DREQ_GET_IS_THREAD_ALIVE:
			dbg_reply_get_is_thread_alive(dbg, !!target);
			continue;
		case DREQ_GET_THREAD_EXTRA_INFO:
			dbg_reply_get_thread_extra_info(
				dbg, target->name().c_str());
			continue;
		case DREQ_SET_CONTINUE_THREAD:
		case DREQ_SET_QUERY_THREAD:
			dbg_reply_select_thread(dbg, !!target);
			continue;
		default:
			/* fall through to next switch stmt */
			break;
		}

		/* These requests require a valid target task.  We
		 * don't trust the debugger to use the information
		 * provided above to only query valid tasks. */
		if (!target) {
			dbg_notify_no_such_thread(dbg, &req);
			continue;
		}
		switch (req.type) {
		case DREQ_GET_AUXV: {
			// FIXME: translate tid to rec_tid
			pid_t tid = req.target.tid > 0 ? req.target.tid : t->tid;
			char filename[] = "/proc/01234567890/auxv";
			int fd;
			struct dbg_auxv_pair auxv[4096];
			ssize_t len;

			snprintf(filename, sizeof(filename) - 1,
				 "/proc/%d/auxv", tid);
			fd = open(filename, O_RDONLY);
			if (0 > fd) {
				dbg_reply_get_auxv(dbg, NULL, -1);
				continue;
			}

			len = read(fd, auxv, sizeof(auxv));
			if (0 > len) {
				dbg_reply_get_auxv(dbg, NULL, -1);
				continue;
			}

			assert(0 == len % sizeof(auxv[0]));
			len /= sizeof(auxv[0]);
			dbg_reply_get_auxv(dbg, auxv, len);
			continue;
		}
		case DREQ_GET_MEM: {
			size_t len;
			byte* mem = read_mem(target, req.mem.addr, req.mem.len,
					     &len);
			dbg_reply_get_mem(dbg, mem, len);
			free(mem);
			continue;
		}
		case DREQ_GET_REG: {
			dbg_regvalue_t val;

			val.value = get_reg(&target->regs(),
					    req.reg, &val.defined);
			dbg_reply_get_reg(dbg, val);
			continue;
		}
		case DREQ_GET_REGS: {
			struct user_regs_struct regs;
			struct dbg_regfile file;
			dbg_regvalue_t* val;

			memset(&file, 0, sizeof(file));
			for (int i = DREG_EAX; i < DREG_NUM_USER_REGS; ++i) {
				val = &file.regs[i];
				val->value = get_reg(&target->regs(),
						     DbgRegister(i),
						     &val->defined);
			}
			val = &file.regs[DREG_ORIG_EAX];
			val->value = get_reg(&regs, DREG_ORIG_EAX,
					     &val->defined);

			dbg_reply_get_regs(dbg, &file);
			continue;
		}
		case DREQ_GET_STOP_REASON: {
			dbg_reply_get_stop_reason(dbg, get_threadid(target),
						  target->child_sig);
			continue;
		}
		case DREQ_SET_SW_BREAK:
			assert_exec(target,
				    (req.mem.len ==
				     sizeof(AddressSpace::breakpoint_insn)),
				    "Debugger setting bad breakpoint insn");
			target->vm()->set_breakpoint(req.mem.addr,
						     TRAP_BKPT_USER);
			dbg_reply_watchpoint_request(dbg, 0);
			continue;
		case DREQ_REMOVE_SW_BREAK:
			target->vm()->remove_breakpoint(req.mem.addr,
							TRAP_BKPT_USER);
			dbg_reply_watchpoint_request(dbg, 0);
			break;
		case DREQ_REMOVE_HW_BREAK:
		case DREQ_REMOVE_RD_WATCH:
		case DREQ_REMOVE_WR_WATCH:
		case DREQ_REMOVE_RDWR_WATCH:
		case DREQ_SET_HW_BREAK:
		case DREQ_SET_RD_WATCH:
		case DREQ_SET_WR_WATCH:
		case DREQ_SET_RDWR_WATCH:
			dbg_reply_watchpoint_request(dbg, -1);
			continue;
		default:
			fatal("Unknown debugger request %d", req.type);
		}
	}
}

/**
 * Return the task that was recorded running the next event in the
 * trace, and should be replayed now.  Return nullptr if the trace was
 * interrupted at this point in recording.  If so, |*intr_t| is set to
 * the task recorded along with that event, if there was one.
 * |intr_t| may be nullptr.
 */
static trace_frame cur_trace_frame;
static Task* schedule_task(Task** intr_t = nullptr)
{
	read_next_trace(&cur_trace_frame);
	if (EV_TRACE_TERMINATION == cur_trace_frame.ev.type) {
		if (intr_t) {
			*intr_t = Task::find(cur_trace_frame.tid);
		}
		return nullptr;
	}

	Task *t = Task::find(cur_trace_frame.tid);
	assert(t != NULL);

	memcpy(&(t->trace), &cur_trace_frame, sizeof(t->trace));

	// Subsequent reschedule-events of the same thread can be
	// combined to a single event.  This meliorization is a
	// tremendous win.
	if (t->trace.ev.type == EV_SCHED) {
		bool combined = false;
		struct trace_frame next_trace;

		peek_next_trace(&next_trace);
		int64_t rbc = t->trace.rbc;
		while (EV_SCHED == next_trace.ev.type
		       && next_trace.tid == t->rec_tid) {
			rbc += next_trace.rbc;
			read_next_trace(&(t->trace));
			peek_next_trace(&next_trace);
			combined = true;
		}

		if (combined) {
			t->trace.rbc = rbc;
		}
	}
	assert(get_global_time() == t->trace.global_time);
	return t;
}

/**
 * Compares the register file as it appeared in the recording phase
 * with the current register file.
 */
static void validate_args(int event, int state, Task* t)
{
	struct user_regs_struct rec_regs = t->trace.recorded_regs;

	/* don't validate anything before execve is done as the actual
	 * process did not start prior to this point */
	if (!validate) {
		return;
	}
	if ((SYS_pwrite64 == event || SYS_pread64 == event)
	    && STATE_SYSCALL_EXIT == state) {
		/* The x86 linux 3.5.0-36 kernel packaged with Ubuntu
		 * 12.04 has been observed to mutate $esi across
		 * syscall entry/exit.  (This has been verified
		 * outside of rr as well; not an rr bug.)  It's not
		 * clear whether this is a ptrace bug or a kernel bug,
		 * but either way it's not supposed to happen.  So we
		 * fudge registers here to cover up that bug. */
		if (t->regs().esi != rec_regs.esi) {
			log_warn("Probably saw kernel bug mutating $esi across pread/write64 call: recorded:0x%lx; replaying:0x%lx.  Fudging registers.",
				 rec_regs.esi, t->regs().esi);
			rec_regs.esi = t->regs().esi;
		}
	}
	assert_child_regs_are(t, &rec_regs);
}

/** Return true when |t|'s $ip points at a syscall instruction. */
static bool entering_syscall_insn(Task* t)
{
	const byte sysenter[] = { 0x0f, 0x34 };
	const byte int_0x80[] = { 0xcd, 0x80 };
	static_assert(sizeof(sysenter) == sizeof(int_0x80), "Must ==");
	byte insn[sizeof(sysenter)];

	t->read_bytes(t->ip(), insn);
	return (!memcmp(insn, sysenter, sizeof(sysenter))
		|| !memcmp(insn, int_0x80, sizeof(int_0x80)));
}

/**
 * Continue until reaching either the "entry" of an emulated syscall,
 * or the entry or exit of an executed syscall.  |emu| is nonzero when
 * we're emulating the syscall.  Return 0 when the next syscall
 * boundary is reached, or nonzero if advancing to the boundary was
 * interrupted by an unknown trap.
 */
enum { EXEC = 0, EMU = 1 };
static int cont_syscall_boundary(Task* t, int emu, int stepi)
{
	bool is_syscall_entry = (STATE_SYSCALL_ENTRY == t->trace.ev.state);
	if (is_syscall_entry) {
		t->stepped_into_syscall = false;
	}

	ResumeRequest resume_how;
	if (emu && stepi) {
		resume_how = RESUME_SYSEMU_SINGLESTEP;
	} else if (emu) {
		resume_how = RESUME_SYSEMU;
	} else if (stepi) {
		resume_how = RESUME_SINGLESTEP;
		// Annoyingly, PTRACE_SINGLESTEP doesn't raise
		// PTRACE_O_SYSGOOD traps.  (Unlike
		// PTRACE_SINGLESTEP_SYSEMU, which does.)  That means
		// if we just blindly stepi'd the tracee to a
		// non-emulated syscall, we'd shoot right past it
		// without knowing.
		//
		// The correct solution to this problem is to emulate
		// all syscalls during replay and then inject the
		// executed ones (as we do for mmap).  But for now,
		// work around this problem by recognizing syscall
		// insns and issuing PTRACE_SYSCALL to enter them
		// instead of PTRACE_SINGLESTEP.
		if (entering_syscall_insn(t) || t->stepped_into_syscall) {
			resume_how = RESUME_SYSCALL;
			// Leave this breadcrumb on syscall entry so
			// that we know to issue PTRACE_SYSCALL to
			// exit the syscall the next time we stepi.
			t->stepped_into_syscall = is_syscall_entry;
		}
	} else {
		resume_how = RESUME_SYSCALL;
	}
	t->resume_execution(resume_how, RESUME_WAIT);

	switch ((t->child_sig = t->pending_sig())) {
	case 0:
		break;
	case SIGCHLD:
		/* SIGCHLD is pending, do not deliver it, wait for it
		 * to appear in the trace SIGCHLD is the only signal
		 * that should ever be generated as all other signals
		 * are emulated! */
		return cont_syscall_boundary(t, emu, stepi);
	case SIGTRAP:
		return 1;
	default:
		assert_exec(t, 0, "Replay got unrecorded signal %d",
			    t->child_sig);
	}

	assert(t->child_sig == 0);

	return 0;
}

/**
 *  Step over the system call instruction to "exit" the emulated
 *  syscall.
 *
 * XXX verify that this can't be interrupted by a breakpoint trap
 */
static void step_exit_syscall_emu(Task *t)
{
	struct user_regs_struct r = t->regs();
	t->cont_sysemu_singlestep();
	t->set_regs(r);

	t->force_status(0);
}

/**
 * Advance to the next syscall entry (or virtual entry) according to
 * |step|.  Return 0 if successful, or nonzero if an unhandled trap
 * occurred.
 */
static int enter_syscall(Task* t,
			 const struct rep_trace_step* step,
			 int stepi)
{
	int ret;
	if ((ret = cont_syscall_boundary(t, step->syscall.emu, stepi))) {
		return ret;
	}
	validate_args(step->syscall.no, STATE_SYSCALL_ENTRY, t);
	return ret;
}

/**
 * Advance past the reti (or virtual reti) according to |step|.
 * Return 0 if successful, or nonzero if an unhandled trap occurred.
 */
static int exit_syscall(Task* t,
			const struct rep_trace_step* step,
			int stepi)
{
	int i, emu = step->syscall.emu;

	if (!emu) {
		int ret = cont_syscall_boundary(t, emu, stepi);
		if (ret) {
			return ret;
		}
	}

	for (i = 0; i < step->syscall.num_emu_args; ++i) {
		t->set_data_from_trace();
	}
	if (step->syscall.emu_ret) {
		t->set_return_value_from_trace();
	}
	validate_args(step->syscall.no, STATE_SYSCALL_EXIT, t);

	if (emu) {
		step_exit_syscall_emu(t);
	}
	return 0;
}

/**
 * Advance |t| to the next signal or trap.  If |stepi| is |STEPI|,
 * then execution resumes by single-stepping.  Otherwise it continues
 * normally.  The delivered signal is recorded in |t->child_sig|.
 */
enum { DONT_STEPI = 0, STEPI };
static void continue_or_step(Task* t, int stepi)
{
	int child_sig_gt_zero;

	ResumeRequest resume_how;
	if (stepi) {
		resume_how = RESUME_SINGLESTEP;
	} else {
		/* We continue with RESUME_SYSCALL for error checking:
		 * since the next event is supposed to be a signal,
		 * entering a syscall here means divergence.  There
		 * shouldn't be any straight-line execution overhead
		 * for SYSCALL vs. CONT, so the difference in cost
		 * should be neglible. */
		resume_how = RESUME_SYSCALL;
	}
	t->resume_execution(resume_how, RESUME_WAIT);

	t->child_sig = t->pending_sig();
	child_sig_gt_zero = (0 < t->child_sig);
	if (child_sig_gt_zero) {
		return;
	}
	assert_exec(t, child_sig_gt_zero,
		    "Replaying `%s': expecting tracee signal or trap, but instead at `%s' (rcb: %lld)",
		    Event(t->trace.ev).str().c_str(),
		    syscallname(t->regs().orig_eax),
		    read_rbc(t->hpc));
}

/**
 * Return nonzero if |t| was stopped for a breakpoint trap (int3),
 * as opposed to a trace trap.  Return zero in the latter case.
 */
static int is_breakpoint_trap(Task* t)
{
	siginfo_t si;

	assert(SIGTRAP == t->child_sig);

	t->get_siginfo(&si);
	assert(SIGTRAP == si.si_signo);

	/* XXX unable to find docs on which of these "should" be
	 * right.  The SI_KERNEL code is seen in the int3 test, so we
	 * at least need to handle that. */
	return SI_KERNEL == si.si_code || TRAP_BRKPT == si.si_code;
}

/**
 * Return one of the (non-zero) enumerated TRAP_* debugger-trap types
 * above if the SIGTRAP generated by the child is intended for the
 * debugger, or zero if it's meant for rr internally.
 *
 * NB: calling this function while advancing the rbc through hpc
 * interrupts when emulating asynchronous signal delivery *will*
 * result in bad results.  Don't call this function from there; it's
 * not necessary.
 */
enum SigDeliveryType { ASYNC, DETERMINISTIC };
enum ExecStateType { UNKNOWN, NOT_AT_TARGET, AT_TARGET };
static TrapType compute_trap_type(Task* t, int target_sig,
				SigDeliveryType delivery,
				ExecStateType exec_state,
				int stepi)
{
	TrapType trap_type;

	assert(SIGTRAP == t->child_sig);

	/* We're not replaying a trap, and it was clearly raised on
	 * behalf of the debugger.  (The debugger will verify
	 * that.) */
	if (SIGTRAP != target_sig
	    /* Replay of deterministic signals never internally
	     * single-steps or sets internal breakpoints. */
	    && (DETERMINISTIC == delivery
		/* Replay of async signals will sometimes internally
		 * single-step when advancing to an execution target,
		 * so the trap was only clearly for the debugger if
		 * the debugger was requesting single-stepping. */
		|| (stepi && NOT_AT_TARGET == exec_state))) {
		return stepi ? TRAP_STEPI : TRAP_BKPT_USER;
	}

	/* We're trying to replay a deterministic SIGTRAP, or we're
	 * replaying an async signal. */

	trap_type = t->vm()->get_breakpoint_type_at_ip(t->ip());
	if (TRAP_BKPT_USER == trap_type || TRAP_BKPT_INTERNAL == trap_type) {
		assert(is_breakpoint_trap(t));
		return trap_type;
	}

	if (is_breakpoint_trap(t)) {
		/* We successfully replayed a recorded deterministic
		 * SIGTRAP.  (Because it must have been raised by an
		 * |int3|, but not one we injected.)  Not for the
		 * debugger, although we'll end up notifying it
		 * anyway. */
		assert(DETERMINISTIC == delivery);
		return TRAP_NONE;
	}

	if (DETERMINISTIC == delivery) {
		/* If the delivery of SIGTRAP is supposed to be
		 * deterministic and we didn't just retire an |int 3|
		 * and this wasn't a breakpoint, we must have been
		 * single stepping.  So definitely for the
		 * debugger. */
		assert(stepi);
		return TRAP_STEPI;
	}

	/* We're replaying an async signal. */

	if (AT_TARGET == exec_state) {
		/* If we're at the target of the async signal
		 * delivery, prefer delivering the signal to retiring
		 * a possible debugger single-step; we'll notify the
		 * debugger anyway. */
		return TRAP_NONE;
	}

	/* Otherwise, we're not at the execution target, so may have
	 * been internally single-stepping.  We'll notify the debugger
	 * if it was also requesting single-stepping.  The debugger
	 * won't care about the rr-internal trap if it wasn't
	 * requesting single-stepping. */
	return stepi ? TRAP_STEPI : TRAP_NONE;
}

/**
 * Shortcut for callers that don't care about internal breakpoints.
 * Return nonzero if |t|'s trap is for the debugger, zero otherwise.
 */
static int is_debugger_trap(Task* t, int target_sig,
			    SigDeliveryType delivery, ExecStateType exec_state,
			    int stepi)
{
	TrapType type = compute_trap_type(t, target_sig, delivery, exec_state,
					  stepi);
	assert(TRAP_BKPT_INTERNAL != type);
	return TRAP_NONE != type;
}

static void guard_overshoot(Task* t,
			    const struct user_regs_struct* target_regs,
			    int64_t target_rcb, int64_t remaining_rcbs)
{
	int remaining_rcbs_gt_0 = remaining_rcbs >= 0;
	if (!remaining_rcbs_gt_0) {
		long target_ip = target_regs->eip;

		log_err("Replay diverged.  Dumping register comparison.");
		/* Cover up the internal breakpoint that we may have
		 * set, and restore the tracee's $ip to what it would
		 * have been had it not hit the breakpoint (if it did
		 * hit the breakpoint).*/
		t->vm()->remove_breakpoint((void*)target_ip,
					   TRAP_BKPT_INTERNAL);
		if (t->regs().eip ==
		    long(target_ip + sizeof(AddressSpace::breakpoint_insn))) {
			t->move_ip_before_breakpoint();
		}
		compare_register_files(t, "rep overshoot", &t->regs(),
				       "rec", target_regs, LOG_MISMATCHES);
		assert_exec(t, remaining_rcbs_gt_0,
			    "overshot target rcb=%lld by %lld",
			    target_rcb, -remaining_rcbs);
	}
}

static void guard_unexpected_signal(Task* t)
{
	Event ev;
	int child_sig_is_zero_or_sigtrap = (0 == t->child_sig
					    || SIGTRAP == t->child_sig);
	/* "0" normally means "syscall", but continue_or_step() guards
	 * against unexpected syscalls.  So the caller must have set
	 * "0" intentionally. */
	if (child_sig_is_zero_or_sigtrap) {
		return;
	}
	if (t->child_sig) {
		ev = SignalEvent(t->child_sig, NONDETERMINISTIC_SIG);
	} else {
		ev = SyscallEvent(max(0L, t->regs().orig_eax));
	}
	assert_exec(t, child_sig_is_zero_or_sigtrap,
		    "Replay got unrecorded event %s while awaiting signal\n",
		    ev.str().c_str());
}

static int is_same_execution_point(Task* t,
				   const struct user_regs_struct* rec_regs,
				   int64_t rcbs_left)
{
	int behavior =
#ifdef DEBUGTAG
				   LOG_MISMATCHES
#else
				   EXPECT_MISMATCHES
#endif
		;
	if (0 != rcbs_left) {
		debug("  not same execution point: %lld rcbs left (@%p)",
		      rcbs_left, (void*)rec_regs->eip);
#ifdef DEBUGTAG
		compare_register_files(t, "(rep)", &t->regs(),
				       "(rec)", rec_regs, LOG_MISMATCHES);
#endif
		return 0;
	}
	if (compare_register_files(t, "rep", &t->regs(), "rec", rec_regs,
				   behavior)) {
		debug("  not same execution point: regs differ (@%p)",
		      (void*)rec_regs->eip);
		return 0;
	}
	debug("  same execution point");
	return 1;
}

/**
 * Run execution forwards for |t| until |*rcb| is reached, and the $ip
 * reaches the recorded $ip.  Return 0 if successful or 1 if an
 * unhandled interrupt occurred.  |sig| is the pending signal to be
 * delivered; it's only used to distinguish debugger-related traps
 * from traps related to replaying execution.  |rcb| is an inout param
 * that will be decremented by branches retired during this attempted
 * step.
 */
static int advance_to(Task* t, const struct user_regs_struct* regs,
		      int sig, int stepi, int64_t* rcb)
{
	pid_t tid = t->tid;
	byte* ip = (byte*)regs->eip;
	int64_t rcbs_left;

	assert(t->hpc->rbc.fd > 0);
	assert(t->child_sig == 0);

	/* Step 1: advance to the target rcb (minus a slack region) as
	 * quickly as possible by programming the hpc. */
	rcbs_left = *rcb - read_rbc(t->hpc);

	debug("advancing %lld rcbs to reach %lld/%p",
	      rcbs_left, *rcb, ip);

	/* XXX should we only do this if (rcb > 10000)? */
	while (rcbs_left - SKID_SIZE > SKID_SIZE) {
		if (SIGTRAP == t->child_sig) {
			/* We proved we're not at the execution
			 * target, and we haven't set any internal
			 * breakpoints, and we're not temporarily
			 * internally single-stepping, so we must have
			 * hit a debugger breakpoint or the debugger
			 * was single-stepping the tracee.  (The
			 * debugging code will verify that.) */
			return 1;
		}
		t->child_sig = 0;

		debug("  programming interrupt for %lld rcbs",
		      rcbs_left - SKID_SIZE);
		*rcb -= read_rbc(t->hpc);
		reset_hpc(t, rcbs_left - SKID_SIZE);

		continue_or_step(t, stepi);
		if (HPC_TIME_SLICE_SIGNAL == t->child_sig
		    || SIGCHLD == t->child_sig) {
			/* Tracees can receive SIGCHLD at pretty much
			 * any time during replay.  If we recorded
			 * delivery, we'll manually replay it
			 * eventually (or already have).  Just ignore
			 * here. */
			t->child_sig = 0;
		}
		guard_unexpected_signal(t);

		/* TODO this assertion won't catch many spurious
		 * signals; should assert that the siginfo says the
		 * source is io-ready and the fd is the child's fd. */
		if (fcntl(t->hpc->rbc.fd, F_GETOWN) != tid) {
			fatal("Scheduled task %d doesn't own hpc; replay divergence", tid);
		}

		rcbs_left = *rcb - read_rbc(t->hpc);
	}
	guard_overshoot(t, regs, *rcb, rcbs_left);

	/* Step 2: more slowly, find our way to the target rcb and
	 * execution point.  We set an internal breakpoint on the
	 * target $ip and then resume execution.  When that *internal*
	 * breakpoint is hit (i.e., not one incidentally also set on
	 * that $ip by the debugger), we check again if we're at the
	 * target rcb and execution point.  If not, we temporarily
	 * remove the breakpoint, single-step over the insn, and
	 * repeat.
	 *
	 * What we really want to do is set a (precise)
	 * retired-instruction interrupt and do away with all this
	 * cruft. */
	while (rcbs_left >= 0) {
		/* Invariants here are
		 *  o rcbs_left is up-to-date
		 *  o rcbs_left >= 0
		 *
		 * Possible state of the execution of |t|
		 *  0. at a debugger trap (breakpoint or stepi)
		 *  1. at an internal breakpoint
		 *  2. at the execution target
		 *  3. not at the execution target, but incidentally
		 *     at the target $ip
		 *  4. otherwise not at the execution target
		 *
		 * Determining whether we're at a debugger trap is
		 * surprisingly complicated, but we delegate the work
		 * to |compute_debugger_trap()|.  The rest can be
		 * straightforwardly computed with rbc value and
		 * registers. */
		int at_target;

		at_target = is_same_execution_point(t, regs, rcbs_left);
		if (SIGTRAP == t->child_sig) {
			TrapType trap_type = compute_trap_type(
				t, sig, ASYNC,
				at_target ? AT_TARGET : NOT_AT_TARGET,
				stepi);
			switch (trap_type) {
			case TRAP_BKPT_USER:
			case TRAP_STEPI:
				/* Case (0) above: interrupt for the
				 * debugger. */
				debug("    trap was debugger interrupt %d",
				      trap_type);
				return 1;
			case TRAP_BKPT_INTERNAL: {
				/* Case (1) above: cover the tracks of
				 * our internal breakpoint, and go
				 * check again if we're at the
				 * target. */
				debug("    trap was for target $ip");
				/* (The breakpoint would have trapped
				 * at the $ip one byte beyond the
				 * target.) */
				assert(!at_target);

				t->child_sig = 0;
				t->move_ip_before_breakpoint();
				/* We just backed up the $ip, but
				 * rewound it over an |int $3|
				 * instruction, which couldn't have
				 * retired a branch.  So we don't need
				 * to adjust |rcb_now|. */
				continue;
			}
			case TRAP_NONE:
				/* Otherwise, we must have been forced
				 * to single-step because the tracee's
				 * $ip was incidentally the same as
				 * the target.  Unfortunately, it's
				 * awkward to assert that here, so we
				 * don't yet.  TODO. */
				debug("    (SIGTRAP; stepi'd target $ip)");
				assert(!stepi);
				t->child_sig = 0;
				break;
			}
		}
		/* We had to keep the internal breakpoint set (if it
		 * was when we entered the loop) for the checks above.
		 * But now we're either done (at the target) or about
		 * to resume execution in one of a variety of ways,
		 * and it's simpler to start out knowing that the
		 * breakpoint isn't set. */
		t->vm()->remove_breakpoint(ip, TRAP_BKPT_INTERNAL);
		
		if (at_target) {
			/* Case (2) above: done. */
			return 0;
		}

		/* At this point, we've proven that we're not at the
		 * target execution point, and we've ensured the
		 * internal breakpoint is unset. */
		if (USE_BREAKPOINT_TARGET && regs->eip != t->regs().eip) {
			/* Case (4) above: set a breakpoint on the
			 * target $ip and PTRACE_CONT in an attempt to
			 * execute as many non-trapped insns as we
			 * can.  (Unless the debugger is stepping, of
			 * course.)  Trapping and checking
			 * are-we-at-target is slow.  It bears
			 * repeating that the ideal implementation
			 * would be programming a precise counter
			 * interrupt (insns-retired best of all), but
			 * we're forced to be conservative by observed
			 * imprecise counters.  This should still be
			 * no slower than single-stepping our way to
			 * the target execution point. */
			debug("    breaking on target $ip");
			t->vm()->set_breakpoint(ip, TRAP_BKPT_INTERNAL);
			continue_or_step(t, stepi);
		} else {
			/* Case (3) above: we can't put a breakpoint
			 * on the $ip, because resuming execution
			 * would just trap and we'd be back where we
			 * started.  Single-step past it. */
			debug("    (single-stepping over target $ip)");
			continue_or_step(t, STEPI);
		}

		if (HPC_TIME_SLICE_SIGNAL == t->child_sig
		    || SIGCHLD == t->child_sig) {
			/* See the long comment in "Step 1" above.
			 *
			 * We don't usually expect a time-slice signal
			 * during this phase, but it's possible for a
			 * SIGCHLD to interrupt the previous step just
			 * as the tracee enters the slack region,
			 * i.e., where an rbc signal was just about to
			 * fire.  (There's not really a non-racy way
			 * to disable the rbc interrupt, and we need
			 * to keep the counter running for overshoot
			 * checking anyway.)  So this is the most
			 * convenient way to squelch that "spurious"
			 * signal. */
			t->child_sig = 0;
		}
		guard_unexpected_signal(t);

		/* Maintain the "'rcbs_left'-is-up-to-date"
		 * invariant. */
		rcbs_left = *rcb - read_rbc(t->hpc);
	}
	guard_overshoot(t, regs, *rcb, rcbs_left);

	return 0;
}

/**
 * Emulates delivery of |sig| to |oldtask|.  Returns nonzero if
 * emulation was interrupted, zero if completed.
 */
static int emulate_signal_delivery(Task* oldtask, int sig, int sigtype)
{
	/* We are now at the exact point in the child where the signal
	 * was recorded, emulate it using the next trace line (records
	 * the state at sighandler entry). */
	Task* t;
	struct trace_frame* trace;

	reset_hpc(oldtask, 0);

	t = schedule_task();
	if (!t) {
		// Trace terminated abnormally.  We'll pop out to code
		// that knows what to do.
		return 1;
	}
	assert_exec(oldtask, t == oldtask, "emulate_signal_delivery changed task");
	trace = &t->trace;

	/* Restore the signal-hander frame data, if there was one. */
	bool restored_sighandler_frame = 0 < t->set_data_from_trace();
	if (restored_sighandler_frame) {
		debug("--> restoring sighandler frame for %s", signalname(sig));
		t->push_event(SignalEvent(sig, sigtype));
		t->ev().transform(EV_SIGNAL_DELIVERY);
		t->ev().transform(EV_SIGNAL_HANDLER);
	} else if (possibly_destabilizing_signal(t, sig)) {
		t->push_event(SignalEvent(sig, sigtype));
		t->ev().transform(EV_SIGNAL_DELIVERY);

		t->destabilize_task_group();

		t->pop_signal_delivery();
	}
	/* If this signal had a user handler, and we just set up the
	 * callframe, and we need to restore the $sp for continued
	 * execution. */
	t->set_regs(trace->recorded_regs);
	/* Delivered the signal. */
	t->child_sig = 0;

	validate_args(trace->ev.type, -1, t);
	return 0;
}

static void assert_at_recorded_rcb(Task* t, const Event& ev)
{
	static const int64_t rbc_slack = 0;
	int64_t rbc_now = t->hpc->started ? read_rbc(t->hpc) : 0;

	if (!validate) {
		return;
	}
	assert_exec(t, (!t->hpc->started
			|| llabs(rbc_now - t->trace.rbc) <= rbc_slack),
		    "rbc mismatch for '%s'; expected %lld, got %lld",
		    ev.str().c_str(), t->trace.rbc, read_rbc(t->hpc));
}

/**
 * Advance to the delivery of the deterministic signal |sig| and
 * update registers to what was recorded.  Return 0 if successful or 1
 * if an unhandled interrupt occurred.
 */
static int emulate_deterministic_signal(Task* t, int sig, int stepi)
{
	Event ev(t->trace.ev);

	continue_or_step(t, stepi);
	if (SIGCHLD == t->child_sig) {
		t->child_sig = 0;
		return emulate_deterministic_signal(t, sig, stepi);
	} else if (SIGTRAP == t->child_sig
		   && is_debugger_trap(t, sig, DETERMINISTIC, UNKNOWN, stepi)) {
		return 1;
	}
	assert_exec(t, t->child_sig == sig,
		    "Replay got unrecorded signal %d (expecting %d)",
		    t->child_sig, sig);
	assert_at_recorded_rcb(t, ev);

	if (EV_SEGV_RDTSC == ev.type()) {
		t->set_regs(t->trace.recorded_regs);
		/* We just "delivered" this pseudosignal. */
		t->child_sig = 0;
		return 0;
	}
	return emulate_signal_delivery(t, sig, DETERMINISTIC_SIG);
}

/**
 * Run execution forwards for |t| until |t->trace.rbc| is reached,
 * and the $ip reaches the recorded $ip.  After that, deliver |sig| if
 * nonzero.  Return 0 if successful or 1 if an unhandled interrupt
 * occurred.
 */
static int emulate_async_signal(Task* t,
				const struct user_regs_struct* regs, int sig,
				int stepi, int64_t* rcb)
{
	if (advance_to(t, regs, 0, stepi, rcb)) {
		return 1;
	}
	if (sig) {
		if (emulate_signal_delivery(t, sig, NONDETERMINISTIC_SIG)) {
			return 1;
		}
	}
	stop_hpc(t);
	return 0;
}

/**
 * Skip over the entry/exit of either an arm-desched-event or
 * disarm-desched-event ioctl(), as described by |ds|.  Return nonzero
 * if an unhandled interrupt occurred, zero if the ioctl() was
 * successfully skipped over.
 */
static int skip_desched_ioctl(Task* t,
			      struct rep_desched_state* ds, int stepi)
{
	int ret, is_desched_syscall;

	/* Skip ahead to the syscall entry. */
	if (DESCHED_ENTER == ds->state
	    && (ret = cont_syscall_boundary(t, EMU, stepi))) {
		return ret;
	}
	ds->state = DESCHED_EXIT;

	is_desched_syscall = (DESCHED_ARM == ds->type ?
			      t->is_arm_desched_event_syscall() :
			      t->is_disarm_desched_event_syscall());
	assert_exec(t, is_desched_syscall,
		    "Failed to reach desched ioctl; at %s(%ld, %ld) instead",
		    syscallname(t->regs().orig_eax),
		    t->regs().ebx, t->regs().ecx);
	/* Emulate a return value of "0".  It's OK for us to hard-code
	 * that value here, because the syscallbuf lib aborts if a
	 * desched ioctl returns non-zero (it doesn't know how to
	 * handle that). */
	struct user_regs_struct r = t->regs();
	r.eax = 0;
	t->set_regs(r);
	step_exit_syscall_emu(t);
	return 0;
}

/**
 * Buffer for recorded syscallbuf bytes.  By definition buffer flushes
 * must be replayed sequentially, so we can use one buffer for all
 * tracees.  At the start of the flush, the recorded bytes are read
 * back into this buffer.  Then they're copied back to the tracee
 * record-by-record, as the tracee exits those syscalls.
 */
static byte syscallbuf_flush_buffer[SYSCALLBUF_BUFFER_SIZE];
/** Convenience header pointer to skip tedious pointer arithmetic. */
static const struct syscallbuf_hdr* flush_hdr =
	(const struct syscallbuf_hdr*)syscallbuf_flush_buffer;

/**
 * Restore the recorded syscallbuf data to the tracee, preparing the
 * tracee for replaying the records.  Return the number of record
 * bytes and a pointer to the first record through outparams.
 */
static void prepare_syscallbuf_records(Task* t,
				       size_t* num_rec_bytes,
				       const struct syscallbuf_record** first_rec_rec,
				       struct syscallbuf_record** first_child_rec)
{
	/* Read the recorded syscall buffer back into the buffer
	 * region. */
	void* rec_addr;
	*num_rec_bytes = read_raw_data_direct(&t->trace,
					      syscallbuf_flush_buffer,
					      sizeof(syscallbuf_flush_buffer),
					      &rec_addr);

	/* The stored num_rec_bytes in the header doesn't include the
	 * header bytes, but the stored trace data does. */
	*num_rec_bytes -= sizeof(sizeof(struct syscallbuf_hdr));
	assert(rec_addr == t->syscallbuf_child);
	assert(flush_hdr->num_rec_bytes == *num_rec_bytes);

	*first_rec_rec = flush_hdr->recs;
	*first_child_rec = t->syscallbuf_hdr->recs;
}

/**
 * Bail if |t| isn't at the buffered syscall |syscallno|.
 */
static void assert_at_buffered_syscall(Task* t, int syscallno)
{
	assert_exec(t, t->is_untraced_syscall(),
		    "Bad ip %p: should have been buffered-syscall ip", t->ip());
	assert_exec(t, t->regs().orig_eax == syscallno,
		    "At %s; should have been at %s(%d)",
		    syscallname(t->regs().orig_eax),
		    syscallname(syscallno), syscallno);
}

/**
 * Bail if |rec_rec| and |rep_rec| haven't been prepared for the same
 * syscall (including desched'd-ness and reserved extra space).
 */
static void assert_same_rec(Task* t,
			    const struct syscallbuf_record* rec_rec,
			    struct syscallbuf_record* rep_rec)
{
	assert_exec(t,
		    rec_rec->syscallno == rep_rec->syscallno
		    && rec_rec->desched == rep_rec->desched
		    && rec_rec->size == rep_rec->size,
		    "Recorded rec { no=%d, desched:%d, size: %u } "
		    "being replayed as { no=%d, desched:%d, size: %u }",
		    rec_rec->syscallno, rec_rec->desched, rec_rec->size,
		    rep_rec->syscallno, rep_rec->desched, rep_rec->size);
}

/**
 * Directly restore the uaddr/uaddr2 outparams that were saved to
 * buffer.  Because the syscallbuf can't use scratch values for the
 * futexes, it can't restore the record data itself.
 */
static void restore_futex_words(Task* t,
				const struct syscallbuf_record* rec)
{
	static_assert(sizeof(uint32_t) == sizeof(long), "NYI: Task::write_int()");
	ssize_t extra_data_size = rec->size - sizeof(*rec);
	bool saved_uaddr2 = (2 * sizeof(uint32_t) == extra_data_size);
	assert_exec(t, sizeof(uint32_t) == extra_data_size || saved_uaddr2,
		    "Futex should have saved 4 or 8 bytes, but instead saved %d bytes",
		    extra_data_size);

	void* child_uaddr = (void*)t->regs().ebx;
	uint32_t rec_uaddr =
		*reinterpret_cast<const uint32_t*>(rec->extra_data);
	t->write_mem(child_uaddr, rec_uaddr);

	if (saved_uaddr2) {
		void* child_uaddr2 = (void*)t->regs().edi;
		uint32_t rec_uaddr2 =
			*reinterpret_cast<const uint32_t*>(rec->extra_data +
							   sizeof(uint32_t));
		t->write_mem(child_uaddr2, rec_uaddr2);
	}
}


/**
 * Try to flush one buffered syscall as described by |flush|.  Return
 * nonzero if an unhandled interrupt occurred, and zero if the syscall
 * was flushed (in which case |flush->state == DONE|).
 */
static int flush_one_syscall(Task* t,
			     struct rep_flush_state* flush, int stepi)
{
	const struct syscallbuf_record* rec_rec = flush->rec_rec;
	struct syscallbuf_record* child_rec = flush->child_rec;
	int call = rec_rec->syscallno;
	int ret;
	// TODO: use syscall_defs table information to determine this.
	int emu = (SYS_madvise == call) ? EXEC : EMU;

	switch (flush->state) {
	case FLUSH_START:
		assert_exec(t, 0 == ((uintptr_t)rec_rec & (sizeof(int) - 1)),
			    "Recorded record must be int-aligned, but instead is %p",
			    rec_rec);
		assert_exec(t, 0 == ((uintptr_t)child_rec & (sizeof(int) - 1)),
			    "Replaying record must be int-aligned, but instead is %p",
			    child_rec);
		assert_exec(t, MAX_SYSCALLNO >= call,
			    "Replaying unknown syscall %d", call);
		assert_exec(t, 0 == rec_rec->desched || 1 == rec_rec->desched,
			    "Recorded record is corrupted: rec->desched is %d",
			    rec_rec->desched);
		// We'll check at syscall entry that the recorded and
		// replayed record values match.

		debug("Replaying buffered `%s' (ret:%ld) which does%s use desched event",
		      syscallname(call), rec_rec->ret,
		      !rec_rec->desched ? " not" : "");

		if (!rec_rec->desched) {
			flush->state = FLUSH_ENTER;
		} else {
			flush->state = FLUSH_ARM;
			flush->desched.type = DESCHED_ARM;
			flush->desched.state = DESCHED_ENTER;
		}
		return flush_one_syscall(t, flush, stepi);

	case FLUSH_ARM:
		/* Skip past the ioctl that armed the desched
		 * notification. */
		debug("  skipping over arm-desched ioctl");
		if ((ret = skip_desched_ioctl(t, &flush->desched, stepi))) {
			return ret;
		}
		flush->state = FLUSH_ENTER;
		return flush_one_syscall(t, flush, stepi);

	case FLUSH_ENTER:
		debug("  advancing to buffered syscall entry");
		if ((ret = cont_syscall_boundary(t, emu, stepi))) {
			return ret;
		}
		assert_at_buffered_syscall(t, call);
		assert_same_rec(t, rec_rec, child_rec);
		flush->state = FLUSH_EXIT;
		return flush_one_syscall(t, flush, stepi);

	case FLUSH_EXIT: {
		debug("  advancing to buffered syscall exit");

		EmuFs::AutoGc gc(call);

		assert_at_buffered_syscall(t, call);

		// Restore saved trace data.
		memcpy(child_rec->extra_data, rec_rec->extra_data,
		       rec_rec->size);

		// Restore return value.
		// TODO: try to share more code with cont_syscall_boundary()
		if (!emu) {
			int ret = cont_syscall_boundary(t, emu, stepi);
			if (ret) {
				return ret;
			}
			assert_at_buffered_syscall(t, call);
		}
		struct user_regs_struct r = t->regs();
		r.eax = rec_rec->ret;
		t->set_regs(r);
		if (emu) {
			step_exit_syscall_emu(t);
		}

		switch (call) {
		case SYS_futex:
			restore_futex_words(t, rec_rec);
			break;
		case SYS_write:
			rep_maybe_replay_stdio_write(t);
			break;
		}

		if (!rec_rec->desched) {
			flush->state = FLUSH_DONE;
			return 0;
		}
		flush->state = FLUSH_DISARM;
		flush->desched.type = DESCHED_DISARM;
		flush->desched.state = DESCHED_ENTER;
		return flush_one_syscall(t, flush, stepi);
	}
	case FLUSH_DISARM:
		/* And skip past the ioctl that disarmed the desched
		 * notification. */
		debug("  skipping over disarm-desched ioctl");
		if ((ret = skip_desched_ioctl(t, &flush->desched, stepi))) {
			return ret;
		}
		flush->state = FLUSH_DONE;
		return 0;

	default:
		fatal("Unknown buffer-flush state %d", flush->state);
		return 0;	/* unreached */
	}
}

/**
 * Replay all the syscalls recorded in the interval between |t|'s
 * current execution point and the next non-syscallbuf event (the one
 * that flushed the buffer).  Return 0 if successful or 1 if an
 * unhandled interrupt occurred.
 */
static int flush_syscallbuf(Task* t, struct rep_trace_step* step,
			    int stepi)
{
	struct rep_flush_state* flush = &step->flush;

	if (flush->need_buffer_restore) {
		prepare_syscallbuf_records(t,
					   &flush->num_rec_bytes_remaining,
					   &flush->rec_rec, &flush->child_rec);
		flush->need_buffer_restore = 0;

		debug("Prepared %d bytes of syscall records",
		      flush->num_rec_bytes_remaining);
	}

	while (flush->num_rec_bytes_remaining > 0) {
		int ret;
		size_t stored_rec_size;

		if ((ret = flush_one_syscall(t, flush, stepi))) {
			return ret;
		}

		assert(FLUSH_DONE == flush->state);

		stored_rec_size = stored_record_size(flush->rec_rec->size);
		flush->rec_rec = (const struct syscallbuf_record*)
				 ((byte*)flush->rec_rec + stored_rec_size);
		flush->child_rec = (struct syscallbuf_record*)
				   ((byte*)flush->child_rec + stored_rec_size);
		flush->num_rec_bytes_remaining -= stored_rec_size;
		flush->state = FLUSH_START;

		debug("  %d bytes remain to flush",
		      flush->num_rec_bytes_remaining);
	}
	return 0;
}

/**
 * Try to execute |step|, adjusting for |req| if needed.  Return 0 if
 * |step| was made, or nonzero if there was a trap or |step| needs
 * more work.
 */
static int try_one_trace_step(Task* t,
			      struct rep_trace_step* step,
			      const struct dbg_request* req)
{
	if (DREQ_DETACH == req->type) {
		return 0;
	}
	int stepi = (DREQ_STEP == req->type && get_threadid(t) == req->target);
	switch (step->action) {
	case TSTEP_RETIRE:
		return 0;
	case TSTEP_ENTER_SYSCALL:
		return enter_syscall(t, step, stepi);
	case TSTEP_EXIT_SYSCALL:
		return exit_syscall(t, step, stepi);
	case TSTEP_DETERMINISTIC_SIGNAL:
		return emulate_deterministic_signal(t, step->signo, stepi);
	case TSTEP_PROGRAM_ASYNC_SIGNAL_INTERRUPT:
		return emulate_async_signal(t,
					    step->target.regs,
					    step->target.signo,
					    stepi,
					    &step->target.rcb);
	case TSTEP_FLUSH_SYSCALLBUF:
		return flush_syscallbuf(t, step, stepi);
	case TSTEP_DESCHED:
		return skip_desched_ioctl(t, &step->desched, stepi);
	default:
		fatal("Unhandled step type %d", step->action);
		return 0;
	}
}

/**
 * The trace was interrupted abnormally at this point during replay.
 * All we can do is notify the debugger, process its final requests,
 * and then die.
 */
static void handle_interrupted_trace(struct dbg_context* dbg,
				     Task* t = nullptr)
{
	if (!probably_not_interactive()) {
		fprintf(stderr,
			"\nrr: (Trace terminated early at this point during recording.)\n\n");
	}
	if (dbg) {
		dbg_notify_stop(dbg, t ? get_threadid(t) : DBG_ALL_THREADS,
				0x05);
		log_info("Processing last round of debugger requests.");
		process_debugger_requests(dbg, t);
	}
	Task::killall();
	log_info("Exiting.");
	exit(0);
}

/** Return true when replaying/debugging should cease after |t| exits. */
static bool is_last_interesting_task(Task* t)
{
	return (0 == debugged_tgid && Task::count() == 1)
		|| (t->tgid() == debugged_tgid
		    && t->task_group()->task_set().size() == 1);
}

static void replay_one_trace_frame(struct dbg_context* dbg, Task* t)
{
	struct dbg_request req;
	struct rep_trace_step step;
	Event ev(t->trace.ev);
	int stop_sig = 0;

	debug("[line %d] %d: replaying %s; state %s",
	      get_global_time(), t->rec_tid,
	      strevent(t->trace.ev), statename(t->trace.ev.state));
	if (t->syscallbuf_hdr) {
		debug("    (syscllbufsz:%u, abrtcmt:%u)",
		      t->syscallbuf_hdr->num_rec_bytes,
		      t->syscallbuf_hdr->abort_commit);
	}

	/* Advance the trace until we've exec()'d the tracee before
	 * processing debugger requests.  Otherwise the debugger host
	 * will be confused about the initial executable image,
	 * rr's. */
	if (validate) {
		req = process_debugger_requests(dbg, t);
		assert(dbg_is_resume_request(&req));
	}

	if (t->child_sig != 0) {
		assert(EV_SIGNAL == ev.type() && t->child_sig == ev.Signal().no);
		t->child_sig = 0;
	}

	/* Ask the trace-interpretation code what to do next in order
	 * to retire the current frame. */
	memset(&step, 0, sizeof(step));

	switch (ev.type()) {
	case EV_UNSTABLE_EXIT:
		t->unstable = 1;
		/* fall through */
	case EV_EXIT: {
		if (is_last_interesting_task(t)) {
			debug("last interesting task in %d is %d (%d)",
			      debugged_tgid, t->rec_tid, t->tid);
			last_task = t;
			return;
		}

		/* If the task was killed by a terminating signal,
		 * then it may have ended abruptly in a syscall or at
		 * some other random execution point.  That's bad for
		 * replay, because we detach from the task after we
		 * replay its "exit".  Since we emulate signal
		 * delivery, the task may happily carry on with
		 * (non-emulated!) execution after we detach.  That
		 * execution might include things like |rm -rf ~|.
		 *
		 * To ensure that the task really dies, we send it a
		 * terminating signal here.  One would like to use
		 * SIGKILL, but for not-understood reasons that causes
		 * shutdown hangs when joining the exited tracee.
		 * Other terminating signals have not been observed to
		 * hang, so that's what's used here.. */
		syscall(SYS_tkill, t->tid, SIGABRT);
		// TODO dissociate address space from file table
		bool file_table_dying = (1 == t->vm()->task_set().size());
		delete t;
		/* Early-return because |t| is gone now. */
		if (file_table_dying) {
			EmuFs::gc();
		}
		return;
	}
	case EV_DESCHED:
		step.action = TSTEP_DESCHED;
		step.desched.type =
			ARMING_DESCHED_EVENT == ev.Desched().state ?
			DESCHED_ARM : DESCHED_DISARM;
		step.desched.state = DESCHED_ENTER;
		break;
	case EV_SYSCALLBUF_ABORT_COMMIT:
		t->syscallbuf_hdr->abort_commit = 1;
		step.action = TSTEP_RETIRE;
		break;
	case EV_SYSCALLBUF_FLUSH:
		step.action = TSTEP_FLUSH_SYSCALLBUF;
		step.flush.need_buffer_restore = 1;
		step.flush.num_rec_bytes_remaining = 0;
		break;
	case EV_SYSCALLBUF_RESET:
		t->syscallbuf_hdr->num_rec_bytes = 0;
		step.action = TSTEP_RETIRE;
		break;
	case EV_SCHED:
		step.action = TSTEP_PROGRAM_ASYNC_SIGNAL_INTERRUPT;
		step.target.rcb = t->trace.rbc;
		step.target.regs = &t->trace.recorded_regs;
		step.target.signo = 0;
		break;
	case EV_SEGV_RDTSC:
		step.action = TSTEP_DETERMINISTIC_SIGNAL;
		step.signo = SIGSEGV;
		break;
	case EV_INTERRUPTED_SYSCALL_NOT_RESTARTED:
		debug("  popping interrupted but not restarted %s",
		      syscallname(t->ev().syscall.no));
		t->pop_syscall_interruption();
		step.action = TSTEP_RETIRE;
		break;
	case EV_EXIT_SIGHANDLER:
		debug("<-- sigreturn from %s", signalname(t->ev().syscall.no));
		t->pop_signal_handler();
		step.action = TSTEP_RETIRE;
		break;
	case EV_SIGNAL:
		step.signo = ev.Signal().no;
		stop_sig = step.signo;
		step.action = (ev.Signal().deterministic ?
			       TSTEP_DETERMINISTIC_SIGNAL :
			       TSTEP_PROGRAM_ASYNC_SIGNAL_INTERRUPT);
		if (TSTEP_PROGRAM_ASYNC_SIGNAL_INTERRUPT == step.action) {
			step.target.signo = step.signo;
			stop_sig = step.target.signo;
			step.target.rcb = t->trace.rbc;
			step.target.regs = &t->trace.recorded_regs;
		}
		break;
	case EV_SYSCALL:
		rep_process_syscall(t, &step);
		break;
	default:
		fatal("Unexpected event %s", ev.str().c_str());
	}

	/* See the comment below about *not* resetting the hpc for
	 * buffer flushes.  Here, we're processing the *other* event,
	 * just after the buffer flush, where the rcb matters.  To
	 * simplify the advance-to-target code that follows (namely,
	 * making debugger interrupts simpler), pretend like the
	 * execution in the BUFFER_FLUSH didn't happen by resetting
	 * the rbc and compensating down the target rcb. */
	if (TSTEP_PROGRAM_ASYNC_SIGNAL_INTERRUPT == step.action) {
		int64_t rcb_now = read_rbc(t->hpc);

		assert(step.target.rcb >= rcb_now);

		step.target.rcb -= rcb_now;
		reset_hpc(t, 0);
	}

	/* Advance until |step| has been fulfilled. */
	while (try_one_trace_step(t, &step, &req)) {
		if (EV_TRACE_TERMINATION == cur_trace_frame.ev.type) {
			// An irregular trace step had to read the
			// next trace frame, and that frame was an
			// early-termination marker.  Otherwise we
			// would have seen the marker at
			// |schedule_task()|.
			return handle_interrupted_trace(dbg, t);
		}

		/* Currently we only understand software breakpoints
		 * and successful stepi's. */
		assert(SIGTRAP == t->child_sig && "Unknown trap");

		if (TRAP_BKPT_USER ==
		    t->vm()->get_breakpoint_type_at_ip(t->ip())) {
			debug("  %d(rec:%d): hit debugger breakpoint at ip %p",
			      t->tid, t->rec_tid, t->ip());
			/* SW breakpoint: $ip is just past the
			 * breakpoint instruction.  Move $ip back
			 * right before it. */
			t->move_ip_before_breakpoint();
		} else {
			debug("  finished debugger stepi");
			/* Successful stepi.  Nothing else to do. */
			assert(DREQ_STEP == req.type
			       && req.target == get_threadid(t));
		}
		/* Don't restart with SIGTRAP anywhere. */
		t->child_sig = 0;

		/* Notify the debugger and process any new requests
		 * that might have triggered before resuming. */
		dbg_notify_stop(dbg, get_threadid(t),	0x05/*gdb mandate*/);
		req = process_debugger_requests(dbg, t);
		assert(dbg_is_resume_request(&req));
	}

	if (DREQ_DETACH == req.type) {
		log_info("(debugger detached from us, rr exiting)");
		dbg_reply_detach(dbg);
		dbg_destroy_context(&dbg);
		// Don't orphan tracees: their VMs are inconsistent
		// because we've been using emulated tracing, so they
		// can't resume normal execution.  And we wouldn't
		// want them continuing to execute even if they could.
		Task::killall();
		exit(0);
	}

	if (TSTEP_ENTER_SYSCALL == step.action) {
		rep_after_enter_syscall(t, step.syscall.no);
	}
	if (STATE_SYSCALL_EXIT == t->trace.ev.state
	    && rr_flags()->check_cached_mmaps) {
		t->vm()->verify(t);
	}

	if (dbg && stop_sig) {
		dbg_notify_stop(dbg, get_threadid(t), stop_sig);
	}

	/* We flush the syscallbuf in response to detecting *other*
	 * events, like signal delivery.  Flushing the syscallbuf is a
	 * sort of side-effect of reaching the other event.  But once
	 * we've flushed the syscallbuf during replay, we still must
	 * reach the execution point of the *other* event.  For async
	 * signals, that requires us to have an "intact" rbc, with the
	 * same value as it was when the last buffered syscall was
	 * retired during replay.  We'll be continuing from that rcb
	 * to reach the rcb we recorded at signal delivery.  So don't
	 * reset the counter for buffer flushes.  (It doesn't matter
	 * for non-async-signal types, which are deterministic.) */
	switch (ev.type()) {
	case EV_SYSCALLBUF_ABORT_COMMIT:
	case EV_SYSCALLBUF_FLUSH:
	case EV_SYSCALLBUF_RESET:
		break;
	case EV_DESCHED:
		/* ARM_DESCHED events are like the SYSCALLBUF_* events
		 * in that they weren't actually observed during
		 * recording, only inferred, so we don't have any
		 * reference to assert against during replay. */
		if (ARMING_DESCHED_EVENT == ev.Desched().state) {
			break;
		}
		// fall through
	default:
		/* We won't necessarily reach the same rcb when
		 * replaying an async signal, due to debugger
		 * interrupts and other implementation details.  This
		 * is checked in |advance_to()| anyway. */
		if (TSTEP_PROGRAM_ASYNC_SIGNAL_INTERRUPT != step.action) {
			assert_at_recorded_rcb(t, ev);
		}
		reset_hpc(t, 0);
	}
	debug_memory(t);
}

/**
 * Return the previous debugger |dbg| if there was one.  Otherwise if
 * the trace has reached the event at which the user wanted a debugger
 * started, then create one and return it.  Otherwise return nullptr.
 */
struct dbg_context* maybe_create_debugger(Task* t, struct dbg_context* dbg)
{
	if (dbg) {
		return dbg;
	}
	// Don't launch the debugger for the initial rr fork child.
	// No one ever wants that to happen.
	if (!validate) {
		return nullptr;
	}
	uint32_t goto_event = rr_flags()->goto_event;
	pid_t target_process = rr_flags()->target_process;
	bool require_exec = (CREATED_EXEC == rr_flags()->process_created_how);
	uint32_t event_now = get_global_time();
	if (event_now < goto_event
	    // NB: we'll happily attach to whichever task within the
	    // group happens to be scheduled here.  We don't take
	    // "attach to process" to mean "attach to thread-group
	    // leader".
	    || (target_process && t->tgid() != target_process)
	    || (require_exec && !t->vm()->execed())) {
		return nullptr;
	}
	assert(!dbg);

	if (goto_event || target_process) {
		fprintf(stderr,	"\a\n"
			"--------------------------------------------------\n"
			" ---> Reached target process %d at event %u.\n"
			"--------------------------------------------------\n",
			target_process, event_now);
	}

	unsigned short port = (rr_flags()->dbgport > 0) ?
			      rr_flags()->dbgport : getpid();
	// Don't probe if the user specified a port.  Explicitly
	// selecting a port is usually done by scripts, which would
	// presumably break if a different port were to be selected by
	// rr (otherwise why would they specify a port in the first
	// place).  So fail with a clearer error message.
	int probe = (rr_flags()->dbgport > 0) ? DONT_PROBE : PROBE_PORT;
	const char* exe = rr_flags()->dont_launch_debugger ? nullptr :
			  t->vm()->exe_image().c_str();
	return dbg_await_client_connection("127.0.0.1", port, probe,
					   debugged_tgid = t->tgid(), exe,
					   parent, debugger_params_pipe[1]);
}

/**
 * Set the blocked-ness of |sig| to |blockedness|.
 */
static void set_sig_blockedness(int sig, int blockedness)
{
	sigset_t sset;
	sigemptyset(&sset);
	sigaddset(&sset, sig);
	if (sigprocmask(blockedness, &sset, NULL)) {
		fatal("Didn't change sigmask.");
	}
}

static void replay_trace_frames(void)
{
	struct dbg_context* dbg = nullptr;
	while (!last_task) {
		Task* intr_t;
		Task* t = schedule_task(&intr_t);
		if (!t) {
			return handle_interrupted_trace(dbg, intr_t);
		}
		dbg = maybe_create_debugger(t, dbg);
		replay_one_trace_frame(dbg, t);
	}
	log_info("Replayer successfully finished.");
	fflush(stdout);

	if (dbg) {
		// TODO return real exit code, if it's useful.
		dbg_notify_exit_code(dbg, 0);
		process_debugger_requests(dbg, last_task);
		fatal("Received continue request after end-of-trace.");
	}
}

static void serve_replay(int argc, char* argv[], char** envp)
{
	string exe_image;
	CharpVector arg_v;
	CharpVector env_p;

	load_recorded_env(argv[0], &argc, &exe_image, &arg_v, &env_p);

	init_libpfm();

	rep_setup_trace_dir(argv[0]);
	open_trace_files();
	rep_init_trace_files();

	// Because we execvpe() the tracee, we must ensure that $PATH
	// is the same as in recording so that libc searches paths in
	// the same order.  So copy that over now.
	//
	// And because we use execvpe(), the exec'd tracee will start
	// with a fresh environment guaranteed to be the same as in
	// replay, so we don't have to worry about any mutation here
	// affecting post-exec execution.
	for (CharpVector::const_iterator it = env_p.begin();
	     *it && it != env_p.end(); ++it) {
		if (!strncmp(*it, "PATH=", sizeof("PATH=") - 1)) {
			// NB: intentionally leaking this string.
			putenv(strdup(*it));
		}
	}

	Task::create(exe_image, arg_v, env_p, get_recorded_main_thread());
	replay_trace_frames();

	close_libpfm();
	close_trace_files();

	debug("debugger server exiting ...");
	exit(0);
}

static bool launch_debugger;
static void handle_signal(int sig)
{
	switch (sig) {
	case SIGINT:
		// Translate the SIGINT into SIGTERM for the debugger
		// server, because it's blocking SIGINT.  We don't use
		// SIGINT for anything, so all it's meant to do is
		// kill us, and SIGTERM works just as well for that.
		if (child > 0) {
			kill(child, SIGTERM);
		}
		break;
	case DBG_SOCKET_READY_SIG:
		launch_debugger = true;
		break;
	default:
		fatal("Unhandled signal %s", signalname(sig));
	}
}

void replay(int argc, char* argv[], char** envp)
{
	// If we're not going to autolaunch the debugger, don't go
	// through the rigamarole to set that up.  All it does is
	// complicate the process tree and confuse users.
	if (rr_flags()->dont_launch_debugger) {
		return serve_replay(argc, argv, envp);
	}

	parent = getpid();

	struct sigaction sa;
	memset(&sa, 0, sizeof(sa));
	sa.sa_handler = handle_signal;
	if (sigaction(SIGINT, &sa, nullptr)
	    || sigaction(DBG_SOCKET_READY_SIG, &sa, nullptr)) {
		fatal("Couldn't set sigaction for SIGINT.");
	}

	if (pipe2(debugger_params_pipe, O_CLOEXEC)) {
		fatal("Couldn't open debugger params pipe.");
	}
	if (0 == (child = fork())) {
		// The parent process (gdb) must be able to receive
		// SIGINT's to interrupt non-stopped tracees.  But the
		// debugger server isn't set up to handle SIGINT.  So
		// block it.
		set_sig_blockedness(SIGINT, SIG_BLOCK);
		serve_replay(argc, argv, envp);
		fatal("Not reached");
	}
	debug("%d: forked debugger server %d", parent, child);

	while (true) {
		int ret, status;
		// NB: we're forced to use waitpid() because linux
		// doesn't provide a way to poll waitpid.  If it did,
		// we would poll both that and the debugger-params
		// pipe and we wouldn't need the signal hack.
		// Alternatively, passing a sigmask to waitpid() in
		// the style of ppoll() would solve the problem.
		//
		// But as it is, there's a potential race condition
		// betweed the params-ready signal being delivered to
		// us and us entering waitpid().  We rely on waitpid()
		// erroring out on signal delivery to wake us up.  If
		// the signal arrives between the if-condition here
		// and entry to waitpid(), then we'll deadlock with
		// the debug server.  This sched_yield() is a
		// poor-man's "please run the next few instructions
		// atomically".
		//
		// FIXME: add pwaitpid() to linux?
		sched_yield();
		// BEGIN CRITICAL SECTION
		if (launch_debugger) {
			dbg_launch_debugger(debugger_params_pipe[0]);
			fatal("Not reached");
		}
		ret = waitpid(child, &status, 0);
		// END CRITICAL SECTION

		int err = errno;
		debug("%d: waitpid(%d) returned %s(%d); status:%#x",
		      getpid(), child, strerror(err), err, status);
		if (child != ret) {
			if (EINTR == err) {
				continue;
			}
			fatal("%d: waitpid(%d) failed", getpid(), child);
		}
		if (WIFEXITED(status) || WIFSIGNALED(status)) {
			log_info("Debugger server died.  Exiting.");
			exit(WIFEXITED(status) ? WEXITSTATUS(status) : 1);
		}
	}
}

void replay_flags_to_args(const struct flags& f,
			  vector<string>* common_args,
			  vector<string>* replay_args)
{
	// Common args.
	if (f.force_enable_debugger) {
		common_args->push_back("-f");
	}
	if (f.check_cached_mmaps) {
		common_args->push_back("-k");
	}
	if (f.mark_stdio) {
		common_args->push_back("-m");
	}
	if (f.cpu_unbound) {
		common_args->push_back("-u");
	}
	if (f.verbose) {
		common_args->push_back("-v");
	}
	// Replay args.
	if (!f.redirect) {
		replay_args->push_back("-q");
	}
	if (f.target_process) {
		switch (f.process_created_how) {
		case CREATED_FORK:
			replay_args->push_back("-f");
			break;
		case CREATED_EXEC:
			replay_args->push_back("-p");
			break;
		default:
			fatal("Not reached");
		}
		stringstream ss;
		ss << f.target_process;
		replay_args->push_back(ss.str());
	}
	if (f.goto_event) {
		replay_args->push_back("-g");
		stringstream ss;
		ss << f.goto_event;
		replay_args->push_back(ss.str());
	}
	if (f.dbgport) {
		replay_args->push_back("-s");
		stringstream ss;
		ss << f.dbgport;
		replay_args->push_back(ss.str());
	}
}

static void restart_replay(struct dbg_context* dbg, struct dbg_request req)
{
	// Update the replayer launch flags with values the debugger
	// has most recently seen from the user.
	struct flags f = *rr_flags();
	f.goto_event = req.restart.event > 0 ?
			req.restart.event : f.goto_event;
	if (f.goto_event && !f.target_process) {
		// We allow the user to change the target event.  It's
		// perfectly acceptable for them to target an event
		// before |debugged_tgid| was launched; rr should Do
		// The Right Thing in that case and attach the
		// debugger at the first event after |f.goto_event| at
		// which |debugged_tgid| exists.
		f.target_process = debugged_tgid;
		f.process_created_how =
			Task::find(debugged_tgid)->vm()->execed() ?
			CREATED_EXEC : CREATED_FORK;
	}
	f.dbgport = req.restart.port;

	// We're going to restart by exec()ing on top of ourselves, in
	// order to keep the process tree intact.  So our tracees
	// won't get the kill signal they usually would if we
	// early-exited.
	//
	// NB: it's critical that *all* allocated rr resources die
	// when our task does, and exec()ing is an easy way to test
	// that we're not potentially leaking anything.
	Task::killall();

	// FIXME: we have to handle crashes and abnormal termination,
	// so these shouldn't be mandatory.  I don't know if they are
	// or not.
	close_libpfm();
	close_trace_files();

	dbg_prepare_restore_after_exec_restart(dbg);

	char exe[PATH_MAX];
	ssize_t nbytes = readlink("/proc/self/exe", exe, sizeof(exe));
	if (-1 == nbytes) {
		fatal("Failed to readlink('/proc/self/exe')");
	}
	exe[nbytes] = '\0';

	vector<string> common_args;
	vector<string> replay_args;
	replay_flags_to_args(f, &common_args, &replay_args);

	CharpVector argv;
	argv.push_back(exe);
	for (size_t j = 0; j < common_args.size(); ++j) {
		argv.push_back(strdup(common_args[j].c_str()));
	}
	argv.push_back(strdup("replay"));
	for (size_t j = 0; j < replay_args.size(); ++j) {
		argv.push_back(strdup(replay_args[j].c_str()));
	}
	argv.push_back(strdup(get_trace_path()));

	argv.push_back(nullptr);

	execv(exe, argv.data());
	fatal("Failed to exec %s", exe);
}

void emergency_debug(Task* t)
{
	if (probably_not_interactive()
	    && !rr_flags()->force_enable_debugger) {
		errno = 0;
		fatal("(session doesn't look interactive, aborting emergency debugging)");
	}

	start_debug_server(t);
	fatal("Can't resume execution from invalid state");
}

void start_debug_server(Task* t)
{
	flush_trace_files();

	// See the comment in |guard_overshoot()| explaining why we do
	// this.  Unlike in that context though, we don't know if |t|
	// overshot an internal breakpoint.  If it did, cover that
	// breakpoint up.
	t->vm()->destroy_all_breakpoints();

	// Don't launch a debugger on fatal errors; the user is most
	// likely already in a debugger, and wouldn't be able to
	// control another session.
	struct dbg_context* dbg =
		dbg_await_client_connection("127.0.0.1", t->tid, PROBE_PORT,
					    t->tgid());

	process_debugger_requests(dbg, t);

	dbg_destroy_context(&dbg);
}
