/**************************************************************************

   Fotoxx      edit photos and manage collections

   Copyright 2007 2008 2009 2010 2011  Michael Cornelison
   Source URL: http://kornelix.squarespace.com/fotoxx
   Contact: kornelix2@googlemail.com
   
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program. If not, see http://www.gnu.org/licenses/.

***************************************************************************/

#define EX extern                                                          //  enable extern declarations
#include "fotoxx.h"


/**************************************************************************

   Image edit functions

***************************************************************************/

//  adjust white balance

double   whitebal_red, whitebal_green, whitebal_blue;

void m_whitebal(GtkWidget *, cchar *)
{
   int    whitebal_dialog_event(zdialog* zd, cchar *event);
   void   whitebal_mousefunc();
   void * whitebal_thread(void *);

   cchar  *wbtitle = ZTX("Adjust White Balance");
   cchar  *wbhelp = ZTX("Click white or gray image location");

   zfuncs::F1_help_topic = "white_balance";                                //  v.10.8
   
   if (! edit_setup("whitebal",1,2)) return;                               //  setup edit: preview

   zdedit = zdialog_new(wbtitle,mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"label","labwbh","hb1",wbhelp,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labpix","hb2","pixel:");
   zdialog_add_widget(zdedit,"label","pixel","hb2","0000 0000");
   zdialog_add_widget(zdedit,"label","labrgb","hb2","   RGB:");
   zdialog_add_widget(zdedit,"label","rgb","hb2","000 000 000");

   zdialog_run(zdedit,whitebal_dialog_event);                              //  run dialog - parallel

   whitebal_red = whitebal_green = whitebal_blue = 1.0;
   start_thread(whitebal_thread,0);                                        //  start working thread

   takeMouse(0,whitebal_mousefunc,dragcursor);                             //  connect mouse function    v.11.03
   return;
}


//  dialog event and completion callback function

int whitebal_dialog_event(zdialog *zd, cchar *event)                       //  dialog event function     v.10.2
{
   if (zd->zstat)
   {
      freeMouse();                                                         //  disconnect mouse function       v.10.12
      if (zd->zstat == 1) edit_done();                                     //  done
      else edit_cancel();                                                  //  cancel or destroy
      return 0;
   }
   
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3
   return 1;
}


void whitebal_mousefunc()                                                  //  mouse function
{
   int         px, py, dx, dy;
   double      red, green, blue, rgbmean;
   char        work[40];
   uint16      *ppix16;
   
   if (! LMclick) return;
   
   LMclick = 0;
   px = Mxclick;                                                           //  mouse click position
   py = Myclick;
   
   if (px < 2) px = 2;                                                     //  pull back from edge
   if (px > E3ww-3) px = E3ww-3;
   if (py < 2) py = 2;
   if (py > E3hh-3) py = E3hh-3;
   
   red = green = blue = 0;

   for (dy = -2; dy <= 2; dy++)                                            //  5x5 block around mouse position
   for (dx = -2; dx <= 2; dx++)
   {
      ppix16 = PXMpix(E1pxm16,px+dx,py+dy);                                //  input image
      red += ppix16[0];
      green += ppix16[1];
      blue += ppix16[2];
   }
   
   red = red / 25.0;                                                       //  mean RGB levels 
   green = green / 25.0;
   blue = blue / 25.0;
   rgbmean = (red + green + blue) / 3.0;

   whitebal_red = rgbmean / red;
   whitebal_green = rgbmean / green;
   whitebal_blue = rgbmean / blue;

   signal_thread();                                                        //  trigger image update

   snprintf(work,40,"%d %d",px,py);
   zdialog_stuff(zdedit,"pixel",work);

   snprintf(work,40,"%7.3f %7.3f %7.3f",red/256,green/256,blue/256);
   zdialog_stuff(zdedit,"rgb",work);
   
   return;
}


//  Update image based on neutral pixel that was clicked

void * whitebal_thread(void *)
{
   void * whitebal_wthread(void *arg);

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker threads
         start_wt(whitebal_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * whitebal_wthread(void *arg)                                         //  worker thread function
{
   int         index = *((int *) arg);
   int         px, py, ii, dist = 0;
   uint16      *pix1, *pix3;
   double      red1, green1, blue1;
   double      red3, green3, blue3;
   double      brmax, dold, dnew;

   for (py = index; py < E1hh; py += Nwt)
   for (px = 0; px < E1ww; px++)
   {
      if (Factivearea) {                                                   //  select area active
         ii = py * Fww + px;
         dist = sa_pixisin[ii];                                            //  distance from edge
         if (! dist) continue;                                             //  pixel outside area
      }

      pix1 = PXMpix(E1pxm16,px,py);                                        //  input pixel
      pix3 = PXMpix(E3pxm16,px,py);                                        //  output pixel
      
      red1 = pix1[0];
      green1 = pix1[1];
      blue1 = pix1[2];
      
      red3 = whitebal_red * red1;                                          //  change color ratios
      green3 = whitebal_green * green1;
      blue3 = whitebal_blue * blue1;

      if (Factivearea && dist < sa_blend) {                                //  select area is active,
         dnew = 1.0 * dist / sa_blend;                                     //    blend changes over sa_blend
         dold = 1.0 - dnew;
         red3 = dnew * red3 + dold * red1;
         green3 = dnew * green3 + dold * green1;
         blue3 = dnew * blue3 + dold * blue1;
      }
      
      brmax = red3;                                                        //  brmax = brightest color
      if (green3 > brmax) brmax = green3;
      if (blue3 > brmax) brmax = blue3;
      
      if (brmax > 65535) {                                                 //  if overflow, reduce
         brmax = 65535 / brmax;
         red3 = red3 * brmax;
         green3 = green3 * brmax;
         blue3 = blue3 * brmax;
      }

      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
   }
   
   exit_wt();
   return 0;                                                               //  not executed, stop gcc warning
}


/**************************************************************************/

//  make a black & white or color negative image

void m_negate(GtkWidget *, cchar *)                                        //  v.10.9
{
   int negate_dialog_event(zdialog *zd, cchar *event);

   zfuncs::F1_help_topic = "make_negative";

   if (! edit_setup("negate",0,0)) return;                                 //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Make Negative"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"radio","b&wpos","dialog","black/white positive");
   zdialog_add_widget(zdedit,"radio","b&wneg","dialog","black/white negative");
   zdialog_add_widget(zdedit,"radio","colpos","dialog","color positive");
   zdialog_add_widget(zdedit,"radio","colneg","dialog","color negative");

   zdialog_stuff(zdedit,"colpos",1);

   zdialog_resize(zdedit,200,0);
   zdialog_run(zdedit,negate_dialog_event);                                //  run dialog - parallel
   return;
}


//  dialog event and completion callback function

int negate_dialog_event(zdialog *zd, cchar *event)
{
   int         mode, px, py;
   int         red, green, blue;
   uint16      *pix1, *pix3;

   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      return 0;
   }
   
   if (strEqu(event,"b&wpos")) mode = 1;
   if (strEqu(event,"b&wneg")) mode = 2;
   if (strEqu(event,"colpos")) mode = 3;
   if (strEqu(event,"colneg")) mode = 4;
   
   edit_zapredo();                                                         //  delete redo copy

   for (py = 0; py < E3hh; py++)
   for (px = 0; px < E3ww; px++)
   {
      pix1 = PXMpix(E1pxm16,px,py);

      red = pix1[0];
      green = pix1[1];
      blue = pix1[2];
      
      if (mode == 1)                                                       //  black and white positive
         red = green = blue = (red + green + blue) / 3;
      
      else if (mode == 2)                                                  //  black and white negative
         red = green = blue = 65535 - (red + green + blue) / 3;
      
      if (mode == 3) {  /** do nothing  **/  }                             //  color positive
      
      if (mode == 4) {                                                     //  color negative
         red = 65535 - red;
         green = 65535 - green;
         blue = 65535 - blue; 
      }
      
      pix3 = PXMpix(E3pxm16,px,py);

      pix3[0] = red;
      pix3[1] = green;
      pix3[2] = blue;
   }
   
   Fmodified = 1;
   mwpaint2();

   return 0;
}


/**************************************************************************/

//  flatten brightness distribution

double   flatten_value = 0;                                                //  flatten value, 0 - 100%
double   flatten_brdist[65536];


void m_flatten(GtkWidget *, cchar *)
{
   int    flatten_dialog_event(zdialog* zd, cchar *event);
   void * flatten_thread(void *);

   cchar  *title = ZTX("Flatten Brightness Distribution");

   zfuncs::F1_help_topic = "flatten";                                      //  v.10.8
   
   if (! edit_setup("flatten",1,2)) return;                                //  setup edit: preview, select area OK

   zdedit = zdialog_new(title,mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=15");
   zdialog_add_widget(zdedit,"label","labfd","hb1",ZTX("Flatten"),"space=5");
   zdialog_add_widget(zdedit,"hscale","flatten","hb1","0|100|1|0","expand");

   zdialog_resize(zdedit,300,0);
   zdialog_run(zdedit,flatten_dialog_event);                               //  run dialog - parallel
   
   flatten_value = 0;
   start_thread(flatten_thread,0);                                         //  start working thread
   return;
}


//  dialog event and completion callback function

int flatten_dialog_event(zdialog *zd, cchar *event)                        //  flatten dialog event function
{
   if (zd->zstat)
   {
      if (zd->zstat == 1) edit_done();                                     //  done
      else edit_cancel();                                                  //  cancel or destroy
      return 0;
   }
   
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3

   if (strEqu(event,"flatten")) {
      zdialog_fetch(zd,"flatten",flatten_value);                           //  get slider value
      signal_thread();                                                     //  trigger update thread
   }
   
   return 1;
}


//  thread function - use multiple working threads

void * flatten_thread(void *)
{
   void  * flatten_wthread(void *arg);

   int         px, py, ii, npix;
   double      bright1;
   uint16      *pix1;

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      for (ii = 0; ii < 65536; ii++)                                       //  clear brightness distribution data
         flatten_brdist[ii] = 0;

      npix = 0;                                                            //  v.11.02 - instead of sa_Npixel

      if (Factivearea)                                                     //  process selected area
      {
         for (ii = 0; ii < Fww * Fhh; ii++)                                //  v.9.6
         {
            if (! sa_pixisin[ii]) continue;
            py = ii / Fww;
            px = ii - py * Fww;
            pix1 = PXMpix(E1pxm16,px,py);
            bright1 = pixbright(pix1);
            flatten_brdist[int(bright1)]++;
            npix++;
         }
         
         for (ii = 1; ii < 65536; ii++)                                    //  cumulative brightness distribution
            flatten_brdist[ii] += flatten_brdist[ii-1];                    //   0 ... npix

         for (ii = 0; ii < 65536; ii++)
            flatten_brdist[ii] = flatten_brdist[ii]                        //  multiplier per brightness level
                               / npix * 65536.0 / (ii + 1);
      }

      else                                                                 //  process whole image
      {
         for (py = 0; py < E1hh; py++)                                     //  compute brightness distribution
         for (px = 0; px < E1ww; px++)
         {
            pix1 = PXMpix(E1pxm16,px,py);
            bright1 = pixbright(pix1);
            flatten_brdist[int(bright1)]++;
         }
         
         for (ii = 1; ii < 65536; ii++)                                    //  cumulative brightness distribution
            flatten_brdist[ii] += flatten_brdist[ii-1];                    //   0 ... (ww1 * hh1)

         for (ii = 0; ii < 65536; ii++)
            flatten_brdist[ii] = flatten_brdist[ii]                        //  multiplier per brightness level 
                               / (E1ww * E1hh) * 65536.0 / (ii + 1);
      }
      
      for (ii = 0; ii < Nwt; ii++)                                         //  start worker threads
         start_wt(flatten_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * flatten_wthread(void *arg)                                          //  worker thread function
{
   int         index = *((int *) (arg));
   int         px, py, ii, dist = 0;
   uint16      *pix1, *pix3;
   double      fold, fnew, dold, dnew, cmax;
   double      red1, green1, blue1, red3, green3, blue3;
   double      bright1, bright2;

   for (py = index; py < E1hh; py += Nwt)                                  //  flatten brightness distribution
   for (px = 0; px < E1ww; px++)
   {
      if (Factivearea) {                                                   //  select area active
         ii = py * Fww + px;
         dist = sa_pixisin[ii];                                            //  distance from edge
         if (! dist) continue;                                             //  outside pixel
      }

      pix1 = PXMpix(E1pxm16,px,py);                                        //  input pixel
      pix3 = PXMpix(E3pxm16,px,py);                                        //  output pixel
         
      fnew = 0.01 * flatten_value;                                         //  0.0 - 1.0  how much to flatten
      fold = 1.0 - fnew;                                                   //  1.0 - 0.0  how much to retain

      red1 = pix1[0];
      green1 = pix1[1];
      blue1 = pix1[2];

      bright1 = pixbright(pix1);                                           //  input brightness
      bright2 = flatten_brdist[int(bright1)];                              //  output brightness adjustment

      red3 = bright2 * red1;                                               //  flattened brightness
      green3 = bright2 * green1;
      blue3 = bright2 * blue1;

      red3 = fnew * red3 + fold * red1;                                    //  blend new and old brightness
      green3 = fnew * green3 + fold * green1;
      blue3 = fnew * blue3 + fold * blue1;

      if (Factivearea && dist < sa_blend) {                                //  select area is active,
         dnew = 1.0 * dist / sa_blend;                                     //    blend changes over sa_blend
         dold = 1.0 - dnew;
         red3 = dnew * red3 + dold * red1;
         green3 = dnew * green3 + dold * green1;
         blue3 = dnew * blue3 + dold * blue1;
      }

      cmax = red3;                                                         //  stop overflow, keep color balance
      if (green3 > cmax) cmax = green3;
      if (blue3 > cmax) cmax = blue3;
      if (cmax > 65535) {
         cmax = 65535 / cmax;
         red3 = red3 * cmax;
         green3 = green3 * cmax;
         blue3 = blue3 * cmax;
      }

      pix3[0] = int(red3 + 0.5);
      pix3[1] = int(green3 + 0.5);
      pix3[2] = int(blue3 + 0.5);
   }

   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


/**************************************************************************/

//  brightness / color / contrast adjustment

void  tune_curve_update(int spc);                                          //  curve update callback function

int      tune_spc;                                                         //  current spline curve 1-5
int      tune_spc_moved[6];                                                //  tracks which curves changed


void m_tune(GtkWidget *, cchar *)                                          //  menu function
{
   int   tune_dialog_event(zdialog *zd, cchar *event);
   void  *tune_thread(void *);

   cchar    *title = ZTX("Adjust Brightness and Color");
   zfuncs::F1_help_topic = "tune";                                         //  v.10.8
   
   if (! edit_setup("britecolor",1,2)) return;                             //  setup edit: preview
   
/***
       --------------------------------------------
      |                                            |
      |                                            |
      |           curve drawing area               |
      |                                            |
      |                                            |
       --------------------------------------------
       darker areas                   lighter areas

      [+++] [---] [+ -] [- +] [+-+] [-+-]
      Curve File: [ Open ] [ Save ]
      (o) brightness         [reset 1]  [reset all]
      (o) color saturation   [histogram]
      (o) red
      (o) green
      (o) blue
                                 [done] [cancel]
***/

   zdedit = zdialog_new(title,mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"frame","fr1","dialog",0,"expand");
   zdialog_add_widget(zdedit,"hbox","hba","dialog");
   zdialog_add_widget(zdedit,"label","labda","hba",Bdarker,"space=5");
   zdialog_add_widget(zdedit,"label","space","hba",0,"expand");
   zdialog_add_widget(zdedit,"label","labba","hba",Blighter,"space=5");

   zdialog_add_widget(zdedit,"hbox","hbb","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"button","b +++","hbb","+++");
   zdialog_add_widget(zdedit,"button","b ---","hbb","‒ ‒ ‒");
   zdialog_add_widget(zdedit,"button","b +-", "hbb"," + ‒ ");
   zdialog_add_widget(zdedit,"button","b -+", "hbb"," ‒ + ");
   zdialog_add_widget(zdedit,"button","b +-+","hbb","+ ‒ +");
   zdialog_add_widget(zdedit,"button","b -+-","hbb","‒ + ‒");

   zdialog_add_widget(zdedit,"hbox","hbcf","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labcf","hbcf",ZTX("Curve File:"),"space=5");
   zdialog_add_widget(zdedit,"button","load","hbcf",Bopen,"space=5");
   zdialog_add_widget(zdedit,"button","save","hbcf",Bsave,"space=5");

   zdialog_add_widget(zdedit,"hbox","hb2","dialog");
   zdialog_add_widget(zdedit,"vbox","vb21","hb2");
   zdialog_add_widget(zdedit,"vbox","vb22","hb2");
   zdialog_add_widget(zdedit,"radio","radbri","vb21",Bbrightness);
   zdialog_add_widget(zdedit,"radio","radsat","vb21",ZTX("color saturation"));
   zdialog_add_widget(zdedit,"radio","radR","vb21",Bred);
   zdialog_add_widget(zdedit,"radio","radG","vb21",Bgreen);
   zdialog_add_widget(zdedit,"radio","radB","vb21",Bblue);
   zdialog_add_widget(zdedit,"hbox","hbrs","vb22",0,"space=5");
   zdialog_add_widget(zdedit,"label","space","hbrs",0,"space=10");
   zdialog_add_widget(zdedit,"button","reset1","hbrs",ZTX(" reset 1 "));
   zdialog_add_widget(zdedit,"button","resetA","hbrs",ZTX("reset all"));
   zdialog_add_widget(zdedit,"hbox","hbhist","vb22",0,"space=6");
   zdialog_add_widget(zdedit,"label","space","hbhist",0,"space=10");
   zdialog_add_widget(zdedit,"button","histo","hbhist",ZTX("histogram"));
   
   GtkWidget *frame = zdialog_widget(zdedit,"fr1");                        //  setup for curve editing
   spcdat *sd = spccurve_init(frame,tune_curve_update);                    //  v.11.01
   Espcdat = sd;

   for (int spc = 0; spc < 6; spc++)                                       //  setup 6 spline curves
   {                                                                       //  no. 0 is active curve
      sd->vert[spc] = 0;                                                   //  all curves are horizontal
      sd->nap[spc] = 3;                                                    //  curves 1-6 are copied to 0 when active
      sd->apx[spc][0] = 0.01;
      sd->apy[spc][0] = 0.5;
      sd->apx[spc][1] = 0.5;
      sd->apy[spc][1] = 0.5;
      sd->apx[spc][2] = 0.98;
      sd->apy[spc][2] = 0.5;
      spccurve_generate(sd,spc);
      tune_spc_moved[spc] = 0;
   }

   sd->Nspc = 1;                                                           //  only one at a time is active
   tune_spc = 1;                                                           //  default curve = brightness

   zdialog_stuff(zdedit,"radbri",1);                                       //  stuff default selection
   
   zdialog_resize(zdedit,0,420);
   zdialog_run(zdedit,tune_dialog_event);                                  //  run dialog - parallel
   start_thread(tune_thread,0);                                            //  start working thread
   return;
}


//  dialog event and completion callback function

int tune_dialog_event(zdialog *zd, cchar *event)
{
   int      Fupdate = 0;
   int      ii, jj, nn;
   double   px, py;
   spcdat   *sd = Espcdat;
   spcdat   sdtemp;
   
   if (zd->zstat)
   {
      if (zd->zstat == 1) edit_done();                                     //  done
      else edit_cancel();                                                  //  cancel or destroy
      return 0;
   }
   
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3
   if (strEqu(event,"histo")) m_histogram(0,0);                            //  popup brightness histogram

   if (strnEqu(event,"rad",3)) {                                           //  new choice of curve
      ii = strcmpv(event,"radbri","radsat","radR","radG","radB",null);
      tune_spc = ii;
      sd->nap[0] = sd->nap[ii];                                            //  copy active curve to curve 0
      for (jj = 0; jj < sd->nap[0]; jj++) {
         sd->apx[0][jj] = sd->apx[ii][jj];
         sd->apy[0][jj] = sd->apy[ii][jj];
      }
      Fupdate++;
   }
   
   ii = tune_spc;                                                          //  current active curve

   if (strnEqu(event,"b ",2)) {                                            //  button to move entire curve
      for (jj = 0; jj < sd->nap[ii]; jj++) {
         px = sd->apx[0][jj];
         py = sd->apy[0][jj];
         if (strEqu(event,"b +++")) py += 0.1;
         if (strEqu(event,"b ---")) py -= 0.1;
         if (strEqu(event,"b +-"))  py += 0.1 - 0.2 * px;
         if (strEqu(event,"b -+"))  py -= 0.1 - 0.2 * px;
         if (strEqu(event,"b +-+")) py -= 0.05 - 0.2 * fabs(px-0.5);
         if (strEqu(event,"b -+-")) py += 0.05 - 0.2 * fabs(px-0.5);
         if (py > 1) py = 1;
         if (py < 0) py = 0;
         sd->apy[0][jj] = py;
      }
      tune_spc_moved[ii] = 1;
      Fupdate++;
   }
   
   if (strEqu(event,"reset1")) {                                           //  reset current curve
      sd->nap[0] = 3;
      sd->apx[0][0] = 0.01;                                                //  3 anchor points, flatline
      sd->apy[0][0] = 0.5;
      sd->apx[0][1] = 0.5;
      sd->apy[0][1] = 0.5;
      sd->apx[0][2] = 0.99;
      sd->apy[0][2] = 0.5;
      Fupdate++;
      tune_spc_moved[ii] = 0;
   }
   
   if (strEqu(event,"resetA")) 
   {
      for (jj = 0; jj < 6; jj++) {                                         //  reset all curves
         sd->nap[jj] = 3;
         sd->apx[jj][0] = 0.01;
         sd->apy[jj][0] = 0.5;
         sd->apx[jj][1] = 0.5;
         sd->apy[jj][1] = 0.5;
         sd->apx[jj][2] = 0.99;
         sd->apy[jj][2] = 0.5;
         spccurve_generate(sd,jj);                                         //  regenerate all
         tune_spc_moved[jj] = 0;
      }
      Fupdate++;
   }

   if (strEqu(event,"load"))                                               //  load 5 saved curves    v.11.02
   {
      sdtemp.Nspc = 5;
      sdtemp.drawarea = 0;
      nn = spccurve_load(&sdtemp);
      if (nn != 1) return 0;

      for (ii = 0; ii < 5; ii++) {
         for (jj = 0; jj < sdtemp.nap[ii]; jj++) {
            sd->apx[ii+1][jj] = sdtemp.apx[ii][jj];
            sd->apy[ii+1][jj] = sdtemp.apy[ii][jj];
         }
         spccurve_generate(sd,ii+1);
         tune_spc_moved[ii+1] = 1;
      }

      ii = tune_spc;
      sd->nap[0] = sd->nap[ii];                                            //  copy active curve to curve 0
      for (jj = 0; jj < sd->nap[0]; jj++) {
         sd->apx[0][jj] = sd->apx[ii][jj];
         sd->apy[0][jj] = sd->apy[ii][jj];
      }
      Fupdate++;
   }

   if (strEqu(event,"save"))                                               //  save 5 curves to file   v.11.02
   {
      sdtemp.Nspc = 5;
      for (ii = 0; ii < 5; ii++) {
         sdtemp.nap[ii] = sd->nap[ii+1];
         for (jj = 0; jj < sd->nap[ii+1]; jj++) {
            sdtemp.apx[ii][jj] = sd->apx[ii+1][jj];
            sdtemp.apy[ii][jj] = sd->apy[ii+1][jj];
         }
      }
      spccurve_save(&sdtemp);
   }

   if (Fupdate)                                                            //  curve has changed
   {
      spccurve_generate(sd,0);                                             //  regenerate curve 0
      
      ii = tune_spc;                                                       //  active curve
      sd->nap[ii] = sd->nap[0];                                            //  copy curve 0 to active curve
      for (jj = 0; jj < sd->nap[0]; jj++) {
         sd->apx[ii][jj] = sd->apx[0][jj];
         sd->apy[ii][jj] = sd->apy[0][jj];
      }
      for (jj = 0; jj < 1000; jj++)
         sd->yval[ii][jj] = sd->yval[0][jj];

      spccurve_draw(0,0,sd);                                               //  draw curve
      signal_thread();                                                     //  trigger image update
   }
   
   return 1;
}


//  this function is called when curve 0 is edited using mouse

void  tune_curve_update(int)
{
   int      ii = tune_spc, jj;
   spcdat   *sd = Espcdat;

   sd->nap[ii] = sd->nap[0];                                               //  copy curve 0 to current curve
   for (jj = 0; jj < sd->nap[0]; jj++) {
      sd->apx[ii][jj] = sd->apx[0][jj];
      sd->apy[ii][jj] = sd->apy[0][jj];
   }
   for (jj = 0; jj < 1000; jj++)
      sd->yval[ii][jj] = sd->yval[0][jj];

   tune_spc_moved[ii] = 1;
   
   signal_thread();                                                        //  trigger image update
   return;
}


//  Update image based on latest settings of all dialog controls.

void * tune_thread(void *)
{
   void * tune_wthread(void *arg);
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker threads
         start_wt(tune_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * tune_wthread(void *arg)                                             //  worker thread function
{
   void  tune1pix(int px, int py);

   int      px, py;
   int      index = *((int *) arg);

   for (py = index; py < E1hh; py += Nwt)
   for (px = 0; px < E1ww; px++)
      tune1pix(px,py);

   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


void tune1pix(int px, int py)                                              //  process one pixel
{
   int         ii, dist = 0;
   uint16      *pix1, *pix3;
   double      red1, green1, blue1, red3, green3, blue3;
   double      xval, brmax, brmean, brout, brightness;
   spcdat      *sd = Espcdat;
   
   if (Factivearea) {                                                      //  select area active
      ii = py * Fww + px;
      dist = sa_pixisin[ii];                                               //  distance from edge
      if (! dist) return;                                                  //  pixel outside area
   }

   pix1 = PXMpix(E1pxm16,px,py);                                           //  input pixel
   pix3 = PXMpix(E3pxm16,px,py);                                           //  output pixel
   
   red1 = red3 = pix1[0];                                                  //  input and output RGB values
   green1 = green3 = pix1[1];
   blue1 = blue3 = pix1[2];

   brightness = 0.25 * red3 + 0.65 * green3 + 0.10 * blue3;                //  perceived brightness   v.10.9
   xval = brightness / 65536.0;                                            //  curve x-value, 0 to 0.999

/* ------------------------------------------------------------------------

      brightness adjustment

      brightness curve values:
           0 = dark
         0.5 = normal, unchanged
         1.0 = 200% brightness, clipped
*/

   if (tune_spc_moved[1])                                                  //  curve has been edited
   {
      brout = spccurve_yval(sd,1,xval);                                    //  brightness factor, 0 - 1
      
      brmax = red3;                                                        //  brmax = brightest color
      if (green3 > brmax) brmax = green3;
      if (blue3 > brmax) brmax = blue3;

      brout = brout / 0.5;                                                 //  0 - 2.0
      red3 = red3 * brout;                                                 //  apply to all colors
      green3 = green3 * brout;                                             //  remove limit check     v.10.9
      blue3 = blue3 * brout;                                               //  (result is better)
   }
      
/* ------------------------------------------------------------------------

      color saturation curve values:
           0 = no color saturation (gray scale)
         0.5 = normal (initial unmodified RGB)
         1.0 = max. color saturation

      0.5 >> 0:  move all RGB values to their mean: (R+G+B)/3
      0.5 >> 1.0:  increase RGB spread to 2x original level
*/

   if (tune_spc_moved[2])                                                  //  curve has been edited
   {
      brout = spccurve_yval(sd,2,xval);                                    //  saturation factor, 0 - 1
      brout = 2.0 * brout - 1.0;                                           //  -1.0 .. +1.0
      brmean = 0.333 * (red3 + green3 + blue3);
      red3 = red3 + brout * (red3 - brmean);                               //  simplified       v.10.9
      green3 = green3 + brout * (green3 - brmean);
      blue3 = blue3 + brout * (blue3 - brmean);
      if (red3 < 0) red3 = 0;
      if (green3 < 0) green3 = 0;
      if (blue3 < 0) blue3 = 0;
   }

/* ------------------------------------------------------------------------

      color balance curve values:
           0 = 0.5 * original color
         0.5 = unmodified
         1.0 = 1.5 * original color, clipped
*/

   if (tune_spc_moved[3]) {                                                //  curve has been edited
      brout = spccurve_yval(sd,3,xval);
      red3 = red3 * (brout + 0.5);
   }

   if (tune_spc_moved[4]) {                                                //  curve has been edited
      brout = spccurve_yval(sd,4,xval);
      green3 = green3 * (brout + 0.5);
   }

   if (tune_spc_moved[5]) {                                                //  curve has been edited
      brout = spccurve_yval(sd,5,xval);
      blue3 = blue3 * (brout + 0.5);
   }

/* ------------------------------------------------------------------------
   
   if working within a select area, blend changes over distance from edge

*/
      
   double      dold, dnew;

   if (Factivearea && dist < sa_blend) {
      dnew = 1.0 * dist / sa_blend;
      dold = 1.0 - dnew;
      red3 = dnew * red3 + dold * red1;
      green3 = dnew * green3 + dold * green1;
      blue3 = dnew * blue3 + dold * blue1;
   }

/* ------------------------------------------------------------------------
   
   prevent clipping and set output RGB values

*/

   if (red3 > 65535) red3 = 65535;
   if (green3 > 65535) green3 = 65535;
   if (blue3 > 65535) blue3 = 65535;

   pix3[0] = int(red3);
   pix3[1] = int(green3);
   pix3[2] = int(blue3);
   
   return;
}


/**************************************************************************/

//  ramp brightness across image, vertical and horizontal gradients        //  new  v.9.3

void m_brightramp(GtkWidget *, cchar *)
{
   int    brramp_dialog_event(zdialog* zd, cchar *event);
   void   brramp_curvedit(int spc);
   void * brramp_thread(void *);
   
   zfuncs::F1_help_topic = "brightness_ramp";                              //  v.10.8

   if (! edit_setup("briteramp",1,2)) return;                              //  setup edit: preview, select area OK

   zdedit = zdialog_new(ZTX("Ramp brightness across image"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5|expand");
   zdialog_add_widget(zdedit,"vbox","vb1","hb1");
   zdialog_add_widget(zdedit,"vbox","vb2","hb1",0,"expand");
   zdialog_add_widget(zdedit,"label","lmax","vb1","+","space=3");
   zdialog_add_widget(zdedit,"label","lspace","vb1",0,"expand");
   zdialog_add_widget(zdedit,"label","lmin","vb1","‒","space=3");
   zdialog_add_widget(zdedit,"label","lspace","vb1");
   zdialog_add_widget(zdedit,"frame","frame","vb2",0,"expand");
   zdialog_add_widget(zdedit,"hbox","hb2","vb2");
   zdialog_add_widget(zdedit,"label","lmin","hb2","‒","space=3");
   zdialog_add_widget(zdedit,"label","lspace","hb2",0,"expand");
   zdialog_add_widget(zdedit,"label","lmax","hb2","+","space=3");

   zdialog_add_widget(zdedit,"hbox","hbcf","dialog",0,"space=3");
   zdialog_add_widget(zdedit,"label","labcf","hbcf",ZTX("Curve File:"),"space=8");
   zdialog_add_widget(zdedit,"button","load","hbcf",Bopen,"space=5");
   zdialog_add_widget(zdedit,"button","save","hbcf",Bsave,"space=5");

   GtkWidget *frame = zdialog_widget(zdedit,"frame");
   spcdat *sd = spccurve_init(frame,brramp_curvedit);                      //  v.11.01
   Espcdat = sd;

   sd->Nspc = 2;
   sd->nap[0] = 4;
   sd->vert[0] = 0;
   sd->apx[0][0] = 0.01;
   sd->apx[0][1] = 0.4;
   sd->apx[0][2] = 0.6;
   sd->apx[0][3] = 0.99;
   sd->apy[0][0] = sd->apy[0][1] = sd->apy[0][2] = sd->apy[0][3] = 0.5;

   sd->nap[1] = 4;
   sd->vert[1] = 1;
   sd->apx[1][0] = 0.01;
   sd->apx[1][1] = 0.4;
   sd->apx[1][2] = 0.6;
   sd->apx[1][3] = 0.99;
   sd->apy[1][0] = sd->apy[1][1] = sd->apy[1][2] = sd->apy[1][3] = 0.5;

   spccurve_generate(sd,0);
   spccurve_generate(sd,1);

   zdialog_resize(zdedit,260,300);
   zdialog_run(zdedit,brramp_dialog_event);                                //  run dialog, parallel

   start_thread(brramp_thread,0);                                          //  start working thread
   signal_thread();
   return;
}


//  dialog event and completion callback function

int brramp_dialog_event(zdialog *zd, cchar *event)
{
   spcdat *sd = Espcdat;

   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      return 1;
   }
   
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3

   if (strEqu(event,"load")) {                                             //  load saved curves      v.11.02
      spccurve_load(sd);
      signal_thread();
      return 0;
   }

   if (strEqu(event,"save")) {                                             //  save curve to file     v.11.02
      spccurve_save(sd);
      return 0;
   }

   return 1;
}


//  this function is called when a curve is edited

void brramp_curvedit(int spc)
{
   signal_thread();
   return;
}


//  brramp thread function

void * brramp_thread(void *arg)
{
   void * brramp_wthread(void *);
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker threads
         start_wt(brramp_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      mwpaint2();                                                          //  update window
      Fmodified = 1;                                                       //  image3 modified
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * brramp_wthread(void *arg)                                           //  worker thread function
{
   int         index = *((int *) arg);
   int         ii, dist = 0, px3, py3;
   int         red1, green1, blue1, maxrgb;
   int         red3, green3, blue3;
   uint16      *pix1, *pix3;
   double      dispx, dispy;
   double      hramp, vramp, tramp;
   double      spanw, spanh, dold, dnew;
   spcdat      *sd = Espcdat;

   if (Factivearea) {                                                      //  if select area active, ramp
      spanw = sa_maxx - sa_minx;                                           //    brightness over enclosing rectangle
      spanh = sa_maxy - sa_miny;
   }
   else {
      spanw = E3ww;                                                        //  else over entire image
      spanh = E3hh;
   }   

   for (py3 = index; py3 < E3hh; py3 += Nwt)                               //  loop output pixels
   for (px3 = 0; px3 < E3ww; px3++)
   {
      if (Factivearea) {                                                   //  select area active        v.10.1
         ii = py3 * E3ww + px3;
         dist = sa_pixisin[ii];                                            //  distance from edge
         if (! dist) continue;                                             //  pixel outside area
         dispx = (px3 - sa_minx) / spanw;
         dispy = (py3 - sa_miny) / spanh;
      }
      else {
         dispx = px3 / spanw;                                              //  left > right = 0 to 1
         dispy = py3 / spanh;                                              //  top > bottom = 0 to 1
      }
      
      hramp = spccurve_yval(sd,0,dispx) - 0.5;                             //  -0.5 to +0.5
      vramp = spccurve_yval(sd,1,dispy) - 0.5;                             //  -0.5 to +0.5
      tramp = 1.0 + hramp + vramp;

      pix1 = PXMpix(E1pxm16,px3,py3);                                      //  input pixel
      pix3 = PXMpix(E3pxm16,px3,py3);                                      //  output pixel
      
      red1 = pix1[0];
      green1 = pix1[1];
      blue1 = pix1[2];

      maxrgb = red1;
      if (green1 > maxrgb) maxrgb = green1;
      if (blue1 > maxrgb) maxrgb = blue1;

      if (tramp * maxrgb > 65535) tramp = 65535.0 / maxrgb;

      red3 = tramp * red1;
      green3 = tramp * green1;
      blue3 = tramp * blue1;

      if (Factivearea && dist < sa_blend) {                                //  blend changes over blendwidth   v.10.1
         dnew = 1.0 * dist / sa_blend;
         dold = 1.0 - dnew;
         red3 = dnew * red3 + dold * red1;
         green3 = dnew * green3 + dold * green1;
         blue3 = dnew * blue3 + dold * blue1;
      }

      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
   }

   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


/**************************************************************************/

//  Clip pixels outside the given low/high range and expand the remaining
//  brightness range back to the full range of 0 to 65535.

double   xbrangeD, xbrangeB;

void m_xbrange(GtkWidget *, cchar *)                                       //  new v.10.1
{
   int      xbrange_dialog_event(zdialog *zd, cchar *event);
   void *   xbrange_thread(void *);
   
   zfuncs::F1_help_topic = "expand_brightness";                            //  v.10.8

   cchar  *title = ZTX("Expand Brightness Range");
   cchar  *labD = ZTX("dark pixels");
   cchar  *labB = ZTX("bright pixels");

   if (! edit_setup("expand",1,2)) return;                                 //  setup: preview, select area OK

   zdedit = zdialog_new(title,mWin,Bdone,Bcancel,null);                    //  revised v.11.01
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"expand");
   zdialog_add_widget(zdedit,"vbox","vb1","hb1",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb2","hb1",0,"homog|expand");
   zdialog_add_widget(zdedit,"label","labD","vb1",labD);
   zdialog_add_widget(zdedit,"label","labB","vb1",labB);
   zdialog_add_widget(zdedit,"hscale","clipD","vb2","0|99|0.5|0");
   zdialog_add_widget(zdedit,"hscale","clipB","vb2","0|99|0.5|0");

   zdialog_resize(zdedit,300,0);
   zdialog_run(zdedit,xbrange_dialog_event);                               //  run dialog, parallel
   
   m_histogram(0,0);                                                       //  show brightness distribution    v.11.02

   xbrangeD = xbrangeB = 0;                                                //  initial clip = 0
   
   start_thread(xbrange_thread,0);                                         //  start working thread
   return;
}


//  dialog event and completion callback function

int xbrange_dialog_event(zdialog *zd, cchar *event)
{
   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      histogram_destroy();                                                 //  v.11.02
      return 1;
   }

   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3

   if (strEqu(event,"clipD")) {
      zdialog_fetch(zd,"clipD",xbrangeD);
      signal_thread();
   }

   if (strEqu(event,"clipB")) {
      zdialog_fetch(zd,"clipB",xbrangeB);
      signal_thread();
   }

   return 0;
}


//  thread function

void * xbrange_thread(void *)
{
   int         ii, px, py, rgb, dist = 0;
   double      dark, bright, b1, b3, bf;
   double      pval1, pval3, f1, f2;
   uint16      *pix1, *pix3;
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      dark = 0.01 * xbrangeD * 65536;                                      //  clipping brightness levels
      bright = (1.0 - 0.01 * xbrangeB) * 65536;

      for (py = 0; py < E3hh; py++)                                        //  loop all image pixels
      for (px = 0; px < E3ww; px++)
      {
         ii = py * E3ww + px;

         if (Factivearea) {                                                //  select area active     bugfix v.9.9
            dist = sa_pixisin[ii];                                         //  distance from edge
            if (! dist) continue;                                          //  pixel is outside area
         }

         pix1 = PXMpix(E1pxm16,px,py);                                     //  input pixel
         pix3 = PXMpix(E3pxm16,px,py);                                     //  output pixel

         b1 = pixbright(pix1);

         if (b1 < dark)                                                    //  clip dark pixels
            b3 = 0;
         else if (b1 > bright)                                             //  clip bright pixels
            b3 = bright;
         else b3 = b1;
         
         if (b3 > dark) b3 = b3 - dark;                                    //  expand the rest
         b3 = b3 * (65535.0 / (bright - dark));

         bf = b3 / (b1 + 1);                                               //  brightness ratio
         
         for (rgb = 0; rgb < 3; rgb++)                                     //  loop 3 RGB colors
         {
            pval1 = pix1[rgb];
            pval3 = bf * pval1;                                            //  apply ratio
            if (pval3 > 65535) pval3 = 65535;

            if (Factivearea && dist < sa_blend) {                          //  select area is active,
               f1 = 1.0 * dist / sa_blend;                                 //    blend changes over sa_blend
               f2 = 1.0 - f1;
               pval3 = int(f1 * pval3 + f2 * pval1);
            }

            pix3[rgb] = pval3;
         }
      }

      Fmodified = 1;
      mwpaint2();
   }

   return 0;                                                               //  not executed, stop g++ warning
}


/**************************************************************************

   Image Tone Mapping function
   enhance local contrast as opposed to overall contrast

   methodology:
   get brightness gradients for each pixel in 4 directions: SE SW NE NW
   amplify gradients using the edit curve (x-axis range 0-max. gradient)
   integrate 4 new brightness surfaces from the amplified gradients:
     - pixel brightness = prior pixel brightness + amplified gradient
     - the Amplify control varies amplification from zero to overamplified
   new pixel brightness = average from 4 calculated brightness surfaces

***************************************************************************/

float    *Tmap_brmap1;
float    *Tmap_brmap3[4];
int      Tmap_contrast99;
double   Tmap_amplify;

void m_tonemap(GtkWidget *, cchar *)                                       //  new v.9.8
{
   int      Tmap_dialog_event(zdialog *zd, cchar *event);
   void     Tmap_curvedit(int);
   void *   Tmap_thread(void *);

   zfuncs::F1_help_topic = "tone_mapping";                                 //  v.10.8

   int         ii, cc, px, py;
   uint16      *pix1;
   cchar       *title = ZTX("Tone Mapping");
   int         jj, sum, limit, condist[100];

   if (! edit_setup("tonemap",0,2)) return;                                //  setup: no preview, select area OK

/***
            _____________________________
           |                             |
           |                             |
           |    curve drawing area       |
           |                             |
           |_____________________________| 
            low       contrast       high

            Amplify ========[]=========

            Curve File: [ Open ] [ Save ]
***/

   zdedit = zdialog_new(title,mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"frame","frame","dialog",0,"expand");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labcL","hb1","low","space=4");
   zdialog_add_widget(zdedit,"label","labcM","hb1","contrast","expand");
   zdialog_add_widget(zdedit,"label","labcH","hb1","high","space=5");

   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labcon","hb2",ZTX("Amplify"),"space=5");
   zdialog_add_widget(zdedit,"hscale","amplify","hb2","0|1|0.01|0","expand");

   zdialog_add_widget(zdedit,"hbox","hbcf","dialog",0,"space=8");
   zdialog_add_widget(zdedit,"label","labcf","hbcf",ZTX("Curve File:"),"space=5");
   zdialog_add_widget(zdedit,"button","load","hbcf",Bopen,"space=5");
   zdialog_add_widget(zdedit,"button","save","hbcf",Bsave,"space=5");

   GtkWidget *frame = zdialog_widget(zdedit,"frame");                      //  set up curve edit
   spcdat *sd = spccurve_init(frame,Tmap_curvedit);                        //  v.11.01
   Espcdat = sd;

   sd->Nspc = 1;
   sd->vert[0] = 0;
   sd->nap[0] = 3;                                                         //  initial curve anchor points
   sd->apx[0][0] = 0.01;
   sd->apy[0][0] = 0.2;
   sd->apx[0][1] = 0.50;
   sd->apy[0][1] = 0.4;
   sd->apx[0][2] = 0.99;
   sd->apy[0][2] = 0.0;

   spccurve_generate(sd,0);                                                //  generate curve data

   cc = Fww * Fhh * sizeof(float);                                         //  allocate brightness map memory
   Tmap_brmap1 = (float *) zmalloc(cc,"tmap.br1");
   for (ii = 0; ii < 4; ii++)
      Tmap_brmap3[ii] = (float *) zmalloc(cc,"tmap.br3");

   for (py = 0; py < Fhh; py++)                                            //  map initial image brightness
   for (px = 0; px < Fww; px++)
   {
      ii = py * Fww + px;
      pix1 = PXMpix(E1pxm16,px,py);
      Tmap_brmap1[ii] = pixbright(pix1);
   }

   for (ii = 0; ii < 100; ii++) 
      condist[ii] = 0;

   for (py = 1; py < Fhh; py++)                                            //  map contrast distribution
   for (px = 1; px < Fww; px++)
   {
      ii = py * Fww + px;
      jj = 0.00152 * fabsf(Tmap_brmap1[ii] - Tmap_brmap1[ii-1]);           //  contrast, ranged 0 - 99
      condist[jj]++;
      jj = 0.00152 * fabsf(Tmap_brmap1[ii] - Tmap_brmap1[ii-Fww]);
      condist[jj]++;
   }
   
   sum = 0;
   limit = 0.99 * 2 * (Fww-1) * (Fhh-1);                                   //  find 99th percentile contrast

   for (ii = 0; ii < 100; ii++) {
      sum += condist[ii];
      if (sum > limit) break;
   }
   
   Tmap_contrast99 = 65535.0 * ii / 100.0;                                 //  0 to 65535
   if (Tmap_contrast99 < 1000) Tmap_contrast99 = 1000;                     //  rescale low-contrast image   v.10.9

   zdialog_resize(zdedit,300,300);
   zdialog_run(zdedit,Tmap_dialog_event);                                  //  run dialog, parallel

   Tmap_amplify = 0;
   start_thread(Tmap_thread,0);                                            //  start working thread
   signal_thread();
   return;
}


//  dialog event and completion callback function

int Tmap_dialog_event(zdialog *zd, cchar *event)
{
   spcdat *sd = Espcdat;

   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      zfree(Tmap_brmap1);                                                  //  free memory
      for (int ii = 0; ii < 4; ii++)
      zfree(Tmap_brmap3[ii]);
      return 1;
   }

   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3

   if (strEqu(event,"amplify")) {
      zdialog_fetch(zd,"amplify",Tmap_amplify);
      signal_thread();
   }

   if (strEqu(event,"load")) {                                             //  load saved curve       v.11.02
      spccurve_load(sd);
      signal_thread();
      return 0;
   }

   if (strEqu(event,"save")) {                                             //  save curve to file     v.11.02
      spccurve_save(sd);
      return 0;
   }
   
   return 0;
}


//  this function is called when the curve is edited

void Tmap_curvedit(int)
{
   signal_thread();
   return;
}


//  thread function

void * Tmap_thread(void *)
{
   void * Tmap_wthread1(void *arg);
   void * Tmap_wthread2(void *arg);
   
   int      ii;
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      if (Factivearea) SB_goal = sa_Npixel;                                //  setup progress tracking   v.10.2
      else  SB_goal = E3ww * E3hh;
      SB_goal = SB_goal * 5;                                               //  5 passes through all pixels   v.11.03
      for (ii = 0; ii < 8; ii++) SB_done[ii] = 0;                          //  v.11.03

      for (ii = 0; ii < 4; ii++)                                           //  start working threads1 (must be 4)
         start_wt(Tmap_wthread1,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      for (ii = 0; ii < Nwt; ii++)                                         //  start working threads2
         start_wt(Tmap_wthread2,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion
      
      SB_goal = 0;

      Fmodified = 1;
      mwpaint2(); 
   }

   return 0;                                                               //  not executed, stop g++ warning
}


//  working threads

void * Tmap_wthread1(void *arg)                                            //  hardened   v.9.9
{
   int         ii, bii, pii, dist = 0;
   int         px, px1, px2, pxinc;
   int         py, py1, py2, pyinc;
   float       b1, b3, xval, yval, grad;
   float       amplify, contrast99;
   spcdat      *sd = Espcdat;

   bii = *((int *) arg);

   if (bii == 0) {                                                         //  direction SE
      px1 = 1; px2 = E3ww; pxinc = 1;
      py1 = 1; py2 = E3hh; pyinc = 1;
      pii = - 1 - E3ww;
   }
   
   else if (bii == 1) {                                                    //  direction SW
      px1 = E3ww-2; px2 = 0; pxinc = -1;
      py1 = 1; py2 = E3hh; pyinc = 1;
      pii = + 1 - E3ww;
   }
   
   else if (bii == 2) {                                                    //  direction NE
      px1 = 1; px2 = E3ww; pxinc = 1;
      py1 = E3hh-2; py2 = 0; pyinc = -1;
      pii = - 1 + E3ww;
   }
   
   else {   /* bii == 3 */                                                 //  direction NW
      px1 = E3ww-2; px2 = 0; pxinc = -1;
      py1 = E3hh-2; py2 = 0; pyinc = -1;
      pii = + 1 + E3ww;
   }

   contrast99 = Tmap_contrast99;                                           //  99th percentile contrast
   contrast99 = 1.0 / contrast99;                                          //  inverted

   amplify = pow(Tmap_amplify,0.2);                                        //  get amplification, 0 to 1    v.11.02

   for (ii = 0; ii < E3ww * E3hh; ii++)                                    //  initial brightness map
      Tmap_brmap3[bii][ii] = Tmap_brmap1[ii];

   for (py = py1; py != py2; py += pyinc)                                  //  loop all image pixels
   for (px = px1; px != px2; px += pxinc)
   {
      SB_done[bii]++;                                                      //  track progress         v.10.2

      ii = py * E3ww + px;

      if (Factivearea) {                                                   //  select area active
         dist = sa_pixisin[ii];                                            //  distance from edge
         if (! dist) continue;                                             //  outside pixel
      }

      b1 = Tmap_brmap1[ii];                                                //  this pixel brightness
      grad = b1 - Tmap_brmap1[ii+pii];                                     //  - prior pixel --> gradient

      xval = fabsf(grad) * contrast99;                                     //  gradient scaled 0 to 1+
      yval = 1.0 + 5 * spccurve_yval(sd,0,xval);                           //  1 to 6
      grad = grad * yval;                                                  //  magnified gradient

      b3 = Tmap_brmap3[bii][ii+pii] + grad;                                //  pixel brightness = prior + gradient
      b3 = (1.0 - amplify) * b1 + amplify * b3;                            //  constrain: push b3 toward b1    v.11.02
      
      if (b3 > 65535) b3 = 65535;                                          //  constrain
      if (b3 < 10) b3 = 10;
      
      Tmap_brmap3[bii][ii] = b3;                                           //  new pixel brightness
   }

   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


void * Tmap_wthread2(void *arg)                                            //  speedup   v.10.2
{
   uint16      *pix1, *pix3;
   int         index, ii, px, py, dist = 0;
   int         rgb, pval1, pval3;
   float       b1, b3, bf, f1, f2;

   index = *((int *) arg);

   for (py = index; py < E3hh; py += Nwt)                                  //  loop all image pixels
   for (px = 0; px < E3ww; px++)
   {
      SB_done[index]++;                                                    //  track progress         v.10.2

      ii = py * E3ww + px;

      if (Factivearea) {                                                   //  select area active     bugfix v.9.9
         dist = sa_pixisin[ii];                                            //  distance from edge
         if (! dist) continue;                                             //  pixel is outside area
      }

      pix1 = PXMpix(E1pxm16,px,py);                                        //  input pixel
      pix3 = PXMpix(E3pxm16,px,py);                                        //  output pixel

      b1 = Tmap_brmap1[ii];                                                //  initial pixel brightness
      b3 = Tmap_brmap3[0][ii] + Tmap_brmap3[1][ii]                         //  new brightness = average of four
         + Tmap_brmap3[2][ii] + Tmap_brmap3[3][ii];                        //    calculated brightness surfaces
      
      bf = 0.25 * b3 / (b1 + 1);                                           //  brightness ratio
      
      for (rgb = 0; rgb < 3; rgb++)                                        //  loop 3 RGB colors
      {
         pval1 = pix1[rgb];
         pval3 = bf * pval1;                                               //  apply ratio
         if (pval3 > 65535) pval3 = 65535;

         if (Factivearea && dist < sa_blend) {                             //  select area is active,
            f1 = 1.0 * dist / sa_blend;                                    //    blend changes over sa_blend
            f2 = 1.0 - f1;
            pval3 = int(f1 * pval3 + f2 * pval1);
         }

         pix3[rgb] = pval3;
      }
   }

   exit_wt();
   return 0;                                                               //  not executed, stop g++ warning
}


/**************************************************************************/

//  red eye removal function

struct sredmem {                                                           //  red-eye struct in memory
   char        type, space[3];
   int         cx, cy, ww, hh, rad, clicks;
   double      thresh, tstep;
};
sredmem  redmem[100];                                                      //  store up to 100 red-eyes

int      Nredmem = 0, maxredmem = 100;
void     redeye_mousefunc();


void m_redeye(GtkWidget *, cchar *)
{
   int      redeye_dialog_event(zdialog *zd, cchar *event);

   cchar    *redeye_message = ZTX(
               "Method 1:\n"
               "  Left-click on red-eye to darken.\n"
               "Method 2:\n"
               "  Drag down and right to enclose red-eye.\n"
               "  Left-click on red-eye to darken.\n"
               "Undo red-eye:\n"
               "  Right-click on red-eye.");

   zfuncs::F1_help_topic = "red_eye";                                      //  v.10.8

   if (! edit_setup("redeye",0,1)) return;                                 //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Red Eye Reduction"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",redeye_message);
   zdialog_add_widget(zdedit,"check","mymouse","dialog",BmyMouse);
   zdialog_run(zdedit,redeye_dialog_event);                                //  run dialog, parallel mode

   Nredmem = 0;
   zdialog_stuff(zdedit,"mymouse",1);                                      //  v.11.03
   takeMouse(zdedit,redeye_mousefunc,dragcursor);                          //  connect mouse function          v.11.03
   return;
}


//  dialog event and completion callback function

int redeye_dialog_event(zdialog *zd, cchar *event)
{
   int      mymouse;

   if (zd->zstat) {
      freeMouse();                                                         //  disconnect mouse function       v.10.12
      if (Nredmem > 0) Fmodified = 1;
      Ftoparc = ptoparc = 0;
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      return 0;
   }

   if (strEqu(event,"mymouse")) {                                          //  toggle mouse capture         v.10.12
      zdialog_fetch(zd,"mymouse",mymouse);
      if (mymouse) takeMouse(zd,redeye_mousefunc,dragcursor);              //  connect mouse function
      else freeMouse();                                                    //  disconnect mouse
   }

   return 0;
}


//  mouse functions to define, darken, and undo red-eyes

int      redeye_createF(int px, int py);                                   //  create 1-click red-eye (type F)
int      redeye_createR(int px, int py, int ww, int hh);                   //  create robust red-eye (type R)
void     redeye_darken(int ii);                                            //  darken red-eye
void     redeye_distr(int ii);                                             //  build pixel redness distribution
int      redeye_find(int px, int py);                                      //  find red-eye at mouse position
void     redeye_remove(int ii);                                            //  remove red-eye at mouse position
int      redeye_radlim(int cx, int cy);                                    //  compute red-eye radius limit


void redeye_mousefunc()
{
   int         ii, px, py, ww, hh;

   if (Nredmem == maxredmem) {
      zmessageACK(mWin,"%d red-eye limit reached",maxredmem);              //  too many red-eyes
      return;
   }

   if (LMclick)                                                            //  left mouse click
   {
      LMclick = 0;

      px = Mxclick;                                                        //  click position
      py = Myclick;
      if (px < 0 || px > E3ww-1 || py < 0 || py > E3hh-1) return;          //  outside image area

      ii = redeye_find(px,py);                                             //  find existing red-eye
      if (ii < 0) ii = redeye_createF(px,py);                              //  or create new type F
      redeye_darken(ii);                                                   //  darken red-eye
   }
   
   if (RMclick)                                                            //  right mouse click
   {
      RMclick = 0;
      px = Mxclick;                                                        //  click position
      py = Myclick;
      ii = redeye_find(px,py);                                             //  find red-eye
      if (ii >= 0) redeye_remove(ii);                                      //  if found, remove
   }

   if (Mxdrag || Mydrag)                                                   //  mouse drag underway
   {
      px = Mxdown;                                                         //  initial position
      py = Mydown;
      ww = Mxdrag - Mxdown;                                                //  increment
      hh = Mydrag - Mydown;
      if (ww < 2 && hh < 2) return;
      if (ww < 2) ww = 2;
      if (hh < 2) hh = 2;
      if (px < 1) px = 1;                                                  //  keep within image area
      if (py < 1) py = 1;      
      if (px + ww > E3ww-1) ww = E3ww-1 - px;
      if (py + hh > E3hh-1) hh = E3hh-1 - py;
      ii = redeye_find(px,py);                                             //  find existing red-eye
      if (ii >= 0) redeye_remove(ii);                                      //  remove it
      ii = redeye_createR(px,py,ww,hh);                                    //  create new red-eye type R
   }

   mwpaint2();
   return;
}


//  create type F redeye (1-click automatic)

int redeye_createF(int cx, int cy)
{
   int         cx0, cy0, cx1, cy1, px, py, rad, radlim;
   int         loops, ii;
   int         Tnpix, Rnpix, R2npix;
   double      rd, rcx, rcy, redpart;
   double      Tsum, Rsum, R2sum, Tavg, Ravg, R2avg;
   double      sumx, sumy, sumr;
   uint16      *ppix;
   
   cx0 = cx;
   cy0 = cy;
   
   for (loops = 0; loops < 8; loops++)
   {
      cx1 = cx;
      cy1 = cy;

      radlim = redeye_radlim(cx,cy);                                       //  radius limit (image edge)
      Tsum = Tavg = Ravg = Tnpix = 0;

      for (rad = 0; rad < radlim-2; rad++)                                 //  find red-eye radius from (cx,cy)
      {
         Rsum = Rnpix = 0;
         R2sum = R2npix = 0;

         for (py = cy-rad-2; py <= cy+rad+2; py++)
         for (px = cx-rad-2; px <= cx+rad+2; px++)
         {
            rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
            ppix = PXMpix(E3pxm16,px,py);
            redpart = pixred(ppix);

            if (rd <= rad + 0.5 && rd > rad - 0.5) {                       //  accum. redness at rad
               Rsum += redpart;
               Rnpix++;
            }
            else if (rd <= rad + 2.5 && rd > rad + 1.5) {                  //  accum. redness at rad+2
               R2sum += redpart;
               R2npix++;
            }
         }
         
         Tsum += Rsum;
         Tnpix += Rnpix;
         Tavg = Tsum / Tnpix;                                              //  avg. redness over 0-rad
         Ravg = Rsum / Rnpix;                                              //  avg. redness at rad
         R2avg = R2sum / R2npix;                                           //  avg. redness at rad+2
         if (R2avg > Ravg || Ravg > Tavg) continue;
         if ((Ravg - R2avg) < 0.2 * (Tavg - Ravg)) break;                  //  0.1 --> 0.2
      }
      
      sumx = sumy = sumr = 0;
      rad = int(1.2 * rad + 1);
      if (rad > radlim) rad = radlim;
      
      for (py = cy-rad; py <= cy+rad; py++)                                //  compute center of gravity for
      for (px = cx-rad; px <= cx+rad; px++)                                //   pixels within rad of (cx,cy)
      {
         rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
         if (rd > rad + 0.5) continue;
         ppix = PXMpix(E3pxm16,px,py);
         redpart = pixred(ppix);                                           //  weight by redness
         sumx += redpart * (px - cx);
         sumy += redpart * (py - cy);
         sumr += redpart;
      }

      rcx = cx + 1.0 * sumx / sumr;                                        //  new center of red-eye
      rcy = cy + 1.0 * sumy / sumr;
      if (fabs(cx0 - rcx) > 0.6 * rad) break;                              //  give up if big movement
      if (fabs(cy0 - rcy) > 0.6 * rad) break;
      cx = int(rcx + 0.5);
      cy = int(rcy + 0.5);
      if (cx == cx1 && cy == cy1) break;                                   //  done if no change
   }

   radlim = redeye_radlim(cx,cy);
   if (rad > radlim) rad = radlim;

   ii = Nredmem++;                                                         //  add red-eye to memory
   redmem[ii].type = 'F';
   redmem[ii].cx = cx;
   redmem[ii].cy = cy;
   redmem[ii].rad = rad;
   redmem[ii].clicks = 0;
   redmem[ii].thresh = 0;
   return ii;
}


//  create type R red-eye (drag an ellipse over red-eye area)

int redeye_createR(int cx, int cy, int ww, int hh)
{
   int      rad, radlim;

   Ftoparc = 1;                                                            //  paint ellipse over image
   toparcx = cx - ww;
   toparcy = cy - hh;
   toparcw = 2 * ww;
   toparch = 2 * hh;

   if (ww > hh) rad = ww;
   else rad = hh;
   radlim = redeye_radlim(cx,cy);
   if (rad > radlim) rad = radlim;

   int ii = Nredmem++;                                                     //  add red-eye to memory
   redmem[ii].type = 'R';
   redmem[ii].cx = cx;
   redmem[ii].cy = cy;
   redmem[ii].ww = 2 * ww;
   redmem[ii].hh = 2 * hh;
   redmem[ii].rad = rad;
   redmem[ii].clicks = 0;
   redmem[ii].thresh = 0;
   return ii;
}


//  darken a red-eye and increase click count

void redeye_darken(int ii)
{
   int         cx, cy, ww, hh, px, py, rad, clicks;
   double      rd, thresh, tstep;
   char        type;
   uint16      *ppix;

   type = redmem[ii].type;
   cx = redmem[ii].cx;
   cy = redmem[ii].cy;
   ww = redmem[ii].ww;
   hh = redmem[ii].hh;
   rad = redmem[ii].rad;
   thresh = redmem[ii].thresh;
   tstep = redmem[ii].tstep;
   clicks = redmem[ii].clicks++;
   
   if (thresh == 0)                                                        //  1st click 
   {
      redeye_distr(ii);                                                    //  get pixel redness distribution
      thresh = redmem[ii].thresh;                                          //  initial redness threshhold
      tstep = redmem[ii].tstep;                                            //  redness step size
      Ftoparc = 0;
   }

   tstep = (thresh - tstep) / thresh;                                      //  convert to reduction factor
   thresh = thresh * pow(tstep,clicks);                                    //  reduce threshhold by total clicks

   for (py = cy-rad; py <= cy+rad; py++)                                   //  darken pixels over threshhold
   for (px = cx-rad; px <= cx+rad; px++)
   {
      if (type == 'R') {
         if (px < cx - ww/2) continue;
         if (px > cx + ww/2) continue;
         if (py < cy - hh/2) continue;
         if (py > cy + hh/2) continue;
      }
      rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
      if (rd > rad + 0.5) continue;
      ppix = PXMpix(E3pxm16,px,py);                                        //  set redness = threshhold
      if (pixred(ppix) > thresh)
         ppix[0] = int(thresh * (0.65 * ppix[1] + 0.10 * ppix[2] + 1) / (25 - 0.25 * thresh));
   }

   return;
}


//  Build a distribution of redness for a red-eye. Use this information 
//  to set initial threshhold and step size for stepwise darkening.

void redeye_distr(int ii)
{
   int         cx, cy, ww, hh, rad, px, py;
   int         bin, npix, dbins[20], bsum, blim;
   double      rd, maxred, minred, redpart, dbase, dstep;
   char        type;
   uint16      *ppix;
   
   type = redmem[ii].type;
   cx = redmem[ii].cx;
   cy = redmem[ii].cy;
   ww = redmem[ii].ww;
   hh = redmem[ii].hh;
   rad = redmem[ii].rad;
   
   maxred = 0;
   minred = 100;

   for (py = cy-rad; py <= cy+rad; py++)
   for (px = cx-rad; px <= cx+rad; px++)
   {
      if (type == 'R') {
         if (px < cx - ww/2) continue;
         if (px > cx + ww/2) continue;
         if (py < cy - hh/2) continue;
         if (py > cy + hh/2) continue;
      }
      rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
      if (rd > rad + 0.5) continue;
      ppix = PXMpix(E3pxm16,px,py);
      redpart = pixred(ppix);
      if (redpart > maxred) maxred = redpart;
      if (redpart < minred) minred = redpart;
   }
   
   dbase = minred;
   dstep = (maxred - minred) / 19.99;

   for (bin = 0; bin < 20; bin++) dbins[bin] = 0;
   npix = 0;

   for (py = cy-rad; py <= cy+rad; py++)
   for (px = cx-rad; px <= cx+rad; px++)
   {
      if (type == 'R') {
         if (px < cx - ww/2) continue;
         if (px > cx + ww/2) continue;
         if (py < cy - hh/2) continue;
         if (py > cy + hh/2) continue;
      }
      rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
      if (rd > rad + 0.5) continue;
      ppix = PXMpix(E3pxm16,px,py);
      redpart = pixred(ppix);
      bin = int((redpart - dbase) / dstep);
      ++dbins[bin];
      ++npix;
   }
   
   bsum = 0;
   blim = int(0.5 * npix);

   for (bin = 0; bin < 20; bin++)                                          //  find redness level for 50% of
   {                                                                       //    pixels within red-eye radius
      bsum += dbins[bin];
      if (bsum > blim) break;
   }

   redmem[ii].thresh = dbase + dstep * bin;                                //  initial redness threshhold
   redmem[ii].tstep = dstep;                                               //  redness step (5% of range)

   return;
}


//  find a red-eye (nearly) overlapping the mouse click position

int redeye_find(int cx, int cy)
{
   for (int ii = 0; ii < Nredmem; ii++)
   {
      if (cx > redmem[ii].cx - 2 * redmem[ii].rad && 
          cx < redmem[ii].cx + 2 * redmem[ii].rad &&
          cy > redmem[ii].cy - 2 * redmem[ii].rad && 
          cy < redmem[ii].cy + 2 * redmem[ii].rad) 
            return ii;                                                     //  found
   }
   return -1;                                                              //  not found
}


//  remove a red-eye from memory

void redeye_remove(int ii)
{
   int      cx, cy, rad, px, py;
   uint16   *pix1, *pix3;

   cx = redmem[ii].cx;
   cy = redmem[ii].cy;
   rad = redmem[ii].rad;

   for (px = cx-rad; px <= cx+rad; px++)
   for (py = cy-rad; py <= cy+rad; py++)
   {
      pix1 = PXMpix(E1pxm16,px,py);
      pix3 = PXMpix(E3pxm16,px,py);
      pix3[0] = pix1[0];
      pix3[1] = pix1[1];
      pix3[2] = pix1[2];
   }
   
   for (ii++; ii < Nredmem; ii++) 
      redmem[ii-1] = redmem[ii];
   Nredmem--;
   
   Ftoparc = 0;
   return;
}


//  compute red-eye radius limit: smaller of 100 and nearest image edge

int redeye_radlim(int cx, int cy)
{
   int radlim = 100;
   if (cx < 100) radlim = cx;
   if (E3ww-1 - cx < 100) radlim = E3ww-1 - cx;
   if (cy < 100) radlim = cy;
   if (E3hh-1 - cy < 100) radlim = E3hh-1 - cy;
   return radlim;
}


/**************************************************************************/

//  image blur function                                                    //  radius steps of 0.5   v.9.2

double      blur_radius;
double      blur_weight[101][101];                                         //  up to blur radius = 99   v.9.2
int         blur_Npixels, blur_pixdone;


void m_blur(GtkWidget *, cchar *)
{
   int    blur_dialog_event(zdialog *zd, cchar *event);
   void * blur_thread(void *);

   zfuncs::F1_help_topic = "blur";                                         //  v.10.8

   if (! edit_setup("blur",0,2)) return;                                   //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Set Blur Radius"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"label","labrad","hb2",Bradius,"space=5");
   zdialog_add_widget(zdedit,"spin","radius","hb2","0|99|0.5|0.5","space=5");
   zdialog_add_widget(zdedit,"button","apply","hb2",Bapply,"space=5");

   zdialog_run(zdedit,blur_dialog_event);                                  //  start dialog
   
   blur_radius = 0.5;
   start_thread(blur_thread,0);                                            //  start working thread
   return;
}


//  dialog event and completion callback function

int blur_dialog_event(zdialog * zd, cchar *event)
{
   if (zd->zstat)
   {
      if (zd->zstat == 1) edit_done();                                     //  done
      else edit_cancel();                                                  //  cancel or destroy
      return 1;
   }
   
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3

   if (strEqu(event,"apply")) {
      zdialog_fetch(zd,"radius",blur_radius);                              //  get blur radius
      if (blur_radius == 0) edit_reset();
      else signal_thread();                                                //  trigger working thread
   }

   return 1;
}


//  image blur thread function

void * blur_thread(void *)
{
   void * blur_wthread(void *arg);

   int         dx, dy;
   double      rad, rad2;
   double      m, d, w, sum;
  
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      rad = blur_radius - 0.2;                                             //  v.9.2
      rad2 = rad * rad;

      for (dx = 0; dx <= rad+1; dx++)                                      //  clear weights array
      for (dy = 0; dy <= rad+1; dy++)
         blur_weight[dx][dy] = 0;

      for (dx = -rad-1; dx <= rad+1; dx++)                                 //  blur_weight[dx][dy] = no. of pixels
      for (dy = -rad-1; dy <= rad+1; dy++)                                 //    at distance (dx,dy) from center
         ++blur_weight[abs(dx)][abs(dy)];

      m = sqrt(rad2 + rad2);                                               //  corner pixel distance from center
      sum = 0;

      for (dx = 0; dx <= rad+1; dx++)                                      //  compute weight of pixel
      for (dy = 0; dy <= rad+1; dy++)                                      //    at distance dx, dy
      {
         d = sqrt(dx*dx + dy*dy);
         w = (m + 1.2 - d) / m;                                            //  v.9.2
         w = w * w;
         sum += blur_weight[dx][dy] * w;
         blur_weight[dx][dy] = w;
      }

      for (dx = 0; dx <= rad+1; dx++)                                      //  make weights add up to 1.0
      for (dy = 0; dy <= rad+1; dy++)
         blur_weight[dx][dy] = blur_weight[dx][dy] / sum;
      
      if (Factivearea) SB_goal = sa_Npixel;
      else  SB_goal = E3ww * E3hh;
      for (int ii = 0; ii < 8; ii++) SB_done[ii] = 0;                      //  v.11.03

      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker threads
         start_wt(blur_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      SB_goal = 0;
      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * blur_wthread(void *arg)                                             //  worker thread function
{
   void  blur_pixel(int px, int py);

   int      index = *((int *) arg);
   int      px, py;
   
   for (py = index; py < E3hh-1; py += Nwt)                                //  loop all image pixels
   for (px = 0; px < E3ww-1; px++)
   {
      blur_pixel(px,py);
      SB_done[index]++;
   }

   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


void blur_pixel(int px, int py)
{
   int         ii, dist = 0;
   int         jj, dx, dy, adx, ady, rad;
   double      red, green, blue;
   double      weight1, weight2, f1, f2;
   uint16      *pix1, *pix3, *pixN;

   if (Factivearea) {                                                      //  select area active
      ii = py * Fww + px;
      dist = sa_pixisin[ii];                                               //  distance from edge
      if (! dist) return;                                                  //  outside pixel
   }

   pix1 = PXMpix(E1pxm16,px,py);                                           //  source pixel
   pix3 = PXMpix(E3pxm16,px,py);                                           //  target pixel
   
   rad = blur_radius;
   red = green = blue = 0;
   weight2 = 0.0;
   
   if (Factivearea)                                                        //  select area active
   {
      for (dy = -rad-1; dy <= rad+1; dy++)                                 //  loop neighbor pixels within radius
      for (dx = -rad-1; dx <= rad+1; dx++)
      {
         if (px+dx < 0 || px+dx > E3ww-1) continue;                        //  omit pixels off edge
         if (py+dy < 0 || py+dy > E3hh-1) continue;
         jj = (py+dy) * E3ww + (px+dx);
         if (! sa_pixisin[jj]) continue;                                   //  omit pixels outside area
         adx = abs(dx);
         ady = abs(dy);
         pixN = pix1 + (dy * E3ww + dx) * 3;
         weight1 = blur_weight[adx][ady];                                  //  weight at distance (dx,dy)
         weight2 += weight1;
         red += pixN[0] * weight1;                                         //  accumulate contributions
         green += pixN[1] * weight1;
         blue += pixN[2] * weight1;
      }
      
      red = red / weight2;                                                 //  weighted average
      green = green / weight2;
      blue = blue / weight2;

      if (dist < sa_blend) {                                               //  select area is active,
         f1 = 1.0 * dist / sa_blend;                                       //    blend changes over sa_blend
         f2 = 1.0 - f1;
         red = f1 * red + f2 * pix1[0];
         green = f1 * green + f2 * pix1[1];
         blue = f1 * blue + f2 * pix1[2];
      }

      pix3[0] = int(red);
      pix3[1] = int(green);
      pix3[2] = int(blue);
   }

   else
   {
      for (dy = -rad-1; dy <= rad+1; dy++)                                 //  loop neighbor pixels within radius
      for (dx = -rad-1; dx <= rad+1; dx++)
      {
         if (px+dx < 0 || px+dx > E3ww-1) continue;                        //  omit pixels off edge
         if (py+dy < 0 || py+dy > E3hh-1) continue;
         adx = abs(dx);
         ady = abs(dy);
         pixN = pix1 + (dy * E3ww + dx) * 3;
         weight1 = blur_weight[adx][ady];                                  //  weight at distance (dx,dy)
         weight2 += weight1;
         red += pixN[0] * weight1;                                         //  accumulate contributions
         green += pixN[1] * weight1;
         blue += pixN[2] * weight1;
      }

      red = red / weight2;                                                 //  weighted average
      green = green / weight2;
      blue = blue / weight2;

      pix3[0] = int(red);
      pix3[1] = int(green);
      pix3[2] = int(blue);
   }
   
   return;
}


/**************************************************************************/

//  image sharpening function

int      sharp_ED_cycles;
int      sharp_ED_reduce;
int      sharp_ED_thresh;
int      sharp_UM_radius;
int      sharp_UM_amount;
int      sharp_UM_thresh;
int      sharp_UM_Fcalc;
int      sharp_GR_amount;
int      sharp_GR_thresh;
char     sharp_function[4];


void m_sharpen(GtkWidget *, cchar *)
{
   int    sharp_dialog_event(zdialog *zd, cchar *event);
   void * sharp_thread(void *);

   zfuncs::F1_help_topic = "sharpen";                                      //  v.10.8

   if (! edit_setup("sharp",0,2)) return;                                  //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Sharpen Image"),mWin,Bdone,Bcancel,null); 

   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb11","hb1",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb12","hb1",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb13","hb1",0,"homog|space=5");
   zdialog_add_widget(zdedit,"button","ED","vb11",ZTX("edge detection"),"space=5");
   zdialog_add_widget(zdedit,"label","lab11","vb12",ZTX("cycles"));
   zdialog_add_widget(zdedit,"label","lab12","vb12",ZTX("reduce"));
   zdialog_add_widget(zdedit,"label","lab13","vb12",Bthresh);
   zdialog_add_widget(zdedit,"spin","cyclesED","vb13","1|30|1|10");
   zdialog_add_widget(zdedit,"spin","reduceED","vb13","50|95|1|80");
   zdialog_add_widget(zdedit,"spin","threshED","vb13","1|99|1|1");

   zdialog_add_widget(zdedit,"hsep","sep2","dialog");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb21","hb2",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb22","hb2",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb23","hb2",0,"homog|space=5");
   zdialog_add_widget(zdedit,"button","UM","vb21",ZTX("unsharp mask"),"space=5");
   zdialog_add_widget(zdedit,"label","lab21","vb22",Bradius);
   zdialog_add_widget(zdedit,"label","lab22","vb22",Bamount);
   zdialog_add_widget(zdedit,"label","lab23","vb22",Bthresh);
   zdialog_add_widget(zdedit,"spin","radiusUM","vb23","1|20|1|2");
   zdialog_add_widget(zdedit,"spin","amountUM","vb23","0|200|1|100");
   zdialog_add_widget(zdedit,"spin","threshUM","vb23","0|100|1|0");

   zdialog_add_widget(zdedit,"hsep","sep3","dialog");
   zdialog_add_widget(zdedit,"hbox","hb3","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb31","hb3",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb32","hb3",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb33","hb3",0,"homog|space=5");
   zdialog_add_widget(zdedit,"button","GR","vb31",ZTX("brightness gradient"),"space=5");
   zdialog_add_widget(zdedit,"label","lab32","vb32",Bamount);
   zdialog_add_widget(zdedit,"label","lab33","vb32",Bthresh);
   zdialog_add_widget(zdedit,"spin","amountGR","vb33","0|400|1|100");
   zdialog_add_widget(zdedit,"spin","threshGR","vb33","0|100|1|0");

   zdialog_run(zdedit,sharp_dialog_event);                                 //  run dialog, parallel

   *sharp_function = 0;
   sharp_UM_Fcalc = 1;
   start_thread(sharp_thread,0);                                           //  start working thread
   return;
}


//  dialog event and completion callback function

int sharp_dialog_event(zdialog *zd, cchar *event)
{
   if (zd->zstat)
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      return 0;
   }
   
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3
   if (strEqu(event,"radiusUM")) sharp_UM_Fcalc = 1;                       //  must recalculate

   if (strcmpv(event,"ED","UM","GR",null))
   {
      edit_reset();                                                        //  restore original image

      zdialog_fetch(zd,"cyclesED",sharp_ED_cycles);                        //  get all input values
      zdialog_fetch(zd,"reduceED",sharp_ED_reduce);
      zdialog_fetch(zd,"threshED",sharp_ED_thresh);
      zdialog_fetch(zd,"radiusUM",sharp_UM_radius);
      zdialog_fetch(zd,"amountUM",sharp_UM_amount);
      zdialog_fetch(zd,"threshUM",sharp_UM_thresh);
      zdialog_fetch(zd,"amountGR",sharp_GR_amount);
      zdialog_fetch(zd,"threshGR",sharp_GR_thresh);

      strcpy(sharp_function,event);                                        //  pass to working thread
      signal_thread();
   }

   return 0;
}


//  sharpen image thread function

void * sharp_thread(void *)
{
   int    sharp_ED();
   int    sharp_UM();
   int    sharp_GR();
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      if (strEqu(sharp_function,"ED")) sharp_ED();                         //  do requested function
      if (strEqu(sharp_function,"UM")) sharp_UM();
      if (strEqu(sharp_function,"GR")) sharp_GR();

      Fmodified = 1;
      mwpaint2(); 
   }

   return 0;                                                               //  not executed, stop g++ warning
}


//  image sharpen function by edge detection and compression

int sharp_ED()
{
   void  sharp_pixel_ED(int px, int py, int thresh);

   int      sharp_thresh1 = 100;                                           //  initial threshold
   double   sharp_thresh2 = 0.01 * sharp_ED_reduce;                        //  decline rate
   int      ii, px, py, thresh, cycles;
   
   thresh = sharp_thresh1;
    
   if (Factivearea) SB_goal = sa_Npixel;                                   //  v.9.6
   else  SB_goal = E3ww * E3hh;
   SB_goal *= sharp_ED_cycles;
   for (ii = 0; ii < 8; ii++) SB_done[ii] = 0;                             //  v.11.03

   for (cycles = 0; cycles < sharp_ED_cycles; cycles++)
   {
      if (cycles > 0) thresh = int(thresh * sharp_thresh2);

      for (py = 2; py < E3hh-2; py++)
      for (px = 2; px < E3ww-2; px++)                                      //  loop all pixels
      {
         sharp_pixel_ED(px,py,thresh);
         SB_done[0]++;
      }
   }

   SB_goal = 0;
   return 1;
}


void sharp_pixel_ED(int px, int py, int thresh)
{
   uint16   *pix1, *pix1u, *pix1d;
   uint16   *pix3, *pix3u, *pix3d, *pix3uu, *pix3dd;
   int      ii, dist = 0;
   int      dd, rgb, pthresh;
   int      dx[4] = { -1, 0, 1, 1 };                                       //  4 directions: NW N NE E
   int      dy[4] = { -1, -1, -1, 0 };
   int      pv2, pv2u, pv2d, pv2uu, pv2dd, pvdiff;
   double   f1, f2;
   
   if (Factivearea) {                                                      //  select area active
      ii = py * Fww + px;
      dist = sa_pixisin[ii];                                               //  distance from edge
      if (! dist) return;                                                  //  outside pixel
   }

   pthresh = sharp_ED_thresh;                                              //  pthresh = larger
   if (thresh > pthresh) pthresh = thresh;

   pix1 = PXMpix(E1pxm16,px,py);                                           //  input pixel
   pix3 = PXMpix(E3pxm16,px,py);                                           //  output pixel

   for (dd = 0; dd < 4; dd++)                                              //  4 directions
   {
      pix3u = pix3 + (dy[dd] * E3ww + dx[dd]) * 3;                         //  upstream pixel
      pix3d = pix3 - (dy[dd] * E3ww - dx[dd]) * 3;                         //  downstream pixel

      for (rgb = 0; rgb < 3; rgb++)                                        //  loop 3 RGB colors
      {
         pv2 = pix3[rgb];
         pv2u = pix3u[rgb];                                                //  brightness difference
         pv2d = pix3d[rgb];                                                //    across target pixel

         pvdiff = pv2d - pv2u;
         if (pvdiff < 0) pvdiff = -pvdiff;
         if (pvdiff < 256 * pthresh) continue;                             //  brightness slope < threshold

         if (pv2u < pv2 && pv2 < pv2d)                                     //  slope up, monotone
         {
            pix3uu = pix3u + (dy[dd] * E3ww + dx[dd]) * 3;                 //  upstream of upstream pixel
            pix3dd = pix3d - (dy[dd] * E3ww - dx[dd]) * 3;                 //  downstream of downstream
            pv2uu = pix3uu[rgb];
            pv2dd = pix3dd[rgb];

            if (pv2uu >= pv2u) {                                           //  shift focus of changes to
               pix3u = pix3;                                               //    avoid up/down/up jaggies
               pv2u = pv2;
            }
            
            if (pv2dd <= pv2d) {
               pix3d = pix3;
               pv2d = pv2;
            }
               
            if (pv2u > 256) pv2u -= 256;
            if (pv2d < 65279) pv2d += 256;
         }
         
         else if (pv2u > pv2 && pv2 > pv2d)                                //  slope down, monotone
         {
            pix3uu = pix3u + (dy[dd] * E3ww + dx[dd]) * 3;
            pix3dd = pix3d - (dy[dd] * E3ww - dx[dd]) * 3;
            pv2uu = pix3uu[rgb];
            pv2dd = pix3dd[rgb];

            if (pv2uu <= pv2u) {
               pix3u = pix3;
               pv2u = pv2;
            }
            
            if (pv2dd >= pv2d) {
               pix3d = pix3;
               pv2d = pv2;
            }

            if (pv2d > 256) pv2d -= 256;
            if (pv2u < 65279) pv2u += 256;
         }

         else continue;                                                    //  slope too small

         if (Factivearea && dist < sa_blend) {                             //  select area is active,
            f1 = 1.0 * dist / sa_blend;                                    //    blend changes over sa_blend
            f2 = 1.0 - f1;
            pix1u = pix1 + (dy[dd] * E1ww + dx[dd]) * 3;                   //  upstream input pixel
            pix1d = pix1 - (dy[dd] * E1ww - dx[dd]) * 3;                   //  downstream input pixel
            pv2u = int(f1 * pv2u + f2 * pix1u[rgb]);
            pv2d = int(f1 * pv2d + f2 * pix1d[rgb]);
         }

         pix3u[rgb] = pv2u;                                                //  modified brightness values
         pix3d[rgb] = pv2d;                                                //    >> image3 pixel
      }
   }

   return;
}


//  image sharpen function using unsharp mask

int sharp_UM()
{
   void * sharp_UM_wthread(void *arg);
   
   int      ii;
   
   if (sharp_UM_Fcalc) {                                                   //  speedup   v.9.6 
      sharp_UM_Fcalc = 0;
      brhood_calc(sharp_UM_radius,'f');
   }   

   if (Factivearea) SB_goal = sa_Npixel;                                   //  v.9.6
   else  SB_goal = E3ww * E3hh;
   for (ii = 0; ii < 8; ii++) SB_done[ii] = 0;                             //  v.11.03

   for (ii = 0; ii < Nwt; ii++)                                            //  start worker threads
      start_wt(sharp_UM_wthread,&wtnx[ii]);
   wait_wts();                                                             //  wait for completion

   SB_goal = 0;
   return 1;
}


void * sharp_UM_wthread(void *arg)                                         //  worker thread function
{
   void  sharp_pixel_UM(int px, int py, int index);

   int      index = *((int *) arg);
   int      px, py;
   
   for (py = index; py < E3hh; py += Nwt)                                  //  loop all image3 pixels
   for (px = 0; px < E3ww; px++)
      sharp_pixel_UM(px,py,index);

   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


void sharp_pixel_UM(int px, int py, int index)                             //  process one pixel
{                                                                          //  revised  v.9.6
   int         ii, dist = 0;
   double      amount, thresh, bright;
   double      mean, incr, ratio, f1, f2;
   int         rgb, cval1, cval3;
   uint16      *pix1, *pix3;
   
   if (Factivearea) {                                                      //  select area active
      ii = py * Fww + px;
      dist = sa_pixisin[ii];                                               //  distance from edge
      if (! dist) return;                                                  //  outside pixel
   }

   amount = 0.01 * sharp_UM_amount;                                        //  0.0 to 2.0
   thresh = 100 * sharp_UM_thresh;                                         //  0 to 10K (64K max. possible)

   pix1 = PXMpix(E1pxm16,px,py);                                           //  input pixel
   pix3 = PXMpix(E3pxm16,px,py);                                           //  output pixel

   bright = pixbright(pix1);
   if (bright < 100) return;                                               //  effectively black
   mean = get_brhood(px,py);
   incr = (bright - mean);
   if (fabs(incr) < thresh) return;                                        //  omit low-contrast pixels

   incr = incr * amount;                                                   //  0.0 to 2.0
   if (bright + incr > 65535) incr = 65535 - bright;
   ratio = (bright + incr) / bright;

   for (rgb = 0; rgb < 3; rgb++)                                           //  loop 3 RGB colors
   {
      cval1 = pix1[rgb];
      cval3 = ratio * cval1;
      if (cval3 < 0) cval3 = 0;
      if (cval3 > 65535) cval3 = 65535;
      
      if (Factivearea && dist < sa_blend) {                                //  select area is active,
         f1 = 1.0 * dist / sa_blend;                                       //    blend changes over sa_blend
         f2 = 1.0 - f1;
         cval3 = f1 * cval3 + f2 * cval1;
      }
      
      pix3[rgb] = cval3;
   }
   
   SB_done[index]++;   
   return;
}


//  sharpen image by increasing brightness gradient                        //  new v.9.8

int sharp_GR()
{
   uint16      *pix1, *pix3;
   int         ii, px, py, dist = 0, rgb;   
   double      amount, thresh;
   double      b1, b1x, b1y, b3x, b3y, b3, bf;
   double      pval, f1, f2;

   amount = 1 + 0.01 * sharp_GR_amount;                                    //  1.0 - 5.0
   thresh = 655.35 * sharp_GR_thresh;                                      //  0 - 64K

   if (Factivearea) SB_goal = sa_Npixel;
   else  SB_goal = E3ww * E3hh;
   for (ii = 0; ii < 8; ii++) SB_done[ii] = 0;                             //  v.11.03

   for (py = 1; py < E1hh; py++)                                           //  loop all image pixels
   for (px = 1; px < E1ww; px++)
   {
      if (Factivearea) {                                                   //  select area active
         ii = py * Fww + px;
         dist = sa_pixisin[ii];                                            //  distance from edge
         if (! dist) continue;                                             //  pixel is outside area
      }

      pix1 = PXMpix(E1pxm16,px,py);                                        //  input pixel
      pix3 = PXMpix(E3pxm16,px,py);                                        //  output pixel

      b1 = pixbright(pix1);                                                //  pixel brightness, 0 - 64K
      b1x = b1 - pixbright(pix1 - 3);                                      //  brightness gradient (x,y)
      b1y = b1 - pixbright(pix1 - 3 * E1ww);
      
      if (abs(b1x + b1y) < thresh)                                         //  moderate brightness change for
         f1 = abs(b1x + b1y) / thresh;                                     //    pixels below threshold gradient
      else  f1 = 1.0;                                                      //  v.10.9
      f2 = 1.0 - f1;

      b1x = b1x * amount;                                                  //  amplified gradient
      b1y = b1y * amount;

      b3x = pixbright(pix1 - 3) + b1x;                                     //  + prior pixel brightness
      b3y = pixbright(pix1 - 3 * E3ww) + b1y;                              //  = new brightness
      b3 = 0.5 * (b3x + b3y);

      b3 = f1 * b3 + f2 * b1;                                              //  possibly moderated     v.10.9

      bf = b3 / b1;                                                        //  ratio of brightness change
      if (bf < 0) bf = 0;
      if (bf > 4) bf = 4;
      
      for (rgb = 0; rgb < 3; rgb++)                                        //  loop 3 RGB colors
      {
         pval = bf * pix3[rgb];                                            //  apply factor
         if (pval > 65535) pval = 65535;
         
         if (Factivearea && dist < sa_blend) {                             //  select area is active,
            f1 = 1.0 * dist / sa_blend;                                    //    blend changes over sa_blend
            f2 = 1.0 - f1;
            pval = int(f1 * pval + f2 * pix1[rgb]);
         }
         
         pix3[rgb] = pval;
      }

      SB_done[0]++;
   }

   SB_goal = 0;
   return 1;
}


/**************************************************************************/

//  image noise reduction

int      denoise_method = 5;                                               //  default algorithm
int      denoise_radius = 4;


void m_denoise(GtkWidget *, cchar *)
{
   int    denoise_dialog_event(zdialog *zd, cchar *event);                 //  dialog event function
   void * denoise_thread(void *);

   cchar  *denoise_message = ZTX(" Press the reduce button to \n"
                                 " reduce noise in small steps. \n"
                                 " Use undo to start over.");

   zfuncs::F1_help_topic = "reduce_noise";                                 //  v.10.8

   if (! edit_setup("noise",0,2)) return;                                  //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Noise Reduction"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",denoise_message,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labalg","hb1",ZTX("algorithm"),"space=5");
   zdialog_add_widget(zdedit,"combo","method","hb1",0,"space=5|expand");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labrad","hb2",Bradius,"space=5");
   zdialog_add_widget(zdedit,"spin","radius","hb2","1|9|1|4","space=5");
   zdialog_add_widget(zdedit,"button","reduce","hb2",Breduce,"space=5");
   
   zdialog_cb_app(zdedit,"method",ZTX("flatten outliers by color (1)"));
   zdialog_cb_app(zdedit,"method",ZTX("flatten outliers by color (2)"));
   zdialog_cb_app(zdedit,"method",ZTX("set median brightness by color"));
   zdialog_cb_app(zdedit,"method",ZTX("top hat filter by color"));
   zdialog_stuff(zdedit,"method",ZTX("top hat filter by color"));          //  default

   zdialog_run(zdedit,denoise_dialog_event);                               //  run dialog
   start_thread(denoise_thread,0);                                         //  start working thread
   return;
}


//  dialog event and completion callback function

int denoise_dialog_event(zdialog * zd, cchar *event)
{
   char     method[40];

   if (zd->zstat)
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      return 0;
   }

   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  trigger update thread

   if (strEqu(event,"radius")) 
      zdialog_fetch(zd,"radius",denoise_radius);

   if (strEqu(event,"method")) 
   {
      zdialog_fetch(zd,"method",method,39);

      if (strEqu(method,"flatten outliers by color (1)")) {
         denoise_method = 1;
         denoise_radius = 1;
      }

      if (strEqu(method,"flatten outliers by color (2)")) {
         denoise_method = 2;
         denoise_radius = 3;
      }

      if (strEqu(method,"set median brightness by color")) {
         denoise_method = 4;
         denoise_radius = 2;
      }

      if (strEqu(method,"top hat filter by color")) {
         denoise_method = 5;
         denoise_radius = 4;
      }
      
      zdialog_stuff(zd,"radius",denoise_radius);
   }
   
   if (strEqu(event,"reduce")) signal_thread();                            //  trigger update thread

   return 1;
}


//  image noise reduction thread

void * denoise_thread(void *)
{
   void * denoise_wthread(void *arg);
   
   int      ii;

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      E9pxm16 = PXM_copy(E3pxm16);                                         //  image3 is reference source
                                                                           //  image9 will be modified
      if (Factivearea) SB_goal = sa_Npixel;
      else  SB_goal = E3ww * E3hh;
      for (ii = 0; ii < 8; ii++) SB_done[ii] = 0;                          //  v.11.03

      for (ii = 0; ii < Nwt; ii++)                                         //  start worker threads
         start_wt(denoise_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      SB_goal = 0;

      mutex_lock(&Fpixmap_lock);
      PXM_free(E3pxm16);                                                   //  image9 >> image3
      E3pxm16 = E9pxm16;
      E9pxm16 = 0;
      mutex_unlock(&Fpixmap_lock);

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * denoise_wthread(void *arg)                                          //  worker thread function
{
   void  denoise_func1(uint16 *pix3, uint16 *pix9);
   void  denoise_func2(uint16 *pix3, uint16 *pix9);
   void  denoise_func4(uint16 *pix3, uint16 *pix9);
   void  denoise_func5(uint16 *pix3, uint16 *pix9);
   
   int         index = *((int *) arg);
   int         ii, px, py, rad, dist = 0;
   double      f1, f2;
   uint16      *pix1, *pix3, *pix9;

   rad = denoise_radius;

   for (py = index+rad; py < E3hh-rad; py += Nwt)                          //  loop all image3 pixels
   for (px = rad; px < E3ww-rad; px++)
   {
      if (Factivearea) {                                                   //  select area active
         ii = py * Fww + px;
         dist = sa_pixisin[ii];                                            //  distance from edge
         if (! dist) continue;                                             //  outside pixel
      }

      pix3 = PXMpix(E3pxm16,px,py);                                        //  source pixel
      pix9 = PXMpix(E9pxm16,px,py);                                        //  target pixel

      if (denoise_method == 1) denoise_func1(pix3,pix9);
      if (denoise_method == 2) denoise_func2(pix3,pix9);
      if (denoise_method == 4) denoise_func4(pix3,pix9);
      if (denoise_method == 5) denoise_func5(pix3,pix9);

      if (Factivearea && dist < sa_blend) {                                //  select area is active,
         f1 = 1.0 * dist / sa_blend;                                       //    blend changes over sa_blend
         f2 = 1.0 - f1;
         pix1 = PXMpix(E1pxm16,px,py);                                     //  source pixel
         pix9[0] = int(f1 * pix9[0] + f2 * pix1[0]);
         pix9[1] = int(f1 * pix9[1] + f2 * pix1[1]);
         pix9[2] = int(f1 * pix9[2] + f2 * pix1[2]);
      }

      SB_done[index]++;
   }

   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


//  flatten outliers within radius, by color 
//  an outlier is the max or min value within a radius

void denoise_func1(uint16 *pix3, uint16 *pix9)
{
   int         dy, dx, rad;
   int         min0, min1, min2, max0, max1, max2;
   uint16      *pixN;

   min0 = min1 = min2 = 65535;
   max0 = max1 = max2 = 0;
   rad = denoise_radius;

   for (dy = -rad; dy <= rad; dy++)                                        //  loop surrounding pixels
   for (dx = -rad; dx <= rad; dx++)
   {
      if (dy == 0 && dx == 0) continue;                                    //  skip self

      pixN = pix3 + (dy * E3ww + dx) * 3;
      if (pixN[0] < min0) min0 = pixN[0];                                  //  find min and max per color
      if (pixN[0] > max0) max0 = pixN[0];
      if (pixN[1] < min1) min1 = pixN[1];
      if (pixN[1] > max1) max1 = pixN[1];
      if (pixN[2] < min2) min2 = pixN[2];
      if (pixN[2] > max2) max2 = pixN[2];
   }
   
   if (pix3[0] <= min0 && min0 < 65279) pix9[0] = min0 + 256;              //  if outlier, flatten a little
   if (pix3[0] >= max0 && max0 > 256) pix9[0] = max0 - 256;
   if (pix3[1] <= min1 && min1 < 65279) pix9[1] = min1 + 256;
   if (pix3[1] >= max1 && max1 > 256) pix9[1] = max1 - 256;
   if (pix3[2] <= min2 && min2 < 65279) pix9[2] = min2 + 256;
   if (pix3[2] >= max2 && max2 > 256) pix9[2] = max2 - 256;
   
   return;
}


//  flatten outliers
//  An outlier pixel has an RGB value outside one sigma of 
//  the mean for all pixels within a given radius of the pixel.

void denoise_func2(uint16 *pix3, uint16 *pix9)
{
   int         rgb, dy, dx, rad, nn;
   double      nn1, val, sum, sum2, mean, variance, sigma;
   uint16      *pixN;

   rad = denoise_radius;
   nn = (rad * 2 + 1);
   nn = nn * nn - 1;
   nn1 = 1.0 / nn;

   for (rgb = 0; rgb < 3; rgb++)                                           //  loop RGB color
   {
      sum = sum2 = 0;

      for (dy = -rad; dy <= rad; dy++)                                     //  loop surrounding pixels
      for (dx = -rad; dx <= rad; dx++)
      {
         if (dy == 0 && dx == 0) continue;                                 //  skip self
         pixN = pix3 + (dy * E3ww + dx) * 3;
         val = pixN[rgb];
         sum += val;
         sum2 += val * val;
      }
      
      mean = nn1 * sum;
      variance = nn1 * (sum2 - 2.0 * mean * sum) + mean * mean;
      sigma = sqrt(variance);

      val = pix3[rgb];      
      if (val > mean + sigma) {                                            //  move value to mean +/- sigma
         val = mean + sigma;
         pix9[rgb] = val;
      }
      else if (val < mean - sigma) {
         val = mean - sigma;
         pix9[rgb] = val;
      }
   }
   
   return;
}


//  use median brightness for pixels within radius

void denoise_func4(uint16 *pix3, uint16 *pix9)
{
   int         dy, dx, rad;
   int         ns, rgb, bsortN[400];
   uint16      *pixN;

   rad = denoise_radius;

   for (rgb = 0; rgb < 3; rgb++)                                           //  loop all RGB colors
   {
      ns = 0;

      for (dy = -rad; dy <= rad; dy++)                                     //  loop surrounding pixels
      for (dx = -rad; dx <= rad; dx++)                                     //  get brightness values
      {
         pixN = pix3 + (dy * E3ww + dx) * 3;
         bsortN[ns] = pixN[rgb];
         ns++;
      }

      HeapSort(bsortN,ns);
      pix9[rgb] = bsortN[ns/2];                                            //  median brightness of ns pixels
   }

   return;
}


//  modified top hat filter: execute with increasing radius from 1 to limit
//  detect outlier by comparing with pixels in outer radius

void denoise_func5(uint16 *pix3, uint16 *pix9)
{
   int         dy, dx, rad;
   int         min0, min1, min2, max0, max1, max2;
   uint16      *pixN;

   for (rad = 1; rad <= denoise_radius; rad++)
   for (int loops = 0; loops < 2; loops++)
   {
      min0 = min1 = min2 = 65535;
      max0 = max1 = max2 = 0;

      for (dy = -rad; dy <= rad; dy++)                                     //  loop all pixels within rad
      for (dx = -rad; dx <= rad; dx++)
      {
         if (dx > -rad && dx < rad) continue;                              //  skip inner pixels
         if (dy > -rad && dy < rad) continue;

         pixN = pix3 + (dy * E3ww + dx) * 3;
         if (pixN[0] < min0) min0 = pixN[0];                               //  find min and max per color
         if (pixN[0] > max0) max0 = pixN[0];                               //    among outermost pixels
         if (pixN[1] < min1) min1 = pixN[1];
         if (pixN[1] > max1) max1 = pixN[1];
         if (pixN[2] < min2) min2 = pixN[2];
         if (pixN[2] > max2) max2 = pixN[2];
      }
      
      if (pix3[0] < min0 && pix9[0] < 65279) pix9[0] += 256;               //  if central pixel is outlier,
      if (pix3[0] > max0 && pix9[0] > 256) pix9[0] -= 256;                 //    moderate its values
      if (pix3[1] < min1 && pix9[1] < 65279) pix9[1] += 256;
      if (pix3[1] > max1 && pix9[1] > 256) pix9[1] -= 256;
      if (pix3[2] < min2 && pix9[2] < 65279) pix9[2] += 256;
      if (pix3[2] > max2 && pix9[2] > 256) pix9[2] -= 256;
   }

   return;
}


/**************************************************************************/

//  Smart Erase menu function - Replace pixels inside a select area 
//    with a reflection of pixels outside the area.

void m_smart_erase(GtkWidget *, const char *)                              //  overhauled    v.11.04
{
   int    smart_erase_dialog_event(zdialog* zd, const char *event);

   int      cc;
   cchar    *erase_message = ZTX("1. Drag mouse to select. \n"
                                 "2. Erase.   3. Repeat. ");

   zfuncs::F1_help_topic = "smart_erase";
   
   if (! edit_setup("smart_erase",0,2)) return;                            //  setup edit
   
/*     ______________________________
      |                              |
      | 1. Drag mouse to select.     |
      | 2. Erase.   3. Repeat.       |
      |                              |
      | [x] my mouse                 |
      | Radius [10|v]  Blur [0.5|v]  |
      | [New Area]  [Erase]  [Undo]  |
      |                              |
      |                    [Done]    |
      |______________________________|
*/

   zdedit = zdialog_new(ZTX("Smart Erase"),mWin,Bdone,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",erase_message,"space=8");

   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"check","mymouse","hb1",BmyMouse,"space=5");

   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labr","hb2",ZTX("Radius"),"space=5");
   zdialog_add_widget(zdedit,"spin","radius","hb2","1|20|1|5");
   zdialog_add_widget(zdedit,"label","labb","hb2",ZTX("Blur"),"space=5");
   zdialog_add_widget(zdedit,"spin","blur","hb2","0|9|0.5|0.5");

   zdialog_add_widget(zdedit,"hbox","hb3","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"button","newarea","hb3",ZTX("New Area"),"space=10");
   zdialog_add_widget(zdedit,"button","erase","hb3",Berase,"space=10");
   zdialog_add_widget(zdedit,"button","undo1","hb3",Bundo,"space=5");

   zdialog_stuff(zdedit,"mymouse",0);
   zdialog_run(zdedit,smart_erase_dialog_event);

   sa_Npixel = sa_blend = sa_calced = Factivearea = 0;                     //  start new select area
   if (sa_pixseq) zfree(sa_pixseq);
   cc = Fww * Fhh;
   sa_pixseq = (uint16 *) zmalloc(2*cc,"pixseq");
   memset(sa_pixseq,0,2*cc);
   sa_currseq = sa_Ncurrseq = 0;
   sa_mode = 6;                                                            //  mode = select by mouse
   sa_stat = 1;                                                            //  status = active edit
   sa_show(1);

   sa_mouseradius = 5;                                                     //  initial mouse select radius
   return;
}


//  dialog event and completion function

int smart_erase_dialog_event(zdialog *zd, const char *event)               //  overhauled    v.11.04
{
   void smart_erase_func(int mode);
   void smart_erase_blur(double radius);
   
   double      radius;
   int         mymouse, cc;
   
   if (zd->zstat) {
      freeMouse();                                                         //  disconnect mouse function
      sa_delete();
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      return 0;
   }
   
   if (strEqu(event,"mymouse")) {                                          //  toggle mouse capture
      zdialog_fetch(zd,"mymouse",mymouse);
      if (mymouse) {
         sa_stat = 1;                                                      //  status = active edit
         takeMouse(zdedit,sa_radius_mousefunc,0);                          //  use select area by mouse function
         sa_show(1);
      }
      else {
         sa_stat = 2;                                                      //  pause edit
         freeMouse();                                                      //  disconnect mouse
      }
   }
   
   if (strEqu(event,"newarea")) {
      sa_delete();
      cc = Fww * Fhh;
      sa_pixseq = (uint16 *) zmalloc(2*cc,"pixseq");
      memset(sa_pixseq,0,2*cc);
      sa_mode = 6;                                                         //  mode = select by mouse
      sa_stat = 1;                                                         //  status = active edit
      sa_show(1);
   }
   
   if (strEqu(event,"radius"))
      zdialog_fetch(zd,"radius",sa_mouseradius);

   if (strEqu(event,"erase")) {                                            //  do smart erase
      sa_finish_auto();
      smart_erase_func(1);
      zdialog_fetch(zd,"blur",radius);                                     //  add optional blur
      if (radius > 0) smart_erase_blur(radius);
      sa_show(0);
      freeMouse();                                                         //  disconnect mouse
   }
   
   if (strEqu(event,"undo1"))                                              //  dialog undo, undo last erase
      smart_erase_func(2);

   if (strEqu(event,"undo"))                                               //  toolbar undo, undo all erases
      edit_undo();

   if (strEqu(event,"redo"))                                               //  toolbar redo, redo all erases
      edit_redo();
   
   return 0;
}


//  erase the area or restore last erased area

void smart_erase_func(int mode)
{
   int         px, py, npx, npy, dx, dy;
   int         qx, qy, sx, sy, tx, ty;
   int         ii, rad, inc, cc;
   int         dist, dist2, mindist2;
   double      slope;
   char        *pmap;
   uint16      *pix1, *pix3;

   if (! Factivearea) return;                                              //  nothing selected       v.11.05
   
   for (py = sa_miny; py <= sa_maxy; py++)                                 //  loop all pixels in area
   for (px = sa_minx; px <= sa_maxx; px++)
   {
      ii = py * Fww + px;
      if (! sa_pixisin[ii]) continue;                                      //  pixel not selected

      pix1 = PXMpix(E1pxm16,px,py);                                        //  input pixel
      pix3 = PXMpix(E3pxm16,px,py);                                        //  output pixel

      pix3[0] = pix1[0];                                                   //  restore pixels inside area
      pix3[1] = pix1[1];
      pix3[2] = pix1[2];
   }

   mwpaint2();                                                             //  update window
   
   if (mode == 2) return;                                                  //  mode = undo, done

   cc = Fww * Fhh;                                                         //  allocate pixel done map
   pmap = (char *) zmalloc(cc,"smart_erase.pmap");
   memset(pmap,0,cc);

   for (py = sa_miny; py < sa_maxy; py++)                                  //  loop all pixels in area
   for (px = sa_minx; px < sa_maxx; px++)
   {
      ii = py * Fww + px;
      if (! sa_pixisin[ii]) continue;                                      //  pixel not selected
      if (pmap[ii]) continue;                                              //  pixel already done

      mindist2 = 999999;                                                   //  find nearest edge
      npx = npy = 0;

      for (rad = 1; rad < 50; rad++)                                       //  50 pixel limit            v.11.05
      {
         for (qx = px-rad; qx <= px+rad; qx++)                             //  search within rad         v.11.05
         for (qy = py-rad; qy <= py+rad; qy++)
         {
            if (qx != px-rad && qx != px+rad &&                            //  exclude within rad-1 (already searched)
                qy != py-rad && qy != py+rad) continue;
            if (qx < 0 || qx >= Fww) continue;                             //  off image edge
            if (qy < 0 || qy >= Fhh) continue;
            ii = qy * Fww + qx;
            if (sa_pixisin[ii]) continue;                                  //  within selected area

            dx = (px - qx) * (px - qx);                                    //  found pixel
            dy = (py - qy) * (py - qy);
            dist2 = dx + dy;                                               //  distance**2
            if (dist2 < mindist2) {
               mindist2 = dist2;
               npx = qx;                                                   //  save nearest pixel found
               npy = qy;
            }
         }

         if (rad * rad >= mindist2) break;                                 //  can quit now
      }

      if (! npx && ! npy) continue;                                        //  edge not found, should not happen

      qx = npx;                                                            //  nearest edge pixel
      qy = npy;
      dist = sqrt(mindist2);                                               //  distance

      if (abs(qy - py) > abs(qx - px)) {                                   //  qx/qy = near edge from px/py
         slope = 1.0 * (qx - px) / (qy - py);
         if (qy > py) inc = 1;
         else inc = -1;
         for (sy = py; sy != qy+inc; sy += inc)                            //  sx/sy = line from px/py to qx/qy
         {
            sx = px + slope * (sy - py);
            ii = sy * Fww + sx;
            pmap[ii] = 1;
            tx = qx + (sx - px) + slope * inc;                             //  tx/ty = parallel line from qx/qy 
            ty = qy + (sy - py) + inc;
            if (tx < 0 || tx > Fww-1) break;
            if (ty < 0 || ty > Fhh-1) break;
            pix1 = PXMpix(E3pxm16,tx,ty);                                  //  copy pixel from tx/ty to sx/sy
            pix3 = PXMpix(E3pxm16,sx,sy);
            pix3[0] = pix1[0];
            pix3[1] = pix1[1];
            pix3[2] = pix1[2];
         }
      }

      else {                                                               //  bugfix, some pixels missed   v.11.05
         slope = 1.0 * (qy - py) / (qx - px);
         if (qx > px) inc = 1;
         else inc = -1;
         for (sx = px; sx != qx+inc; sx += inc) 
         {
            sy = py + slope * (sx - px);
            ii = sy * Fww + sx;
            pmap[ii] = 1;
            tx = qx + (sx - px) + inc;
            ty = qy + (sy - py) + slope * inc;
            if (tx < 0 || tx > Fww-1) break;
            if (ty < 0 || ty > Fhh-1) break;
            pix1 = PXMpix(E3pxm16,tx,ty);
            pix3 = PXMpix(E3pxm16,sx,sy);
            pix3[0] = pix1[0];
            pix3[1] = pix1[1];
            pix3[2] = pix1[2];
         }
      }
   }

   zfree(pmap);                                                            //  free memory
   Fmodified = 1;
   mwpaint2();                                                             //  update window
   return;
}


//  add blur to the erased area to help mask the side-effects

int smart_erase_blur(double radius)
{
   int         ii, px, py, dx, dy, adx, ady;
   double      blur_weight[10][10];                                        //  up to blur radius = 9
   double      rad, rad2;
   double      m, d, w, sum;
   double      red, green, blue;
   double      weight1, weight2;
   uint16      *pix9, *pix3, *pixN;
   
   if (! Factivearea) return 0;
  
   rad = radius - 0.2;
   rad2 = rad * rad;

   for (dx = 0; dx <= rad+1; dx++)                                         //  clear weights array
   for (dy = 0; dy <= rad+1; dy++)
      blur_weight[dx][dy] = 0;

   for (dx = -rad-1; dx <= rad+1; dx++)                                    //  blur_weight[dx][dy] = no. of pixels
   for (dy = -rad-1; dy <= rad+1; dy++)                                    //    at distance (dx,dy) from center
      ++blur_weight[abs(dx)][abs(dy)];

   m = sqrt(rad2 + rad2);                                                  //  corner pixel distance from center
   sum = 0;

   for (dx = 0; dx <= rad+1; dx++)                                         //  compute weight of pixel
   for (dy = 0; dy <= rad+1; dy++)                                         //    at distance dx, dy
   {
      d = sqrt(dx*dx + dy*dy);
      w = (m + 1.2 - d) / m;
      w = w * w;
      sum += blur_weight[dx][dy] * w;
      blur_weight[dx][dy] = w;
   }

   for (dx = 0; dx <= rad+1; dx++)                                         //  make weights add up to 1.0
   for (dy = 0; dy <= rad+1; dy++)
      blur_weight[dx][dy] = blur_weight[dx][dy] / sum;
   
   E9pxm16 = PXM_copy(E3pxm16);                                            //  copy edited image
   
   for (py = sa_miny; py < sa_maxy; py++)                                  //  loop all pixels in area
   for (px = sa_minx; px < sa_maxx; px++)
   {
      ii = py * Fww + px;
      if (! sa_pixisin[ii]) continue;                                      //  pixel not in area

      pix9 = PXMpix(E9pxm16,px,py);                                        //  source pixel
      pix3 = PXMpix(E3pxm16,px,py);                                        //  target pixel
      
      rad = radius;
      red = green = blue = 0;
      weight2 = 0.0;
   
      for (dy = -rad-1; dy <= rad+1; dy++)                                 //  loop neighbor pixels within radius
      for (dx = -rad-1; dx <= rad+1; dx++)
      {
         if (px+dx < 0 || px+dx > E3ww-1) continue;                        //  omit pixels off edge
         if (py+dy < 0 || py+dy > E3hh-1) continue;
         adx = abs(dx);
         ady = abs(dy);
         pixN = pix9 + (dy * E3ww + dx) * 3;
         weight1 = blur_weight[adx][ady];                                  //  weight at distance (dx,dy)
         weight2 += weight1;
         red += pixN[0] * weight1;                                         //  accumulate contributions
         green += pixN[1] * weight1;
         blue += pixN[2] * weight1;
      }
      
      red = red / weight2;                                                 //  weighted average
      green = green / weight2;
      blue = blue / weight2;

      pix3[0] = int(red);
      pix3[1] = int(green);
      pix3[2] = int(blue);
   }

   PXM_free(E9pxm16);

   Fmodified = 1;
   mwpaint2();                                                             //  update window
   return 0;
}


/**************************************************************************/

//  find and remove "dust" from an image (e.g. from a scanned dusty slide)
//  dust is defined as small dark areas surrounded by brighter areas
//  image 1   original with prior edits
//  image 3   accumulated dust removals that have been comitted
//  image 9   comitted dust removals + pending removal (work in process)

namespace dust_names 
{
   int         spotspann;                                                  //  max. dustspot spann, pixels
   int         spotspann2;                                                 //  spotspann **2
   double      brightness;                                                 //  brightness limit, 0 to 1 = white
   double      contrast;                                                   //  min. contrast, 0 to 1 = black/white
   int         *pixgroup;                                                  //  maps (px,py) to pixel group no.
   int         Fred;                                                       //  red pixels are on

   int         Nstack;

   struct spixstack {
      uint16      px, py;                                                  //  pixel group search stack
      uint16      direc;
   }           *pixstack;

   #define maxgroups 1000000
   int         Ngroups;
   int         groupcount[maxgroups];                                      //  count of pixels in each group
   double      groupbright[maxgroups];                                     //  
   int         edgecount[maxgroups];                                       //  group edge pixel count
   double      edgebright[maxgroups];                                      //  group edge pixel brightness sum

   typedef struct {
      uint16      px1, py1, px2, py2;                                      //  pixel group extreme pixels
      int         spann2;                                                  //  spann from px1/py1 to px2/py2
   }  sgroupspann;
   
   sgroupspann    groupspann[maxgroups];
}


void m_dust(GtkWidget *, const char *)                                     //  new v.11.05
{
   using namespace dust_names;

   int    dust_dialog_event(zdialog *zd, cchar *event);
   void * dust_find_thread(void *);
   
   zfuncs::F1_help_topic = "remove_dust";                                  //  v.11.05.1

   if (! edit_setup("dust",0,2)) return;                                   //  setup edit: no preview

   E9pxm16 = PXM_copy(E3pxm16);                                            //  image 9 = copy of image3
   Fred = 0;

   int cc = Fww * Fhh * sizeof(int);
   pixgroup = (int *) zmalloc(cc,"dust.pixgroup");                         //  maps pixels to assigned groups

   cc = Fww * Fhh * sizeof(spixstack);
   pixstack = (spixstack *) zmalloc(cc,"dust.pixstack");                   //  pixel group search stack
   
/***
                     Remove Dust

        spot size limit    =========[]===========
        max. brightness    =============[]=======
        min. contrast      ========[]============
        [erase] [red] [undo last] [apply]

                                  [Done] [Cancel]
***/

   zdedit = zdialog_new(ZTX("Remove Dust"),mWin,Bdone,Bcancel,null);

   zdialog_add_widget(zdedit,"hbox","hbssl","dialog",0,"space=1");
   zdialog_add_widget(zdedit,"label","labssl","hbssl",ZTX("spot size limit"),"space=5");
   zdialog_add_widget(zdedit,"hscale","spotspann","hbssl","1|50|1|20","space=5|expand");
   zdialog_add_widget(zdedit,"hbox","hbmb","dialog",0,"space=1");
   zdialog_add_widget(zdedit,"label","labmb","hbmb",ZTX("max. brightness"),"space=5");
   zdialog_add_widget(zdedit,"hscale","brightness","hbmb","1|999|1|700","space=5|expand");
   zdialog_add_widget(zdedit,"hbox","hbmc","dialog",0,"space=1");
   zdialog_add_widget(zdedit,"label","labmb","hbmc",ZTX("min. contrast"),"space=5");
   zdialog_add_widget(zdedit,"hscale","contrast","hbmc","1|500|1|50","space=5|expand");
   zdialog_add_widget(zdedit,"hbox","hbbutts","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"button","erase","hbbutts",Berase,"space=5");
   zdialog_add_widget(zdedit,"button","red","hbbutts",Bred,"space=5");
   zdialog_add_widget(zdedit,"button","undo1","hbbutts",Bundolast,"space=5");
   zdialog_add_widget(zdedit,"button","apply","hbbutts",Bapply,"space=5");

   zdialog_fetch(zdedit,"spotspann",spotspann);                            //  max. dustspot spann (pixels)
   spotspann2 = spotspann * spotspann;

   zdialog_fetch(zdedit,"brightness",brightness);                          //  max. dustspot brightness
   brightness = 0.001 * brightness;                                        //  scale 0 to 1 = white

   zdialog_fetch(zdedit,"contrast",contrast);                              //  min. dustspot contrast
   contrast = 0.001 * contrast;                                            //  scale 0 to 1 = black/white

   zdialog_resize(zdedit,300,0);
   zdialog_run(zdedit,dust_dialog_event);                                  //  run dialog - parallel

   start_thread(dust_find_thread,0);                                       //  start working thread
   signal_thread();

   return;
}


//  dialog event and completion callback function

int dust_dialog_event(zdialog *zd, cchar *event)
{
   using namespace dust_names;
   
   void dust_erase();

   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) {                                                //  done, use committed changes
         mutex_lock(&Fpixmap_lock);
         PXM_free(E3pxm16);
         E3pxm16 = E9pxm16;                                                //  image 3 = image 9
         E9pxm16 = 0;
         mutex_unlock(&Fpixmap_lock);
         edit_done();
      }
      else {                                                               //  cancel, discard changes
         PXM_free(E9pxm16);
         edit_cancel();
      }

      zfree(pixgroup);                                                     //  free memory
      zfree(pixstack);
      return 0;
   }
   
   if (strEqu(event,"undo")) edit_undo();
   if (strEqu(event,"redo")) edit_redo();
   
   if (strEqu(event,"spotspann") || strEqu(event,"brightness") 
          || strEqu(event,"contrast") || strEqu(event,"red"))
   {
      zdialog_fetch(zd,"spotspann",spotspann);                             //  max. dustspot spann (pixels)
      spotspann2 = spotspann * spotspann;

      zdialog_fetch(zd,"brightness",brightness);                           //  max. dustspot brightness
      brightness = 0.001 * brightness;                                     //  scale 0 to 1 = white
      
      zdialog_fetch(zdedit,"contrast",contrast);                           //  min. dustspot contrast
      contrast = 0.001 * contrast;                                         //  scale 0 to 1 = black/white

      signal_thread();                                                     //  do the work
   }
   
   if (strEqu(event,"erase")) dust_erase();
   if (strEqu(event,"blendwidth")) dust_erase();
   
   if (strEqu(event,"undo1")) {
      mutex_lock(&Fpixmap_lock);                                           //  image 3 = copy of image 9
      PXM_free(E3pxm16);
      E3pxm16 = PXM_copy(E9pxm16);
      mutex_unlock(&Fpixmap_lock);
      Fred = 0;
      mwpaint2();
   }
   
   if (strEqu(event,"apply")) {
      if (Fred) dust_erase();
      PXM_free(E9pxm16);                                                   //  image 9 = copy of image 3
      E9pxm16 = PXM_copy(E3pxm16);
      Fmodified = 1;
   }
   
   return 0;
}


//  dust find thread function - find the dust particles and mark them

void * dust_find_thread(void *)
{
   using namespace dust_names;

   int         xspann, yspann, spann2;
   int         group, cc, ii, kk, Nremoved;
   int         px, py, dx, dy, ppx, ppy, npx, npy;
   double      gbright, pbright, pcontrast;
   double      ff = 1.0 / 65536.0;
   uint16      direc, *pix3;
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      mutex_lock(&Fpixmap_lock);                                           //  image 3 = copy of image 9
      PXM_free(E3pxm16);
      E3pxm16 = PXM_copy(E9pxm16);
      mutex_unlock(&Fpixmap_lock);
      mwpaint2();

      cc = Fww * Fhh * sizeof(int);                                        //  clear group arrays
      memset(pixgroup,0,cc);
      cc = maxgroups * sizeof(int);
      memset(groupcount,0,cc);
      memset(edgecount,0,cc);
      cc = maxgroups * sizeof(double);
      memset(groupbright,0,cc);
      memset(edgebright,0,cc);
      cc = maxgroups * sizeof(sgroupspann);
      memset(groupspann,0,cc);

      group = 0;
      
      for (py = 0; py < Fhh; py++)                                         //  loop all pixels
      for (px = 0; px < Fww; px++)
      {
         ii = py * Fww + px;
         if (Factivearea && ! sa_pixisin[ii]) continue;                    //  not in active area
         if (pixgroup[ii]) continue;                                       //  already assigned to a group

         pix3 = PXMpix(E3pxm16,px,py);                                     //  get pixel brightness
         gbright = ff * pixbright(pix3);                                   //  0 to 1.0 = white
         if (gbright > brightness) continue;                               //  ignore bright pixel

         if (group == maxgroups-1) break;                                  //  too many groups, make no more

         pixgroup[ii] = ++group;                                           //  assign next group
         groupcount[group] = 1;
         groupbright[group] = gbright;

         pixstack[0].px = px;                                              //  put pixel into stack with
         pixstack[0].py = py;                                              //    direction = ahead
         pixstack[0].direc = 0;
         Nstack = 1;

         while (Nstack)
         {
            kk = Nstack - 1;                                               //  get last pixel in stack
            px = pixstack[kk].px;
            py = pixstack[kk].py;
            direc = pixstack[kk].direc;                                    //  next search direction
            
            if (direc == 'x') {
               Nstack--;                                                   //  none left
               continue;
            }

            if (Nstack > 1) {
               ii = Nstack - 2;                                            //  get prior pixel in stack
               ppx = pixstack[ii].px;
               ppy = pixstack[ii].py;
            }
            else {
               ppx = px - 1;                                               //  if only one, assume prior = left
               ppy = py;
            }
            
            dx = px - ppx;                                                 //  vector from prior to this pixel
            dy = py - ppy;
            
            switch (direc) 
            {
               case 0:
                  npx = px + dx;
                  npy = py + dy;
                  pixstack[kk].direc = 1;
                  break;

               case 1:
                  npx = px + dy;
                  npy = py + dx;
                  pixstack[kk].direc = 3;
                  break;
            
               case 2:
                  npx = px - dx;                                           //  back to prior pixel
                  npy = py - dy;                                           //  (this path never taken)
                  zappcrash("stack search bug");
                  break;
            
               case 3:
                  npx = px - dy;
                  npy = py - dx;
                  pixstack[kk].direc = 4;
                  break;
            
               case 4:
                  npx = px - dx;
                  npy = py + dy;
                  pixstack[kk].direc = 5;
                  break;
            
               case 5:
                  npx = px - dy;
                  npy = py + dx;
                  pixstack[kk].direc = 6;
                  break;
            
               case 6:
                  npx = px + dx;
                  npy = py - dy;
                  pixstack[kk].direc = 7;
                  break;
            
               case 7:
                  npx = px + dy;
                  npy = py - dx;
                  pixstack[kk].direc = 'x';
                  break;
               
               default:
                  npx = npy = 0;
                  zappcrash("stack search bug");
            }

            if (npx < 0 || npx > Fww-1) continue;                          //  pixel off the edge
            if (npy < 0 || npy > Fhh-1) continue;
            
            ii = npy * Fww + npx;
            if (pixgroup[ii]) continue;                                    //  pixel already assigned
            if (Factivearea && ! sa_pixisin[ii]) continue;                 //  pixel outside area

            pix3 = PXMpix(E3pxm16,npx,npy);                                //  pixel brightness
            pbright = ff * pixbright(pix3);
            if (pbright > brightness) continue;                            //  brighter than limit

            pixgroup[ii] = group;                                          //  assign pixel to group
            ++groupcount[group];                                           //  count pixels in group
            groupbright[group] += pbright;                                 //  sum brightness for group

            kk = Nstack++;                                                 //  put pixel into stack
            pixstack[kk].px = npx;
            pixstack[kk].py = npy;
            pixstack[kk].direc = 0;                                        //  search direction
         }
      }
      
      Ngroups = group;                                                     //  group numbers are 1-Ngroups
      Nremoved = 0;
      
      for (py = 0; py < Fhh; py++)                                         //  loop all pixels
      for (px = 0; px < Fww; px++)
      {
         ii = py * Fww + px;
         group = pixgroup[ii];
         if (! group) continue;
         if (groupspann[group].px1 == 0) {                                 //  first pixel found in this group
            groupspann[group].px1 = px;                                    //  group px1/py1 = this pixel
            groupspann[group].py1 = py;
            continue;
         }
         xspann = groupspann[group].px1 - px;                              //  spann from group px1/py1 to this pixel
         yspann = groupspann[group].py1 - py;
         spann2 = xspann * xspann + yspann * yspann;
         if (spann2 > groupspann[group].spann2) {   
            groupspann[group].spann2 = spann2;                             //  if greater, group px2/py2 = this pixel
            groupspann[group].px2 = px;
            groupspann[group].py2 = py;
         }
      }

      for (py = 0; py < Fhh; py++)                                         //  loop all pixels
      for (px = 0; px < Fww; px++)
      {
         ii = py * Fww + px;
         group = pixgroup[ii];
         if (! group) continue;
         if (groupspann[group].spann2 > spotspann2) continue;
         xspann = groupspann[group].px2 - px;                              //  spann from this pixel to group px2/py2
         yspann = groupspann[group].py2 - py;
         spann2 = xspann * xspann + yspann * yspann;
         if (spann2 > groupspann[group].spann2) {   
            groupspann[group].spann2 = spann2;                             //  if greater, group px1/py1 = this pixel
            groupspann[group].px1 = px;
            groupspann[group].py1 = py;
         }
      }

      for (py = 0; py < Fhh; py++)                                         //  loop all pixels
      for (px = 0; px < Fww; px++)
      {
         ii = py * Fww + px;                                               //  eliminate group if spann > limit
         group = pixgroup[ii];
         if (! group) continue;
         if (! groupcount[group]) pixgroup[ii] = 0;
         else if (groupspann[group].spann2 > spotspann2) {
            pixgroup[ii] = 0;
            groupcount[group] = 0;
            Nremoved++;
         }
      }
      
      for (py = 1; py < Fhh-1; py++)                                       //  loop all pixels except image edges
      for (px = 1; px < Fww-1; px++)
      {
         ii = py * Fww + px;
         group = pixgroup[ii];
         if (group) continue;                                              //  find pixels bordering group pixels
         pix3 = PXMpix(E3pxm16,px,py);
         pbright = ff * pixbright(pix3);
         
         group = pixgroup[ii-Fww-1];
         if (group) {
            ++edgecount[group];                                            //  accumulate pixel count and
            edgebright[group] += pbright;                                  //      bordering the groups
         }

         group = pixgroup[ii-Fww];
         if (group) {
            ++edgecount[group];
            edgebright[group] += pbright;
         }

         group = pixgroup[ii-Fww+1];
         if (group) {
            ++edgecount[group];
            edgebright[group] += pbright;
         }

         group = pixgroup[ii-1];
         if (group) {
            ++edgecount[group];
            edgebright[group] += pbright;
         }

         group = pixgroup[ii+1];
         if (group) {
            ++edgecount[group];
            edgebright[group] += pbright;
         }

         group = pixgroup[ii+Fww-1];
         if (group) {
            ++edgecount[group];
            edgebright[group] += pbright;
         }

         group = pixgroup[ii+Fww];
         if (group) {
            ++edgecount[group];
            edgebright[group] += pbright;
         }

         group = pixgroup[ii+Fww+1];
         if (group) {
            ++edgecount[group];
            edgebright[group] += pbright;
         }
      }

      for (group = 1; group <= Ngroups; group++)                           //  compute group pixel and edge pixel
      {                                                                    //    mean brightness
         if (groupcount[group] && edgecount[group]) {
            edgebright[group] = edgebright[group] / edgecount[group];
            groupbright[group] = groupbright[group] / groupcount[group];
            pcontrast = edgebright[group] - groupbright[group];            //  edge - group contrast
            if (pcontrast < contrast) {
               groupcount[group] = 0;
               Nremoved++;
            }
         }
      }

      for (py = 0; py < Fhh; py++)                                         //  loop all pixels
      for (px = 0; px < Fww; px++)
      {
         ii = py * Fww + px;                                               //  eliminate group if low contrast
         group = pixgroup[ii];
         if (! group) continue;
         if (! groupcount[group]) pixgroup[ii] = 0;
      }

      for (py = 0; py < Fhh; py++)                                         //  loop all pixels
      for (px = 0; px < Fww; px++)
      {
         ii = py * Fww + px;
         if (! pixgroup[ii]) continue;                                     //  not a dust pixel
         pix3 = PXMpix(E3pxm16,px,py);                                     //  paint it red
         pix3[0] = 65535;
         pix3[1] = pix3[2] = 0;
      }

      Fred = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


//  erase the selected dust areas

void dust_erase()
{
   using namespace dust_names;
   
   int         cc, ii, px, py, rgb, inc;
   int         qx, qy, dx, dy, npx, npy;
   int         sx, sy, tx, ty;
   int         rad, dist, dist2, mindist2;
   double      slope, f1, f2;
   uint16      *pix1, *pix3;
   char        *pmap;

   Ffuncbusy++;

   mutex_lock(&Fpixmap_lock);                                              //  image 3 = copy of image 9
   PXM_free(E3pxm16);
   E3pxm16 = PXM_copy(E9pxm16);
   mutex_unlock(&Fpixmap_lock);

   cc = Fww * Fhh;                                                         //  allocate pixel done map
   pmap = (char *) zmalloc(cc,"dust.pmap");
   memset(pmap,0,cc);

   for (py = 0; py < Fhh; py++)                                            //  loop all pixels
   for (px = 0; px < Fww; px++)
   {
      ii = py * Fww + px;
      if (! pixgroup[ii]) continue;                                        //  not a dust pixel
      if (pmap[ii]) continue;                                              //  skip pixels already done
      
      mindist2 = 999999;
      npx = npy = 0;

      for (rad = 1; rad < 10; rad++)                                       //  find nearest edge (10 pixel limit)
      {
         for (qx = px-rad; qx <= px+rad; qx++)                             //  search within rad
         for (qy = py-rad; qy <= py+rad; qy++)
         {
            if (qx != px-rad && qx != px+rad &&                            //  exclude within rad-1 (already searched)
                qy != py-rad && qy != py+rad) continue;
            if (qx < 0 || qx >= Fww) continue;                             //  off image edge
            if (qy < 0 || qy >= Fhh) continue;
            ii = qy * Fww + qx;
            if (pixgroup[ii]) continue;                                    //  within dust area

            dx = (px - qx) * (px - qx);                                    //  found pixel
            dy = (py - qy) * (py - qy);
            dist2 = dx + dy;                                               //  distance**2
            if (dist2 < mindist2) {
               mindist2 = dist2;
               npx = qx;                                                   //  save nearest pixel found
               npy = qy;
            }
         }

         if (rad * rad >= mindist2) break;                                 //  can quit now
      }

      if (! npx && ! npy) continue;                                        //  should not happen

      qx = npx;                                                            //  nearest edge pixel
      qy = npy;
      dist = sqrt(mindist2);                                               //  distance

      if (abs(qy - py) > abs(qx - px)) {                                   //  qx/qy = near edge from px/py
         slope = 1.0 * (qx - px) / (qy - py);
         if (qy > py) inc = 1;
         else inc = -1;
         for (sy = py; sy != qy+inc; sy += inc)                            //  sx/sy = line from px/py to qx/qy
         {
            sx = px + slope * (sy - py);
            ii = sy * Fww + sx;
            pmap[ii] = 1;
            tx = qx + (sx - px) + slope * inc;                             //  tx/ty = parallel line from qx/qy 
            ty = qy + (sy - py) + inc;
            if (tx < 0 || tx > Fww-1) break;
            if (ty < 0 || ty > Fhh-1) break;
            pix1 = PXMpix(E3pxm16,tx,ty);                                  //  copy pixel from tx/ty to sx/sy
            pix3 = PXMpix(E3pxm16,sx,sy);
            pix3[0] = pix1[0];
            pix3[1] = pix1[1];
            pix3[2] = pix1[2];
         }
      }

      else {
         slope = 1.0 * (qy - py) / (qx - px);
         if (qx > px) inc = 1;
         else inc = -1;
         for (sx = px; sx != qx+inc; sx += inc) 
         {
            sy = py + slope * (sx - px);
            ii = sy * Fww + sx;
            pmap[ii] = 1;
            tx = qx + (sx - px) + inc;
            ty = qy + (sy - py) + slope * inc;
            if (tx < 0 || tx > Fww-1) break;
            if (ty < 0 || ty > Fhh-1) break;
            pix1 = PXMpix(E3pxm16,tx,ty);
            pix3 = PXMpix(E3pxm16,sx,sy);
            pix3[0] = pix1[0];
            pix3[1] = pix1[1];
            pix3[2] = pix1[2];
         }
      }
   }

   zfree(pmap);

   if (Factivearea)                                                        //  area edge blending
   {
      for (ii = 0; ii < Fww * Fhh; ii++)                                   //  find pixels in select area
      {
         dist = sa_pixisin[ii];
         if (! dist || dist >= sa_blend) continue;

         py = ii / Fww;
         px = ii - py * Fww;
         pix1 = PXMpix(E1pxm16,px,py);                                     //  input pixel, unchanged image
         pix3 = PXMpix(E3pxm16,px,py);                                     //  output pixel, changed image

         f2 = 1.0 * dist / sa_blend;                                       //  changes over distance sa_blend
         f1 = 1.0 - f2;

         for (rgb = 0; rgb < 3; rgb++)                                     //  blend pixels
            pix3[rgb] = int(f1 * pix1[rgb] + f2 * pix3[rgb]);
      }
   }

   Fred = 0;
   Ffuncbusy--;
   mwpaint2();                                                             //  update window
   return;
}


/**************************************************************************/

//  trim image - use mouse to select image region to retain

int         trimx1, trimy1, trimx2, trimy2;                                //  current trim rectangle
int         trimpx1, trimpy1, trimpx2, trimpy2;                            //  prior trim rectangle
double      trimR;                                                         //  trim ratio, width/height

void  trim_mousefunc();                                                    //  edit trim margins with mouse
void  trim_dialog();
void  trim_trim(int mode);                                                 //  show trim area, trim image


void m_trim(GtkWidget *, cchar *)
{
   zfuncs::F1_help_topic = "trim_image";                                   //  v.10.8

   if (! edit_setup("trim",0,0)) return;                                   //  setup edit
   
   if (! trimbuttons[0] || strEqu(trimbuttons[0],"undefined")) {
      trimbuttons[0] = strdupz("1:1",8);                                   //  default trim buttons   v.10.10.2
      trimbuttons[1] = strdupz("2:1",8);
      trimbuttons[2] = strdupz("3:2",8);
      trimbuttons[3] = strdupz("4:3",8);
      trimbuttons[4] = strdupz("16:9",8);
      trimbuttons[5] = strdupz("gold",8);

      trimratios[0] = strdupz("1:1",8);                                    //  default trim ratios    v.10.10.2
      trimratios[1] = strdupz("2:1",8);
      trimratios[2] = strdupz("3:2",8);
      trimratios[3] = strdupz("4:3",8);
      trimratios[4] = strdupz("16:9",8);
      trimratios[5] = strdupz("1.618:1",8);                                //  fix inverted ratio     v.10.10.3
   }

   if (trimsize[0] < 0.95 * iww && trimsize[1] < 0.95 * ihh) {             //  use last trim size if within limits
      trimx1 = Iorgx + 0.5 * (iww - trimsize[0]);
      trimx2 = trimx1 + trimsize[0];
      trimy1 = Iorgy + 0.5 * (ihh - trimsize[1]);
      trimy2 = trimy1 + trimsize[1];
   }
   else {                                                                  //  else use 90% current dimensions
      trimx1 = Iorgx + 0.05 * iww;
      trimx2 = Iorgx + 0.95 * iww;
      trimy1 = Iorgy + 0.05 * ihh;
      trimy2 = Iorgy + 0.95 * ihh;
   }
   
   trimpx1 = Iorgx;                                                        //  prior trim rectangle
   trimpx2 = Iorgx + iww;                                                  //    = 100% of image
   trimpy1 = Iorgy;
   trimpy2 = Iorgy + ihh;

   trimsize[0] = (trimx2 - trimx1);
   trimsize[1] = (trimy2 - trimy1);
   trimR = 1.0 * trimsize[0] / trimsize[1];

   trim_trim(0);                                                           //  show trim area in image
   takeMouse(0,trim_mousefunc,dragcursor);                                 //  connect mouse function       v.11.03
   trim_dialog();                                                          //  start dialog

   return;
}


//  dialog function is called from two places

void trim_dialog()
{
   int    trim_dialog_event(zdialog *zd, cchar *event);

   cchar       *trim_message = ZTX("Drag middle to move, drag corners to resize.");
   char        text[20];
   int         ii;

/**
      Drag middle to move, drag corners to resize

      width [___]  height [___]  ratio [___]                               //  width/height inputs    v.11.05
      [1:1] [2:1] [3:2] [4:3] [16:9] [gold] [invert]
      [x] Lock Ratio  [x] my mouse

                     [customize] [Done] [Cancel]
**/

   zdedit = zdialog_new(ZTX("Trim Image"),mWin,ZTX("customize"),Bdone,Bcancel,null);

   zdialog_add_widget(zdedit,"label","lab1","dialog",trim_message,"space=8");

   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=4");
   zdialog_add_widget(zdedit,"label","labW","hb1",ZTX("width"),"space=3");
   zdialog_add_widget(zdedit,"spin","width","hb1","20|9999|1|1000");
   zdialog_add_widget(zdedit,"label","space","hb1",0,"space=5");
   zdialog_add_widget(zdedit,"label","labH","hb1",ZTX("height"),"space=3");
   zdialog_add_widget(zdedit,"spin","height","hb1","20|9999|1|600");
   zdialog_add_widget(zdedit,"label","space","hb1",0,"space=5");
   zdialog_add_widget(zdedit,"label","labR","hb1",ZTX("ratio"),"space=3");
   zdialog_add_widget(zdedit,"label","ratio","hb1","1.67   ");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=4");           //  ratio buttons
   zdialog_add_widget(zdedit,"hbox","hb3","dialog",0,"space=4");
   zdialog_add_widget(zdedit,"check","lock","hb3",ZTX("Lock Ratio"),"space=3");
   zdialog_add_widget(zdedit,"check","mymouse","hb3",BmyMouse,"space=8");

   for (ii = 0; ii < 6; ii++)                                              //  6 custom buttons    v.10.10.2
      zdialog_add_widget(zdedit,"button",trimbuttons[ii],"hb2",trimbuttons[ii]);

   zdialog_add_widget(zdedit,"button","invert","hb2",ZTX("invert"));       //  [invert] button

   zdialog_stuff(zdedit,"width",trimsize[0]);                              //  stuff width, height, ratio   v.11.05
   zdialog_stuff(zdedit,"height",trimsize[1]);
   sprintf(text,"%.2f  ",trimR);
   zdialog_stuff(zdedit,"ratio",text);
   zdialog_stuff(zdedit,"mymouse",1);                                      //  v.11.05

   zdialog_run(zdedit,trim_dialog_event);                                  //  run dialog
   return;
}


//  dialog event and completion callback function

int trim_dialog_event(zdialog *zd, cchar *event)                           //  overhauled    v.11.05
{
   void  trim_customize();

   static int  flip = 0;
   int         width, height, delta;
   int         ii, rlock, mymouse;
   double      r1, r2, ratio = 0;
   char        text[20];
   cchar       *pp;
   
   if (zd->zstat)                                                          //  dialog complete
   {
      freeMouse();                                                         //  disconnect mouse function       v.10.12
      
      paint_toplines(2);                                                   //  erase rectangle outline         v.11.05

      if (zd->zstat == 1) {                                                //  customize buttons
         zdialog_free(zdedit);                                             //  kill trim dialog
         trim_customize();                                                 //  do customize dialog
         trim_trim(0);                                                     //  show trim area in image         v.11.05
         takeMouse(0,trim_mousefunc,dragcursor);                           //  connect mouse function          v.11.05
         trim_dialog();                                                    //  restart dialog
         return 0;
      }

      if (zd->zstat != 2) {                                                //  cancel
         edit_cancel();
         return 0;
      }

      trimsize[0] = trimx2 - trimx1;                                       //  apply
      trimsize[1] = trimy2 - trimy1;

      trim_trim(1);                                                        //  do trim on image
      Fzoom = 0;                                                           //  v.11.01
      edit_done();
      return 0;
   }
   
   if (strEqu(event,"mymouse")) {                                          //  my mouse toggle              v.11.05
      zdialog_fetch(zd,"mymouse",mymouse);
      if (mymouse)
         takeMouse(0,trim_mousefunc,dragcursor);                           //  connect mouse function
      else freeMouse();                                                    //  disconnect mouse
      return 0;
   }
   
   if (strstr("width height",event))                                       //  get direct width/height inputs
   {                                                                       //                               v.11.05
      zdialog_fetch(zd,"width",width);
      zdialog_fetch(zd,"height",height);
      zdialog_fetch(zd,"lock",rlock);                                      //  lock ratio on/off

      
      if (strEqu(event,"width")) {                                         //  v.11.05
         if (width > iww) width = iww;
         if (rlock) {                                                      //  ratio locked
            height = width / trimR + 0.5;                                  //  try to keep ratio
            if (height > ihh) height = ihh;
            zdialog_stuff(zd,"height",height);
         }
      }

      if (strEqu(event,"height")) {
         if (height > ihh) height = ihh;
         if (rlock) {
            width = height * trimR + 0.5;
            if (width > iww) width = iww;
            zdialog_stuff(zd,"width",width);
         }
      }

      flip = 1 - flip;                                                     //  alternates 0, 1, 0, 1 ...

      delta = width - trimsize[0];

      if (delta > 0) {                                                     //  increase width
         trimx1 = trimx1 - delta / 2;                                      //  left and right sides equally
         trimx2 = trimx2 + delta / 2;
         if (delta % 2) {                                                  //  if increase is odd
            trimx1 = trimx1 - flip;                                        //    add 1 alternatively to each side
            trimx2 = trimx2 + 1 - flip;
         }
      }
      
      if (delta < 0) {                                                     //  decrease width
         trimx1 = trimx1 - delta / 2;
         trimx2 = trimx2 + delta / 2;
         if (delta % 2) {
            trimx1 = trimx1 + flip;
            trimx2 = trimx2 - 1 + flip;
         }
      }

      delta = height - trimsize[1];

      if (delta > 0) {                                                     //  increase width
         trimy1 = trimy1 - delta / 2;                                      //  left and right sides equally
         trimy2 = trimy2 + delta / 2;
         if (delta % 2) {                                                  //  if increase is odd
            trimy1 = trimy1 - flip;                                        //    add 1 alternatively to each side
            trimy2 = trimy2 + 1 - flip;
         }
      }
      
      if (delta < 0) {                                                     //  decrease width
         trimy1 = trimy1 - delta / 2;
         trimy2 = trimy2 + delta / 2;
         if (delta % 2) {
            trimy1 = trimy1 + flip;
            trimy2 = trimy2 - 1 + flip;
         }
      }

      if (trimx1 < Iorgx) trimx1 = Iorgx;
      if (trimx2 > Iorgx + iww) trimx2 = Iorgx + iww;
      if (trimy1 < Iorgy) trimy1 = Iorgy;
      if (trimy2 > Iorgy + ihh) trimy2 = Iorgy + ihh;
      
      width = trimx2 - trimx1;                                             //  new width and height
      height = trimy2 - trimy1;
      
      zdialog_stuff(zd,"width",width);                                     //  update dialog values
      zdialog_stuff(zd,"height",height);

      trimsize[0] = width;                                                 //  new rectangle dimensions
      trimsize[1] = height;

      if (! rlock)                                                         //  set new ratio if not locked
         trimR = 1.0 * trimsize[0] / trimsize[1];

      sprintf(text,"%.2f  ",trimR);                                        //  stuff new ratio
      zdialog_stuff(zd,"ratio",text);

      trim_trim(0);                                                        //  show trim area in image
      return 0;
   }

   for (ii = 0; ii < 6; ii++)                                              //  trim ratio buttons
      if (strEqu(event,trimbuttons[ii])) break;
   if (ii < 6) {
      r1 = r2 = ratio = 0;
      pp = strField(trimratios[ii],':',1);
      if (pp) r1 = atof(pp);
      pp = strField(trimratios[ii],':',2);
      if (pp) r2 = atof(pp);
      if (r1 > 0 && r2 > 0) ratio = r1/r2;
      if (ratio < 0.1 || ratio > 10) ratio = 1.0;
      if (! ratio) return 0;
      zdialog_stuff(zd,"lock",1);                                          //  assume lock is wanted
      trimR = ratio;
   }
   
   if (strEqu(event,"invert"))                                             //  invert ratio button
      if (trimR) ratio = 1.0 / trimR;
   
   if (ratio)                                                              //  ratio was changed
   {
      trimR = ratio;

      if (trimx2 - trimx1 > trimy2 - trimy1)
         trimy2 = trimy1 + (trimx2 - trimx1) / trimR;                      //  adjust smaller dimension
      else 
         trimx2 = trimx1 + (trimy2 - trimy1) * trimR;

      if (trimx2 > Iorgx + iww) {                                          //  if off the right edge,
         trimx2 = Iorgx + iww;                                             //  adjust height
         trimy2 = trimy1 + (trimx2 - trimx1) / trimR;
      }

      if (trimy2 > Iorgy + ihh) {                                          //  if off the bottom edge,
         trimy2 = Iorgy + ihh;                                             //  adjust width
         trimx2 = trimx1 + (trimy2 - trimy1) * trimR;
      }

      trimsize[0] = trimx2 - trimx1;                                       //  new rectangle dimensions
      trimsize[1] = trimy2 - trimy1;

      zdialog_stuff(zd,"width",trimsize[0]);                               //  stuff width, height, ratio   v.11.05
      zdialog_stuff(zd,"height",trimsize[1]);
      sprintf(text,"%.2f  ",trimR);
      zdialog_stuff(zd,"ratio",text);

      trim_trim(0);                                                        //  show trim area in image
      return 0;
   }
   
   return 0;
}


//  trim mouse function

void trim_mousefunc()
{
   int         mpx, mpy, xdrag, ydrag, rlock;
   int         corner, chop, moveall = 0;
   int         dx, dy, dd, d1, d2, d3, d4;
   char        text[20];
   double      drr;
   
   if (LMclick || Mxdrag || Mydrag)                                        //  mouse click or drag
   {
      if (LMclick) {
         mpx = Mxclick;                                                    //  click
         mpy = Myclick;
         xdrag = ydrag = 0;
         LMclick = 0;
      }
      else {
         mpx = Mxdrag;                                                     //  drag
         mpy = Mydrag;
         xdrag = Mxdrag - Mxdown;
         ydrag = Mydrag - Mydown;
         Mxdown = Mxdrag;                                                  //  reset drag origin
         Mydown = Mydrag;
      }
      
      if (Mxdrag || Mydrag) {
         moveall = 1;
         dd = 0.1 * (trimx2 - trimx1);                                     //  test if mouse is in the broad
         if (mpx < trimx1 + dd) moveall = 0;                               //    middle of the rectangle
         if (mpx > trimx2 - dd) moveall = 0;
         dd = 0.1 * (trimy2 - trimy1);
         if (mpy < trimy1 + dd) moveall = 0;
         if (mpy > trimy2 - dd) moveall = 0;
      }

      if (moveall) {                                                       //  yes, move the whole rectangle
         trimx1 += xdrag;
         trimx2 += xdrag;
         trimy1 += ydrag;
         trimy2 += ydrag;
         corner = 0;
      }

      else {                                                               //  no, find closest corner
         dx = mpx - trimx1;
         dy = mpy - trimy1;
         d1 = sqrt(dx*dx + dy*dy);                                         //  distance from NW corner
         
         dx = mpx - trimx2;
         dy = mpy - trimy1;
         d2 = sqrt(dx*dx + dy*dy);
         
         dx = mpx - trimx2;
         dy = mpy - trimy2;
         d3 = sqrt(dx*dx + dy*dy);
         
         dx = mpx - trimx1;
         dy = mpy - trimy2;
         d4 = sqrt(dx*dx + dy*dy);
         
         corner = 1;                                                       //  NW
         dd = d1;
         if (d2 < dd) { corner = 2; dd = d2; }                             //  NE
         if (d3 < dd) { corner = 3; dd = d3; }                             //  SE
         if (d4 < dd) { corner = 4; dd = d4; }                             //  SW
         
         if (corner == 1) { trimx1 = mpx; trimy1 = mpy; }                  //  move this corner to mouse
         if (corner == 2) { trimx2 = mpx; trimy1 = mpy; }
         if (corner == 3) { trimx2 = mpx; trimy2 = mpy; }
         if (corner == 4) { trimx1 = mpx; trimy2 = mpy; }
      }
      
      if (trimx1 < Iorgx) trimx1 = Iorgx;                                  //  keep within visible area     v.11.05
      if (trimx2 > Iorgx + iww) trimx2 = Iorgx + iww;
      if (trimy1 < Iorgy) trimy1 = Iorgy;
      if (trimy2 > Iorgy + ihh) trimy2 = Iorgy + ihh;

      if (trimx1 > trimx2-10) trimx1 = trimx2-10;                          //  sanity limits
      if (trimy1 > trimy2-10) trimy1 = trimy2-10;

      zdialog_fetch(zdedit,"lock",rlock);                                  //  w/h ratio locked
      if (rlock && corner) {
         if (corner < 3)                                                   //  bugfix   v.10.3.1
            trimy2 = trimy1 + 1.0 * (trimx2 - trimx1) / trimR;
         else
            trimy1 = trimy2 - 1.0 * (trimx2 - trimx1) / trimR;
      }

      chop = 0;
      if (trimx1 < Iorgx) {                                                //  look for off the edge   v.10.1
         trimx1 = Iorgx; 
         chop = 1; 
      }

      if (trimx2 > Iorgx + iww) {                                          //  after corner move    v.10.3.1
         trimx2 = Iorgx + iww;
         chop = 2; 
      }

      if (trimy1 < Iorgy) { 
         trimy1 = Iorgy; 
         chop = 3; 
      }

      if (trimy2 > Iorgy + ihh) { 
         trimy2 = Iorgy + ihh;
         chop = 4; 
      }

      if (rlock && chop) {                                                 //  keep ratio if off edge   v.10.1
         if (chop < 3)
            trimy2 = trimy1 + 1.0 * (trimx2 - trimx1) / trimR;
         else
            trimx2 = trimx1 + 1.0 * (trimy2 - trimy1) * trimR;
      }

      if (trimx1 > trimx2-10) trimx1 = trimx2-10;                          //  sanity limits
      if (trimy1 > trimy2-10) trimy1 = trimy2-10;

      if (trimx1 < Iorgx) trimx1 = Iorgx;                                  //  keep within visible area     v.11.05
      if (trimx2 > Iorgx + iww) trimx2 = Iorgx + iww;
      if (trimy1 < Iorgy) trimy1 = Iorgy;
      if (trimy2 > Iorgy + ihh) trimy2 = Iorgy + ihh;

      trimsize[0] = trimx2 - trimx1;                                       //  new rectangle dimensions
      trimsize[1] = trimy2 - trimy1;

      drr = 1.0 * trimsize[0] / trimsize[1];                               //  new w/h ratio
      if (! rlock) trimR = drr;

      zdialog_stuff(zdedit,"width",trimsize[0]);                           //  stuff width, height, ratio   v.11.05
      zdialog_stuff(zdedit,"height",trimsize[1]);
      sprintf(text,"%.2f  ",trimR);
      zdialog_stuff(zdedit,"ratio",text);
      
      trim_trim(0);                                                        //  show trim area in image
   }

   return;
}


//  darken image pixels outside of current trim margins
//  messy logic: update pixmaps only for changed pixels to increase speed

void trim_trim(int mode)
{
   int      ox1, oy1, ox2, oy2;                                            //  outer trim rectangle
   int      nx1, ny1, nx2, ny2;                                            //  inner trim rectangle
   int      px, py, px1, py1, px2, py2;
   uint16   *pix1, *pix3;
 
   if (mode == 1)                                                          //  do the final trim
   {
      mutex_lock(&Fpixmap_lock);
      PXM_free(E3pxm16);
      E3pxm16 = PXM_make(trimsize[0],trimsize[1],16);                      //  new pixmap with requested size
      E3ww = trimsize[0];
      E3hh = trimsize[1];
      
      for (py1 = trimy1; py1 < trimy2; py1++)                              //  copy pixels
      for (px1 = trimx1; px1 < trimx2; px1++)
      {
         px2 = px1 - trimx1;
         py2 = py1 - trimy1;
         pix1 = PXMpix(E1pxm16,px1,py1);
         pix3 = PXMpix(E3pxm16,px2,py2);
         pix3[0] = pix1[0];
         pix3[1] = pix1[1];
         pix3[2] = pix1[2];
      }

      Fmodified = 1;
      mutex_unlock(&Fpixmap_lock);
      mwpaint2();                                                          //  update window
      return;
   }

   if (trimx1 < Iorgx) trimx1 = Iorgx;                                     //  keep within visible area     v.11.05
   if (trimx2 > Iorgx + iww) trimx2 = Iorgx + iww;
   if (trimy1 < Iorgy) trimy1 = Iorgy;
   if (trimy2 > Iorgy + ihh) trimy2 = Iorgy + ihh;

   if (trimpx1 < trimx1) ox1 = trimpx1;                                    //  outer rectangle
   else ox1 = trimx1;
   if (trimpx2 > trimx2) ox2 = trimpx2;
   else ox2 = trimx2;
   if (trimpy1 < trimy1) oy1 = trimpy1;
   else oy1 = trimy1;
   if (trimpy2 > trimy2) oy2 = trimpy2;
   else oy2 = trimy2;

   if (trimpx1 > trimx1) nx1 = trimpx1;                                    //  inner rectangle
   else nx1 = trimx1;
   if (trimpx2 < trimx2) nx2 = trimpx2;
   else nx2 = trimx2;
   if (trimpy1 > trimy1) ny1 = trimpy1;
   else ny1 = trimy1;
   if (trimpy2 < trimy2) ny2 = trimpy2;
   else ny2 = trimy2;
   
   trimpx1 = trimx1;                                                       //  set prior trim rectangle 
   trimpx2 = trimx2;                                                       //    from current trim rectangle
   trimpy1 = trimy1;
   trimpy2 = trimy2;
   
   if (ox1 > 0) ox1--;
   if (ox2 < E3ww-1) ox2++;
   if (oy1 > 0) oy1--;
   if (oy2 < E3hh-1) oy2++;

   for (py = oy1; py < ny1; py++)                                          //  top band of pixels
   for (px = ox1; px < ox2; px++) 
   {
      pix1 = PXMpix(E1pxm16,px,py);
      pix3 = PXMpix(E3pxm16,px,py);
      if (px < trimx1 || px > trimx2 || py < trimy1 || py > trimy2) {
         pix3[0] = pix1[0] / 2;                                            //  outside trim margins
         pix3[1] = pix1[1] / 2;                                            //  50% brightness
         pix3[2] = pix1[2] / 2;
      }
      else {
         pix3[0] = pix1[0];                                                //  inside trim margins
         pix3[1] = pix1[1];                                                //  100% brightness
         pix3[2] = pix1[2];
      }
   }

   for (py = oy1; py < oy2; py++)                                          //  right band
   for (px = nx2; px < ox2; px++)
   {
      pix1 = PXMpix(E1pxm16,px,py);
      pix3 = PXMpix(E3pxm16,px,py);
      if (px < trimx1 || px > trimx2 || py < trimy1 || py > trimy2) {
         pix3[0] = pix1[0] / 2;
         pix3[1] = pix1[1] / 2;
         pix3[2] = pix1[2] / 2;
      }
      else {
         pix3[0] = pix1[0];
         pix3[1] = pix1[1];
         pix3[2] = pix1[2];
      }
   }

   for (py = ny2; py < oy2; py++)                                          //  bottom band
   for (px = ox1; px < ox2; px++)
   {
      pix1 = PXMpix(E1pxm16,px,py);
      pix3 = PXMpix(E3pxm16,px,py);
      if (px < trimx1 || px > trimx2 || py < trimy1 || py > trimy2) {
         pix3[0] = pix1[0] / 2;
         pix3[1] = pix1[1] / 2;
         pix3[2] = pix1[2] / 2;
      }
      else {
         pix3[0] = pix1[0];
         pix3[1] = pix1[1];
         pix3[2] = pix1[2];
      }
   }

   for (py = oy1; py < oy2; py++)                                          //  left band
   for (px = ox1; px < nx1; px++)
   {
      pix1 = PXMpix(E1pxm16,px,py);
      pix3 = PXMpix(E3pxm16,px,py);
      if (px < trimx1 || px > trimx2 || py < trimy1 || py > trimy2) {
         pix3[0] = pix1[0] / 2;
         pix3[1] = pix1[1] / 2;
         pix3[2] = pix1[2] / 2;
      }
      else {
         pix3[0] = pix1[0];
         pix3[1] = pix1[1];
         pix3[2] = pix1[2];
      }
   }
   
   mwpaint3(ox1,oy1,ox2-ox1+1,ny1-oy1+1);                                  //  4 updated rectangles
   mwpaint3(nx2,oy1,ox2-nx2+1,oy2-oy1+1);                                  //  ww+1, hh+1                v.11.05
   mwpaint3(ox1,ny2,ox2-ox1+1,oy2-ny2+1);
   mwpaint3(ox1,oy1,nx1-ox1+1,oy2-oy1+1);

   Ntoplines = 4;                                                          //  outline trim rectangle    v.11.05
   toplinex1[0] = trimx1;
   topliney1[0] = trimy1;
   toplinex2[0] = trimx2;
   topliney2[0] = trimy1;
   toplinex1[1] = trimx2;
   topliney1[1] = trimy1;
   toplinex2[1] = trimx2;
   topliney2[1] = trimy2;
   toplinex1[2] = trimx2;
   topliney1[2] = trimy2;
   toplinex2[2] = trimx1;
   topliney2[2] = trimy2;
   toplinex1[3] = trimx1;
   topliney1[3] = trimy2;
   toplinex2[3] = trimx1;
   topliney2[3] = trimy1;
   paint_toplines(1);

   return;
}


//  dialog to get custom trim button names and corresponding ratios

void trim_customize()
{
   char        text[20], blab[8], rlab[8];
   double      r1, r2, ratio;
   cchar       *pp;
   int         ii, zstat;
   
   zdedit = zdialog_new(ZTX("Trim Buttons"),mWin,Bdone,Bcancel,null);      //  start dialog
   zdialog_add_widget(zdedit,"hbox","hbb","dialog",0,"homog|space=5");
   zdialog_add_widget(zdedit,"hbox","hbr","dialog",0,"homog|space=5");
   
   strcpy(blab,"butt-0");
   strcpy(rlab,"ratio-0");
   
   for (ii = 0; ii < 6; ii++) 
   {
      blab[5] = '0' + ii;
      rlab[6] = '0' + ii;
      zdialog_add_widget(zdedit,"entry",blab,"hbb",trimbuttons[ii],"scc=6");
      zdialog_add_widget(zdedit,"entry",rlab,"hbr",trimratios[ii],"scc=6");
   }
   
   zdialog_run(zdedit);
   zstat = zdialog_wait(zdedit);

   if (zstat == 1)                                                         //  apply
   {
      for (ii = 0; ii < 6; ii++)                                           //  get custom button names
      {
         blab[5] = '0' + ii;
         zdialog_fetch(zdedit,blab,text,12);
         strTrim2(text);
         if (! *text) continue;
         zfree(trimbuttons[ii]);
         trimbuttons[ii] = strdupz(text);
      }

      for (ii = 0; ii < 6; ii++)                                           //  get custom ratios
      {
         rlab[6] = '0' + ii;
         zdialog_fetch(zdedit,rlab,text,12);
         strTrim2(text);
         r1 = r2 = ratio = 0;
         pp = strField(text,':',1);
         if (! pp) continue;
         r1 = atof(pp);
         pp = strField(text,':',2);
         if (! pp) continue;
         r2 = atof(pp);
         if (r1 > 0 && r2 > 0) ratio = r1/r2;
         if (ratio < 0.1 || ratio > 10) continue;
         zfree(trimratios[ii]);
         trimratios[ii] = strdupz(text);
      }
   }

   zdialog_free(zdedit);                                                   //  kill dialog
   return;
}


/**************************************************************************/

//  Resize (rescale) image
//
//  Output pixels are composites of input pixels, e.g. 2/3 size means 
//  that 3x3 input pixels are mapped into 2x2 output pixels, and an 
//  image size of 1000 x 600 becomes 667 x 400.

int      resize_ww0, resize_hh0;                                           //  original size
int      resize_ww1, resize_hh1;                                           //  new size


void m_resize(GtkWidget *, cchar *)
{
   int    resize_dialog_event(zdialog *zd, cchar *event);
   void * resize_thread(void *);

   cchar  *lockmess = ZTX("Lock aspect ratio");

   zfuncs::F1_help_topic = "resize";                                       //  v.10.8

   if (! edit_setup("resize",0,0)) return;                                 //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Resize Image"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"vbox","vb11","hb1",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb12","hb1",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb13","hb1",0,"homog|space=5");
   zdialog_add_widget(zdedit,"label","placeholder","vb11",0);              //            pixels     percent
   zdialog_add_widget(zdedit,"label","labw","vb11",Bwidth);                //    width   [______]   [______]
   zdialog_add_widget(zdedit,"label","labh","vb11",Bheight);               //    height  [______]   [______]
   zdialog_add_widget(zdedit,"label","labpix","vb12","pixels");            //
   zdialog_add_widget(zdedit,"spin","wpix","vb12","20|9999|1|0");          //    presets [2/3] [1/2] [1/3] [1/4] [Prev]
   zdialog_add_widget(zdedit,"spin","hpix","vb12","20|9999|1|0");          //
   zdialog_add_widget(zdedit,"label","labpct","vb13",Bpercent);            //    [_] lock aspect ratio
   zdialog_add_widget(zdedit,"spin","wpct","vb13","1|500|0.1|100");        //
   zdialog_add_widget(zdedit,"spin","hpct","vb13","1|500|0.1|100");        //                      [done] [cancel]
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","preset","hb2",Bpresets,"space=5");
   zdialog_add_widget(zdedit,"button","b 3/4","hb2","3/4");
   zdialog_add_widget(zdedit,"button","b 2/3","hb2","2/3");
   zdialog_add_widget(zdedit,"button","b 1/2","hb2","1/2");
   zdialog_add_widget(zdedit,"button","b 1/3","hb2","1/3");
   zdialog_add_widget(zdedit,"button","b 1/4","hb2","1/4");
   zdialog_add_widget(zdedit,"button","prev","hb2",ZTX("Prev"));
   zdialog_add_widget(zdedit,"hbox","hb3","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"check","lock","hb3",lockmess,"space=5");

   resize_ww0 = Fpxm16->ww;                                                //  original width, height
   resize_hh0 = Fpxm16->hh;

   zdialog_stuff(zdedit,"wpix",resize_ww0);
   zdialog_stuff(zdedit,"hpix",resize_hh0);
   zdialog_stuff(zdedit,"lock",1);

   start_thread(resize_thread,0);                                          //  start working thread   

   zdialog_run(zdedit,resize_dialog_event);                                //  run dialog
   zdialog_wait(zdedit);                                                   //  wait for completion

   return;
}


//  dialog event and completion callback function

int resize_dialog_event(zdialog *zd, cchar * event)
{
   int         lock;
   double      wpct1, hpct1;
   
   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat != 1) {                                                //  user cancel
         edit_cancel();
         return 0;
      }

      editresize[0] = resize_ww1;                                          //  remember size used     v.10.10
      editresize[1] = resize_hh1;

      Fzoom = 0;                                                           //  v.11.01
      edit_done();
      return 0;
   }

   zdialog_fetch(zd,"wpix",resize_ww1);                                    //  get all widget values
   zdialog_fetch(zd,"hpix",resize_hh1);
   zdialog_fetch(zd,"wpct",wpct1);
   zdialog_fetch(zd,"hpct",hpct1);
   zdialog_fetch(zd,"lock",lock);

   if (strEqu(event,"b 3/4")) {
      resize_ww1 = (3 * resize_ww0 + 3) / 4;
      resize_hh1 = (3 * resize_hh0 + 3) / 4;
   }

   if (strEqu(event,"b 2/3")) {
      resize_ww1 = (2 * resize_ww0 + 2) / 3;
      resize_hh1 = (2 * resize_hh0 + 2) / 3;
   }

   if (strEqu(event,"b 1/2")) {
      resize_ww1 = (resize_ww0 + 1) / 2;
      resize_hh1 = (resize_hh0 + 1) / 2;
   }

   if (strEqu(event,"b 1/3")) {
      resize_ww1 = (resize_ww0 + 2) / 3;
      resize_hh1 = (resize_hh0 + 2) / 3;
   }

   if (strEqu(event,"b 1/4")) {
      resize_ww1 = (resize_ww0 + 3) / 4;
      resize_hh1 = (resize_hh0 + 3) / 4;
   }

   if (strEqu(event,"prev")) {                                             //  use previous resize values   v.10.10
      resize_ww1 = editresize[0];
      resize_hh1 = editresize[1];
   }

   if (strEqu(event,"wpct"))                                               //  width % - set pixel width
      resize_ww1 = int(wpct1 / 100.0 * resize_ww0 + 0.5);

   if (strEqu(event,"hpct"))                                               //  height % - set pixel height
      resize_hh1 = int(hpct1 / 100.0 * resize_hh0 + 0.5);
   
   if (lock && event[0] == 'w')                                            //  preserve width/height ratio
      resize_hh1 = int(resize_ww1 * (1.0 * resize_hh0 / resize_ww0) + 0.5);
   if (lock && event[0] == 'h') 
      resize_ww1 = int(resize_hh1 * (1.0 * resize_ww0 / resize_hh0) + 0.5);
   
   hpct1 = 100.0 * resize_hh1 / resize_hh0;                                //  set percents to match pixels
   wpct1 = 100.0 * resize_ww1 / resize_ww0;
   
   zdialog_stuff(zd,"wpix",resize_ww1);                                    //  index all widget values
   zdialog_stuff(zd,"hpix",resize_hh1);
   zdialog_stuff(zd,"wpct",wpct1);
   zdialog_stuff(zd,"hpct",hpct1);

   signal_thread();                                                        //  do the update, don't wait for idle
   return 1;
}


//  do the resize job

void * resize_thread(void *)                                               //  v.10.12
{
   Fmodified = 0;
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for signal
      
      mutex_lock(&Fpixmap_lock);
      PXM_free(E3pxm16);
      E3pxm16 = PXM_rescale(Fpxm16,resize_ww1,resize_hh1);                 //  rescale the edit image
      E3ww = resize_ww1;
      E3hh = resize_hh1;
      Fmodified = 1;
      mutex_unlock(&Fpixmap_lock);
      mwpaint2();
   }

   return 0;
}


/**************************************************************************/

//  menu function - batch resize - resize many files at once
//  resized files are .jpg files

void m_batchresize(GtkWidget *, cchar *)                                   //  new v.10.8
{
   int   batchresize_dialog_event(zdialog *zd, cchar *event);

   zdialog     *zd;                                                        //  batch resize files dialog

   zfuncs::F1_help_topic = "batch_resize";                                 //  v.10.8

   if (mod_keep()) return;                                                 //  unsaved edits
   if (! menulock(1)) return;                                              //  lock menus

   //  [select files]  N files selected
   //  new max. width   [____]
   //  new max. height  [____]
   //  (o) replace originals
   //  (o) export to location [browse] [x] copy EXIF
   //  [___________________________________________]
   //
   //                   [resize]  [cancel]

   zd = zdialog_new(ZTX("Batch Resize"),mWin,Bproceed,Bcancel,null);
   zdialog_add_widget(zd,"hbox","hbf","dialog",0,"space=5");
   zdialog_add_widget(zd,"button","files","hbf",Bselectfiles,"space=5");
   zdialog_add_widget(zd,"label","fcount","hbf","0 files selected","space=10");
   zdialog_add_widget(zd,"hbox","hbwh","dialog","space=10");
   zdialog_add_widget(zd,"vbox","vbwh1","hbwh",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vbwh2","hbwh",0,"homog|space=5");
   zdialog_add_widget(zd,"label","labw","vbwh1",ZTX("new max. width"));
   zdialog_add_widget(zd,"label","labh","vbwh1",ZTX("new max. height"));
   zdialog_add_widget(zd,"entry","maxw","vbwh2","1000","scc=5");
   zdialog_add_widget(zd,"entry","maxh","vbwh2","700","scc=5");
   zdialog_add_widget(zd,"hbox","hbsep","dialog",0,"space=5");
   zdialog_add_widget(zd,"hbox","hbloc","dialog");
   zdialog_add_widget(zd,"vbox","vbloc1","hbloc",0,"homog");
   zdialog_add_widget(zd,"vbox","vbloc2","hbloc",0,"homog");
   zdialog_add_widget(zd,"radio","replace","vbloc1",ZTX("replace originals"));
   zdialog_add_widget(zd,"radio","export","vbloc1",ZTX("export to location"));
   zdialog_add_widget(zd,"label","space","vbloc2");
   zdialog_add_widget(zd,"hbox","hbloc2","vbloc2","space=5");
   zdialog_add_widget(zd,"button","browse","hbloc2",Bbrowse,"space=5");
   zdialog_add_widget(zd,"check","copyexif","hbloc2",ZTX("copy EXIF"),"space=5");
   zdialog_add_widget(zd,"hbox","hbloc3","dialog",0,"expand");
   zdialog_add_widget(zd,"entry","location","hbloc3",0,"expand|space=5");

   zdialog_stuff(zd,"replace",0);                                          //  defaults: export, no EXIF
   zdialog_stuff(zd,"export",1);
   zdialog_stuff(zd,"copyexif",0);
   zdialog_stuff(zd,"maxw",batchresize[0]);                                //  v.10.9
   zdialog_stuff(zd,"maxh",batchresize[1]);
   
   zdialog_run(zd,batchresize_dialog_event);                               //  run dialog
   zdialog_wait(zd);                                                       //  wait for completion

   menulock(0);
   return;
}


//  dialog event and completion callback function

int batchresize_dialog_event(zdialog *zd, cchar *event)
{
   static char    **flist = 0;
   char           countmess[32];
   int            Freplace, maxw, maxh, Fcopyexif, yn;
   int            ii, err, lcc, fcc, ww, hh;
   char           location[maxfcc], *ploc, *pfile, *pext;
   char           *oldfile, *newfile, *tempfile;
   double         scale, wscale, hscale;
   PXM            *pxmout;
   struct stat    statb;

   if (strEqu(event,"files"))                                              //  select images to resize
   {
      if (flist) {                                                         //  free prior list
         for (ii = 0; flist[ii]; ii++) 
            zfree(flist[ii]);
         zfree(flist);
      }
      
      flist = image_gallery_getfiles(0,mWin);                              //  get list of files to resize   v.10.9

      if (flist)                                                           //  count files selected
         for (ii = 0; flist[ii]; ii++);
      else ii = 0;

      sprintf(countmess,"%d files selected",ii);                           //  update dialog
      zdialog_stuff(zd,"fcount",countmess);
   }
   
   if (strEqu(event,"browse"))
   {
      ploc = zgetfile1(ZTX("Select directory"),"folder",curr_dirk);        //  get directory          v.10.9
      if (! ploc) return 0;
      zdialog_stuff(zd,"location",ploc);
      zfree(ploc);
      zdialog_stuff(zd,"export",1);
   }

   if (! zd->zstat) return 0;                                              //  dialog still busy

   if (zd->zstat != 1) goto cleanup;                                       //  dialog canceled
   
   if (! flist) {                                                          //  no files selected
      zd->zstat = 0;                                                       //  keep dialog active
      return 0;
   }

   zdialog_fetch(zd,"maxw",maxw);                                          //  new max width
   zdialog_fetch(zd,"maxh",maxh);                                          //  new max height
   zdialog_fetch(zd,"replace",Freplace);                                   //  replace originals y/n
   zdialog_fetch(zd,"copyexif",Fcopyexif);                                 //  copy EXIF y/n
   zdialog_fetch(zd,"location",location,maxfcc);                           //  output location (Freplace = 0)
   strTrim2(location,location);                                            //  trim leading and trailing blanks

   if (Freplace) {
      yn = zmessageYN(mWin,ZTX("replace original files? (max. %d x %d)"),maxw,maxh);
      if (! yn) {
         zd->zstat = 0;
         return 0;
      }
   }
   else {
      yn = zmessageYN(mWin,ZTX("copy files? (max. %d x %d) \n"
                               " to location %s"),maxw,maxh,location);
      if (! yn) {
         zd->zstat = 0;
         return 0;
      }
   }

   if (! Freplace) {
      err = stat(location,&statb);
      if (err || ! S_ISDIR(statb.st_mode)) {
         zmessageACK(mWin,ZTX("location is not a valid directory"));
         zd->zstat = 0;
         return 0;
      }
   }

   if (maxw < 20 || maxh < 20) {
      zmessageACK(mWin,ZTX("max. size %d x %d is not reasonable"),maxw,maxh);
      zd->zstat = 0;
      return 0;
   }
   
   write_popup_text("open","Resizing files",500,200,mWin);                 //  status monitor popup window

   lcc = strlen(location);
   if (! Freplace && location[lcc-1] == '/') lcc--;                        //  remove trailing '/'

   for (ii = 0; flist[ii]; ii++)                                           //  loop selected files
   {
      oldfile = flist[ii];

      if (Freplace)
         newfile = strdupz(oldfile,8,"batchresize.old");                   //  new file = old file
      else {
         pfile = strrchr(oldfile,'/');
         if (! pfile) continue;
         fcc = strlen(pfile);
         newfile = strdupz(location,fcc+9,"batchresize.new");
         strcpy(newfile+lcc,pfile);                                        //  new file at location
      }
      
      pfile = strrchr(newfile,'/');                                        //  force .jpg extension      v.10.9
      pext = strrchr(pfile,'.');
      if (! pext) pext = pfile + strlen(pfile);
      if (strlen(pext) > 5) pext = pext + strlen(pext);
      strcpy(pext,".jpg");
      
      write_popup_text("write",newfile);                                   //  report progress
      zmainloop();

      if (! Freplace) {
         err = stat(newfile,&statb);                                       //  if export, check if file exists
         if (! err) {
            snprintf(command,ccc,"%s", ZTX("new file already exists"));
            write_popup_text("write",command);
            zfree(newfile);
            continue;
         }
      }
      
      tempfile = strdupz(newfile,12,"batchresize.temp");                   //  temp file needed for EXIF copy
      strcat(tempfile,"-temp.jpg");                                        //  v.10.9
      
      f_open(oldfile,0);                                                   //  read old file

      wscale = hscale = 1.0;
      if (Fww > maxw) wscale = 1.0 * maxw / Fww;                           //  compute new size
      if (Fhh > maxh) hscale = 1.0 * maxh / Fhh;
      if (wscale < hscale) scale = wscale;
      else scale = hscale;
      ww = Fww * scale;
      hh = Fhh * scale;
      pxmout = PXM_rescale(Fpxm8,ww,hh);                                   //  rescale file

      PXBwrite(pxmout,tempfile);                                           //  write to temp file

      if (Freplace || Fcopyexif)                                           //  copy EXIF if files replaced
         exif_copy(oldfile,tempfile,0,0,0);                                //    or if requested for export
      
      snprintf(command,ccc,"cp -p \"%s\" \"%s\" ",tempfile,newfile);       //  copy tempfile to newfile
      err = system(command);
      if (err) write_popup_text("write",wstrerror(err));

      remove(tempfile);                                                    //  remove tempfile

      if (Freplace && ! err && strNeq(oldfile,newfile))                    //  remove oldfile if not
         remove(oldfile);                                                  //    already replaced

      zfree(newfile);
      zfree(tempfile);
      PXM_free(pxmout);
   }

   write_popup_text("write","COMPLETED");
   write_popup_text("close",0);
   
   batchresize[0] = maxw;                                                  //  save preferred size    v.10.9
   batchresize[1] = maxh;

cleanup:

   if (flist) {                                                            //  free memory
      for (ii = 0; flist[ii]; ii++) 
         zfree(flist[ii]);
      zfree(flist);
      flist = 0;
   }

   zdialog_free(zd);                                                       //  kill dialog
   return 0;
}


/**************************************************************************/

//  edit image annotation - write text on the image itself

char     annotate_file[1000] = "";                                         //  file for annotation data
int      annotate_mode = 0;                                                //  0/1/2 = 1st write / re-write / erase
int      annotate_px, annotate_py;                                         //  text position on image
PXM      *annotate_pxm = 0;                                                //  buffer for rendered annotation text
PXM      *annotate_pxm_transp = 0;                                         //  buffer for transparency data

void  annotate_dialog_stuff(zdialog *zd);                                  //  stuff dialog widgets from memory data
int   annotate_dialog_event(zdialog *zd, cchar *event);                    //  dialog event function
void  annotate_mousefunc();                                                //  mouse event function
void  annotate_gettext();                                                  //  render text into graphic image
void  annotate_write();                                                    //  write text on image
void  annotate_load();                                                     //  load annotation data from file
void  annotate_save();                                                     //  save annotation data to file


void m_annotate(GtkWidget *, cchar *menu)                                  //  overhauled  v.10.11
{
   char     *pp;
   cchar    *intro = ZTX("Enter text, click/drag on image.\n"
                         "Right click to remove");

   zfuncs::F1_help_topic = "annotate";                                     //  user guide topic

   if (! edit_setup("annotate",0,1)) return;                               //  setup edit

   if (! annotate_text) 
      annotate_text = strdupz("enter text",0,"annotate");

   if (! annotate_font || strEqu(annotate_font,"undefined"))
      annotate_font = strdupz("FreeMono Bold Italic 44",4,"annotate");     //  default font and size
   else annotate_font = strdupz(annotate_font,4,"annotate");               //  add extra space      bugfix  v.11.02
   

   if (! annotate_color[0] || strEqu(annotate_color[0],"undefined")) {     //  default colors
      annotate_color[0] = strdupz("255|0|0",0,"annotate");                 //  text
      annotate_color[1] = strdupz("0|0|255",0,"annotate");                 //  background
      annotate_color[2] = strdupz("0|255|0",0,"annotate");                 //  text outline
      annotate_towidth = 6;                                                //  outline width
   }
   
   pp = annotate_color[0];                                                 //  insure at least 20 bytes
   annotate_color[0] = strdupz(pp,20,"annotate");
   zfree(pp);
   pp = annotate_color[1];
   annotate_color[1] = strdupz(pp,20,"annotate");
   zfree(pp);
   pp = annotate_color[2];
   annotate_color[2] = strdupz(pp,20,"annotate");
   zfree(pp);

//                Annotate Image
//
//      Text [__________________________________]                          //  dialog
//
//      [font]    size [___|v]    angle [___|v]
//
//                     text     backing   outline                          //  v.11.04
//      Color         [color]   [color]   [color]
//      Transparency  [___|v]   [___|v]   [___|v]
//                               Width    [___|v]
//
//      [x] my mouse    Annotation File: [Open] [Save]
//                                       [Done] [Cancel]

   zdedit = zdialog_new(ZTX("Annotate Image"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","intro","dialog",intro,"space=3");
   zdialog_add_widget(zdedit,"hbox","hbtext","dialog",0,"space=3");
   zdialog_add_widget(zdedit,"label","labtext","hbtext",ZTX("Text"),"space=5");
   zdialog_add_widget(zdedit,"frame","frtext","hbtext",0,"expand");
   zdialog_add_widget(zdedit,"edit","text","frtext",0,"expand");

   zdialog_add_widget(zdedit,"hbox","hbfont","dialog",0,"space=8");
   zdialog_add_widget(zdedit,"button","font","hbfont",Bfont,"space=5");
   zdialog_add_widget(zdedit,"label","space","hbfont","","space=5");
   zdialog_add_widget(zdedit,"label","labsize","hbfont",ZTX("Size"),"space=3");
   zdialog_add_widget(zdedit,"spin","size","hbfont","6|99|1|20","space=3");
   zdialog_add_widget(zdedit,"label","space","hbfont","","space=5");
   zdialog_add_widget(zdedit,"label","labangle","hbfont",ZTX("Angle"),"space=3");
   zdialog_add_widget(zdedit,"spin","angle","hbfont","-180|180|0.2|0","space=3");
   
   zdialog_add_widget(zdedit,"hsep","hs1","dialog",0,"space=3");
   zdialog_add_widget(zdedit,"hbox","hbcol","dialog",0,"space=3");
   zdialog_add_widget(zdedit,"vbox","vbcol1","hbcol",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vbcol2","hbcol",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vbcol3","hbcol",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vbcol4","hbcol",0,"homog|space=5");
   
   zdialog_add_widget(zdedit,"label","space","vbcol1");
   zdialog_add_widget(zdedit,"label","labcol","vbcol1",ZTX("Color"));
   zdialog_add_widget(zdedit,"label","labcol","vbcol1",ZTX("Transparency"));
   zdialog_add_widget(zdedit,"label","space","vbcol1");
   
   zdialog_add_widget(zdedit,"label","labtext","vbcol2","text");
   zdialog_add_widget(zdedit,"colorbutt","fgcolor","vbcol2","0|0|0");
   zdialog_add_widget(zdedit,"spin","fgtrans","vbcol2","0|100|1|0");
   zdialog_add_widget(zdedit,"label","space","vbcol2");
   
   zdialog_add_widget(zdedit,"label","labback","vbcol3","backing");
   zdialog_add_widget(zdedit,"colorbutt","bgcolor","vbcol3","255|255|255");
   zdialog_add_widget(zdedit,"spin","bgtrans","vbcol3","0|100|1|0");
   zdialog_add_widget(zdedit,"label","labw","vbcol3",ZTX("Outline\n Width"));
   
   zdialog_add_widget(zdedit,"label","laboutln","vbcol4","outline");
   zdialog_add_widget(zdedit,"colorbutt","tocolor","vbcol4","255|0|0");
   zdialog_add_widget(zdedit,"spin","totrans","vbcol4","0|100|1|0");
   zdialog_add_widget(zdedit,"spin","towidth","vbcol4","0|9|1|0");

   zdialog_add_widget(zdedit,"hsep","hs1","dialog",0,"space=3");
   zdialog_add_widget(zdedit,"hbox","hbaf","dialog",0,"space=8");
   zdialog_add_widget(zdedit,"check","mymouse","hbaf",BmyMouse,"space=3");
   zdialog_add_widget(zdedit,"label","space","hbaf",0,"space=12");
   zdialog_add_widget(zdedit,"label","labbg","hbaf",ZTX("Annotation File:"));
   zdialog_add_widget(zdedit,"button","load","hbaf",Bopen,"space=8");
   zdialog_add_widget(zdedit,"button","save","hbaf",Bsave,"space=5");

   annotate_dialog_stuff(zdedit);                                          //  stuff dialog widgets from memory data

   zdialog_run(zdedit,annotate_dialog_event);                              //  start dialog

   zdialog_stuff(zdedit,"mymouse",1);
   takeMouse(zdedit,annotate_mousefunc,dragcursor);                        //  connect mouse function          v.10.12

   annotate_mode = 0;                                                      //  no write on image (yet)
   annotate_px = annotate_py = -1;                                         //  no location on image (yet)
   return;
}


//  stuff all dialog widgets from annotation data in memory

void  annotate_dialog_stuff(zdialog *zd)
{
   int      size;
   char     *pp;

   zdialog_stuff(zd,"text",annotate_text);
   zdialog_stuff(zd,"angle",annotate_angle);
   zdialog_stuff(zd,"fgtrans",annotate_trans[0]);
   zdialog_stuff(zd,"bgtrans",annotate_trans[1]);
   zdialog_stuff(zd,"totrans",annotate_trans[2]);
   zdialog_stuff(zd,"fgcolor",annotate_color[0]);
   zdialog_stuff(zd,"bgcolor",annotate_color[1]);
   zdialog_stuff(zd,"tocolor",annotate_color[2]);
   zdialog_stuff(zd,"towidth",annotate_towidth);

   pp = annotate_font + strlen(annotate_font);
   while (*pp != ' ') pp--;
   if (pp > annotate_font) {
      size = atoi(pp);
      if (size >= 6 && size <= 99) 
         zdialog_stuff(zd,"size",size);
   }
   
   annotate_gettext();                                                     //  build text buffer from annotation data
   
   return;
}


//  dialog event and completion callback function

int annotate_dialog_event(zdialog *zd, cchar *event)
{
   GtkWidget   *font_dialog;
   int         size, mymouse;
   char        *pp, text[1000];
   
   if (zd->zstat)
   {
      freeMouse();                                                         //  disconnect mouse function       v.10.12
      if (zd->zstat == 1 && Fmodified) edit_done();                        //  Done, complete pending edit
      else edit_cancel();                                                  //  Cancel or kill
      return 0;
   }
   
   if (strEqu(event,"mymouse")) {                                          //  toggle mouse capture         v.10.12
      zdialog_fetch(zd,"mymouse",mymouse);
      if (mymouse) takeMouse(zd,annotate_mousefunc,dragcursor);            //  connect mouse function       v.11.03
      else freeMouse();                                                    //  disconnect mouse
   }

   if (strEqu(event,"text")) {
      zdialog_fetch(zd,"text",text,999);                                   //  get text from dialog
      if (annotate_text) zfree(annotate_text);
      annotate_text = 0;
      if (*text) annotate_text = strdupz(text,0,"annotate");
   }
   
   if (strEqu(event,"font")) {                                             //  new font
      font_dialog = gtk_font_selection_dialog_new(ZTX("select font"));
      gtk_font_selection_dialog_set_font_name(GTK_FONT_SELECTION_DIALOG(font_dialog),annotate_font);
      gtk_dialog_run(GTK_DIALOG(font_dialog));
      pp = gtk_font_selection_dialog_get_font_name(GTK_FONT_SELECTION_DIALOG(font_dialog));
      gtk_widget_destroy(GTK_WIDGET(font_dialog));

      if (pp) {
         zfree(annotate_font);
         annotate_font = strdupz(pp,6,"annotate");                         //  update font and size
         g_free(pp);
         pp = annotate_font + strlen(annotate_font);
         while (*pp != ' ') pp--;
         if (pp > annotate_font) {
            size = atoi(pp);
            if (size >= 6 && size <= 99) zdialog_stuff(zd,"size",size);
         }
      }
   }

   if (strEqu(event,"load")) {                                             //  load annotate data from file
      annotate_load();
      annotate_dialog_stuff(zd);
      return 1;
   }

   if (strEqu(event,"save")) {                                             //  save annotate data to file
      annotate_save();
      return 1;
   }
   
   if (strEqu(event,"size")) {                                             //  new font size
      zdialog_fetch(zd,"size",size);
      pp = annotate_font + strlen(annotate_font);                          //  "fontname NN"
      while (*pp != ' ') pp--;                                             //  back-up to " NN"
      if (pp > annotate_font) sprintf(pp," %d",size);                      //  replace NN with new size
   }

   if (strEqu(event,"angle"))
      zdialog_fetch(zd,"angle",annotate_angle);
   
   if (strEqu(event,"fgcolor"))                                            //  foreground (text) color
      zdialog_fetch(zd,"fgcolor",annotate_color[0],20);
   
   if (strEqu(event,"bgcolor"))                                            //  background color
      zdialog_fetch(zd,"bgcolor",annotate_color[1],20);
   
   if (strEqu(event,"tocolor"))                                            //  text outline color
      zdialog_fetch(zd,"tocolor",annotate_color[2],20);
   
   if (strEqu(event,"fgtrans"))                                            //  foreground transparency
      zdialog_fetch(zd,"fgtrans",annotate_trans[0]);
   
   if (strEqu(event,"bgtrans"))                                            //  background transparency
      zdialog_fetch(zd,"bgtrans",annotate_trans[1]);

   if (strEqu(event,"totrans"))                                            //  text outline transparency
      zdialog_fetch(zd,"totrans",annotate_trans[2]);
   
   if (strEqu(event,"towidth"))                                            //  text outline width
      zdialog_fetch(zd,"towidth",annotate_towidth);

   annotate_gettext();                                                     //  rebuild text buffer
   annotate_write();                                                       //  write on image
   zmainloop();          

   return 1;
}


//  mouse function, set position for annotation text on image

void annotate_mousefunc()
{
   static int     dragging = 0;

   if (LMclick) {                                                          //  left mouse click
      LMclick = 0;
      annotate_px = Mxclick;                                               //  new text position on image
      annotate_py = Myclick;
      annotate_write();                                                    //  write text on image
   }
   
   if (RMclick) {                                                          //  right mouse click
      RMclick = 0;
      annotate_mode = 2;                                                   //  erase text on image
      annotate_write();
      annotate_px = annotate_py = -1;                                      //  no location on image
   }
   
   if (Mxdrag || Mydrag)                                                   //  mouse dragged
   {
      if (! dragging && annotate_mode) dragging = 1;

      if (dragging) {
         annotate_px = Mxdrag;                                             //  new text position on image
         annotate_py = Mydrag;
         annotate_write();                                                 //  write text on image
      }
   }
   else  dragging = 0;                                                     //  no drag underway

   return;
}


//  write annotation text on image at designated place, if wanted
//  annotate_mode: 0/1/2 = 1st write / re-write / erase

void annotate_write()
{
   uint8       *pix1, *pix2;
   uint16      *pix3, *pix31, *pix33;
   int         px1, py1, px3, py3;
   double      e3part, f256 = 1.0 / 256.0;

   static int  orgx1 = 0, orgy1 = 0, ww1 = 0, hh1 = 0;                     //  old text image overlap rectangle
   int         orgx2, orgy2, ww2, hh2;                                     //  new overlap rectangle
   
   if (annotate_mode)                                                      //  re-write or erase
   {
      for (py3 = orgy1; py3 < orgy1 + hh1; py3++)                          //  erase prior text image
      for (px3 = orgx1; px3 < orgx1 + ww1; px3++)                          //  replace E3 pixels with E1 pixels
      {                                                                    //    in prior overlap rectangle
         if (px3 < 0 || px3 >= E3ww) continue;
         if (py3 < 0 || py3 >= E3hh) continue;
         pix31 = PXMpix(E1pxm16,px3,py3);
         pix33 = PXMpix(E3pxm16,px3,py3);
         pix33[0] = pix31[0];
         pix33[1] = pix31[1];
         pix33[2] = pix31[2];
      }
      
      Fmodified = 0;
   }

   if (annotate_mode == 2) {                                               //  erase only
      annotate_mode = 0;                                                   //  next time is a 1st write
      mwpaint3(orgx1,orgy1,ww1,hh1);                                       //  update window
      return;
   }   

   if (! annotate_text) {                                                  //  no text to write
      annotate_mode = 0;                                                   //  next time is a 1st write
      mwpaint3(orgx1,orgy1,ww1,hh1);                                       //  update window
      return;
   }
   
   if (annotate_px < 0 || annotate_py < 0) {                               //  no image position to write
      annotate_mode = 0;                                                   //  next time is a 1st write
      mwpaint3(orgx1,orgy1,ww1,hh1);                                       //  update window
      return;
   }

   annotate_mode = 1;                                                      //  next time is a re-write

   ww2 = annotate_pxm->ww;                                                 //  text image buffer
   hh2 = annotate_pxm->hh;

   orgx2 = annotate_px - ww2/2;                                            //  copy-to image3 location
   orgy2 = annotate_py - hh2/2;

   for (py1 = 0; py1 < hh2; py1++)                                         //  loop all pixels in text image
   for (px1 = 0; px1 < ww2; px1++)
   {
      px3 = orgx2 + px1;                                                   //  copy-to image3 pixel
      py3 = orgy2 + py1;

      if (px3 < 0 || px3 >= E3ww) continue;                                //  omit parts beyond edges
      if (py3 < 0 || py3 >= E3hh) continue;

      pix1 = PXMpix8(annotate_pxm,px1,py1);                                //  copy-from text pixel
      pix2 = PXMpix8(annotate_pxm_transp,px1,py1);                         //  copy-from transparency
      pix3 = PXMpix(E3pxm16,px3,py3);                                      //  copy-to image pixel
      
      e3part = pix2[0] * f256;                                             //  image part visible through text
      
      pix3[0] = (pix1[0] << 8) + e3part * pix3[0];                         //  combine text part + image part
      pix3[1] = (pix1[1] << 8) + e3part * pix3[1];
      pix3[2] = (pix1[2] << 8) + e3part * pix3[2];
   }
   
   mwpaint3(orgx1,orgy1,ww1,hh1);                                          //  restore prior overlap rectangle
   mwpaint3(orgx2,orgy2,ww2,hh2);                                          //  update new overlap rectangle

   Fmodified = 1;
   
   orgx1 = orgx2;                                                          //  remember overlap rectangle
   orgy1 = orgy2;                                                          //    for next call

   ww1 = ww2;
   hh1 = hh2;
   return;
}


//  Create a graphic image with text, white on black, any font, any angle.

void annotate_gettext()                                                    //  revised   v.10.12
{
   PXM * annotate_outline(PXM *);

   PangoFontDescription    *pfont;
   static GdkColormap      *colormap = 0;
   static GdkColor         black, white;
   PangoLayout             *playout;
   GdkPixmap               *pixmap;
   GdkGC                   *gdkgc;
   GdkPixbuf               *pixbuf;

   char           *text = annotate_text;
   double         angle = annotate_angle;
   PXM            *pxm_temp1, *pxm_temp2, *pxm_temp3;
   uint8          *pix1, *pix2, *pix3, *ppix;
   int            px, py, ww, hh, rs, fontsize;
   char           *pp, font[100];
   cchar          *ppc;
   int            fgred, fggreen, fgblue;
   int            bgred, bggreen, bgblue;
   int            tored, togreen, toblue;
   int            red, green, blue;
   double         fgtrans, bgtrans, totrans, fgpart, bgpart;
   double         f256 = 1.0 / 256.0;

   if (! annotate_text || ! *annotate_text) return;                        //  no annotation text

   strncpy0(font,annotate_font,99);
   pp = font + strlen(font);                                               //  save current font size
   while (*pp != ' ') pp--;
   fontsize = atoi(pp);
   if (fontsize < 6 || fontsize > 99) fontsize = 20;

   strcpy(pp+1,"99");                                                      //  use large size for text generation

   if (! colormap) {
      colormap = gtk_widget_get_colormap(drWin);
      black.red = black.green = black.blue = 0;
      white.red = white.green = white.blue = 0xffff;
      gdk_rgb_find_color(colormap,&black);
      gdk_rgb_find_color(colormap,&white);
   }
   
   pfont = pango_font_description_from_string(font);                       //  make layout with text
   playout = gtk_widget_create_pango_layout(drWin,null);
   pango_layout_set_font_description(playout,pfont);
   pango_layout_set_text(playout,text,-1);
   pango_layout_get_pixel_size(playout,&ww,&hh);

   ww += 10;                                                               //  sometimes it is a little too small
   hh += 2;
   pixmap = gdk_pixmap_new(drWin->window,ww,hh,-1);                        //  then make pixmap
   gdkgc = gdk_gc_new(pixmap);
   gdk_draw_rectangle(pixmap,gdkgc,1,0,0,ww,hh);
   gdk_draw_layout_with_colors(pixmap,gdkgc,0,0,playout,&white,&black);

   pixbuf = gdk_pixbuf_get_from_drawable(null,pixmap,0,0,0,0,0,ww,hh);     //  then make pixbuf

   ww = gdk_pixbuf_get_width(pixbuf);                                      //  pixbuf dimensions
   hh = gdk_pixbuf_get_height(pixbuf);
   rs = gdk_pixbuf_get_rowstride(pixbuf);
   ppix = gdk_pixbuf_get_pixels(pixbuf);
   
   pxm_temp1 = PXM_make(ww,hh,8);

   for (py = 0; py < hh; py++)                                             //  copy pixbuf to PXM
   for (px = 0; px < ww; px++)
   {                                                                       //  text color is gray/white
      pix1 = ppix + rs * py + 3 * px;                                      //  can erase all but one color
      pix2 = PXMpix8(pxm_temp1,px,py);
      pix2[0] = pix1[0];
      pix2[1] = pix2[2] = 0;
   }

   g_object_unref(playout);
   g_object_unref(pixmap);
   g_object_unref(gdkgc);
   g_object_unref(pixbuf);
   
   pxm_temp2 = annotate_outline(pxm_temp1);                                //  add text outline color if any
   if (pxm_temp2) {
      PXM_free(pxm_temp1);
      pxm_temp1 = pxm_temp2;
   }

   if (fabs(angle) > 0.1) {                                                //  rotate text if wanted
      pxm_temp2 = PXM_rotate(pxm_temp1,angle);
      PXM_free(pxm_temp1);
      pxm_temp1 = pxm_temp2;
   }

   fgred = fggreen = fgblue = 0;
   bgred = bggreen = bgblue = 255;
   tored = togreen = toblue = 0;

   ppc = strField(annotate_color[0],'|',1);                                //  get text foreground color
   if (ppc) fgred = atoi(ppc);                                             //  0 - 255 per RGB color
   ppc = strField(annotate_color[0],'|',2);
   if (ppc) fggreen = atoi(ppc);
   ppc = strField(annotate_color[0],'|',3);
   if (ppc) fgblue = atoi(ppc);

   ppc = strField(annotate_color[1],'|',1);                                //  get text background color
   if (ppc) bgred = atoi(ppc);
   ppc = strField(annotate_color[1],'|',2);
   if (ppc) bggreen = atoi(ppc);
   ppc = strField(annotate_color[1],'|',3);
   if (ppc) bgblue = atoi(ppc);

   ppc = strField(annotate_color[2],'|',1);                                //  get text outline color
   if (ppc) tored = atoi(ppc);
   ppc = strField(annotate_color[2],'|',2);
   if (ppc) togreen = atoi(ppc);
   ppc = strField(annotate_color[2],'|',3);
   if (ppc) toblue = atoi(ppc);

   fgtrans = 0.01 * annotate_trans[0];                                     //  get transparencies
   bgtrans = 0.01 * annotate_trans[1];                                     //  text, background, text outline
   totrans = 0.01 * annotate_trans[2];

   ww = pxm_temp1->ww;                                                     //  text image input pixmap
   hh = pxm_temp1->hh;

   pxm_temp2 = PXM_make(ww,hh,8);                                          //  text image output pixmap
   pxm_temp3 = PXM_make(ww,hh,8);                                          //  output transparency map

   for (py = 0; py < hh; py++)                                             //  loop all pixels in text image
   for (px = 0; px < ww; px++)
   {
      pix1 = PXMpix8(pxm_temp1,px,py);                                     //  copy-from pixel
      pix2 = PXMpix8(pxm_temp2,px,py);                                     //  copy-to pixel
      pix3 = PXMpix8(pxm_temp3,px,py);                                     //  copy-to transparency

      fgpart = pix1[0] * f256;                                             //  white part = text foreground, 0 - 1
      bgpart = 1.0 - fgpart;                                               //  rest = text background part, 1 - 0
      
      if (pix1[1])                                                         //  use text outline color
      {
         fgpart = fgpart * (1.0 - totrans);                                //  reduce for transparencies
         bgpart = bgpart * (1.0 - bgtrans);                                
         red = tored * fgpart + bgred * bgpart;                            //  red part for text outline + background
         green = togreen * fgpart + bggreen * bgpart;                      //  same for green
         blue = toblue * fgpart + bgblue * bgpart;                         //  same for blue
      }
      
      else                                                                 //  use text foreground color
      {
         fgpart = fgpart * (1.0 - fgtrans);                                //  reduce for transparencies
         bgpart = bgpart * (1.0 - bgtrans);                                
         red = fgred * fgpart + bgred * bgpart;                            //  red part for text + text background
         green = fggreen * fgpart + bggreen * bgpart;                      //  same for green
         blue = fgblue * fgpart + bgblue * bgpart;                         //  same for blue
      }

      pix2[0] = red;                                                       //  output total red, green blue
      pix2[1] = green;
      pix2[2] = blue;

      pix3[0] = 255 * (1.0 - fgpart - bgpart);                             //  image part visible through text
   }
   
   ww = ww * fontsize / 99.0 + 0.5;                                        //  resize from size 99 font
   hh = hh * fontsize / 99.0 + 0.5;                                        //    to requested font size
   PXM_free(pxm_temp1);
   pxm_temp1 = PXM_rescale(pxm_temp2,ww,hh);
   PXM_free(pxm_temp2);
   PXM_free(annotate_pxm);
   annotate_pxm = pxm_temp1;
   
   pxm_temp1 = PXM_rescale(pxm_temp3,ww,hh);                               //  resize transparency map
   PXM_free(pxm_temp3);
   PXM_free(annotate_pxm_transp);
   annotate_pxm_transp = pxm_temp1;

   return;
}


//  add an outline color to the text character edges

PXM * annotate_outline(PXM *pxm1)                                          //  new  v.10.12
{
   PXM         *pxm2;
   int         toww, toww2;
   int         ww1, hh1, ww2, hh2;
   int         px1, py1, px2, py2, ii, diff;
   uint8       *pix1, *pix2;

   toww = annotate_towidth;                                                //  text outline color width
   if (toww == 0) return 0;                                                //  zero
   toww2 = 2 * toww;
   
   ww1 = pxm1->ww;                                                         //  input PXM dimensions
   hh1 = pxm1->hh;
   
   ww2 = ww1 + toww2;                                                      //  output PXM with added margins
   hh2 = hh1 + toww2;
   pxm2 = PXM_make(ww2,hh2,8);
   
   memset(pxm2->bmp,0,ww2*hh2*3);                                          //  clear output to black
   
   for (py1 = 0; py1 < hh1; py1++)                                         //  copy input to output,
   for (px1 = 0; px1 < ww1; px1++)                                         //    displaced for margins
   {
      pix1 = PXMpix8(pxm1,px1,py1);
      pix2 = PXMpix8(pxm2,px1+toww,py1+toww);
      pix2[0] = pix1[0];
      pix2[1] = pix1[1];
      pix2[2] = pix1[2];
   }
   
   for (py1 = 0; py1 < hh1; py1++)
   for (px1 = 0; px1 < ww1-toww-2; px1++)                                  //  horizontal forward scan
   {
      pix1 = PXMpix8(pxm1,px1,py1);
      diff = (pix1+6)[0] - pix1[0];
      if (diff < 200) continue;
      
      px2 = px1 + toww;
      py2 = py1 + toww;
      pix2 = PXMpix8(pxm2,px2-toww/2,py2);
      
      for (ii = 0; ii < toww + 1; ii++)
      {
         pix2[0] = pix1[0];
         pix2[1] = 1;
         pix1 += 3;
         pix2 += 3;
      }
   }

   for (py1 = 0; py1 < hh1; py1++)
   for (px1 = ww1-1; px1 > toww+2; px1--)                                  //  horizontal reverse scan
   {
      pix1 = PXMpix8(pxm1,px1,py1);
      diff = (pix1-6)[0] - pix1[0];
      if (diff < 200) continue;
      
      px2 = px1 + toww;
      py2 = py1 + toww;
      pix2 = PXMpix8(pxm2,px2+toww/2,py2);
      
      for (ii = 0; ii < toww + 1; ii++)
      {
         pix2[0] = pix1[0];
         pix2[1] = 1;
         pix1 -= 3;
         pix2 -= 3;
      }
   }

   for (px1 = 0; px1 < ww1; px1++)
   for (py1 = 0; py1 < hh1-toww-2; py1++)                                  //  vertical forward scan
   {
      pix1 = PXMpix8(pxm1,px1,py1);
      diff = (pix1+6*ww1)[0] - pix1[0];
      if (diff < 200) continue;
      
      px2 = px1 + toww;
      py2 = py1 + toww;
      pix2 = PXMpix8(pxm2,px2,py2-toww/2);
      
      for (ii = 0; ii < toww + 1; ii++)
      {
         if (pix2[0] < pix1[0]) pix2[0] = pix1[0];
         pix2[1] = 1;
         pix1 += 3*ww1;
         pix2 += 3*ww2;
      }
   }

   for (px1 = 0; px1 < ww1; px1++)
   for (py1 = hh1-1; py1 > toww+2; py1--)                                  //  vertical reverse scan
   {
      pix1 = PXMpix8(pxm1,px1,py1);
      diff = (pix1-6*ww1)[0] - pix1[0];
      if (diff < 200) continue;
      
      px2 = px1 + toww;
      py2 = py1 + toww;
      pix2 = PXMpix8(pxm2,px2,py2+toww/2);
      
      for (ii = 0; ii < toww + 1; ii++)
      {
         if (pix2[0] < pix1[0]) pix2[0] = pix1[0];
         pix2[1] = 1;
         pix1 -= 3*ww1;
         pix2 -= 3*ww2;
      }
   }

   return pxm2;
}


//  load annotation data from a file

void annotate_load()                                                       //  v.10.11
{
   FILE     *fid;
   int      cc, err;
   char     *pp, *file, buff[1200];
   cchar    *dialogtitle = "load annotation data from a file";

   file = zgetfile1(dialogtitle,"open",annotations_dirk);                  //  get input file from user
   if (! file) return;
   
   fid = fopen(file,"r");                                                  //  open for read
   if (! fid) {
      zmessageACK(mWin,"%s",strerror(errno));
      zfree(file);
      return;
   }

   while (true)
   {
      pp = fgets_trim(buff,1200,fid,1);
      if (! pp) break;
      
      if (strnEqu(pp,"annotate_text  ",15)) {
         if (annotate_text) zfree(annotate_text);
         cc = strlen(pp+15) + 100;
         annotate_text = zmalloc(cc,"annotate");
         repl_1str(pp+15,annotate_text,"\\n","\n");                        //  replace "\n" with real newline char.
      }
      
      if (strnEqu(pp,"annotate_font  ",15))
         strcpy(annotate_font,pp+15);
         
      if (strnEqu(pp,"annotate_angle  ",16))
         convSD(pp+16,annotate_angle,-180,+180);
      
      if (strnEqu(pp,"annotate_fgcolor  ",18))
         strcpy(annotate_color[0],pp+18);

      if (strnEqu(pp,"annotate_bgcolor  ",18))
         strcpy(annotate_color[1],pp+18);

      if (strnEqu(pp,"annotate_tocolor  ",18))
         strcpy(annotate_color[2],pp+18);

      if (strnEqu(pp,"annotate_fgtrans  ",18))
         convSI(pp+18,annotate_trans[0],0,100);

      if (strnEqu(pp,"annotate_bgtrans  ",18))
         convSI(pp+18,annotate_trans[1],0,100);

      if (strnEqu(pp,"annotate_totrans  ",18))
         convSI(pp+18,annotate_trans[2],0,100);

      if (strnEqu(pp,"annotate_towidth  ",18))
         convSI(pp+18,annotate_towidth,0,9);
   }

   err = fclose(fid);
   if (err) {
      zmessageACK(mWin,"%s",strerror(errno));
      zfree(file);
      return;
   }
   
   strcpy(annotate_file,file);                                             //  update current file
   zfree(file);

   return;
}


//  save annotation data to a file

void annotate_save()                                                       //  v.10.11
{
   FILE     *fid;
   char     *file, text[1100];
   cchar    *dialogtitle = "save annotation data to a file";
   int      err;

   file = zgetfile1(dialogtitle,"save",annotations_dirk);                  //  get output file from user
   if (! file) return;
   
   fid = fopen(file,"w");                                                  //  open for write
   if (! fid) {
      zmessageACK(mWin,"%s",strerror(errno));
      zfree(file);
      return;
   }
   
   repl_1str(annotate_text,text,"\n","\\n");                               //  replace newlines with "\n"

   fprintf(fid,"annotate_text  %s \n", text);
   fprintf(fid,"annotate_font  %s \n", annotate_font);
   fprintf(fid,"annotate_angle  %.1f \n", annotate_angle);
   fprintf(fid,"annotate_fgcolor  %s \n", annotate_color[0]);
   fprintf(fid,"annotate_bgcolor  %s \n", annotate_color[1]);
   fprintf(fid,"annotate_tocolor  %s \n", annotate_color[2]);
   fprintf(fid,"annotate_fgtrans  %d \n", annotate_trans[0]);
   fprintf(fid,"annotate_bgtrans  %d \n", annotate_trans[1]);
   fprintf(fid,"annotate_totrans  %d \n", annotate_trans[2]);
   fprintf(fid,"annotate_towidth  %d \n", annotate_towidth);

   fprintf(fid,"\n");
   
   err = fclose(fid);
   if (err) {
      zmessageACK(mWin,"file I/O error %s",file);
      zfree(file);
      return;
   }
   
   strcpy(annotate_file,file);                                             //  update current file
   zfree(file);

   return;
}


/**************************************************************************/

//  rotate image through any arbitrary angle

double      rotate_angle = 0;                                              //  E3 rotatation vs. F
double      rotate_delta = 0;
int         rotate_trim = 0;


void m_rotate(GtkWidget *, cchar *menu)                                    //  menu function
{
   int    rotate_dialog_event(zdialog *zd, cchar *event);
   void * rotate_thread(void *);
   void   rotate_mousefunc();
   
   cchar  *rotmess = ZTX("Use buttons or drag right edge with mouse");

   zfuncs::F1_help_topic = "rotate";                                       //  v.10.8

   if (! edit_setup("rotate",1,0)) return;                                 //  setup edit: use preview

   zdedit = zdialog_new(ZTX("Rotate Image"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","labrot","dialog",ZTX(rotmess),"space=5");
   zdialog_add_widget(zdedit,"label","labdeg","dialog",ZTX("degrees"),"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb1","hb1",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb2","hb1",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb3","hb1",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb4","hb1",0,"space=5");
   zdialog_add_widget(zdedit,"button"," +0.1  ","vb1"," + 0.1 ");          //  button name is increment to use
   zdialog_add_widget(zdedit,"button"," -0.1  ","vb1"," - 0.1 ");
   zdialog_add_widget(zdedit,"button"," +1.0  ","vb2"," + 1   ");
   zdialog_add_widget(zdedit,"button"," -1.0  ","vb2"," - 1   ");
   zdialog_add_widget(zdedit,"button"," +10.0 ","vb3"," + 10  ");
   zdialog_add_widget(zdedit,"button"," -10.0 ","vb3"," - 10  ");
   zdialog_add_widget(zdedit,"button"," +90.0 ","vb4"," + 90  ");
   zdialog_add_widget(zdedit,"button"," -90.0 ","vb4"," - 90  ");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"button","trim","hb2",ZTX("Trim"),"space=10");

   zdialog_run(zdedit,rotate_dialog_event);                                //  run dialog - parallel

   takeMouse(zdedit,rotate_mousefunc,dragcursor);                          //  connect mouse function          v.11.03

   rotate_angle = rotate_delta = rotate_trim = 0;
   start_thread(rotate_thread,0);                                          //  start working thread
   return;
}


//  dialog event and completion callback function

int rotate_dialog_event(zdialog *zd, cchar * event)
{
   int         err, trim = 0;
   double      incr;
   char        text[20];
   
   if (zd->zstat)                                                          //  dialog complete
   {
      freeMouse();                                                         //  disconnect mouse function       v.10.12

      if (zd->zstat == 1) {
         rotate_delta = rotate_angle;                                      //  rotate main image
         rotate_angle = 0;
         edit_done();
      }
      else edit_cancel();

      rotate_angle = rotate_delta = rotate_trim = 0;
      return 0;
   }
   
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3

   if (strEqu(event,"trim")) {
      rotate_trim = 1 - rotate_trim;                                       //  toggle trim button
      if (rotate_trim) zdialog_stuff(zd,"trim",ZTX("Undo Trim"));
      else zdialog_stuff(zd,"trim",ZTX("Trim"));
      trim = 1;                                                            //  v.10.3
   }
   
   if (strpbrk(event,"+-")) {
      err = convSD(event,incr);                                            //  button name is increment to use
      if (err) return 0;
      rotate_delta += incr;
   }

   if (rotate_delta || trim) {
      trim = 0;
      zdialog_stuff(zd,"labdeg","computing");
      signal_thread();                                                     //  do rotate in thread
      wait_thread_idle();
      sprintf(text,ZTX("degrees: %.1f"),rotate_angle);                     //  update dialog angle display
      zdialog_stuff(zd,"labdeg",text);
   }

   return 1;
}


//  rotate mouse function - drag right edge of image up/down for rotation

void rotate_mousefunc()
{
   static int     mpx0 = 0, mpy0 = 0;
   static int     mpy1, mpy2, dist;

   if (! Mxdrag && ! Mydrag) return;                                       //  no drag underway
   if (Mxdrag < 0.8 * E3ww) return;                                        //  not right edge of image

   if (Mxdown != mpx0 || Mydown != mpy0) {
      mpx0 = Mxdown;                                                       //  new drag started
      mpy0 = mpy1 = Mydown;
   }
   
   mpy2 = Mydrag;
   dist = mpy2 - mpy1;                                                     //  drag distance
   mpy1 = mpy2;                                                            //  reset origin for next time
   if (! dist) return;

   rotate_delta = 30.0 * dist / E3ww;                                      //  convert to angle
   rotate_dialog_event(zdedit,"mouse");
   return;
}


//  rotate thread function

void * rotate_thread(void *)
{
   int         px3, py3, px9, py9;
   int         wwcut, hhcut, ww, hh;
   double      trim_angle, radians;
   uint16      *pix3, *pix9;
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      mutex_lock(&Fpixmap_lock);

      rotate_angle += rotate_delta;                                        //  accum. net rotation   
      rotate_delta = 0;                                                    //    from dialog widget
      
      if (rotate_angle >= 360) rotate_angle -=360;
      if (rotate_angle <= -360) rotate_angle +=360;
      if (fabs(rotate_angle) < 0.01) rotate_angle = 0;

      if (! rotate_angle) {
         PXM_free(E3pxm16);                                                //  E1 >> E3
         E3pxm16 = PXM_copy(E1pxm16);
         E3ww = E1ww;
         E3hh = E1hh;
         Fmodified = 0;
      }
      
      if (rotate_angle) {
         PXM_free(E3pxm16);
         E3pxm16 = PXM_rotate(E1pxm16,rotate_angle);                       //  E3 is rotated E1
         E3ww = E3pxm16->ww;
         E3hh = E3pxm16->hh;
         Fmodified = 1;
      }

      if (rotate_trim)
      {                                                                    //  auto trim
         trim_angle = fabs(rotate_angle);
         while (trim_angle > 45) trim_angle -= 90;
         radians = fabs(trim_angle / 57.296);
         wwcut = int(E3pxm16->hh * sin(radians) + 1);                      //  amount to trim
         hhcut = int(E3pxm16->ww * sin(radians) + 1);
         ww = E3pxm16->ww - 2 * wwcut;
         hh = E3pxm16->hh - 2 * hhcut;
         if (ww > 0 && hh > 0) {
            E9pxm16 = PXM_make(ww,hh,16);
            
            for (py3 = hhcut; py3 < E3hh-hhcut; py3++)                     //  E9 = trimmed E3
            for (px3 = wwcut; px3 < E3ww-wwcut; px3++)
            {
               px9 = px3 - wwcut;
               py9 = py3 - hhcut;
               pix3 = PXMpix(E3pxm16,px3,py3);
               pix9 = PXMpix(E9pxm16,px9,py9);
               pix9[0] = pix3[0];
               pix9[1] = pix3[1];
               pix9[2] = pix3[2];
            }

            PXM_free(E3pxm16);                                             //  E3 = E9
            E3pxm16 = E9pxm16;
            E9pxm16 = 0;
            E3ww = ww;
            E3hh = hh;
         }
      }
      
      mutex_unlock(&Fpixmap_lock);
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


/**************************************************************************/

//  flip an image horizontally or vertically

void m_flip(GtkWidget *, cchar *)
{
   int flip_dialog_event(zdialog *zd, cchar *event);

   zfuncs::F1_help_topic = "flip_image";                                   //  v.10.8

   if (! edit_setup("flip",0,0)) return;                                   //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Flip Image"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"button","horz","hb1",ZTX("horizontal"),"space=5");
   zdialog_add_widget(zdedit,"button","vert","hb1",ZTX("vertical"),"space=5");

   zdialog_run(zdedit,flip_dialog_event);                                  //  run dialog - parallel
   return;
}


//  dialog event and completion callback function

int flip_dialog_event(zdialog *zd, cchar *event)
{
   int flip_horz();
   int flip_vert();

   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      return 0;
   }

   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"horz")) flip_horz();
   if (strEqu(event,"vert")) flip_vert();
   return 0;
}


int flip_horz()
{
   int      px, py;
   uint16   *pix3, *pix9;
   
   edit_zapredo();                                                         //  delete redo copy    v.10.3

   E9pxm16 = PXM_copy(E3pxm16);
   
   for (py = 0; py < E3hh; py++)
   for (px = 0; px < E3ww; px++)
   {
      pix3 = PXMpix(E3pxm16,px,py);                                        //  image9 = flipped image3
      pix9 = PXMpix(E9pxm16,E3ww-1-px,py);
      pix9[0] = pix3[0];
      pix9[1] = pix3[1];
      pix9[2] = pix3[2];
   }
   
   mutex_lock(&Fpixmap_lock);
   PXM_free(E3pxm16);                                                      //  image9 >> image3
   E3pxm16 = E9pxm16;
   E9pxm16 = 0;
   mutex_unlock(&Fpixmap_lock);

   Fmodified = 1;
   mwpaint2();
   return 0;
}


int flip_vert()
{
   int      px, py;
   uint16   *pix3, *pix9;
   
   edit_zapredo();                                                         //  delete redo copy    v.10.3

   E9pxm16 = PXM_copy(E3pxm16);
   
   for (py = 0; py < E3hh; py++)
   for (px = 0; px < E3ww; px++)
   {
      pix3 = PXMpix(E3pxm16,px,py);                                        //  image9 = flipped image3
      pix9 = PXMpix(E9pxm16,px,E3hh-1-py);
      pix9[0] = pix3[0];
      pix9[1] = pix3[1];
      pix9[2] = pix3[2];
   }
   
   mutex_lock(&Fpixmap_lock);
   PXM_free(E3pxm16);                                                      //  image9 >> image3
   E3pxm16 = E9pxm16;
   E9pxm16 = 0;
   mutex_unlock(&Fpixmap_lock);

   Fmodified = 1;
   mwpaint2();
   return 0;
}


/**************************************************************************/

//  unbend an image
//  straighten curvature added by pano or improve perspective

double   unbend_lin_horz, unbend_lin_vert;                                 //  unbend values from dialog
double   unbend_cur_horz, unbend_cur_vert;
double   unbend_x1, unbend_x2, unbend_y1, unbend_y2;                       //  unbend axes scaled 0 to 1
int      unbend_hx1, unbend_hy1, unbend_hx2, unbend_hy2;
int      unbend_vx1, unbend_vy1, unbend_vx2, unbend_vy2;


void m_unbend(GtkWidget *, cchar *)                                        //  overhauled    v.11.04
{
   int    unbend_dialog_event(zdialog* zd, cchar *event);
   void * unbend_thread(void *);
   void   unbend_mousefunc();

   zfuncs::F1_help_topic = "unbend";

   if (! edit_setup("unbend",1,0)) return;                                 //  setup edit: preview

/***         ___________________________________
            |                                   |
            |           Unbend Image            |
            |                                   |
            |                linear    curved   |
            |  vertical      [__|v]    [__|v]   |
            |  horizontal    [__|v]    [__|v]   |
            |                                   |
            |                [done]  [cancel]   |
            |___________________________________|
***/

   zdedit = zdialog_new(ZTX("Unbend Image"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=8");
   zdialog_add_widget(zdedit,"vbox","vb1","hb1",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb2","hb1",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb3","hb1",0,"homog|space=5");
   zdialog_add_widget(zdedit,"label","labspace","vb1","");
   zdialog_add_widget(zdedit,"label","labvert","vb1",ZTX("vertical"));
   zdialog_add_widget(zdedit,"label","labhorz","vb1",ZTX("horizontal"));
   zdialog_add_widget(zdedit,"label","lablin","vb2",ZTX("linear"));
   zdialog_add_widget(zdedit,"spin","splinvert","vb2","-99|99|1|0");
   zdialog_add_widget(zdedit,"spin","splinhorz","vb2","-99|99|1|0");
   zdialog_add_widget(zdedit,"label","labhorz","vb3",ZTX("curved"));
   zdialog_add_widget(zdedit,"spin","spcurvert","vb3","-99|99|1|0");
   zdialog_add_widget(zdedit,"spin","spcurhorz","vb3","-99|99|1|0");

   zdialog_run(zdedit,unbend_dialog_event);                                //  run dialog, parallel

   unbend_x1 = unbend_x2 = unbend_y1 = unbend_y2 = 0.5;                    //  initial axes thru image middle
   unbend_lin_horz = unbend_lin_vert = 0;
   unbend_cur_horz = unbend_cur_vert = 0;

   takeMouse(zdedit,unbend_mousefunc,dragcursor);                          //  connect mouse function          v.11.03
   
   start_thread(unbend_thread,0);                                          //  start working thread
   signal_thread();

   return;
}


//  dialog event and completion callback function

int unbend_dialog_event(zdialog *zd, cchar *event)
{
   if (zd->zstat)                                                          //  dialog complete
   {
      paint_toplines(2);                                                   //  erase axes-lines
      freeMouse();                                                         //  disconnect mouse function       v.10.12

      if (zd->zstat != 1) {
         edit_cancel();                                                    //  canceled
         return 0;
      }

      if (unbend_cur_vert || unbend_cur_horz ||                            //  image3 modified
          unbend_lin_vert || unbend_lin_horz) Fmodified = 1;
      else Fmodified = 0;
      edit_done();                                                         //  commit changes to image3
      return 1;
   }

   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3

   if (strstr(event,"splinvert")) {                                        //  get new unbend value
      zdialog_fetch(zd,"splinvert",unbend_lin_vert);
      signal_thread();                                                     //  trigger thread
   }

   if (strstr(event,"splinhorz")) {
      zdialog_fetch(zd,"splinhorz",unbend_lin_horz);
      signal_thread();
   }

   if (strstr(event,"spcurvert")) {
      zdialog_fetch(zd,"spcurvert",unbend_cur_vert);
      signal_thread();
   }

   if (strstr(event,"spcurhorz")) {
      zdialog_fetch(zd,"spcurhorz",unbend_cur_horz);
      signal_thread();
   }

   return 1;
}


//  unbend mouse function                                                  //  adjustable axes

void unbend_mousefunc()   
{
   cchar       *close;
   double      dist1, dist2;
   double      mpx = 0, mpy = 0;
   
   if (LMclick) {                                                          //  left mouse click
      LMclick = 0;
      mpx = Mxclick;
      mpy = Myclick;
   }
   
   if (Mxdrag || Mydrag) {                                                 //  mouse dragged
      mpx = Mxdrag;
      mpy = Mydrag;
   }
   
   if (! mpx && ! mpy) return;

   mpx = 1.0 * mpx / E3ww;                                                 //  scale mouse position 0 to 1
   mpy = 1.0 * mpy / E3hh;

   if (mpx < 0.2 || mpx > 0.8 ) {                                          //  check reasonable position
      if (mpy < 0.1 || mpy > 0.9) return;
   }
   else if (mpy < 0.2 || mpy > 0.8) {
      if (mpx < 0.1 || mpx > 0.9) return;
   }
   else return;

   close = "?";                                                            //  find closest axis end-point
   dist1 = 2;

   dist2 = mpx * mpx + (mpy-unbend_y1) * (mpy-unbend_y1);
   if (dist2 < dist1) {
      dist1 = dist2;
      close = "left";
   }

   dist2 = (1-mpx) * (1-mpx) + (mpy-unbend_y2) * (mpy-unbend_y2);
   if (dist2 < dist1) {
      dist1 = dist2;
      close = "right";
   }

   dist2 = (mpx-unbend_x1) * (mpx-unbend_x1) + mpy * mpy;
   if (dist2 < dist1) {
      dist1 = dist2;
      close = "top";
   }

   dist2 = (mpx-unbend_x2) * (mpx-unbend_x2) + (1-mpy) * (1-mpy);
   if (dist2 < dist1) {
      dist1 = dist2;
      close = "bottom";
   }
   
   if (strEqu(close,"left")) unbend_y1 = mpy;                              //  set new axis end-point
   if (strEqu(close,"right")) unbend_y2 = mpy;
   if (strEqu(close,"top")) unbend_x1 = mpx;
   if (strEqu(close,"bottom")) unbend_x2 = mpx;

   signal_thread();                                                        //  trigger thread 

   return ;
}


//  unbend thread function

void * unbend_thread(void *arg)
{
   void * unbend_wthread(void *);

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      unbend_hx1 = 0;                                                      //  scale axes to E3ww/hh
      unbend_hy1 = unbend_y1 * E3hh;
      unbend_hx2 = E3ww;
      unbend_hy2 = unbend_y2 * E3hh;

      unbend_vx1 = unbend_x1 * E3ww;
      unbend_vy1 = 0;
      unbend_vx2 = unbend_x2 * E3ww;
      unbend_vy2 = E3hh;

      if (Fpreview) {                                                      //  omit for final unbend
         Ntoplines = 2;
         toplinex1[0] = unbend_hx1;                                        //  lines on window
         topliney1[0] = unbend_hy1;
         toplinex2[0] = unbend_hx2;
         topliney2[0] = unbend_hy2;
         toplinex1[1] = unbend_vx1;
         topliney1[1] = unbend_vy1;
         toplinex2[1] = unbend_vx2;
         topliney2[1] = unbend_vy2;
      }
      
      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker threads
         start_wt(unbend_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}

   
void * unbend_wthread(void *arg)                                           //  worker thread function
{
   int         index = *((int *) arg);
   int         vstat, px3, py3, cx3, cy3;
   double      dispx, dispy, dispx2, dispy2;
   double      px1, py1, vx1, vx2, hy1, hy2;
   double      curvert, curhorz, linvert, linhorz;
   uint16      vpix[3], *pix3;

   curvert = int(unbend_cur_vert * 0.01 * E3hh);                           //  -0.99 to +0.99
   curhorz = int(unbend_cur_horz * 0.01 * E3ww);
   linvert = int(unbend_lin_vert * 0.0013 * E3hh);                         //  -0.13 to +0.13
   linhorz = int(unbend_lin_horz * 0.0013 * E3ww);
   
   vx1 = unbend_vx1;
   vx2 = unbend_vx2;
   hy1 = unbend_hy1;
   hy2 = unbend_hy2;

   for (py3 = index; py3 < E3hh; py3 += Nwt)                               //  step through F3 pixels
   for (px3 = 0; px3 < E3ww; px3++)
   {
      cx3 = vx1 + (vx2 - vx1) * py3 / E3hh;                                //  center of unbend
      cy3 = hy1 + (hy2 - hy1) * px3 / E3ww;
      dispx = 2.0 * (px3 - cx3) / E3ww;                                    //  -1.0 ..  0.0 .. +1.0  (roughly)
      dispy = 2.0 * (py3 - cy3) / E3hh;                                    //  -1.0 ..  0.0 .. +1.0
      dispx2 = -cos(0.8 * dispx) + 1;                                      //   curved                v.11.03
      dispy2 = -cos(0.8 * dispy) + 1;                                      //                         v.11.04
      
      pix3 = PXMpix(E3pxm16,px3,py3);                                      //  output pixel
      px1 = px3;                                                           //  input pixel = output
      py1 = py3;
      
      px1 += dispx * dispy * linhorz;                                      //  move input pixel
      py1 += dispy * dispx * linvert;
      px1 += dispx * dispy2 * curhorz;
      py1 += dispy * dispx2 * curvert;

      vstat = vpixel(E1pxm16,px1,py1,vpix);
      if (vstat) {
         pix3[0] = vpix[0];                                                //  input pixel >> output pixel
         pix3[1] = vpix[1];
         pix3[2] = vpix[2];
      }
      else pix3[0] = pix3[1] = pix3[2] = 0;
   }
      
   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


/**************************************************************************/

//  warp/distort area - select image area and pull with mouse

float       *WarpAx, *WarpAy;                                              //  memory of all displaced pixels
float       WarpAmem[4][100];                                              //  undo memory, last 100 warps
int         NWarpA;                                                        //  WarpA mem count
int         WarpA_started;

void  WarpA_warpfunc(float wdx, float wdy, float wdw, float wdh, int acc);


void m_warp_area(GtkWidget *, cchar *)
{
   int      WarpA_dialog_event(zdialog *zd, cchar *event);
   
   cchar  *WarpA_message = ZTX(
             " Select an area to warp using select area function. \n"
             " Press [start warp] and pull area with mouse. \n"
             " Make multiple mouse pulls until satisfied. \n"
             " When finished, select another area or press [done]."); 
   
   int         px, py, ii;

   zfuncs::F1_help_topic = "warp_area";

   if (! edit_setup("warp_area",0,2)) return;                              //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Warp Image (area)"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",WarpA_message,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"button","swarp","hb1",ZTX("start warp"),"space=5");
   zdialog_add_widget(zdedit,"button","undlast","hb1",Bundolast,"space=5");
   zdialog_add_widget(zdedit,"button","undall","hb1",Bundoall,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"check","mymouse","hb2",BmyMouse,"space=5");

   zdialog_run(zdedit,WarpA_dialog_event);                                 //  run dialog

   WarpAx = (float *) zmalloc(E3ww * E3hh * sizeof(float),"warp.ax");      //  get memory for pixel displacements
   WarpAy = (float *) zmalloc(E3ww * E3hh * sizeof(float),"warp.ay");
   
   NWarpA = 0;                                                             //  no warp data
   WarpA_started = 0;

   for (py = 0; py < E3hh; py++)                                           //  no pixel displacements
   for (px = 0; px < E3ww; px++)
   {
      ii = py * E3ww + px;
      WarpAx[ii] = WarpAy[ii] = 0.0;
   }
   
   return;
}


//  dialog event and completion callback function

int WarpA_dialog_event(zdialog * zd, cchar *event)
{
   void     WarpA_mousefunc(void);

   int         px, py, ii, mymouse;
   float       wdx, wdy, wdw, wdh;

   if (zd->zstat)                                                          //  dialog complete
   {
      freeMouse();                                                         //  disconnect mouse function       v.10.12

      if (NWarpA) Fmodified = 1;
      else Fmodified = 0;

      if (zd->zstat == 1) edit_done();
      else edit_cancel();

      zfree(WarpAx);                                                       //  release undo memory
      zfree(WarpAy);
      return 0;
   }
   
   if (strEqu(event,"swarp"))                                              //  start warp
   {
      if (! Factivearea || sa_mode == 7) {                                 //  no select area active        v.11.01
         zmessageACK(mWin,ZTX("Select area first"));
         return 0;
      }
      sa_edgecalc();                                                       //  calculate area edge distances
      takeMouse(zd,WarpA_mousefunc,dragcursor);                            //  connect mouse function          v.11.03
      WarpA_started = 1;
   }
   
   if (! WarpA_started) {
      zdialog_stuff(zd,"mymouse",0);
      return 0;
   }

   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3

   if (strEqu(event,"mymouse")) {                                          //  toggle mouse capture         v.10.12
      zdialog_fetch(zd,"mymouse",mymouse);
      if (mymouse)                                                         //  connect mouse function       v.11.03
         takeMouse(zd,WarpA_mousefunc,dragcursor);
      else freeMouse();                                                    //  disconnect mouse
   }

   if (strEqu(event,"undlast")) {
      if (NWarpA) {                                                        //  undo most recent warp
         ii = --NWarpA;
         wdx = WarpAmem[0][ii];
         wdy = WarpAmem[1][ii];
         wdw = WarpAmem[2][ii];
         wdh = WarpAmem[3][ii];
         WarpA_warpfunc(wdx,wdy,-wdw,-wdh,0);                              //  unwarp image
         WarpA_warpfunc(wdx,wdy,-wdw,-wdh,1);                              //  unwarp memory
      }
   }

   if (strEqu(event,"undall"))                                             //  undo all warps
   {
      edit_reset();                                                        //  v.10.3

      for (py = 0; py < E3hh; py++)                                        //  reset pixel displacements
      for (px = 0; px < E3ww; px++)
      {
         ii = py * E3ww + px;
         WarpAx[ii] = WarpAy[ii] = 0.0;
      }

      NWarpA = 0;                                                          //  erase undo memory
      mwpaint2(); 
   }

   return 1;
}


//  warp mouse function

void  WarpA_mousefunc(void)
{
   static float   wdx, wdy, wdw, wdh;
   static int     ii, warped = 0;

   if (Mxdrag || Mydrag)                                                   //  mouse drag underway
   {
      wdx = Mxdown;                                                        //  drag origin, image coordinates
      wdy = Mydown;
      wdw = Mxdrag - Mxdown;                                               //  drag increment
      wdh = Mydrag - Mydown;
      WarpA_warpfunc(wdx,wdy,wdw,wdh,0);                                   //  warp image
      warped = 1;
      return;
   }
   
   else if (warped) 
   {
      warped = 0;
      WarpA_warpfunc(wdx,wdy,wdw,wdh,1);                                   //  drag done, add to warp memory

      if (NWarpA == 100)                                                   //  if full, throw away oldest
      {
         NWarpA = 99;
         for (ii = 0; ii < NWarpA; ii++)
         {
            WarpAmem[0][ii] = WarpAmem[0][ii+1];
            WarpAmem[1][ii] = WarpAmem[1][ii+1];
            WarpAmem[2][ii] = WarpAmem[2][ii+1];
            WarpAmem[3][ii] = WarpAmem[3][ii+1];
         }
      }

      ii = NWarpA;
      WarpAmem[0][ii] = wdx;                                               //  save warp for undo
      WarpAmem[1][ii] = wdy;
      WarpAmem[2][ii] = wdw;
      WarpAmem[3][ii] = wdh;
      NWarpA++;
   }
   
   return;
}


//  warp image and accumulate warp memory

void  WarpA_warpfunc(float wdx, float wdy, float wdw, float wdh, int acc)
{
   int            ii, jj, px, py, ww, hh, vstat;
   double         ddx, ddy, dpe, dpm, mag, dispx, dispy;
   uint16         vpix[3], *pix3;

   edit_zapredo();                                                         //  delete redo copy    v.10.3
   
   for (py = sa_miny; py <= sa_maxy; py++)                                 //  loop all pixels in area   v.10.11
   for (px = sa_minx; px <= sa_maxx; px++)
   {
      ii = py * Fww + px;
      dpe = sa_pixisin[ii];                                                //  distance from area edge
      if (! dpe) continue;

      ddx = (px - wdx);
      ddy = (py - wdy);
      dpm = sqrt(ddx*ddx + ddy*ddy);                                       //  distance from drag origin

      if (dpm < 1.0) mag = 1.0;                                            //  revised   v.10.8
      else  mag = 1.0 - dpm / (dpm + dpe);
      mag = mag * mag;

      dispx = -wdw * mag;                                                  //  warp = drag * magnification
      dispy = -wdh * mag;
      
      jj = py * E3ww + px;

      if (acc) {                                                           //  mouse drag done,
         WarpAx[jj] += dispx;                                              //    accumulate warp memory
         WarpAy[jj] += dispy;
         continue;
      }

      dispx += WarpAx[jj];                                                 //  add this warp to prior
      dispy += WarpAy[jj];

      vstat = vpixel(E1pxm16,px+dispx,py+dispy,vpix);                      //  input virtual pixel
      if (vstat) {
         pix3 = PXMpix(E3pxm16,px,py);                                     //  output pixel
         pix3[0] = vpix[0];
         pix3[1] = vpix[1];
         pix3[2] = vpix[2];
      }
   }

   ww = sa_maxx - sa_minx;                                                 //  update window    v.10.11
   hh = sa_maxy - sa_miny;
   mwpaint3(sa_minx,sa_miny,ww,hh);

   Fmodified = 1;                                                          //  v.10.2
   return;
}


/**************************************************************************/

//  warp/distort whole image with a curved transform
//  fix perspective problems (e.g. curved walls, leaning buildings)

float       *WarpCx, *WarpCy;                                              //  memory of all dragged pixels
float       WarpCmem[4][100];                                              //  undo memory, last 100 drags
int         NWarpC;                                                        //  WarpCmem count
int         WarpCdrag;
int         WarpCww, WarpChh;

void  WarpC_warpfunc(float wdx, float wdy, float wdw, float wdh, int acc);


void m_warp_curved(GtkWidget *, cchar *)
{
   int      WarpC_dialog_event(zdialog *zd, cchar *event);
   void     WarpC_mousefunc(void);

   cchar  *WarpC_message = ZTX(
             " Pull an image position using the mouse. \n"
             " Make multiple mouse pulls until satisfied. \n"
             " When finished, press [done]."); 
   
   int         px, py, ii;

   zfuncs::F1_help_topic = "warp_curved";

   if (! edit_setup("warp_curved",1,0)) return;                            //  setup edit: use preview

   zdedit = zdialog_new(ZTX("Warp Image (curved)"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",WarpC_message,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"button","undlast","hb1",Bundolast,"space=5");
   zdialog_add_widget(zdedit,"button","undall","hb1",Bundoall,"space=5");

   zdialog_run(zdedit,WarpC_dialog_event);                                 //  run dialog

   NWarpC = WarpCdrag = 0;                                                 //  no drag data

   WarpCx = (float *) zmalloc(E3ww * E3hh * sizeof(float),"warp.ix");      //  get memory for pixel displacements
   WarpCy = (float *) zmalloc(E3ww * E3hh * sizeof(float),"warp.iy");
   
   for (py = 0; py < E3hh; py++)                                           //  no pixel displacements
   for (px = 0; px < E3ww; px++)
   {
      ii = py * E3ww + px;
      WarpCx[ii] = WarpCy[ii] = 0.0;
   }
   
   WarpCww = E3ww;                                                         //  preview dimensions
   WarpChh = E3hh;
   
   takeMouse(zdedit,WarpC_mousefunc,dragcursor);                           //  connect mouse function          v.11.03
   return;
}


//  dialog event and completion callback function

int WarpC_dialog_event(zdialog * zd, cchar *event)
{
   int         px, py, ii;
   float       wdx, wdy, wdw, wdh;
   int         fpx, fpy, epx, epy, vstat;
   double      scale, dispx, dispy;
   uint16      vpix[3], *pix3;

   if (zd->zstat) goto complete;

   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3

   if (strEqu(event,"undlast")) 
   {
      if (NWarpC == 1) event = "undall";
      else if (NWarpC) {                                                   //  undo most recent drag
         ii = --NWarpC;
         wdx = WarpCmem[0][ii];
         wdy = WarpCmem[1][ii];
         wdw = WarpCmem[2][ii];
         wdh = WarpCmem[3][ii];
         WarpC_warpfunc(wdx,wdy,-wdw,-wdh,0);                              //  undrag image
         WarpC_warpfunc(wdx,wdy,-wdw,-wdh,1);                              //  undrag memory
      }
   }

   if (strEqu(event,"undall"))                                             //  undo all drags
   {
      NWarpC = 0;                                                          //  erase undo memory

      for (py = 0; py < E3hh; py++)                                        //  reset pixel displacements
      for (px = 0; px < E3ww; px++)
      {
         ii = py * E3ww + px;
         WarpCx[ii] = WarpCy[ii] = 0.0;
      }
      edit_reset();                                                        //  restore image 1  v.10.3
   }
   
   return 1;

complete:

   freeMouse();                                                            //  disconnect mouse

   if (zd->zstat != 1) edit_cancel();
   else if (NWarpC == 0) edit_cancel();
   else 
   {
      edit_fullsize();                                                     //  get full-size E1/E3

      scale = 1.0 * (E3ww + E3hh) / (WarpCww + WarpChh);

      for (fpy = 0; fpy < E3hh; fpy++)                                     //  scale net pixel displacements
      for (fpx = 0; fpx < E3ww; fpx++)                                     //    to full image size
      {
         epx = WarpCww * fpx / E3ww;
         epy = WarpChh * fpy / E3hh;
         ii = epy * WarpCww + epx;
         dispx = WarpCx[ii] * scale;
         dispy = WarpCy[ii] * scale;

         vstat = vpixel(E1pxm16,fpx+dispx,fpy+dispy,vpix);                 //  input virtual pixel
         pix3 = PXMpix(E3pxm16,fpx,fpy);                                   //  output pixel
         if (vstat) {
            pix3[0] = vpix[0];
            pix3[1] = vpix[1];
            pix3[2] = vpix[2];
         }
         else pix3[0] = pix3[1] = pix3[2] = 0;
      }

      edit_done();
   }

   zfree(WarpCx);                                                          //  release memory
   zfree(WarpCy);
   return 0;
}


//  WarpC mouse function

void  WarpC_mousefunc(void)
{
   static float   wdx, wdy, wdw, wdh;
   int            ii;

   if (Mxdrag || Mydrag)                                                   //  mouse drag underway
   {
      wdx = Mxdown;                                                        //  drag origin, window coordinates
      wdy = Mydown;
      wdw = Mxdrag - Mxdown;                                               //  drag increment
      wdh = Mydrag - Mydown;
      WarpC_warpfunc(wdx,wdy,wdw,wdh,0);                                   //  drag image
      WarpCdrag = 1;
      return;
   }
   
   else if (WarpCdrag) 
   {
      WarpCdrag = 0;
      WarpC_warpfunc(wdx,wdy,wdw,wdh,1);                                   //  drag done, add to memory

      if (NWarpC == 100)                                                   //  if full, throw away oldest
      {
         NWarpC = 99;
         for (ii = 0; ii < NWarpC; ii++)
         {
            WarpCmem[0][ii] = WarpCmem[0][ii+1];
            WarpCmem[1][ii] = WarpCmem[1][ii+1];
            WarpCmem[2][ii] = WarpCmem[2][ii+1];
            WarpCmem[3][ii] = WarpCmem[3][ii+1];
         }
      }

      ii = NWarpC;
      WarpCmem[0][ii] = wdx;                                               //  save drag for undo
      WarpCmem[1][ii] = wdy;
      WarpCmem[2][ii] = wdw;
      WarpCmem[3][ii] = wdh;
      NWarpC++;
   }
   
   return;
}


//  warp image and accumulate warp memory
//  mouse at (mx,my) is moved (mw,mh) pixels

void  WarpC_warpfunc(float mx, float my, float mw, float mh, int acc)
{
   int         ii, px, py, vstat;
   double      mag, dispx, dispy;
   double      d1, d2, d3, d4;
   uint16      vpix[3], *pix3;
   
   edit_zapredo();                                                         //  delete redo copy    v.10.3
   
   d1 = (mx-0) * (mx-0) + (my-0) * (my-0);                                 //  distance, mouse to 4 corners
   d2 = (E3ww-mx) * (E3ww-mx) + (my-0) * (my-0);
   d3 = (E3ww-mx) * (E3ww-mx) + (E3hh-my) * (E3hh-my);
   d4 = (mx-0) * (mx-0) + (E3hh-my) * (E3hh-my);
   
   if (d2 > d1) d1 = d2;                                                   //  d1 = greatest       v.10.11
   if (d3 > d1) d1 = d3;
   if (d4 > d1) d1 = d4;

   for (py = 0; py < E3hh; py++)                                           //  process all pixels
   for (px = 0; px < E3ww; px++)
   {
      d2 = (px-mx)*(px-mx) + (py-my)*(py-my);
      mag = (1.0 - d2 / d1);
      mag = mag * mag;                                                     //  faster than pow(mag,16);
      mag = mag * mag;
      //mag = mag * mag;

      dispx = -mw * mag;                                                   //  displacement = drag * mag
      dispy = -mh * mag;
      
      ii = py * E3ww + px;

      if (acc) {                                                           //  drag done, accumulate drag sum
         WarpCx[ii] += dispx;
         WarpCy[ii] += dispy;
         continue;
      }

      dispx += WarpCx[ii];                                                 //  add this drag to prior sum
      dispy += WarpCy[ii];

      vstat = vpixel(E1pxm16,px+dispx,py+dispy,vpix);                      //  input virtual pixel
      pix3 = PXMpix(E3pxm16,px,py);                                        //  output pixel
      if (vstat) {
         pix3[0] = vpix[0];
         pix3[1] = vpix[1];
         pix3[2] = vpix[2];
      }
      else pix3[0] = pix3[1] = pix3[2] = 0;
   }

   Fmodified = 1;
   mwpaint2();                                                             //  update window
   return;
}


/**************************************************************************/

//  warp/distort whole image with a linear transform
//  fix perspective problems (e.g. curved walls, leaning buildings)

float       *WarpLx, *WarpLy;                                              //  memory of all dragged pixels
float       WarpLmem[4][100];                                              //  undo memory, last 100 drags
int         NWarpL;                                                        //  WarpLmem count
int         WarpLdrag;
int         WarpLww, WarpLhh;

void  WarpL_warpfunc(float wdx, float wdy, float wdw, float wdh, int acc);


void m_warp_linear(GtkWidget *, cchar *)                                   //  new v.10.11
{
   int      WarpL_dialog_event(zdialog *zd, cchar *event);
   void     WarpL_mousefunc(void);

   cchar  *WarpL_message = ZTX(
             " Pull an image position using the mouse. \n"
             " Make multiple mouse pulls until satisfied. \n"
             " When finished, press [done]."); 
   
   int         px, py, ii;

   zfuncs::F1_help_topic = "warp_linear";

   if (! edit_setup("warp_linear",1,0)) return;                            //  setup edit: use preview

   zdedit = zdialog_new(ZTX("Warp Image (linear)"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",WarpL_message,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"button","undlast","hb1",Bundolast,"space=5");
   zdialog_add_widget(zdedit,"button","undall","hb1",Bundoall,"space=5");

   zdialog_run(zdedit,WarpL_dialog_event);                                 //  run dialog

   NWarpL = WarpLdrag = 0;                                                 //  no drag data

   WarpLx = (float *) zmalloc(E3ww * E3hh * sizeof(float),"warp.ix");      //  get memory for pixel displacements
   WarpLy = (float *) zmalloc(E3ww * E3hh * sizeof(float),"warp.iy");
   
   for (py = 0; py < E3hh; py++)                                           //  no pixel displacements
   for (px = 0; px < E3ww; px++)
   {
      ii = py * E3ww + px;
      WarpLx[ii] = WarpLy[ii] = 0.0;
   }
   
   WarpLww = E3ww;                                                         //  preview dimensions
   WarpLhh = E3hh;
   
   takeMouse(zdedit,WarpL_mousefunc,dragcursor);                           //  connect mouse function          v.11.03
   return;
}


//  dialog event and completion callback function

int WarpL_dialog_event(zdialog * zd, cchar *event)
{
   int         px, py, ii;
   float       wdx, wdy, wdw, wdh;
   int         fpx, fpy, epx, epy, vstat;
   double      scale, dispx, dispy;
   uint16      vpix[3], *pix3;

   if (zd->zstat) goto complete;

   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3

   if (strEqu(event,"undlast")) 
   {
      if (NWarpL == 1) event = "undall";
      else if (NWarpL) {                                                   //  undo most recent drag
         ii = --NWarpL;
         wdx = WarpLmem[0][ii];
         wdy = WarpLmem[1][ii];
         wdw = WarpLmem[2][ii];
         wdh = WarpLmem[3][ii];
         WarpL_warpfunc(wdx,wdy,-wdw,-wdh,0);                              //  undrag image
         WarpL_warpfunc(wdx,wdy,-wdw,-wdh,1);                              //  undrag memory
      }
   }

   if (strEqu(event,"undall"))                                             //  undo all drags
   {
      NWarpL = 0;                                                          //  erase undo memory

      for (py = 0; py < E3hh; py++)                                        //  reset pixel displacements
      for (px = 0; px < E3ww; px++)
      {
         ii = py * E3ww + px;
         WarpLx[ii] = WarpLy[ii] = 0.0;
      }
      edit_reset();                                                        //  restore image 1  v.10.3
   }

   return 1;

complete:

   freeMouse();                                                            //  disconnect mouse

   if (zd->zstat != 1) edit_cancel();
   else if (NWarpL == 0) edit_cancel();
   else 
   {
      edit_fullsize();                                                     //  get full-size E1/E3

      scale = 1.0 * (E3ww + E3hh) / (WarpLww + WarpLhh);

      for (fpy = 0; fpy < E3hh; fpy++)                                     //  scale net pixel displacements
      for (fpx = 0; fpx < E3ww; fpx++)                                     //    to full image size
      {
         epx = WarpLww * fpx / E3ww;
         epy = WarpLhh * fpy / E3hh;
         ii = epy * WarpLww + epx;
         dispx = WarpLx[ii] * scale;
         dispy = WarpLy[ii] * scale;

         vstat = vpixel(E1pxm16,fpx+dispx,fpy+dispy,vpix);                 //  input virtual pixel
         pix3 = PXMpix(E3pxm16,fpx,fpy);                                   //  output pixel
         if (vstat) {
            pix3[0] = vpix[0];
            pix3[1] = vpix[1];
            pix3[2] = vpix[2];
         }
         else pix3[0] = pix3[1] = pix3[2] = 0;
      }

      edit_done();
   }

   zfree(WarpLx);                                                          //  release memory
   zfree(WarpLy);
   return 0;
}


//  WarpL mouse function

void  WarpL_mousefunc(void)
{
   static float   wdx, wdy, wdw, wdh;
   int            ii;

   if (Mxdrag || Mydrag)                                                   //  mouse drag underway
   {
      wdx = Mxdown;                                                        //  drag origin, window coordinates
      wdy = Mydown;
      wdw = Mxdrag - Mxdown;                                               //  drag increment
      wdh = Mydrag - Mydown;
      WarpL_warpfunc(wdx,wdy,wdw,wdh,0);                                   //  drag image
      WarpLdrag = 1;
      return;
   }
   
   else if (WarpLdrag) 
   {
      WarpLdrag = 0;
      WarpL_warpfunc(wdx,wdy,wdw,wdh,1);                                   //  drag done, add to memory

      if (NWarpL == 100)                                                   //  if full, throw away oldest
      {
         NWarpL = 99;
         for (ii = 0; ii < NWarpL; ii++)
         {
            WarpLmem[0][ii] = WarpLmem[0][ii+1];
            WarpLmem[1][ii] = WarpLmem[1][ii+1];
            WarpLmem[2][ii] = WarpLmem[2][ii+1];
            WarpLmem[3][ii] = WarpLmem[3][ii+1];
         }
      }

      ii = NWarpL;
      WarpLmem[0][ii] = wdx;                                               //  save drag for undo
      WarpLmem[1][ii] = wdy;
      WarpLmem[2][ii] = wdw;
      WarpLmem[3][ii] = wdh;
      NWarpL++;
   }
   
   return;
}


//  warp image and accumulate warp memory
//  mouse at (mx,my) is moved (mw,mh) pixels

void  WarpL_warpfunc(float mx, float my, float mw, float mh, int acc)
{
   int         ii, px, py, vstat;
   double      mag, dispx, dispy;
   double      d1, d2, d3, d4;
   uint16      vpix[3], *pix3;
   
   edit_zapredo();                                                         //  delete redo copy    v.10.3
   
   d1 = (mx-0) * (mx-0) + (my-0) * (my-0);                                 //  distance, mouse to 4 corners
   d2 = (E3ww-mx) * (E3ww-mx) + (my-0) * (my-0);
   d3 = (E3ww-mx) * (E3ww-mx) + (E3hh-my) * (E3hh-my);
   d4 = (mx-0) * (mx-0) + (E3hh-my) * (E3hh-my);
   
   if (d2 > d1) d1 = d2;                                                   //  d1 = greatest       v.10.11
   if (d3 > d1) d1 = d3;
   if (d4 > d1) d1 = d4;
   
   d1 = sqrt(d1);

   for (py = 0; py < E3hh; py++)                                           //  process all pixels
   for (px = 0; px < E3ww; px++)
   {
      d2 = (px-mx)*(px-mx) + (py-my)*(py-my);
      d2 = sqrt(d2);
      mag = (1.0 - d2 / d1);

      dispx = -mw * mag;                                                   //  displacement = drag * mag
      dispy = -mh * mag;
      
      ii = py * E3ww + px;

      if (acc) {                                                           //  drag done, accumulate drag sum
         WarpLx[ii] += dispx;
         WarpLy[ii] += dispy;
         continue;
      }

      dispx += WarpLx[ii];                                                 //  add this drag to prior sum
      dispy += WarpLy[ii];

      vstat = vpixel(E1pxm16,px+dispx,py+dispy,vpix);                      //  input virtual pixel
      pix3 = PXMpix(E3pxm16,px,py);                                        //  output pixel
      if (vstat) {
         pix3[0] = vpix[0];
         pix3[1] = vpix[1];
         pix3[2] = vpix[2];
      }
      else pix3[0] = pix3[1] = pix3[2] = 0;
   }

   Fmodified = 1;
   mwpaint2();                                                             //  update window
   return;
}


/**************************************************************************/

//  warp/distort whole image using affine transform 
//  (straight lines remain straight)

double      WarpF_old[3][2];                                               //  3 original image points 
double      WarpF_new[3][2];                                               //  corresponding warped points
double      WarpF_coeff[6];                                                //  transform coefficients
double      WarpF_Icoeff[6];                                               //  inverse transform coefficients
int         WarpF_ftf;                                                     //  first time flag

void  WarpF_warpfunc();                                                    //  image warp function
void  WarpF_affine(double po[3][2], double pn[3][2], double coeff[6]);     //  compute affine transform coefficients
void  WarpF_invert(double coeff[6], double Icoeff[6]);                     //  compute reverse transform coefficients


void m_warp_affine(GtkWidget *, cchar *)
{
   int      WarpF_dialog_event(zdialog *zd, cchar *event);
   void     WarpF_mousefunc(void);

   cchar  *WarpF_message = ZTX(
             " Pull on an image corner using the mouse. \n"
             " Make multiple mouse pulls until satisfied. \n"
             " When finished, press [done]."); 
   
   zfuncs::F1_help_topic = "warp_affine";

   if (! edit_setup("warp_affine",1,0)) return;                            //  setup edit: use preview

   zdedit = zdialog_new(ZTX("Warp Image (affine)"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",WarpF_message,"space=5");

   zdialog_run(zdedit,WarpF_dialog_event);                                 //  run dialog, parallel

   WarpF_ftf = 1;                                                          //  1st warp flag

   takeMouse(zdedit,WarpF_mousefunc,dragcursor);                           //  connect mouse function          v.11.03
   return;
}


//  dialog event and completion callback function

int WarpF_dialog_event(zdialog *zd, cchar *event)
{
   double      scale;
   int         ww, hh;

   if (zd->zstat) goto complete;

   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   return 1;

complete:

   freeMouse();                                                            //  disconnect mouse

   if (zd->zstat != 1) edit_cancel();
   
   else 
   {
      ww = E3ww;                                                           //  preview image dimensions
      hh = E3hh;

      edit_fullsize();                                                     //  get full-size images

      scale = 1.0 * (E3ww + E3hh) / (ww + hh);                             //  preview to full-size scale factor
      
      WarpF_old[0][0] = WarpF_old[0][0] * scale;                           //  re-scale new and old points
      WarpF_old[0][1] = WarpF_old[0][1] * scale;
      WarpF_old[1][0] = WarpF_old[1][0] * scale;
      WarpF_old[1][1] = WarpF_old[1][1] * scale;
      WarpF_old[2][0] = WarpF_old[2][0] * scale;
      WarpF_old[2][1] = WarpF_old[2][1] * scale;

      WarpF_new[0][0] = WarpF_new[0][0] * scale;
      WarpF_new[0][1] = WarpF_new[0][1] * scale;
      WarpF_new[1][0] = WarpF_new[1][0] * scale;
      WarpF_new[1][1] = WarpF_new[1][1] * scale;
      WarpF_new[2][0] = WarpF_new[2][0] * scale;
      WarpF_new[2][1] = WarpF_new[2][1] * scale;
      
      WarpF_warpfunc();                                                    //  warp full-size image
      edit_done();
   }

   return 0;
}


//  WarpF mouse function

void  WarpF_mousefunc(void)
{
   int      mdx1, mdy1, mdx2, mdy2;
   double   x1o, y1o, x2o, y2o, x3o, y3o;
   double   x1n, y1n, x2n, y2n, x3n, y3n;
   double   a, b, c, d, e, f;
   
   if (Mxdrag + Mydrag == 0) return;

   mdx1 = Mxdown;                                                          //  mouse drag origin
   mdy1 = Mydown;
   mdx2 = Mxdrag;                                                          //  mouse drag position
   mdy2 = Mydrag;

   Mxdown = Mxdrag;                                                        //  reset origin for next time
   Mydown = Mydrag;
   
   x1n = mdx1;                                                             //  point 1 = drag origin
   y1n = mdy1;
   x2n = E3ww - x1n;                                                       //  point 2 = mirror of point1
   y2n = E3hh - y1n;
   x3n = E3ww * (y2n / E3hh);
   y3n = E3hh * (1.0 - (x2n / E3ww));

   if (WarpF_ftf)                                                          //  first warp
   {
      WarpF_ftf = 0;
      x1o = x1n;                                                           //  old = current positions
      y1o = y1n;
      x2o = x2n;
      y2o = y2n;
      x3o = x3n;
      y3o = y3n;
   }
   else
   {
      WarpF_invert(WarpF_coeff,WarpF_Icoeff);                              //  get inverse coefficients
      a = WarpF_Icoeff[0];
      b = WarpF_Icoeff[1];
      c = WarpF_Icoeff[2];
      d = WarpF_Icoeff[3];
      e = WarpF_Icoeff[4];
      f = WarpF_Icoeff[5];
      
      x1o = a * x1n + b * y1n + c;                                         //  compute old from current positions
      y1o = d * x1n + e * y1n + f;
      x2o = a * x2n + b * y2n + c;
      y2o = d * x2n + e * y2n + f;
      x3o = a * x3n + b * y3n + c;
      y3o = d * x3n + e * y3n + f;
   }
      
   WarpF_old[0][0] = x1o;                                                  //  set up 3 old points and corresponding
   WarpF_old[0][1] = y1o;                                                  //    new points for affine translation
   WarpF_old[1][0] = x2o;
   WarpF_old[1][1] = y2o;
   WarpF_old[2][0] = x3o;
   WarpF_old[2][1] = y3o;

   x1n = mdx2;                                                             //  point 1 new position = drag position
   y1n = mdy2;
   x2n = E3ww - x1n;                                                       //  point 2 new = mirror of point1 new
   y2n = E3hh - y1n;

   WarpF_new[0][0] = x1n;                                                  //  3 new points 
   WarpF_new[0][1] = y1n;
   WarpF_new[1][0] = x2n;
   WarpF_new[1][1] = y2n;
   WarpF_new[2][0] = x3n;
   WarpF_new[2][1] = y3n;
   
   WarpF_warpfunc();                                                       //  do the warp

   return;
}


//  warp image and accumulate warp memory

void  WarpF_warpfunc()
{
   double      a, b, c, d, e, f;
   int         px3, py3, vstat;
   double      px1, py1;
   uint16      vpix1[3], *pix3;
   
   edit_zapredo();                                                         //  delete redo copy    v.10.3

   WarpF_affine(WarpF_old, WarpF_new, WarpF_coeff);                        //  get coefficients for forward transform
   WarpF_invert(WarpF_coeff, WarpF_Icoeff);                                //  get coefficients for reverse transform
   
   a = WarpF_Icoeff[0];                                                    //  coefficients to map output pixels
   b = WarpF_Icoeff[1];                                                    //    to corresponding input pixels
   c = WarpF_Icoeff[2];
   d = WarpF_Icoeff[3];
   e = WarpF_Icoeff[4];
   f = WarpF_Icoeff[5];
   
   for (py3 = 0; py3 < E3hh; py3++)                                        //  loop all output pixels
   for (px3 = 0; px3 < E3ww; px3++)
   {
      px1 = a * px3 + b * py3 + c;                                         //  corresponding input pixel
      py1 = d * px3 + e * py3 + f;

      vstat = vpixel(E1pxm16,px1,py1,vpix1);                               //  input virtual pixel
      pix3 = PXMpix(E3pxm16,px3,py3);                                      //  output pixel

      if (vstat) {
         pix3[0] = vpix1[0];
         pix3[1] = vpix1[1];
         pix3[2] = vpix1[2];
      }
      else pix3[0] = pix3[1] = pix3[2] = 0;
   }

   Fmodified = 1;
   mwpaint2();                                                             //  update window
   return;
}


/**************************************************************************

   Compute affine transformation of an image (warp image).

   Given 3 new (warped) positions for 3 image points, derive the 
   coefficients of the translation function to warp the entire image.

   Inputs:
      pold[3][2]  (x,y) coordinates for 3 points in original image
      pnew[3][2]  (x,y) coordinates for same points in warped image
   
   Output: 
      coeff[6]  coefficients of translation function which can be used
                to convert all image points to their warped positions

   If coeff[6] = (a, b, c, d, e, f) then the following formula
   can be used to convert an image point to its warped position:

      Xnew = a * Xold + b * Yold + c
      Ynew = d * Xold + e * Yold + f

***************************************************************************/

void WarpF_affine(double pold[3][2], double pnew[3][2], double coeff[6])
{
   double   x11, y11, x12, y12, x13, y13;                                  //  original points
   double   x21, y21, x22, y22, x23, y23;                                  //  moved points
   double   a, b, c, d, e, f;                                              //  coefficients
   double   A1, A2, B1, B2, C1, C2;
   
   x11 = pold[0][0];
   y11 = pold[0][1];
   x12 = pold[1][0];
   y12 = pold[1][1];
   x13 = pold[2][0];
   y13 = pold[2][1];

   x21 = pnew[0][0];
   y21 = pnew[0][1];
   x22 = pnew[1][0];
   y22 = pnew[1][1];
   x23 = pnew[2][0];
   y23 = pnew[2][1];
   
   A1 = x11 - x12;
   A2 = x12 - x13;
   B1 = y11 - y12;
   B2 = y12 - y13;
   C1 = x21 - x22;
   C2 = x22 - x23;
   
   a = (B1 * C2 - B2 * C1) / (A2 * B1 - A1 * B2);
   b = (A1 * C2 - A2 * C1) / (A1 * B2 - A2 * B1);
   c = x23 - a * x13 - b * y13;
   
   C1 = y21 - y22;
   C2 = y22 - y23;
   
   d = (B1 * C2 - B2 * C1) / (A2 * B1 - A1 * B2);
   e = (A1 * C2 - A2 * C1) / (A1 * B2 - A2 * B1);
   f = y23 - d * x13 - e * y13;
   
   coeff[0] = a;
   coeff[1] = b;
   coeff[2] = c;
   coeff[3] = d;
   coeff[4] = e;
   coeff[5] = f;

   return;
}   


/**************************************************************************

   Invert affine transform

   Input:
      coeff[6]  coefficients of translation function to convert
                image points to their warped positions
   Output: 
      Icoeff[6]  coefficients of translation function to convert
                 warped image points to their original positions

   If Icoeff[6] = (a, b, c, d, e, f) then the following formula can be
      used to translate a warped image point to its original position:

      Xold = a * Xnew + b * Ynew + c
      Yold = d * Xnew + e * Ynew + f

***************************************************************************/

void WarpF_invert(double coeff[6], double Icoeff[6])
{
   double   a, b, c, d, e, f, Z;
   
   a = coeff[0];
   b = coeff[1];
   c = coeff[2];
   d = coeff[3];
   e = coeff[4];
   f = coeff[5];
   
   Z = 1.0 / (a * e - b * d);
   
   Icoeff[0] = e * Z;
   Icoeff[1] = - b * Z;
   Icoeff[2] = Z * (b * f - c * e);
   Icoeff[3] = - d * Z;
   Icoeff[4] = a * Z;
   Icoeff[5] = Z * (c * d - a * f);
   
   return;
}


/**************************************************************************/

//  image color-depth reduction

int      colordep_depth = 16;                                              //  bits per RGB color


void m_colordep(GtkWidget *, cchar *)
{
   int    colordep_dialog_event(zdialog *zd, cchar *event);
   void * colordep_thread(void *);

   cchar  *colmess = ZTX("Set color depth to 1-16 bits");
   
   zfuncs::F1_help_topic = "color_depth";                                  //  v.10.8

   if (! edit_setup("colordepth",1,2)) return;                             //  setup edit: preview

   zdedit = zdialog_new(ZTX("Set Color Depth"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",colmess,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"spin","colors","hb1","1|16|1|16","space=5");

   zdialog_run(zdedit,colordep_dialog_event);                              //  run dialog - parallel
   
   colordep_depth = 16;
   start_thread(colordep_thread,0);                                        //  start working thread
   return;
}


//  dialog event and completion callback function

int colordep_dialog_event(zdialog *zd, cchar *event)
{
   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      return 0;
   }

   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3

   if (strEqu(event,"colors")) {
      zdialog_fetch(zd,"colors",colordep_depth);
      signal_thread();
   }
   
   return 0;
}


//  image color depth thread function

void * colordep_thread(void *)
{
   int         ii, px, py, rgb, dist = 0;
   uint16      m1, m2, val1, val3;
   uint16      *pix1, *pix3;
   double      fmag, f1, f2;
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
   
      m1 = 0xFFFF << (16 - colordep_depth);                                //  5 > 1111100000000000
      m2 = 0x8000 >> colordep_depth;                                       //  5 > 0000010000000000
      
      fmag = 65535.0 / m1;

      for (py = 0; py < E3hh; py++)
      for (px = 0; px < E3ww; px++)
      {
         if (Factivearea) {                                                //  select area active
            ii = py * Fww + px;
            dist = sa_pixisin[ii];                                         //  distance from edge
            if (! dist) continue;                                          //  outside pixel
         }

         pix1 = PXMpix(E1pxm16,px,py);                                     //  input pixel
         pix3 = PXMpix(E3pxm16,px,py);                                     //  output pixel
         
         for (rgb = 0; rgb < 3; rgb++)
         {
            val1 = pix1[rgb];
            if (val1 < m1) val3 = (val1 + m2) & m1;
            else val3 = m1;
            val3 = uint(val3 * fmag);

            if (Factivearea && dist < sa_blend) {                          //  select area is active,
               f2 = 1.0 * dist / sa_blend;                                 //    blend changes over sa_blend
               f1 = 1.0 - f2;
               val3 = int(f1 * val1 + f2 * val3);
            }

            pix3[rgb] = val3;
         }
      }

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


/**************************************************************************/

//  convert image to simulate a drawing

int      draw_contrast;
int      draw_threshold;
int      draw_pixcon;
int      draw_reverse;
double   draw_trfunc[256];
double   draw_pixcon3;
uint8    *draw_pixcon_map = 0;

void m_draw(GtkWidget *, cchar *)
{
   int    draw_dialog_event(zdialog* zd, cchar *event);
   void * draw_thread(void *);

   cchar       *title = ZTX("Simulate Drawing");
   uint16      *pix1, *pix2;
   int         ii, px, py, qx, qy;
   int         red, green, blue, con, maxcon;

   zfuncs::F1_help_topic = "drawing";                                      //  v.10.8

   if (! edit_setup("draw",0,2)) return;                                   //  setup edit: no preview

   draw_pixcon_map = (uint8 *) zmalloc(E1ww*E1hh,"pixconmap");             //  set up pixel contrast map
   memset(draw_pixcon_map,0,E1ww*E1hh);

   for (py = 1; py < E1hh-1; py++)                                         //  scan image pixels
   for (px = 1; px < E1ww-1; px++)
   {
      pix1 = PXMpix(E1pxm16,px,py);                                        //  pixel at (px,py)
      red = pix1[0];                                                       //  pixel RGB levels
      green = pix1[1];
      blue = pix1[2];
      maxcon = 0;

      for (qy = py-1; qy < py+2; qy++)                                     //  loop 3x3 block of neighbor
      for (qx = px-1; qx < px+2; qx++)                                     //    pixels around pix1
      {
         pix2 = PXMpix(E1pxm16,qx,qy);                                     //  find max. contrast with
         con = abs(red-pix2[0]) + abs(green-pix2[1]) + abs(blue-pix2[2]);  //    neighbor pixel
         if (con > maxcon) maxcon = con;
      }

      ii = py * E1ww + px;
      draw_pixcon_map[ii] = (maxcon/3) >> 8;                               //  contrast for (px,py) 0-255
   }

   zdedit = zdialog_new(title,mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"vbox","vb1","hb1",0,"homog");
   zdialog_add_widget(zdedit,"vbox","vb2","hb1",0,"homog|expand");
   zdialog_add_widget(zdedit,"label","lab1","vb1",ZTX("contrast"));
   zdialog_add_widget(zdedit,"label","lab2","vb1",Bthresh);
   zdialog_add_widget(zdedit,"label","lab3","vb1",ZTX("outlines"));
   zdialog_add_widget(zdedit,"hscale","contrast","vb2","0|100|1|0","expand");
   zdialog_add_widget(zdedit,"hscale","threshold","vb2","0|100|1|0","expand");
   zdialog_add_widget(zdedit,"hscale","pixcon","vb2","0|255|1|0","expand");
   zdialog_add_widget(zdedit,"hbox","hb4","dialog");
   zdialog_add_widget(zdedit,"radio","pencil","hb4",ZTX("pencil"),"space=10");
   zdialog_add_widget(zdedit,"radio","chalk","hb4",ZTX("chalk"),"space=10");

   zdialog_resize(zdedit,300,0);
   zdialog_run(zdedit,draw_dialog_event);                                  //  run dialog - parallel
   
   start_thread(draw_thread,0);                                            //  start working thread
   return;
}


//  dialog event and completion callback function

int draw_dialog_event(zdialog *zd, cchar *event)                           //  draw dialog event function
{
   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      zfree(draw_pixcon_map);
      return 0;
   }

   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3

   if (strcmpv(event,"contrast","threshold","pixcon","chalk",null)) 
   {
      zdialog_fetch(zd,"contrast",draw_contrast);                          //  get slider values
      zdialog_fetch(zd,"threshold",draw_threshold);
      zdialog_fetch(zd,"pixcon",draw_pixcon);
      zdialog_fetch(zd,"chalk",draw_reverse);
      signal_thread();                                                     //  trigger update thread
   }

   return 1;
}


//  thread function - use multiple working threads

void * draw_thread(void *)
{
   void  * draw_wthread(void *arg);

   int         ii;
   double      threshold, contrast, trf;
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      threshold = 0.01 * draw_threshold;                                   //  range 0 to 1
      contrast = 0.01 * draw_contrast;                                     //  range 0 to 1
      
      for (ii = 0; ii < 256; ii++)                                         //  brightness transfer function
      {
         trf = 1.0 - 0.003906 * (256 - ii) * contrast;                     //  ramp-up from 0-1 to 1
         if (ii < 256 * threshold) trf = 0;                                //  0 if below threshold
         draw_trfunc[ii] = trf;
      }
      
      for (ii = 0; ii < Nwt; ii++)                                         //  start worker threads
         start_wt(draw_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * draw_wthread(void *arg)                                             //  worker thread function
{
   void  draw_1pix(int px, int py);

   int         index = *((int *) (arg));
   int         px, py;
   double      pixcon;

   pixcon = draw_pixcon / 255.0;                                           //  0-1 linear ramp
   draw_pixcon3 = 255 * pixcon * pixcon * pixcon;                          //  0-255 cubic ramp

   for (py = index+1; py < E1hh-1; py += Nwt)                              //  process all pixels
   for (px = 1; px < E1ww-1; px++)
      draw_1pix(px,py);

   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


void draw_1pix(int px, int py)                                             //  process one pixel
{
   uint16      *pix1, *pix3;
   int         ii, dist = 0;
   int         bright1, bright2;
   int         red1, green1, blue1;
   int         red3, green3, blue3;
   double      dold, dnew;
   double      pixcon = draw_pixcon3;

   if (Factivearea) {                                                      //  select area active
      ii = py * Fww + px;
      dist = sa_pixisin[ii];                                               //  distance from edge
      if (! dist) return;                                                  //  outside pixel
   }
   
   pix1 = PXMpix(E1pxm16,px,py);                                           //  input pixel
   pix3 = PXMpix(E3pxm16,px,py);                                           //  output pixel
      
   red1 = pix1[0];
   green1 = pix1[1];
   blue1 = pix1[2];
   
   bright1 = ((red1 + green1 + blue1) / 3) >> 8;                           //  old brightness  0-255
   bright2 = bright1 * draw_trfunc[bright1];                               //  new brightness  0-255
   
   ii = py * E1ww + px;
   if (draw_pixcon_map[ii] < pixcon) bright2 = 255;

   if (pixcon > 1 && bright2 > draw_threshold) bright2 = 255;              //  empirical !!!

   if (draw_reverse) bright2 = 255 - bright2;                              //  negate if "chalk"

   red3 = green3 = blue3 = bright2 << 8;                                   //  gray scale, new brightness

   if (Factivearea && dist < sa_blend) {                                   //  select area is active,
      dnew = 1.0 * dist / sa_blend;
      dold = 1.0 - dnew;
      red3 = dnew * red3 + dold * red1;
      green3 = dnew * green3 + dold * green1;
      blue3 = dnew * blue3 + dold * blue1;
   }

   pix3[0] = red3;
   pix3[1] = green3;
   pix3[2] = blue3;
   
   return;
}


/**************************************************************************/

//  make an image outline drawing from edge pixels
//  superimpose outlines and image and vary brightness of each

double   outlines_olth, outlines_olww, outlines_imbr;

void m_outlines(GtkWidget *, cchar *)
{
   int    outlines_dialog_event(zdialog* zd, cchar *event);
   void * outlines_thread(void *);

   cchar  *title = ZTX("Add Image Outlines");

   zfuncs::F1_help_topic = "outlines";
   
   if (! edit_setup("outlines",0,2)) return;                               //  setup edit: no preview, select area OK

   zdedit = zdialog_new(title,mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","lab1","hb1",ZTX("outline threshold"),"space=5");
   zdialog_add_widget(zdedit,"hscale","olth","hb1","0|100|1|90","expand|space=5");   
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","lab2","hb2",ZTX("outline width"),"space=5");
   zdialog_add_widget(zdedit,"hscale","olww","hb2","0|100|1|50","expand|space=5");
   zdialog_add_widget(zdedit,"hbox","hb3","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","lab3","hb3",ZTX("image brightness"),"space=5");
   zdialog_add_widget(zdedit,"hscale","imbr","hb3","0|100|1|10","expand|space=5");
   
   outlines_olth = 90;
   outlines_olww = 50;
   outlines_imbr = 10;

   zdialog_resize(zdedit,300,0);
   zdialog_run(zdedit,outlines_dialog_event);                              //  run dialog - parallel
   start_thread(outlines_thread,0);                                        //  start working thread
   signal_thread();
   return;
}


//  dialog event and completion callback function

int outlines_dialog_event(zdialog *zd, cchar *event)                       //  dialog event function
{
   if (zd->zstat)
   {
      if (zd->zstat == 1) edit_done();                                     //  done
      else edit_cancel();                                                  //  cancel or destroy
      PXM_free(E8pxm16);
      PXM_free(E9pxm16);
      return 0;
   }
   
   if (strEqu(event,"olth")) {
      zdialog_fetch(zd,"olth",outlines_olth);                              //  get outline threshold 0-100
      signal_thread();
   }

   if (strEqu(event,"olww")) {
      zdialog_fetch(zd,"olww",outlines_olww);                              //  get outline width 0-100
      signal_thread();
   }
   
   if (strEqu(event,"imbr")) {
      zdialog_fetch(zd,"imbr",outlines_imbr);                              //  get image brightness 0-100
      signal_thread();
   }
   
   if (strEqu(event,"undo")) edit_undo();
   if (strEqu(event,"redo")) edit_redo();
   if (strEqu(event,"blendwidth")) signal_thread();

   return 1;
}


//  thread function to update image

void * outlines_thread(void *)
{
   void * outlines_wthread(void *arg);
   
   int         px, py, ww, hh;
   double      olww, red3, green3, blue3;
   double      red3h, red3v, green3h, green3v, blue3h, blue3v;
   uint16      *pix8, *pix9;
   uint16      *pixa, *pixb, *pixc, *pixd, *pixe, *pixf, *pixg, *pixh;

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      olww = 0.01 * outlines_olww;                                         //  outline width, 0 - 1.0
      olww = 1.0 - olww;                                                   //  1.0 - 0
      olww = 0.8 * olww + 0.2;                                             //  1.0 - 0.2

      ww = Fww * olww + 0.5;                                               //  create smaller outline image
      hh = Fhh * olww + 0.5;
      
      if (! E9pxm16 || ww != E9pxm16->ww)                                  //  initial or changed outline brightness
      {
         PXM_free(E8pxm16);
         PXM_free(E9pxm16);

         E8pxm16 = PXM_rescale(E1pxm16,ww,hh);
         E9pxm16 = PXM_copy(E8pxm16);

         for (py = 1; py < hh-1; py++)
         for (px = 1; px < ww-1; px++)
         {
            pix8 = PXMpix(E8pxm16,px,py);                                  //  input pixel
            pix9 = PXMpix(E9pxm16,px,py);                                  //  output pixel
            
            pixa = pix8-3*ww-3;                                            //  8 neighboring pixels are used
            pixb = pix8-3*ww;                                              //    to get edge brightness using
            pixc = pix8-3*ww+3;                                            //       a Sobel filter
            pixd = pix8-3;
            pixe = pix8+3;
            pixf = pix8+3*ww-3;
            pixg = pix8+3*ww;
            pixh = pix8+3*ww+3;
            
            red3h = -pixa[0] -2 * pixb[0] -pixc[0] + pixf[0] + 2 * pixg[0] + pixh[0];
            red3v = -pixa[0] -2 * pixd[0] -pixf[0] + pixc[0] + 2 * pixe[0] + pixh[0];
            green3h = -pixa[1] -2 * pixb[1] -pixc[1] + pixf[1] + 2 * pixg[1] + pixh[1];
            green3v = -pixa[1] -2 * pixd[1] -pixf[1] + pixc[1] + 2 * pixe[1] + pixh[1];
            blue3h = -pixa[2] -2 * pixb[2] -pixc[2] + pixf[2] + 2 * pixg[2] + pixh[2];
            blue3v = -pixa[2] -2 * pixd[2] -pixf[2] + pixc[2] + 2 * pixe[2] + pixh[2];
            
            red3 = (abs(red3h) + abs(red3v)) / 2;                          //  average vertical and horizontal brightness
            green3 = (abs(green3h) + abs(green3v)) / 2;
            blue3 = (abs(blue3h) + abs(blue3v)) / 2;
            
            if (red3 > 65535) red3 = 65535;
            if (green3 > 65535) green3 = 65535;
            if (blue3 > 65535) blue3 = 65535;

            pix9[0] = red3;
            pix9[1] = green3;
            pix9[2] = blue3;
         }
      }
      
      PXM_free(E8pxm16);                                                   //  scale back to full-size
      E8pxm16 = PXM_rescale(E9pxm16,Fww,Fhh);

      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker threads
         start_wt(outlines_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * outlines_wthread(void *arg)                                         //  worker thread function
{
   int         index = *((int *) arg);
   int         px, py, ii, dist = 0;
   double      olth, imbr, br8, dold, dnew;
   double      red1, green1, blue1, red3, green3, blue3;
   uint16      *pix1, *pix8, *pix3;
   
   olth = 0.01 * outlines_olth;                                            //  outline threshold, 0 - 1.0
   olth = 1.0 - olth;                                                      //                     1.0 - 0
   imbr = 0.01 * outlines_imbr;                                            //  image brightness,  0 - 1.0

   for (py = index+1; py < Fhh-1; py += Nwt)
   for (px = 1; px < Fww-1; px++)
   {
      if (Factivearea) {                                                   //  select area active
         ii = py * Fww + px;
         dist = sa_pixisin[ii];                                            //  distance from edge
         if (! dist) continue;                                             //  pixel outside area
      }
      
      pix1 = PXMpix(E1pxm16,px,py);                                        //  input image pixel
      pix8 = PXMpix(E8pxm16,px,py);                                        //  input outline pixel
      pix3 = PXMpix(E3pxm16,px,py);                                        //  output image pixel
      
      red1 = pix1[0];                                                      //  input image pixel
      green1 = pix1[1];
      blue1 = pix1[2];
      
      br8 = pixbright(pix8);                                               //  outline brightness
      br8 = br8 / 65536.0;                                                 //  scale 0 - 1.0

      if (br8 > olth) {                                                    //  > threshold, use outline pixel
         red3 = pix8[0];
         green3 = pix8[1];
         blue3 = pix8[2];
      }
      else {                                                               //  use image pixel dimmed by user control
         red3 = red1 * imbr;
         green3 = green1 * imbr;
         blue3 = blue1 * imbr;
      }
      
      if (Factivearea && dist < sa_blend) {                                //  select area is active,
         dnew = 1.0 * dist / sa_blend;                                     //    blend changes over sa_blend
         dold = 1.0 - dnew;
         red3 = dnew * red3 + dold * red1;
         green3 = dnew * green3 + dold * green1;
         blue3 = dnew * blue3 + dold * blue1;
      }
      
      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
   }
   
   exit_wt();
   return 0;                                                               //  not executed, stop gcc warning
}


/**************************************************************************/

//  convert image to simulate an embossing

int      emboss_radius, emboss_color;
double   emboss_depth;
double   emboss_kernel[20][20];                                            //  support radius <= 9

void m_emboss(GtkWidget *, cchar *)
{
   int    emboss_dialog_event(zdialog* zd, cchar *event);
   void * emboss_thread(void *);

   cchar  *title = ZTX("Simulate Embossing");

   zfuncs::F1_help_topic = "embossing";                                    //  v.10.8

   if (! edit_setup("emboss",0,2)) return;                                 //  setup edit: no preview

   zdedit = zdialog_new(title,mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"label","lab1","hb1",Bradius,"space=5");
   zdialog_add_widget(zdedit,"spin","radius","hb1","0|9|1|0");
   zdialog_add_widget(zdedit,"label","lab2","hb1",ZTX("depth"),"space=5");
   zdialog_add_widget(zdedit,"spin","depth","hb1","0|99|1|0");
   zdialog_add_widget(zdedit,"check","color","hb1",ZTX("color"),"space=8");

   zdialog_run(zdedit,emboss_dialog_event);                                //  run dialog - parallel
   
   start_thread(emboss_thread,0);                                          //  start working thread
   return;
}


//  dialog event and completion callback function

int emboss_dialog_event(zdialog *zd, cchar *event)                         //  emboss dialog event function
{
   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      return 0;
   }

   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3

   if (strcmpv(event,"radius","depth","color",null))
   {
      zdialog_fetch(zd,"radius",emboss_radius);                            //  get user inputs
      zdialog_fetch(zd,"depth",emboss_depth);
      zdialog_fetch(zd,"color",emboss_color);
      signal_thread();                                                     //  trigger update thread
   }

   return 1;
}


//  thread function - use multiple working threads

void * emboss_thread(void *)
{
   void  * emboss_wthread(void *arg);

   int         ii, dx, dy, rad;
   double      depth, kern, coeff;
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      rad = emboss_radius;
      depth = emboss_depth;

      coeff = 0.1 * depth / (rad * rad + 1);

      for (dy = -rad; dy <= rad; dy++)                                     //  build kernel with radius and depth
      for (dx = -rad; dx <= rad; dx++)
      {
         kern = coeff * (dx + dy);
         emboss_kernel[dx+rad][dy+rad] = kern;
      }
      
      emboss_kernel[rad][rad] = 1;                                         //  kernel center cell = 1

      for (ii = 0; ii < Nwt; ii++)                                         //  start worker threads
         start_wt(emboss_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for comletion

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * emboss_wthread(void *arg)                                           //  worker thread function
{
   void  emboss_1pix(int px, int py);

   int         index = *((int *) (arg));
   int         px, py;

   for (py = index; py < E1hh; py += Nwt)                                  //  process all pixels
   for (px = 0; px < E1ww; px++)
      emboss_1pix(px,py);

   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


void emboss_1pix(int px, int py)                                           //  process one pixel
{
   int         ii, dist = 0;
   int         bright1, bright3;
   int         rgb, dx, dy, rad;
   uint16      *pix1, *pix3, *pixN;
   double      sumpix, kern, dold, dnew;
   
   if (Factivearea) {                                                      //  select area active
      ii = py * Fww + px;
      dist = sa_pixisin[ii];                                               //  distance from edge
      if (! dist) return;                                                  //  outside pixel
   }

   rad = emboss_radius;

   if (px < rad || py < rad) return;
   if (px > E3ww-rad-1 || py > E3hh-rad-1) return;
   
   pix1 = PXMpix(E1pxm16,px,py);                                           //  input pixel
   pix3 = PXMpix(E3pxm16,px,py);                                           //  output pixel
   
   if (emboss_color) 
   {
      for (rgb = 0; rgb < 3; rgb++)
      {      
         sumpix = 0;
         
         for (dy = -rad; dy <= rad; dy++)                                  //  loop surrounding block of pixels
         for (dx = -rad; dx <= rad; dx++)
         {
            pixN = pix1 + (dy * E1ww + dx) * 3;
            kern = emboss_kernel[dx+rad][dy+rad];
            sumpix += kern * pixN[rgb];
      
            bright1 = pix1[rgb];
            bright3 = sumpix;
            if (bright3 < 0) bright3 = 0;
            if (bright3 > 65535) bright3 = 65535;

            if (Factivearea && dist < sa_blend) {                          //  select area is active,
               dnew = 1.0 * dist / sa_blend;                               //    blend changes over sa_blend
               dold = 1.0 - dnew;
               bright3 = dnew * bright3 + dold * bright1;
            }

            pix3[rgb] = bright3;
         }
      }
   }
   
   else                                                                    //  use gray scale
   {
      sumpix = 0;
         
      for (dy = -rad; dy <= rad; dy++)                                     //  loop surrounding block of pixels
      for (dx = -rad; dx <= rad; dx++)
      {
         pixN = pix1 + (dy * E1ww + dx) * 3;
         kern = emboss_kernel[dx+rad][dy+rad];
         sumpix += kern * (pixN[0] + pixN[1] + pixN[2]);
      }
      
      bright1 = 0.3333 * (pix1[0] + pix1[1] + pix1[2]);
      bright3 = 0.3333 * sumpix;
      if (bright3 < 0) bright3 = 0;
      if (bright3 > 65535) bright3 = 65535;
      
      if (Factivearea && dist < sa_blend) {                                //  select area is active,
         dnew = 1.0 * dist / sa_blend;                                     //    blend changes over sa_blend
         dold = 1.0 - dnew;
         bright3 = dnew * bright3 + dold * bright1;
      }

      pix3[0] = pix3[1] = pix3[2] = bright3;
   }

   return;
}


/**************************************************************************/

//  convert image to simulate square tiles

int         tile_size, tile_gap;
uint16      *tile_pixmap = 0;


void m_tiles(GtkWidget *, cchar *)
{
   int    tile_dialog_event(zdialog *zd, cchar *event);
   void * tile_thread(void *);

   zfuncs::F1_help_topic = "tiles";                                        //  v.10.8

   if (! edit_setup("tiles",0,2)) return;                                  //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Simulate Tiles"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labt","hb1",ZTX("tile size"),"space=5");
   zdialog_add_widget(zdedit,"spin","size","hb1","1|99|1|5","space=5");
   zdialog_add_widget(zdedit,"button","apply","hb1",Bapply,"space=10");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labg","hb2",ZTX("tile gap"),"space=5");
   zdialog_add_widget(zdedit,"spin","gap","hb2","0|9|1|1","space=5");

   zdialog_run(zdedit,tile_dialog_event);                                  //  start dialog

   tile_size = 5;
   tile_gap = 1;

   tile_pixmap = (uint16 *) zmalloc(E1ww*E1hh*6,"tile.pixmap");            //  set up pixel color map
   memset(tile_pixmap,0,E1ww*E1hh*6);

   start_thread(tile_thread,0);                                            //  start working thread
   return;
}


//  dialog event and completion callback function

int tile_dialog_event(zdialog * zd, cchar *event)
{
   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      zfree(tile_pixmap);
      return 0;
   }

   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3

   if (strNeq(event,"apply")) return 0;

   zdialog_fetch(zd,"size",tile_size);                                     //  get tile size
   zdialog_fetch(zd,"gap",tile_gap);                                       //  get tile gap 

   if (tile_size < 2) {
      edit_reset();                                                        //  restore original image
      return 0;
   }
   
   signal_thread();                                                        //  trigger working thread
   return 1;
}


//  image tiles thread function

void * tile_thread(void *)
{
   int         sg, gg;
   int         sumpix, red, green, blue;
   int         ii, jj, px, py, qx, qy, dist;
   double      dnew, dold;
   uint16      *pix1, *pix3;

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      sg = tile_size + tile_gap;
      gg = tile_gap;

      for (py = 0; py < E1hh; py += sg)                                    //  initz. pixel color map for
      for (px = 0; px < E1ww; px += sg)                                    //    given pixel size
      {
         sumpix = red = green = blue = 0;

         for (qy = py + gg; qy < py + sg; qy++)                            //  get mean color for pixel block
         for (qx = px + gg; qx < px + sg; qx++)
         {
            if (qy > E1hh-1 || qx > E1ww-1) continue;

            pix1 = PXMpix(E1pxm16,qx,qy);
            red += pix1[0];
            green += pix1[1];
            blue += pix1[2];
            sumpix++;
         }

         if (sumpix) {
            red = (red / sumpix);
            green = (green / sumpix);
            blue = (blue / sumpix);
         }
         
         for (qy = py; qy < py + sg; qy++)                                 //  set color for pixels in block
         for (qx = px; qx < px + sg; qx++)
         {
            if (qy > E1hh-1 || qx > E1ww-1) continue;
            
            jj = (qy * E1ww + qx) * 3;

            if (qx-px < gg || qy-py < gg) {
               tile_pixmap[jj] = tile_pixmap[jj+1] = tile_pixmap[jj+2] = 0;
               continue;
            }

            tile_pixmap[jj] = red;
            tile_pixmap[jj+1] = green;
            tile_pixmap[jj+2] = blue;
         }
      }

      if (Factivearea)                                                     //  process selected area
      {
         for (ii = 0; ii < Fww * Fhh; ii++)                                //  find pixels in select area
         {                                                                 //  v.9.6
            if (! sa_pixisin[ii]) continue;
            py = ii / Fww;
            px = ii - py * Fww;
            dist = sa_pixisin[ii];
            pix3 = PXMpix(E3pxm16,px,py);
            jj = (py * E3ww + px) * 3;

            if (dist >= sa_blend) {                                        //  blend changes over sa_blend   v.10.2
               pix3[0] = tile_pixmap[jj];
               pix3[1] = tile_pixmap[jj+1];                                //  apply block color to member pixels
               pix3[2] = tile_pixmap[jj+2];
            }
            else {
               dnew = 1.0 * dist / sa_blend;
               dold = 1.0 - dnew;
               pix1 = PXMpix(E1pxm16,px,py);
               pix3[0] = dnew * tile_pixmap[jj] + dold * pix1[0];
               pix3[1] = dnew * tile_pixmap[jj+1] + dold * pix1[1];
               pix3[2] = dnew * tile_pixmap[jj+2] + dold * pix1[2];
            }
         }
      }

      else                                                                 //  process entire image
      {
         for (py = 0; py < E3hh-1; py++)                                   //  loop all image pixels
         for (px = 0; px < E3ww-1; px++)
         {
            pix3 = PXMpix(E3pxm16,px,py);                                  //  target pixel
            jj = (py * E3ww + px) * 3;                                     //  color map for (px,py)
            pix3[0] = tile_pixmap[jj];
            pix3[1] = tile_pixmap[jj+1];
            pix3[2] = tile_pixmap[jj+2];
         }
      }

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


/**************************************************************************/

//  convert image to dot grid (like Roy Lichtenstein)

int      dot_size;                                                         //  tile size enclosing dots

void m_dots(GtkWidget *, cchar *)                                          //  v.11.01
{
   int    dots_dialog_event(zdialog *zd, cchar *event);
   void * dots_thread(void *);

   zfuncs::F1_help_topic = "dot_matrix";

   if (! edit_setup("dots",0,2)) return;                                   //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Convert Image to Dots"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labt","hb1",ZTX("dot size"),"space=5");
   zdialog_add_widget(zdedit,"spin","size","hb1","3|99|1|9","space=5");
   zdialog_add_widget(zdedit,"button","apply","hb1",Bapply,"space=10");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");

   zdialog_run(zdedit,dots_dialog_event);                                  //  start dialog

   dot_size = 9;

   start_thread(dots_thread,0);                                            //  start working thread
   return;
}


//  dialog event and completion callback function

int dots_dialog_event(zdialog * zd, cchar *event)
{
   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      return 0;
   }

   if (strEqu(event,"undo")) edit_undo();
   if (strEqu(event,"redo")) edit_redo();
   if (strEqu(event,"blendwidth")) signal_thread();
   if (strNeq(event,"apply")) return 0;                                    //  wait for [apply]

   zdialog_fetch(zd,"size",dot_size);                                      //  get dot size
   signal_thread();                                                        //  trigger working thread
   return 1;
}


//  image dots thread function

void * dots_thread(void *)
{
   int         ds, sumpix, red, green, blue;
   int         cc, maxrgb, ii, dist, shift1, shift2;
   int         px, py, px1, px2, py1, py2;
   int         qx, qy, qx1, qx2, qy1, qy2;
   uint16      *pix1, *pix3;
   double      relmax, radius, radius2a, radius2b, f1, f2;
   double      f64K = 1.0 / 65536.0;
   double      fpi = 1.0 / 3.14159;
   double      dcx, dcy, qcx, qcy, qdist2;
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      mutex_lock(&Fpixmap_lock); 

      ds = dot_size;                                                       //  dot size and tile size

      cc = E3ww * E3hh * 6;                                                //  clear output image to black
      memset(E3pxm16->bmp,0,cc);
      
      px1 = 0;                                                             //  limits for tiles 
      px2 = Fww - ds;
      py1 = 0;
      py2 = Fhh - ds;
      
      if (Factivearea) {                                                   //  reduce for active area
         px1 = sa_minx;
         px2 = sa_maxx;
         py1 = sa_miny;
         py2 = sa_maxy;
         if (px2 > Fww - ds) px2 = Fww - ds;
         if (py2 > Fhh - ds) py2 = Fhh - ds;
      }
      
      shift1 = 0;

      for (py = py1; py < py2; py += ds)                                   //  loop all tiles in input image
      {
         shift1 = 1 - shift1;                                              //  offset alternate rows   v.11.02
         shift2 = 0.5 * shift1 * ds;

         for (px = px1 + shift2; px < px2; px += ds)
         {
            sumpix = red = green = blue = 0;

            for (qy = py; qy < py + ds; qy++)                              //  loop all pixels in tile
            for (qx = px; qx < px + ds; qx++)                              //  get mean RGB levels for tile
            {
               pix1 = PXMpix(E1pxm16,qx,qy);
               red += pix1[0];
               green += pix1[1];
               blue += pix1[2];
               sumpix++;
            }
            
            red = (red / sumpix);                                          //  mean RGB levels, 0 to 64K
            green = (green / sumpix);
            blue = (blue / sumpix);
            
            maxrgb = red;                                                  //  max. mean RGB level, 0 to 64K
            if (green > maxrgb) maxrgb = green;
            if (blue > maxrgb) maxrgb = blue;
            relmax = f64K * maxrgb;                                        //  max. RGB as 0 to 0.999
            
            radius = ds * sqrt(fpi * relmax);                              //  radius of dot with maximized color
            radius = radius * 0.9;                                         //  deliberate reduction      v.11.05
            if (radius < 0.5) continue;
            
            red = 65535.0 * red / maxrgb;                                  //  dot color, maximized
            green = 65535.0 * green / maxrgb;
            blue = 65535.0 * blue / maxrgb;

            dcx = px + 0.5 * ds;                                           //  center of dot / tile
            dcy = py + 0.5 * ds;
            
            qx1 = dcx - radius;                                            //  pixels within dot radius of center
            qx2 = dcx + radius;
            qy1 = dcy - radius;
            qy2 = dcy + radius;
            
            radius2a = (radius + 0.5) * (radius + 0.5);
            radius2b = (radius - 0.5) * (radius - 0.5);
            
            for (qy = qy1; qy <= qy2; qy++)                                //  loop all pixels within dot radius
            for (qx = qx1; qx <= qx2; qx++)
            {
               qcx = qx + 0.5;                                             //  center of pixel
               qcy = qy + 0.5;
               qdist2 = (qcx-dcx)*(qcx-dcx) + (qcy-dcy)*(qcy-dcy);         //  pixel distance**2 from center of dot
               if (qdist2 > radius2a) f1 = 0.0;                            //  pixel outside dot, no color
               else if (qdist2 < radius2b) f1 = 1.0;                       //  pixel inside dot, full color
               else f1 = 1.0 - sqrt(qdist2) + radius - 0.5;                //  pixel straddles edge, some color
               pix3 = PXMpix(E3pxm16,qx,qy);
               pix3[0] = f1 * red;
               pix3[1] = f1 * green;
               pix3[2] = f1 * blue;
            }
         }
      }
      
      if (Factivearea)                                                     //  select area active
      {
         pix1 = (uint16 *) E1pxm16->bmp;
         pix3 = (uint16 *) E3pxm16->bmp;
         
         for (ii = 0; ii < Fww * Fhh; ii++)                                //  loop all pixels
         {
            dist = sa_pixisin[ii];
            if (dist == 0) memmove(pix3,pix1,6);                           //  pixel outside area, output pixel = input
            else if (dist < sa_blend) {
               f1 = 1.0 * dist / sa_blend;                                 //  pixel in blend region
               f2 = 1.0 - f1;
               pix3[0] = f1 * pix3[0] + f2 * pix1[0];                      //  output pixel is mix of input and output
               pix3[1] = f1 * pix3[1] + f2 * pix1[1];
               pix3[2] = f1 * pix3[2] + f2 * pix1[2];
            }
            pix1 += 3;
            pix3 += 3;
         }
      }

      mutex_unlock(&Fpixmap_lock); 

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


/**************************************************************************/

//  convert image to simulate a painting
//  processing a 10 megapixel image needs 140 MB of main memory

namespace paint_names 
{
   int         color_depth;
   int         group_area;
   double      color_match;
   int         borders;

   typedef struct  {
      int16       px, py;
      char        direc;
   }  spixstack;

   int         Nstack;
   spixstack   *pixstack;                                                  //  pixel group search memory
   int         *pixgroup;                                                  //  maps (px,py) to pixel group no.
   int         *groupcount;                                                //  count of pixels in each group

   int         group;
   char        direc;
   uint16      gcolor[3];
}


void m_painting(GtkWidget *, cchar *)
{
   using namespace paint_names;

   int    painting_dialog_event(zdialog *zd, cchar *event);
   void * painting_thread(void *);

   zfuncs::F1_help_topic = "painting";                                     //  v.10.8

   if (! edit_setup("paint",0,2)) return;                                  //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Simulate Painting"),mWin,Bdone,Bcancel,null);

   zdialog_add_widget(zdedit,"hbox","hbcd","dialog",0,"space=1");
   zdialog_add_widget(zdedit,"label","lab1","hbcd",ZTX("color depth"),"space=5");
   zdialog_add_widget(zdedit,"spin","colordepth","hbcd","1|5|1|3","space=5");

   zdialog_add_widget(zdedit,"hbox","hbts","dialog",0,"space=1");
   zdialog_add_widget(zdedit,"label","labts","hbts",ZTX("patch area goal"),"space=5");
   zdialog_add_widget(zdedit,"spin","grouparea","hbts","0|9999|10|1000","space=5");          // v.11.04

   zdialog_add_widget(zdedit,"hbox","hbcm","dialog",0,"space=1");
   zdialog_add_widget(zdedit,"label","labcm","hbcm",ZTX("req. color match"),"space=5");
   zdialog_add_widget(zdedit,"spin","colormatch","hbcm","0|99|1|50","space=5");

   zdialog_add_widget(zdedit,"hbox","hbbd","dialog",0,"space=1");
   zdialog_add_widget(zdedit,"label","labbd","hbbd",ZTX("borders"),"space=5");
   zdialog_add_widget(zdedit,"check","borders","hbbd",0,"space=2");
   zdialog_add_widget(zdedit,"button","apply","hbbd",Bapply,"space=10");

   zdialog_run(zdedit,painting_dialog_event);                              //  run dialog - parallel

   start_thread(painting_thread,0);                                        //  start working thread
   return;
}


//  dialog event and completion callback function

int painting_dialog_event(zdialog *zd, cchar *event)
{
   using namespace paint_names;

   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      return 0;
   }

   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3

   if (strEqu(event,"apply")) {                                            //  apply user settings
      zdialog_fetch(zd,"colordepth",color_depth);                          //  color depth
      zdialog_fetch(zd,"grouparea",group_area);                            //  target group area (pixels)
      zdialog_fetch(zd,"colormatch",color_match);                          //  req. color match to combine groups
      zdialog_fetch(zd,"borders",borders);                                 //  borders wanted
      color_match = 0.01 * color_match;                                    //  scale 0 to 1

      signal_thread();                                                     //  do the work
      wait_thread_idle();
   }
   
   return 0;
}


//  painting thread function

void * painting_thread(void *)
{
   void  paint_colordepth();
   void  paint_pixgroups();
   void  paint_mergegroups();
   void  paint_paintborders();
   void  paint_blend();

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      paint_colordepth();                                                  //  set new color depth
      paint_pixgroups();                                                   //  group pixel patches of a color
      paint_mergegroups();                                                 //  merge smaller into larger groups
      paint_paintborders();                                                //  add borders around groups
      paint_blend();                                                       //  blend edges of selected area

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


//  set the specified color depth, 1-5 bits/color

void paint_colordepth()
{
   using namespace paint_names;

   int            ii, px, py, rgb;
   double         fmag;
   uint16         m1, m2, val1, val3;
   uint16         *pix1, *pix3;

   m1 = 0xFFFF << (16 - color_depth);                                      //  5 > 1111100000000000
   m2 = 0x8000 >> color_depth;                                             //  5 > 0000010000000000

   fmag = 65535.0 / m1;                                                    //  full brightness range

   if (Factivearea)                                                        //  process select area
   {
      for (ii = 0; ii < Fww * Fhh; ii++)
      {                                                                    //  v.9.6
         if (! sa_pixisin[ii]) continue;
         py = ii / Fww;
         px = ii - py * Fww;

         pix1 = PXMpix(E1pxm16,px,py);                                     //  input pixel
         pix3 = PXMpix(E3pxm16,px,py);                                     //  output pixel

         for (rgb = 0; rgb < 3; rgb++)
         {
            val1 = pix1[rgb];
            if (val1 < m1) val3 = (val1 + m2) & m1;
            else val3 = m1;
            val3 = uint(val3 * fmag);
            pix3[rgb] = val3;
         }
      }
   }
   
   else                                                                    //  process entire image
   {
      for (py = 0; py < E3hh; py++)                                        //  loop all pixels
      for (px = 0; px < E3ww; px++)
      {
         pix1 = PXMpix(E1pxm16,px,py);                                     //  input pixel
         pix3 = PXMpix(E3pxm16,px,py);                                     //  output pixel
         
         for (rgb = 0; rgb < 3; rgb++)
         {
            val1 = pix1[rgb];
            if (val1 < m1) val3 = (val1 + m2) & m1;
            else val3 = m1;
            val3 = uint(val3 * fmag);
            pix3[rgb] = val3;
         }
      }
   }

   return;
}


//  find all groups of contiguous pixels with the same color

void paint_pixgroups()
{
   using namespace paint_names;

   void  paint_pushpix(int px, int py);

   int            cc1, cc2;
   int            ii, kk, px, py;
   int            ppx, ppy, npx, npy;
   uint16         *pix3;

   cc1 = E3ww * E3hh;

   cc2 = cc1 * sizeof(int);
   pixgroup = (int *) zmalloc(cc2,"pixgroup");                             //  maps pixel to assigned group
   memset(pixgroup,0,cc2);
   
   if (Factivearea) cc1 = sa_Npixel;

   cc2 = cc1 * sizeof(spixstack);
   pixstack = (spixstack *) zmalloc(cc2,"pixstack");                       //  memory stack for pixel search
   memset(pixstack,0,cc2);
   
   cc2 = cc1 * sizeof(int);
   groupcount = (int *) zmalloc(cc2,"groupcount");                         //  counts pixels per group
   memset(groupcount,0,cc2);
   
   group = 0;
   
   for (py = 0; py < E3hh; py++)                                           //  loop all pixels
   for (px = 0; px < E3ww; px++)
   {
      kk = py * E3ww + px;
      if (Factivearea && ! sa_pixisin[kk]) continue;
      if (pixgroup[kk]) continue;                                          //  already assigned to group

      pixgroup[kk] = ++group;                                              //  assign next group
      ++groupcount[group];

      pix3 = PXMpix(E3pxm16,px,py);
      gcolor[0] = pix3[0];
      gcolor[1] = pix3[1];
      gcolor[2] = pix3[2];

      pixstack[0].px = px;                                                 //  put pixel into stack with
      pixstack[0].py = py;                                                 //    direction = ahead          v.11.04
      pixstack[0].direc = 'a';
      Nstack = 1;

      while (Nstack)                                                       //  overhauled    v.11.02
      {
         kk = Nstack - 1;                                                  //  get last pixel in stack
         px = pixstack[kk].px;
         py = pixstack[kk].py;
         direc = pixstack[kk].direc;
         
         if (direc == 'x') {
            Nstack--;
            continue;
         }

         if (Nstack > 1) {
            ii = Nstack - 2;                                               //  get prior pixel in stack
            ppx = pixstack[ii].px;
            ppy = pixstack[ii].py;
         }
         else {
            ppx = px - 1;                                                  //  if only one, assume prior = left
            ppy = py;
         }

         if (direc == 'a') {                                               //  next ahead pixel             v.11.04
            npx = px + px - ppx;
            npy = py + py - ppy;
            pixstack[kk].direc = 'r';                                      //  next search direction
         }

         else if (direc == 'r') {                                          //  next right pixel             v.11.04
            npx = px + py - ppy;
            npy = py + px - ppx;
            pixstack[kk].direc = 'l';
         }

         else { /*  direc = 'l'  */                                        //  next left pixel              v.11.04
            npx = px + ppy - py;
            npy = py + ppx - px;
            pixstack[kk].direc = 'x';
         }

         if (npx < 0 || npx > E3ww-1) continue;                            //  pixel off the edge           v.11.04
         if (npy < 0 || npy > E3hh-1) continue;
         
         kk = npy * E3ww + npx;

         if (Factivearea)
            if (! sa_pixisin[kk]) continue;                                //  pixel outside area

         if (pixgroup[kk]) continue;                                       //  pixel already assigned

         pix3 = PXMpix(E3pxm16,npx,npy);
         if (pix3[0] != gcolor[0] || pix3[1] != gcolor[1]                  //  not same color as group
                                  || pix3[2] != gcolor[2]) continue;
         
         pixgroup[kk] = group;                                             //  assign pixel to group
         ++groupcount[group];

         kk = Nstack++;                                                    //  put pixel into stack
         pixstack[kk].px = npx;
         pixstack[kk].py = npy;
         pixstack[kk].direc = 'a';                                         //  direction = ahead            v.11.04
      }
   }
   
   return;
}      


//  merge small pixel groups into adjacent larger groups with best color match

void paint_mergegroups()
{
   using namespace paint_names;

   int         ii, jj, kk, px, py, npx, npy;
   int         nccc, mcount, group2;
   double      ff = 1.0 / 65536.0;
   double      fred, fgreen, fblue, match;
   int         nnpx[4] = {  0, -1, +1, 0 };
   int         nnpy[4] = { -1, 0,  0, +1 };
   uint16      *pix3, *pixN;

   typedef struct  {
      int         group;
      double      match;
      uint16      pixM[3];
   }  snewgroup;

   snewgroup      *newgroup;
   
   nccc = (group + 1) * sizeof(snewgroup);
   newgroup = (snewgroup *) zmalloc(nccc,"newgroup");
   
   if (Factivearea)                                                        //  process select area
   {
      while (true)
      {
         memset(newgroup,0,nccc);

         for (ii = 0; ii < Fww * Fhh; ii++)                                //  find pixels in select area
         {                                                                 //  v.9.6
            if (! sa_pixisin[ii]) continue;
            py = ii / Fww;
            px = ii - py * Fww;

            kk = E3ww * py + px;                                           //  get assigned group
            group = pixgroup[kk];
            if (groupcount[group] >= group_area) continue;                 //  group count large enough

            pix3 = PXMpix(E3pxm16,px,py);

            for (jj = 0; jj < 4; jj++)                                     //  get 4 neighbor pixels
            {
               npx = px + nnpx[jj];
               npy = py + nnpy[jj];

               if (npx < 0 || npx >= E3ww) continue;                       //  off the edge
               if (npy < 0 || npy >= E3hh) continue;

               kk = E3ww * npy + npx;
               if (! sa_pixisin[kk]) continue;                             //  pixel outside area
               if (pixgroup[kk] == group) continue;                        //  already in same group
               
               pixN = PXMpix(E3pxm16,npx,npy);                             //  match color of group neighbor
               fred = ff * abs(pix3[0] - pixN[0]);                         //    to color of group
               fgreen = ff * abs(pix3[1] - pixN[1]);
               fblue = ff * abs(pix3[2] - pixN[2]);
               match = (1.0 - fred) * (1.0 - fgreen) * (1.0 - fblue);      //  color match, 0 to 1.0
               if (match < color_match) continue;

               if (match > newgroup[group].match) {
                  newgroup[group].match = match;                           //  remember best match
                  newgroup[group].group = pixgroup[kk];                    //  and corresp. group no.
                  newgroup[group].pixM[0] = pixN[0];                       //  and corresp. new color
                  newgroup[group].pixM[1] = pixN[1];
                  newgroup[group].pixM[2] = pixN[2];
               }
            }
         }

         mcount = 0;

         for (ii = 0; ii < Fww * Fhh; ii++)                                //  find pixels in select area
         {                                                                 //  v.9.6
            if (! sa_pixisin[ii]) continue;
            py = ii / Fww;
            px = ii - py * Fww;

            kk = E3ww * py + px;
            group = pixgroup[kk];                                          //  test for new group assignment
            group2 = newgroup[group].group;
            if (! group2) continue;
            
            if (groupcount[group] > groupcount[group2]) continue;          //  accept only bigger new group

            pixgroup[kk] = group2;                                         //  make new group assignment
            --groupcount[group];
            ++groupcount[group2];

            pix3 = PXMpix(E3pxm16,px,py);                                  //  make new color assignment
            pix3[0] = newgroup[group].pixM[0];
            pix3[1] = newgroup[group].pixM[1];
            pix3[2] = newgroup[group].pixM[2];

            mcount++;
         }
         
         if (mcount == 0) break;
      }
   }

   else                                                                    //  process entire image
   {
      while (true)
      {
         memset(newgroup,0,nccc);

         for (py = 0; py < E3hh; py++)                                     //  loop all pixels
         for (px = 0; px < E3ww; px++)
         {
            kk = E3ww * py + px;                                           //  get assigned group
            group = pixgroup[kk];
            if (groupcount[group] >= group_area) continue;                 //  group count large enough

            pix3 = PXMpix(E3pxm16,px,py);

            for (jj = 0; jj < 4; jj++)                                     //  get 4 neighbor pixels
            {
               npx = px + nnpx[jj];
               npy = py + nnpy[jj];

               if (npx < 0 || npx >= E3ww) continue;                       //  off the edge
               if (npy < 0 || npy >= E3hh) continue;
               
               kk = E3ww * npy + npx;
               if (pixgroup[kk] == group) continue;                        //  in same group

               pixN = PXMpix(E3pxm16,npx,npy);                             //  match color of group neighbor
               fred = ff * abs(pix3[0] - pixN[0]);                         //    to color of group
               fgreen = ff * abs(pix3[1] - pixN[1]);
               fblue = ff * abs(pix3[2] - pixN[2]);
               match = (1.0 - fred) * (1.0 - fgreen) * (1.0 - fblue);      //  color match, 0 to 1.0
               if (match < color_match) continue;

               if (match > newgroup[group].match) {
                  newgroup[group].match = match;                           //  remember best match
                  newgroup[group].group = pixgroup[kk];                    //  and corresp. group no.
                  newgroup[group].pixM[0] = pixN[0];                       //  and corresp. new color
                  newgroup[group].pixM[1] = pixN[1];
                  newgroup[group].pixM[2] = pixN[2];
               }
            }
         }

         mcount = 0;

         for (py = 0; py < E3hh; py++)                                     //  loop all pixels
         for (px = 0; px < E3ww; px++)
         {
            kk = E3ww * py + px;
            group = pixgroup[kk];                                          //  test for new group assignment
            group2 = newgroup[group].group;
            if (! group2) continue;
            
            if (groupcount[group] > groupcount[group2]) continue;          //  accept only bigger new group

            pixgroup[kk] = group2;                                         //  make new group assignment
            --groupcount[group];
            ++groupcount[group2];

            pix3 = PXMpix(E3pxm16,px,py);                                  //  make new color assignment
            pix3[0] = newgroup[group].pixM[0];
            pix3[1] = newgroup[group].pixM[1];
            pix3[2] = newgroup[group].pixM[2];

            mcount++;
         }
         
         if (mcount == 0) break;
      }
   }

   zfree(pixgroup);
   zfree(pixstack);
   zfree(groupcount);
   zfree(newgroup);

   return;
}


//  paint borders between the groups of contiguous pixels

void paint_paintborders()
{
   using namespace paint_names;

   int            ii, kk, px, py, cc;
   uint16         *pix3, *pixL, *pixA;
   
   if (! borders) return;
   
   cc = E3ww * E3hh;
   char * pixblack = zmalloc(cc,"pixblack");
   memset(pixblack,0,cc);

   if (Factivearea)
   {
      for (ii = 0; ii < Fww * Fhh; ii++)                                   //  find pixels in select area
      {                                                                    //  v.9.6
         if (! sa_pixisin[ii]) continue;
         py = ii / Fww;
         px = ii - py * Fww;
         if (px < 1 || py < 1) continue;

         pix3 = PXMpix(E3pxm16,px,py);
         pixL = PXMpix(E3pxm16,px-1,py);
         pixA = PXMpix(E3pxm16,px,py-1);
         
         if (pix3[0] != pixL[0] || pix3[1] != pixL[1] || pix3[2] != pixL[2])
         {
            kk = ii - 1;
            if (pixblack[kk]) continue;
            kk += 1;
            pixblack[kk] = 1;
            continue;
         }

         if (pix3[0] != pixA[0] || pix3[1] != pixA[1] || pix3[2] != pixA[2])
         {
            kk = ii - E3ww;
            if (pixblack[kk]) continue;
            kk += E3ww;
            pixblack[kk] = 1;
         }
      }

      for (ii = 0; ii < Fww * Fhh; ii++)                                   //  find pixels in select area
      {                                                                    //  v.9.6
         if (! sa_pixisin[ii]) continue;
         py = ii / Fww;
         px = ii - py * Fww;
         if (px < 1 || py < 1) continue;

         if (! pixblack[ii]) continue;
         pix3 = PXMpix(E3pxm16,px,py);
         pix3[0] = pix3[1] = pix3[2] = 0;
      }
   }
         
   else
   {
      for (py = 1; py < E3hh; py++)                                        //  loop all pixels
      for (px = 1; px < E3ww; px++)                                        //  omit top and left
      {
         pix3 = PXMpix(E3pxm16,px,py);                                     //  output pixel
         pixL = PXMpix(E3pxm16,px-1,py);                                   //  pixel to left
         pixA = PXMpix(E3pxm16,px,py-1);                                   //  pixel above
         
         if (pix3[0] != pixL[0] || pix3[1] != pixL[1] || pix3[2] != pixL[2])
         {
            kk = E3ww * py + px-1;                                         //  have horiz. transition
            if (pixblack[kk]) continue;
            kk += 1;
            pixblack[kk] = 1;
            continue;
         }

         if (pix3[0] != pixA[0] || pix3[1] != pixA[1] || pix3[2] != pixA[2])
         {
            kk = E3ww * (py-1) + px;                                       //  have vertical transition
            if (pixblack[kk]) continue;
            kk += E3ww;
            pixblack[kk] = 1;
         }
      }

      for (py = 1; py < E3hh; py++)
      for (px = 1; px < E3ww; px++)
      {
         kk = E3ww * py + px;
         if (! pixblack[kk]) continue;
         pix3 = PXMpix(E3pxm16,px,py);
         pix3[0] = pix3[1] = pix3[2] = 0;
      }
   }
   
   zfree(pixblack);
   return;
}


//  blend edges of selected area

void paint_blend()
{
   int         ii, px, py, rgb, dist;
   uint16      *pix1, *pix3;
   double      f1, f2;
   
   if (Factivearea && sa_blend > 0)
   {
      for (ii = 0; ii < Fww * Fhh; ii++)                                   //  find pixels in select area
      {                                                                    //  v.9.6
         dist = sa_pixisin[ii];
         if (! dist || dist >= sa_blend) continue;

         py = ii / Fww;
         px = ii - py * Fww;
         pix1 = PXMpix(E1pxm16,px,py);                                     //  input pixel
         pix3 = PXMpix(E3pxm16,px,py);                                     //  output pixel

         f2 = 1.0 * dist / sa_blend;                                       //  changes over distance sa_blend
         f1 = 1.0 - f2;

         for (rgb = 0; rgb < 3; rgb++)
            pix3[rgb] = int(f1 * pix1[rgb] + f2 * pix3[rgb]);
      }
   }

   return;
}


/**************************************************************************/

//  pixel edit function - edit individual pixels

#define pixed_undomaxmem (1000 * mega)                                     //  pixel edit max. memory          v.11.04
#define pixed_undomaxpix (mega)                                            //  pixel edit max. pixel blocks

void  pixed_mousefunc();
void  pixed_dopixels(int px, int py);
void  pixed_undo1();
void  pixed_freeundo();

int      pixed_RGB[3];
int      pixed_mode;
int      pixed_radius;
double   pixed_kernel[200][200];                                           //  radius <= 99

int      pixed_undototpix = 0;                                             //  total undo pixel blocks
int      pixed_undototmem = 0;                                             //  total undo memory allocated
int      pixed_undoseq = 0;                                                //  undo sequence no.
char     pixed_undomemmessage[100];                                        //  translated undo memory message

typedef struct {                                                           //  pixel block before edit
   int         seq;                                                        //  undo sequence no.
   uint16      npix;                                                       //  no. pixels in this block
   uint16      px, py;                                                     //  center pixel (radius org.)
   uint16      radius;                                                     //  radius of pixel block
   uint16      pixel[][3];                                                 //  array of pixel[npix][3] 
}  pixed_savepix;

pixed_savepix   **pixed_undopixmem = 0;                                    //  array of *pixed_savepix


void m_pixedit(GtkWidget *, cchar *)
{
   int   pixed_dialog_event(zdialog* zd, cchar *event);

   char        undomemmessage[100];

   zfuncs::F1_help_topic = "edit_pixels";                                  //  v.10.8

   if (! edit_setup("pixedit",0,1)) return;                                //  setup edit: no preview

   strncpy0(pixed_undomemmessage,ZTX("Undo Memory %d%c"),99);              //  translate undo memory message

   zdedit = zdialog_new(ZTX("Edit Pixels"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hbc","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labc","hbc",ZTX("color"),"space=8");
   zdialog_add_widget(zdedit,"colorbutt","color","hbc","100|100|100");
   zdialog_add_widget(zdedit,"label","space","hbc",0,"space=10");
   zdialog_add_widget(zdedit,"radio","radio1","hbc",ZTX("pick"),"space=3");
   zdialog_add_widget(zdedit,"radio","radio2","hbc",ZTX("paint"),"space=3");
   zdialog_add_widget(zdedit,"radio","radio3","hbc",ZTX("erase"),"space=3");
   zdialog_add_widget(zdedit,"hbox","hbbri","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vbbr1","hbbri",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vbbr2","hbbri",0,"homog|space=5");
   zdialog_add_widget(zdedit,"label","space","hbbri",0,"space=10");
   zdialog_add_widget(zdedit,"vbox","vbbr3","hbbri",0,"homog|space=3");
   zdialog_add_widget(zdedit,"hbox","hbrad","vbbr1",0,"space=3");
   zdialog_add_widget(zdedit,"label","space","hbrad",0,"expand");
   zdialog_add_widget(zdedit,"label","labbr","hbrad",ZTX("paintbrush radius"));
   zdialog_add_widget(zdedit,"label","labtc","vbbr1",ZTX("transparency center"));
   zdialog_add_widget(zdedit,"label","labte","vbbr1",ZTX("transparency edge"));
   zdialog_add_widget(zdedit,"spin","radius","vbbr2","1|99|1|2");
   zdialog_add_widget(zdedit,"spin","trcent","vbbr2","0|99|1|60");
   zdialog_add_widget(zdedit,"spin","tredge","vbbr2","0|99|1|99");
   zdialog_add_widget(zdedit,"button","undlast","vbbr3",Bundolast);
   zdialog_add_widget(zdedit,"button","undall","vbbr3",Bundoall);
   zdialog_add_widget(zdedit,"hbox","hb4","dialog");
   zdialog_add_widget(zdedit,"check","mymouse","hb4",BmyMouse,"space=20");
   zdialog_add_widget(zdedit,"label","labmem","hb4");

   zdialog_run(zdedit,pixed_dialog_event);                                 //  run dialog - parallel

   zdialog_send_event(zdedit,"radius");                                    //  get kernel initialized

   snprintf(undomemmessage,99,pixed_undomemmessage,0,'%');                 //  stuff undo memory status
   zdialog_stuff(zdedit,"labmem",undomemmessage);
   
   pixed_RGB[0] = pixed_RGB[1] = pixed_RGB[2] = 100;                       //  initialize color
   
   pixed_mode = 1;                                                         //  mode = pick color

   pixed_undopixmem = 0;                                                   //  no undo data
   pixed_undototpix = 0;
   pixed_undototmem = 0;
   pixed_undoseq = 0;

   takeMouse(zdedit,pixed_mousefunc,drawcursor);                           //  connect mouse function          v.11.03
   return;
}


//  dialog event and completion callback function

int pixed_dialog_event(zdialog *zd, cchar *event)                          //  pixedit dialog event function
{
   char        color[20];
   cchar       *pp;
   int         radius, dx, dy, brad, mymouse;
   double      rad, kern, trcent, tredge;
   
   if (zd->zstat) goto complete;
   
   paint_toparc(2);                                                        //  v.11.04

   if (strEqu(event,"mymouse")) {                                          //  toggle mouse capture         v.10.12
      zdialog_fetch(zd,"mymouse",mymouse);
      if (mymouse)                                                         //  connect mouse function       v.11.03
         takeMouse(zd,pixed_mousefunc,drawcursor);
      else freeMouse();                                                    //  disconnect mouse
   }

   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3

   zdialog_fetch(zd,"radio1",brad);                                        //  pick
   if (brad) pixed_mode = 1;
   zdialog_fetch(zd,"radio2",brad);                                        //  paint
   if (brad) pixed_mode = 2;
   zdialog_fetch(zd,"radio3",brad);                                        //  erase
   if (brad) pixed_mode = 3;
   
   if (strEqu(event,"color")) 
   {
      zdialog_fetch(zd,"color",color,19);                                  //  get color from color wheel
      pp = strField(color,"|",1);
      if (pp) pixed_RGB[0] = atoi(pp);
      pp = strField(color,"|",2);
      if (pp) pixed_RGB[1] = atoi(pp);
      pp = strField(color,"|",3);
      if (pp) pixed_RGB[2] = atoi(pp);
   }
   
   if (strstr("radius trcent tredge",event))                               //  get new brush attributes
   {
      zdialog_fetch(zd,"radius",radius);                                   //  radius
      zdialog_fetch(zd,"trcent",trcent);                                   //  center transparency
      zdialog_fetch(zd,"tredge",tredge);                                   //  edge transparency

      pixed_radius = radius;
      trcent = 0.01 * trcent;                                              //  scale 0 ... 1
      tredge = 0.01 * tredge;
      tredge = (1 - trcent) * (1 - tredge);
      tredge = 1 - tredge;
      trcent = sqrt(trcent);                                               //  speed up the curve
      tredge = sqrt(tredge);

      for (dy = -radius; dy <= radius; dy++)                               //  build kernel
      for (dx = -radius; dx <= radius; dx++)
      {
         rad = sqrt(dx*dx + dy*dy);
         kern = (radius - rad) / radius;                                   //  1 ... 0 
         kern = kern * (trcent - tredge) + tredge;                         //  trcent ... tredge
         if (rad > radius) kern = 1;
         if (kern < 0) kern = 0;
         if (kern > 1) kern = 1;
         pixed_kernel[dx+radius][dy+radius] = kern;
      }
   }
   
   if (strEqu(event,"undlast"))                                            //  undo last edit (click or drag)
      pixed_undo1();

   if (strEqu(event,"undall")) {                                           //  undo all edits
      edit_reset();                                                        //  v.10.3
      pixed_freeundo();
   }

   return 1;

complete:

   freeMouse();                                                            //  disconnect mouse

   if (zd->zstat == 1) edit_done();                                        //  done
   else edit_cancel();                                                     //  cancel or destroy

   pixed_freeundo();                                                       //  free undo memory
   return 0;
}


//  pixel edit mouse function

void pixed_mousefunc()
{
   static int  pmxdown = 0, pmydown = 0;
   int         px, py;
   char        color[20];
   uint16      *ppix3;
   
   if (LMclick)                                                            //  left mouse click
   {
      LMclick = 0;
      px = Mxclick;
      py = Myclick;

      if (pixed_mode == 1)                                                 //  pick new color from image
      {
         ppix3 = PXMpix(E3pxm16,px,py);
         pixed_RGB[0] = ppix3[0] / 256;
         pixed_RGB[1] = ppix3[1] / 256;
         pixed_RGB[2] = ppix3[2] / 256;
         snprintf(color,19,"%d|%d|%d",pixed_RGB[0],pixed_RGB[1],pixed_RGB[2]);
         if (zdedit) zdialog_stuff(zdedit,"color",color);
      }
      else {                                                               //  paint or erase
         pixed_undoseq++;                                                  //  new undo seq. no.
         paint_toparc(2);
         pixed_dopixels(px,py);                                            //  do 1 block of pixels
      }
   }
   
   if (RMclick) 
   {
      RMclick = 0;
      paint_toparc(2);
      pixed_undo1();                                                       //  undo last paint      v.10.11
   }
   
   if (Mxdrag || Mydrag)                                                   //  drag in progress
   {
      px = Mxdrag;
      py = Mydrag;
      Mxdrag = Mydrag = 0;

      if (Mxdown != pmxdown || Mydown != pmydown) {                        //  new drag
         pixed_undoseq++;                                                  //  new undo seq. no.
         pmxdown = Mxdown;
         pmydown = Mydown;
      }
      paint_toparc(2);
      pixed_dopixels(px,py);                                               //  do 1 block of pixels
   }
   
   toparcx = Mxposn - pixed_radius;                                        //  define brush outline circle
   toparcy = Myposn - pixed_radius;
   toparcw = toparch = 2 * pixed_radius;
   if (pixed_mode == 1) Ftoparc = 0;
   else Ftoparc = 1;
   if (Ftoparc) paint_toparc(3);
   
   return;
}


//  paint or erase 1 block of pixels within radius of px, py

void pixed_dopixels(int px, int py)
{
   void  pixed_saveundo(int px, int py);

   uint16      *ppix1, *ppix3;
   int         radius, dx, dy, qx, qy, ii, ww, dist;
   int         red, green, blue;
   double      kern;

   edit_zapredo();                                                         //  delete redo copy    v.10.3
   
   pixed_saveundo(px,py);                                                  //  save pixels for poss. undo   v.10.12.1

   red = 256 * pixed_RGB[0];
   green = 256 * pixed_RGB[1];
   blue = 256 * pixed_RGB[2];

   radius = pixed_radius;

   for (dy = -radius; dy <= radius; dy++)                                  //  loop surrounding block of pixels
   for (dx = -radius; dx <= radius; dx++)
   {
      qx = px + dx;
      qy = py + dy;
      
      if (qx < 0 || qx > E3ww-1) continue;
      if (qy < 0 || qy > E3hh-1) continue;
      
      if (Factivearea) {                                                   //  select area active     v.10.11
         ii = qy * E3ww + qx;
         dist = sa_pixisin[ii];
         if (! dist) continue;                                             //  pixel is outside area
      }
      
      kern = pixed_kernel[dx+radius][dy+radius];
      ppix1 = PXMpix(E1pxm16,qx,qy);                                       //  original image pixel
      ppix3 = PXMpix(E3pxm16,qx,qy);                                       //  edited image pixel

      if (pixed_mode == 2)                                                 //  color pixels transparently
      {
         ppix3[0] = (1.0 - kern) * red   + kern * ppix3[0];
         ppix3[1] = (1.0 - kern) * green + kern * ppix3[1];
         ppix3[2] = (1.0 - kern) * blue  + kern * ppix3[2];
         Fmodified = 1;
      }

      if (pixed_mode == 3)                                                 //  restore org. pixels transparently
      {
         ppix3[0] = (1.0 - kern) * ppix1[0] + kern * ppix3[0];
         ppix3[1] = (1.0 - kern) * ppix1[1] + kern * ppix3[1];
         ppix3[2] = (1.0 - kern) * ppix1[2] + kern * ppix3[2];
      }
   }

   px = px - radius - 1;
   py = py - radius - 1;
   ww = 2 * radius + 3;
   mwpaint3(px,py,ww,ww);                                                  //  v.10.11
   return;
}


//  save 1 block of pixels for possible undo

void pixed_saveundo(int px, int py)
{
   int            npix, radius, dx, dy;
   uint16         *ppix3;
   pixed_savepix  *ppixsave1;
   char           undomemmessage[100];
   int            mempercent;
   static int     ppercent = 0;

   if (! pixed_undopixmem)                                                 //  first call
   {
      pixed_undopixmem = (pixed_savepix **) zmalloc(pixed_undomaxpix * sizeof(void *),"pixed.undomem");
      pixed_undototpix = 0;
      pixed_undototmem = 0;
   }
   
   if (pixed_undototmem > pixed_undomaxmem) 
   {
      zmessageACK(mWin,ZTX("Undo memory limit has been reached. \n"
                           "Save work with [done], then resume editing."));
      Mdrag = 0;
      return;
   }

   radius = pixed_radius;
   npix = 0;

   for (dy = -radius; dy <= radius; dy++)                                  //  count pixels in block
   for (dx = -radius; dx <= radius; dx++)
   {
      if (px + dx < 0 || px + dx > E3ww-1) continue;
      if (py + dy < 0 || py + dy > E3hh-1) continue;
      npix++;
   }
   
   ppixsave1 = (pixed_savepix *) zmalloc(npix*6+12,"pixed.pixsave");       //  allocate memory for block
   pixed_undopixmem[pixed_undototpix] = ppixsave1;
   pixed_undototpix += 1;
   pixed_undototmem += npix * 6 + 12;
   
   ppixsave1->seq = pixed_undoseq;                                         //  save pixel block poop
   ppixsave1->npix = npix;
   ppixsave1->px = px;
   ppixsave1->py = py;
   ppixsave1->radius = radius;

   npix = 0;

   for (dy = -radius; dy <= radius; dy++)                                  //  save pixels in block
   for (dx = -radius; dx <= radius; dx++)
   {
      if (px + dx < 0 || px + dx > E3ww-1) continue;
      if (py + dy < 0 || py + dy > E3hh-1) continue;
      ppix3 = PXMpix(E3pxm16,(px+dx),(py+dy));                             //  edited image pixel
      ppixsave1->pixel[npix][0] = ppix3[0];
      ppixsave1->pixel[npix][1] = ppix3[1];
      ppixsave1->pixel[npix][2] = ppix3[2];
      npix++;
   }

   mempercent = int(100.0 * pixed_undototmem / pixed_undomaxmem);          //  update undo memory status
   if (mempercent != ppercent) {
      ppercent = mempercent;
      snprintf(undomemmessage,99,pixed_undomemmessage,mempercent,'%');
      zdialog_stuff(zdedit,"labmem",undomemmessage);
   }

   return;
}


//  undo last undo sequence number

void pixed_undo1()
{
   int            pindex, npix, radius, mempercent;
   int            ww, px, py, dx, dy;
   uint16         *ppix3;
   pixed_savepix  *ppixsave1;
   char           undomemmessage[100];
   
   pindex = pixed_undototpix;
   
   while (pindex > 0)
   {
      --pindex;
      ppixsave1 = pixed_undopixmem[pindex];
      if (ppixsave1->seq != pixed_undoseq) break;
      px = ppixsave1->px;
      py = ppixsave1->py;
      radius = ppixsave1->radius;

      npix = 0;
      for (dy = -radius; dy <= radius; dy++)
      for (dx = -radius; dx <= radius; dx++)
      {
         if (px + dx < 0 || px + dx > E3ww-1) continue;
         if (py + dy < 0 || py + dy > E3hh-1) continue;
         ppix3 = PXMpix(E3pxm16,(px+dx),(py+dy));
         ppix3[0] = ppixsave1->pixel[npix][0];
         ppix3[1] = ppixsave1->pixel[npix][1];
         ppix3[2] = ppixsave1->pixel[npix][2];
         npix++;
      }

      px = px - radius - 1;                                                //  v.10.12
      py = py - radius - 1;
      ww = 2 * radius + 3;
      mwpaint3(px,py,ww,ww);

      npix = ppixsave1->npix;
      zfree(ppixsave1);
      pixed_undopixmem[pindex] = 0;
      pixed_undototmem -= (npix * 6 + 12);
      --pixed_undototpix;
   }
   
   if (pixed_undoseq > 0) --pixed_undoseq;

   mempercent = int(100.0 * pixed_undototmem / pixed_undomaxmem);          //  update undo memory status
   snprintf(undomemmessage,99,pixed_undomemmessage,mempercent,'%');
   zdialog_stuff(zdedit,"labmem",undomemmessage);

   return;
}


//  free all undo memory

void pixed_freeundo()
{
   int            pindex;
   pixed_savepix  *ppixsave1;
   char           undomemmessage[100];

   pindex = pixed_undototpix;
   
   while (pindex > 0)
   {
      --pindex;
      ppixsave1 = pixed_undopixmem[pindex];
      zfree(ppixsave1);
   }
   
   if (pixed_undopixmem) zfree(pixed_undopixmem);
   pixed_undopixmem = 0;
   
   pixed_undoseq = 0;
   pixed_undototpix = 0;
   pixed_undototmem = 0;

   if (zdedit) {
      snprintf(undomemmessage,99,pixed_undomemmessage,0,'%');              //  undo memory = 0%
      zdialog_stuff(zdedit,"labmem",undomemmessage);
   }

   return;
}


/**************************************************************************

   File scope variables and functions for composite images
      used by HDR, HDF, STP, STN, Panorama.

***************************************************************************/

int      cimNF;                                                   //  image count, <= 10
char     *cimFile[10];                                            //  image files
PXM      *cimPXMf[10];                                            //  original images
PXM      *cimPXMs[10];                                            //  alignment images, scaled and curved (pano)
PXM      *cimPXMw[10];                                            //  alignment images, warped

struct cimoffs  {                                                 //  image alignment offsets
   double   xf, yf, tf;                                           //  x, y, theta offsets
   double   wx[4], wy[4];                                         //  x/y corner warps, 0=NW, 1=NE, 2=SE, 3=SW
};
cimoffs  cimOffs[10];                                             //  image alignment data in E3 output image

double   cimScale;                                                //  alignment image size relative to full image
double   cimBlend;                                                //  image blend width at overlap, pixels (pano)
double   cimSearchRange;                                          //  alignment search range, pixels
double   cimSearchStep;                                           //  alignment search step, vpixels
double   cimWarpRange;                                            //  alignment corner warp range, pixels
double   cimWarpStep;                                             //  alignment corner warp step, vpixels
double   cimSampSize;                                             //  pixel sample size
int      cimOv1xlo, cimOv1xhi, cimOv1ylo, cimOv1yhi;              //  rectangle enclosing overlap area,
int      cimOv2xlo, cimOv2xhi, cimOv2ylo, cimOv2yhi;              //    image 1 and image2 coordinates
double   cimRGBmf1[3][65536];                                     //  RGB matching factors for pixel comparisons:
double   cimRGBmf2[3][65536];                                     //  cimRGBmf1[*][pix1[*]] == cimRGBmf2[*][pix2[*]]
char     *cimRedpix = 0;                                          //  maps high-contrast pixels for alignment
int      cimRedImage;                                             //  which image has red pixels
int      cimNsearch;                                              //  alignment search counter
int      cimShowIm1, cimShowIm2;                                  //  two images for cim_show_images()
int      cimShowAll;                                              //  if > 0, show all images
int      cimShrink;                                               //  image shrinkage from pano image curving
int      cimPano;                                                 //  pano mode flag for cim_align_image()
int      cimPanoV;                                                //  vertical pano flag

int      cim_load_files();                                        //  load and check selected files
void     cim_fixblue(PXM *pxm);                                   //  replace blue = 0 pixels with blue = 1
void     cim_scale_image(int im, PXM **);                         //  scale image, 1.0 to cimScale (normally < 1)
double   cim_get_overlap(int im1, int im2, PXM **);               //  get overlap area for images (horiz or vert)
void     cim_match_colors(int im1, int im2, PXM **);              //  match image RGB levels >> match data
void     cim_adjust_colors(PXM *pxm, int fwhich);                 //  adjust RGB levels from match data
void     cim_get_redpix(int im1);                                 //  find high-contrast pixels in overlap area
void     cim_curve_image(int im);                                 //  curve cimPXMs[im] using lens parameters
void     cim_curve_Vimage(int im);                                //  vertical pano version
void     cim_warp_image(int im);                                  //  warp image corners: cimPXMs[im] >> cimPXMw[im]
void     cim_warp_image_pano(int im, int fblend);                 //  pano version, all / left side / blend stripe
void     cim_warp_image_Vpano(int im, int fblend);                //  vertical pans version: bottom side corners
void     cim_align_image(int im1, int im2);                       //  align image im2 to im1, modify im2 offsets
double   cim_match_images(int im1, int im2);                      //  compute match for overlapped images
void     cim_show_images(int fnew, int fblend);                   //  combine images >> E3pxm16 >> main window
void     cim_show_Vimages(int fnew, int fblend);                  //  vertical pano version
void     cim_trim();                                              //  cut-off edges where all images do not overlap
void     cim_dump_offsets(cchar *text);                           //  diagnostic tool


//  load image file into pixmaps cimPXMf[*] and check for errors
//  returns 0 if error

int cim_load_files()                                                       //  v.10.7
{
   PXM      *pxm;

   for (int imx = 0; imx < cimNF; imx++)
   {
      PXM_free(cimPXMf[imx]);
      pxm = f_load(cimFile[imx],16);                                       //  will diagnose errors
      if (! pxm) return 0;
      cimPXMf[imx] = pxm;
      
      cim_fixblue(pxm);
   }

   return 1;
}


//  replace blue = 0 pixels with blue = 1
//  (blue = 0 reserved for pixels voided by warp or overlay offsets)

void cim_fixblue(PXM *pxm)                                                 //  v.10.7
{
   int      ww, hh, px, py;
   uint16   *pixel;

   ww = pxm->ww;
   hh = pxm->hh;

   for (py = 0; py < hh; py++)
   for (px = 0; px < ww; px++) 
   {
      pixel = PXMpix(pxm,px,py);
      if (! pixel[2]) pixel[2] = 1;
   }

   return;
}


//  scale image from full size) to cimScale (normally < 1.0)

void cim_scale_image(int im, PXM** pxmout)                                 //  v.10.7
{
   int      ww, hh;
   
   ww = cimScale * cimPXMf[im]->ww;
   hh = cimScale * cimPXMf[im]->hh;

   PXM_free(pxmout[im]);   
   pxmout[im] = PXM_rescale(cimPXMf[im],ww,hh);
   
   cim_fixblue(pxmout[im]);

   return;
}


//  get overlap area for a pair of images im1 and im2
//  outputs are coordinates of overlap area in im1 and in im2
//  returns overlap width as fraction of image width <= 1.0

double cim_get_overlap(int im1, int im2, PXM **pxmx)                       //  v.11.04
{
   double      x1, y1, t1, x2, y2, t2;
   double      xoff, yoff, toff, costf, sintf;
   int         ww1, ww2, hh1, hh2, pxM;
   PXM         *pxm1, *pxm2;

   x1 = cimOffs[im1].xf;                                                   //  im1, im2 absolute offsets
   y1 = cimOffs[im1].yf;
   t1 = cimOffs[im1].tf;
   x2 = cimOffs[im2].xf;
   y2 = cimOffs[im2].yf;
   t2 = cimOffs[im2].tf;
   
   xoff = (x2 - x1) * cos(t1) + (y2 - y1) * sin(t1);                       //  offset of im2 relative to im1
   yoff = (y2 - y1) * cos(t1) - (x2 - x1) * sin(t1);
   toff = t2 - t1;

   costf = cos(toff);
   sintf = sin(toff);

   pxm1 = pxmx[im1];
   pxm2 = pxmx[im2];

   ww1 = pxm1->ww;
   hh1 = pxm1->hh;
   ww2 = pxm2->ww;
   hh2 = pxm2->hh;
   
   cimOv1xlo = 0;                                                          //  lowest x overlap
   if (xoff > 0) cimOv1xlo = xoff;

   cimOv1xhi = ww1-1;                                                      //  highest x overlap
   if (cimOv1xhi > xoff + ww2-1) cimOv1xhi = xoff + ww2-1;

   cimOv1ylo = 0;                                                          //  lowest y overlap
   if (yoff > 0) cimOv1ylo = yoff;

   cimOv1yhi = hh1-1;                                                      //  highest y overlap
   if (cimOv1yhi > yoff + hh2-1) cimOv1yhi = yoff + hh2-1;
   
   if (toff < 0) cimOv1xlo -= toff * (cimOv1yhi - cimOv1ylo);              //  reduce for theta offset
   if (toff < 0) cimOv1yhi += toff * (cimOv1xhi - cimOv1xlo);
   if (toff > 0) cimOv1xhi -= toff * (cimOv1yhi - cimOv1ylo);
   if (toff > 0) cimOv1ylo += toff * (cimOv1xhi - cimOv1xlo);
   
   cimOv1xlo += cimShrink + 3;                                             //  account for void areas from
   cimOv1xhi += - cimShrink - 3;                                           //    image shrinkage from 
   cimOv1ylo += cimShrink + 3;                                             //      cim_curve_image()
   cimOv1yhi += - cimShrink - 3;                                           //  v.11.04
   
   if (cimPanoV) {
      if (cimBlend && cimBlend < (cimOv1yhi - cimOv1ylo)) {                //  reduce y range to cimBlend   v.11.04
         pxM = (cimOv1yhi + cimOv1ylo) / 2;
         cimOv1ylo = pxM - cimBlend / 2;
         cimOv1yhi = pxM + cimBlend / 2;
      }
   }
   else {
      if (cimBlend && cimBlend < (cimOv1xhi - cimOv1xlo)) {                //  reduce x range to cimBlend
         pxM = (cimOv1xhi + cimOv1xlo) / 2;
         cimOv1xlo = pxM - cimBlend / 2;
         cimOv1xhi = pxM + cimBlend / 2;
      }
   }

   cimOv2xlo = costf * (cimOv1xlo - xoff) + sintf * (cimOv1ylo - yoff);    //  overlap area in im2 coordinates
   cimOv2xhi = costf * (cimOv1xhi - xoff) + sintf * (cimOv1yhi - yoff);
   cimOv2ylo = costf * (cimOv1ylo - yoff) + sintf * (cimOv1xlo - xoff);
   cimOv2yhi = costf * (cimOv1yhi - yoff) + sintf * (cimOv1xhi - xoff);

   if (cimOv1xlo < 0) cimOv1xlo = 0;                                       //  take care of limits
   if (cimOv1ylo < 0) cimOv1ylo = 0;
   if (cimOv2xlo < 0) cimOv2xlo = 0;
   if (cimOv2ylo < 0) cimOv2ylo = 0;
   if (cimOv1xhi > ww1-1) cimOv1xhi = ww1-1;
   if (cimOv1yhi > hh1-1) cimOv1yhi = hh1-1;
   if (cimOv2xhi > ww2-1) cimOv2xhi = ww2-1;
   if (cimOv2yhi > hh2-1) cimOv2yhi = hh2-1;
   
   if (cimPanoV) return 1.0 * (cimOv1yhi - cimOv1ylo) / hh1;               //  return overlap height <= 1.0  v.11.04
   else return 1.0 * (cimOv1xhi - cimOv1xlo) / ww1;                        //  return overlap width <= 1.0  v.11.03
}


//  Get the RGB brightness distribution in the overlap area for each image.
//  Compute matching factors to compare pixels within the overlap area.
//  compare  cimRGBmf1[rgb][pix1[rgb]]  to  cimRGBmf2[rgb][pix2[rgb]]

void cim_match_colors(int im1, int im2, PXM **pxmx)                        //  v.10.7
{
   double      Bratios1[3][256];                //  image2/image1 brightness ratio per color per level
   double      Bratios2[3][256];                //  image1/image2 brightness ratio per color per level

   uint16      *pix1, vpix2[3];
   int         vstat2, px1, py1;
   int         ii, jj, rgb;
   int         npix, npix1, npix2, npix3;
   int         brdist1[3][256], brdist2[3][256];
   double      x1, y1, t1, x2, y2, t2;
   double      xoff, yoff, toff, costf, sintf;
   double      px2, py2;
   double      brlev1[3][256], brlev2[3][256];
   double      a1, a2, b1, b2, bratio = 1;
   double      r256 = 1.0 / 256.0;
   PXM         *pxm1, *pxm2;
   
   pxm1 = pxmx[im1];
   pxm2 = pxmx[im2];
   
   x1 = cimOffs[im1].xf;                                                   //  im1, im2 absolute offsets
   y1 = cimOffs[im1].yf;
   t1 = cimOffs[im1].tf;
   x2 = cimOffs[im2].xf;
   y2 = cimOffs[im2].yf;
   t2 = cimOffs[im2].tf;
   
   xoff = (x2 - x1) * cos(t1) + (y2 - y1) * sin(t1);                       //  offset of im2 relative to im1
   yoff = (y2 - y1) * cos(t1) - (x2 - x1) * sin(t1);
   toff = t2 - t1;

   costf = cos(toff);
   sintf = sin(toff);

   for (rgb = 0; rgb < 3; rgb++)                                           //  clear distributions
   for (ii = 0; ii < 256; ii++)
      brdist1[rgb][ii] = brdist2[rgb][ii] = 0;

   npix = 0;

   for (py1 = cimOv1ylo; py1 < cimOv1yhi; py1++)                           //  loop overlapped rows
   for (px1 = cimOv1xlo; px1 < cimOv1xhi; px1++)                           //  loop overlapped columns
   {
      pix1 = PXMpix(pxm1,px1,py1);                                         //  image1 pixel
      if (! pix1[2]) continue;                                             //  ignore void pixels

      px2 = costf * (px1 - xoff) + sintf * (py1 - yoff);                   //  corresponding image2 pixel
      py2 = costf * (py1 - yoff) - sintf * (px1 - xoff);
      vstat2 = vpixel(pxm2,px2,py2,vpix2);
      if (! vstat2) continue;                                              //  does not exist

      ++npix;                                                              //  count overlapping pixels
      
      for (rgb = 0; rgb < 3; rgb++)                                        //  accumulate distributions
      {                                                                    //    by color in 256 bins
         ++brdist1[rgb][int(r256*pix1[rgb])];
         ++brdist2[rgb][int(r256*vpix2[rgb])];
      }
   }
   
   npix1 = npix / 256;                                                     //  1/256th of total pixels
   
   for (rgb = 0; rgb < 3; rgb++)                                           //  get brlev1[rgb][N] = mean bright
   for (ii = jj = 0; jj < 256; jj++)                                       //    for Nth group of image1 pixels
   {                                                                       //      for color rgb
      brlev1[rgb][jj] = 0;
      npix2 = npix1;                                                       //  1/256th of total pixels

      while (npix2 > 0 && ii < 256)                                        //  next 1/256th group from distr,
      {
         npix3 = brdist1[rgb][ii];
         if (npix3 == 0) { ++ii; continue; }
         if (npix3 > npix2) npix3 = npix2;
         brlev1[rgb][jj] += ii * npix3;                                    //  brightness * (pixels with)
         brdist1[rgb][ii] -= npix3;
         npix2 -= npix3;
      }

      brlev1[rgb][jj] = brlev1[rgb][jj] / npix1;                           //  mean brightness for group, 0-255
   }

   for (rgb = 0; rgb < 3; rgb++)                                           //  do same for image2
   for (ii = jj = 0; jj < 256; jj++)
   {
      brlev2[rgb][jj] = 0;
      npix2 = npix1;

      while (npix2 > 0 && ii < 256)
      {
         npix3 = brdist2[rgb][ii];
         if (npix3 == 0) { ++ii; continue; }
         if (npix3 > npix2) npix3 = npix2;
         brlev2[rgb][jj] += ii * npix3;
         brdist2[rgb][ii] -= npix3;
         npix2 -= npix3;
      }

      brlev2[rgb][jj] = brlev2[rgb][jj] / npix1;
   }

   for (rgb = 0; rgb < 3; rgb++)                                           //  color
   for (ii = jj = 0; ii < 256; ii++)                                       //  brlev1 brightness, 0 to 255
   {
      if (ii == 0) bratio = 1;
      while (ii > brlev2[rgb][jj] && jj < 256) ++jj;                       //  find matching brlev2 brightness
      a2 = brlev2[rgb][jj];                                                //  next higher value
      b2 = brlev1[rgb][jj];
      if (a2 > 0 && b2 > 0) {
         if (jj > 0) {
            a1 = brlev2[rgb][jj-1];                                        //  next lower value
            b1 = brlev1[rgb][jj-1];
         }
         else   a1 = b1 = 0;
         if (ii == 0)  bratio = b2 / a2;
         else   bratio = (b1 + (ii-a1)/(a2-a1) * (b2-b1)) / ii;            //  interpolate
      }

      if (bratio < 0.2) bratio = 0.2;                                      //  contain outliers
      if (bratio > 5) bratio = 5;
      Bratios2[rgb][ii] = bratio;
   }

   for (rgb = 0; rgb < 3; rgb++)                                           //  color
   for (ii = jj = 0; ii < 256; ii++)                                       //  brlev2 brightness, 0 to 255
   {
      if (ii == 0) bratio = 1;
      while (ii > brlev1[rgb][jj] && jj < 256) ++jj;                       //  find matching brlev1 brightness
      a2 = brlev1[rgb][jj];                                                //  next higher value
      b2 = brlev2[rgb][jj];
      if (a2 > 0 && b2 > 0) {
         if (jj > 0) {
            a1 = brlev1[rgb][jj-1];                                        //  next lower value
            b1 = brlev2[rgb][jj-1];
         }
         else   a1 = b1 = 0;
         if (ii == 0)  bratio = b2 / a2;
         else   bratio = (b1 + (ii-a1)/(a2-a1) * (b2-b1)) / ii;            //  interpolate
      }

      if (bratio < 0.2) bratio = 0.2;                                      //  contain outliers
      if (bratio > 5) bratio = 5;
      Bratios1[rgb][ii] = bratio;
   }
   
   for (ii = 0; ii < 65536; ii++)                                          //  convert brightness ratios into
   {                                                                       //    conversion factors
      jj = ii / 256;

      for (rgb = 0; rgb < 3; rgb++)
      {
         cimRGBmf1[rgb][ii] = sqrt(Bratios1[rgb][jj]) * ii;                //  use sqrt(ratio) so that adjustment
         cimRGBmf2[rgb][ii] = sqrt(Bratios2[rgb][jj]) * ii;                //    can be applied to both images
      }
   }
   
   return;
}


//  Use color match data from cim_match_colors() to 
//    modify images so the colors match.

void cim_adjust_colors(PXM *pxm, int fwhich)                               //  v.10.7
{
   int         ww, hh, px, py;
   int         red, green, blue, max;
   uint16      *pix;
   double      f1;
   
   ww = pxm->ww;
   hh = pxm->hh;

   for (py = 0; py < hh; py++)
   for (px = 0; px < ww; px++)
   {
      pix = PXMpix(pxm,px,py);
      red = pix[0];
      green = pix[1];
      blue = pix[2];
      if (! blue) continue;

      if (fwhich == 1) {
         red = cimRGBmf1[0][red];
         green = cimRGBmf1[1][green];
         blue = cimRGBmf1[2][blue];
      }

      if (fwhich == 2) {
         red = cimRGBmf2[0][red];
         green = cimRGBmf2[1][green];
         blue = cimRGBmf2[2][blue];
      }

      if (red > 65535 || green > 65535 || blue > 65535) {
         max = red;
         if (green > max) max = green;
         if (blue > max) max = blue;
         f1 = 65535.0 / max;
         red = red * f1;
         green = green * f1;
         blue = blue * f1;
      }
      
      if (! blue) blue = 1;                                                //  avoid 0    v.10.7

      pix[0] = red;
      pix[1] = green;
      pix[2] = blue;
   }

   return;
}


//  find pixels of greatest contrast within overlap area
//  flag high-contrast pixels to use in each image compare region

void cim_get_redpix(int im1)                                               //  v.10.7
{
   int         ww, hh, samp, xzone, yzone;
   int         pxL, pxH, pyL, pyH;
   int         px, py, ii, jj, npix;
   int         red1, green1, blue1, red2, green2, blue2, tcon;
   int         ov1xlo, ov1xhi, ov1ylo, ov1yhi;
   int         Hdist[256], Vdist[256], Hmin, Vmin;
   double      s8 = 1.0 / 770.0;
   double      zsamp[16] = { 4,6,6,4,6,9,9,6,6,9,9,6,4,6,6,4 };            //  % sample per zone, sum = 100
   uchar       *Hcon, *Vcon;
   uint16      *pix1, *pix2;
   PXM         *pxm;
   
   pxm = cimPXMs[im1];                                                     //  v.11.04
   ww = pxm->ww;
   hh = pxm->hh;

   if (cimRedpix) zfree(cimRedpix);                                        //  clear prior
   cimRedpix = zmalloc(ww*hh,"cimRedpix");
   memset(cimRedpix,0,ww*hh);

   cimRedImage = im1;                                                      //  image with red pixels
   
   ov1xlo = cimOv1xlo + cimSearchRange;                                    //  stay within x/y search range
   ov1xhi = cimOv1xhi - cimSearchRange;                                    //    so that red pixels persist
   ov1ylo = cimOv1ylo + cimSearchRange;                                    //      over offset changes
   ov1yhi = cimOv1yhi - cimSearchRange;
   
   for (yzone = 0; yzone < 4; yzone++)                                     //  loop 16 zones       v.10.8
   for (xzone = 0; xzone < 4; xzone++)
   {
      pxL = ov1xlo + 0.25 * xzone     * (ov1xhi - ov1xlo);                 //  px and py zone limits
      pxH = ov1xlo + 0.25 * (xzone+1) * (ov1xhi - ov1xlo);
      pyL = ov1ylo + 0.25 * yzone     * (ov1yhi - ov1ylo);
      pyH = ov1ylo + 0.25 * (yzone+1) * (ov1yhi - ov1ylo);
      
      npix = (pxH - pxL) * (pyH - pyL);                                    //  zone pixels
      Hcon = (uchar *) zmalloc(npix,"redpix.hcon");                        //  horizontal pixel contrast 0-255
      Vcon = (uchar *) zmalloc(npix,"redpix.vcon");                        //  vertical pixel contrast 0-255
      
      ii = 4 * yzone + xzone;
      samp = cimSampSize * 0.01 * zsamp[ii];                               //  sample size for zone
      if (samp > 0.1 * npix) samp = 0.1 * npix;                            //  limit to 10% of zone pixels
      
      for (py = pyL; py < pyH; py++)                                       //  scan image pixels in zone
      for (px = pxL; px < pxH; px++)
      {
         ii = (py-pyL) * (pxH-pxL) + (px-pxL);
         Hcon[ii] = Vcon[ii] = 0;                                          //  horiz. = vert. contrast = 0

         if (py < 8 || py > hh-9) continue;                                //  keep away from image edges
         if (px < 8 || px > ww-9) continue;

         pix1 = PXMpix(pxm,px,py-6);                                       //  verify not near void areas
         if (! pix1[2]) continue;
         pix1 = PXMpix(pxm,px+6,py);
         if (! pix1[2]) continue;
         pix1 = PXMpix(pxm,px,py+6);
         if (! pix1[2]) continue;
         pix1 = PXMpix(pxm,px-6,py);
         if (! pix1[2]) continue;

         pix1 = PXMpix(pxm,px,py);                                         //  candidate red pixel
         red1 = pix1[0];
         green1 = pix1[1];
         blue1 = pix1[2];

         pix2 = PXMpix(pxm,px+2,py);                                       //  2 pixels to right
         red2 = pix2[0];
         green2 = pix2[1];
         blue2 = pix2[2];

         tcon = abs(red1-red2) + abs(green1-green2) + abs(blue1-blue2);    //  horizontal contrast
         Hcon[ii] = int(tcon * s8);                                        //  scale  0 - 255

         pix2 = PXMpix(pxm,px,py+2);                                       //  2 pixels below
         red2 = pix2[0];
         green2 = pix2[1];
         blue2 = pix2[2];

         tcon = abs(red1-red2) + abs(green1-green2) + abs(blue1-blue2);    //  vertical contrast
         Vcon[ii] = int(tcon * s8);
      }

      for (ii = 0; ii < 256; ii++) Hdist[ii] = Vdist[ii] = 0;              //  clear contrast distributions

      for (py = pyL; py < pyH; py++)                                       //  scan image pixels
      for (px = pxL; px < pxH; px++)
      {                                                                    //  build contrast distributions
         ii = (py-pyL) * (pxH-pxL) + (px-pxL);
         ++Hdist[Hcon[ii]];
         ++Vdist[Vcon[ii]];
      }
      
      for (npix = 0, ii = 255; ii > 0; ii--)                               //  find minimum contrast needed to get
      {                                                                    //    enough pixels for sample size
         npix += Hdist[ii];                                                //      (horizontal contrast pixels)
         if (npix > samp) break; 
      }
      Hmin = ii; 

      for (npix = 0, ii = 255; ii > 0; ii--)                               //  (verticle contrast pixels)
      {
         npix += Vdist[ii];
         if (npix > samp) break;
      }
      Vmin = ii;

      for (py = pyL; py < pyH; py++)                                       //  scan zone pixels
      for (px = pxL; px < pxH; px++)
      {
         ii = (py-pyL) * (pxH-pxL) + (px-pxL);
         jj = py * ww + px;
         if (Hcon[ii] > Hmin) cimRedpix[jj] = 1;                           //  flag pixels above min. contrast
         if (Vcon[ii] > Vmin) cimRedpix[jj] = 1;
      }

      zfree(Hcon);
      zfree(Vcon);

      for (py = pyL; py < pyH; py++)                                       //  scan zone pixels
      for (px = pxL; px < pxH; px++)
      {
         ii = (py-pyL) * (pxH-pxL) + (px-pxL);
         jj = py * ww + px;
         if (! cimRedpix[jj]) continue;
         npix = cimRedpix[jj-1] + cimRedpix[jj+1];                         //  eliminate flagged pixels with no
         npix += cimRedpix[jj-ww] + cimRedpix[jj+ww];                      //    neighboring flagged pixels
         npix += cimRedpix[jj-ww-1] + cimRedpix[jj+ww-1];                  //  v.11.03
         npix += cimRedpix[jj-ww+1] + cimRedpix[jj+ww+1];
         if (npix < 2) cimRedpix[jj] = 0;
      }

      for (py = pyL; py < pyH; py++)                                       //  scan zone pixels
      for (px = pxL; px < pxH; px++)
      {
         ii = (py-pyL) * (pxH-pxL) + (px-pxL);
         jj = py * ww + px;

         if (cimRedpix[jj] == 1) {                                         //  flag horizontal group of 3
            cimRedpix[jj+1] = 2;
            cimRedpix[jj+2] = 2;
            cimRedpix[jj+ww] = 2;                                          //  and vertical group of 3
            cimRedpix[jj+2*ww] = 2;
         }
      }
   }
   
   return;
}


//  curve image based on lens parameters (pano)
//  replaces cimPXMs[im] with curved version

void cim_curve_image(int im)                                               //  overhauled    v.11.03
{
   int         px, py, ww, hh, vstat;
   double      ww2, hh2;
   double      dx, dy;
   double      F = lens_mm;                                                //  lens focal length, 35mm equivalent
   double      S = 35.0;                                                   //  corresponding image width
   double      R1, R2, G, T, bow;
   PXM         *pxmin, *pxmout;
   uint16      vpix[3], *pix;
   
   pxmin = cimPXMs[im];                                                    //  input and output image
   ww = pxmin->ww;                  //    200
   hh = pxmin->hh;
   ww2 = 0.5 * ww;                  //    100
   hh2 = 0.5 * hh;
   
   if (hh > ww) S = S * ww / hh;                                           //  vertical format
   F = F / S;                       //    28 / 35                          //  scale to image dimensions
   S = ww2;                         //    100
   F = F * ww;                      //    160
   R1 = F;                                                                 //  cylinder tangent to image plane
   
   bow = -lens_bow * 0.01 / hh2 / hh2;                                     //  lens bow % to fraction
   if (hh > ww) 
      bow = -lens_bow * 0.01 / ww2 / ww2;

   pxmout = PXM_make(ww,hh,16);                                            //  temp. output PXM
   
   for (py = 0; py < hh; py++)                                             //  cylindrical projection    v.11.03
   for (px = 0; px < ww; px++)
   {
      dx = px - ww2;
      dy = py - hh2;
      T = dx / R1;
      dx = F * tan(T);
      R2 = sqrt(dx * dx + F * F);
      G = R1 - R2;
      dy = (dy * R2) / (R2 + G);
      dx += bow * dx * dy * dy;                                            //  barrel distortion
      dx += ww2;
      dy += hh2;
      vstat = vpixel(pxmin,dx,dy,vpix);                                    //  input virtual pixel
      pix = PXMpix(pxmout,px,py);                                          //  output real pixel
      if (vstat) {  
         pix[0] = vpix[0];
         pix[1] = vpix[1];
         pix[2] = vpix[2];
      }
      else pix[0] = pix[1] = pix[2] = 0;                                   //  voided pixels are (0,0,0)
   }

   for (px = 1; px < ww2; px++) {                                          //  compute image shrinkage
      pix = PXMpix(pxmout,px,hh/2);
      if (pix[2]) break;
   }
   cimShrink = px-1;                                                       //  = 0 if no curvature
   
   PXM_free(pxmin);                                                        //  replace input with output PXM
   cimPXMs[im] = pxmout;

   return;
}


//  version for vertical panorama

void cim_curve_Vimage(int im)                                              //  v.11.04
{
   int         px, py, ww, hh, vstat;
   double      ww2, hh2;
   double      dx, dy;
   double      F = lens_mm;                                                //  lens focal length, 35mm equivalent
   double      S = 35.0;                                                   //  corresponding image width
   double      R1, R2, G, T, bow;
   PXM         *pxmin, *pxmout;
   uint16      vpix[3], *pix;
   
   pxmin = cimPXMs[im];                                                    //  input and output image
   ww = pxmin->ww;                  //    200
   hh = pxmin->hh;
   ww2 = 0.5 * ww;                  //    100
   hh2 = 0.5 * hh;
   
   if (hh > ww) S = S * ww / hh;                                           //  vertical format
   F = F / S;                       //    28 / 35                          //  scale to image dimensions
   S = ww2;                         //    100
   F = F * ww;                      //    160
   R1 = F;                                                                 //  cylinder tangent to image plane
   
   bow = -lens_bow * 0.01 / hh2 / hh2;                                     //  lens bow % to fraction
   if (hh > ww) 
      bow = -lens_bow * 0.01 / ww2 / ww2;

   pxmout = PXM_make(ww,hh,16);                                            //  temp. output PXM
   
   for (py = 0; py < hh; py++)                                             //  cylindrical projection    v.11.03
   for (px = 0; px < ww; px++)
   {
      dx = px - ww2;
      dy = py - hh2;
      T = dy / R1;
      dy = F * tan(T);
      R2 = sqrt(dy * dy + F * F);
      G = R1 - R2;
      dx = (dx * R2) / (R2 + G);
      dy += bow * dy * dx * dx;                                            //  barrel distortion
      dx += ww2;
      dy += hh2;
      vstat = vpixel(pxmin,dx,dy,vpix);                                    //  input virtual pixel
      pix = PXMpix(pxmout,px,py);                                          //  output real pixel
      if (vstat) {  
         pix[0] = vpix[0];
         pix[1] = vpix[1];
         pix[2] = vpix[2];
      }
      else pix[0] = pix[1] = pix[2] = 0;                                   //  voided pixels are (0,0,0)
   }

   for (py = 1; py < hh2; py++) {                                          //  compute image shrinkage
      pix = PXMpix(pxmout,ww/2,py);
      if (pix[2]) break;
   }
   cimShrink = py-1;                                                       //  = 0 if no curvature
   
   PXM_free(pxmin);                                                        //  replace input with output PXM
   cimPXMs[im] = pxmout;

   return;
}


//  Warp 4 image corners according to cimOffs[im].wx[ii] and .wy[ii]
//  corner = 0 = NW,  1 = NE,  2 = SE,  3 = SW
//  4 corners move by these pixel amounts and center does not move.
//  input: cimPXMs[im] (flat or curved) output: cimPXMw[im]

namespace cim_warp_image_names {
   PXM         *pxmin, *pxmout;
   double      ww, hh, wwi, hhi;
   double      wx0, wy0, wx1, wy1, wx2, wy2, wx3, wy3;
}

void cim_warp_image(int im)                                                //  caller function     v.10.8
{
   using namespace cim_warp_image_names;

   void * cim_warp_image_wthread(void *arg);
   
   pxmin = cimPXMs[im];                                                    //  input and output pixmaps
   pxmout = cimPXMw[im];
   
   PXM_free(pxmout);                                                       //  v.11.04
   pxmout = PXM_copy(pxmin);
   cimPXMw[im] = pxmout;

   ww = pxmin->ww;
   hh = pxmin->hh;
   wwi = 1.0 / ww;
   hhi = 1.0 / hh;

   wx0 = cimOffs[im].wx[0];                                                //  corner warps
   wy0 = cimOffs[im].wy[0];
   wx1 = cimOffs[im].wx[1];
   wy1 = cimOffs[im].wy[1];
   wx2 = cimOffs[im].wx[2];
   wy2 = cimOffs[im].wy[2];
   wx3 = cimOffs[im].wx[3];
   wy3 = cimOffs[im].wy[3];

   for (int ii = 0; ii < Nwt; ii++)                                        //  start worker threads
      start_wt(cim_warp_image_wthread,&wtnx[ii]);
   wait_wts();                                                             //  wait for completion

   return;
}


void * cim_warp_image_wthread(void *arg)                                   //  worker thread function  v.10.8
{
   using namespace cim_warp_image_names;

   int         index = *((int *) arg);
   int         pxm, pym, vstat;
   uint16      vpix[3], *pixm;
   double      px, py, dx, dy, coeff;
   
   for (pym = index; pym < hh; pym += Nwt)                                 //  loop all pixels for this thread
   for (pxm = 0; pxm < ww; pxm++)
   {
      dx = dy = 0.0;

      coeff = (1.0 - pym * hhi - pxm * wwi);                               //  corner 0  NW
      if (coeff > 0) {
         dx += coeff * wx0;
         dy += coeff * wy0;
      }
      coeff = (1.0 - pym * hhi - (ww - pxm) * wwi);                        //  corner 1  NE
      if (coeff > 0) {
         dx += coeff * wx1;
         dy += coeff * wy1;
      }
      coeff = (1.0 - (hh - pym) * hhi - (ww - pxm) * wwi);                 //  corner 2  SE
      if (coeff > 0) {
         dx += coeff * wx2;
         dy += coeff * wy2;
      }
      coeff = (1.0 - (hh - pym) * hhi - pxm * wwi);                        //  corner 3  SW
      if (coeff > 0) {
         dx += coeff * wx3;
         dy += coeff * wy3;
      }

      px = pxm + dx;                                                       //  source pixel location
      py = pym + dy;

      vstat = vpixel(pxmin,px,py,vpix);                                    //  input virtual pixel
      pixm = PXMpix(pxmout,pxm,pym);                                       //  output real pixel

      if (vstat) {  
         pixm[0] = vpix[0];
         pixm[1] = vpix[1];
         pixm[2] = vpix[2];
      }
      else pixm[0] = pixm[1] = pixm[2] = 0;
   }
   
   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


//  warp image for pano, left side corners only, reduced warp range
//  input: cimPXMs[im] (curved) 
//  output: cimPXMw[im] (warped)
//  fblend: 0 = process entire image
//          1 = process left half only
//          2 = process blend stripe only

void cim_warp_image_pano(int im, int fblend)                               //  v.10.8
{
   int         ww, hh, ww2, hh2, pxL, pxH;
   int         pxm, pym, vstat;
   uint16      vpix[3], *pixm;
   double      ww2i, hh2i, pxs, pys, xdisp, ydisp;
   double      wx0, wy0, wx3, wy3;
   PXM         *pxmin, *pxmout;
   
   pxmin = cimPXMs[im];                                                    //  input and output pixmaps
   pxmout = cimPXMw[im];
   
   PXM_free(pxmout);                                                       //  v.11.04
   pxmout = PXM_copy(pxmin);
   cimPXMw[im] = pxmout;

   ww = pxmin->ww;
   hh = pxmin->hh;

   ww2 = ww / 2;
   hh2 = hh / 2;

   ww2i = 1.0 / ww2;                                                       //  v.10.8
   hh2i = 1.0 / hh2;

   wx0 = cimOffs[im].wx[0];                                                //  NW corner warp
   wy0 = cimOffs[im].wy[0];

   wx3 = cimOffs[im].wx[3];                                                //  SW corner warp
   wy3 = cimOffs[im].wy[3];

   pxL = 0;                                                                //  entire image        v.10.8
   pxH = ww;
   
   if (fblend == 1)                                                        //  left half
      pxH = ww2;

   if (fblend == 2) {
      pxL = cimOv2xlo;                                                     //  limit to overlap/blend width
      pxH = cimOv2xhi;
   }
   
   for (pym = 0; pym < hh; pym++)                                          //  loop all output pixels
   for (pxm = pxL; pxm < pxH; pxm++)
   {
      pixm = PXMpix(pxmout,pxm,pym);                                       //  output pixel
      
      xdisp = (pxm - ww2) * ww2i;                                          //  -1 ... 0 ... +1
      ydisp = (pym - hh2) * hh2i;                                          //  v.11.04
      
      if (xdisp > 0) {                                                     //  right half, no warp
         pxs = pxm;
         pys = pym;
      }
      else if (ydisp < 0) {                                                //  use NW corner warp
         pxs = pxm + wx0 * xdisp * ydisp;
         pys = pym + wy0 * xdisp * ydisp;
      }
      else {                                                               //  use SW corner warp
         pxs = pxm + wx3 * xdisp * ydisp;
         pys = pym + wy3 * xdisp * ydisp;
      }

      vstat = vpixel(pxmin,pxs,pys,vpix);                                  //  input virtual pixel

      if (vstat) {  
         pixm[0] = vpix[0];
         pixm[1] = vpix[1];
         pixm[2] = vpix[2];
      }
      else pixm[0] = pixm[1] = pixm[2] = 0;
   }

   for (pxm = 1; pxm < ww2; pxm++) {                                       //  compute image shrinkage
      pixm = PXMpix(pxmout,pxm,hh2);                                       //    used by cim_get_overlap()
      if (pixm[2]) break;
   }
   cimShrink = pxm-1;

   return;
}


//  vertical pano version - warp top side corners (NW, NE)

void cim_warp_image_Vpano(int im, int fblend)                              //  v.11.04
{
   int         ww, hh, ww2, hh2, pyL, pyH;
   int         pxm, pym, vstat;
   uint16      vpix[3], *pixm;
   double      ww2i, hh2i, pxs, pys, xdisp, ydisp;
   double      wx0, wy0, wx1, wy1;
   PXM         *pxmin, *pxmout;
   
   pxmin = cimPXMs[im];                                                    //  input and output pixmaps
   pxmout = cimPXMw[im];
   
   PXM_free(pxmout);                                                       //  v.11.04
   pxmout = PXM_copy(pxmin);
   cimPXMw[im] = pxmout;

   ww = pxmin->ww;
   hh = pxmin->hh;

   ww2 = ww / 2;
   hh2 = hh / 2;

   ww2i = 1.0 / ww2;                                                       //  v.10.8
   hh2i = 1.0 / hh2;

   wx0 = cimOffs[im].wx[0];                                                //  NW corner warp
   wy0 = cimOffs[im].wy[0];

   wx1 = cimOffs[im].wx[1];                                                //  NE corner warp
   wy1 = cimOffs[im].wy[1];

   pyL = 0;                                                                //  entire image        v.10.8
   pyH = hh;
   
   if (fblend == 1)                                                        //  top half
      pyH = hh2;

   if (fblend == 2) {
      pyL = cimOv2ylo;                                                     //  limit to overlap/blend width
      pyH = cimOv2yhi;
   }
   
   for (pym = pyL; pym < pyH; pym++)                                       //  loop all output pixels
   for (pxm = 0; pxm < ww; pxm++)
   {
      pixm = PXMpix(pxmout,pxm,pym);                                       //  output pixel
      
      xdisp = (pxm - ww2) * ww2i;                                          //  -1 ... 0 ... +1
      ydisp = (pym - hh2) * hh2i;
      
      if (ydisp > 0) {                                                     //  bottom half, no warp
         pxs = pxm;
         pys = pym;
      }
      else if (xdisp < 0) {                                                //  use NW corner warp
         pxs = pxm + wx0 * xdisp * ydisp;
         pys = pym + wy0 * xdisp * ydisp;
      }
      else {                                                               //  use NE corner warp
         pxs = pxm + wx1 * xdisp * ydisp;
         pys = pym + wy1 * xdisp * ydisp;
      }

      vstat = vpixel(pxmin,pxs,pys,vpix);                                  //  input virtual pixel

      if (vstat) {  
         pixm[0] = vpix[0];
         pixm[1] = vpix[1];
         pixm[2] = vpix[2];
      }
      else pixm[0] = pixm[1] = pixm[2] = 0;
   }

   for (pym = 1; pym < hh2; pym++) {                                       //  compute image shrinkage
      pixm = PXMpix(pxmout,ww2,pym);                                       //    used by cim_get_overlap()
      if (pixm[2]) break;
   }
   cimShrink = pym-1;

   return;
}


//  fine-align a pair of images im1 and im2
//  cimPXMs[im2] is aligned with cimPXMs[im1]
//  inputs are cimOffs[im1] and cimOffs[im2] (x/y/t and corner offsets)
//  output is adjusted offsets and corner warp values for im2 only
//  (im1 is used as-is without corner warps)

void cim_align_image(int im1, int im2)                                     //  speedup v.11.03
{
   int         ii, corner1, cornerstep, cornerN, pass;
   double      xyrange, xystep, trange, tstep, wrange, wstep;
   double      xfL, xfH, yfL, yfH, tfL, tfH;
   double      wxL, wxH, wyL, wyH;
   double      match, matchB;
   cimoffs     offsets0, offsetsB;
   
   offsets0 = cimOffs[im2];                                                //  initial offsets
   offsetsB = offsets0;                                                    //  = best offsets so far
   matchB = cim_match_images(im1,im2);                                     //  = best image match level

   if (cimPanoV) cim_show_Vimages(0,0);                                    //  v.11.04
   else cim_show_images(0,0);                                              //  show with 50/50 blend

   for (pass = 1; pass <=2; pass++)                                        //  main pass and 2nd pass    v.10.8
   {
      xyrange = cimSearchRange;                                            //  x/y search range and step
      xystep = cimSearchStep;   

      trange = xyrange / (cimOv1yhi - cimOv1ylo);                          //  angle range, radians
      tstep = trange * xystep / xyrange;
      
      if (pass == 2) {
         xyrange = 0.5 * xyrange;                                          //  2nd pass, reduce range and step
         xystep = 0.5 * xystep;                                            //  v.11.04
         trange = 0.5 * trange;
         tstep = 0.5 * tstep;
      }
      
      //  search x/y/t range for best match

      xfL = cimOffs[im2].xf - xyrange;
      xfH = cimOffs[im2].xf + xyrange + 0.5 * xystep;
      yfL = cimOffs[im2].yf - xyrange;
      yfH = cimOffs[im2].yf + xyrange + 0.5 * xystep;
      tfL = cimOffs[im2].tf - trange;
      tfH = cimOffs[im2].tf + trange + 0.5 * tstep;

      for (cimOffs[im2].xf = xfL; cimOffs[im2].xf < xfH; cimOffs[im2].xf += xystep)
      for (cimOffs[im2].yf = yfL; cimOffs[im2].yf < yfH; cimOffs[im2].yf += xystep)
      for (cimOffs[im2].tf = tfL; cimOffs[im2].tf < tfH; cimOffs[im2].tf += tstep)
      {
         match = cim_match_images(im1,im2);                                //  get match level

         if (sigdiff(match,matchB,0.00001) > 0) {
            matchB = match;                                                //  save best match
            offsetsB = cimOffs[im2];
         }

         sprintf(SB_text,"align: %d  match: %.5f",cimNsearch++,matchB);    //  update status bar
         zmainloop();
      }
      
      cimOffs[im2] = offsetsB;                                             //  restore best match

      if (cimPanoV) cim_show_Vimages(0,0);                                 //  v.11.04
      else cim_show_images(0,0);
      
      //  warp corners and search for best match

      wrange = cimWarpRange;                                               //  corner warp range and step
      wstep = cimWarpStep;
      if (! wrange) continue;

      if (pass == 2) {                                                     //  2nd pass, 1/4 range and 1/2 step
         wrange = wrange / 4;
         wstep = wstep / 2;
      }
      
      corner1 = 0;                                                         //  process all 4 corners
      cornerN = 3;
      cornerstep = 1;

      if (cimPano) {
         corner1 = 0;                                                      //  left side corners 0, 3
         cornerN = 3;
         cornerstep = 3;
      }
      
      if (cimPanoV) {
         corner1 = 0;                                                      //  top side corners 0, 1   v.11.04
         cornerN = 1;
         cornerstep = 1;
      }

      matchB = cim_match_images(im1,im2);                                  //  initial image match level
      
      for (ii = corner1; ii <= cornerN; ii += cornerstep)                  //  modify one corner at a time
      {
         wxL = cimOffs[im2].wx[ii] - wrange;
         wxH = cimOffs[im2].wx[ii] + wrange + 0.5 * wstep;
         wyL = cimOffs[im2].wy[ii] - wrange;
         wyH = cimOffs[im2].wy[ii] + wrange + 0.5 * wstep;

         for (cimOffs[im2].wx[ii] = wxL; cimOffs[im2].wx[ii] < wxH; cimOffs[im2].wx[ii] += wstep)
         for (cimOffs[im2].wy[ii] = wyL; cimOffs[im2].wy[ii] < wyH; cimOffs[im2].wy[ii] += wstep)
         {
            match = cim_match_images(im1,im2);                             //  get match level

            if (sigdiff(match,matchB,0.00001) > 0) {
               matchB = match;                                             //  save best match
               offsetsB = cimOffs[im2];
            }

            sprintf(SB_text,"warp: %d  match: %.5f",cimNsearch++,matchB);
            zmainloop();
         }

         cimOffs[im2] = offsetsB;                                          //  restore best match
      }
      
      if (cimPano) cim_warp_image_pano(im2,1);                             //  apply corner warps     v.11.04
      else if (cimPanoV) cim_warp_image_Vpano(im2,1);
      else  cim_warp_image(im2);

      if (cimPanoV) cim_show_Vimages(0,0);                                 //  v.11.04
      else cim_show_images(0,0);
   }

   return;
}


//  Compare 2 pixels using precalculated brightness ratios
//  1.0 = perfect match   0 = total mismatch (black/white)

inline double cim_match_pixels(uint16 *pix1, uint16 *pix2)                 //  v.10.7
{
   double      red1, green1, blue1, red2, green2, blue2;
   double      reddiff, greendiff, bluediff, match;
   double      ff = 1.0 / 65536.0;

   red1 = pix1[0];
   green1 = pix1[1];
   blue1 = pix1[2];

   red2 = pix2[0];
   green2 = pix2[1];
   blue2 = pix2[2];

   reddiff = ff * fabs(red1-red2);                                         //  0 = perfect match
   greendiff = ff * fabs(green1-green2);                                   //  1 = total mismatch
   bluediff = ff * fabs(blue1-blue2);
   
   match = (1.0 - reddiff) * (1.0 - greendiff) * (1.0 - bluediff);         //  1 = perfect match
   return match;
}


//  Compare two images in overlapping areas.
//  Use the high-contrast pixels from cim_get_redpix()
//  return: 1 = perfect match, 0 = total mismatch (black/white)
//  cimPXMs[im1] is matched to cimPXMs[im2] + virtual warps

double cim_match_images(int im1, int im2)                                  //  v.11.03
{
   uint16      *pix1, vpix2[3];
   int         ww, hh, ww2, hh2;
   int         px1, py1, ii, vstat;
   double      wwi, hhi, ww2i, hh2i, xdisp, ydisp;
   double      wx0, wy0, wx1, wy1, wx2, wy2, wx3, wy3;
   double      dx, dy, px2, py2;
   double      x1, y1, t1, x2, y2, t2;
   double      xoff, yoff, toff, costf, sintf, coeff;
   double      match, cmatch, maxcmatch;
   PXM         *pxm1, *pxm2;
   
   x1 = cimOffs[im1].xf;                                                   //  im1, im2 absolute offsets
   y1 = cimOffs[im1].yf;
   t1 = cimOffs[im1].tf;
   x2 = cimOffs[im2].xf;
   y2 = cimOffs[im2].yf;
   t2 = cimOffs[im2].tf;
   
   xoff = (x2 - x1) * cos(t1) + (y2 - y1) * sin(t1);                       //  offset of im2 relative to im1
   yoff = (y2 - y1) * cos(t1) - (x2 - x1) * sin(t1);
   toff = t2 - t1;

   costf = cos(toff);
   sintf = sin(toff);

   wx0 = cimOffs[im2].wx[0];                                               //  im2 corner warps
   wy0 = cimOffs[im2].wy[0];
   wx1 = cimOffs[im2].wx[1];
   wy1 = cimOffs[im2].wy[1];
   wx2 = cimOffs[im2].wx[2];
   wy2 = cimOffs[im2].wy[2];
   wx3 = cimOffs[im2].wx[3];
   wy3 = cimOffs[im2].wy[3];

   pxm1 = cimPXMs[im1];                                                    //  base image
   pxm2 = cimPXMs[im2];                                                    //  comparison image (virtual warps)

   ww = pxm1->ww;   
   hh = pxm1->hh;
   ww2 = ww / 2;
   hh2 = hh / 2;
   
   wwi = 1.0 / ww;
   hhi = 1.0 / hh;
   ww2i = 1.0 / ww2;
   hh2i = 1.0 / hh2;

   cmatch = 0;
   maxcmatch = 1;
   
   if (cimPano)
   {
      for (py1 = cimOv1ylo; py1 < cimOv1yhi; py1++)                        //  loop overlapping pixels
      for (px1 = cimOv1xlo; px1 < cimOv1xhi; px1++)
      {
         ii = py1 * ww + px1;                                              //  skip low-contrast pixels
         if (! cimRedpix[ii]) continue;

         pix1 = PXMpix(pxm1,px1,py1);                                      //  image1 pixel
         if (! pix1[2]) continue;                                          //  ignore void pixels

         px2 = costf * (px1 - xoff) + sintf * (py1 - yoff);                //  corresponding image2 pixel
         py2 = costf * (py1 - yoff) - sintf * (px1 - xoff);
         
         dx = dy = 0.0;                                                    //  corner warp

         xdisp = (px2 - ww2) * ww2i;                                       //  -1 ... 0 ... +1
         ydisp = (py2 - hh2) * hh2i;                                       //  v.11.04
         
         if (xdisp > 0)                                                    //  right half, no warp
            dx = dy = 0;

         else if (ydisp < 0) {                                             //  use NW corner warp
            dx = wx0 * xdisp * ydisp;
            dy = wy0 * xdisp * ydisp;
         }

         else {                                                            //  use SW corner warp
            dx = wx3 * xdisp * ydisp;
            dy = wy3 * xdisp * ydisp;
         }

         px2 += dx;                                                        //  source pixel location
         py2 += dy;                                                        //    after corner warps
         
         vstat = vpixel(pxm2,px2,py2,vpix2);
         if (! vstat) continue;

         match = cim_match_pixels(pix1,vpix2);                             //  compare brightness adjusted
         cmatch += match;                                                  //  accumulate total match
         maxcmatch += 1.0;
      }
   }

   else if (cimPanoV)
   {
      for (py1 = cimOv1ylo; py1 < cimOv1yhi; py1++)                        //  loop overlapping pixels
      for (px1 = cimOv1xlo; px1 < cimOv1xhi; px1++)
      {
         ii = py1 * ww + px1;                                              //  skip low-contrast pixels
         if (! cimRedpix[ii]) continue;

         pix1 = PXMpix(pxm1,px1,py1);                                      //  image1 pixel
         if (! pix1[2]) continue;                                          //  ignore void pixels

         px2 = costf * (px1 - xoff) + sintf * (py1 - yoff);                //  corresponding image2 pixel
         py2 = costf * (py1 - yoff) - sintf * (px1 - xoff);
         
         dx = dy = 0.0;                                                    //  corner warp

         xdisp = (px2 - ww2) * ww2i;                                       //  -1 ... 0 ... +1
         ydisp = (py2 - hh2) * hh2i;
         
         if (ydisp > 0)                                                    //  bottom half, no warp
            dx = dy = 0;

         else if (xdisp < 0) {                                             //  use NW corner warp
            dx = wx0 * xdisp * ydisp;
            dy = wy0 * xdisp * ydisp;
         }

         else {                                                            //  use NE corner warp
            dx = wx1 * xdisp * ydisp;
            dy = wy1 * xdisp * ydisp;
         }

         px2 += dx;                                                        //  source pixel location
         py2 += dy;                                                        //    after corner warps
         
         vstat = vpixel(pxm2,px2,py2,vpix2);
         if (! vstat) continue;

         match = cim_match_pixels(pix1,vpix2);                             //  compare brightness adjusted
         cmatch += match;                                                  //  accumulate total match
         maxcmatch += 1.0;
      }
   }

   else
   {
      for (py1 = cimOv1ylo; py1 < cimOv1yhi; py1++)                        //  loop overlapping pixels
      for (px1 = cimOv1xlo; px1 < cimOv1xhi; px1++)
      {
         ii = py1 * ww + px1;                                              //  skip low-contrast pixels
         if (! cimRedpix[ii]) continue;

         pix1 = PXMpix(pxm1,px1,py1);                                      //  image1 pixel
         if (! pix1[2]) continue;                                          //  ignore void pixels

         px2 = costf * (px1 - xoff) + sintf * (py1 - yoff);                //  corresponding image2 pixel
         py2 = costf * (py1 - yoff) - sintf * (px1 - xoff);
         
         dx = dy = 0.0;                                                    //  corner warp

         coeff = (1.0 - py2 * hhi - px2 * wwi);                            //  corner 0  NW
         if (coeff > 0) {
            dx += coeff * wx0;
            dy += coeff * wy0;
         }
         coeff = (1.0 - py2 * hhi - (ww - px2) * wwi);                     //  corner 1  NE
         if (coeff > 0) {
            dx += coeff * wx1;
            dy += coeff * wy1;
         }
         coeff = (1.0 - (hh - py2) * hhi - (ww - px2) * wwi);              //  corner 2  SE
         if (coeff > 0) {
            dx += coeff * wx2;
            dy += coeff * wy2;
         }
         coeff = (1.0 - (hh - py2) * hhi - px2 * wwi);                     //  corner 3  SW
         if (coeff > 0) {
            dx += coeff * wx3;
            dy += coeff * wy3;
         }

         px2 += dx;                                                        //  source pixel location
         py2 += dy;                                                        //    after corner warps
         
         vstat = vpixel(pxm2,px2,py2,vpix2);
         if (! vstat) continue;

         match = cim_match_pixels(pix1,vpix2);                             //  compare brightness adjusted
         cmatch += match;                                                  //  accumulate total match
         maxcmatch += 1.0;
      }
   }
   
   return cmatch / maxcmatch;
}


//  combine and show all images
//  fnew >> make new E3 output image and adjust x and y offsets
//  cimPXMw[*] >> E3pxm16 >> main window
//  fblend:  0 > 50/50 blend,  1 > gradual blend

namespace  cim_show_images_names {
   int         im1, im2, iminc, fblendd;
   int         wwlo[10], wwhi[10];
   int         hhlo[10], hhhi[10];
   double      costf[10], sintf[10];
}

void cim_show_images(int fnew, int fblend)                                 //  v.10.7
{
   using namespace cim_show_images_names;

   void * cim_show_images_wthread(void *arg);

   int         imx, pxr, pyr, ii, px3, py3;
   int         ww, hh, wwmin, wwmax, hhmin, hhmax, bmid;
   double      xf, yf, tf;
   uint16      *pix3;
   
   mutex_lock(&Fpixmap_lock);                                              //  stop window updates
   
   fblendd = fblend;                                                       //  blend 50/50 or gradual ramp

   im1 = cimShowIm1;                                                       //  two images to show
   im2 = cimShowIm2;
   iminc = im2 - im1;                                                      //  v.10.9
   
   if (cimShowAll) {                                                       //  show all images     v.10.9
      im1 = 0;
      im2 = cimNF-1;
      iminc = 1;
   }

   for (imx = 0; imx < cimNF; imx++) {                                     //  pre-calculate
      costf[imx] = cos(cimOffs[imx].tf);
      sintf[imx] = sin(cimOffs[imx].tf);
   }

   if (fnew) PXM_free(E3pxm16);                                            //  force new output pixmap
   
   if (! E3pxm16)                                                          //  allocate output pixmap
   {
      wwmin = hhmin = 9999;                                                //  initial values
      wwmax = cimPXMw[im2]->ww;
      hhmax = cimPXMw[im2]->hh;
      
      for (imx = im1; imx <= im2; imx += iminc)                            //  find min and max ww and hh extents
      {
         xf = cimOffs[imx].xf;
         yf = cimOffs[imx].yf;
         tf = cimOffs[imx].tf;
         ww = cimPXMw[imx]->ww;
         hh = cimPXMw[imx]->hh;
         if (xf < wwmin) wwmin = xf;
         if (xf - tf * hh < wwmin) wwmin = xf + tf * hh;
         if (xf + ww > wwmax) wwmax = xf + ww;
         if (xf + ww - tf * hh > wwmax) wwmax = xf + ww - tf * hh;
         if (yf < hhmin) hhmin = yf;
         if (yf + tf * ww < hhmin) hhmin = yf + tf * ww;
         if (yf + hh > hhmax) hhmax = yf + hh;
         if (yf + hh + tf * ww > hhmax) hhmax = yf + hh + tf * ww;
      }

      for (imx = im1; imx <= im2; imx += iminc) {                          //  align to top and left edges
         cimOffs[imx].xf -= wwmin;
         cimOffs[imx].yf -= hhmin;
      }
      wwmax = wwmax - wwmin;
      hhmax = hhmax - hhmin;
      wwmin = hhmin = 0;

      if (cimPano) {
         for (imx = im1; imx <= im2; imx += iminc)                         //  deliberate margins      v.11.03
            cimOffs[imx].yf += 10;
         hhmax += 20;
      }

      if (cimPanoV) {
         for (imx = im1; imx <= im2; imx += iminc)                         //  deliberate margins      v.11.04
            cimOffs[imx].xf += 10;
         wwmax += 20;
      }

      E3pxm16 = PXM_make(wwmax,hhmax,16);                                  //  allocate output image
      E3ww = wwmax;
      E3hh = hhmax;
   }

   for (imx = im1; imx <= im2; imx += iminc)                               //  get ww range of each image
   {
      ww = cimPXMw[imx]->ww;
      hh = cimPXMw[imx]->hh;
      tf = cimOffs[imx].tf;
      wwlo[imx] = cimOffs[imx].xf;
      wwhi[imx] = wwlo[imx] + ww;
      wwlo[imx] -= 0.5 * tf * hh;                                          //  use midpoint of sloping edges
      wwhi[imx] -= 0.5 * tf * hh;
   }

   if (cimBlend) {                                                         //  blend width active
      for (imx = im1; imx <= im2-1; imx += iminc)                          //  reduce for blend width
      {
         if (wwhi[imx] - wwlo[imx+1] > cimBlend) {
            bmid = (wwhi[imx] + wwlo[imx+1]) / 2;
            wwlo[imx+1] = bmid - cimBlend / 2;
            wwhi[imx] = bmid + cimBlend / 2;
         }
      }
   }

   for (ii = 0; ii < Nwt; ii++)                                            //  start worker threads   v.10.7
      start_wt(cim_show_images_wthread,&wtnx[ii]);
   wait_wts();                                                             //  wait for completion

   if (cimRedpix) 
   {
      imx = cimRedImage;                                                   //  paint red pixels for current image
      ww = cimPXMw[imx]->ww;                                               //    being aligned
      hh = cimPXMw[imx]->hh;

      for (ii = 0; ii < ww * hh; ii++)
      {
         if (cimRedpix[ii]) {
            pyr = ii / ww;                                                 //  red pixel
            pxr = ii - pyr * ww;
            px3 = cimOffs[imx].xf + pxr * costf[imx] - pyr * sintf[imx] + 0.5;
            py3 = cimOffs[imx].yf + pyr * costf[imx] + pxr * sintf[imx] + 0.5;
            pix3 = PXMpix(E3pxm16,px3,py3);
            pix3[0] = 65535; pix3[1] = pix3[2] = 1;
         }
      }
   }

   mutex_unlock(&Fpixmap_lock);
   mwpaint2();                                                             //  update window
   zmainloop();

   return;
}


void * cim_show_images_wthread(void *arg)                                  //  working thread   v.10.7
{
   using namespace cim_show_images_names;

   int         index = *((int *) (arg));
   int         imx, imy;
   int         px3, py3;
   int         vstat, vstat1, vstat2;
   int         red1, green1, blue1;
   int         red2, green2, blue2;
   int         red3, green3, blue3;
   double      f1, f2, px, py;
   uint16      vpix[3], *pix3;
   
   red1 = green1 = blue1 = 0;

   f1 = f2 = 0.5;                                                          //  to use if no fblend flag

   for (py3 = index; py3 < E3hh; py3 += Nwt)                               //  loop E3 rows
   for (px3 = 0; px3 < E3ww; px3++)                                        //  loop E3 columns
   {
      vstat1 = vstat2 = 0;
      
      for (imx = imy = im1; imx <= im2; imx += iminc)                      //  find which images overlap this pixel
      {
         if (px3 < wwlo[imx] || px3 > wwhi[imx]) continue;
         px = costf[imx] * (px3 - cimOffs[imx].xf) + sintf[imx] * (py3 - cimOffs[imx].yf);
         py = costf[imx] * (py3 - cimOffs[imx].yf) - sintf[imx] * (px3 - cimOffs[imx].xf);
         vstat = vpixel(cimPXMw[imx],px,py,vpix);
         if (! vstat) continue;

         if (! vstat1) {                                                   //  first overlapping image
            vstat1 = 1;
            imy = imx;
            red1 = vpix[0];
            green1 = vpix[1];
            blue1 = vpix[2];
         }
         else {                                                            //  second image
            vstat2 = 1;
            red2 = vpix[0];
            green2 = vpix[1];
            blue2 = vpix[2];
            break;
         }
      }
      
      imx = imy;                                                           //  first of 1 or 2 overlapping images

      if (vstat1) {
         if (! vstat2) {
            red3 = red1;                                                   //  use image1 pixel
            green3 = green1;
            blue3 = blue1; 
         }
         else {                                                            //  use blended image1 + image2 pixels
            if (fblendd) {
               f1 = wwhi[imx] - px3;                                       //  gradual blend
               f2 = px3 - wwlo[imx+1];
               f1 = f1 / (f1 + f2);
               f2 = 1.0 - f1;
            }
            red3 = f1 * red1 + f2 * red2 + 0.5;
            green3 = f1 * green1 + f2 * green2 + 0.5;
            blue3 = f1 * blue1 + f2 * blue2 + 0.5;
         }
      }

      else red3 = green3 = blue3 = 0;                                      //  no overlapping image, use black pixel

      pix3 = PXMpix(E3pxm16,px3,py3);                                      //  output pixel
      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
   }

   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


//  version for vertical panorama

void cim_show_Vimages(int fnew, int fblend)                                //  v.11.04
{
   using namespace cim_show_images_names;

   void * cim_show_Vimages_wthread(void *arg);

   int         imx, pxr, pyr, ii, px3, py3;
   int         ww, hh, wwmin, wwmax, hhmin, hhmax, bmid;
   double      xf, yf, tf;
   uint16      *pix3;
   
   mutex_lock(&Fpixmap_lock);                                              //  stop window updates
   
   fblendd = fblend;                                                       //  blend 50/50 or gradual ramp

   im1 = 0;                                                                //  show all images (pano)
   im2 = cimNF-1;

   for (imx = 0; imx < cimNF; imx++) {                                     //  pre-calculate
      costf[imx] = cos(cimOffs[imx].tf);
      sintf[imx] = sin(cimOffs[imx].tf);
   }

   if (fnew) PXM_free(E3pxm16);                                            //  force new output pixmap
   
   if (! E3pxm16)                                                          //  allocate output pixmap
   {
      wwmin = hhmin = 9999;
      wwmax = hhmax = 0;
      
      for (imx = im1; imx <= im2; imx++)                                   //  find min and max ww and hh extents
      {
         xf = cimOffs[imx].xf;
         yf = cimOffs[imx].yf;
         tf = cimOffs[imx].tf;
         ww = cimPXMw[imx]->ww;
         hh = cimPXMw[imx]->hh;
         if (xf < wwmin) wwmin = xf;
         if (xf - tf * hh < wwmin) wwmin = xf + tf * hh;
         if (xf + ww > wwmax) wwmax = xf + ww;
         if (xf + ww - tf * hh > wwmax) wwmax = xf + ww - tf * hh;
         if (yf < hhmin) hhmin = yf;
         if (yf + tf * ww < hhmin) hhmin = yf + tf * ww;
         if (yf + hh > hhmax) hhmax = yf + hh;
         if (yf + hh + tf * ww > hhmax) hhmax = yf + hh + tf * ww;
      }

      for (imx = im1; imx <= im2; imx++) {                                 //  align to top and left edges
         cimOffs[imx].xf -= wwmin;
         cimOffs[imx].yf -= hhmin;
      }
      wwmax = wwmax - wwmin;
      hhmax = hhmax - hhmin;
      wwmin = hhmin = 0;

      for (imx = im1; imx <= im2; imx++)                                   //  deliberate margins
         cimOffs[imx].xf += 10;
      wwmax += 20;

      E3pxm16 = PXM_make(wwmax,hhmax,16);                                  //  allocate output image
      E3ww = wwmax;
      E3hh = hhmax;
   }

   for (imx = im1; imx <= im2; imx++)                                      //  get hh range of each image
   {
      ww = cimPXMw[imx]->ww;
      hh = cimPXMw[imx]->hh;
      tf = cimOffs[imx].tf;
      hhlo[imx] = cimOffs[imx].yf;
      hhhi[imx] = hhlo[imx] + hh;
      hhlo[imx] += 0.5 * tf * ww;                                          //  use midpoint of sloping edges
      hhhi[imx] += 0.5 * tf * ww;
   }

   if (cimBlend) {                                                         //  blend width active
      for (imx = im1; imx <= im2-1; imx++)                                 //  reduce for blend width
      {
         if (hhhi[imx] - hhlo[imx+1] > cimBlend) {
            bmid = (hhhi[imx] + hhlo[imx+1]) / 2;
            hhlo[imx+1] = bmid - cimBlend / 2;
            hhhi[imx] = bmid + cimBlend / 2;
         }
      }
   }

   for (ii = 0; ii < Nwt; ii++)                                            //  start worker threads   v.10.7
      start_wt(cim_show_Vimages_wthread,&wtnx[ii]);
   wait_wts();                                                             //  wait for completion

   if (cimRedpix) 
   {
      imx = cimRedImage;                                                   //  paint red pixels for current image
      ww = cimPXMw[imx]->ww;                                               //    being aligned
      hh = cimPXMw[imx]->hh;

      for (ii = 0; ii < ww * hh; ii++)
      {
         if (cimRedpix[ii]) {
            pyr = ii / ww;                                                 //  red pixel
            pxr = ii - pyr * ww;
            px3 = cimOffs[imx].xf + pxr * costf[imx] - pyr * sintf[imx] + 0.5;
            py3 = cimOffs[imx].yf + pyr * costf[imx] + pxr * sintf[imx] + 0.5;
            pix3 = PXMpix(E3pxm16,px3,py3);
            pix3[0] = 65535; pix3[1] = pix3[2] = 1;
         }
      }
   }

   mutex_unlock(&Fpixmap_lock);
   mwpaint2();                                                             //  update window
   zmainloop();

   return;
}


void * cim_show_Vimages_wthread(void *arg)                                 //  working thread   v.11.04
{
   using namespace cim_show_images_names;

   int         index = *((int *) (arg));
   int         imx, imy;
   int         px3, py3;
   int         vstat, vstat1, vstat2;
   int         red1, green1, blue1;
   int         red2, green2, blue2;
   int         red3, green3, blue3;
   double      f1, f2, px, py;
   uint16      vpix[3], *pix3;
   
   red1 = green1 = blue1 = 0;

   f1 = f2 = 0.5;                                                          //  to use if no fblend flag

   for (py3 = index; py3 < E3hh; py3 += Nwt)                               //  loop E3 rows
   for (px3 = 0; px3 < E3ww; px3++)                                        //  loop E3 columns
   {
      vstat1 = vstat2 = 0;
      
      for (imx = imy = im1; imx <= im2; imx++)                             //  find which images overlap this pixel
      {
         if (py3 < hhlo[imx] || py3 > hhhi[imx]) continue;
         px = costf[imx] * (px3 - cimOffs[imx].xf) + sintf[imx] * (py3 - cimOffs[imx].yf);
         py = costf[imx] * (py3 - cimOffs[imx].yf) - sintf[imx] * (px3 - cimOffs[imx].xf);
         vstat = vpixel(cimPXMw[imx],px,py,vpix);
         if (! vstat) continue;

         if (! vstat1) {                                                   //  first overlapping image
            vstat1 = 1;
            imy = imx;
            red1 = vpix[0];
            green1 = vpix[1];
            blue1 = vpix[2];
         }
         else {                                                            //  second image
            vstat2 = 1;
            red2 = vpix[0];
            green2 = vpix[1];
            blue2 = vpix[2];
            break;
         }
      }
      
      imx = imy;                                                           //  first of 1 or 2 overlapping images

      if (vstat1) {
         if (! vstat2) {
            red3 = red1;                                                   //  use image1 pixel
            green3 = green1;
            blue3 = blue1; 
         }
         else {                                                            //  use blended image1 + image2 pixels
            if (fblendd) {
               f1 = hhhi[imx] - py3;                                       //  gradual blend
               f2 = py3 - hhlo[imx+1];
               f1 = f1 / (f1 + f2);
               f2 = 1.0 - f1;
            }
            red3 = f1 * red1 + f2 * red2 + 0.5;
            green3 = f1 * green1 + f2 * green2 + 0.5;
            blue3 = f1 * blue1 + f2 * blue2 + 0.5;
         }
      }

      else red3 = green3 = blue3 = 0;                                      //  no overlapping image, use black pixel

      pix3 = PXMpix(E3pxm16,px3,py3);                                      //  output pixel
      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
   }

   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


//  cut-off edges of output image where all input images do not overlap
//  (HDR HDF Stack)

void cim_trim()                                                            //  v.10.9
{
   int      edgex[8] =  {  0, 1,  2, 2,  2, 1,  0, 0 };                    //  4 corners and 4 midpoints of rectangle
   int      edgey[8] =  {  0, 0,  0, 1,  2, 2,  2, 1 };                    //  0 and 2 mark corners, 1 marks midpoints
   int      edgewx[4] = { +1, -1, -1, +1 };
   int      edgewy[4] = { +1, +1, -1, -1 };

   int      imx, ii, jj, ww, hh, px3, py3, px9, py9;
   int      wwmin, wwmax, hhmin, hhmax;
   double   xf, yf, tf, sintf, costf, px, py, wx, wy;
   uint16   *pix3, *pix9;

   wwmin = hhmin = 0;
   wwmax = E3ww;
   hhmax = E3hh;
   
   for (imx = 0; imx < cimNF; imx++)                                       //  loop all images
   {
      ww = cimPXMw[imx]->ww;                                               //  image size
      hh = cimPXMw[imx]->hh;
      xf = cimOffs[imx].xf;                                                //  alignment offsets
      yf = cimOffs[imx].yf;
      tf = cimOffs[imx].tf;
      sintf = sin(tf);
      costf = cos(tf);
      
      for (ii = 0; ii < 8; ii++)                                           //  8 points around image rectangle
      {
         px = ww * edgex[ii] / 2;                                          //  coordinates before warping
         py = hh * edgey[ii] / 2;
         
         if (edgex[ii] != 1 && edgey[ii] != 1) {                           //  if a corner
            jj = ii / 2;
            wx = cimOffs[imx].wx[jj];                                      //  corner warp
            wy = cimOffs[imx].wy[jj];
            if (edgewx[jj] > 0 && wx < 0) px -= wx;                        //  if warp direction inwards,
            if (edgewx[jj] < 0 && wx > 0) px -= wx;                        //    reduce px/py by warp
            if (edgewy[jj] > 0 && wy < 0) py -= wy;
            if (edgewy[jj] < 0 && wy > 0) py -= wy;
         }

         px3 = xf + px * costf - py * sintf;                               //  map px/py to output image px3/py3
         py3 = yf + py * costf + px * sintf;
         
         if (edgex[ii] != 1) {
            if (px3 < ww/2 && px3 > wwmin) wwmin = px3;                    //  remember px3/py3 extremes
            if (px3 > ww/2 && px3 < wwmax) wwmax = px3;
         }

         if (edgey[ii] != 1) {
            if (py3 < hh/2 && py3 > hhmin) hhmin = py3;
            if (py3 > hh/2 && py3 < hhmax) hhmax = py3;
         }
      }
   }
   
   wwmin += 2;                                                             //  compensate rounding
   wwmax -= 2;
   hhmin += 2;
   hhmax -= 2;

   ww = wwmax - wwmin;                                                     //  new image size
   hh = hhmax - hhmin;

   if (ww < 0.7 * E3ww) return;                                            //  sanity check
   if (hh < 0.7 * E3hh) return;
   
   E9pxm16 = PXM_make(ww,hh,16);

   for (py3 = hhmin; py3 < hhmax; py3++)                                   //  E9 = trimmed E3
   for (px3 = wwmin; px3 < wwmax; px3++)
   {
      px9 = px3 - wwmin;
      py9 = py3 - hhmin;
      pix3 = PXMpix(E3pxm16,px3,py3);
      pix9 = PXMpix(E9pxm16,px9,py9);
      pix9[0] = pix3[0];
      pix9[1] = pix3[1];
      pix9[2] = pix3[2];
   }

   PXM_free(E3pxm16);                                                      //  E3 = E9
   E3pxm16 = E9pxm16;
   E9pxm16 = 0;
   E3ww = ww;
   E3hh = hh;

   return;
}


//  dump offsets to stdout - diagnostic tool

void cim_dump_offsets(cchar *text)
{
   printf("\n offsets: %s \n",text);

   for (int imx = 0; imx < cimNF; imx++)
   {
      printf(" imx %d  x/y/t: %.1f %.1f %.4f  w0: %.1f %.1f  w1: %.1f %.1f  w2: %.1f %.1f  w3: %.1f %.1f \n",
          imx, cimOffs[imx].xf, cimOffs[imx].yf, cimOffs[imx].tf,
               cimOffs[imx].wx[0], cimOffs[imx].wy[0], cimOffs[imx].wx[1], cimOffs[imx].wy[1], 
               cimOffs[imx].wx[2], cimOffs[imx].wy[2], cimOffs[imx].wx[3], cimOffs[imx].wy[3]);
   }

   return;
}


/**************************************************************************

   Make an HDR (high dynamic range) image from several images of the same
   subject with different exposure levels. The composite image has better
   visibility of detail in both the brightest and darkest areas.

***************************************************************************/

int      HDRstat;                                                          //  1 = OK, 0 = failed or canceled
double   HDRinitAlignSize = 160;                                           //  initial align image size
double   HDRimageIncrease = 1.6;                                           //  image size increase per align cycle
double   HDRsampSize = 6000;                                               //  pixel sample size   11.03

double   HDRinitSearchRange = 8.0;                                         //  initial search range, +/- pixels
double   HDRinitSearchStep = 1.0;                                          //  initial search step, pixels 
double   HDRinitWarpRange = 3.0;                                           //  initial corner warp range, +/- pixels
double   HDRinitWarpStep = 0.67;                                           //  initial corner warp step, pixels 
double   HDRsearchRange = 2.0;                                             //  normal search range, +/- pixels
double   HDRsearchStep = 0.67;                                             //  normal search step, pixels 
double   HDRwarpRange = 2.0;                                               //  normal corner warp range, +/- pixels
double   HDRwarpStep = 0.67;                                               //  normal corner warp step, pixels

float    *HDRbright = 0;                                                   //  maps brightness per pixel
zdialog  *HDRzd = 0;                                                       //  tweak dialog
double   HDR_respfac[10][1000];                                            //  contribution / image / pixel brightness

void * HDR_align_thread(void *);                                           //  align 2 images
void   HDR_brightness();                                                   //  compute pixel brightness levels
void   HDR_tweak();                                                        //  adjust image contribution curves
void * HDR_combine_thread(void *);                                         //  combine images per contribution curves


//  menu function

void m_HDR(GtkWidget *, cchar *)                                           //  v.10.7
{
   char        **flist, *ftemp;
   int         imx, jj, err, px, py, ww, hh;
   double      diffw, diffh;
   double      fbright[10], btemp;
   double      pixsum, fnorm = 3.0 / 65536.0;
   uint16      *pixel;
   PXM         *pxmtemp;
   
   zfuncs::F1_help_topic = "HDR";                                          //  help topic

   if (mod_keep()) return;                                                 //  warn unsaved changes

   for (imx = 0; imx < 10; imx++)
   {                                                                       //  clear all file and PXM data
      cimFile[imx] = 0;
      cimPXMf[imx] = cimPXMs[imx] = cimPXMw[imx] = 0;
   }

   cimNF = 0;   
   HDRbright = 0;

   flist = zgetfileN(ZTX("Select 2 to 9 files"),"openN",curr_file);        //  select images to combine
   if (! flist) return;

   for (imx = 0; flist[imx]; imx++);                                       //  count selected files
   if (imx < 2 || imx > 9) {
      zmessageACK(mWin,ZTX("Select 2 to 9 files"));
      goto cleanup;
   }

   cimNF = imx;                                                            //  file count
   for (imx = 0; imx < cimNF; imx++)
      cimFile[imx] = strdupz(flist[imx],0,"HDR.file");                     //  set up file list
   
   if (! cim_load_files()) goto cleanup;                                   //  load and check all files

   ww = cimPXMf[0]->ww;
   hh = cimPXMf[0]->hh;
   
   for (imx = 1; imx < cimNF; imx++)                                       //  check image compatibility
   {
      diffw = abs(ww - cimPXMf[imx]->ww);
      diffw = diffw / ww;
      diffh = abs(hh - cimPXMf[imx]->hh);
      diffh = diffh / hh;

      if (diffw > 0.02 || diffh > 0.02) {
         zmessageACK(mWin,ZTX("Images are not all the same size"));
         goto cleanup;
      }
   }
   
   free_resources();                                                       //  ready to commit

   err = f_open(cimFile[0],0);                                             //  curr_file = 1st file in list
   if (err) goto cleanup;

   if (! edit_setup("HDR",0,0)) goto cleanup;                              //  setup edit, no preview

   for (imx = 0; imx < cimNF; imx++)                                       //  compute image brightness levels
   {
      pixsum = 0;
      for (py = 0; py < Fhh; py++)
      for (px = 0; px < Fww; px++)
      {
         pixel = PXMpix(cimPXMf[imx],px,py);
         pixsum += fnorm * (pixel[0] + pixel[1] + pixel[2]);
      }
      fbright[imx] = pixsum / (Fww * Fhh);
   }

   for (imx = 0; imx < cimNF; imx++)                                       //  sort file and pixmap lists
   for (jj = imx+1; jj < cimNF; jj++)                                      //    by decreasing brightness
   {
      if (fbright[jj] > fbright[imx]) {                                    //  bubble sort
         btemp = fbright[jj];
         fbright[jj] = fbright[imx];
         fbright[imx] = btemp;
         ftemp = cimFile[jj];
         cimFile[jj] = cimFile[imx];
         cimFile[imx] = ftemp;
         pxmtemp = cimPXMf[jj];
         cimPXMf[jj] = cimPXMf[imx];
         cimPXMf[imx] = pxmtemp;
      }
   }
   
   start_thread(HDR_align_thread,0);                                       //  align each pair of images
   wrapup_thread(0);                                                       //  wait for completion
   if (HDRstat != 1) goto cancel;

   HDR_brightness();                                                       //  compute pixel brightness levels
   if (HDRstat != 1) goto cancel;
   
   HDR_tweak();                                                            //  combine images based on user inputs
   if (HDRstat != 1) goto cancel;

   Fmodified = 1;                                                          //  done
   edit_done();
   goto cleanup;

cancel:
   edit_cancel();

cleanup:

   if (flist) {
      for (imx = 0; flist[imx]; imx++)                                     //  free file list
         zfree(flist[imx]);
      zfree(flist);
   }

   for (imx = 0; imx < cimNF; imx++) {                                     //  free cim file and PXM data
      if (cimFile[imx]) zfree(cimFile[imx]);
      if (cimPXMf[imx]) PXM_free(cimPXMf[imx]);
      if (cimPXMs[imx]) PXM_free(cimPXMs[imx]);
      if (cimPXMw[imx]) PXM_free(cimPXMw[imx]);
   }
   
   if (HDRbright) zfree(HDRbright);
   *SB_text = 0;

   return;
}


//  HDR align each pair of input images, output combined image to E3pxm16
//  cimPXMf[*]  original image
//  cimPXMs[*]  scaled and color adjusted for pixel comparisons
//  cimPXMw[*]  warped for display

void * HDR_align_thread(void *)                                            //  v.10.7
{
   int         imx, im1, im2, ww, hh, ii, nn;
   double      R, maxtf, mintf, midtf;
   double      xoff, yoff, toff, dxoff, dyoff;
   cimoffs     offsets[10];                                                //  x/y/t offsets after alignment
   
   Fzoom = 0;                                                              //  fit to window if big
   Fblowup = 1;                                                            //  scale up to window if small
   Ffuncbusy++;                                                            //  v.11.01
   cimShrink = 0;                                                          //  no warp shrinkage (pano)
   cimPano = cimPanoV = 0;                                                 //  no pano mode

   for (imx = 0; imx < cimNF; imx++)                                       //  bugfix     v.10.8
      memset(&offsets[imx],0,sizeof(cimoffs));

   for (im1 = 0; im1 < cimNF-1; im1++)                                     //  loop each pair of images
   {
      im2 = im1 + 1;

      memset(&cimOffs[im1],0,sizeof(cimoffs));                             //  initial image offsets = 0
      memset(&cimOffs[im2],0,sizeof(cimoffs));

      ww = cimPXMf[im1]->ww;                                               //  image dimensions
      hh = cimPXMf[im1]->hh;

      nn = ww;                                                             //  use larger of ww, hh
      if (hh > ww) nn = hh;
      cimScale = HDRinitAlignSize / nn;                                    //  initial align image size
      if (cimScale > 1.0) cimScale = 1.0;

      cimBlend = 0;                                                        //  no blend width (use all)
      cim_get_overlap(im1,im2,cimPXMf);                                    //  get overlap area
      cim_match_colors(im1,im2,cimPXMf);                                   //  get color matching factors

      cimSearchRange = HDRinitSearchRange;                                 //  initial align search range
      cimSearchStep = HDRinitSearchStep;                                   //  initial align search step
      cimWarpRange = HDRinitWarpRange;                                     //  initial align corner warp range
      cimWarpStep = HDRinitWarpStep;                                       //  initial align corner warp step
      cimSampSize = HDRsampSize;                                           //  pixel sample size for align/compare
      cimNsearch = 0;                                                      //  reset align search counter

      while (true)                                                         //  loop, increasing image size
      {
         cim_scale_image(im1,cimPXMs);                                     //  scale images to cimScale
         cim_scale_image(im2,cimPXMs);

         cim_adjust_colors(cimPXMs[im1],1);                                //  apply color adjustments
         cim_adjust_colors(cimPXMs[im2],2);

         cim_warp_image(im1);                                              //  make warped images to show
         cim_warp_image(im2);      

         cimShowIm1 = im1;                                                 //  show two images with 50/50 blend
         cimShowIm2 = im2;
         cimShowAll = 0;
         cim_show_images(1,0);                                             //  (x/y offsets can change)

         cim_get_overlap(im1,im2,cimPXMs);                                 //  get overlap area             v.11.04
         cim_get_redpix(im1);                                              //  get high-contrast pixels

         cim_align_image(im1,im2);                                         //  align im2 to im1

         zfree(cimRedpix);                                                 //  clear red pixels
         cimRedpix = 0;

         if (cimScale == 1.0) break;                                       //  done

         R = HDRimageIncrease;                                             //  next larger image size
         cimScale = cimScale * R;
         if (cimScale > 0.85) {                                            //  if close to end, jump to end
            R = R / cimScale;
            cimScale = 1.0;
         }

         cimOffs[im1].xf *= R;                                             //  scale offsets for larger image
         cimOffs[im1].yf *= R;
         cimOffs[im2].xf *= R;
         cimOffs[im2].yf *= R;

         for (ii = 0; ii < 4; ii++) {
            cimOffs[im1].wx[ii] *= R;
            cimOffs[im1].wy[ii] *= R;
            cimOffs[im2].wx[ii] *= R;
            cimOffs[im2].wy[ii] *= R;
         }

         cimSearchRange = HDRsearchRange;                                  //  align search range
         cimSearchStep = HDRsearchStep;                                    //  align search step size
         cimWarpRange = HDRwarpRange;                                      //  align corner warp range
         cimWarpStep = HDRwarpStep;                                        //  align corner warp step size
      }

      offsets[im2].xf = cimOffs[im2].xf - cimOffs[im1].xf;                 //  save im2 offsets from im1
      offsets[im2].yf = cimOffs[im2].yf - cimOffs[im1].yf;
      offsets[im2].tf = cimOffs[im2].tf - cimOffs[im1].tf;
      
      for (ii = 0; ii < 4; ii++) {
         offsets[im2].wx[ii] = cimOffs[im2].wx[ii] - cimOffs[im1].wx[ii];
         offsets[im2].wy[ii] = cimOffs[im2].wy[ii] - cimOffs[im1].wy[ii];
      }
   }
   
   for (imx = 0; imx < cimNF; imx++)                                       //  offsets[*] >> cimOffs[*]
      cimOffs[imx] = offsets[imx];

   cimOffs[0].xf = cimOffs[0].yf = cimOffs[0].tf = 0;                      //  image 0 at (0,0,0)

   for (im1 = 0; im1 < cimNF-1; im1++)                                     //  absolute offsets for image 1 to last
   {
      im2 = im1 + 1;
      cimOffs[im2].xf += cimOffs[im1].xf;                                  //  x/y/t offsets are additive
      cimOffs[im2].yf += cimOffs[im1].yf;
      cimOffs[im2].tf += cimOffs[im1].tf;

      for (ii = 0; ii < 4; ii++) {                                         //  corner warps are additive
         cimOffs[im2].wx[ii] += cimOffs[im1].wx[ii];
         cimOffs[im2].wy[ii] += cimOffs[im1].wy[ii];
      }
   }

   for (imx = 1; imx < cimNF; imx++)                                       //  re-warp to absolute       v.10.8
      cim_warp_image(imx);

   toff = cimOffs[0].tf;                                                   //  balance +/- thetas
   maxtf = mintf = toff;
   for (imx = 1; imx < cimNF; imx++) {
      toff = cimOffs[imx].tf;
      if (toff > maxtf) maxtf = toff;
      if (toff < mintf) mintf = toff;
   }
   midtf = 0.5 * (maxtf + mintf);
   
   for (imx = 0; imx < cimNF; imx++)
      cimOffs[imx].tf -= midtf;

   for (im1 = 0; im1 < cimNF-1; im1++)                                     //  adjust x/y offsets for images after im1
   for (im2 = im1+1; im2 < cimNF; im2++)                                   //    due to im1 theta offset
   {
      toff = cimOffs[im1].tf;
      xoff = cimOffs[im2].xf - cimOffs[im1].xf;
      yoff = cimOffs[im2].yf - cimOffs[im1].yf;
      dxoff = yoff * sin(toff);
      dyoff = xoff * sin(toff);
      cimOffs[im2].xf -= dxoff;
      cimOffs[im2].yf += dyoff;
   }

   Fzoom = Fblowup = 0;
   Ffuncbusy--;
   HDRstat = 1;
   thread_exit();
   return 0;                                                               //  not executed
}


//  Compute mean image pixel brightness levels.
//  (basis for setting image contributions per brightness level)

void HDR_brightness()                                                      //  v.10.7
{
   int         px3, py3, ww, hh, imx, kk, vstat;
   double      px, py, red, green, blue;
   double      bright, maxbright, minbright;
   double      xoff, yoff, sintf[10], costf[10];
   double      norm, fnorm = 1.0 / 65536.0;
   uint16      vpix[3], *pix3;

   cimScale = 1.0;

   for (imx = 0; imx < cimNF; imx++)                                       //  replace alignment images
   {                                                                       //   (color adjusted for pixel matching)
      PXM_free(cimPXMs[imx]);                                              //    with the original images
      cimPXMs[imx] = PXM_copy(cimPXMf[imx]);
      cim_warp_image(imx);                                                 //  re-apply warps
   }

   for (imx = 0; imx < cimNF; imx++)                                       //  pre-calculate trig functions
   {
      sintf[imx] = sin(cimOffs[imx].tf);
      costf[imx] = cos(cimOffs[imx].tf);
   }
   
   ww = E3pxm16->ww;
   hh = E3pxm16->hh;

   HDRbright = (float *) zmalloc(ww*hh*sizeof(int),"hdr.brightmem");       //  get memory for brightness array
   
   minbright = 1.0;
   maxbright = 0.0;
   
   for (py3 = 0; py3 < hh; py3++)                                          //  step through all output pixels
   for (px3 = 0; px3 < ww; px3++)
   {
      red = green = blue = 0;
      vstat = 0;

      for (imx = 0; imx < cimNF; imx++)                                    //  step through all input images
      {
         xoff = cimOffs[imx].xf;
         yoff = cimOffs[imx].yf;

         px = costf[imx] * (px3 - xoff) + sintf[imx] * (py3 - yoff);       //  image N pixel, after offsets
         py = costf[imx] * (py3 - yoff) - sintf[imx] * (px3 - xoff);
         vstat = vpixel(cimPXMw[imx],px,py,vpix);
         if (! vstat) break;
         
         red += fnorm * vpix[0];                                           //  sum input pixels
         green += fnorm * vpix[1];
         blue += fnorm * vpix[2];
      }
      
      if (! vstat) {                                                       //  pixel outside some image
         pix3 = PXMpix(E3pxm16,px3,py3);                                   //  output pixel = black
         pix3[0] = pix3[1] = pix3[2] = 0;
         kk = py3 * ww + px3;
         HDRbright[kk] = 0;
         continue;
      }
      
      bright = (red + green + blue) / (3 * cimNF);                         //  mean pixel brightness, 0.0 to 1.0
      kk = py3 * ww + px3;
      HDRbright[kk] = bright;
      
      if (bright > maxbright) maxbright = bright;
      if (bright < minbright) minbright = bright;

      pix3 = PXMpix(E3pxm16,px3,py3);                                      //  output pixel
      pix3[0] = red * 65535.0 / cimNF;
      pix3[1] = green * 65535.0 / cimNF;
      pix3[2] = blue * 65535.0 / cimNF;
   }
   
   norm = 0.999 / (maxbright - minbright);                                 //  normalize to range 0.0 to 0.999   

   for (int ii = 0; ii < ww * hh; ii++)
      HDRbright[ii] = (HDRbright[ii] - minbright) * norm;

   mwpaint2();                                                             //  update window
   return;
}


//  Dialog for user to control the contributions of each input image
//  while watching the output image which is updated in real time.

void HDR_tweak()                                                           //  v.10.7
{
   int    HDR_tweak_event(zdialog *zd, cchar *event);
   void   HDR_curvedit(int);

   int         imx;
   double      cww = 1.0 / (cimNF-1);
   
   HDRzd = zdialog_new(ZTX("Adjust Image Contributions"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(HDRzd,"frame","brframe","dialog",0,"expand|space=2");
   zdialog_add_widget(HDRzd,"hbox","hb1","dialog",0);
   zdialog_add_widget(HDRzd,"label","lab11","hb1",ZTX("dark pixels"),"space=3");
   zdialog_add_widget(HDRzd,"label","lab12","hb1",0,"expand");
   zdialog_add_widget(HDRzd,"label","lab13","hb1",ZTX("light pixels"),"space=3");
   zdialog_add_widget(HDRzd,"hbox","hb2","dialog",0,"space=3");
   zdialog_add_widget(HDRzd,"label","labf1","hb2",ZTX("file:"),"space=3");
   zdialog_add_widget(HDRzd,"label","labf2","hb2","*");

   zdialog_add_widget(HDRzd,"hbox","hbcf","dialog",0,"space=5");
   zdialog_add_widget(HDRzd,"label","labcf","hbcf",ZTX("Curve File:"),"space=5");
   zdialog_add_widget(HDRzd,"button","load","hbcf",Bopen,"space=5");
   zdialog_add_widget(HDRzd,"button","save","hbcf",Bsave,"space=5");

   GtkWidget *brframe = zdialog_widget(HDRzd,"brframe");                   //  set up curve edit
   spcdat *sd = spccurve_init(brframe,HDR_curvedit);                       //  v.11.01
   Espcdat = sd;

   sd->Nspc = cimNF;                                                       //  no. curves = no. files
   
   for (imx = 0; imx < cimNF; imx++)                                       //  set up initial response curve
   {                                                                       //    anchor points
      sd->vert[imx] = 0;
      sd->nap[imx] = 2;
      sd->apx[imx][0] = 0.01;                                              //  flatter curves, v.9.3
      sd->apx[imx][1] = 0.99;
      sd->apy[imx][0] = 0.9 - imx * 0.8 * cww;
      sd->apy[imx][1] = 0.1 + imx * 0.8 * cww;
      spccurve_generate(sd,imx);
   }
   
   start_thread(HDR_combine_thread,0);                                     //  start working thread
   signal_thread();

   zdialog_resize(HDRzd,400,360);                                          //  run dialog
   zdialog_run(HDRzd,HDR_tweak_event);
   zdialog_wait(HDRzd);                                                    //  wait for completion

   return;
}


//  dialog event and completion callback function

int HDR_tweak_event(zdialog *zd, cchar *event)
{
   spcdat *sd = Espcdat;

   if (strEqu(event,"load")) {                                             //  load saved curve       v.11.02
      spccurve_load(sd);
      zdialog_stuff(HDRzd,"labf2","*");
      signal_thread();
      return 0;
   }

   if (strEqu(event,"save")) {                                             //  save curve to file     v.11.02
      spccurve_save(sd);
      return 0;
   }

   if (zd->zstat)                                                          //  dialog complete
   {
      wrapup_thread(8);
      if (zd->zstat == 1) HDRstat = 1;
      else HDRstat = 0;
      zfree(Espcdat);                                                      //  free curve edit data      v.11.01
      Espcdat = 0;
      zdialog_free(HDRzd);
      if (HDRstat == 1) cim_trim();                                        //  cut-off edges       v.10.9
   }
   
   return 1;
}


//  this function is called when a curve is edited

void HDR_curvedit(int spc)
{
   cchar  *pp; 
   
   pp = strrchr(cimFile[spc],'/');
   zdialog_stuff(HDRzd,"labf2",pp+1);
   signal_thread();
   return;
}


//  Combine all input images >> E3pxm16 based on image response curves.

void * HDR_combine_thread(void *)
{
   void * HDR_combine_wthread(void *arg);

   int         imx, ii;
   double      xlo, xhi, xval, yval, sumrf;
   spcdat      *sd = Espcdat;

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      for (imx = 0; imx < cimNF; imx++)                                    //  loop input images
      {
         ii = sd->nap[imx];                                                //  get low and high anchor points
         xlo = sd->apx[imx][0];                                            //    for image response curve
         xhi = sd->apx[imx][ii-1];
         if (xlo < 0.02) xlo = 0;                                          //  snap-to scale end points
         if (xhi > 0.98) xhi = 1;

         for (ii = 0; ii < 1000; ii++)                                     //  loop all brightness levels
         {
            HDR_respfac[imx][ii] = 0;
            xval = 0.001 * ii;
            if (xval < xlo || xval > xhi) continue;                        //  no influence for brightness level
            yval = spccurve_yval(sd,imx,xval);                             //  response curve value for brightness
            HDR_respfac[imx][ii] = yval;                                   //    = contribution of this input image
         }
      }

      for (ii = 0; ii < 1000; ii++)                                        //  normalize the factors so that
      {                                                                    //    they sum to 1.0
         sumrf = 0;
         for (imx = 0; imx < cimNF; imx++)
            sumrf += HDR_respfac[imx][ii];
         if (! sumrf) continue;
         for (imx = 0; imx < cimNF; imx++)
            HDR_respfac[imx][ii] = HDR_respfac[imx][ii] / sumrf;
      }

      mutex_lock(&Fpixmap_lock);                                           //  stop window updates

      for (ii = 0; ii < Nwt; ii++)                                         //  start worker threads      v.10.7
         start_wt(HDR_combine_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion
   
      mutex_unlock(&Fpixmap_lock);
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed
}


void * HDR_combine_wthread(void *arg)                                      //  working thread
{
   int         index = *((int *) (arg));
   int         imx, ww, hh, ii, px3, py3, vstat;
   double      sintf[10], costf[10], xoff, yoff;
   double      px, py, red, green, blue, bright, factor;
   uint16      vpix[3], *pix3;

   for (imx = 0; imx < cimNF; imx++)                                       //  pre-calculate trig functions
   {
      sintf[imx] = sin(cimOffs[imx].tf);
      costf[imx] = cos(cimOffs[imx].tf);
   }

   ww = E3pxm16->ww;
   hh = E3pxm16->hh;

   for (py3 = index; py3 < hh; py3 += Nwt)                                 //  step through all output pixels
   for (px3 = 0; px3 < ww; px3++)
   {
      ii = py3 * ww + px3;
      bright = HDRbright[ii];                                              //  mean brightness, 0.0 to 1.0
      ii = 1000 * bright;
      
      red = green = blue = 0;
      
      for (imx = 0; imx < cimNF; imx++)                                    //  loop input images
      {
         factor = HDR_respfac[imx][ii];                                    //  image contribution to this pixel
         if (! factor) continue;                                           //  none
         
         xoff = cimOffs[imx].xf;
         yoff = cimOffs[imx].yf;

         px = costf[imx] * (px3 - xoff) + sintf[imx] * (py3 - yoff);       //  input virtual pixel mapping to
         py = costf[imx] * (py3 - yoff) - sintf[imx] * (px3 - xoff);       //    this output pixel

         vstat = vpixel(cimPXMw[imx],px,py,vpix);                          //  get input pixel
         if (! vstat) continue;

         red += factor * vpix[0];                                          //  accumulate brightness contribution
         green += factor * vpix[1];
         blue += factor * vpix[2];
      }
         
      pix3 = PXMpix(E3pxm16,px3,py3);                                      //  output pixel

      pix3[0] = red;                                                       //  = sum of input pixel contributions
      pix3[1] = green;
      pix3[2] = blue;
   }
   
   exit_wt();
   return 0;                                                               //  not executed
}


/**************************************************************************

   Make an HDF (high depth of field) image from several images of the same
   subject with different focus settings. Combine the images and allow the
   user to "paint" the output composite image using the mouse and choosing 
   the sharpest input image for each area of the output image. The result 
   is an image with a depth of field that exceeds the camera capability.
   
   The images are aligned at the center, but small differences in camera 
   position (hand-held photos) will cause parallax errors that prevent 
   perfect alignment of the images. Also, the images with nearer focus
   will be slightly larger than those with farther focus. These problems
   can be compensated by dragging and warping the images using the mouse.                 v.10.7 

**************************************************************************/

int      HDFstat;                                                          //  1 = OK, 0 = failed or canceled
double   HDFinitAlignSize = 160;                                           //  initial align image size
double   HDFimageIncrease = 1.6;                                           //  image size increase per align cycle
double   HDFsampSize = 6000;                                               //  pixel sample size

double   HDFinitSearchRange = 8.0;                                         //  initial search range, +/- pixels
double   HDFinitSearchStep = 1.0;                                          //  initial search step, pixels
double   HDFinitWarpRange = 4.0;                                           //  initial corner warp range
double   HDFinitWarpStep = 1.0;                                            //  initial corner warp step
double   HDFsearchRange = 2.0;                                             //  normal search range
double   HDFsearchStep = 1.0;                                              //  normal search step
double   HDFwarpRange = 1.0;                                               //  normal corner warp range     v.11.03
double   HDFwarpStep = 0.67;                                               //  normal corner warp step

void * HDF_align_thread(void *);
void   HDF_tweak();
void   HDF_mousefunc();
void * HDF_combine_thread(void *);


//  menu function

void m_HDF(GtkWidget *, cchar *)                                           //  v.10.7
{
   char        **flist;
   int         imx, err, ww, hh;
   double      diffw, diffh;
   
   zfuncs::F1_help_topic = "HDF";                                          //  help topic

   if (mod_keep()) return;                                                 //  warn unsaved changes

   for (imx = 0; imx < 10; imx++)
   {                                                                       //  clear all file and PXM data
      cimFile[imx] = 0;
      cimPXMf[imx] = cimPXMs[imx] = cimPXMw[imx] = 0;
   }

   cimNF = 0;   

   flist = zgetfileN(ZTX("Select 2 to 9 files"),"openN",curr_file);        //  select images to combine
   if (! flist) return;

   for (imx = 0; flist[imx]; imx++);                                       //  count selected files
   if (imx < 2 || imx > 9) {
      zmessageACK(mWin,ZTX("Select 2 to 9 files"));
      goto cleanup;
   }

   cimNF = imx;                                                            //  file count
   for (imx = 0; imx < cimNF; imx++)
      cimFile[imx] = strdupz(flist[imx],0,"HDF.file");                     //  set up file list
   
   if (! cim_load_files()) goto cleanup;                                   //  load and check all files

   ww = cimPXMf[0]->ww;
   hh = cimPXMf[0]->hh;
   
   for (imx = 1; imx < cimNF; imx++)                                       //  check image compatibility
   {
      diffw = abs(ww - cimPXMf[imx]->ww);
      diffw = diffw / ww;
      diffh = abs(hh - cimPXMf[imx]->hh);
      diffh = diffh / hh;

      if (diffw > 0.02 || diffh > 0.02) {
         zmessageACK(mWin,ZTX("Images are not all the same size"));
         goto cleanup;
      }
   }

   free_resources();                                                       //  ready to commit

   err = f_open(cimFile[0],0);                                             //  curr_file = 1st file in list
   if (err) goto cleanup;

   if (! edit_setup("HDF",0,0)) goto cleanup;                              //  setup edit, no preview

   start_thread(HDF_align_thread,0);                                       //  align each pair of images
   wrapup_thread(0);                                                       //  wait for completion
   if (HDFstat != 1) goto cancel;

   HDF_tweak();                                                            //  combine images based on user inputs
   if (HDFstat != 1) goto cancel;

   Fmodified = 1;                                                          //  done
   edit_done();
   goto cleanup;

cancel:
   edit_cancel();

cleanup:

   if (flist) {
      for (imx = 0; flist[imx]; imx++)                                     //  free file list
         zfree(flist[imx]);
      zfree(flist);
   }

   for (imx = 0; imx < cimNF; imx++) {                                     //  free cim file and PXM data
      if (cimFile[imx]) zfree(cimFile[imx]);
      if (cimPXMf[imx]) PXM_free(cimPXMf[imx]);
      if (cimPXMs[imx]) PXM_free(cimPXMs[imx]);
      if (cimPXMw[imx]) PXM_free(cimPXMw[imx]);
   }
   
   *SB_text = 0;
   return;
}


//  HDF align each pair of input images, output combined image to E3pxm16
//  cimPXMf[*]  original image
//  cimPXMs[*]  scaled and color adjusted for pixel comparisons
//  cimPXMw[*]  warped for display

void * HDF_align_thread(void *)                                            //  v.10.7
{
   int         imx, im1, im2, ww, hh, ii, nn;
   double      R, maxtf, mintf, midtf;
   double      xoff, yoff, toff, dxoff, dyoff;
   cimoffs     offsets[10];                                                //  x/y/t offsets after alignment
   
   Fzoom = 0;                                                              //  fit to window if big
   Fblowup = 1;                                                            //  scale up to window if small
   Ffuncbusy++;                                                            //  v.11.01
   cimShrink = 0;                                                          //  no warp shrinkage (pano)
   cimPano = cimPanoV = 0;                                                 //  no pano mode

   for (imx = 0; imx < cimNF; imx++)                                       //  bugfix     v.10.8
      memset(&offsets[imx],0,sizeof(cimoffs));
   
   for (im1 = 0; im1 < cimNF-1; im1++)                                     //  loop each pair of images
   {
      im2 = im1 + 1;

      memset(&cimOffs[im1],0,sizeof(cimoffs));                             //  initial image offsets = 0
      memset(&cimOffs[im2],0,sizeof(cimoffs));
      
      ww = cimPXMf[im1]->ww;                                               //  image dimensions
      hh = cimPXMf[im1]->hh;

      nn = ww;                                                             //  use larger of ww, hh
      if (hh > ww) nn = hh;
      cimScale = HDFinitAlignSize / nn;                                    //  initial align image size
      if (cimScale > 1.0) cimScale = 1.0;

      cimBlend = 0;                                                        //  no blend width (use all)
      cim_get_overlap(im1,im2,cimPXMf);                                    //  get overlap area
      cim_match_colors(im1,im2,cimPXMf);                                   //  get color matching factors

      cimSearchRange = HDFinitSearchRange;                                 //  initial align search range
      cimSearchStep = HDFinitSearchStep;                                   //  initial align search step
      cimWarpRange = HDFinitWarpRange;                                     //  initial align corner warp range
      cimWarpStep = HDFinitWarpStep;                                       //  initial align corner warp step
      cimSampSize = HDFsampSize;                                           //  pixel sample size for align/compare
      cimNsearch = 0;                                                      //  reset align search counter

      while (true)                                                         //  loop, increasing image size
      {
         cim_scale_image(im1,cimPXMs);                                     //  scale images to cimScale
         cim_scale_image(im2,cimPXMs);

         cim_adjust_colors(cimPXMs[im1],1);                                //  apply color adjustments
         cim_adjust_colors(cimPXMs[im2],2);

         cim_warp_image(im1);                                              //  warp images for show
         cim_warp_image(im2);

         cimShowIm1 = im1;                                                 //  show these two images
         cimShowIm2 = im2;                                                 //    with 50/50 blend
         cimShowAll = 0;
         cim_show_images(1,0);                                             //  (y offset can change)

         cim_get_overlap(im1,im2,cimPXMs);                                 //  get overlap area             v.11.04
         cim_get_redpix(im1);                                              //  get high-contrast pixels

         cim_align_image(im1,im2);                                         //  align im2 to im1
         
         zfree(cimRedpix);                                                 //  clear red pixels
         cimRedpix = 0;

         if (cimScale == 1.0) break;                                       //  done

         R = HDFimageIncrease;                                             //  next larger image size
         cimScale = cimScale * R;
         if (cimScale > 0.85) {                                            //  if close to end, jump to end
            R = R / cimScale;
            cimScale = 1.0;
         }

         cimOffs[im1].xf *= R;                                             //  scale offsets for larger image
         cimOffs[im1].yf *= R;
         cimOffs[im2].xf *= R;
         cimOffs[im2].yf *= R;

         for (ii = 0; ii < 4; ii++) {
            cimOffs[im1].wx[ii] *= R;
            cimOffs[im1].wy[ii] *= R;
            cimOffs[im2].wx[ii] *= R;
            cimOffs[im2].wy[ii] *= R;
         }

         cimSearchRange = HDFsearchRange;                                  //  align search range
         cimSearchStep = HDFsearchStep;                                    //  align search step size
         cimWarpRange = HDFwarpRange;                                      //  align corner warp range
         cimWarpStep = HDFwarpStep;                                        //  align corner warp step size
      }
      
      offsets[im2].xf = cimOffs[im2].xf - cimOffs[im1].xf;                 //  save im2 offsets from im1
      offsets[im2].yf = cimOffs[im2].yf - cimOffs[im1].yf;
      offsets[im2].tf = cimOffs[im2].tf - cimOffs[im1].tf;

      for (ii = 0; ii < 4; ii++) {
         offsets[im2].wx[ii] = cimOffs[im2].wx[ii] - cimOffs[im1].wx[ii];
         offsets[im2].wy[ii] = cimOffs[im2].wy[ii] - cimOffs[im1].wy[ii];
      }
   }

   for (imx = 0; imx < cimNF; imx++)                                       //  offsets[*] >> cimOffs[*]
      cimOffs[imx] = offsets[imx];
   
   cimOffs[0].xf = cimOffs[0].yf = cimOffs[0].tf = 0;                      //  image 0 at (0,0,0)

   for (im1 = 0; im1 < cimNF-1; im1++)                                     //  absolute offsets for image 1 to last
   {
      im2 = im1 + 1;
      cimOffs[im2].xf += cimOffs[im1].xf;                                  //  x/y/t offsets are additive
      cimOffs[im2].yf += cimOffs[im1].yf;
      cimOffs[im2].tf += cimOffs[im1].tf;

      for (ii = 0; ii < 4; ii++) {                                         //  corner warps are additive
         cimOffs[im2].wx[ii] += cimOffs[im1].wx[ii];
         cimOffs[im2].wy[ii] += cimOffs[im1].wy[ii];
      }
   }

   for (imx = 1; imx < cimNF; imx++)                                       //  re-warp to absolute       v.10.8
      cim_warp_image(imx);

   toff = cimOffs[0].tf;                                                   //  balance +/- thetas
   maxtf = mintf = toff;
   for (imx = 1; imx < cimNF; imx++) {
      toff = cimOffs[imx].tf;
      if (toff > maxtf) maxtf = toff;
      if (toff < mintf) mintf = toff;
   }
   midtf = 0.5 * (maxtf + mintf);
   
   for (imx = 0; imx < cimNF; imx++)
      cimOffs[imx].tf -= midtf;

   for (im1 = 0; im1 < cimNF-1; im1++)                                     //  adjust x/y offsets for images after im1
   for (im2 = im1+1; im2 < cimNF; im2++)                                   //    due to im1 theta offset
   {
      toff = cimOffs[im1].tf;
      xoff = cimOffs[im2].xf - cimOffs[im1].xf;
      yoff = cimOffs[im2].yf - cimOffs[im1].yf;
      dxoff = yoff * sin(toff);
      dyoff = xoff * sin(toff);
      cimOffs[im2].xf -= dxoff;
      cimOffs[im2].yf += dyoff;
   }
   
   for (imx = 0; imx < cimNF; imx++)                                       //  use final warped images as basis
   {                                                                       //    for manual align adjustments
      PXM_free(cimPXMs[imx]);                                              //  bugfix     v.11.04
      cimPXMs[imx] = PXM_copy(cimPXMw[imx]);
   }

   Fzoom = Fblowup = 0;
   Ffuncbusy--;
   HDFstat = 1;
   thread_exit();
   return 0;                                                               //  not executed
}


//  paint and warp output image

zdialog  *HDFzd = 0;                                                       //  paint dialog
int      HDFmode;                                                          //  mode: paint or warp
int      HDFimage;                                                         //  current image (0 based)
int      HDFradius;                                                        //  paint mode radius
char     *HDFpixmap = 0;                                                   //  map input image per output pixel
float    *HDFwarpx[10], *HDFwarpy[10];                                     //  warp memory, pixel displacements


void HDF_tweak()                                                           //  v.10.7
{
   char     imageN[8] = "imageN", labN[4] = "0";
   int      cc, imx, ww, hh;

   int HDF_tweak_dialog_event(zdialog *zd, cchar *event);

   //    image   (o) 1  (o) 2  (o) 3  ...
   //    (o) paint  radius [___]
   //    (o) warp [__]
   //    [x] my mouse

   HDFzd = zdialog_new(ZTX("Paint and Warp Image"),mWin,Bdone,Bcancel,null);

   zdialog_add_widget(HDFzd,"hbox","hbim","dialog",0,"space=3");
   zdialog_add_widget(HDFzd,"label","labim","hbim",ZTX("image"),"space=5");
   zdialog_add_widget(HDFzd,"hbox","hbpw","dialog",0,"space=3");
   zdialog_add_widget(HDFzd,"vbox","vbpw1","hbpw",0,"homog|space=5");
   zdialog_add_widget(HDFzd,"vbox","vbpw2","hbpw",0,"homog|space=5");
   zdialog_add_widget(HDFzd,"radio","paint","vbpw1",ZTX("paint"));
   zdialog_add_widget(HDFzd,"radio","warp","vbpw1",ZTX("warp"));
   zdialog_add_widget(HDFzd,"hbox","hbp","vbpw2");
   zdialog_add_widget(HDFzd,"label","labpr","hbp",Bradius,"space=5");
   zdialog_add_widget(HDFzd,"spin","radius","hbp","1|400|1|100");
   zdialog_add_widget(HDFzd,"label","space","vbpw2");
   zdialog_add_widget(HDFzd,"hbox","hbsr","dialog");
   zdialog_add_widget(HDFzd,"check","mymouse","hbsr",BmyMouse,"space=5");

   for (imx = 0; imx < cimNF; imx++) {                                     //  add radio button for each image
      imageN[5] = '1' + imx;
      labN[0] = '1' + imx;
      zdialog_add_widget(HDFzd,"radio",imageN,"hbim",labN);
   }
   
   zdialog_stuff(HDFzd,"paint",1);                                         //  paint button on
   zdialog_stuff(HDFzd,"warp",0);                                          //  warp button off
   zdialog_stuff(HDFzd,"image1",1);                                        //  initial image = 1st

   HDFmode = 0;                                                            //  start in paint mode
   HDFimage = 0;                                                           //  initial image
   HDFradius = 100;                                                        //  paint radius

   takeMouse(HDFzd,HDF_mousefunc,0);                                       //  connect mouse function          v.10.12
   
   cc = E3ww * E3hh;                                                       //  allocate pixel map
   HDFpixmap = zmalloc(cc,"HDF.pixmap");
   memset(HDFpixmap,cimNF,cc);                                             //  initial state, blend all images

   for (imx = 0; imx < cimNF; imx++) {                                     //  allocate warp memory
      ww = cimPXMw[imx]->ww;
      hh = cimPXMw[imx]->hh;
      HDFwarpx[imx] = (float *) zmalloc(ww * hh * sizeof(float),"hdf.warp");
      HDFwarpy[imx] = (float *) zmalloc(ww * hh * sizeof(float),"hdf.warp");
   }
   
   start_thread(HDF_combine_thread,0);                                     //  start working thread
   signal_thread();

   zdialog_run(HDFzd,HDF_tweak_dialog_event);                              //  run dialog, parallel
   zdialog_wait(HDFzd);                                                    //  wait for completion

   return;
}


//  dialog event and completion callback function

int HDF_tweak_dialog_event(zdialog *zd, cchar *event)                      //  v.10.7
{
   int      imx, nn, mymouse;
   
   if (zd->zstat)                                                          //  dialog finish
   {
      freeMouse();                                                         //  disconnect mouse function    v.10.12
      signal_thread();
      wrapup_thread(8);
      if (zd->zstat == 1) HDFstat = 1;
      else HDFstat = 0;
      if (HDFstat == 1) cim_trim();                                        //  cut-off edges       v.10.9
      zdialog_free(HDFzd);
      HDFmode = 0;
      zfree(HDFpixmap);                                                    //  free pixel map
      for (imx = 0; imx < cimNF; imx++) {
         zfree(HDFwarpx[imx]);                                             //  free warp memory
         zfree(HDFwarpy[imx]);
      }
   }
   
   if (strEqu(event,"paint")) {                                            //  set paint mode
      zdialog_fetch(zd,"paint",nn);
      if (! nn) return 1;
      HDFmode = 0;
      gdk_window_set_cursor(drWin->window,0);                              //  no drag cursor            v.11.03
   }
   
   if (strEqu(event,"warp")) {                                             //  set warp mode
      zdialog_fetch(zd,"warp",nn);
      if (! nn) return 1;
      HDFmode = 1;
      paint_toparc(2);                                                     //  stop brush outline
      zdialog_fetch(zd,"mymouse",mymouse);
      if (mymouse) 
         gdk_window_set_cursor(drWin->window,dragcursor);                  //  set drag cursor           v.11.03
   }
   
   if (strnEqu(event,"image",5)) {                                         //  image radio button
      nn = event[5] - '0';                                                 //  1 to cimNF
      if (nn > 0 && nn <= cimNF) 
         HDFimage = nn - 1;                                                //  0 to cimNF-1
      signal_thread();
   }

   if (strEqu(event,"radius"))                                             //  change paint radius
      zdialog_fetch(zd,"radius",HDFradius);

   if (strEqu(event,"mymouse")) {                                          //  toggle mouse capture      v.10.12
      zdialog_fetch(zd,"mymouse",mymouse);
      if (mymouse) {
         takeMouse(zd,HDF_mousefunc,0);                                    //  connect mouse function
         if (HDFmode == 1) 
            gdk_window_set_cursor(drWin->window,dragcursor);               //  warp mode, drag cursor    v.11.03
         signal_thread();
      }
      else freeMouse();                                                    //  disconnect mouse
   }

   return 1;
}


//  HDF dialog mouse function
//  paint:  during drag, selected image >> HDFpixmap (within paint radius) >> E3
//  warp:   for selected image, cimPXMs >> warp >> cimPXMw >> E3

void HDF_mousefunc()                                                       //  v.10.7
{
   uint16      vpix1[3], *pix2, *pix3;
   int         imx, radius, radius2, vstat1;
   int         mx, my, dx, dy, px3, py3;
   char        imageN[8] = "imageN";
   double      px1, py1;
   double      xoff, yoff, sintf[10], costf[10];
   int         ii, px, py, ww, hh;
   double      mag, dispx, dispy, d1, d2;
   PXM         *pxm1, *pxm2;
   
   if (HDFmode == 0) goto paint;
   if (HDFmode == 1) goto warp;
   return;

paint:

   radius = HDFradius;                                                     //  paintbrush radius
   radius2 = radius * radius;

   toparcx = Mxposn - radius;                                              //  paintbrush outline circle
   toparcy = Myposn - radius;
   toparcw = toparch = 2 * radius;
   Ftoparc = 1;
   paint_toparc(3);

   if (LMclick || RMclick) {                                               //  mouse click
      LMclick = RMclick = 0;
      return;                                                              //  ignore     v.10.8
   }

   else if (Mxdrag || Mydrag) {                                            //  drag in progress
      mx = Mxdrag;
      my = Mydrag;
   }
   
   else return;

   if (mx < 0 || mx > E3ww-1 || my < 0 || my > E3hh-1)                     //  mouse outside image area
      return;

   for (imx = 0; imx < cimNF; imx++)                                       //  pre-calculate trig funcs
   {
      sintf[imx] = sin(cimOffs[imx].tf);
      costf[imx] = cos(cimOffs[imx].tf);
   }

   for (dy = -radius; dy <= radius; dy++)                                  //  loop pixels around mouse
   for (dx = -radius; dx <= radius; dx++)
   {
      if (dx*dx + dy*dy > radius2) continue;                               //  outside radius

      px3 = mx + dx;                                                       //  output pixel
      py3 = my + dy;
      if (px3 < 0 || px3 > E3ww-1) continue;                               //  outside image
      if (py3 < 0 || py3 > E3hh-1) continue;

      pix3 = PXMpix(E3pxm16,px3,py3);                                      //  output pixel

      imx = py3 * E3ww + px3;                                              //  update pixmap to selected image
      HDFpixmap[imx] = HDFimage;
      
      imx = HDFimage;
      xoff = cimOffs[imx].xf;
      yoff = cimOffs[imx].yf;
      px1 = costf[imx] * (px3 - xoff) + sintf[imx] * (py3 - yoff);         //  input virtual pixel
      py1 = costf[imx] * (py3 - yoff) - sintf[imx] * (px3 - xoff);
      vstat1 = vpixel(cimPXMw[imx],px1,py1,vpix1);
      if (vstat1) {
         pix3[0] = vpix1[0];
         pix3[1] = vpix1[1];
         pix3[2] = vpix1[2];
      }
      else pix3[0] = pix3[1] = pix3[2] = 0;
   }

   mx = mx - radius - 1;                                                   //  update window    v.10.12
   my = my - radius - 1;
   ww = 2 * radius + 3;
   paint_toparc(2);
   mwpaint3(mx,my,ww,ww);
   Ftoparc = 1;
   paint_toparc(3);

   return;

warp:

   if (LMclick || RMclick) {                                               //  mouse click
      LMclick = RMclick = 0;                                               //  ignore     v.10.8
      return;
   }

   else if (Mxdrag || Mydrag) {                                            //  drag in progress
      mx = Mxdrag;
      my = Mydrag;
   }
   
   else return;

   if (mx < 0 || mx > E3ww-1 || my < 0 || my > E3hh-1)                     //  mouse outside image area
      return;

   imx = my * E3ww + mx;                                                   //  if pixel has been painted, 
   imx = HDFpixmap[imx];                                                   //    select corresp. image to warp
   if (imx == cimNF) return;                                               //  else no action      v.10.8
   
   if (imx != HDFimage) {
      HDFimage = imx;                                                      //  update selected image and
      imageN[5] = '1' + imx;                                               //    dialog radio button
      zdialog_stuff(HDFzd,imageN,1);
   }

   pxm1 = cimPXMs[imx];                                                    //  input image
   pxm2 = cimPXMw[imx];                                                    //  output image
   ww = pxm2->ww;
   hh = pxm2->hh;

   mx = Mxdown;                                                            //  drag origin, image coordinates
   my = Mydown;
   dx = Mxdrag - Mxdown;                                                   //  drag increment
   dy = Mydrag - Mydown;
   Mxdown = Mxdrag;                                                        //  next drag origin
   Mydown = Mydrag;

   d1 = ww * ww + hh * hh;
   
   for (py = 0; py < hh; py++)                                             //  process all output pixels
   for (px = 0; px < ww; px++)
   {
      d2 = (px-mx)*(px-mx) + (py-my)*(py-my);
      mag = (1.0 - d2 / d1);
      mag = mag * mag * mag * mag;
      mag = mag * mag * mag * mag;
      mag = mag * mag * mag * mag;

      dispx = -dx * mag;                                                   //  displacement = drag * mag
      dispy = -dy * mag;
      
      ii = py * ww + px;
      HDFwarpx[imx][ii] += dispx;                                          //  add this drag to prior sum
      HDFwarpy[imx][ii] += dispy;

      dispx = HDFwarpx[imx][ii];
      dispy = HDFwarpy[imx][ii];

      vstat1 = vpixel(pxm1,px+dispx,py+dispy,vpix1);                       //  input virtual pixel
      pix2 = PXMpix(pxm2,px,py);                                           //  output pixel
      if (vstat1) {
         pix2[0] = vpix1[0];
         pix2[1] = vpix1[1];
         pix2[2] = vpix1[2];
      }
      else pix2[0] = pix2[1] = pix2[2] = 0;
   }

   signal_thread();                                                        //  combine images >> E3 >> main window
   return;
}


//  Combine images in E3pxm16 (not reallocated). Update main window.

void * HDF_combine_thread(void *)                                          //  v.10.7
{
   void * HDF_combine_wthread(void *);
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      mutex_lock(&Fpixmap_lock);                                           //  stop window updates

      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker threads   v.10.7
         start_wt(HDF_combine_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      mutex_unlock(&Fpixmap_lock);                                         //  update window
      mwpaint2();
   }

   return 0;                                                               //  not executed
}


void * HDF_combine_wthread(void *arg)                                      //  worker thread
{
   int         index = *((int *) (arg));                                   //  no more paint and warp modes  v.10.8
   int         px3, py3, vstat1;
   int         imx, red, green, blue;
   double      px, py;
   double      xoff, yoff, sintf[10], costf[10];
   uint16      vpix1[3], *pix3;

   for (imx = 0; imx < cimNF; imx++)                                       //  pre-calculate trig funcs
   {
      sintf[imx] = sin(cimOffs[imx].tf);
      costf[imx] = cos(cimOffs[imx].tf);
   }

   for (py3 = index+1; py3 < E3hh-1; py3 += Nwt)                           //  step through output pixels
   for (px3 = 1; px3 < E3ww-1; px3++)
   {
      pix3 = PXMpix(E3pxm16,px3,py3);
      
      imx = py3 * E3ww + px3;
      imx = HDFpixmap[imx];

      if (imx < cimNF)                                                     //  specific image maps to pixel
      {
         xoff = cimOffs[imx].xf;
         yoff = cimOffs[imx].yf;
         px = costf[imx] * (px3 - xoff) + sintf[imx] * (py3 - yoff);
         py = costf[imx] * (py3 - yoff) - sintf[imx] * (px3 - xoff);
         vstat1 = vpixel(cimPXMw[imx],px,py,vpix1);                        //  corresp. input vpixel
         if (vstat1) {
            pix3[0] = vpix1[0];
            pix3[1] = vpix1[1];
            pix3[2] = vpix1[2];
         }
         else pix3[0] = pix3[1] = pix3[2] = 0;
      }
      
      else                                                                 //  use blend of all images
      {
         red = green = blue = 0;

         for (imx = 0; imx < cimNF; imx++)
         {
            xoff = cimOffs[imx].xf;
            yoff = cimOffs[imx].yf;
            px = costf[imx] * (px3 - xoff) + sintf[imx] * (py3 - yoff);
            py = costf[imx] * (py3 - yoff) - sintf[imx] * (px3 - xoff);
            vstat1 = vpixel(cimPXMw[imx],px,py,vpix1);
            if (vstat1) {
               red += vpix1[0];
               green += vpix1[1];
               blue += vpix1[2];
            }
         }
         
         pix3[0] = red / cimNF;
         pix3[1] = green / cimNF;
         pix3[2] = blue / cimNF;
      }
   }

   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


/**************************************************************************

   Stack/Paint function
   Combine multiple images of one subject taken at different times from
   (almost) the same camera position. Align the images and allow the user 
   to choose which input image to use for each area of the output image, 
   by "painting" with the mouse. Use this to remove tourists and cars that
   move in and out of a scene being photographed.
   
**************************************************************************/

int      STPstat;                                                          //  1 = OK, 0 = failed or canceled
double   STPinitAlignSize = 160;                                           //  initial align image size
double   STPimageIncrease = 1.6;                                           //  image size increase per align cycle
double   STPsampSize = 10000;                                              //  pixel sample size    v.11.03

double   STPinitSearchRange = 5.0;                                         //  initial search range, +/- pixels
double   STPinitSearchStep = 1.0;                                          //  initial search step, pixels
double   STPinitWarpRange = 2.0;                                           //  initial corner warp range
double   STPinitWarpStep = 1.0;                                            //  initial corner warp step
double   STPsearchRange = 2.0;                                             //  normal search range
double   STPsearchStep = 1.0;                                              //  normal search step
double   STPwarpRange = 1.0;                                               //  normal corner warp range
double   STPwarpStep = 0.67;                                               //  normal corner warp step

void * STP_align_thread(void *);
void   STP_tweak();
void   STP_mousefunc();
void * STP_combine_thread(void *);


//  menu function

void m_STP(GtkWidget *, cchar *)                                           //  v.11.02
{
   char        **flist;
   int         imx, err, ww, hh;
   double      diffw, diffh;
   
   zfuncs::F1_help_topic = "stack_paint";                                  //  help topic

   if (mod_keep()) return;                                                 //  warn unsaved changes

   for (imx = 0; imx < 10; imx++)
   {                                                                       //  clear all file and PXM data
      cimFile[imx] = 0;
      cimPXMf[imx] = cimPXMs[imx] = cimPXMw[imx] = 0;
   }

   cimNF = 0;   

   flist = zgetfileN(ZTX("Select 2 to 9 files"),"openN",curr_file);        //  select images to combine
   if (! flist) return;

   for (imx = 0; flist[imx]; imx++);                                       //  count selected files
   if (imx < 2 || imx > 9) {
      zmessageACK(mWin,ZTX("Select 2 to 9 files"));
      goto cleanup;
   }

   cimNF = imx;                                                            //  file count
   for (imx = 0; imx < cimNF; imx++)
      cimFile[imx] = strdupz(flist[imx],0,"STP.file");                     //  set up file list
   
   if (! cim_load_files()) goto cleanup;                                   //  load and check all files

   ww = cimPXMf[0]->ww;
   hh = cimPXMf[0]->hh;
   
   for (imx = 1; imx < cimNF; imx++)                                       //  check image compatibility
   {
      diffw = abs(ww - cimPXMf[imx]->ww);
      diffw = diffw / ww;
      diffh = abs(hh - cimPXMf[imx]->hh);
      diffh = diffh / hh;

      if (diffw > 0.02 || diffh > 0.02) {
         zmessageACK(mWin,ZTX("Images are not all the same size"));
         goto cleanup;
      }
   }

   free_resources();                                                       //  ready to commit

   err = f_open(cimFile[0],0);                                             //  curr_file = 1st file in list
   if (err) goto cleanup;

   if (! edit_setup("STP",0,0)) goto cleanup;                              //  setup edit, no preview

   start_thread(STP_align_thread,0);                                       //  align each pair of images
   wrapup_thread(0);                                                       //  wait for completion
   if (STPstat != 1) goto cancel;

   STP_tweak();                                                            //  combine images based on user inputs
   if (STPstat != 1) goto cancel;

   Fmodified = 1;                                                          //  done
   edit_done();
   goto cleanup;

cancel:
   edit_cancel();

cleanup:

   if (flist) {
      for (imx = 0; flist[imx]; imx++)                                     //  free file list
         zfree(flist[imx]);
      zfree(flist);
   }

   for (imx = 0; imx < cimNF; imx++) {                                     //  free cim file and PXM data
      if (cimFile[imx]) zfree(cimFile[imx]);
      if (cimPXMf[imx]) PXM_free(cimPXMf[imx]);
      if (cimPXMs[imx]) PXM_free(cimPXMs[imx]);
      if (cimPXMw[imx]) PXM_free(cimPXMw[imx]);
   }
   
   *SB_text = 0;
   return;
}


//  align each pair of input images, output combined image to E3pxm16
//  cimPXMf[*]  original image
//  cimPXMs[*]  scaled and color adjusted for pixel comparisons
//  cimPXMw[*]  warped for display

void * STP_align_thread(void *)                                            //  v.11.02
{
   int         imx, im1, im2, ww, hh, ii, nn;
   double      R, maxtf, mintf, midtf;
   double      xoff, yoff, toff, dxoff, dyoff;
   cimoffs     offsets[10];                                                //  x/y/t offsets after alignment
   
   Fzoom = 0;                                                              //  fit to window if big
   Fblowup = 1;                                                            //  scale up to window if small
   Ffuncbusy++;
   cimShrink = 0;                                                          //  no warp shrinkage (pano)
   cimPano = cimPanoV = 0;                                                 //  no pano mode

   for (imx = 0; imx < cimNF; imx++)
      memset(&offsets[imx],0,sizeof(cimoffs));
   
   for (im1 = 0; im1 < cimNF-1; im1++)                                     //  loop each pair of images
   {
      im2 = im1 + 1;

      memset(&cimOffs[im1],0,sizeof(cimoffs));                             //  initial image offsets = 0
      memset(&cimOffs[im2],0,sizeof(cimoffs));
      
      ww = cimPXMf[im1]->ww;                                               //  image dimensions
      hh = cimPXMf[im1]->hh;

      nn = ww;                                                             //  use larger of ww, hh
      if (hh > ww) nn = hh;
      cimScale = STPinitAlignSize / nn;                                    //  initial align image size
      if (cimScale > 1.0) cimScale = 1.0;

      cimBlend = 0;                                                        //  no blend width (use all)
      cim_get_overlap(im1,im2,cimPXMf);                                    //  get overlap area
      cim_match_colors(im1,im2,cimPXMf);                                   //  get color matching factors

      cimSearchRange = STPinitSearchRange;                                 //  initial align search range
      cimSearchStep = STPinitSearchStep;                                   //  initial align search step
      cimWarpRange = STPinitWarpRange;                                     //  initial align corner warp range
      cimWarpStep = STPinitWarpStep;                                       //  initial align corner warp step
      cimSampSize = STPsampSize;                                           //  pixel sample size for align/compare
      cimNsearch = 0;                                                      //  reset align search counter

      while (true)                                                         //  loop, increasing image size
      {
         cim_scale_image(im1,cimPXMs);                                     //  scale images to cimScale
         cim_scale_image(im2,cimPXMs);

         cim_adjust_colors(cimPXMs[im1],1);                                //  apply color adjustments
         cim_adjust_colors(cimPXMs[im2],2);

         cim_warp_image(im1);                                              //  warp images for show
         cim_warp_image(im2);

         cimShowIm1 = im1;                                                 //  show these two images
         cimShowIm2 = im2;                                                 //    with 50/50 blend
         cimShowAll = 0;
         cim_show_images(1,0);                                             //  (y offset can change)

         cim_get_overlap(im1,im2,cimPXMs);                                 //  get overlap area             v.11.04
         cim_get_redpix(im1);                                              //  get high-contrast pixels

         cim_align_image(im1,im2);                                         //  align im2 to im1
         
         zfree(cimRedpix);                                                 //  clear red pixels
         cimRedpix = 0;

         if (cimScale == 1.0) break;                                       //  done

         R = STPimageIncrease;                                             //  next larger image size
         cimScale = cimScale * R;
         if (cimScale > 0.85) {                                            //  if close to end, jump to end
            R = R / cimScale;
            cimScale = 1.0;
         }

         cimOffs[im1].xf *= R;                                             //  scale offsets for larger image
         cimOffs[im1].yf *= R;
         cimOffs[im2].xf *= R;
         cimOffs[im2].yf *= R;

         for (ii = 0; ii < 4; ii++) {
            cimOffs[im1].wx[ii] *= R;
            cimOffs[im1].wy[ii] *= R;
            cimOffs[im2].wx[ii] *= R;
            cimOffs[im2].wy[ii] *= R;
         }

         cimSearchRange = STPsearchRange;                                  //  align search range
         cimSearchStep = STPsearchStep;                                    //  align search step size
         cimWarpRange = STPwarpRange;                                      //  align corner warp range
         cimWarpStep = STPwarpStep;                                        //  align corner warp step size
      }

      offsets[im2].xf = cimOffs[im2].xf - cimOffs[im1].xf;                 //  save im2 offsets from im1
      offsets[im2].yf = cimOffs[im2].yf - cimOffs[im1].yf;
      offsets[im2].tf = cimOffs[im2].tf - cimOffs[im1].tf;

      for (ii = 0; ii < 4; ii++) {
         offsets[im2].wx[ii] = cimOffs[im2].wx[ii] - cimOffs[im1].wx[ii];
         offsets[im2].wy[ii] = cimOffs[im2].wy[ii] - cimOffs[im1].wy[ii];
      }
   }

   for (imx = 0; imx < cimNF; imx++)                                       //  offsets[*] >> cimOffs[*]
      cimOffs[imx] = offsets[imx];
   
   cimOffs[0].xf = cimOffs[0].yf = cimOffs[0].tf = 0;                      //  image 0 at (0,0,0)

   for (im1 = 0; im1 < cimNF-1; im1++)                                     //  absolute offsets for image 1 to last
   {
      im2 = im1 + 1;
      cimOffs[im2].xf += cimOffs[im1].xf;                                  //  x/y/t offsets are additive
      cimOffs[im2].yf += cimOffs[im1].yf;
      cimOffs[im2].tf += cimOffs[im1].tf;

      for (ii = 0; ii < 4; ii++) {                                         //  corner warps are additive
         cimOffs[im2].wx[ii] += cimOffs[im1].wx[ii];
         cimOffs[im2].wy[ii] += cimOffs[im1].wy[ii];
      }
   }

   for (imx = 1; imx < cimNF; imx++)                                       //  re-warp to absolute
      cim_warp_image(imx);

   toff = cimOffs[0].tf;                                                   //  balance +/- thetas
   maxtf = mintf = toff;
   for (imx = 1; imx < cimNF; imx++) {
      toff = cimOffs[imx].tf;
      if (toff > maxtf) maxtf = toff;
      if (toff < mintf) mintf = toff;
   }
   midtf = 0.5 * (maxtf + mintf);
   
   for (imx = 0; imx < cimNF; imx++)
      cimOffs[imx].tf -= midtf;

   for (im1 = 0; im1 < cimNF-1; im1++)                                     //  adjust x/y offsets for images after im1
   for (im2 = im1+1; im2 < cimNF; im2++)                                   //    due to im1 theta offset
   {
      toff = cimOffs[im1].tf;
      xoff = cimOffs[im2].xf - cimOffs[im1].xf;
      yoff = cimOffs[im2].yf - cimOffs[im1].yf;
      dxoff = yoff * sin(toff);
      dyoff = xoff * sin(toff);
      cimOffs[im2].xf -= dxoff;
      cimOffs[im2].yf += dyoff;
   }

   Fzoom = Fblowup = 0;
   Ffuncbusy--;
   STPstat = 1;
   thread_exit();
   return 0;                                                               //  not executed
}


//  paint output image

zdialog  *STPzd = 0;                                                       //  paint dialog
int      STPimage;                                                         //  current image (0 based)
int      STPradius;                                                        //  paint mode radius
char     *STPpixmap = 0;                                                   //  map input image per output pixel


void STP_tweak()                                                           //  v.11.02
{
   char     imageN[8] = "imageN", labN[4] = "0";
   int      cc, imx;

   int STP_tweak_dialog_event(zdialog *zd, cchar *event);

   //    image   (o) 1  (o) 2  (o) 3  ...
   //    [x] my mouse    radius [___]

   STPzd = zdialog_new(ZTX("Select and Paint Image"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(STPzd,"hbox","hbim","dialog",0,"space=3");
   zdialog_add_widget(STPzd,"label","labim","hbim",ZTX("image"),"space=5");
   zdialog_add_widget(STPzd,"hbox","hbmr","dialog",0,"space=3");
   zdialog_add_widget(STPzd,"check","mymouse","hbmr",BmyMouse,"space=5");
   zdialog_add_widget(STPzd,"label","labr","hbmr",Bradius,"space=5");
   zdialog_add_widget(STPzd,"spin","radius","hbmr","1|400|1|100");

   for (imx = 0; imx < cimNF; imx++) {                                     //  add radio button for each image
      imageN[5] = '1' + imx;
      labN[0] = '1' + imx;
      zdialog_add_widget(STPzd,"radio",imageN,"hbim",labN);
   }
   
   zdialog_stuff(STPzd,"image1",1);                                        //  initial image = 1st

   STPimage = 0;                                                           //  initial image
   STPradius = 100;                                                        //  paint radius

   takeMouse(STPzd,STP_mousefunc,0);                                       //  connect mouse function
   
   cc = E3ww * E3hh;                                                       //  allocate pixel map
   STPpixmap = zmalloc(cc,"STP.pixmap");
   memset(STPpixmap,cimNF,cc);                                             //  initial state, blend all images

   start_thread(STP_combine_thread,0);                                     //  start working thread
   signal_thread();

   zdialog_run(STPzd,STP_tweak_dialog_event);                              //  run dialog, parallel
   zdialog_wait(STPzd);                                                    //  wait for completion

   return;
}


//  dialog event and completion callback function

int STP_tweak_dialog_event(zdialog *zd, cchar *event)                      //  v.11.02
{
   int      nn, mymouse;
   
   if (zd->zstat)                                                          //  dialog finish
   {
      freeMouse();                                                         //  disconnect mouse function
      signal_thread();
      wrapup_thread(8);
      if (zd->zstat == 1) STPstat = 1;
      else STPstat = 0;
      if (STPstat == 1) cim_trim();                                        //  cut-off edges
      zdialog_free(STPzd);
      zfree(STPpixmap);                                                    //  free pixel map
   }
   
   if (strnEqu(event,"image",5)) {                                         //  image radio button
      nn = event[5] - '0';                                                 //  1 to cimNF
      if (nn > 0 && nn <= cimNF) 
         STPimage = nn - 1;                                                //  0 to cimNF-1
      signal_thread();
   }

   if (strEqu(event,"radius"))                                             //  change paint radius
      zdialog_fetch(zd,"radius",STPradius);

   if (strEqu(event,"mymouse")) {                                          //  toggle mouse capture
      zdialog_fetch(zd,"mymouse",mymouse);
      if (mymouse) {
         takeMouse(zd,STP_mousefunc,0);                                    //  connect mouse function
         signal_thread();
      }
      else freeMouse();                                                    //  disconnect mouse
   }

   return 1;
}


//  STP dialog mouse function
//  paint:  during drag, selected image >> STPpixmap (within paint radius) >> E3
//  warp:   for selected image, cimPXMs >> warp >> cimPXMw >> E3

void STP_mousefunc()                                                       //  v.11.02
{
   uint16      vpix1[3], *pix3;
   int         imx, radius, radius2, vstat1;
   int         mx, my, dx, dy, px3, py3, ww;
   double      px1, py1;
   double      xoff, yoff, sintf[10], costf[10];
   
   radius = STPradius;                                                     //  paintbrush radius
   radius2 = radius * radius;

   toparcx = Mxposn - radius;                                              //  paintbrush outline circle
   toparcy = Myposn - radius;
   toparcw = toparch = 2 * radius;
   Ftoparc = 1;
   paint_toparc(3);

   if (LMclick || RMclick) {                                               //  mouse click
      LMclick = RMclick = 0;
      return;                                                              //  ignore
   }

   else if (Mxdrag || Mydrag) {                                            //  drag in progress
      mx = Mxdrag;
      my = Mydrag;
   }
   
   else return;

   if (mx < 0 || mx > E3ww-1 || my < 0 || my > E3hh-1)                     //  mouse outside image area
      return;

   for (imx = 0; imx < cimNF; imx++)                                       //  pre-calculate trig funcs
   {
      sintf[imx] = sin(cimOffs[imx].tf);
      costf[imx] = cos(cimOffs[imx].tf);
   }

   for (dy = -radius; dy <= radius; dy++)                                  //  loop pixels around mouse
   for (dx = -radius; dx <= radius; dx++)
   {
      if (dx*dx + dy*dy > radius2) continue;                               //  outside radius

      px3 = mx + dx;                                                       //  output pixel
      py3 = my + dy;
      if (px3 < 0 || px3 > E3ww-1) continue;                               //  outside image
      if (py3 < 0 || py3 > E3hh-1) continue;

      pix3 = PXMpix(E3pxm16,px3,py3);                                      //  output pixel

      imx = py3 * E3ww + px3;                                              //  update pixmap to selected image
      STPpixmap[imx] = STPimage;
      
      imx = STPimage;
      xoff = cimOffs[imx].xf;
      yoff = cimOffs[imx].yf;
      px1 = costf[imx] * (px3 - xoff) + sintf[imx] * (py3 - yoff);         //  input virtual pixel
      py1 = costf[imx] * (py3 - yoff) - sintf[imx] * (px3 - xoff);
      vstat1 = vpixel(cimPXMw[imx],px1,py1,vpix1);
      if (vstat1) {
         pix3[0] = vpix1[0];
         pix3[1] = vpix1[1];
         pix3[2] = vpix1[2];
      }
      else pix3[0] = pix3[1] = pix3[2] = 0;
   }

   mx = mx - radius - 1;                                                   //  update window
   my = my - radius - 1;
   ww = 2 * radius + 3;
   paint_toparc(2);
   mwpaint3(mx,my,ww,ww);
   Ftoparc = 1;
   paint_toparc(3);
   return;
}


//  Combine images in E3pxm16 (not reallocated). Update main window.

void * STP_combine_thread(void *)                                          //  v.11.02
{
   void * STP_combine_wthread(void *);
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      mutex_lock(&Fpixmap_lock);                                           //  stop window updates

      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker threads
         start_wt(STP_combine_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      mutex_unlock(&Fpixmap_lock);                                         //  update window
      mwpaint2();
   }

   return 0;                                                               //  not executed
}


void * STP_combine_wthread(void *arg)                                      //  worker thread
{
   int         index = *((int *) (arg));
   int         px3, py3, vstat1;
   int         imx, red, green, blue;
   double      px, py;
   double      xoff, yoff, sintf[10], costf[10];
   uint16      vpix1[3], *pix3;

   for (imx = 0; imx < cimNF; imx++)                                       //  pre-calculate trig funcs
   {
      sintf[imx] = sin(cimOffs[imx].tf);
      costf[imx] = cos(cimOffs[imx].tf);
   }

   for (py3 = index+1; py3 < E3hh-1; py3 += Nwt)                           //  step through output pixels
   for (px3 = 1; px3 < E3ww-1; px3++)
   {
      pix3 = PXMpix(E3pxm16,px3,py3);
      
      imx = py3 * E3ww + px3;
      imx = STPpixmap[imx];

      if (imx < cimNF)                                                     //  specific image maps to pixel
      {
         xoff = cimOffs[imx].xf;
         yoff = cimOffs[imx].yf;
         px = costf[imx] * (px3 - xoff) + sintf[imx] * (py3 - yoff);
         py = costf[imx] * (py3 - yoff) - sintf[imx] * (px3 - xoff);
         vstat1 = vpixel(cimPXMw[imx],px,py,vpix1);                        //  corresp. input vpixel
         if (vstat1) {
            pix3[0] = vpix1[0];
            pix3[1] = vpix1[1];
            pix3[2] = vpix1[2];
         }
         else pix3[0] = pix3[1] = pix3[2] = 0;
      }
      
      else                                                                 //  use blend of all images
      {
         red = green = blue = 0;

         for (imx = 0; imx < cimNF; imx++)
         {
            xoff = cimOffs[imx].xf;
            yoff = cimOffs[imx].yf;
            px = costf[imx] * (px3 - xoff) + sintf[imx] * (py3 - yoff);
            py = costf[imx] * (py3 - yoff) - sintf[imx] * (px3 - xoff);
            vstat1 = vpixel(cimPXMw[imx],px,py,vpix1);
            if (vstat1) {
               red += vpix1[0];
               green += vpix1[1];
               blue += vpix1[2];
            }
         }
         
         pix3[0] = red / cimNF;
         pix3[1] = green / cimNF;
         pix3[2] = blue / cimNF;
      }
   }

   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


/**************************************************************************

   Stack/Noise function
   Combine multiple photos of the same subject and average the 
   pixels for noise reduction.

**************************************************************************/

double   STN_initAlignSize = 160;                                          //  initial align image size
double   STN_imageIncrease = 1.6;                                          //  image size increase per align cycle
double   STN_sampSize = 6000;                                              //  pixel sample size

double   STN_initSearchRange = 5.0;                                        //  initial search range, +/- pixels
double   STN_initSearchStep = 1.0;                                         //  initial search step, pixels
double   STN_initWarpRange = 2.0;                                          //  initial corner warp range
double   STN_initWarpStep = 1.0;                                           //  initial corner warp step
double   STN_searchRange = 2.0;                                            //  normal search range
double   STN_searchStep = 1.0;                                             //  normal search step
double   STN_warpRange = 1.0;                                              //  normal corner warp range
double   STN_warpStep = 0.67;                                              //  normal corner warp step

int      STN_stat;                                                         //  1 = OK, 0 = failed or canceled
int      STN_average = 1, STN_median = 0;                                  //  use average/median of input pixels
int      STN_exlow = 0, STN_exhigh = 0;                                    //  exclude low/high pixel

void * STN_align_thread(void *);
void   STN_tweak();
void * STN_combine_thread(void *);


//  menu function

void m_STN(GtkWidget *, cchar *)                                           //  new v.10.9
{
   char        **flist;
   int         imx, err, ww, hh;
   double      diffw, diffh;
   
   zfuncs::F1_help_topic = "stack_noise";                                  //  help topic

   if (mod_keep()) return;                                                 //  warn unsaved changes

   for (imx = 0; imx < 10; imx++)
   {                                                                       //  clear all file and PXM data
      cimFile[imx] = 0;
      cimPXMf[imx] = cimPXMs[imx] = cimPXMw[imx] = 0;
   }

   cimNF = 0;   

   flist = zgetfileN(ZTX("Select 2 to 9 files"),"openN",curr_file);        //  select images to combine
   if (! flist) return;

   for (imx = 0; flist[imx]; imx++);                                       //  count selected files
   if (imx < 2 || imx > 9) {
      zmessageACK(mWin,ZTX("Select 2 to 9 files"));
      goto cleanup;
   }

   cimNF = imx;                                                            //  file count
   for (imx = 0; imx < cimNF; imx++)
      cimFile[imx] = strdupz(flist[imx],0,"STN.file");                     //  set up file list
   
   if (! cim_load_files()) goto cleanup;                                   //  load and check all files

   ww = cimPXMf[0]->ww;
   hh = cimPXMf[0]->hh;
   
   for (imx = 1; imx < cimNF; imx++)                                       //  check image compatibility
   {
      diffw = abs(ww - cimPXMf[imx]->ww);
      diffw = diffw / ww;
      diffh = abs(hh - cimPXMf[imx]->hh);
      diffh = diffh / hh;

      if (diffw > 0.02 || diffh > 0.02) {
         zmessageACK(mWin,ZTX("Images are not all the same size"));
         goto cleanup;
      }
   }

   free_resources();                                                       //  ready to commit

   err = f_open(cimFile[0],0);                                             //  curr_file = 1st file in list
   if (err) goto cleanup;

   if (! edit_setup("STN",0,0)) goto cleanup;                              //  setup edit, no preview

   start_thread(STN_align_thread,0);                                       //  align each pair of images
   wrapup_thread(0);                                                       //  wait for completion
   if (STN_stat != 1) goto cancel;

   STN_tweak();                                                            //  combine images based on user inputs
   if (STN_stat != 1) goto cancel;

   Fmodified = 1;                                                          //  done
   edit_done();
   goto cleanup;

cancel:
   edit_cancel();

cleanup:

   if (flist) {
      for (imx = 0; flist[imx]; imx++)                                     //  free file list
         zfree(flist[imx]);
      zfree(flist);
   }

   for (imx = 0; imx < cimNF; imx++) {                                     //  free cim file and PXM data
      if (cimFile[imx]) zfree(cimFile[imx]);
      if (cimPXMf[imx]) PXM_free(cimPXMf[imx]);
      if (cimPXMs[imx]) PXM_free(cimPXMs[imx]);
      if (cimPXMw[imx]) PXM_free(cimPXMw[imx]);
   }
   
   *SB_text = 0;
   return;
}


//  align each image 2nd-last to 1st image
//  cimPXMf[*]  original image
//  cimPXMs[*]  scaled and color adjusted for pixel comparisons
//  cimPXMw[*]  warped for display

void * STN_align_thread(void *)                                            //  v.10.9
{
   int         imx, im1, im2, ww, hh, ii, nn;
   double      R, maxtf, mintf, midtf;
   double      xoff, yoff, toff, dxoff, dyoff;
   
   Fzoom = 0;                                                              //  fit to window if big
   Fblowup = 1;                                                            //  scale up to window if small
   Ffuncbusy++;                                                            //  v.11.01
   cimShrink = 0;                                                          //  no warp shrinkage (pano)
   cimPano = cimPanoV = 0;                                                 //  no pano mode

   for (imx = 1; imx < cimNF; imx++)                                       //  loop 2nd to last image
   {
      im1 = 0;                                                             //  images to align
      im2 = imx;

      memset(&cimOffs[im1],0,sizeof(cimoffs));                             //  initial image offsets = 0
      memset(&cimOffs[im2],0,sizeof(cimoffs));
      
      ww = cimPXMf[im1]->ww;                                               //  image dimensions
      hh = cimPXMf[im1]->hh;

      nn = ww;                                                             //  use larger of ww, hh
      if (hh > ww) nn = hh;
      cimScale = STN_initAlignSize / nn;                                   //  initial align image size
      if (cimScale > 1.0) cimScale = 1.0;

      cimBlend = 0;                                                        //  no blend width (use all)
      cim_get_overlap(im1,im2,cimPXMf);                                    //  get overlap area
      cim_match_colors(im1,im2,cimPXMf);                                   //  get color matching factors

      cimSearchRange = STN_initSearchRange;                                //  initial align search range
      cimSearchStep = STN_initSearchStep;                                  //  initial align search step
      cimWarpRange = STN_initWarpRange;                                    //  initial align corner warp range
      cimWarpStep = STN_initWarpStep;                                      //  initial align corner warp step
      cimSampSize = STN_sampSize;                                          //  pixel sample size for align/compare
      cimNsearch = 0;                                                      //  reset align search counter

      while (true)                                                         //  loop, increasing image size
      {
         cim_scale_image(im1,cimPXMs);                                     //  scale images to cimScale
         cim_scale_image(im2,cimPXMs);

         cim_adjust_colors(cimPXMs[im1],1);                                //  apply color adjustments
         cim_adjust_colors(cimPXMs[im2],2);

         cim_warp_image(im1);                                              //  warp images for show
         cim_warp_image(im2);

         cimShowIm1 = im1;                                                 //  show these two images
         cimShowIm2 = im2;                                                 //    with 50/50 blend
         cimShowAll = 0;
         cim_show_images(1,0);                                             //  (y offset can change)

         cim_get_overlap(im1,im2,cimPXMs);                                 //  get overlap area             v.11.04
         cim_get_redpix(im1);                                              //  get high-contrast pixels

         cim_align_image(im1,im2);                                         //  align im2 to im1
         
         zfree(cimRedpix);                                                 //  clear red pixels
         cimRedpix = 0;

         if (cimScale == 1.0) break;                                       //  done

         R = STN_imageIncrease;                                            //  next larger image size
         cimScale = cimScale * R;
         if (cimScale > 0.85) {                                            //  if close to end, jump to end
            R = R / cimScale;
            cimScale = 1.0;
         }

         cimOffs[im1].xf *= R;                                             //  scale offsets for larger image
         cimOffs[im1].yf *= R;
         cimOffs[im2].xf *= R;
         cimOffs[im2].yf *= R;

         for (ii = 0; ii < 4; ii++) {
            cimOffs[im1].wx[ii] *= R;
            cimOffs[im1].wy[ii] *= R;
            cimOffs[im2].wx[ii] *= R;
            cimOffs[im2].wy[ii] *= R;
         }

         cimSearchRange = STN_searchRange;                                 //  align search range
         cimSearchStep = STN_searchStep;                                   //  align search step size
         cimWarpRange = STN_warpRange;                                     //  align corner warp range
         cimWarpStep = STN_warpStep;                                       //  align corner warp step size
      }
   }

   toff = cimOffs[0].tf;                                                   //  balance +/- thetas
   maxtf = mintf = toff;
   for (imx = 1; imx < cimNF; imx++) {
      toff = cimOffs[imx].tf;
      if (toff > maxtf) maxtf = toff;
      if (toff < mintf) mintf = toff;
   }
   midtf = 0.5 * (maxtf + mintf);
   
   for (imx = 0; imx < cimNF; imx++)
      cimOffs[imx].tf -= midtf;

   for (im1 = 0; im1 < cimNF-1; im1++)                                     //  adjust x/y offsets for images after im1
   for (im2 = im1+1; im2 < cimNF; im2++)                                   //    due to im1 theta offset
   {
      toff = cimOffs[im1].tf;
      xoff = cimOffs[im2].xf - cimOffs[im1].xf;
      yoff = cimOffs[im2].yf - cimOffs[im1].yf;
      dxoff = yoff * sin(toff);
      dyoff = xoff * sin(toff);
      cimOffs[im2].xf -= dxoff;
      cimOffs[im2].yf += dyoff;
   }

   Fzoom = Fblowup = 0;
   Ffuncbusy--;
   STN_stat = 1;
   thread_exit();
   return 0;                                                               //  not executed
}


//  change pixel combination according to user input

void STN_tweak()                                                           //  v.10.9
{
   zdialog     *zd;

   int STN_tweak_dialog_event(zdialog *zd, cchar *event);

   //    Adjust Pixel Composition
   //
   //    (o) use average  (o) use median
   //    [x] omit lowest value
   //    [x] omit highest value

   zd = zdialog_new(ZTX("Adjust Pixel Composition"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
   zdialog_add_widget(zd,"radio","average","hb1","use average","space=3");
   zdialog_add_widget(zd,"radio","median","hb1","use median","space=3");
   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=3");
   zdialog_add_widget(zd,"check","exlow","hb2","omit low pixel","space=3");
   zdialog_add_widget(zd,"check","exhigh","hb2","omit high pixel","space=3");
   
   zdialog_stuff(zd,"average",1);                                          //  default = average
   zdialog_stuff(zd,"median",0);
   zdialog_stuff(zd,"exlow",0);
   zdialog_stuff(zd,"exhigh",0);
   
   STN_average = 1;
   STN_median = 0;
   STN_exlow = 0;
   STN_exhigh = 0;
   
   start_thread(STN_combine_thread,0);                                     //  start working thread
   signal_thread();

   zdialog_resize(zd,250,0);
   zdialog_run(zd,STN_tweak_dialog_event);                                 //  run dialog, parallel
   zdialog_wait(zd);                                                       //  wait for completion

   return;
}


//  dialog event and completion callback function

int STN_tweak_dialog_event(zdialog *zd, cchar *event)                      //  v.10.9
{
   if (zd->zstat) {                                                        //  dialog finish
      if (zd->zstat == 1) STN_stat = 1;
      else STN_stat = 0;
      wrapup_thread(8);
      zdialog_free(zd);
      if (STN_stat == 1) cim_trim();                                       //  trim edges    v.10.9
   }
   
   if (strEqu(event,"average")) {
      zdialog_fetch(zd,"average",STN_average);
      signal_thread();
   }
   
   if (strEqu(event,"median")) {
      zdialog_fetch(zd,"median",STN_median);
      signal_thread();
   }
   
   if (strEqu(event,"exlow")) {
      zdialog_fetch(zd,"exlow",STN_exlow);
      signal_thread();
   }
   
   if (strEqu(event,"exhigh")) {
      zdialog_fetch(zd,"exhigh",STN_exhigh);
      signal_thread();
   }
   
   return 1;
}


//  compute mean/median mix for each output pixel and update E3 image

void * STN_combine_thread(void *)                                          //  v.10.9
{
   void * STN_combine_wthread(void *arg);                                  //  worker thread

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker threads
         start_wt(STN_combine_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed
}


//  worker thread

void * STN_combine_wthread(void *arg)                                      //  v.10.9
{
   int         index = *((int *) arg);
   int         imx, vstat, px3, py3;
   int         red, green, blue;
   int         ii, ns, ns1, ns2;
   int         Rlist[10], Glist[10], Blist[10];
   double      px, py;
   double      xoff, yoff, sintf[10], costf[10];
   uint16      *pix3, vpix[3];

   //    input layers     0      1      2      3      4      5      6      7      8      9     10
   int   nsx[11][2] = { {0,0}, {0,0}, {0,1}, {1,1}, {1,2}, {2,2}, {2,3}, {2,4}, {2,5}, {3,5}, {3,6} };
   
   for (imx = 0; imx < cimNF; imx++)                                       //  pre-calculate trig funcs
   {
      sintf[imx] = sin(cimOffs[imx].tf);
      costf[imx] = cos(cimOffs[imx].tf);
   }

   for (py3 = index+1; py3 < E3hh-1; py3 += Nwt)                           //  step through output pixels
   for (px3 = 1; px3 < E3ww-1; px3++)
   {
      for (imx = ns = 0; imx < cimNF; imx++)                               //  get aligned input pixels
      {
         xoff = cimOffs[imx].xf;
         yoff = cimOffs[imx].yf;
         px = costf[imx] * (px3 - xoff) + sintf[imx] * (py3 - yoff);
         py = costf[imx] * (py3 - yoff) - sintf[imx] * (px3 - xoff);

         vstat = vpixel(cimPXMw[imx],px,py,vpix);
         if (vstat) {
            Rlist[ns] = vpix[0];                                           //  add pixel RGB values to list
            Glist[ns] = vpix[1];
            Blist[ns] = vpix[2];
            ns++;
         }
      }
      
      if (! ns) continue;
      
      if (STN_exlow || STN_exhigh || STN_median) {                         //  RGB values must be sorted
         HeapSort(Rlist,ns);
         HeapSort(Glist,ns);
         HeapSort(Blist,ns);
      }

      red = green = blue = 0;

      if (STN_average)                                                     //  average the input pixels
      {
         ns1 = 0;                                                          //  low and high RGB values
         ns2 = ns - 1;
         
         if (STN_exlow) {                                                  //  exclude low
            ns1++;
            if (ns1 > ns2) ns1--;
         }

         if (STN_exhigh) {                                                 //  exclude high
            ns2--;
            if (ns1 > ns2) ns2++;
         }
         
         for (ii = ns1; ii <= ns2; ii++)                                   //  sum remaining RGB levels
         {
            red += Rlist[ii];
            green += Glist[ii];
            blue += Blist[ii];
         }

         ns = ns2 - ns1 + 1;                                               //  sample count

         red = red / ns;                                                   //  output RGB = average
         green = green / ns;
         blue = blue / ns;
      }
      
      if (STN_median)                                                      //  use median input pixels
      {
         ns1 = nsx[ns][0];                                                 //  middle group of pixels
         ns2 = nsx[ns][1];
         
         for (ii = ns1; ii <= ns2; ii++)
         {
            red += Rlist[ii];
            green += Glist[ii];
            blue += Blist[ii];
         }

         ns = ns2 - ns1 + 1;                                               //  sample count

         red = red / ns;                                                   //  output RGB = average
         green = green / ns;
         blue = blue / ns;
      }

      pix3 = PXMpix(E3pxm16,px3,py3);                                      //  output pixel
      pix3[0] = red;
      pix3[1] = green;
      pix3[2] = blue;
   }

   exit_wt();
   return 0;                                                               //  not executed
}


/**************************************************************************

    Panorama function: join 2, 3, or 4 images.

***************************************************************************/

int      panStat;                                                          //  1 = OK
zdialog  *panozd = 0;                                                      //  pre-align dialog

double   panPreAlignSize = 1000;                                           //  pre-align image size (ww)
double   panInitAlignSize = 200;                                           //  initial align image size
double   panImageIncrease = 1.6;                                           //  image size increase per align cycle
double   panSampSize = 10000;                                              //  pixel sample size

double   panPreAlignBlend = 0.30;                                          //  pre-align blend width * ww
double   panInitBlend = 0.20;                                              //  initial blend width during auto-align
double   panFinalBlend = 0.08;                                             //  final blend width * ww
double   panBlendDecrease = 0.8;                                           //  blend width reduction per align cycle

double   panInitSearchRange = 5.0;                                         //  initial search range, +/- pixels
double   panInitSearchStep = 0.7;                                          //  initial search step, pixels
double   panInitWarpRange = 4.0;                                           //  initial corner warp range, +/- pixels
double   panInitWarpStep = 1.0;                                            //  initial corner warp step, pixels
double   panSearchRange = 3.0;                                             //  normal search range, +/- pixels
double   panSearchStep = 1.0;                                              //  normal search step, pixels
double   panWarpRange = 2.0;                                               //  normal corner warp range, +/- pixels
double   panWarpStep = 1.0;                                                //  normal corner warp step, pixels

void  pano_prealign();                                                     //  manual pre-align
void  pano_align();                                                        //  auto fine-align
void  pano_tweak();                                                        //  user color tweak


//  menu function

void m_pano(GtkWidget *, cchar *)                                          //  v.10.7
{
   int      imx, err;
   char     **flist = 0;

   zfuncs::F1_help_topic = "panorama";                                     //  help topic

   if (mod_keep()) return;                                                 //  warn unsaved changes

   for (imx = 0; imx < 10; imx++)
   {                                                                       //  clear all file and PXM data
      cimFile[imx] = 0;
      cimPXMf[imx] = cimPXMs[imx] = cimPXMw[imx] = 0;
   }
   cimNF = 0;   

   flist = zgetfileN(ZTX("Select 2 to 4 files"),"openN",curr_file);        //  select images to combine
   if (! flist) return;

   for (imx = 0; flist[imx]; imx++);                                       //  count selected files
   if (imx < 2 || imx > 4) {
      zmessageACK(mWin,ZTX("Select 2 to 4 files"));
      goto cleanup;
   }

   cimNF = imx;                                                            //  file count
   for (imx = 0; imx < cimNF; imx++)
      cimFile[imx] = strdupz(flist[imx],0,"pano.file");                    //  set up file list
   
   if (! cim_load_files()) goto cleanup;                                   //  load and check all files

   free_resources();                                                       //  ready to commit

   err = f_open(cimFile[0],0);                                             //  curr_file = 1st file in list
   if (err) goto cleanup;

   if (! edit_setup("pano",0,0)) goto cleanup;                             //  setup edit, no preview

   cimShowAll = 1;                                                         //  for cim_show_images(), show all    v.10.9
   cimShrink = 0;                                                          //  no warp shrinkage                  v.11.04
   cimPano = 1;                                                            //  horizontal pano mode               v.11.04
   cimPanoV = 0;
   
   pano_prealign();                                                        //  manual pre-alignment
   if (panStat != 1) goto cancel;

   pano_align();                                                           //  auto full alignment
   if (panStat != 1) goto cancel;

   pano_tweak();                                                           //  manual color adjustment
   if (panStat != 1) goto cancel;

   Fmodified = 1;                                                          //  done
   edit_done();
   goto cleanup;

cancel:                                                                    //  failed or canceled
   edit_cancel();

cleanup:

   if (flist) {
      for (imx = 0; flist[imx]; imx++)                                     //  free file list
         zfree(flist[imx]);
      zfree(flist);
   }

   for (imx = 0; imx < cimNF; imx++) {                                     //  free cim file and PXM data
      if (cimFile[imx]) zfree(cimFile[imx]);
      if (cimPXMf[imx]) PXM_free(cimPXMf[imx]);
      if (cimPXMs[imx]) PXM_free(cimPXMs[imx]);
      if (cimPXMw[imx]) PXM_free(cimPXMw[imx]);
   }
   
   *SB_text = 0;
   return;
}


//  perform manual pre-align of all images
//  returns alignment data in cimOffs[*]
//  lens_mm and lens_bow may also be altered

void pano_prealign()                                                       //  v.10.7
{
   int    pano_prealign_event(zdialog *zd, cchar *event);                  //  dialog event function
   void * pano_prealign_thread(void *);                                    //  working thread

   int         imx, ww, err = 0;
   cchar       *exifkey = { exif_focal_length_key };
   char        lensname[40], **pp = 0;

   cchar  *align_mess = ZTX("Drag images into rough alignment.\n"
                            "To rotate, drag from lower edge.");
   cchar  *search_mess = ZTX("Search for lens mm and bow");

   pp = exif_get(curr_file,&exifkey,1);                                    //  get lens mm from EXIF if available
   if (pp && *pp) {
      err = convSD(*pp, lens_mm, 20, 1000);                                //  leave lens_bow unchanged (no source)
      strcpy(lensname,"(EXIF)");                                           //  lens name = EXIF
   }

   if (! pp || ! *pp || err) {                                             //  not available
      lens_mm = lens4_mm[curr_lens];                                       //  get curr. lens mm, bow, name
      lens_bow = lens4_bow[curr_lens];
      *lensname = 0;
      strncatv(lensname,40,"(",lens4_name[curr_lens],")",null);
   }
   
   for (imx = 0; imx < 10; imx++)                                          //  set all alignment offsets = 0
      memset(&cimOffs[imx],0,sizeof(cimoffs));
      
   for (imx = ww = 0; imx < cimNF; imx++)                                  //  sum image widths
      ww += cimPXMf[imx]->ww;
   
   cimScale = 1.4 * panPreAlignSize / ww;                                  //  set alignment image scale
   if (cimScale > 1.0) cimScale = 1.0;                                     //  (* 0.7 after overlaps)

   for (imx = 0; imx < cimNF; imx++)                                       //  scale images > cimPXMs[*]
      cim_scale_image(imx,cimPXMs);

   for (imx = 0; imx < cimNF; imx++) {                                     //  curve images, cimPXMs[*] replaced
      cim_curve_image(imx);
      cimPXMw[imx] = PXM_copy(cimPXMs[imx]);                               //  copy to cimPXMw[*] for display
   }
   
   cimOffs[0].xf = cimOffs[0].yf = 0;                                      //  first image at (0,0)

   for (imx = 1; imx < cimNF; imx++)                                       //  position images with 30% overlap
   {                                                                       //    in horizontal row
      cimOffs[imx].xf = cimOffs[imx-1].xf + 0.7 * cimPXMw[imx-1]->ww;
      cimOffs[imx].yf = cimOffs[imx-1].yf;
   }
   
   Fzoom = 0;                                                              //  scale image to fit window
   Fblowup = 1;                                                            //  magnify small image to window size

   cimBlend = panPreAlignBlend * cimPXMw[1]->ww;                           //  overlap in align window
   cim_show_images(1,0);                                                   //  combine and show images in main window

   panozd = zdialog_new(ZTX("Pre-align Images"),mWin,Bproceed,Bcancel,null);     //  start pre-align dialog
   zdialog_add_widget(panozd,"label","lab1","dialog",align_mess,"space=5");
   zdialog_add_widget(panozd,"hbox","hb1","dialog",0,"space=2");
   zdialog_add_widget(panozd,"spin","spmm","hb1","22|200|0.1|35","space=5");     //  [ 35  ]    lens mm  (source)
   zdialog_add_widget(panozd,"label","labmm","hb1",ZTX("lens mm"));              //  [ 0.3 ]    lens bow
   zdialog_add_widget(panozd,"label","lablens","hb1","","space=5");              //  [resize]   resize window
   zdialog_add_widget(panozd,"hbox","hb2","dialog",0,"space=2");                 //  [search]   search lens mm and bow
   zdialog_add_widget(panozd,"spin","spbow","hb2","-9|9|0.01|0","space=5");
   zdialog_add_widget(panozd,"label","labbow","hb2",ZTX("lens bow"));
   zdialog_add_widget(panozd,"hbox","hb3","dialog",0,"space=2");
   zdialog_add_widget(panozd,"button","resize","hb3",ZTX("Resize"),"space=5");
   zdialog_add_widget(panozd,"label","labsiz","hb3",ZTX("resize window"),"space=5");
   zdialog_add_widget(panozd,"hbox","hb4","dialog",0,"space=2");
   zdialog_add_widget(panozd,"button","search","hb4",Bsearch,"space=5");
   zdialog_add_widget(panozd,"label","labsearch","hb4",search_mess,"space=5");

   zdialog_stuff(panozd,"spmm",lens_mm);                                   //  stuff lens data
   zdialog_stuff(panozd,"spbow",lens_bow);
   zdialog_stuff(panozd,"lablens",lensname);                               //  show source of lens data

   panStat = -1;                                                           //  busy status
   gdk_window_set_cursor(drWin->window,dragcursor);                        //  set drag cursor              v.11.03
   zdialog_run(panozd,pano_prealign_event);                                //  start dialog
   start_thread(pano_prealign_thread,0);                                   //  start working thread
   zdialog_wait(panozd);                                                   //  wait for dialog completion
   gdk_window_set_cursor(drWin->window,0);                                 //  restore normal cursor        v.11.03
   Fzoom = Fblowup = 0;
   return;
}


//  pre-align dialog event function

int pano_prealign_event(zdialog *zd, cchar *event)                         //  v.10.7
{
   int      imx;
   double   overlap;

   if (strstr("spmm spbow",event)) {
      zdialog_fetch(zd,"spmm",lens_mm);                                    //  get revised lens data
      zdialog_fetch(zd,"spbow",lens_bow);
   }
   
   if (strEqu(event,"resize"))                                             //  allocate new E3 image
      cim_show_images(1,0);

   if (strEqu(event,"search")) {                                           //  search for optimal lens parms
      if (cimNF != 2)
         zmessageACK(mWin,ZTX("use two images only"));
      else  panStat = 2;                                                   //  tell thread to search
      return 0;
   }

   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1)                                                  //  proceed
         panStat = 1;
      else                                                                 //  cancel or other
         panStat = 0;

      zdialog_free(panozd);                                                //  kill dialog
      wrapup_thread(0);                                                    //  wait for thread
      
      if (! panStat) return 0;                                             //  canceled

      for (imx = 0; imx < cimNF-1; imx++)                                  //  check for enough overlap
      {
         overlap = cim_get_overlap(imx,imx+1,cimPXMs);                     //  v.11.04
         if (overlap < 0.1) {
            zmessageACK(mWin,ZTX("Too little overlap, cannot align"));
            panStat = 0;
            return 0;
         }
      }
   }

   return 0;
}


//  pre-align working thread
//  convert mouse and KB events into image movements                       //  overhauled   v.11.02

void * pano_prealign_thread(void *)
{
   void   pano_autolens();

   cimoffs     offstemp;
   PXM         *pxmtemp;
   char        *ftemp;
   int         im1, im2, imm, imx;
   int         mx0, my0, mx, my;                                           //  mouse drag origin, position
   int         xoff, yoff, lox, hix;
   int         sepx, minsep;
   int         ww, hh, rotate, midx;
   double      lens_mm0, lens_bow0;
   double      dx, dy, t1, t2, dt;
      
   imm = ww = hh = rotate = xoff = yoff = 0;                               //  stop compiler warnings

   lens_mm0 = lens_mm;                                                     //  to detect changes
   lens_bow0 = lens_bow;

   mx0 = my0 = 0;                                                          //  no drag in progress
   Mcapture = KBcapture = 1;                                               //  capture mouse drag and KB keys

   cimBlend = 0;                                                           //  full blend during pre-align

   while (true)                                                            //  loop and align until done
   {
      zsleep(0.05);                                                        //  logic simplified
      
      if (panStat == 2) {                                                  //  dialog search button
         panStat = -1;                                                     //  back to busy status
         pano_autolens();
      }
      
      if (panStat != -1) break;                                            //  quit signal from dialog
      
      if (lens_mm != lens_mm0 || lens_bow != lens_bow0) {                  //  change in lens parameters
         lens_mm0 = lens_mm;
         lens_bow0 = lens_bow;

         for (imx = 0; imx < cimNF; imx++) {                               //  re-curve images
            cim_scale_image(imx,cimPXMs);
            cim_curve_image(imx);
            PXM_free(cimPXMw[imx]);
            cimPXMw[imx] = PXM_copy(cimPXMs[imx]);
         }

         cim_show_images(1,0);                                             //  combine and show images
         continue;
      }
      
      if (KBkey) {                                                         //  KB input
         if (KBkey == GDK_Left)  cimOffs[imm].xf -= 0.5;                   //  tweak alignment offsets
         if (KBkey == GDK_Right) cimOffs[imm].xf += 0.5;
         if (KBkey == GDK_Up)    cimOffs[imm].yf -= 0.5;
         if (KBkey == GDK_Down)  cimOffs[imm].yf += 0.5;
         if (KBkey == GDK_r)     cimOffs[imm].tf += 0.0005;
         if (KBkey == GDK_l)     cimOffs[imm].tf -= 0.0005;
         KBkey = 0;

         cim_show_images(0,0);                                             //  combine and show images
         continue;
      }

      if (! Mxdrag && ! Mydrag)                                            //  no drag underway
         mx0 = my0 = 0;                                                    //  reset drag origin

      if (Mxdrag || Mydrag)                                                //  mouse drag underway
      {
         mx = Mxdrag;                                                      //  mouse position in image
         my = Mydrag;

         if (! mx0 && ! my0)                                               //  new drag
         {
            mx0 = mx;                                                      //  set drag origin
            my0 = my;
            minsep = 9999;
            
            for (imx = 0; imx < cimNF; imx++)                              //  find image with midpoint
            {                                                              //    closest to mouse x
               lox = cimOffs[imx].xf;
               hix = lox + cimPXMw[imx]->ww;
               midx = (lox + hix) / 2;
               sepx = abs(midx - mx0);
               if (sepx < minsep) {
                  minsep = sepx;
                  imm = imx;                                               //  image to drag or rotate
               }
            }
            
            xoff = cimOffs[imm].xf;
            yoff = cimOffs[imm].yf;
            ww = cimPXMw[imm]->ww;
            hh = cimPXMw[imm]->hh;

            rotate = 0;                                                    //  if drag at bottom edge,
            if (my0 > yoff + 0.85 * hh) rotate = 1;                        //    set rotate flag         v.11.04
         }
         
         if (mx != mx0 || my != my0)                                       //  drag is progressing
         {
            dx = mx - mx0;                                                 //  mouse movement
            dy = my - my0;
            
            if (rotate && my0 > yoff && my > yoff)                         //  rotation
            {
               if (imm > 0) {
                  lox = cimOffs[imm].xf;                                   //  if there is an image to the left,
                  hix = cimOffs[imm-1].xf + cimPXMw[imm-1]->ww;            //    midx = midpoint of overlap
                  midx = (lox + hix) / 2;
               }
               else midx = 0;                                              //  this is the leftmost image

               t1 = atan(1.0 * (mx0-xoff) / (my0-yoff));
               t2 = atan(1.0 * (mx-xoff) / (my-yoff));
               dt = t1 - t2;                                               //  angle change
               dx = dt * (hh/2 + yoff);                                    //  pivot = middle of overlap on left
               dy = -dt * (midx-xoff);
            }

            else  dt = 0;                                                  //  x/y drag

            cimOffs[imm].xf += dx;                                         //  update image
            cimOffs[imm].yf += dy;
            cimOffs[imm].tf += dt;
            xoff = cimOffs[imm].xf;                                        //  v.11.04
            yoff = cimOffs[imm].yf;

            cim_show_images(0,0);                                          //  show combined images

            mx0 = mx;                                                      //  next drag origin = current mouse
            my0 = my;
         }
      }

      for (im1 = 0; im1 < cimNF-1; im1++)                                  //  track image order changes
      {
         im2 = im1 + 1;
         if (cimOffs[im2].xf < cimOffs[im1].xf) 
         {
            ftemp = cimFile[im2];                                          //  switch filespecs
            cimFile[im2] = cimFile[im1];
            cimFile[im1] = ftemp;
            pxmtemp = cimPXMf[im2];                                        //  switch images
            cimPXMf[im2] = cimPXMf[im1];
            cimPXMf[im1] = pxmtemp;
            pxmtemp = cimPXMs[im2];                                        //  scaled images
            cimPXMs[im2] = cimPXMs[im1];
            cimPXMs[im1] = pxmtemp;
            pxmtemp = cimPXMw[im2];                                        //  warped images
            cimPXMw[im2] = cimPXMw[im1];
            cimPXMw[im1] = pxmtemp;
            offstemp = cimOffs[im2];                                       //  offsets
            cimOffs[im2] = cimOffs[im1];
            cimOffs[im1] = offstemp;
            if (imm == im1) imm = im2;                                     //  current drag image
            else if (imm == im2) imm = im1;
            break;
         }
      }
   }
   
   KBcapture = Mcapture = 0;
   thread_exit();
   return 0;                                                               //  not executed, stop g++ warning
}


//  optimize lens parameters
//  inputs and outputs:
//     pre-aligned images cimPXMw[0] and [1]
//     offsets in cimOffs[0] and [1]
//     lens_mm, lens_bow

void pano_autolens()                                                       //  v.10.7
{
   double      mm_range, bow_range, xf_range, yf_range, tf_range;
   double      squeeze, xf_rfinal, rnum, matchB, matchlev;
   double      overlap, lens_mmB, lens_bowB;
   int         imx, randcount = 0;
   cimoffs     offsetsB;

   overlap = cim_get_overlap(0,1,cimPXMs);                                 //  v.11.04
   if (overlap < 0.1) {
      threadmessage = ZTX("Too little overlap, cannot align");
      return;
   }

   Ffuncbusy++;                                                            //  v.11.01

   cimSampSize = 2000;                                                     //  v.11.03
   cimNsearch = 0;

   mm_range = 0.1 * lens_mm;                                               //  set initial search ranges    v.11.03
   bow_range = 0.3 * lens_bow;
   if (bow_range < 0.5) bow_range = 0.5;
   xf_range = 7;
   yf_range = 7;
   tf_range = 0.01;
   xf_rfinal = 0.3;                                                        //  final xf range - when to quit

   cim_match_colors(0,1,cimPXMw);                                          //  adjust colors for image matching
   cim_adjust_colors(cimPXMs[0],1);
   cim_adjust_colors(cimPXMw[0],1);
   cim_adjust_colors(cimPXMs[1],2);
   cim_adjust_colors(cimPXMw[1],2);

   lens_mmB = lens_mm;                                                     //  starting point
   lens_bowB = lens_bow;
   offsetsB = cimOffs[1];
   cimSearchRange = 7;

   matchB = 0;

   while (true)
   {
      srand48(time(0) + randcount++);
      lens_mm = lens_mmB + mm_range * (drand48() - 0.5);                   //  new random lens factors
      lens_bow = lens_bowB + bow_range * (drand48() - 0.5);                //     within search range
      
      for (imx = 0; imx <= 1; imx++) {                                     //  re-curve images
         cim_scale_image(imx,cimPXMs);
         cim_curve_image(imx);
         PXM_free(cimPXMw[imx]);
         cimPXMw[imx] = PXM_copy(cimPXMs[imx]);
      }

      cim_get_redpix(0);                                                   //  get high-contrast pixels     v.11.03
      cim_show_images(0,0);                                                //  combine and show images

      squeeze = 0.97;                                                      //  search range reduction       v.10.7
         
      for (int ii = 0; ii < 1000; ii++)                                    //  loop random x/y/t alignments
      {                                                                    
         rnum = drand48();
         if (rnum < 0.33)                                                  //  random change some alignment offset 
            cimOffs[1].xf = offsetsB.xf + xf_range * (drand48() - 0.5);
         else if (rnum < 0.67)
            cimOffs[1].yf = offsetsB.yf + yf_range * (drand48() - 0.5);
         else
            cimOffs[1].tf = offsetsB.tf + tf_range * (drand48() - 0.5);

         matchlev = cim_match_images(0,1);                                 //  test quality of image alignment

         sprintf(SB_text,"align: %d  match: %.5f  lens: %.1f %.2f",        //  update status bar
                           ++cimNsearch, matchB, lens_mmB, lens_bowB);
         zmainloop();

         if (sigdiff(matchlev,matchB,0.00001) > 0) {
            matchB = matchlev;                                             //  save new best fit
            lens_mmB = lens_mm;                                            //  alignment is better
            lens_bowB = lens_bow;
            offsetsB = cimOffs[1];
            cim_show_images(0,0);
            squeeze = 1;                                                   //  keep same search range as long
            break;                                                         //    as improvements are found
         }

         if (panStat != -1) goto done;                                     //  user kill
      }
      
      if (xf_range < xf_rfinal) goto done;                                 //  finished

      sprintf(SB_text,"align: %d  match: %.5f  lens: %.1f %.2f",           //  update status bar
                        cimNsearch, matchB, lens_mmB, lens_bowB);
      zmainloop();

      mm_range = squeeze * mm_range;                                       //  reduce search range if no 
      if (mm_range < 0.02 * lens_mmB) mm_range = 0.02 * lens_mmB;          //    improvements were found
      bow_range = squeeze * bow_range;
      if (bow_range < 0.1 * lens_bowB) bow_range = 0.1 * lens_bowB;
      if (bow_range < 0.2) bow_range = 0.2;
      xf_range = squeeze * xf_range;
      yf_range = squeeze * yf_range;
      tf_range = squeeze * tf_range;
   }

done:
   zfree(cimRedpix);
   cimRedpix = 0;

   lens_mm = lens_mmB;                                                     //  save best lens params found
   lens_bow = lens_bowB;
   if (panStat == -1 && panozd) {                                          //  unless killed 
      zdialog_stuff(panozd,"spmm",lens_mm);
      zdialog_stuff(panozd,"spbow",lens_bow);
   }

   cimSampSize = panSampSize;                                              //  restore
   Ffuncbusy--;
   cim_show_images(1,0);                                                   //  images are left color-matched
   return;
}


//  fine-alignment
//  start with very small image size
//  search around offset values for best match
//  increase image size and loop until full-size

void pano_align()                                                          //  v.10.7
{
   int         imx, im1, im2, ww;
   double      R, dx, dy, dt;
   double      overlap;
   cimoffs     offsets0;
   
   Fzoom = 0;                                                              //  scale E3 to fit window
   Fblowup = 1;                                                            //  magnify small image to window size
   Ffuncbusy++;                                                            //  v.11.01

   for (imx = 0; imx < cimNF; imx++) {
      cimOffs[imx].xf = cimOffs[imx].xf / cimScale;                        //  scale x/y offsets for full-size images
      cimOffs[imx].yf = cimOffs[imx].yf / cimScale;
   }

   cimScale = 1.0;                                                         //  full-size

   for (imx = 0; imx < cimNF; imx++) {
      PXM_free(cimPXMs[imx]);
      cimPXMs[imx] = PXM_copy(cimPXMf[imx]);                               //  copy full-size images
      cim_curve_image(imx);                                                //  curve them
   }

   cimBlend = 0.3 * cimPXMs[0]->ww;
   cim_get_overlap(0,1,cimPXMs);                                           //  match images 0 & 1 in overlap area
   cim_match_colors(0,1,cimPXMs);
   cim_adjust_colors(cimPXMf[0],1);                                        //  image 0 << profile 1
   cim_adjust_colors(cimPXMf[1],2);                                        //  image 1 << profile 2

   if (cimNF > 2) {
      cimBlend = 0.3 * cimPXMs[1]->ww;
      cim_get_overlap(1,2,cimPXMs);
      cim_match_colors(1,2,cimPXMs);
      cim_adjust_colors(cimPXMf[0],1);
      cim_adjust_colors(cimPXMf[1],1);
      cim_adjust_colors(cimPXMf[2],2);
   }

   if (cimNF > 3) {
      cimBlend = 0.3 * cimPXMs[2]->ww;
      cim_get_overlap(2,3,cimPXMs);
      cim_match_colors(2,3,cimPXMs);
      cim_adjust_colors(cimPXMf[0],1);
      cim_adjust_colors(cimPXMf[1],1);
      cim_adjust_colors(cimPXMf[2],1);
      cim_adjust_colors(cimPXMf[3],2);
   }

   cimScale = panInitAlignSize / cimPXMf[1]->hh;                           //  initial align image scale
   if (cimScale > 1.0) cimScale = 1.0;

   for (imx = 0; imx < cimNF; imx++) {                                     //  scale offsets for image scale
      cimOffs[imx].xf = cimOffs[imx].xf * cimScale;
      cimOffs[imx].yf = cimOffs[imx].yf * cimScale;
   }
   
   cimSearchRange = panInitSearchRange;                                    //  initial align search range
   cimSearchStep = panInitSearchStep;                                      //  initial align search step
   cimWarpRange = panInitWarpRange;                                        //  initial align corner warp range
   cimWarpStep = panInitWarpStep;                                          //  initial align corner warp step
   ww = cimPXMf[0]->ww * cimScale;                                         //  initial align image width
   cimBlend = ww * panInitBlend;                                           //  initial align blend width
   cimSampSize = panSampSize;                                              //  pixel sample size for align/compare
   cimNsearch = 0;                                                         //  reset align search counter
   
   while (true)                                                            //  loop, increasing image size
   {
      for (imx = 0; imx < cimNF; imx++) {                                  //  prepare images
         cim_scale_image(imx,cimPXMs);                                     //  scale to new size
         cim_curve_image(imx);                                             //  curve based on lens params
         cim_warp_image_pano(imx,1);                                       //  apply corner warps
      }

      cim_show_images(1,0);                                                //  show with 50/50 blend in overlaps

      for (im1 = 0; im1 < cimNF-1; im1++)                                  //  fine-align each image with left neighbor
      {
         im2 = im1 + 1;

         offsets0 = cimOffs[im2];                                          //  save initial alignment offsets
         overlap = cim_get_overlap(im1,im2,cimPXMs);                       //  get overlap area          v.11.04
         if (overlap < panFinalBlend-2) {
            zmessageACK(mWin,ZTX("Too little overlap, cannot align"));     //  v.11.03
            goto fail;
         }
         cim_get_redpix(im1);                                              //  get high-contrast pixels
         
         cim_align_image(im1,im2);                                         //  search for best offsets and warps

         zfree(cimRedpix);                                                 //  clear red pixels
         cimRedpix = 0;

         dx = cimOffs[im2].xf - offsets0.xf;                               //  changes from initial offsets
         dy = cimOffs[im2].yf - offsets0.yf;
         dt = cimOffs[im2].tf - offsets0.tf;
         
         for (imx = im2+1; imx < cimNF; imx++)                             //  propagate to following images
         {
            cimOffs[imx].xf += dx;
            cimOffs[imx].yf += dy;
            cimOffs[imx].tf += dt;
            ww = cimOffs[imx].xf - cimOffs[im2].xf;
            cimOffs[imx].yf += ww * dt;
         }
      }

      if (cimScale == 1.0) goto success;                                   //  done

      R = panImageIncrease;                                                //  next larger image size
      cimScale = cimScale * R;
      if (cimScale > 0.85) {                                               //  if close to end, jump to end
         R = R / cimScale;
         cimScale = 1.0;
      }

      for (imx = 0; imx < cimNF; imx++)                                    //  scale offsets for new size
      {
         cimOffs[imx].xf *= R;
         cimOffs[imx].yf *= R;

         for (int ii = 0; ii < 4; ii++) {
            cimOffs[imx].wx[ii] *= R;
            cimOffs[imx].wy[ii] *= R;
         }
      }

      cimSearchRange = panSearchRange;                                     //  align search range
      cimSearchStep = panSearchStep;                                       //  align search step size
      cimWarpRange = panWarpRange;                                         //  align corner warp range
      cimWarpStep = panWarpStep;                                           //  align corner warp step size

      cimBlend = cimBlend * panBlendDecrease * R;                          //  blend width, reduced
      ww = cimPXMf[0]->ww * cimScale;
      if (cimBlend < panFinalBlend * ww) 
         cimBlend = panFinalBlend * ww;                                    //  stay above minimum
   }

success: 
   panStat = 1;
   goto align_done;
fail:
   panStat = 0;
align_done:
   cimBlend = 1;                                                           //  tiny blend (increase in tweak if wanted)
   Fzoom = Fblowup = 0;
   Ffuncbusy--;
   cim_show_images(0,0);
   return;
}


//  get user inputs for RGB changes and blend width, update cimPXMw[*]

void pano_tweak()                                                          //  v.10.7
{
   int    pano_tweak_event(zdialog *zd, cchar *event);                     //  dialog event function
   
   cchar    *tweaktitle = ZTX("Match Brightness and Color");
   char     imageN[8] = "imageN";
   int      imx;
   
   cimBlend = 1;                                                           //  init. blend width
   
   panozd = zdialog_new(tweaktitle,mWin,Bdone,Bcancel,null);

   zdialog_add_widget(panozd,"hbox","hbim","dialog",0,"space=5");
   zdialog_add_widget(panozd,"label","labim","hbim",ZTX("image"),"space=5");      //  image  (o)  (o)  (o)  (o)
   zdialog_add_widget(panozd,"hbox","hbc1","dialog",0,"homog");                   //
   zdialog_add_widget(panozd,"label","labred","hbc1",Bred);                       //     red     green    blue
   zdialog_add_widget(panozd,"label","labgreen","hbc1",Bgreen);                   //   [_____]  [_____]  [_____]
   zdialog_add_widget(panozd,"label","labblue","hbc1",Bblue);                     //
   zdialog_add_widget(panozd,"hbox","hbc2","dialog",0,"homog");                   //   brightness [___]  [apply]
   zdialog_add_widget(panozd,"spin","red","hbc2","50|200|0.1|100","space=5");     //
   zdialog_add_widget(panozd,"spin","green","hbc2","50|200|0.1|100","space=5");   //   --------------------------
   zdialog_add_widget(panozd,"spin","blue","hbc2","50|200|0.1|100","space=5");    //
   zdialog_add_widget(panozd,"hbox","hbbri","dialog",0,"space=5");                //   [auto color]  [file color]
   zdialog_add_widget(panozd,"label","labbr","hbbri",Bbrightness,"space=5");      //
   zdialog_add_widget(panozd,"spin","bright","hbbri","50|200|0.1|100");           //   --------------------------
   zdialog_add_widget(panozd,"button","brapp","hbbri",Bapply,"space=10");         //
   zdialog_add_widget(panozd,"hsep","hsep","dialog",0,"space=5");                 //   blend width [___] [apply]   
   zdialog_add_widget(panozd,"hbox","hbc3","dialog",0,"space=5");                 //
   zdialog_add_widget(panozd,"button","auto","hbc3",ZTX("auto color"),"space=5"); //          [done]  [cancel]
   zdialog_add_widget(panozd,"button","file","hbc3",ZTX("file color"),"space=5");
   zdialog_add_widget(panozd,"hsep","hsep","dialog",0,"space=5");
   zdialog_add_widget(panozd,"hbox","hbblen","dialog",0);
   zdialog_add_widget(panozd,"label","labbl","hbblen",Bblendwidth,"space=5");
   zdialog_add_widget(panozd,"spin","blend","hbblen","1|300|1|1");
   zdialog_add_widget(panozd,"button","blapp","hbblen",Bapply,"space=15");
   
   for (imx = 0; imx < cimNF; imx++) {                                     //  add radio button per image
      imageN[5] = '0' + imx;
      zdialog_add_widget(panozd,"radio",imageN,"hbim",0,"space=5");
   }
   
   zdialog_stuff(panozd,"image0",1);                                       //  pre-select 1st image
   zdialog_resize(panozd,300,0);

   panStat = -1;                                                           //  busy status
   zdialog_run(panozd,pano_tweak_event);                                   //  run dialog, parallel
   zdialog_wait(panozd);                                                   //  wait for dialog completion
   return;
}


//  dialog event function

int pano_tweak_event(zdialog *zd, cchar *event)                            //  v.10.7
{
   char        imageN[8] = "imageN";
   double      red, green, blue, bright, bright2;
   double      red1, green1, blue1;
   int         nn, im0, imx, im1, im2, ww, hh, px, py;
   uint16      *pixel;
   
   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) panStat = 1;                                     //  done
      if (zd->zstat == 2) panStat = 0;                                     //  cancel
      zdialog_free(panozd);                                                //  kill dialog
      return 0;
   }

   for (im0 = 0; im0 < cimNF; im0++) {                                     //  get which image is selected
      imageN[5] = '0' + im0;                                               //    by the radio buttons
      zdialog_fetch(zd,imageN,nn);
      if (nn) break;
   }
   if (im0 == cimNF) return 1;

   zdialog_fetch(zd,"red",red);                                            //  get color adjustments
   zdialog_fetch(zd,"green",green);
   zdialog_fetch(zd,"blue",blue);
   zdialog_fetch(zd,"bright",bright);                                      //  brightness adjustment

   bright2 = (red + green + blue) / 3;                                     //  RGB brightness
   bright = bright / bright2;                                              //  bright setpoint / RGB brightness
   red = red * bright;                                                     //  adjust RGB brightness
   green = green * bright;
   blue = blue * bright;
   
   bright = (red + green + blue) / 3;
   zdialog_stuff(zd,"red",red);                                            //  force back into consistency
   zdialog_stuff(zd,"green",green);
   zdialog_stuff(zd,"blue",blue);
   zdialog_stuff(zd,"bright",bright);
   
   if (strEqu(event,"brapp"))                                              //  apply color & brightness changes
   {
      red = red / 100;                                                     //  normalize 0.5 ... 2.0
      green = green / 100;
      blue = blue / 100;
      
      cim_warp_image_pano(im0,0);                                          //  refresh cimPXMw from cimPXMs

      ww = cimPXMw[im0]->ww;
      hh = cimPXMw[im0]->hh;
      
      for (py = 0; py < hh; py++)                                          //  loop all image pixels
      for (px = 0; px < ww; px++)
      {
         pixel = PXMpix(cimPXMw[im0],px,py);
         red1 = red * pixel[0];                                            //  apply color factors
         green1 = green * pixel[1];
         blue1 = blue * pixel[2];
         if (! blue1) continue;

         if (red1 > 65535 || green1 > 65535 || blue1 > 65535) {
            bright = red1;                                                 //  avoid overflow
            if (green1 > bright) bright = green1;
            if (blue1 > bright) bright = blue1;
            bright = 65535.0 / bright;
            red1 = red1 * bright;
            green1 = green1 * bright;
            blue1 = blue1 * bright;
         }
         
         if (blue1 < 1) blue1 = 1;                                         //  avoid 0       v.10.7

         pixel[0] = red1;
         pixel[1] = green1;
         pixel[2] = blue1;
      }
      
      cimBlend = 1;
      zdialog_stuff(zd,"blend",cimBlend);                                  //  v.11.04
      cim_show_images(0,0);                                                //  combine and show with 50/50 blend
   }
   
   if (strEqu(event,"auto"))                                               //  auto match color of selected image
   {
      for (im1 = im0; im1 < cimNF-1; im1++)                                //  from selected image to last image
      {
         im2 = im1 + 1;
         cimBlend = 0.3 * cimPXMw[im2]->ww;
         cim_get_overlap(im1,im2,cimPXMw);                                 //  match images in overlap area
         cim_match_colors(im1,im2,cimPXMw);
         cim_adjust_colors(cimPXMw[im1],1);                                //  image im1 << profile 1
         cim_adjust_colors(cimPXMw[im2],2);                                //  image im2 << profile 2
         for (imx = im1-1; imx >= im0; imx--)
            cim_adjust_colors(cimPXMw[imx],1);
         cimBlend = 1;
         zdialog_stuff(zd,"blend",cimBlend);                               //  v.11.04
         cim_show_images(0,0);
      }

      for (im1 = im0-1; im1 >= 0; im1--)                                   //  from selected image to 1st image
      {
         im2 = im1 + 1;
         cimBlend = 0.3 * cimPXMw[im2]->ww;
         cim_get_overlap(im1,im2,cimPXMw);                                 //  match images in overlap area
         cim_match_colors(im1,im2,cimPXMw);
         cim_adjust_colors(cimPXMw[im1],1);                                //  image im1 << profile 1
         cim_adjust_colors(cimPXMw[im2],2);                                //  image im2 << profile 2
         for (imx = im2+1; imx < cimNF; imx++)
            cim_adjust_colors(cimPXMw[imx],2);
         cimBlend = 1;
         zdialog_stuff(zd,"blend",cimBlend);                               //  v.11.04
         cim_show_images(0,0);
      }
   }
   
   if (strEqu(event,"file"))                                               //  use original file colors
   {
      if (! cim_load_files()) return 1;

      for (imx = 0; imx < cimNF; imx++) {
         PXM_free(cimPXMs[imx]);
         cimPXMs[imx] = PXM_copy(cimPXMf[imx]);
         cim_curve_image(imx);                                             //  curve and warp
         cim_warp_image_pano(imx,0);
      }

      cimBlend = 1;
      zdialog_stuff(zd,"blend",cimBlend);                                  //  v.11.04
      cim_show_images(0,0);
   }
   
   if (strEqu(event,"blapp"))                                              //  apply new blend width
   {
      zdialog_fetch(zd,"blend",cimBlend);                                  //  can be zero
      cim_show_images(0,1);                                                //  show with gradual blend
   }

   return 1;
}


/**************************************************************************

    Vertical Panorama function: join 2, 3, or 4 images.

***************************************************************************/

void  vpano_prealign();                                                    //  manual pre-align
void  vpano_align();                                                       //  auto fine-align
void  vpano_tweak();                                                       //  user color tweak


//  menu function

void m_vpano(GtkWidget *, cchar *)                                         //  v.11.04
{
   int      imx, err;
   char     **flist = 0;

   zfuncs::F1_help_topic = "panorama";                                     //  help topic

   if (mod_keep()) return;                                                 //  warn unsaved changes

   for (imx = 0; imx < 10; imx++)
   {                                                                       //  clear all file and PXM data
      cimFile[imx] = 0;
      cimPXMf[imx] = cimPXMs[imx] = cimPXMw[imx] = 0;
   }
   cimNF = 0;   

   flist = zgetfileN(ZTX("Select 2 to 4 files"),"openN",curr_file);        //  select images to combine
   if (! flist) return;

   for (imx = 0; flist[imx]; imx++);                                       //  count selected files
   if (imx < 2 || imx > 4) {
      zmessageACK(mWin,ZTX("Select 2 to 4 files"));
      goto cleanup;
   }

   cimNF = imx;                                                            //  file count
   for (imx = 0; imx < cimNF; imx++)
      cimFile[imx] = strdupz(flist[imx],0,"pano.file");                    //  set up file list
   
   if (! cim_load_files()) goto cleanup;                                   //  load and check all files

   free_resources();                                                       //  ready to commit

   err = f_open(cimFile[0],0);                                             //  curr_file = 1st file in list
   if (err) goto cleanup;

   if (! edit_setup("vpano",0,0)) goto cleanup;                            //  setup edit, no preview

   cimShowAll = 1;                                                         //  for cim_show_images(), show all
   cimShrink = 0;                                                          //  no warp shrinkage                  v.11.04
   cimPano = 0;                                                            //  vertical pano mode                 v.11.04
   cimPanoV = 1;

   vpano_prealign();                                                       //  manual pre-alignment
   if (panStat != 1) goto cancel;

   vpano_align();                                                          //  auto full alignment
   if (panStat != 1) goto cancel;

   vpano_tweak();                                                          //  manual color adjustment
   if (panStat != 1) goto cancel;

   Fmodified = 1;                                                          //  done
   edit_done();
   goto cleanup;

cancel:                                                                    //  failed or canceled
   edit_cancel();

cleanup:

   if (flist) {
      for (imx = 0; flist[imx]; imx++)                                     //  free file list
         zfree(flist[imx]);
      zfree(flist);
   }

   for (imx = 0; imx < cimNF; imx++) {                                     //  free cim file and PXM data
      if (cimFile[imx]) zfree(cimFile[imx]);
      if (cimPXMf[imx]) PXM_free(cimPXMf[imx]);
      if (cimPXMs[imx]) PXM_free(cimPXMs[imx]);
      if (cimPXMw[imx]) PXM_free(cimPXMw[imx]);
   }
   
   *SB_text = 0;
   return;
}


//  perform manual pre-align of all images
//  returns alignment data in cimOffs[*]
//  lens_mm and lens_bow may also be altered

void vpano_prealign()
{
   int    vpano_prealign_event(zdialog *zd, cchar *event);                 //  dialog event function
   void * vpano_prealign_thread(void *);                                   //  working thread

   int         imx, hh, err = 0;
   cchar       *exifkey = { exif_focal_length_key };
   char        lensname[40], **pp = 0;

   cchar  *align_mess = ZTX("Drag images into rough alignment.\n"
                            "To rotate, drag from right edge.");

   pp = exif_get(curr_file,&exifkey,1);                                    //  get lens mm from EXIF if available
   if (pp && *pp) {
      err = convSD(*pp, lens_mm, 20, 1000);                                //  leave lens_bow unchanged (no source)
      strcpy(lensname,"(EXIF)");                                           //  lens name = EXIF
   }

   if (! pp || ! *pp || err) {                                             //  not available
      lens_mm = lens4_mm[curr_lens];                                       //  get curr. lens mm, bow, name
      lens_bow = lens4_bow[curr_lens];
      *lensname = 0;
      strncatv(lensname,40,"(",lens4_name[curr_lens],")",null);
   }
   
   for (imx = 0; imx < 10; imx++)                                          //  set all alignment offsets = 0
      memset(&cimOffs[imx],0,sizeof(cimoffs));
      
   for (imx = hh = 0; imx < cimNF; imx++)                                  //  sum image heights
      hh += cimPXMf[imx]->hh;
   
   cimScale = 1.4 * panPreAlignSize / hh;                                  //  set alignment image scale
   if (cimScale > 1.0) cimScale = 1.0;                                     //  (* 0.7 after overlaps)

   for (imx = 0; imx < cimNF; imx++)                                       //  scale images > cimPXMs[*]
      cim_scale_image(imx,cimPXMs);

   for (imx = 0; imx < cimNF; imx++) {                                     //  curve images, cimPXMs[*] replaced
      cim_curve_Vimage(imx);
      cimPXMw[imx] = PXM_copy(cimPXMs[imx]);                               //  copy to cimPXMw[*] for display
   }
   
   cimOffs[0].xf = cimOffs[0].yf = 0;                                      //  first image at (0,0)

   for (imx = 1; imx < cimNF; imx++)                                       //  position images with 30% overlap
   {                                                                       //    in vertical row
      cimOffs[imx].yf = cimOffs[imx-1].yf + 0.7 * cimPXMw[imx-1]->hh;
      cimOffs[imx].xf = cimOffs[imx-1].xf;
   }
   
   Fzoom = 0;                                                              //  scale image to fit window
   Fblowup = 1;                                                            //  magnify small image to window size

   cimBlend = panPreAlignBlend * cimPXMw[1]->hh;                           //  overlap in align window
   cim_show_Vimages(1,0);                                                  //  combine and show images in main window

   panozd = zdialog_new(ZTX("Pre-align Images"),mWin,Bproceed,Bcancel,null);     //  start pre-align dialog
   zdialog_add_widget(panozd,"label","lab1","dialog",align_mess,"space=5");
   zdialog_add_widget(panozd,"hbox","hb1","dialog",0,"space=2");
   zdialog_add_widget(panozd,"spin","spmm","hb1","22|200|0.1|35","space=5");     //  [ 35  ]    lens mm  (source)
   zdialog_add_widget(panozd,"label","labmm","hb1",ZTX("lens mm"));              //  [ 0.3 ]    lens bow
   zdialog_add_widget(panozd,"label","lablens","hb1","","space=5");              //  [resize]   resize window
   zdialog_add_widget(panozd,"hbox","hb2","dialog",0,"space=2");
   zdialog_add_widget(panozd,"spin","spbow","hb2","-9|9|0.01|0","space=5");
   zdialog_add_widget(panozd,"label","labbow","hb2",ZTX("lens bow"));
   zdialog_add_widget(panozd,"hbox","hb3","dialog",0,"space=2");
   zdialog_add_widget(panozd,"button","resize","hb3",ZTX("Resize"),"space=5");
   zdialog_add_widget(panozd,"label","labsiz","hb3",ZTX("resize window"),"space=5");

   zdialog_stuff(panozd,"spmm",lens_mm);                                   //  stuff lens data
   zdialog_stuff(panozd,"spbow",lens_bow);
   zdialog_stuff(panozd,"lablens",lensname);                               //  show source of lens data

   panStat = -1;                                                           //  busy status
   gdk_window_set_cursor(drWin->window,dragcursor);                        //  set drag cursor 
   zdialog_run(panozd,vpano_prealign_event);                               //  start dialog
   start_thread(vpano_prealign_thread,0);                                  //  start working thread
   zdialog_wait(panozd);                                                   //  wait for dialog completion
   gdk_window_set_cursor(drWin->window,0);                                 //  restore normal cursor
   Fzoom = Fblowup = 0;
   return;
}


//  pre-align dialog event function

int vpano_prealign_event(zdialog *zd, cchar *event)
{
   int      imx;
   double   overlap;

   if (strstr("spmm spbow",event)) {
      zdialog_fetch(zd,"spmm",lens_mm);                                    //  get revised lens data
      zdialog_fetch(zd,"spbow",lens_bow);
   }
   
   if (strEqu(event,"resize"))                                             //  allocate new E3 image
      cim_show_Vimages(1,0);

   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1)                                                  //  proceed
         panStat = 1;
      else                                                                 //  cancel or other
         panStat = 0;

      zdialog_free(panozd);                                                //  kill dialog
      wrapup_thread(0);                                                    //  wait for thread
      
      if (! panStat) return 0;                                             //  canceled

      for (imx = 0; imx < cimNF-1; imx++)                                  //  check for enough overlap
      {
         overlap = cim_get_overlap(imx,imx+1,cimPXMs);                     //  v.11.04
         if (overlap < 0.1) {
            zmessageACK(mWin,ZTX("Too little overlap, cannot align"));
            panStat = 0;
            return 0;
         }
      }
   }

   return 0;
}


//  pre-align working thread
//  convert mouse and KB events into image movements                       //  overhauled

void * vpano_prealign_thread(void *)
{
   cimoffs     offstemp;
   PXM         *pxmtemp;
   char        *ftemp;
   int         im1, im2, imm, imx;
   int         mx0, my0, mx, my;                                           //  mouse drag origin, position
   int         xoff, yoff, loy, hiy;
   int         sepy, minsep;
   int         ww, hh, rotate, midy;
   double      lens_mm0, lens_bow0;
   double      dx, dy, t1, t2, dt;
      
   imm = ww = hh = rotate = xoff = yoff = 0;                               //  stop compiler warnings

   lens_mm0 = lens_mm;                                                     //  to detect changes
   lens_bow0 = lens_bow;

   mx0 = my0 = 0;                                                          //  no drag in progress
   Mcapture = KBcapture = 1;                                               //  capture mouse drag and KB keys

   cimBlend = 0;                                                           //  full blend during pre-align

   while (true)                                                            //  loop and align until done
   {
      zsleep(0.05);                                                        //  logic simplified
      
      if (panStat != -1) break;                                            //  quit signal from dialog
      
      if (lens_mm != lens_mm0 || lens_bow != lens_bow0) {                  //  change in lens parameters
         lens_mm0 = lens_mm;
         lens_bow0 = lens_bow;

         for (imx = 0; imx < cimNF; imx++) {                               //  re-curve images
            cim_scale_image(imx,cimPXMs);
            cim_curve_Vimage(imx);
            PXM_free(cimPXMw[imx]);
            cimPXMw[imx] = PXM_copy(cimPXMs[imx]);
         }

         cim_show_Vimages(1,0);                                            //  combine and show images
         continue;
      }
      
      if (KBkey) {                                                         //  KB input
         if (KBkey == GDK_Left)  cimOffs[imm].xf -= 0.5;                   //  tweak alignment offsets
         if (KBkey == GDK_Right) cimOffs[imm].xf += 0.5;
         if (KBkey == GDK_Up)    cimOffs[imm].yf -= 0.5;
         if (KBkey == GDK_Down)  cimOffs[imm].yf += 0.5;
         if (KBkey == GDK_r)     cimOffs[imm].tf += 0.0005;
         if (KBkey == GDK_l)     cimOffs[imm].tf -= 0.0005;
         KBkey = 0;

         cim_show_Vimages(0,0);                                            //  combine and show images
         continue;
      }

      if (! Mxdrag && ! Mydrag)                                            //  no drag underway
         mx0 = my0 = 0;                                                    //  reset drag origin

      if (Mxdrag || Mydrag)                                                //  mouse drag underway
      {
         mx = Mxdrag;                                                      //  mouse position in image
         my = Mydrag;

         if (! mx0 && ! my0)                                               //  new drag
         {
            mx0 = mx;                                                      //  set drag origin
            my0 = my;
            minsep = 9999;
            
            for (imx = 0; imx < cimNF; imx++)                              //  find image with midpoint
            {                                                              //    closest to mouse y
               loy = cimOffs[imx].yf;
               hiy = loy + cimPXMw[imx]->hh;
               midy = (loy + hiy) / 2;
               sepy = abs(midy - my0);
               if (sepy < minsep) {
                  minsep = sepy;
                  imm = imx;                                               //  image to drag or rotate
               }
            }

            xoff = cimOffs[imm].xf;
            yoff = cimOffs[imm].yf;
            ww = cimPXMw[imm]->ww;
            hh = cimPXMw[imm]->hh;

            rotate = 0;                                                    //  if drag at right edge,
            if (mx0 > xoff + 0.85 * ww) rotate = 1;                        //    set rotate flag
         }
         
         if (mx != mx0 || my != my0)                                       //  drag is progressing
         {
            dx = mx - mx0;                                                 //  mouse movement
            dy = my - my0;
            
            if (rotate && my0 > yoff && my > yoff)                         //  rotation
            {
               if (imm > 0) {
                  loy = cimOffs[imm].yf;                                   //  if there is an image above,
                  hiy = cimOffs[imm-1].yf + cimPXMw[imm-1]->hh;            //    midy = midpoint of overlap
                  midy = (loy + hiy) / 2;
               }
               else midy = 0;                                              //  this is the topmist image
               
               t1 = atan(1.0 * (my0-yoff) / (mx0-xoff));
               t2 = atan(1.0 * (my-yoff) / (mx-xoff));
               dt = t2 - t1;                                               //  angle change
               dy = - dt * ww / 2;                                         //  pivot = middle of overlap above
               dx = dt * (midy-yoff);
            }

            else  dt = 0;                                                  //  x/y drag

            cimOffs[imm].xf += dx;                                         //  update image
            cimOffs[imm].yf += dy;
            cimOffs[imm].tf += dt;
            xoff = cimOffs[imm].xf;                                        //  v.11.04
            yoff = cimOffs[imm].yf;

            cim_show_Vimages(0,0);                                         //  show combined images

            mx0 = mx;                                                      //  next drag origin = current mouse
            my0 = my;
         }
      }

      for (im1 = 0; im1 < cimNF-1; im1++)                                  //  track image order changes
      {
         im2 = im1 + 1;
         if (cimOffs[im2].yf < cimOffs[im1].yf) 
         {
            ftemp = cimFile[im2];                                          //  switch filespecs
            cimFile[im2] = cimFile[im1];
            cimFile[im1] = ftemp;
            pxmtemp = cimPXMf[im2];                                        //  switch images
            cimPXMf[im2] = cimPXMf[im1];
            cimPXMf[im1] = pxmtemp;
            pxmtemp = cimPXMs[im2];                                        //  scaled images
            cimPXMs[im2] = cimPXMs[im1];
            cimPXMs[im1] = pxmtemp;
            pxmtemp = cimPXMw[im2];                                        //  warped images
            cimPXMw[im2] = cimPXMw[im1];
            cimPXMw[im1] = pxmtemp;
            offstemp = cimOffs[im2];                                       //  offsets
            cimOffs[im2] = cimOffs[im1];
            cimOffs[im1] = offstemp;
            if (imm == im1) imm = im2;                                     //  current drag image
            else if (imm == im2) imm = im1;
            break;
         }
      }
   }
   
   KBcapture = Mcapture = 0;
   thread_exit();
   return 0;                                                               //  not executed, stop g++ warning
}


//  fine-alignment
//  start with very small image size
//  search around offset values for best match
//  increase image size and loop until full-size

void vpano_align()
{
   int         imx, im1, im2, ww, hh;
   double      R, dx, dy, dt;
   double      overlap;
   cimoffs     offsets0;
   
   Fzoom = 0;                                                              //  scale E3 to fit window
   Fblowup = 1;                                                            //  magnify small image to window size
   Ffuncbusy++;

   for (imx = 0; imx < cimNF; imx++) {
      cimOffs[imx].xf = cimOffs[imx].xf / cimScale;                        //  scale x/y offsets for full-size images
      cimOffs[imx].yf = cimOffs[imx].yf / cimScale;
   }

   cimScale = 1.0;                                                         //  full-size

   for (imx = 0; imx < cimNF; imx++) {
      PXM_free(cimPXMs[imx]);
      cimPXMs[imx] = PXM_copy(cimPXMf[imx]);                               //  copy full-size images
      cim_curve_Vimage(imx);                                               //  curve them
   }

   cimBlend = 0.3 * cimPXMs[0]->hh;
   cim_get_overlap(0,1,cimPXMs);                                           //  match images 0 & 1 in overlap area
   cim_match_colors(0,1,cimPXMs);
   cim_adjust_colors(cimPXMf[0],1);                                        //  image 0 << profile 1
   cim_adjust_colors(cimPXMf[1],2);                                        //  image 1 << profile 2

   if (cimNF > 2) {
      cimBlend = 0.3 * cimPXMs[1]->hh;
      cim_get_overlap(1,2,cimPXMs);
      cim_match_colors(1,2,cimPXMs);
      cim_adjust_colors(cimPXMf[0],1);
      cim_adjust_colors(cimPXMf[1],1);
      cim_adjust_colors(cimPXMf[2],2);
   }

   if (cimNF > 3) {
      cimBlend = 0.3 * cimPXMs[2]->hh;
      cim_get_overlap(2,3,cimPXMs);
      cim_match_colors(2,3,cimPXMs);
      cim_adjust_colors(cimPXMf[0],1);
      cim_adjust_colors(cimPXMf[1],1);
      cim_adjust_colors(cimPXMf[2],1);
      cim_adjust_colors(cimPXMf[3],2);
   }

   cimScale = panInitAlignSize / cimPXMf[1]->hh;                           //  initial align image scale
   if (cimScale > 1.0) cimScale = 1.0;

   for (imx = 0; imx < cimNF; imx++) {                                     //  scale offsets for image scale
      cimOffs[imx].xf = cimOffs[imx].xf * cimScale;
      cimOffs[imx].yf = cimOffs[imx].yf * cimScale;
   }
   
   cimSearchRange = panInitSearchRange;                                    //  initial align search range
   cimSearchStep = panInitSearchStep;                                      //  initial align search step
   cimWarpRange = panInitWarpRange;                                        //  initial align corner warp range
   cimWarpStep = panInitWarpStep;                                          //  initial align corner warp step
   hh = cimPXMf[0]->hh * cimScale;                                         //  initial align image width
   cimBlend = hh * panInitBlend;                                           //  initial align blend width
   cimSampSize = panSampSize;                                              //  pixel sample size for align/compare
   cimNsearch = 0;                                                         //  reset align search counter
   
   while (true)                                                            //  loop, increasing image size
   {
      for (imx = 0; imx < cimNF; imx++) {                                  //  prepare images
         cim_scale_image(imx,cimPXMs);                                     //  scale to new size
         cim_curve_Vimage(imx);                                            //  curve based on lens params
         cim_warp_image_Vpano(imx,1);                                      //  apply corner warps
      }
      
      cim_show_Vimages(1,0);                                               //  show with 50/50 blend in overlaps

      for (im1 = 0; im1 < cimNF-1; im1++)                                  //  fine-align each image with top neighbor
      {
         im2 = im1 + 1;

         offsets0 = cimOffs[im2];                                          //  save initial alignment offsets
         overlap = cim_get_overlap(im1,im2,cimPXMs);                       //  get overlap area                v.11.04
         if (overlap < panFinalBlend-2) {
            zmessageACK(mWin,ZTX("Too little overlap, cannot align"));
            goto fail;
         }

         cim_get_redpix(im1);                                              //  get high-contrast pixels
         
         cim_align_image(im1,im2);                                         //  search for best offsets and warps
         
         zfree(cimRedpix);                                                 //  clear red pixels
         cimRedpix = 0;

         dx = cimOffs[im2].xf - offsets0.xf;                               //  changes from initial offsets
         dy = cimOffs[im2].yf - offsets0.yf;
         dt = cimOffs[im2].tf - offsets0.tf;
         
         for (imx = im2+1; imx < cimNF; imx++)                             //  propagate to following images
         {
            cimOffs[imx].xf += dx;
            cimOffs[imx].yf += dy;
            cimOffs[imx].tf += dt;
            ww = cimOffs[imx].xf - cimOffs[im2].xf;
            cimOffs[imx].yf += ww * dt;
         }
      }
      
      if (cimScale == 1.0) goto success;                                   //  done

      R = panImageIncrease;                                                //  next larger image size
      cimScale = cimScale * R;
      if (cimScale > 0.85) {                                               //  if close to end, jump to end
         R = R / cimScale;
         cimScale = 1.0;
      }

      for (imx = 0; imx < cimNF; imx++)                                    //  scale offsets for new size
      {
         cimOffs[imx].xf *= R;
         cimOffs[imx].yf *= R;

         for (int ii = 0; ii < 4; ii++) {
            cimOffs[imx].wx[ii] *= R;
            cimOffs[imx].wy[ii] *= R;
         }
      }

      cimSearchRange = panSearchRange;                                     //  align search range
      cimSearchStep = panSearchStep;                                       //  align search step size
      cimWarpRange = panWarpRange;                                         //  align corner warp range
      cimWarpStep = panWarpStep;                                           //  align corner warp step size

      cimBlend = cimBlend * panBlendDecrease * R;                          //  blend width, reduced
      hh = cimPXMf[0]->hh * cimScale;
      if (cimBlend < panFinalBlend * hh) 
         cimBlend = panFinalBlend * hh;                                    //  stay above minimum
   }

success: 
   panStat = 1;
   goto align_done;
fail:
   panStat = 0;
align_done:
   Fzoom = Fblowup = 0;
   Ffuncbusy--;
   cimBlend = 1;                                                           //  tiny blend (increase in tweak if wanted)
   cim_show_Vimages(0,0);
   return;
}


//  get user inputs for RGB changes and blend width, update cimPXMw[*]

void vpano_tweak()
{
   int    vpano_tweak_event(zdialog *zd, cchar *event);                    //  dialog event function
   
   cchar    *tweaktitle = ZTX("Match Brightness and Color");
   char     imageN[8] = "imageN";
   int      imx;
   
   cimBlend = 1;                                                           //  init. blend width
   
   panozd = zdialog_new(tweaktitle,mWin,Bdone,Bcancel,null);

   zdialog_add_widget(panozd,"hbox","hbim","dialog",0,"space=5");
   zdialog_add_widget(panozd,"label","labim","hbim",ZTX("image"),"space=5");      //  image  (o)  (o)  (o)  (o)
   zdialog_add_widget(panozd,"hbox","hbc1","dialog",0,"homog");                   //
   zdialog_add_widget(panozd,"label","labred","hbc1",Bred);                       //     red     green    blue
   zdialog_add_widget(panozd,"label","labgreen","hbc1",Bgreen);                   //   [_____]  [_____]  [_____]
   zdialog_add_widget(panozd,"label","labblue","hbc1",Bblue);                     //
   zdialog_add_widget(panozd,"hbox","hbc2","dialog",0,"homog");                   //   brightness [___]  [apply]
   zdialog_add_widget(panozd,"spin","red","hbc2","50|200|0.1|100","space=5");     //
   zdialog_add_widget(panozd,"spin","green","hbc2","50|200|0.1|100","space=5");   //   --------------------------
   zdialog_add_widget(panozd,"spin","blue","hbc2","50|200|0.1|100","space=5");    //
   zdialog_add_widget(panozd,"hbox","hbbri","dialog",0,"space=5");                //   [auto color]  [file color]
   zdialog_add_widget(panozd,"label","labbr","hbbri",Bbrightness,"space=5");      //
   zdialog_add_widget(panozd,"spin","bright","hbbri","50|200|0.1|100");           //   --------------------------
   zdialog_add_widget(panozd,"button","brapp","hbbri",Bapply,"space=10");         //
   zdialog_add_widget(panozd,"hsep","hsep","dialog",0,"space=5");                 //   blend width [___] [apply]   
   zdialog_add_widget(panozd,"hbox","hbc3","dialog",0,"space=5");                 //
   zdialog_add_widget(panozd,"button","auto","hbc3",ZTX("auto color"),"space=5"); //          [done]  [cancel]
   zdialog_add_widget(panozd,"button","file","hbc3",ZTX("file color"),"space=5");
   zdialog_add_widget(panozd,"hsep","hsep","dialog",0,"space=5");
   zdialog_add_widget(panozd,"hbox","hbblen","dialog",0);
   zdialog_add_widget(panozd,"label","labbl","hbblen",Bblendwidth,"space=5");
   zdialog_add_widget(panozd,"spin","blend","hbblen","1|300|1|1");
   zdialog_add_widget(panozd,"button","blapp","hbblen",Bapply,"space=15");
   
   for (imx = 0; imx < cimNF; imx++) {                                     //  add radio button per image
      imageN[5] = '0' + imx;
      zdialog_add_widget(panozd,"radio",imageN,"hbim",0,"space=5");
   }
   
   zdialog_stuff(panozd,"image0",1);                                       //  pre-select 1st image
   zdialog_resize(panozd,300,0);

   panStat = -1;                                                           //  busy status
   zdialog_run(panozd,vpano_tweak_event);                                  //  run dialog, parallel
   zdialog_wait(panozd);                                                   //  wait for dialog completion
   return;
}


//  dialog event function

int vpano_tweak_event(zdialog *zd, cchar *event)
{
   char        imageN[8] = "imageN";
   double      red, green, blue, bright, bright2;
   double      red1, green1, blue1;
   int         nn, im0, imx, im1, im2, ww, hh, px, py;
   uint16      *pixel;
   
   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) panStat = 1;                                     //  done
      if (zd->zstat == 2) panStat = 0;                                     //  cancel
      zdialog_free(panozd);                                                //  kill dialog
      return 0;
   }

   for (im0 = 0; im0 < cimNF; im0++) {                                     //  get which image is selected
      imageN[5] = '0' + im0;                                               //    by the radio buttons
      zdialog_fetch(zd,imageN,nn);
      if (nn) break;
   }
   if (im0 == cimNF) return 1;

   zdialog_fetch(zd,"red",red);                                            //  get color adjustments
   zdialog_fetch(zd,"green",green);
   zdialog_fetch(zd,"blue",blue);
   zdialog_fetch(zd,"bright",bright);                                      //  brightness adjustment

   bright2 = (red + green + blue) / 3;                                     //  RGB brightness
   bright = bright / bright2;                                              //  bright setpoint / RGB brightness
   red = red * bright;                                                     //  adjust RGB brightness
   green = green * bright;
   blue = blue * bright;
   
   bright = (red + green + blue) / 3;
   zdialog_stuff(zd,"red",red);                                            //  force back into consistency
   zdialog_stuff(zd,"green",green);
   zdialog_stuff(zd,"blue",blue);
   zdialog_stuff(zd,"bright",bright);
   
   if (strEqu(event,"brapp"))                                              //  apply color & brightness changes
   {
      red = red / 100;                                                     //  normalize 0.5 ... 2.0
      green = green / 100;
      blue = blue / 100;
      
      cim_warp_image_Vpano(im0,0);                                         //  refresh cimPXMw from cimPXMs

      ww = cimPXMw[im0]->ww;
      hh = cimPXMw[im0]->hh;
      
      for (py = 0; py < hh; py++)                                          //  loop all image pixels
      for (px = 0; px < ww; px++)
      {
         pixel = PXMpix(cimPXMw[im0],px,py);
         red1 = red * pixel[0];                                            //  apply color factors
         green1 = green * pixel[1];
         blue1 = blue * pixel[2];
         if (! blue1) continue;

         if (red1 > 65535 || green1 > 65535 || blue1 > 65535) {
            bright = red1;                                                 //  avoid overflow
            if (green1 > bright) bright = green1;
            if (blue1 > bright) bright = blue1;
            bright = 65535.0 / bright;
            red1 = red1 * bright;
            green1 = green1 * bright;
            blue1 = blue1 * bright;
         }
         
         if (blue1 < 1) blue1 = 1;                                         //  avoid 0

         pixel[0] = red1;
         pixel[1] = green1;
         pixel[2] = blue1;
      }
      
      cimBlend = 1;
      zdialog_stuff(zd,"blend",cimBlend);
      cim_show_Vimages(0,0);                                               //  combine and show with 50/50 blend
   }

   if (strEqu(event,"auto"))                                               //  auto match color of selected image
   {
      for (im1 = im0; im1 < cimNF-1; im1++)                                //  from selected image to last image
      {
         im2 = im1 + 1;
         cimBlend = 0.3 * cimPXMw[im2]->hh;
         cim_get_overlap(im1,im2,cimPXMw);                                 //  match images in overlap area
         cim_match_colors(im1,im2,cimPXMw);
         cim_adjust_colors(cimPXMw[im1],1);                                //  image im1 << profile 1
         cim_adjust_colors(cimPXMw[im2],2);                                //  image im2 << profile 2
         for (imx = im1-1; imx >= im0; imx--)
            cim_adjust_colors(cimPXMw[imx],1);
         cimBlend = 1;
         zdialog_stuff(zd,"blend",cimBlend);
         cim_show_Vimages(0,0);
      }

      for (im1 = im0-1; im1 >= 0; im1--)                                   //  from selected image to 1st image
      {
         im2 = im1 + 1;
         cimBlend = 0.3 * cimPXMw[im2]->hh;
         cim_get_overlap(im1,im2,cimPXMw);                                 //  match images in overlap area
         cim_match_colors(im1,im2,cimPXMw);
         cim_adjust_colors(cimPXMw[im1],1);                                //  image im1 << profile 1
         cim_adjust_colors(cimPXMw[im2],2);                                //  image im2 << profile 2
         for (imx = im2+1; imx < cimNF; imx++)
            cim_adjust_colors(cimPXMw[imx],2);
         cimBlend = 1;
         zdialog_stuff(zd,"blend",cimBlend);
         cim_show_Vimages(0,0);
      }
   }
   
   if (strEqu(event,"file"))                                               //  use original file colors
   {
      if (! cim_load_files()) return 1;

      for (imx = 0; imx < cimNF; imx++) {
         PXM_free(cimPXMs[imx]);
         cimPXMs[imx] = PXM_copy(cimPXMf[imx]);
         cim_curve_Vimage(imx);                                            //  curve and warp
         cim_warp_image_Vpano(imx,0);
      }

      cimBlend = 1;
      zdialog_stuff(zd,"blend",cimBlend);
      cim_show_Vimages(0,0);
   }
   
   if (strEqu(event,"blapp"))                                              //  apply new blend width
   {
      zdialog_fetch(zd,"blend",cimBlend);                                  //  can be zero
      cim_show_Vimages(0,1);                                               //  show with gradual blend
   }

   return 1;
}


