/**************************************************************************

   Fotoxx      edit photos and manage collections

   Copyright 2007 2008 2009 2010 2011  Michael Cornelison
   Source URL: http://kornelix.squarespace.com/fotoxx
   Contact: kornelix2@googlemail.com
   
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program. If not, see http://www.gnu.org/licenses/.

***************************************************************************/

#define EX                                                                 //  disable extern declarations
#include "fotoxx.h"

#define fversion "Fotoxx v.11.05.2"
#define flicense "Free software - GNU General Public License v.3"
#define fhomepage "http://kornelix.squarespace.com/fotoxx"
#define ftranslators                                                                \
      "Translators: Jie Luo, Eugenio Baldi, Justa, Peter Landgren, \n"              \
      "Alexander Krasnopolsky, Arthur Kalverboer, André Campos Rodovalho, \n"       \
      "Stanislas Zeller, Doriano Blengino"
#define fcredits "Software used: GTK, libtiff, ufraw-batch, exiftool"
#define fcontact "Bug reports: kornelix2@googlemail.com"


int main(int argc, char *argv[])
{
   using namespace zfuncs;

   char           lang[8] = "", *pp;
   int            Fclone = 0, ww, hh;
   GdkScreen      *screen;
   
   if (argc > 1 && strEqu(argv[1],"-v")) {                                 //  print fotoxx version and exit
      printf(fversion "\n");              
      return 0;
   }
   
   //  initialize externals                                                //  v.11.05
   
   maxcolor = 0xffff;
   for (int ii = 0; ii < 8; ii++) wtnx[ii] = ii;
   strcpy(jpeg_quality,def_jpeg_quality);
   Dww = mwinww;
   Dhh = mwinhh;
   Mscale = 1.0;
   gridspace[0] = gridspace[1] = 50;
   trimsize[0] = 1600;
   trimsize[1] = 1000;
   editresize[0] = 1600;
   editresize[1] = 1200;
   batchresize[0] = 1600;
   batchresize[1] = 1200;
   emailsize[0] = 1000;
   emailsize[1] = 800;
   sa_pixRGB = &green;
   
   //  command line args

   gtk_init(&argc,&argv);                                                  //  initz. GTK
   zinitapp("fotoxx",null);                                                //  get app directories             v.11.05
   load_params();                                                          //  restore parameters from last session

   for (int ii = 1; ii < argc; ii++)                                       //  command line parameters
   {
      if (strEqu(argv[ii],"-d"))                                           //  -d (debug flag)
         Fdebug = 1;
      else if (strEqu(argv[ii],"-l") && argc > ii+1)                       //  -l language code
         strncpy0(lang,argv[++ii],7);
      else if (strEqu(argv[ii],"-c"))                                      //  -c ww hh  clone instance        v.10.12
         Fclone = 1;
      else if (strEqu(argv[ii],"-slideshow") && argc > ii+1)               //  -slideshow /image/file.jpg      v.11.04
         SS_imagefile = strdupz(argv[++ii],0,"SS_imagefile");
      else if (strEqu(argv[ii],"-music") && argc > ii+1)                   //  -music /music/file.ogg          v.11.04
         SS_musicfile = strdupz(argv[++ii],0,"SS_musicfile");
      else  last_file = strdupz(argv[ii],0,"last_file");                   //  initial file or directory
   }

   ZTXinit(lang);                                                          //  setup translations

   mWin = gtk_window_new(GTK_WINDOW_TOPLEVEL);                             //  create main window
   gtk_window_set_title(MWIN,fversion);
   
   mVbox = gtk_vbox_new(0,0);                                              //  add vert. packing box
   gtk_container_add(GTK_CONTAINER(mWin),mVbox);

   mMbar = create_menubar(mVbox);                                          //  add menus / sub-menus

   GtkWidget *mFile = add_menubar_item(mMbar,ZTX("File"),topmenufunc);
      add_submenu_item(mFile,       ZTX("Image Gallery"),         m_gallery);
      add_submenu_item(mFile,       ZTX("Open Image File"),       m_open);
      add_submenu_item(mFile,       ZTX("Open Previous File"),    m_previous);
      add_submenu_item(mFile,       ZTX("Open Recent File"),      m_recent);
      add_submenu_item(mFile,       ZTX("Save to Same File"),     m_save);
      add_submenu_item(mFile,       ZTX("Save to New File"),      m_saveas);
      add_submenu_item(mFile,       ZTX("Create Blank Image"),    m_create);
      add_submenu_item(mFile,       ZTX("Trash Image File"),      m_trash);
      add_submenu_item(mFile,       ZTX("Rename Image File"),     m_rename);
      add_submenu_item(mFile,       ZTX("Batch Rename Files"),    m_batchrename);
      add_submenu_item(mFile,       ZTX("Create Collection"),     m_create_collection);
      add_submenu_item(mFile,       ZTX("Open Collection"),       m_open_collection);
      add_submenu_item(mFile,       ZTX("Delete Collection"),     m_delete_collection);
      add_submenu_item(mFile,       ZTX("Edit Collection"),       m_edit_collection);
      add_submenu_item(mFile,       ZTX("Print Image File"),      m_print);
      add_submenu_item(mFile,       ZTX("Quit fotoxx"),           m_quit);

   GtkWidget *mTools = add_menubar_item(mMbar,ZTX("Tools"),topmenufunc);
      add_submenu_item(mTools,      ZTX("Check Monitor"),         m_moncheck);
      add_submenu_item(mTools,      ZTX("Monitor Gamma"),         m_mongamma);
      add_submenu_item(mTools,      ZTX("Brightness Graph"),      m_histogram);
      add_submenu_item(mTools,      ZTX("Clone fotoxx"),          m_clone);
      add_submenu_item(mTools,      ZTX("Slide Show"),            m_slideshow);
      add_submenu_item(mTools,      ZTX("Show RGB"),              m_showRGB);
      add_submenu_item(mTools,      ZTX("Grid Lines"),            m_gridlines);
      add_submenu_item(mTools,      ZTX("Lens Parameters"),       m_lensparms);
      add_submenu_item(mTools,      ZTX("Change Language"),       m_lang);
      add_submenu_item(mTools,      ZTX("Add Menu and Launcher"), m_menu_launcher);
      add_submenu_item(mTools,      ZTX("Convert RAW files"),     m_conv_raw);
      add_submenu_item(mTools,      ZTX("Burn Images to CD/DVD"), m_burn);
      add_submenu_item(mTools,      ZTX("E-mail Images"),         m_email);
      add_submenu_item(mTools,      ZTX("Synchronize Files"),     m_syncfiles);
      add_submenu_item(mTools,      ZTX("Memory Usage"),          m_memory_usage);

   GtkWidget *mInfo = add_menubar_item(mMbar,"Info",topmenufunc);
      add_submenu_item(mInfo,       ZTX("Edit Comments"),         m_edit_comments);
      add_submenu_item(mInfo,       ZTX("Edit Caption"),          m_edit_caption);
      add_submenu_item(mInfo,       ZTX("Edit Tags"),             m_edit_tags);
      add_submenu_item(mInfo,       ZTX("Manage Tags"),           m_manage_tags);
      add_submenu_item(mInfo,       ZTX("Batch Add Tags"),        m_batchAddTags);
      add_submenu_item(mInfo,       ZTX("Batch Delete Tag"),      m_batchDelTag);
      add_submenu_item(mInfo,       ZTX("Search Images"),         m_search_images);
      add_submenu_item(mInfo,       ZTX("View Info (short)"),     m_exif_view_short);
      add_submenu_item(mInfo,       ZTX("View Info (long)"),      m_exif_view_long);
      add_submenu_item(mInfo,       ZTX("Edit Info"),             m_exif_edit);
      add_submenu_item(mInfo,       ZTX("Delete Info"),           m_exif_delete);

   GtkWidget *mSelect = add_menubar_item(mMbar,ZTX("Select"),topmenufunc);
      add_submenu_item(mSelect,     ZTX("Select"),                m_select);
      add_submenu_item(mSelect,     ZTX("Show"),                  m_select_show);
      add_submenu_item(mSelect,     ZTX("Hide"),                  m_select_hide);
      add_submenu_item(mSelect,     ZTX("Enable"),                m_select_enable);
      add_submenu_item(mSelect,     ZTX("Disable"),               m_select_disable);
      add_submenu_item(mSelect,     ZTX("Invert"),                m_select_invert);
      add_submenu_item(mSelect,     ZTX("Delete"),                m_select_delete);
      add_submenu_item(mSelect,     ZTX("Copy"),                  m_select_copy);
      add_submenu_item(mSelect,     ZTX("Paste"),                 m_select_paste);
      add_submenu_item(mSelect,     ZTX("Open"),                  m_select_open);
      add_submenu_item(mSelect,     ZTX("Save"),                  m_select_save);
      add_submenu_item(mSelect,     ZTX("Whole Image"),           m_select_whole_image);
      add_submenu_item(mSelect,     ZTX("Select and Edit"),       m_select_edit);

   GtkWidget *mRetouch = add_menubar_item(mMbar,ZTX("Retouch"),topmenufunc);
      add_submenu_item(mRetouch,    ZTX("White Balance"),         m_whitebal);
      add_submenu_item(mRetouch,    ZTX("Make Negative"),         m_negate);
      add_submenu_item(mRetouch,    ZTX("Flatten Brightness"),    m_flatten);
      add_submenu_item(mRetouch,    ZTX("Brightness/Color"),      m_tune);
      add_submenu_item(mRetouch,    ZTX("Brightness Ramp"),       m_brightramp);
      add_submenu_item(mRetouch,    ZTX("Expand Brightness"),     m_xbrange);
      add_submenu_item(mRetouch,    ZTX("Tone Mapping"),          m_tonemap);
      add_submenu_item(mRetouch,    ZTX("Red Eyes"),              m_redeye);
      add_submenu_item(mRetouch,    ZTX("Blur Image"),            m_blur);
      add_submenu_item(mRetouch,    ZTX("Sharpen Image"),         m_sharpen);
      add_submenu_item(mRetouch,    ZTX("Reduce Noise"),          m_denoise);
      add_submenu_item(mRetouch,    ZTX("Smart Erase"),           m_smart_erase);
      add_submenu_item(mRetouch,    ZTX("Remove Dust"),           m_dust);

   GtkWidget *mTransf = add_menubar_item(mMbar,ZTX("Transform"),topmenufunc);
      add_submenu_item(mTransf,     ZTX("Trim Image"),            m_trim);
      add_submenu_item(mTransf,     ZTX("Resize Image"),          m_resize);
      add_submenu_item(mTransf,     ZTX("Batch Resize/Export"),   m_batchresize);
      add_submenu_item(mTransf,     ZTX("Annotate Image"),        m_annotate);
      add_submenu_item(mTransf,     ZTX("Rotate Image"),          m_rotate);
      add_submenu_item(mTransf,     ZTX("Flip Image"),            m_flip);
      add_submenu_item(mTransf,     ZTX("Unbend Image"),          m_unbend);
      add_submenu_item(mTransf,     ZTX("Warp Image (area)"),     m_warp_area);
      add_submenu_item(mTransf,     ZTX("Warp Image (curved)"),   m_warp_curved);
      add_submenu_item(mTransf,     ZTX("Warp Image (linear)"),   m_warp_linear);
      add_submenu_item(mTransf,     ZTX("Warp Image (affine)"),   m_warp_affine);

   GtkWidget *mArt = add_menubar_item(mMbar,ZTX("Art"),topmenufunc);
      add_submenu_item(mArt,        ZTX("Color Depth"),           m_colordep);
      add_submenu_item(mArt,        ZTX("Drawing"),               m_draw);
      add_submenu_item(mArt,        ZTX("Outlines"),              m_outlines);
      add_submenu_item(mArt,        ZTX("Embossing"),             m_emboss);
      add_submenu_item(mArt,        ZTX("Tiles"),                 m_tiles);
      add_submenu_item(mArt,        ZTX("Dots"),                  m_dots);
      add_submenu_item(mArt,        ZTX("Painting"),              m_painting);
      add_submenu_item(mArt,        ZTX("Edit Pixels"),           m_pixedit);

   GtkWidget *mComb = add_menubar_item(mMbar,ZTX("Combine"),topmenufunc);
      add_submenu_item(mComb,       ZTX("High Dynamic Range"),    m_HDR);
      add_submenu_item(mComb,       ZTX("High Depth of Field"),   m_HDF);
      add_submenu_item(mComb,       ZTX("Stack / Paint"),         m_STP);
      add_submenu_item(mComb,       ZTX("Stack / Noise"),         m_STN);
      add_submenu_item(mComb,       ZTX("Panorama"),              m_pano);
      add_submenu_item(mComb,       ZTX("Vertical Panorama"),     m_vpano);

   GtkWidget *mPlugins = add_menubar_item(mMbar,"Plugins",topmenufunc);    //  build plugin menu     v.11.03
      add_submenu_item(mPlugins,ZTX("Edit Plugins"),m_edit_plugins);
      for (int ii = 0; ii < Nplugins; ii++) {
         pp = strstr(plugins[ii]," = ");
         if (! pp) continue;
         *pp = 0;
         add_submenu_item(mPlugins,plugins[ii],m_run_plugin);
         *pp = ' ';
      }
   
   GtkWidget *mHelp = add_menubar_item(mMbar,ZTX("Help"),topmenufunc);
      add_submenu_item(mHelp,       ZTX("About"),                 m_help);
      add_submenu_item(mHelp,       ZTX("User Guide"),            m_help);
      add_submenu_item(mHelp,       "README",                     m_help);
      add_submenu_item(mHelp,       ZTX("Change Log"),            m_help);
      add_submenu_item(mHelp,       ZTX("Translate"),             m_help);
      add_submenu_item(mHelp,       ZTX("Home Page"),             m_help);

   mTbar = create_toolbar(mVbox);                                          //  toolbar buttons
      add_toolbar_button(mTbar,  ZTX("Gallery"),   ZTX("Image Gallery"),         "gallery.png", m_gallery);
      add_toolbar_button(mTbar,  ZTX("Open"),      ZTX("Open Image File"),       "open.png",    m_open);
      add_toolbar_button(mTbar,  ZTX("Prev"),      ZTX("Open Previous File"),    "prev.png",    m_prev);
      add_toolbar_button(mTbar,  ZTX("Next"),      ZTX("Open Next File"),        "next.png",    m_next);
      add_toolbar_button(mTbar,  ZTX("Undo"),      ZTX("Undo One Edit"),         "undo.png",    m_undo);
      add_toolbar_button(mTbar,  ZTX("Redo"),      ZTX("Redo One Edit"),         "redo.png",    m_redo);
      add_toolbar_button(mTbar,  "Zoom+",          ZTX("Zoom-in (bigger)"),      "zoom+.png",   m_zoom);
      add_toolbar_button(mTbar,  "Zoom-",          ZTX("Zoom-out (smaller)"),    "zoom-.png",   m_zoom);
      add_toolbar_button(mTbar,  "separator",      null,                         null,          null);      //  v.11.04
      add_toolbar_button(mTbar,  "separator",      null,                         null,          null);
      add_toolbar_button(mTbar,  ZTX("Save"),      ZTX("Save to Same File"),     "save.png",    m_save);
      add_toolbar_button(mTbar,  ZTX("Save As"),   ZTX("Save to New File"),      "saveas.png",  m_saveas);
      add_toolbar_button(mTbar,  ZTX("Trash"),     ZTX("Move Image to Trash"),   "trash.png",   m_trash);
      add_toolbar_button(mTbar,  "separator",      null,                         null,          null);
      add_toolbar_button(mTbar,  "separator",      null,                         null,          null);
      add_toolbar_button(mTbar,  ZTX("Quit"),      ZTX("Quit fotoxx"),           "quit.png",    m_quit);
      add_toolbar_button(mTbar,  ZTX("Help"),      ZTX("Fotoxx Essentials"),     "help.png",    m_help);
   
   drWin = gtk_drawing_area_new();                                         //  add drawing window
   gtk_box_pack_start(GTK_BOX(mVbox),drWin,1,1,0);

   STbar = create_stbar(mVbox);                                            //  add status bar

   G_SIGNAL(mWin,"delete_event",delete_event,0)                            //  connect signals to windows
   G_SIGNAL(mWin,"destroy",destroy_event,0)
   G_SIGNAL(drWin,"expose-event",mwpaint,0)

   gtk_widget_add_events(drWin,GDK_BUTTON_PRESS_MASK);                     //  connect mouse events
   gtk_widget_add_events(drWin,GDK_BUTTON_RELEASE_MASK);
   gtk_widget_add_events(drWin,GDK_BUTTON_MOTION_MASK);                    //  all buttons
   gtk_widget_add_events(drWin,GDK_POINTER_MOTION_MASK);                   //  pointer motion
   G_SIGNAL(drWin,"button-press-event",mouse_event,0)
   G_SIGNAL(drWin,"button-release-event",mouse_event,0)
   G_SIGNAL(drWin,"motion-notify-event",mouse_event,0)

   G_SIGNAL(mWin,"key-press-event",KBpress,0)                              //  connect KB events
   G_SIGNAL(mWin,"key-release-event",KBrelease,0)
   
   drag_drop_connect(drWin,m_open_drag);                                   //  connect drag-drop event

   gtk_window_set_position(MWIN,GTK_WIN_POS_CENTER);
   gtk_window_set_default_size(MWIN,Dww,Dhh);
   gtk_widget_show_all(mWin);                                              //  show all widgets

   if (Fclone) {                                                           //  clone: use left screen    v.10.12
      screen = gdk_screen_get_default();
      ww = gdk_screen_get_width(screen);
      hh = gdk_screen_get_height(screen);
      ww = ww / 2 - 20;
      hh = hh - 50;
      gdk_window_move_resize(mWin->window,10,10,ww,hh);
   }

   gdkgc = gdk_gc_new(drWin->window);                                      //  initz. graphics context

   black.red = black.green = black.blue = 0;                               //  set up colors
   white.red = white.green = white.blue = maxcolor;
   gray.red = gray.green = gray.blue = 0.9 * maxcolor;
   red.red = maxcolor;  red.green = red.blue = 0;
   green.green = maxcolor; green.red = green.blue = 0;

   colormap = gtk_widget_get_colormap(drWin);
   gdk_rgb_find_color(colormap,&black);
   gdk_rgb_find_color(colormap,&white);
   gdk_rgb_find_color(colormap,&gray);
   gdk_rgb_find_color(colormap,&red);
   gdk_rgb_find_color(colormap,&green);
   
   gdk_gc_set_foreground(gdkgc,&black);
   gdk_gc_set_background(gdkgc,&gray);
   gdk_window_set_background(drWin->window,&gray);                         //  v.11.01
   fg_color = black;

   gdk_gc_set_line_attributes(gdkgc,1,LINEATTRIBUTES);

   arrowcursor = gdk_cursor_new(GDK_TOP_LEFT_ARROW);                       //  cursor for selection
   dragcursor = gdk_cursor_new(GDK_CROSSHAIR);                             //  cursor for dragging
   drawcursor = gdk_cursor_new(GDK_PENCIL);                                //  cursor for drawing lines

   busycursor = zmakecursor("busy.png");                                   //  cursor for function busy
   if (! busycursor)
      busycursor = gdk_cursor_new(GDK_WATCH);
   
   gtk_init_add((GtkFunction) gtkinitfunc,0);                              //  set initz. call from gtk_main()
   gtk_main();                                                             //  start processing window events
   return 0;
}


/**************************************************************************/

//  initial function called from gtk_main() at startup

int gtkinitfunc(void *data)
{
   int            err, flag, npid, ftype;
   char           procfile[20], *ppv;
   cchar          *pp, *pp2;
   struct stat    statb;
   cchar          *initz_message = ZTX("Run Tools > Synchronize Files so that gallery windows \n"
                                       "will be fast and Search Images will work correctly.");
   
   menulock(1);                                                            //  block menus until initz. done   v.11.04

   Badd = ZTX("Add");   
   Baddall = ZTX("Add All");
   Bamount = ZTX("Amount");
   Bapply = ZTX("Apply");
   Bblendwidth = ZTX("Blend Width");
   Bblue = ZTX("Blue");
   Bbrightness = ZTX("Brightness");
   Bbrowse = ZTX("Browse");
   Bcancel = ZTX("Cancel");
   Bclear = ZTX("Clear");
   Bcolor = ZTX("Color");
   Bcommit = ZTX("Commit");
   Bcopy = ZTX("Copy");
   Bcut = ZTX("Cut");
   Bdarker = ZTX("Darker Areas");
   Bdelete = ZTX("Delete");
   Bdisable = ZTX("Disable");
   Bdone = ZTX("Done");
   Bedit = ZTX("Edit");
   Benable = ZTX("Enable");
   Berase = ZTX("Erase");
   Bexiftoolmissing = ZTX("package libimage-exiftool-perl is required");
   Bfetch = ZTX("Fetch");
   Bfinish = ZTX("Finish");
   Bfont = ZTX("Font");
   Bgreen = ZTX("Green");
   Bheight = ZTX("Height");
   Bhide = ZTX("Hide");
   Binsert = ZTX("Insert");
   Binvert = ZTX("Invert");
   Blighter = ZTX("Lighter Areas");
   Blimit = ZTX("limit");
   BmyMouse = ZTX("my mouse");
   Bnext = ZTX("Next");
   BOK = ZTX("OK");
   Bopen = ZTX("Open");
   Bopenrawfile = ZTX("Open RAW File");
   Bpaste = ZTX("Paste");
   Bpause = ZTX("Pause");
   Bpercent = ZTX("Percent");
   Bpresets = ZTX("Presets");
   Bproceed = ZTX("Proceed");
   Bradius = ZTX("Radius");
   Brange = ZTX("range");
   Bred = ZTX("Red");
   Bredo = ZTX("Redo");
   Breduce = ZTX("Reduce");
   Bremove = ZTX("Remove");
   Bsave = ZTX("Save");
   Bsavetoedit = ZTX("Unknown file type, save as tiff/jpeg/png to edit");
   Bsearch = ZTX("Search");
   Bselect = ZTX("Select");
   Bselectfiles = ZTX("Select Files");
   Bshow = ZTX("Show");
   Bstart = ZTX("Start");
   Bthresh = ZTX("Threshold");
   Bundoall = ZTX("Undo All");
   Bundolast = ZTX("Undo Last");
   Bundo = ZTX("Undo");
   Bwidth = ZTX("Width");

   tid_fmain = pthread_self();                                             //  get main() thread ID
   snprintf(PIDstring,11,"%06d",getpid());                                 //  get fotoxx process PID 6-digits
   
   printf(fversion "\n");                                                  //  fotoxx version
   printf("install location: %s \n",get_zinstalloc());                     //  install location       v.11.05

   Nwt = sysconf(_SC_NPROCESSORS_ONLN);                                    //  get SMP CPU count
   if (Nwt <= 0) Nwt = 1;
   if (Nwt > max_threads) Nwt = max_threads;                               //  compile time limit
   printf("using %d threads \n",Nwt);

   err = system("which exiftool");                                         //  check for exiftool     v.11.05
   if (! err) Fexiftool = 1;
   err = system("which xdg-open");                                         //  check for xdg-open
   if (! err) Fxdgopen = 1;
   err = system("which ufraw-batch");                                      //  check for ufraw-batch
   if (! err) Fufraw = 1;
   
   undo_files = zmalloc(200,"undofiles");                                  //  look for orphaned undo files
   *undo_files = 0;
   strncatv(undo_files,199,get_zuserdir(),"/*_undo_*",null);               //  /home/user/.fotoxx/*_undo_*
   flag = 1;
   while ((pp = SearchWild(undo_files,flag)))
   {
      pp2 = strstr(pp,".fotoxx/");
      if (! pp2) continue;
      npid = atoi(pp2+8);                                                  //  pid of file owner
      snprintf(procfile,19,"/proc/%d",npid);
      err = stat(procfile,&statb);
      if (! err) continue;                                                 //  pid is active, keep file
      printf("orphaned undo file deleted: %s \n",pp);
      err = remove(pp);                                                    //  delete file         v.11.03
   }

   *undo_files = 0;                                                        //  setup undo filespec template
   strncatv(undo_files,199,get_zuserdir(),"/",PIDstring,"_undo_nn",null);  //  /home/user/.fotoxx/pppppp_undo_nn

   if (! Fsyncfiles) {                                                     //  ask user to run initialization function
      printf("Synchronize Files is needed \n");
      zmessage_help(mWin,"sync_files",initz_message);                      //  v.11.04
   }

   tags_defined_file = zmalloc(200,"tagsdefinedfile");                     //  tags defined file      v.10.5
   *tags_defined_file = 0;
   strncatv(tags_defined_file,199,get_zuserdir(),"/tags_defined",null);    //  /home/user/.fotoxx/tags_defined

   search_index_file = zmalloc(200,"searchindexfile");                     //  search index file      v.11.02
   *search_index_file = 0;
   strncatv(search_index_file,199,get_zuserdir(),"/search_index",null);    //  /home/user/.fotoxx/search_index
   
   err = stat(search_index_file,&statb);                                   //  is search index file present?
   if (err) {
      printf("search index file is missing \n");
      if (Fsyncfiles)                                                      //  don't repeat if already done
         zmessage_help(mWin,"sync_files",initz_message);                   //  v.11.04
   }

   editlog = pvlist_create(maxedits);                                      //  history log of image edits done
   for (int ii = 0; ii < maxedits; ii++)                                   //  pre-load all pvlist entries     v.10.2
      pvlist_append(editlog,"nothing");

   saved_areas_dirk = zmalloc(200,"selectareadirk");                       //  directory for saved areas       v.9.9
   *saved_areas_dirk = 0;
   strncatv(saved_areas_dirk,199,get_zuserdir(),"/saved_areas/",null);     //  /home/user/.fotoxx/saved_areas/
   err = stat(saved_areas_dirk,&statb);
   if (err) mkdir(saved_areas_dirk,0750);

   collections_dirk = zmalloc(200,"collectionsdirk");                      //  directory for saved collections    v.11.02
   *collections_dirk = 0;
   strncatv(collections_dirk,199,get_zuserdir(),"/collections/",null);     //  /home/user/.fotoxx/collections/
   err = stat(collections_dirk,&statb);
   if (err) mkdir(collections_dirk,0750);

   saved_curves_dirk = zmalloc(200,"curvedatadirk");                       //  directory for saved curves      v.11.02
   *saved_curves_dirk = 0;
   strncatv(saved_curves_dirk,199,get_zuserdir(),"/saved_curves/",null);   //  /home/user/.fotoxx/saved_curves/
   err = stat(saved_curves_dirk,&statb);
   if (err) mkdir(saved_curves_dirk,0750);

   annotations_dirk = zmalloc(200,"annotationsdirk");                      //  directory for annotations       v.11.03
   *annotations_dirk = 0;
   strncatv(annotations_dirk,199,get_zuserdir(),"/annotations/",null);     //  /home/user/.fotoxx/annotations/
   err = stat(annotations_dirk,&statb);
   if (err) mkdir(annotations_dirk,0750);

   mutex_init(&Fpixmap_lock,0);                                            //  setup lock for edit pixmaps

   TIFFSetWarningHandler(tiffwarninghandler);                              //  intercept TIFF warning messages
   
   g_timeout_add(100,gtimefunc,0);                                         //  start periodic function (100 ms)  v.11.03

//  set up current file and directory from command line input or last session parameters        v.11.04

   ppv = get_current_dir_name();                                           //  save current directory
   if (ppv) {
      curr_dirk = strdupz(ppv,0,"curr_dirk");
      free(ppv);
      err = chdir(curr_dirk);
   }
   else curr_dirk = 0;
   
   if (last_file) {                                                        //  from command line or parameters
      ppv = realpath(last_file,0);                                         //  prepend directory if needed
      if (ppv) {
         curr_file = strdupz(ppv,0,"curr_file");
         free(ppv);
      }
      else {
         printf("invalid file: %s \n",last_file);                          //  invalid file
         zfree(last_file);
         last_file = 0;
      }
   }

   if (curr_file) {
      ftype = image_file_type(curr_file);
      if (ftype == 0) {                                                    //  invalid
         printf("invalid file: %s \n",curr_file);
         zfree(curr_file);
         curr_file = 0;
      }
      else if (ftype == 1) {                                               //  directory
         if (curr_dirk) zfree(curr_dirk);
         curr_dirk = curr_file;
         curr_file = 0;
      }
      else if (ftype == 2) {                                               //  supported image file type
         if (curr_dirk) zfree(curr_dirk);
         curr_dirk = strdupz(curr_file,0,"curr_dirk");                     //  set current directory from file
         ppv = strrchr(curr_dirk,'/');
         if (ppv) *ppv = 0;
      }
      else {
         printf("unsupported file type: %s \n",curr_file);                 //  unsupported file
         zfree(curr_file);
         curr_file = 0;
      }
   }

   if (curr_dirk) err = chdir(curr_dirk);                                  //  set current directory
   
   menulock(0);                                                            //  enable menus                 v.11.04
   
   if (SS_imagefile) {                                                     //  start slide show if req.     v.11.04
      f_open(SS_imagefile,1);
      if (SS_musicfile) {
         sprintf(command,"xdg-open \"%s\" ",SS_musicfile);
         printf("command: %s \n",command);
         err = system(command);
      }
      m_slideshow(0,0);
      return 0;
   }

   if (curr_file) f_open(curr_file,1);                                     //  open current file
   else if (curr_dirk) m_gallery(0,0);                                     //  show directory gallery       v.11.05

   return 0;
}


/**************************************************************************/

//  Periodic function - runs every 0.1 seconds.
//  Avoid any thread usage of gtk/gdk functions.

int gtimefunc(void *arg)
{
   static double     secs;
   static int        fbusy = 0;
   static zdialog    *zdmessage = 0;

   if (Wrepaint) {                                                         //  update drawing window
      Wrepaint = 0;
      mwpaint();
   }

   update_statusbar();                                                     //  update every cycle       v.11.03
   
   if (Fslideshow) slideshow_next("timer");                                //  show next image if time is up

   if (Ffuncbusy && ! fbusy) {                                             //  function busy and default cursor  v.10.12
      secs = get_seconds();                                                //  start delay timer
      fbusy = 1;
   }

   if (Ffuncbusy && fbusy == 1) {
      if (get_seconds() - secs > 1.0) {                                    //  busy since 1 second
         fbusy = 2;
         gdk_window_set_cursor(mWin->window,busycursor);                   //  set main window hourglass cursor
         if (zdedit) zdialog_set_cursor(zdedit,busycursor);                //  and also for dialog box
      }
   }
   
   if (! Ffuncbusy && fbusy) {                                             //  no function busy
      if (fbusy == 2) {                                                    //  if hourglass cursor
         gdk_window_set_cursor(mWin->window,0);                            //  restore normal cursors       v.10.12
         if (zdedit) zdialog_set_cursor(zdedit,0);
      }
      fbusy = 0;                                                           //  reset toggle
   }
   
   if (threadmessage && ! zdmessage)                                       //  display message in behalf of thread process
      zdmessage = zmessage_post(mWin,0,threadmessage);                     //  (avoid use of GTK in threads)      v.11.03

   if (zdmessage && ! threadmessage)
      zdialog_free(zdmessage);
   
   if (zdmessage && zdmessage->sentinel != zdsentinel) {
      zdmessage = 0;
      threadmessage = 0;
   }

   return 1;
}


/**************************************************************************/

//  update status bar with image data and status
//  called from timer function

void update_statusbar()
{
   static double  time1 = 0, time2, cpu1 = 0, cpu2, cpuload;
   char           text1[300], text2[100];
   int            ii, ww, hh, scale, bpc, done;
   double         file_MB = 1.0 / mega * curr_file_size;
   
   *text1 = *text2 = 0;
   
   time2 = get_seconds();                                                  //  compute process cpu load     v.11.03

   if (time2 - time1 > 0.5) {                                              //  reduce jitter
      cpu2 = CPUtime();
      cpuload = 100.0 * (cpu2 - cpu1) / (time2 - time1);
      time1 = time2;
      cpu1 = cpu2;
   }

   sprintf(text1,"CPU %03.0f%c",cpuload,'%');                              //  CPU 023%
   
   if (curr_file)                                                          //  v.11.04
   {
      if (E3pxm16) {
         ww = E3ww;
         hh = E3hh;
         bpc = 16;
      }
      else if (Fpxm16) {
         ww = Fww;
         hh = Fhh;
         bpc = 16;
      }
      else {
         ww = Fww;
         hh = Fhh;
         bpc = curr_file_bpc;
      }

      snprintf(text2,99,"  %dx%dx%d",ww,hh,bpc);                           //  2345x1234x16 (preview) 1.56MB 45%
      strcat(text1,text2);
      if (Fpreview) strcat(text1," (preview)");
      sprintf(text2," %.2fMB",file_MB);
      strcat(text1,text2);
      scale = int(Mscale * 100 + 0.5);
      sprintf(text2," %d%c",scale,'%');
      strcat(text1,text2);
   
      if (Pundo)                                                           //  edit undo stack depth 
      {
         snprintf(text2,99,"  edits: %d",Pundo);
         strcat(text1,text2);
      }
   }

   if (Fmenulock) strcat(text1,"  menu locked");
   if (Factivearea) strcat(text1,"  area active");                         //  v.11.01
   
   if (SB_goal) {                                                          //  progress monitor       v.9.6
      for (ii = done = 0; ii < 8; ii++) 
         done += SB_done[ii];                                              //  counter per thread     v.11.03
      done = 100.0 * done / SB_goal;
      snprintf(text2,99,"  progress %d%c",done,'%');
      strcat(text1,text2);
   }

   if (*SB_text) {                                                         //  application text       v.10.7
      strcat(text1,"  ");
      strcat(text1,SB_text);
   }

   stbar_message(STbar,text1);

   return;
}


/**************************************************************************/

//  functions for main window event signals

int delete_event()                                                         //  main window closed
{
   printf("main window delete event \n");
   if (mod_keep()) return 1;                                               //  allow user bailout
   Fshutdown++;                                                            //  shutdown in progress
   save_params();                                                          //  save state for next session
   free_resources();                                                       //  delete undo files
   return 0;
}

void destroy_event()                                                       //  main window destroyed
{
   printf("main window destroy event \n");
   exit(1);                                                                //  instead of gtk_main_quit();
   return;
}


//  process top-level menu entry

void topmenufunc(GtkWidget *, cchar *menu)
{
   topmenu = (char *) menu;                                                //  remember top-level menu in case
   return;                                                                 //    this is needed somewhere
}


/**************************************************************************/

//  Cause (modified) output image to get repainted immediately.
//  The entire image is repainted (that part within the window).
//  This function may be called from threads.
//  (window paint is done from periodic function in main() thread)

void mwpaint2()
{
   Irefresh++;
   Wrepaint++;                                                             //  req. repaint by periodic function
   return;
}


//  Repaint a rectangular part of the image being edited.
//  Called from edit functions. Not thread callable.
//
//  px3, py3, ww3, hh3: modified area within E3pxm16 to be repainted
//  Dpxm16: 1x copy of E3pxm16 area currently visible in main window
//  px2, py2, ww2, hh2: E3pxm16 area copied into Dpxm16
//  Dpxm8: window image, Dpxm16 scaled to window, converted to 8 bits

void mwpaint3(int px3, int py3, int ww3, int hh3)                          //  overhauled v.10.11
{
   int         px2, py2, ww2, hh2, dx, dy;
   uint16      *pix2, *pix3;
   uint8       *pxm8area;
   
   if (! Dpxm16) zappcrash("mwpaint3() no Dpxm16");                        //  v.10.12
   
   px2 = px3 - Iorgx;                                                      //  map modified area into Dpxm16
   py2 = py3 - Iorgy;
   ww2 = ww3;
   hh2 = hh3;

   if (px2 < 0) {                                                          //  if beyond Dpxm16 edge, reduce
      px3 -= px2;
      ww2 += px2;
      px2 = 0;
   }
   if (px2 + ww2 > iww) ww2 = iww - px2;

   if (py2 < 0) {
      py3 -= py2;
      hh2 += py2;
      py2 = 0;
   }
   if (py2 + hh2 > ihh) hh2 = ihh - py2;

   if (ww2 <= 0 || hh2 <= 0) return;                                       //  completely off Dpxm16 image

   for (dy = 0; dy < hh2; dy++)                                            //  copy pixels from E3pxm16 to Dpxm16
   for (dx = 0; dx < ww2; dx++)
   {
      pix3 = PXMpix(E3pxm16,px3+dx,py3+dy);
      pix2 = PXMpix(Dpxm16,px2+dx,py2+dy);
      pix2[0] = pix3[0];
      pix2[1] = pix3[1];
      pix2[2] = pix3[2];
   }

   PXM_update(Dpxm16,Dpxm8,px2,py2,ww2,hh2);                               //  copy/rescale Dpxm16 area into Dpxm8 area

   px2 = px2 * Mscale - 1;                                                 //  Dpxm8 area impacted   v.10.12
   py2 = py2 * Mscale - 1;
   ww2 = ww2 * Mscale + 1 / Mscale + 2;
   hh2 = hh2 * Mscale + 1 / Mscale + 2;
   
   if (px2 < 0) px2 = 0;                                                   //  stay within image edge
   if (py2 < 0) py2 = 0;
   if (px2 + ww2 > dww) ww2 = dww - px2;
   if (py2 + hh2 > dhh) hh2 = dhh - py2;
   if (ww2 <= 0 || hh2 <= 0) return;                                       //  completely off Dpxm8 image

   pxm8area = (uint8 *) Dpxm8->bmp + 3 * (dww * py2 + px2);                //  origin of area to copy

   gdk_draw_rgb_image(drWin->window, gdkgc, px2 + Dorgx, py2 + Dorgy,      //  copy into window
                                  ww2, hh2, NODITHER, pxm8area, dww*3);
   return;
}


/**************************************************************************/

//  paint window when created, exposed, resized, or modified (edited)

int mwpaint()
{
   GdkRectangle   wrect;
   int            incrx, incry;                                            //  mouse drag
   static int     pincrx = 0, pincry = 0;                                  //  prior mouse drag
   static int     pdww = 0, pdhh = 0;                                      //  prior window size
   static int     piorgx = 0, piorgy = 0;                                  //  prior image origin in window
   static double  pscale = 1;                                              //  prior scale
   double         wscale, hscale;
   int            refresh = 0;                                             //  flag, image refesh needed
   PXM            *pxmtemp;

   void sa_show(int flag);                                                 //  show area function, defined later

   if (Fshutdown) return 1;                                                //  shutdown underway
   if (! Fpxm8) return 1;                                                  //  no image

   Dww = drWin->allocation.width;                                          //  (new) drawing window size
   Dhh = drWin->allocation.height;
   if (Dww < 20 || Dhh < 20) return 1;                                     //  too small

   if (mutex_trylock(&Fpixmap_lock) != 0) {                                //  lock pixmaps
      Wrepaint++;                                                          //  cannot, come back later
      return 1;
   }
   
   if (Irefresh) {                                                         //  image was updated   v.10.10.2
      refresh++;
      Irefresh = 0;
   }

   if (Dww != pdww || Dhh != pdhh) {                                       //  window size changed
      refresh++;                                                           //  image refresh needed
      pdww = Dww; 
      pdhh = Dhh;
   }

   if (E3pxm16) {                                                          //  get image size
      Iww = E3ww;
      Ihh = E3hh;                                                          //  edit in progress
   }
   else {
      Iww = Fww;                                                           //  no edit
      Ihh = Fhh;
   }

   if (Fzoom == 0) {                                                       //  scale to fit window
      wscale = 1.0 * Dww / Iww;
      hscale = 1.0 * Dhh / Ihh;
      if (wscale < hscale) Mscale = wscale;                                //  use greatest ww/hh ratio
      else  Mscale = hscale;
      if (Iww < Dww && Ihh < Dhh && ! Fblowup) Mscale = 1.0;               //  small image 1x unless Fblowup
      zoomx = zoomy = 0;
   }
   else Mscale = Fzoom;                                                    //  scale to Fzoom level
   
   if (Mscale != pscale) refresh++;                                        //  scale changed, image refresh needed

   if (Mscale > pscale) {                                                  //  zoom increased
      Iorgx += iww * 0.5 * (1.0 - pscale / Mscale);                        //  keep current image center
      Iorgy += ihh * 0.5 * (1.0 - pscale / Mscale);
   }
   pscale = Mscale;
   
   iww = Dww / Mscale;                                                     //  image space fitting in window
   if (iww > Iww) iww = Iww;
   ihh = Dhh / Mscale;
   if (ihh > Ihh) ihh = Ihh;

   if (zoomx || zoomy) {                                                   //  req. zoom center
      Iorgx = zoomx - 0.5 * iww;                                           //  corresp. image origin
      Iorgy = zoomy - 0.5 * ihh;
      zoomx = zoomy = 0;
   }

   if ((Mxdrag || Mydrag) && ! Mcapture) {                                 //  scroll via mouse drag
      incrx = (Mxdrag - Mxdown) * 1.3 * Iww / iww;                         //  scale
      incry = (Mydrag - Mydown) * 1.3 * Ihh / ihh;
      if (pincrx > 0 && incrx < 0) incrx = 0;                              //  stop bounce at extremes
      if (pincrx < 0 && incrx > 0) incrx = 0;
      pincrx = incrx;
      if (pincry > 0 && incry < 0) incry = 0;
      if (pincry < 0 && incry > 0) incry = 0;
      pincry = incry;
      Iorgx += incrx;                                                      //  new image origin after scroll
      Iorgy += incry;
      Mxdown = Mxdrag + incrx;                                             //  new drag origin
      Mydown = Mydrag + incry;
      Mxdrag = Mydrag = 0;
   }

   if (iww == Iww) {                                                       //  scaled image <= window width
      Iorgx = 0;                                                           //  center image in window
      Dorgx = 0.5 * (Dww - Iww * Mscale);
   }
   else Dorgx = 0;                                                         //  image > window, use entire window

   if (ihh == Ihh) {                                                       //  same for image height
      Iorgy = 0;
      Dorgy = 0.5 * (Dhh - Ihh * Mscale);
   }
   else Dorgy = 0;
   
   if (Iorgx + iww > Iww) Iorgx = Iww - iww;                               //  set limits
   if (Iorgy + ihh > Ihh) Iorgy = Ihh - ihh;
   if (Iorgx < 0) Iorgx = 0;
   if (Iorgy < 0) Iorgy = 0;
   
   if (Iorgx != piorgx || Iorgy != piorgy) {                               //  image origin changed
      refresh++;                                                           //  image refresh needed
      piorgx = Iorgx;
      piorgy = Iorgy;
   }
   
   if (refresh)                                                            //  image refresh is needed  v.10.10.2
   {
      if (E3pxm16) {                                                       //  edit in progress
         PXM_free(Dpxm16);
         Dpxm16 = PXM_copy_area(E3pxm16,Iorgx,Iorgy,iww,ihh);              //  copy E3 image area, PXM-16
         pxmtemp = PXM_convbpc(Dpxm16);                                    //  convert to PXM-8
      }
      else pxmtemp = PXM_copy_area(Fpxm8,Iorgx,Iorgy,iww,ihh);             //  no edit, copy PXM-8 image area

      dww = iww * Mscale;                                                  //  scale to window
      dhh = ihh * Mscale;
      PXM_free(Dpxm8);
      Dpxm8 = PXM_rescale(pxmtemp,dww,dhh);
      PXM_free(pxmtemp);
   }
   
   wrect.x = wrect.y = 0;                                                  //  stop flicker
   wrect.width = Dww;
   wrect.height = Dhh;
   gdk_window_begin_paint_rect(drWin->window,&wrect);

   gdk_draw_rgb_image(drWin->window, gdkgc, Dorgx, Dorgy, dww, dhh,        //  draw scaled image to window
                      NODITHER, (uint8 *) Dpxm8->bmp, dww*3);

   if (Ntoplines) paint_toplines(1);                                       //  draw line overlays
   if (Ftoparc) paint_toparc(1);                                           //  draw arc overlay
   if (Fgrid[0] || Fgrid[1]) paint_gridlines();                            //  draw grid lines           v.10.3.1
   if (Fshowarea && ! Fpreview) sa_show(1);                                //  draw select area outline  v.9.7

   gdk_window_end_paint(drWin->window);                                    //  release all window updates

   mutex_unlock(&Fpixmap_lock);                                            //  unlock pixmaps
   Wpainted++;                                                             //  notify edit function of repaint
   histogram_paint();                                                      //  update brightness histogram if exists
   return 1;
}


/**************************************************************************/

//  mouse event function - capture buttons and drag movements

void mouse_event(GtkWidget *, GdkEventButton *event, void *)
{
   void mouse_convert(int &xpos, int &ypos);

   static int     bdtime = 0, butime = 0, mbusy = 0;
   int            button, time, type;

   type = event->type;
   button = event->button;                                                 //  button, 1/3 = left/right
   time = event->time;
   Mxposn = int(event->x);                                                 //  mouse position in window
   Myposn = int(event->y);

   mouse_convert(Mxposn,Myposn);                                           //  convert to image space

   if (type == GDK_MOTION_NOTIFY) {
      if (mbusy) return;                                                   //  discard excess motion events
      mbusy++;
      zmainloop();
      mbusy = 0;
   }

   if (type == GDK_BUTTON_PRESS) {                                         //  button down
      bdtime = time;                                                       //  time of button down
      Mxdown = Mxposn;                                                     //  position at button down time
      Mydown = Myposn;
      if (button) {
         Mdrag++;                                                          //  possible drag start
         Mbutton = button;
      }
      Mxdrag = Mydrag = 0;
   }

   if (type == GDK_BUTTON_RELEASE) {                                       //  button up
      Mxclick = Myclick  = 0;                                              //  reset click status
      butime = time;                                                       //  time of button up
      if (butime - bdtime < 400)                                           //  less than 0.4 secs down
         if (Mxposn == Mxdown && Myposn == Mydown) {                       //       and not moving
            if (Mbutton == 1) LMclick++;                                   //  left mouse click
            if (Mbutton == 3) RMclick++;                                   //  right mouse click
            Mxclick = Mxdown;                                              //  click = button down position
            Myclick = Mydown;
         }
      Mxdown = Mydown = Mxdrag = Mydrag = Mdrag = Mbutton = 0;             //  forget buttons and drag
   }
   
   if (type == GDK_MOTION_NOTIFY && Mdrag) {                               //  drag underway
      Mxdrag = Mxposn;
      Mydrag = Myposn;
   }
   
   if (mouseCBfunc) {                                                      //  pass to handler function
      if (mbusy) return;
      mbusy++;                                                             //  stop re-entrance       v.10.8
      (* mouseCBfunc)();
      mbusy = 0;
      return;
   }

   if (LMclick && ! Mcapture) {                                            //  left click = zoom request
      LMclick = 0;
      zoomx = Mxclick;                                                     //  zoom center = mouse
      zoomy = Myclick;
      m_zoom(null, (char *) "+");
   }

   if (RMclick && ! Mcapture) {                                            //  right click = reset zoom
      RMclick = 0;
      zoomx = zoomy = 0;
      m_zoom(null, (char *) "-");
   }

   if ((Mxdrag || Mydrag) && ! Mcapture) mwpaint();                        //  drag = scroll
   return;
}


//  convert mouse position from window space to image space

void mouse_convert(int &xpos, int &ypos)
{
   xpos = int((xpos - Dorgx) / Mscale + Iorgx + 0.5);
   ypos = int((ypos - Dorgy) / Mscale + Iorgy + 0.5);

   if (xpos < 0) xpos = 0;                                                 //  if outside image put at edge
   if (ypos < 0) ypos = 0;

   if (E3pxm16) { 
      if (xpos >= E3ww) xpos = E3ww-1;
      if (ypos >= E3hh) ypos = E3hh-1;
   }
   else {
      if (xpos >= Fww) xpos = Fww-1;
      if (ypos >= Fhh) ypos = Fhh-1;
   }
   
   return;
}


/**************************************************************************/

//  keyboard event function - some toolbar buttons have KB equivalents
//  GDK key symbols: /usr/include/gtk-2.0/gdk/gdkkeysyms.h

int   KBcontrolkey = 0;
int   KBzmalloclog = 0;

int KBpress(GtkWidget *win, GdkEventKey *event, void *)                    //  prevent propagation of key-press
{                                                                          //    events to toolbar buttons
   KBkey = event->keyval;
   if (KBkey == 65507) KBcontrolkey = 1;                                   //  Ctrl key is pressed
   return 1;
}

int KBrelease(GtkWidget *win, GdkEventKey *event, void *)
{
   int      angle = 0;
   PXM      *pxm;
   
   KBkey = event->keyval;

   if (KBcapture) return 1;                                                //  let function handle it

   if (KBkey == 65507) KBcontrolkey = 0;                                   //  Ctrk key released
   
   if (KBcontrolkey) {
      if (KBkey == GDK_s) m_save(0,0);                                     //  Ctrl-* shortcuts
      if (KBkey == GDK_S) m_saveas(0,0);  
      if (KBkey == GDK_q) m_quit(0,0);  
      if (KBkey == GDK_Q) m_quit(0,0);
   }

   if (KBkey == GDK_G || KBkey == GDK_g) {                                 //  grid lines on/off      v.10.11
      Fgrid[0] = 1 - Fgrid[0];
      Fgrid[1] = 1 - Fgrid[1];
      mwpaint2();
   }

   if (KBkey == GDK_Left) {                                                //  arrow keys  >>  prev/next image
      if (Fslideshow) slideshow_next("prev");                              //  slide show function    v.11.01
      else m_prev(0,0);
   }

   if (KBkey == GDK_Right) {
      if (Fslideshow) slideshow_next("next");
      else m_next(0,0);
   }

   if (KBkey == GDK_plus) m_zoom(null, (char *) "+");                      //  +/- keys  >>  zoom in/out
   if (KBkey == GDK_equal) m_zoom(null, (char *) "+");                     //  = key: same as +
   if (KBkey == GDK_minus) m_zoom(null, (char *) "-");
   if (KBkey == GDK_KP_Add) m_zoom(null, (char *) "+");                    //  keypad +
   if (KBkey == GDK_KP_Subtract) m_zoom(null, (char *) "-");               //  keypad -
   
   if (KBkey == GDK_Z) m_zoom(null, (char *) "Z");                         //  Z key: zoom to 100%
   if (KBkey == GDK_z) m_zoom(null, (char *) "Z");

   if (KBkey == GDK_Escape)                                                //  escape
      if (Fslideshow) m_slideshow(0,0);                                    //  exit slideshow mode

   if (KBkey == GDK_Delete) m_trash(0,0);                                  //  delete  >>  trash

   if (KBkey == GDK_F1)                                                    //  F1  >>  user guide
      showz_userguide(zfuncs::F1_help_topic);                              //  show topic if there, or page 1

   if (! E1pxm16) {                                                        //  if no edit in progress,
      if (KBkey == GDK_R || KBkey == GDK_r) angle = 90;                    //    allow manual rotations
      if (KBkey == GDK_L || KBkey == GDK_l) angle = -90;
      if (angle) {
         mutex_lock(&Fpixmap_lock);                                        //  v.10.5
         pxm = PXM_rotate(Fpxm8,angle);
         PXM_free(Fpxm8);
         Fpxm8 = pxm;
         Fww = pxm->ww;
         Fhh = pxm->hh;
         mutex_unlock(&Fpixmap_lock);
         mwpaint2();
         m_save(0,0);                                                      //  auto save uprighted image   v.10.12
      }
   }

   if (KBcontrolkey && KBkey == GDK_m) {                                   //  zmalloc() activity log
      KBzmalloclog = 1 - KBzmalloclog;                                     //  Ctrl+M = toggle on/off
      if (KBzmalloclog) zmalloc_log(999999);
      else zmalloc_log(0);
      zmalloc_report();                                                    //  v.10.8
   }
   
   if (KBkey == GDK_T || KBkey == GDK_t) m_trim(0,0);                      //  Key T = Trim function        v.10.3.1
   if (KBkey == GDK_B || KBkey == GDK_b) m_tune(0,0);                      //  Key B = brightness/color     v.10.3.1
   
   KBkey = 0;
   return 1;
}


/**************************************************************************/

//  paint a grid of horizontal and vertical lines

void paint_gridlines()                                                     //  revised  v.10.10.2
{
   int      px, py, stepx, stepy;
   
   if (! Fpxm8) return;                                                    //  no image
   
   stepx = gridspace[0];                                                   //  space between grid lines
   stepy = gridspace[1];

   if (gridcount[0]) stepx = dww / (1 + gridcount[0]);                     //  if line counts specified,    v.10.11
   if (gridcount[1]) stepy = dhh / ( 1 + gridcount[1]);                    //    set spacing accordingly
   
   gdk_gc_set_foreground(gdkgc,&white);                                    //  white lines

   if (Fgrid[0])
      for (px = Dorgx+stepx; px < Dorgx+dww; px += stepx)
         gdk_draw_line(drWin->window,gdkgc,px,Dorgy,px,Dorgy+dhh);

   if (Fgrid[1])
      for (py = Dorgy+stepy; py < Dorgy+dhh; py += stepy)
         gdk_draw_line(drWin->window,gdkgc,Dorgx,py,Dorgx+dww,py);

   gdk_gc_set_foreground(gdkgc,&black);                                    //  adjacent black lines
   fg_color = black;                                                       //  v.10.12

   if (Fgrid[0]) 
      for (px = Dorgx+stepx+1; px < Dorgx+dww; px += stepx)
         gdk_draw_line(drWin->window,gdkgc,px,Dorgy,px,Dorgy+dhh);

   if (Fgrid[1])
      for (py = Dorgy+stepy+1; py < Dorgy+dhh; py += stepy)
         gdk_draw_line(drWin->window,gdkgc,Dorgx,py,Dorgx+dww,py);
   
   return;
}


/**************************************************************************/

//  refresh overlay lines on top of image
//  arg = 1:   paint lines only (because window repainted)
//        2:   erase lines and forget them
//        3:   erase old lines, paint new lines, save new in old

void paint_toplines(int arg)
{
   int      ii;

   if (arg == 2 || arg == 3)                                               //  erase old lines
      for (ii = 0; ii < Nptoplines; ii++)
         erase_line(ptoplinex1[ii],ptopliney1[ii],ptoplinex2[ii],ptopliney2[ii]);
   
   if (arg == 1 || arg == 3)                                               //  draw new lines
      for (ii = 0; ii < Ntoplines; ii++)
         draw_line(toplinex1[ii],topliney1[ii],toplinex2[ii],topliney2[ii]);

   if (arg == 2) {
      Nptoplines = Ntoplines = 0;                                          //  forget lines
      return;
   }

   for (ii = 0; ii < Ntoplines; ii++)                                      //  save for future erase
   {
      ptoplinex1[ii] = toplinex1[ii];
      ptopliney1[ii] = topliney1[ii];
      ptoplinex2[ii] = toplinex2[ii];
      ptopliney2[ii] = topliney2[ii];
   }

   Nptoplines = Ntoplines;

   return;
}


/**************************************************************************/

//  refresh overlay arc (circle/ellipse) on top of image
//  arg = 1:   paint arc only (because window repainted)
//        2:   erase arc and forget it
//        3:   erase old arc, paint new arc, save new in old

void paint_toparc(int arg)
{
   int      arcx, arcy, arcw, arch;

   if (ptoparc && (arg == 2 || arg == 3)) {                                //  erase old arc
      arcx = int((ptoparcx-Iorgx) * Mscale + Dorgx + 0.5);                 //  image to window space
      arcy = int((ptoparcy-Iorgy) * Mscale + Dorgy + 0.5);
      arcw = int(ptoparcw * Mscale);
      arch = int(ptoparch * Mscale);

      gdk_gc_set_function(gdkgc,GDK_INVERT);                               //  invert pixels
      gdk_draw_arc(drWin->window,gdkgc,0,arcx,arcy,arcw,arch,0,64*360);    //  draw arc
      gdk_gc_set_function(gdkgc,GDK_COPY);
   }
   
   if (Ftoparc && (arg == 1 || arg == 3)) {                                //  draw new arc
      arcx = int((toparcx-Iorgx) * Mscale + Dorgx + 0.5);                  //  image to window space
      arcy = int((toparcy-Iorgy) * Mscale + Dorgy + 0.5);
      arcw = int(toparcw * Mscale);
      arch = int(toparch * Mscale);

      gdk_gc_set_function(gdkgc,GDK_INVERT);                               //  invert pixels
      gdk_draw_arc(drWin->window,gdkgc,0,arcx,arcy,arcw,arch,0,64*360);    //  draw arc
      gdk_gc_set_function(gdkgc,GDK_COPY);
   }

   if (arg == 2) {
      Ftoparc = ptoparc = 0;                                               //  forget arcs
      return;
   }
   
   ptoparc = Ftoparc;                                                      //  save for future erase
   ptoparcx = toparcx;
   ptoparcy = toparcy;
   ptoparcw = toparcw;
   ptoparch = toparch;

   return;
}


/**************************************************************************/

//  draw dotted line. coordinates are in image space.

void draw_line(int ix1, int iy1, int ix2, int iy2)
{
   void draw_line_pixel(double pxm, double pym);

   double      x1, y1, x2, y2;   
   double      pxm, pym, slope;
   
   x1 = Mscale * (ix1-Iorgx);                                              //  image to window space
   y1 = Mscale * (iy1-Iorgy);
   x2 = Mscale * (ix2-Iorgx);
   y2 = Mscale * (iy2-Iorgy);
   
   if (abs(y2 - y1) > abs(x2 - x1)) {
      slope = 1.0 * (x2 - x1) / (y2 - y1);
      if (y2 > y1) {
         for (pym = y1; pym <= y2; pym++) {
            pxm = round(x1 + slope * (pym - y1));
            draw_line_pixel(pxm,pym);
         }
      }
      else {
         for (pym = y1; pym >= y2; pym--) {
            pxm = round(x1 + slope * (pym - y1));
            draw_line_pixel(pxm,pym);
         }
      }
   }
   else {
      slope = 1.0 * (y2 - y1) / (x2 - x1);
      if (x2 > x1) {
         for (pxm = x1; pxm <= x2; pxm++) {
            pym = round(y1 + slope * (pxm - x1));
            draw_line_pixel(pxm,pym);
         }
      }
      else {
         for (pxm = x1; pxm >= x2; pxm--) {
            pym = round(y1 + slope * (pxm - x1));
            draw_line_pixel(pxm,pym);
         }
      }
   }

   gdk_gc_set_foreground(gdkgc,&black);
   fg_color = black;

   return;
}

void draw_line_pixel(double px, double py)
{
   int            pxn, pyn;
   static int     flip = 0;
   
   pxn = int(px);
   pyn = int(py);
   
   if (pxn < 0 || pxn > dww-1) return;
   if (pyn < 0 || pyn > dhh-1) return;
   
   if (++flip > 4) flip = -5;                                              //  black/white line     v.10.10
   if (flip < 0) gdk_gc_set_foreground(gdkgc,&black);
   else gdk_gc_set_foreground(gdkgc,&white);
   gdk_draw_point(drWin->window, gdkgc, pxn + Dorgx, pyn + Dorgy);
   
   return;
}


//  erase line. refresh line path from Dpxm8 pixels.

void erase_line(int ix1, int iy1, int ix2, int iy2)
{
   void erase_line_pixel(double pxm, double pym);

   double      x1, y1, x2, y2;   
   double      pxm, pym, slope;
   
   if (! Dpxm8) zappcrash("Dpxm8 = 0");                                    //  v.10.3

   x1 = Mscale * (ix1-Iorgx);
   y1 = Mscale * (iy1-Iorgy);
   x2 = Mscale * (ix2-Iorgx);
   y2 = Mscale * (iy2-Iorgy);
   
   if (abs(y2 - y1) > abs(x2 - x1)) {
      slope = 1.0 * (x2 - x1) / (y2 - y1);
      if (y2 > y1) {
         for (pym = y1; pym <= y2; pym++) {
            pxm = x1 + slope * (pym - y1);
            erase_line_pixel(pxm,pym);
         }
      }
      else {
         for (pym = y1; pym >= y2; pym--) {
            pxm = x1 + slope * (pym - y1);
            erase_line_pixel(pxm,pym);
         }
      }
   }
   else {
      slope = 1.0 * (y2 - y1) / (x2 - x1);
      if (x2 > x1) {
         for (pxm = x1; pxm <= x2; pxm++) {
            pym = y1 + slope * (pxm - x1);
            erase_line_pixel(pxm,pym);
         }
      }
      else {
         for (pxm = x1; pxm >= x2; pxm--) {
            pym = y1 + slope * (pxm - x1);
            erase_line_pixel(pxm,pym);
         }
      }
   }

   return;
}

void erase_line_pixel(double px, double py)
{
   int            pxn, pyn;
   
   pxn = int(px);
   pyn = int(py);
   
   if (pxn < 0 || pxn > dww-1) return;
   if (pyn < 0 || pyn > dhh-1) return;

   uint8 *pixel = (uint8 *) Dpxm8->bmp + (pyn * dww + pxn) * 3;
   gdk_draw_rgb_image(drWin->window, gdkgc, pxn + Dorgx, pyn + Dorgy, 
                                 1, 1, NODITHER, pixel, dww * 3);         
   return;
}


//  draw one pixel using given color, R/G/B = red/green/black

void draw_pixel(int px, int py, GdkColor *color)                           //  v.10.12
{
   int            qx, qy;
   static int     pqx, pqy;
   
   qx = Mscale * (px-Iorgx) + 0.5;                                         //  image to window space
   qy = Mscale * (py-Iorgy) + 0.5;

   if (qx == pqx && qy == pqy) return;                                     //  avoid redundant points   v.9.7

   pqx = qx;
   pqy = qy;
   
   qx = qx + Dorgx;                                                        //  image origin in drawing window
   qy = qy + Dorgy;

   if (qx < 0 || qx > Dww-1) return;                                       //  bugfix v.11.03.1
   if (qy < 0 || qy > Dhh-1) return;

   if (color != &fg_color) {
      fg_color = *color;
      gdk_gc_set_foreground(gdkgc,&fg_color);
   }
   
   gdk_draw_point(drWin->window,gdkgc,qx,qy);                              //  draw pixel

   return;
}


//  draw a fat pixel using given color, R/G/B = red/green/black

void draw_fat_pixel(int px, int py, GdkColor *color)                       //  speedup    v.11.04
{
   int            qx, qy;
   static int     pqx, pqy;
   
   qx = Mscale * (px-Iorgx) + 0.5;                                         //  image to window space
   qy = Mscale * (py-Iorgy) + 0.5;

   if (qx == pqx && qy == pqy) return;                                     //  avoid redundant points   v.9.7

   pqx = qx;
   pqy = qy;
   
   qx = qx + Dorgx;                                                        //  image origin in drawing window
   qy = qy + Dorgy;
   
   if (qx < 0 || qy < 0) return;
   
   if (color != &fg_color) {
      fg_color = *color;
      gdk_gc_set_foreground(gdkgc,&fg_color);
   }
   
   if (qx < Dww-1 && qy < Dhh-1 && Mscale > 1.0)                     
      gdk_draw_rectangle(drWin->window,gdkgc,0,qx,qy,2,2);                 //  draw fat pixel 2x2

   else if (qx < Dww && qy < Dhh)
      gdk_draw_point(drWin->window,gdkgc,qx,qy);                           //  on edge, draw pixel

   return;
}


/**************************************************************************

   spline curve setup and edit functions

   Support multiple frames with curves in parallel                         //  v.11.01
   (edit curve(s) and image mask curve)

   Usage:
   Add frame widget to dialog
   Set up drawing in frame: sd = curve_init(frame, callback_func)
   Initialize no. of curves in frame (1-10): sd->Nspc = n
   Initialize vert/horz flag for curve cc: sd->vert[cc] = hv
   Initialize anchor points for curve cc: sd->nap[cc], sd->apx[cc][xx], sd->apy[cc][yy]
   Generate data for curve cc:  spccurve_generate(sd,cc)
   Curves will now be shown and edited inside the frame when window is realized
   The callback_func(cc) will be called when curve cc is edited
   Change curve in program: set anchor points, call spccurve_generate(sd, cc)
   Get y-value for curve cc and given x-value: yval = spccurve_yval(sd, cc, xval)

***/

//  initialize for spline curve editing
//  initial anchor points are pre-loaded into spcdat before window is realized

spcdat * spccurve_init(GtkWidget *frame, void func(int spc))
{
   int      cc = sizeof(spcdat);                                           //  allocate spc curve data area
   spcdat * sd = (spcdat *) zmalloc(cc,"spcdat");
   memset(sd,0,cc);

   sd->drawarea = gtk_drawing_area_new();                                  //  drawing area for curves
   gtk_container_add(GTK_CONTAINER(frame),sd->drawarea);
   sd->spcfunc = func;                                                     //  user callback function

   gtk_widget_add_events(sd->drawarea,GDK_BUTTON_PRESS_MASK);              //  connect mouse events to drawing area
   gtk_widget_add_events(sd->drawarea,GDK_BUTTON_RELEASE_MASK);
   gtk_widget_add_events(sd->drawarea,GDK_BUTTON1_MOTION_MASK); 
   G_SIGNAL(sd->drawarea,"motion-notify-event",spccurve_adjust,sd)
   G_SIGNAL(sd->drawarea,"button-press-event",spccurve_adjust,sd)
   G_SIGNAL(sd->drawarea,"expose-event",spccurve_draw,sd)
   
   return sd;
}


//  modify anchor points in curve using mouse

int spccurve_adjust(void *, GdkEventButton *event, spcdat *sd)
{
   int            ww, hh, kk;
   int            mx, my, button, evtype;
   static int     spc, ap, mbusy = 0, Fdrag = 0;                           //  drag continuation logic   v.9.7
   int            minspc, minap;
   double         mxval, myval, cxval, cyval;
   double         dist2, mindist2 = 0;
   
   mx = int(event->x);                                                     //  mouse position in drawing area
   my = int(event->y);
   evtype = event->type;
   button = event->button;

   if (evtype == GDK_MOTION_NOTIFY) {
      if (mbusy) return 0;                                                 //  discard excess motion events    v.11.01
      mbusy++;
      zmainloop();
      mbusy = 0;
   }

   if (evtype == GDK_BUTTON_RELEASE) {
      Fdrag = 0;
      return 0;
   }
   
   ww = sd->drawarea->allocation.width;                                    //  drawing area size
   hh = sd->drawarea->allocation.height;
   
   if (mx < 0) mx = 0;                                                     //  limit edge excursions
   if (mx > ww) mx = ww;
   if (my < 0) my = 0;
   if (my > hh) my = hh;
   
   if (evtype == GDK_BUTTON_PRESS) Fdrag = 0;                              //  left or right click
   
   if (Fdrag)                                                              //  continuation of drag
   {
      if (sd->vert[spc]) {
         mxval = 1.0 * my / hh;                                            //  mouse position in curve space
         myval = 1.0 * mx / ww;
      }
      else {
         mxval = 1.0 * mx / ww;
         myval = 1.0 * (hh - my) / hh;
      }

      if (ap < sd->nap[spc]-1 && sd->apx[spc][ap+1] - mxval < 0.05)        //  disallow < 0.05 from next
            return 0;
      if (ap > 0 && mxval - sd->apx[spc][ap-1] < 0.05) return 0;           //    or prior anchor point 
   }
   
   else                                                                    //  mouse click or new drag begin
   {
      minspc = minap = -1;                                                 //  find closest curve/anchor point
      mindist2 = 999999;

      for (spc = 0; spc < sd->Nspc; spc++)                                 //  loop curves
      {
         if (sd->vert[spc]) {
            mxval = 1.0 * my / hh;                                         //  mouse position in curve space
            myval = 1.0 * mx / ww;
         }
         else {
            mxval = 1.0 * mx / ww;
            myval = 1.0 * (hh - my) / hh;
         }

         for (ap = 0; ap < sd->nap[spc]; ap++)                             //  loop anchor points
         {
            cxval = sd->apx[spc][ap];
            cyval = sd->apy[spc][ap];
            dist2 = (mxval-cxval)*(mxval-cxval) 
                  + (myval-cyval)*(myval-cyval);
            if (dist2 < mindist2) {
               mindist2 = dist2;                                           //  remember closest anchor point
               minspc = spc;
               minap = ap;
            }
         }
      }

      if (minspc < 0) return 0;                                            //  impossible
      spc = minspc;
      ap = minap;
   }
   
   if (evtype == GDK_BUTTON_PRESS && button == 3)                          //  right click, remove anchor point
   {
      if (sqrt(mindist2) > 0.05) return 0;                                 //  not close enough
      if (sd->nap[spc] < 3) return 0;                                      //  < 2 anchor points would remain
      for (kk = ap; kk < sd->nap[spc]-1; kk++) {
         sd->apx[spc][kk] = sd->apx[spc][kk+1];
         sd->apy[spc][kk] = sd->apy[spc][kk+1];
      }
      sd->nap[spc]--;
      spccurve_generate(sd,spc);                                           //  regenerate data for modified curve
      spccurve_draw(0,0,sd);                                               //  regen and redraw all curves
      sd->spcfunc(spc);                                                    //  call user function
      return 0;
   }

   if (! Fdrag)                                                            //  new drag or left click
   {
      if (sd->vert[spc]) {
         mxval = 1.0 * my / hh;                                            //  mouse position in curve space
         myval = 1.0 * mx / ww;
      }
      else {
         mxval = 1.0 * mx / ww;
         myval = 1.0 * (hh - my) / hh;
      }

      if (sqrt(mindist2) < 0.05)                                           //  existing point close enough,
      {                                                                    //    move this anchor point to mouse
         if (ap < sd->nap[spc]-1 && sd->apx[spc][ap+1] - mxval < 0.05)
               return 0;                                                   //  disallow < 0.05 from next
         if (ap > 0 && mxval - sd->apx[spc][ap-1] < 0.05) return 0;        //    or prior anchor point 
      }
   
      else                                                                 //  none close, add new anchor point
      {
         minspc = -1;                                                      //  find closest curve to mouse
         mindist2 = 999999;

         for (spc = 0; spc < sd->Nspc; spc++)                              //  loop curves
         {
            if (sd->vert[spc]) {
               mxval = 1.0 * my / hh;                                      //  mouse position in curve space
               myval = 1.0 * mx / ww;
            }
            else {
               mxval = 1.0 * mx / ww;
               myval = 1.0 * (hh - my) / hh;
            }

            cyval = spccurve_yval(sd,spc,mxval);
            dist2 = fabs(myval - cyval);
            if (dist2 < mindist2) {
               mindist2 = dist2;                                           //  remember closest curve
               minspc = spc;
            }
         }

         if (minspc < 0) return 0;                                         //  impossible
         if (mindist2 > 0.05) return 0;                                    //  not close enough to any curve
         spc = minspc;

         if (sd->nap[spc] > 49) {
            zmessageACK(mWin,ZTX("Exceed 50 anchor points"));
            return 0;
         }

         if (sd->vert[spc]) {
            mxval = 1.0 * my / hh;                                         //  mouse position in curve space
            myval = 1.0 * mx / ww;
         }
         else {
            mxval = 1.0 * mx / ww;
            myval = 1.0 * (hh - my) / hh;
         }

         for (ap = 0; ap < sd->nap[spc]; ap++)                             //  find anchor point with next higher x
            if (mxval <= sd->apx[spc][ap]) break;                          //    (ap may come out 0 or nap)

         if (ap < sd->nap[spc] && sd->apx[spc][ap] - mxval < 0.05)         //  disallow < 0.05 from next
               return 0;                                                   //    or prior anchor point
         if (ap > 0 && mxval - sd->apx[spc][ap-1] < 0.05) return 0;

         for (kk = sd->nap[spc]; kk > ap; kk--) {                          //  make hole for new point
            sd->apx[spc][kk] = sd->apx[spc][kk-1];
            sd->apy[spc][kk] = sd->apy[spc][kk-1];
         }

         sd->nap[spc]++;                                                   //  up point count
      }
   }

   sd->apx[spc][ap] = mxval;                                               //  new or moved anchor point
   sd->apy[spc][ap] = myval;                                               //    at mouse position

   spccurve_generate(sd,spc);                                              //  regenerate data for modified curve
   spccurve_draw(0,0,sd);                                                  //  regen and redraw all curves
   sd->spcfunc(spc);                                                       //  call user function
   
   if (evtype == GDK_MOTION_NOTIFY) Fdrag = 1;                             //  remember drag is underway
   return 0;
}


//  draw all curves based on current anchor points
//  (refresh all curves in drawing area)

int spccurve_draw(void *, void *, spcdat *sd)
{
   int         ww, hh, px, py, qx, qy, spc, ap;
   double      xval, yval;
   
   ww = sd->drawarea->allocation.width;                                    //  drawing area size
   hh = sd->drawarea->allocation.height;
   if (ww < 50 || hh < 50) return 0;

   gdk_window_clear(sd->drawarea->window);                                 //  clear window
   gdk_gc_set_foreground(gdkgc,&black);
   fg_color = black;
   
   for (spc = 0; spc < sd->Nspc; spc++)
   {
      if (sd->vert[spc])                                                   //  vert. curve
      {
         for (py = 0; py < hh; py++)                                       //  generate all points for curve
         {
            xval = 1.0 * py / hh;                                          //  remove anchor point limits   v.10.9
            yval = spccurve_yval(sd,spc,xval);
            px = ww * yval + 0.49;                                         //  "almost" round - erratic floating point
            gdk_draw_point(sd->drawarea->window,gdkgc,px,py);              //    causes "bumps" in a flat curve
         }
         
         for (ap = 0; ap < sd->nap[spc]; ap++)                             //  draw boxes at anchor points
         {
            xval = sd->apx[spc][ap];
            yval = sd->apy[spc][ap];
            px = ww * yval;
            py = hh * xval;
            for (qx = -2; qx < 3; qx++)
            for (qy = -2; qy < 3; qy++) {
               if (px+qx < 0 || px+qx >= ww) continue;
               if (py+qy < 0 || py+qy >= hh) continue;
               gdk_draw_point(sd->drawarea->window,gdkgc,px+qx,py+qy);
            }
         }
      }
      else                                                                 //  horz. curve
      {
         for (px = 0; px < ww; px++)                                       //  generate all points for curve
         {
            xval = 1.0 * px / ww;                                          //  remove anchor point limits   v.10.9
            yval = spccurve_yval(sd,spc,xval);
            py = hh - hh * yval + 0.49;                                    //  almost round - erratic FP in Intel CPUs
            gdk_draw_point(sd->drawarea->window,gdkgc,px,py);              //    causes "bumps" in a flat curve
         }
         
         for (ap = 0; ap < sd->nap[spc]; ap++)                             //  draw boxes at anchor points
         {
            xval = sd->apx[spc][ap];
            yval = sd->apy[spc][ap];
            px = ww * xval;
            py = hh - hh * yval;
            for (qx = -2; qx < 3; qx++)
            for (qy = -2; qy < 3; qy++) {
               if (px+qx < 0 || px+qx >= ww) continue;
               if (py+qy < 0 || py+qy >= hh) continue;
               gdk_draw_point(sd->drawarea->window,gdkgc,px+qx,py+qy);
            }
         }
      }
   }

   return 0;
}


//  generate all curve data points when anchor points are modified

int spccurve_generate(spcdat *sd, int spc)
{
   int      kk, kklo, kkhi;
   double   xval, yvalx;

   spline1(sd->nap[spc],sd->apx[spc],sd->apy[spc]);                        //  compute curve fitting anchor points

   kklo = 1000 * sd->apx[spc][0] - 30;                                     //  xval range = anchor point range
   if (kklo < 0) kklo = 0;                                                 //    + 0.03 extra below/above      v.9.5
   kkhi = 1000 * sd->apx[spc][sd->nap[spc]-1] + 30;
   if (kkhi > 1000) kkhi = 1000;

   for (kk = 0; kk < 1000; kk++)                                           //  generate all points for curve
   {
      xval = 0.001 * kk;                                                   //  remove anchor point limits   v.10.9
      yvalx = spline2(xval);
      if (yvalx < 0) yvalx = 0;                                            //  yval < 0 not allowed, > 1 OK    v.9.5
      sd->yval[spc][kk] = yvalx;
   }

   return 0;
}


//  retrieve curve data using interpolation of saved table of values

double spccurve_yval(spcdat *sd, int spc, double xval)
{
   int      ii;
   double   x1, x2, y1, y2, y3;
   
   if (xval <= 0) return sd->yval[spc][0];
   if (xval >= 0.999) return sd->yval[spc][999];
   
   x2 = 1000.0 * xval;
   ii = int(x2);
   x1 = ii;
   y1 = sd->yval[spc][ii];
   y2 = sd->yval[spc][ii+1];
   y3 = y1 + (y2 - y1) * (x2 - x1);
   return y3;
}


//  load curve data from a file
//  returns 0 if fail (invalid file data), sd not modified
//  returns 1 if succcess, sd is initialized from file data

int spccurve_load(spcdat *sd)                                              //  v.11.02
{
   char        *pfile;
   int         nn, ii, jj;
   FILE        *fid = 0;
   int         Nspc, vert[10], nap[10];
   double      apx[10][50], apy[10][50];

   zfuncs::F1_help_topic = "curve_edit";

   pfile = zgetfile1(ZTX("load curve from a file"),"open",saved_curves_dirk);
   if (! pfile) return 0;

   fid = fopen(pfile,"r");
   zfree(pfile);
   if (! fid) goto fail;
   
   nn = fscanf(fid,"%d ",&Nspc);                                           //  no. of curves
   if (nn != 1) goto fail;
   if (Nspc < 1 || Nspc > 10) goto fail;
   if (Nspc != sd->Nspc) goto fail2;

   for (ii = 0; ii < Nspc; ii++)                                           //  loop each curve
   {
      nn = fscanf(fid,"%d  %d ",&vert[ii],&nap[ii]);                       //  vertical flag, no. anchor points
      if (nn != 2) goto fail;
      if (vert[ii] < 0 || vert[ii] > 1) goto fail;
      if (nap[ii] < 2 || nap[ii] > 50) goto fail;

      for (jj = 0; jj < nap[ii]; jj++)                                     //  anchor point values 
      {
         nn = fscanf(fid,"%lf/%lf  ",&apx[ii][jj],&apy[ii][jj]);
         if (nn != 2) goto fail;
         if (apx[ii][jj] < 0 || apx[ii][jj] > 1) goto fail;
         if (apy[ii][jj] < 0 || apy[ii][jj] > 1) goto fail;
      }
   }
   
   fclose(fid);
   
   sd->Nspc = Nspc;                                                        //  copy curve data to caller's arg

   for (ii = 0; ii < Nspc; ii++)
   {
      sd->vert[ii] = vert[ii];
      sd->nap[ii] = nap[ii];
      
      for (jj = 0; jj < nap[ii]; jj++)
      {
         sd->apx[ii][jj] = apx[ii][jj];
         sd->apy[ii][jj] = apy[ii][jj];
      }
   }

   for (ii = 0; ii < Nspc; ii++)                                           //  generate curve data from anchor points
      spccurve_generate(sd,ii);
   
   if (sd->drawarea) spccurve_draw(0,0,sd);                                //  regen and redraw all curves

   return 1;
   
fail:
   if (fid) fclose(fid);
   zmessageACK(mWin,ZTX("curve file is invalid"));
   return 0;

fail2:
   if (fid) fclose(fid);
   zmessageACK(mWin,ZTX("curve file has different no. of curves"));
   return 0;
}


//  save curve data to a file

int spccurve_save(spcdat *sd)                                              //  v.11.02
{
   char        *pfile, *pp;
   int         ii, jj;
   FILE        *fid;

   zfuncs::F1_help_topic = "curve_edit";

   pp = zgetfile1(ZTX("save curve to a file"),"save",saved_curves_dirk);
   if (! pp) return 0;

   pfile = strdupz(pp,8,"spccurve_save");
   zfree(pp);

   pp = strrchr(pfile,'/');                                                //  force .curve extension
   if (pp) pp = strrchr(pp,'.');
   if (pp) strcpy(pp,".curve");
   else strcat(pfile,".curve");

   fid = fopen(pfile,"w");
   zfree(pfile);
   if (! fid) return 0;
   
   fprintf(fid,"%d \n",sd->Nspc);                                          //  no. of curves

   for (ii = 0; ii < sd->Nspc; ii++)                                       //  loop each curve
   {
      fprintf(fid,"%d  %d \n",sd->vert[ii],sd->nap[ii]);                   //  vertical flag, no. anchor points
      for (jj = 0; jj < sd->nap[ii]; jj++)                                 //  anchor point values 
         fprintf(fid,"%.4f/%.4f  ",sd->apx[ii][jj],sd->apy[ii][jj]);
      fprintf(fid,"\n");
   }
   
   fclose(fid);
   return 0;
}


/**************************************************************************
      zdialog mouse capture and release
***************************************************************************/

void takeMouse(zdialog *zd, CBfunc func, GdkCursor *cursor)                //  capture mouse for dialog     v.11.03
{
   freeMouse();
   if (zd) zdialog_stuff(zd,"mymouse",1);                                  //  set zdialog radio button on
   mouse_zd = zd;
   mouseCBfunc = func;
   Mcapture = 1;
   gdk_window_set_cursor(drWin->window,cursor);                            //  v.11.03
   return;
}


void freeMouse()                                                           //  free mouse for main window   v.10.12
{
   if (mouse_zd) zdialog_stuff(mouse_zd,"mymouse",0);                      //  set radio button off
   mouse_zd = 0;
   mouseCBfunc = 0;
   Mcapture = 0;
   paint_toparc(2);                                                        //  remove mouse circle          v.11.04
   gdk_window_set_cursor(drWin->window,0);                                 //  set normal cursor            v.11.03
   return;
}


/**************************************************************************
      file menu functions
***************************************************************************/

//  display image gallery (thumbnails) in a separate window

void m_gallery(GtkWidget *, cchar *)
{
   zfuncs::F1_help_topic = "navigation";
   
   if (curr_file)
      image_gallery(0,"paint1",curr_file_posn,m_gallery2,mWin);            //  overlay main window    v.10.9
   else {
      char *pp = get_current_dir_name();
      if (pp) {
         image_gallery(pp,"init",0,m_gallery2,mWin);                       //  use current directory   v.11.04
         image_gallery(0,"paint1");
         free(pp);
      }
      curr_file_posn = 0;                                                  //  v.11.05
   }

   return;
}


//  clicked thumbnail will call this function

void m_gallery2(int Nth)
{
   char     *file;

   if (Nth == -1) return;                                                  //  gallery window closed

   file = image_gallery(0,"find",Nth);
   if (! file) return;

   f_open(file,1,Nth);                                                     //  clicked file = current file
   zfree(file);

   gtk_window_present(MWIN);                                               //  bring main window to front   v.10.12
   return;
}


/**************************************************************************/

//  open file menu function

void m_open(GtkWidget *, cchar *)
{
   zfuncs::F1_help_topic = "open_image_file";
   f_open(null,1);
   return;
}


/**************************************************************************/

//  open drag-drop file

void m_open_drag(int x, int y, char *file)
{
   zfuncs::F1_help_topic = "open_image_file";
   f_open(file,1);
   return;
}


/**************************************************************************/

//  open the previous file opened (not the same as toolbar [prev] button)
//  repeated use will cycle back and forth between two most recent files

void m_previous(GtkWidget *, cchar *)
{
   int      ii, err;
   
   zfuncs::F1_help_topic = "open_previous_file";

   for (ii = 1; ii < Nrecentfiles; ii++)                                   //  v.10.12
   {
      err = f_open(recentfiles[ii],1);                                     //  get first one that is still there
      if (! err) return;
   }

   return;
}


/**************************************************************************/

//  open an image file from the list of recent image files

void m_recent(GtkWidget *, cchar *)                                        //  overhauled    v.11.01
{
   void recentfile2(int Nth);

   int            ii, typ;
   char           filespec[200];
   FILE           *fid;

   zfuncs::F1_help_topic = "open_recent_file";

   if (! recentfiles[0]) return;
   if (! menulock(1)) return;
   
   snprintf(filespec,199,"%s/recent_files",get_zuserdir());                //  create file of recent files
   fid = fopen(filespec,"w");
   if (! fid) return;
   
   for (ii = 0; ii < Nrecentfiles; ii++)
   {
      if (! recentfiles[ii]) continue;
      typ = image_file_type(recentfiles[ii]);
      if (typ != 2) continue;
      fprintf(fid,"%s \n",recentfiles[ii]);
   }

   fclose(fid);
   
   image_gallery(filespec,"initF",0,recentfile2,mWin);                     //  generate gallery of recent files
   image_gallery(0,"paint1",0);                                            //  show new image gallery window

   menulock(0);
   return;
}


//  clicked thumbnail will call this function

void recentfile2(int Nth)
{
   char     *file;

   if (Nth == -1) {                                                        //  gallery window cancelled
      if (curr_file) image_gallery(curr_file,"init");                      //  reset gallery from current file    v.11.05
      return;
   }
   
   file = image_gallery(0,"find",Nth);
   image_gallery(file,"init");                                             //  initz. gallery from chosen file    v.11.05
   image_gallery(0,"close");
   f_open(file,1);                                                         //  open the file
   zfree(file);
   return;
}


/**************************************************************************/

//  add a file to the list of recent files

void add_recent_file(cchar *file)
{
   int      ii;

   for (ii = 0; ii < Nrecentfiles-1 && recentfiles[ii]; ii++)              //  find file in recent list
      if (strEqu(file,recentfiles[ii])) break;                             //    (or find last entry in list)
   if (recentfiles[ii]) zfree(recentfiles[ii]);                            //  free this slot in list
   for (; ii > 0; ii--) recentfiles[ii] = recentfiles[ii-1];               //  move list down to fill hole
   recentfiles[0] = strdupz(file,0,"recentfile");                          //  current file >> first in list
   return;
}


/**************************************************************************/

//  Open a file and initialize PXM bitmap.
//  If flock and menu is locked, do nothing and return.
//  Otherwise open the file and display in main window.
//  If Nth matches the file position in current file set, curr_file_posn 
//  will be set to Nth, otherwise it is searched and set correctly.
//  (a file can be present multiple times in a collection file set)
//  Returns: 0 = OK, +N = error.

int f_open(cchar *filespec, int flock, int Nth)
{
   PXM         *temp8;
   int         err, cc, yn, fposn, nfiles, nimages;
   char        *pp, *file, titlebar[250];
   char        fname[100], fdirk[100];

   if (flock && Fmenulock) return 1;                                       //  v.10.5
   if (mod_keep()) return 2;                                               //  unsaved edits

   if (filespec) 
      file = strdupz(filespec,0,"f_open");                                 //  use passed filespec
   else 
   {                                                                       //  no passed file
      file = zgetfile1(ZTX("Open Image File"),"open",curr_file);           //  dialog to get filespec
      if (! file) return 3;                                                //  canceled
      if (strlen(file) >= maxfcc) return 4;                                //  disallow humongous file names
      if (image_file_type(file) != 2) return 3;                            //  not a supported image file      v.11.05
   }

   temp8 = f_load(file,8);                                                 //  load image as PXM-8 pixmap
   if (! temp8) {
      zfree(file);                                                         //  bad image
      return 5;
   }

   fposn = image_gallery_position(file,Nth);                               //  discard gallery file list?      v.11.05
   if (fposn < 0 && image_navi::gallerytype == 2) {
      yn = zmessageYN(mWin,ZTX("Discard current gallery list? \n %s"),image_navi::galleryname);
      if (! yn) {
         zfree(file);                                                      //  user says no
         zfree(temp8);
         return 6;
      }
   }

   if (flock && ! menulock(1)) {                                           //  lock menu     v.10.8.4
      zfree(file);
      zfree(temp8);
      return 1;
   }
   
   free_resources();                                                       //  free resources for old image file

   if (curr_file) zfree(curr_file);                                        //  current image filespec
   curr_file = strdupz(file,8,"curr_file");
   zfree(file);

   if (curr_dirk) zfree(curr_dirk);                                        //  set current directory
   curr_dirk = strdupz(curr_file,0,"curr_dirk");                           //    for new current file
   pp = strrchr(curr_dirk,'/');
   *pp = 0;
   err = chdir(curr_dirk);
   
   Fpxm8 = temp8;                                                          //  pixmap for current image
   Fww = Fpxm8->ww;
   Fhh = Fpxm8->hh;
   
   strcpy(curr_file_type,f_load_type);                                     //  set curr_file_xxx from f_load_xxx
   curr_file_bpc = f_load_bpc;
   curr_file_size = f_load_size;

   fposn = image_gallery_position(curr_file,Nth);                          //  file position in gallery        v.11.05
   if (fposn < 0) {                                                        //  image file not in list
      image_gallery(curr_file,"init");                                     //  generate new gallery list
      image_gallery(0,"paint2");                                           //  refresh gallery window if active
      fposn = image_gallery_position(curr_file,0);                         //  position and count in gallery list
   }

   nfiles = image_navi::nfiles;                                            //  total gallery files (incl. directories)
   nimages = image_navi::nimages;                                          //  total image files               v.11.05

   curr_file_posn = fposn;                                                 //  keep track of file position
   curr_file_count = image_navi::nimages;                                  //  and image gallery count

   add_recent_file(curr_file);                                             //  first in recent files list

   Fzoom = 0;                                                              //  zoom level = fit window
   zoomx = zoomy = 0;                                                      //  no zoom center

   pp = (char *) strrchr(curr_file,'/');
   strncpy0(fname,pp+1,99);                                                //  file name
   cc = pp - curr_file;
   if (cc < 99) strncpy0(fdirk,curr_file,cc+2);                            //  get dirk/path/ if short enough
   else {
      strncpy(fdirk,curr_file,96);                                         //  or use /dirk/path...
      strcpy(fdirk+95,"...");
   }

   fposn = fposn + 1 - nfiles + nimages;                                   //  position among images, 1-based
   snprintf(titlebar,250,"%s   %d/%d  %s  %s",                             //  window title bar
                  fversion,fposn,curr_file_count,fname,fdirk);
   gtk_window_set_title(MWIN,titlebar);

   zmainloop();                                                            //  bring main window to front   v.11.04
   gtk_window_present(MWIN);
   mwpaint2();                                                             //  v.11.04

   if (zdrename) m_rename(0,0);                                            //  update active rename dialog
   if (zdexifview) exif_view(0);                                           //    "  EXIF view window        v.10.2
   if (zdexifedit) m_exif_edit(0,0);                                       //    "  EXIF edit window        v.10.11
   if (zdedittags) m_edit_tags(0,0);                                       //    "  edit tags dialog
   if (zdeditcomm) m_edit_comments(0,0);                                   //    "  edit comments dialog    v.10.10
   if (zdeditcapt) m_edit_caption(0,0);                                    //    "  edit caption dialog     v.10.12

   if (flock) menulock(0);                                                 //  v.10.8.4

   return 0;
}


/**************************************************************************/

//  open previous or next file in current gallery list

void m_prev(GtkWidget *, cchar *)
{
   int      err, Nth;

   zfuncs::F1_help_topic = "open_image_file";

   if (! curr_file) return;
   if (mod_keep()) return;

   for (Nth = curr_file_posn-1; Nth >= 0; Nth--)                           //  v.11.05
   {
      char *pp = image_gallery(0,"find",Nth);
      if (! pp) continue;
      err = f_open(pp,1,Nth);
      zfree(pp);
      if (! err) break;
   }
      
   return;
}

void m_next(GtkWidget *, cchar *)
{
   int      err, Nth;
   int      nfiles = image_navi::nfiles;

   zfuncs::F1_help_topic = "open_image_file";

   if (! curr_file) return;
   if (mod_keep()) return;
   
   for (Nth = curr_file_posn+1; Nth < nfiles; Nth++)                       //  v.11.05
   {
      char *pp = image_gallery(0,"find",Nth);
      if (! pp) continue;
      err = f_open(pp,1,Nth);
      zfree(pp);
      if (! err) break;
   }
      
   return;
}


/**************************************************************************/

//  save (modified) image to same file - no confirmation of overwrite.

void m_save(GtkWidget *, cchar *)
{   
   if (! curr_file) return;

   zfuncs::F1_help_topic = "save_file";
   
   strcpy(jpeg_quality,def_jpeg_quality);                                  //  default jpeg save quality

   if (strEqu(curr_file_type,"other"))                                     //  if gif, bmp, etc. use jpg    v.11.03
      strcpy(curr_file_type,"jpg");

   f_save(curr_file,curr_file_type,curr_file_bpc);                         //  save file
   
   strcpy(curr_file_type,f_save_type);                                     //  update curr_file_xxx from f_save_xxx
   curr_file_size = f_save_size;
   
   return;
}


/**************************************************************************/

//  save (modified) image to new file, confirm if overwrite existing file.

GtkWidget   *saveas_fchooser;

void m_saveas(GtkWidget *, cchar *)
{
   void  saveas_radiobutt(void *, int button);
   void  saveas_kbkey(void *, GdkEventKey *event);

   GtkWidget      *fdialog, *hbox;
   GtkWidget      *tiff8, *tiff16, *jpeg, *png, *jqlab, *jqval, *newvers;
   char           *outfile = 0, *outfile2 = 0, *pext, *pvers;
   cchar          *type, *delim;
   int            ii, err, yn, nvers, bpc, status;
   struct stat    fstat;
   cchar          *exts = ".jpg.JPG.jpeg.JPEG.tif.TIF.tiff.TIFF.png.PNG";

   if (! curr_file) return;

   zfuncs::F1_help_topic = "save_file";

   fdialog = gtk_dialog_new_with_buttons(ZTX("Save File"),                 //  build file save dialog
                           MWIN, GTK_DIALOG_MODAL,
                           GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, 
                           GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT, null);
   gtk_window_set_default_size(GTK_WINDOW(fdialog),600,500);

   saveas_fchooser = gtk_file_chooser_widget_new(GTK_FILE_CHOOSER_ACTION_SAVE);
   gtk_container_add(GTK_CONTAINER(GTK_DIALOG(fdialog)->vbox),saveas_fchooser);

   //  set_filename() should select the file but does nothing      GTK bug ?     v.10.9.1       /////
   //  set_current_name() puts file name in input box, but does not select the file
   gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(saveas_fchooser),curr_file);
   char *fname = strrchr(curr_file,'/') + 1;
   gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(saveas_fchooser),fname);
   
   hbox = gtk_hbox_new(0,0);
   gtk_container_add(GTK_CONTAINER(GTK_DIALOG(fdialog)->vbox),hbox);
   gtk_box_set_child_packing(GTK_BOX(GTK_DIALOG(fdialog)->vbox),hbox,0,0,10,GTK_PACK_END);

   tiff8 = gtk_radio_button_new_with_label(null,"tiff-8");
   tiff16 = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(tiff8),"tiff-16");
   png = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(tiff8),"png");
   jpeg = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(tiff8),"jpeg");
   jqlab = gtk_label_new(ZTX("quality"));
   jqval = gtk_entry_new();
   newvers = gtk_check_button_new_with_label(ZTX("make new version"));

   gtk_entry_set_width_chars(GTK_ENTRY(jqval),2);
   gtk_box_pack_start(GTK_BOX(hbox),tiff8,0,0,5);                          //  (o) tiff8  (o) tiff16  (o) png
   gtk_box_pack_start(GTK_BOX(hbox),tiff16,0,0,5);
   gtk_box_pack_start(GTK_BOX(hbox),png,0,0,5);
   gtk_box_pack_start(GTK_BOX(hbox),jpeg,0,0,5);                           //  (o) jpeg  jpeg quality [__]
   gtk_box_pack_start(GTK_BOX(hbox),jqlab,0,0,0);
   gtk_box_pack_start(GTK_BOX(hbox),jqval,0,0,3);
   gtk_box_pack_end(GTK_BOX(hbox),newvers,0,0,10);                         //  [x] new version
   
   G_SIGNAL(tiff8,"pressed",saveas_radiobutt,0)                            //  connect file type radio buttons
   G_SIGNAL(tiff16,"pressed",saveas_radiobutt,1)                           //    to handler function
   G_SIGNAL(png,"pressed",saveas_radiobutt,2)
   G_SIGNAL(jpeg,"pressed",saveas_radiobutt,3)
   G_SIGNAL(fdialog,"key-release-event",saveas_kbkey,0)

   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(jpeg),1);                //  set default file type = jpeg
   gtk_entry_set_text(GTK_ENTRY(jqval),def_jpeg_quality);                  //  default jpeg save quality
   
   if (strEqu(curr_file_type,"png"))                                       //  default matches file type
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(png),1);              //    if png or tiff
   if (strEqu(curr_file_type,"tiff"))
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tiff8),1);
   if (curr_file_bpc == 16)
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tiff16),1);

dialog_run:

   gtk_widget_show_all(fdialog);                                           //  run dialog

   status = gtk_dialog_run(GTK_DIALOG(fdialog));
   if (status != GTK_RESPONSE_ACCEPT) {                                    //  user cancelled
      gtk_widget_destroy(fdialog);
      return;
   }

   outfile2 = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(saveas_fchooser));
   if (! outfile2) goto dialog_run;
   outfile = strdupz(outfile2,12,"curr_file");                             //  add space for possible .vNN and .ext
   g_free(outfile2);

   type = "jpg";                                                           //  default output type
   bpc = 8;

   if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(png)))
      type = "png";
   if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tiff8)))
      type = "tif";
   if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tiff16))) {
      type = "tif";
      bpc = 16;
   }

   if (strEqu(type,"jpg")) {                                               //  set jpeg save quality
      ii = atoi(gtk_entry_get_text(GTK_ENTRY(jqval)));
      if (ii < 1 || ii > 100) {
         zmessageACK(mWin,ZTX("jpeg quality must be 1-100"));
         goto dialog_run;
      }
      sprintf(jpeg_quality,"%d",ii);
   }
   
   pext = strrchr(outfile,'/');                                            //  locate file .ext
   if (pext) pext = strrchr(pext,'.');
   if (pext && ! strstr(exts,pext)) pext = 0;                              //  keep .ext and append new    v.10.12.1
   if (! pext) {
      pext = outfile + strlen(outfile);                                    //  missing, add one
      strcpy(pext,".");
      strcpy(pext+1,type);
   }

   nvers = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(newvers));       //  new version checked?
   if (nvers) {
      pvers = pext - 4;                                                    //  find curr. version: filename.vNN.ext
      if (! strnEqu(pvers,".v",2)) nvers = 0;                              //  (new version format .vNN)    v.10.12
      else {
         err = convSI(pvers+2,nvers,1,98,&delim);                          //  convert NN to number 1-98
         if (err > 1) nvers = 0;                                           //  conversion error
         if (delim != pext) nvers = 0;                                     //  check format NN.ext
      }
      if (nvers == 0) {                                                    //  no version in file name
         pvers = pext;
         pext += 4;
         memmove(pext,pvers,6);                                            //  make space for .vNN before .ext
         strncpy(pvers,".vNN",4);
      }

      for (ii = 98; ii > nvers; ii--)                                      //  look for higher file versions
      {
         pvers[2] = ii/10 + '0';                                           //  build filename.vNN.ext
         pvers[3] = ii - 10 * (ii/10) + '0';
         err = stat(outfile,&fstat);
         if (! err) break;
      }
      ii++;                                                                //  use next version 1-99
      nvers = ii;
      pvers[2] = ii/10 + '0';                                              //  build filename.vNN.ext
      pvers[3] = ii - 10 * (ii/10) + '0';
   }

   gtk_widget_destroy(fdialog);                                            //  kill dialog    /// gtk bug, can crash

   err = stat(outfile,&fstat);                                             //  check if file exists
   if (! err) {
      yn = zmessageYN(mWin,ZTX("Overwrite file? \n %s"),outfile);          //  confirm overwrite
      if (! yn) {
         zfree(outfile);
         return;
      }
   }

   f_save(outfile,type,bpc);                                               //  save the file

   if (samedirk(outfile,curr_file))                                        //  if no directory change,         v.10.11
      f_open(outfile,1);                                                   //    make it the current file      v.11.05

   zfree(outfile);
   return;
}


//  set dialog file type from user selection of file type radio button

void saveas_radiobutt(void *, int button)                                  //  v.9.4
{
   cchar    *filetypes[4] = { ".tif", ".tif", ".png", ".jpg" };            //  v.10.9
   char     *filespec;
   char     *filename, *pp;

   filespec = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(saveas_fchooser));
   if (! filespec) return;
   filename = strrchr(filespec,'/');
   if (! filename) return;
   filename = strdupz(filename+1,6,"saveas");
   pp = strrchr(filename,'.');
   if (! pp || strlen(pp) > 5) pp = filename + strlen(filename);           //  bugfix        v.10.9.1
   strcpy(pp,filetypes[button]);
   gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(saveas_fchooser),filename);
   gtk_widget_show_all(saveas_fchooser);
   zfree(filename);
   g_free(filespec);                                                       //  bugfix, leak    v.10.9.1
   return;
}


//  response function for KB key release

void saveas_kbkey(void *, GdkEventKey *event)                              //  v.10.5
{
   if (event->keyval == GDK_F1) 
      showz_userguide(zfuncs::F1_help_topic);
   return;
}


/**************************************************************************/

//  create a new blank image with desired background color

void m_create(GtkWidget *, cchar *)                                        //  v.11.01
{
   int   create_dialog_event(zdialog *zd, cchar *event);

   zdialog     *zd;
   int         zstat;
   char        *prev_file = 0;

   zfuncs::F1_help_topic = "create";
   
   if (mod_keep()) return;                                                 //  unsaved edits
   if (! menulock(1)) return;                                              //  lock menus

   if (curr_file) prev_file = strdupz(curr_file);

//    file name [___________________________] .jpg                         //  v.11.05
//    width [____]  height [____] (pixels)
//    color [____]
   
   zd = zdialog_new(ZTX("Create Blank Image"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zd,"hbox","hbf","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labf","hbf","file name","space=3");
   zdialog_add_widget(zd,"entry","file","hbf","no-name","space=3|expand");
   zdialog_add_widget(zd,"label","ftype","hbf",".jpg  ");
   zdialog_add_widget(zd,"hbox","hbz","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","space","hbz",0,"space=3");
   zdialog_add_widget(zd,"label","labw","hbz",ZTX("width"));
   zdialog_add_widget(zd,"spin","width","hbz","100|9999|1|1600");
   zdialog_add_widget(zd,"label","space","hbz",0,"space=5");
   zdialog_add_widget(zd,"label","labh","hbz",ZTX("height"));
   zdialog_add_widget(zd,"spin","height","hbz","100|9999|1|1000");
   zdialog_add_widget(zd,"label","space","hbz",0,"space=3");
   zdialog_add_widget(zd,"label","labp","hbz","(pixels) ");
   zdialog_add_widget(zd,"hbox","hbc","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","space","hbc",0,"space=3");
   zdialog_add_widget(zd,"label","labc","hbc",ZTX("color"));
   zdialog_add_widget(zd,"colorbutt","color","hbc","200|200|200");
   
   create_dialog_event(zd,"init");
   zdialog_run(zd,create_dialog_event);
   zstat = zdialog_wait(zd);
   zdialog_free(zd);

   if (zstat != 1) {
      if (prev_file) {
         f_open(prev_file,0);
         zfree(prev_file);
         mwpaint2();
      }
   }

   menulock(0);
   return;
}


//  dialog event and completion function

int create_dialog_event(zdialog *zd, cchar *event)                         //  overhauled    v.11.05
{
   char           color[20], fname[100], *filespec;
   cchar          *pp;
   int            fncc, ww, hh, err;
   static int     red, green, blue;
   uint8          *pixel;
   struct stat    statb;
   
   if (zd->zstat != 1) return 0;                                           //  [done]
   
   zdialog_fetch(zd,"file",fname,100);                                     //  get new file name
   strTrim2(fname);
   if (*fname <= ' ') strcpy(fname,"no-name");
   fncc = strlen(fname);
   
   filespec = strdupz(curr_dirk,fncc+8,"m_create");                        //  make full filespec
   strcat(filespec,"/");
   strcat(filespec,fname);
   strcat(filespec,".jpg");
   
   err = stat(filespec,&statb);                                            //  make sure it does not exist
   if (! err) {
      zmessageACK(mWin,"file already exists");
      zfree(filespec);
      zd->zstat = 0;                                                       //  keep dialog alive
      return 0;
   }

   zdialog_fetch(zd,"width",ww);                                           //  get image dimensions
   zdialog_fetch(zd,"height",hh);

   zdialog_fetch(zd,"color",color,19);                                     //  get image color
   pp = strField(color,"|",1);
   if (pp) red = atoi(pp);
   pp = strField(color,"|",2);
   if (pp) green = atoi(pp);
   pp = strField(color,"|",3);
   if (pp) blue = atoi(pp);
   
   mutex_lock(&Fpixmap_lock);                                              //  lock pixmaps

   PXM_free(Fpxm8);                                                        //  create new PXM image
   Fpxm8 = PXM_make(ww,hh,8);
   Fww = Fpxm8->ww;
   Fhh = Fpxm8->hh;
   pixel = (uint8 *) Fpxm8->bmp;

   for (int ii = 0; ii < ww * hh * 3; ii += 3) {
      pixel[ii] = red;
      pixel[ii+1] = green;
      pixel[ii+2] = blue;
   }

   mutex_unlock(&Fpixmap_lock);

   strcpy(jpeg_quality,def_jpeg_quality);
 
   err = f_save(filespec,"jpg",8);                                         //  save to disk
   if (err) {
      zmessageACK(mWin,"cannot save file");
      zfree(filespec);
      return 0;
   }

   f_open(filespec,0);                                                     //  make it the current file
   zfree(filespec);

   return 0;
}


/**************************************************************************/

//  Delete image file - move curr_file to trash.
//  Use new Linux standard trash function.                                 //  v.10.8
//  If not available, revert to Desktop folder for fotoxx trash.

void m_trash(GtkWidget *, cchar *)
{
   int            err, yn;
   char           *pp, trashdir[200];
   struct stat    trstat;
   static int     gstat, trashworks = 1;                                   //  assume it works until otherwise
   GError         *gerror = 0;
   GFile          *gfile = 0;
   cchar          *gerrmess = ZTX("Linux standard trash is not supported. \n"
                                  "Desktop trash folder will be created.");
   
   zfuncs::F1_help_topic = "trash";                                        //  v.10.8

   if (! curr_file) return;                                                //  nothing to trash

   if (! menulock(1)) return;                                              //  check lock but leave unlocked
   menulock(0);

   err = stat(curr_file,&trstat);                                          //  get file status
   if (err) {
      zmessLogACK(mWin,strerror(errno));
      return;
   }

   if (! (trstat.st_mode & S_IWUSR)) {                                     //  check permission
      yn = zmessageYN(mWin,ZTX("Move read-only file to trash?"));
      if (! yn) return;
      trstat.st_mode |= S_IWUSR;
      chmod(curr_file,trstat.st_mode);
   }
   
   if (trashworks)                                                         //  try Linux standard trash
   {
      gfile = g_file_new_for_path(curr_file);
      gstat = g_file_trash(gfile,0,&gerror);
      if (! gstat) {
         printf("g_file_trash() error: %s \n",gerror->message);
         zmessageACK(mWin,gerrmess);
         trashworks = 0;                                                   //  did not work
      }
   }

   if (! trashworks)
   {
      snprintf(trashdir,199,"%s/%s",getenv("HOME"),ftrash);                //  use fotoxx trash filespec

      trstat.st_mode = 0;
      err = stat(trashdir,&trstat);
      if (! S_ISDIR(trstat.st_mode)) {
         err = mkdir(trashdir,0750);                                       //  v.11.03
         if (err) {
            zmessLogACK(mWin,ZTX("Cannot create trash folder: %s"),wstrerror(err));
            return;
         }
      }

      snprintf(command,ccc,"cp \"%s\" \"%s\" ",curr_file,trashdir);        //  copy image file to trash
      err = system(command);
      if (err) {
         zmessLogACK(mWin,ZTX("error: %s"),wstrerror(err));
         return;
      }

      err = remove(curr_file);                                             //  remove original file      v.11.03
      if (err) {
         zmessLogACK(mWin,ZTX("error: %s"),wstrerror(err));
         return;
      }
   }
   
   delete_search_index(curr_file);                                         //  delete in search index file
   image_gallery(0,"delete",curr_file_posn);                               //  delete in gallery list
   image_gallery(0,"paint2");                                              //  refresh gallery window if active

   pp = image_gallery(0,"find",curr_file_posn);                            //  open next file (now in current position)
   if (pp) f_open(pp,1);                                                   //                                 v.11.05
   if (pp) zfree(pp);

   return;
}


/**************************************************************************/

//  rename menu function
//  activate rename dialog, stuff data from current file
//  dialog remains active when new file is opened

char     rename_old[100] = "";
char     rename_new[100] = "";

void m_rename(GtkWidget *, cchar *)
{
   int rename_dialog_event(zdialog *zd, cchar *event);
   
   char     *pdir, *pfile, *pext;

   if (! curr_file) return;
   
   zfuncs::F1_help_topic = "rename";                                       //  v.10.8

   if (! zdrename)                                                         //  restart dialog
   {
      zdrename = zdialog_new(ZTX("Rename Image File"),mWin,Bcancel,null);
      zdialog_add_widget(zdrename,"hbox","hb1","dialog",0,"space=10");
      zdialog_add_widget(zdrename,"vbox","vb1","hb1",0,"homog|space=5");
      zdialog_add_widget(zdrename,"vbox","vb2","hb1",0,"homog|expand");

      zdialog_add_widget(zdrename,"button","Bold","vb1",ZTX("old name"));
      zdialog_add_widget(zdrename,"button","Bnew","vb1",ZTX("rename to"));
      zdialog_add_widget(zdrename,"button","Bprev","vb1",ZTX("previous"));

      zdialog_add_widget(zdrename,"hbox","hb21","vb2",0);                  //  [ old name ] [ oldname  ]
      zdialog_add_widget(zdrename,"hbox","hb22","vb2",0);                  //  [ new name ] [ newname  ] [+1]
      zdialog_add_widget(zdrename,"hbox","hb23","vb2",0);                  //  [ previous ] [ prevname ]

      zdialog_add_widget(zdrename,"label","Lold","hb21");
      zdialog_add_widget(zdrename,"entry","Enew","hb22",0,"expand|scc=30");
      zdialog_add_widget(zdrename,"button","B+1","hb22"," +1 ","space=5");
      zdialog_add_widget(zdrename,"label","Lprev","hb23");

      zdialog_run(zdrename,rename_dialog_event);                           //  run dialog
   }

   parsefile(curr_file,&pdir,&pfile,&pext);
   strncpy0(rename_old,pfile,99);
   strncpy0(rename_new,pfile,99);
   zdialog_stuff(zdrename,"Lold",rename_old);                              //  current file name
   zdialog_stuff(zdrename,"Enew",rename_new);                              //  entered file name

   return;
}


//  dialog event and completion callback function

int rename_dialog_event(zdialog *zd, cchar *event)
{
   char           *pp, *pdir, *pfile, *pext, *pnew, *pold;
   int            nseq, digits, ccp, ccn, ccx, err;
   struct stat    statb;
   
   if (zd->zstat) {                                                        //  complete
      zdialog_free(zdrename);                                              //  kill dialog
      return 0;
   }
   
   if (strEqu(event,"Bold"))                                               //  reset to current file name
      zdialog_stuff(zd,"Enew",rename_old);

   if (strEqu(event,"Bprev")) {                                            //  previous name >> new name
      zdialog_fetch(zd,"Lprev",rename_new,99);
      zdialog_stuff(zd,"Enew",rename_new);
   }

   if (strEqu(event,"B+1"))                                                //  increment sequence number
   {
      zdialog_fetch(zd,"Enew",rename_new,94);                              //  get entered filename
      pp = rename_new + strlen(rename_new);
      digits = 0;
      while (pp[-1] >= '0' && pp[-1] <= '9') {
         pp--;                                                             //  look for NNN in filenameNNN
         digits++;
      }
      nseq = 1 + atoi(pp);                                                 //  NNN + 1
      if (nseq > 9999) nseq = 0;
      if (digits < 2) digits = 2;                                          //  keep digit count if enough
      if (nseq > 99 && digits < 3) digits = 3;                             //  use leading zeros
      if (nseq > 999 && digits < 4) digits = 4;
      snprintf(pp,digits+1,"%0*d",digits,nseq);
      zdialog_stuff(zd,"Enew",rename_new);
   }

   if (strEqu(event,"Bnew"))                                               //  [rename to] button
   {
      if (! menulock(1)) return 0;                                         //  check lock but leave unlocked
      menulock(0);

      parsefile(curr_file,&pdir,&pfile,&pext);                             //  existing /directories/file.ext

      zdialog_fetch(zd,"Enew",rename_new,94);                              //  new file name from user

      ccp = strlen(pdir);                                                  //  length of /directories/
      ccn = strlen(rename_new);                                            //  length of file
      if (pext) ccx = strlen(pext);                                        //  length of .ext
      else ccx = 0;

      pnew = zmalloc(ccp + ccn + ccx + 1,"rename.new");                    //  put it all together
      strncpy(pnew,curr_file,ccp);                                         //   /directories/file.ext
      strcpy(pnew+ccp,rename_new);
      if (ccx) strcpy(pnew+ccp+ccn,pext);
      
      err = stat(pnew,&statb);                                             //  check if new name exists
      if (! err) {
         zmessageACK(mWin,ZTX("The target file already exists"));
         zfree(pnew);
         return 0;
      }
      
      snprintf(command,ccc,"cp -p \"%s\" \"%s\"",curr_file,pnew);          //  copy to new file   -p v.10.3
      err = system(command);
      if (err) {
         zmessageACK(mWin,ZTX("Rename failed \n %s"),wstrerror(err));
         printf("command: %s \n",command);
         zfree(pnew);
         return 0;
      }

      zdialog_stuff(zd,"Lprev",rename_new);                                //  set previous name in dialog

      load_filetags(pnew);                                                 //  update search index file
      update_search_index(pnew);

      pold = strdupz(curr_file,0,"curr_file");                             //  save file name to be deleted
      delete_search_index(pold);                                           //  delete in search index       v.9.7
      err = remove(pold);                                                  //  delete file                  v.11.03

      m_next(0,0);
      image_gallery(curr_file,"init");                                     //  update gallery               v.11.05
      image_gallery(0,"paint2");                                           //  refresh gallery window if active

      zfree(pnew);
      zfree(pold);
   }
   
   return 0;
}


/**************************************************************************/

//  menu function - batch rename files

char     **batchrename_filelist = 0;
int      batchrename_filecount = 0;

void m_batchrename(GtkWidget *, cchar *)                                   //  new v.9.7
{
   int   batchrename_dialog_event(zdialog *zd, cchar *event);
   
   zdialog  *zd;

   if (zdrename) return;                                                   //  interactive rename is active

   if (mod_keep()) return;                                                 //  unsaved edits
   if (! menulock(1)) return;                                              //  lock menus       v.10.8

   zfuncs::F1_help_topic = "batch_rename";                                 //  v.10.8

   zd = zdialog_new(ZTX("Batch Rename"),mWin,Bproceed,Bcancel,null);

   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"button","files","hb1",Bselectfiles,"space=10");
   zdialog_add_widget(zd,"label","labcount","hb1","0 files selected","space=10");
   zdialog_add_widget(zd,"hbox","hb2","dialog","space=5");
   zdialog_add_widget(zd,"label","lab2","hb2",ZTX("new base name"),"space=10");
   zdialog_add_widget(zd,"entry","basename","hb2");
   zdialog_add_widget(zd,"hbox","hb3","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","lab31","hb3",ZTX("starting sequence"),"space=10");
   zdialog_add_widget(zd,"entry","sequence","hb3","100","scc=5");
   zdialog_add_widget(zd,"label","lab32","hb3",ZTX("increment"),"space=10");
   zdialog_add_widget(zd,"entry","increment","hb3","01","scc=3");

   batchrename_filelist = 0;
   batchrename_filecount = 0;
   
   zdialog_run(zd,batchrename_dialog_event);                               //  run dialog
   zdialog_wait(zd);                                                       //  wait for completion
   
   zdialog_free(zd);
   menulock(0);
   return;
}


//  dialog event and completion callback function

int batchrename_dialog_event(zdialog *zd, cchar *event)
{
   char           **flist = batchrename_filelist, countmess[32];
   cchar          *selectmess = ZTX("select files to rename");
   int            ii, err, cc, ccp, ccn, ccx;
   int            sequence, increment, adder;
   char           basename[100], filename[120], *oldfile, *newfile;
   char           *pdir, *pfile, *pext;
   cchar          *errmess = ZTX("base name / sequence / increment not reasonable");
   struct stat    statb;
   
   if (strEqu(event,"files"))                                              //  select images to rename
   {
      if (flist) {                                                         //  free prior list
         for (ii = 0; flist[ii]; ii++) 
            zfree(flist[ii]);
         zfree(flist);
      }

      flist = zgetfileN(selectmess,"openN",curr_file);                     //  get file list from user
      batchrename_filelist = flist;

      if (flist)                                                           //  count files in list
         for (ii = 0; flist[ii]; ii++);
      else ii = 0;
      batchrename_filecount = ii;

      sprintf(countmess,"%d files selected",batchrename_filecount);
      zdialog_stuff(zd,"labcount",countmess);
   }
   
   if (! zd->zstat) return 0;                                              //  dialog active

   if (zd->zstat != 1) goto cleanup;                                       //  dialog canceled
   if (! batchrename_filecount) goto cleanup;                              //  no files selected

   zdialog_fetch(zd,"basename",basename,99);
   zdialog_fetch(zd,"sequence",sequence);
   zdialog_fetch(zd,"increment",increment);

   if (strlen(basename) < 1 || sequence < 1 || increment < 1) {
      zd->zstat = 0;                                                       //  keep dialog alive      bugfix v.10.7
      zmessageACK(mWin,errmess);
      return 0;
   }
   
   write_popup_text("open","Renaming files",500,200,mWin);                 //  status monitor popup window    v.10.3

   for (ii = 0; flist[ii]; ii++)
   {
      oldfile = flist[ii];
      parsefile(oldfile,&pdir,&pfile,&pext);
      ccp = strlen(pdir);
      if (pext) ccx = strlen(pext);
      else ccx = 0;

      adder = sequence + ii * increment;
      snprintf(filename,119,"%s%d",basename,adder);                        //  removed "-" between        v.10.7
      ccn = strlen(filename);

      newfile = zmalloc(ccp + ccn + ccx + 1,"newfile");                    //  construct /path/filename.ext
      strcpy(newfile,pdir);
      strcpy(newfile+ccp,filename);
      if (ccx) strcpy(newfile+ccp+ccn,pext);

      err = stat(newfile,&statb);
      if (! err) {
         snprintf(command,ccc,"%s %s",ZTX("new file already exists:"),newfile);
         write_popup_text("write",command);
         zfree(newfile);
         break;
      }

      cc = snprintf(command,ccc,"cp -p \"%s\" \"%s\"",oldfile,newfile);    //  copy to new file   -p v.10.3
      if (cc >= maxfcc*2) {
         snprintf(command,ccc,"%s %s",ZTX("filespec too long:"),oldfile);
         write_popup_text("write",command);
         zfree(newfile);
         break;
      }

      write_popup_text("write",command);                                   //  report progress
      zmainloop();

      err = system(command);
      if (err) {
         snprintf(command,ccc,"%s %s",ZTX("Rename failed:"),wstrerror(err));
         write_popup_text("write",command);
         zfree(newfile);
         break;
      }

      load_filetags(newfile);                                              //  update search index file
      update_search_index(newfile);
      zfree(newfile);

      err = remove(oldfile);                                               //  delete old file           v.11.03
      delete_search_index(oldfile);                                        //  remove from search index
      zmainloop();
   }

   write_popup_text("write","COMPLETED");
   write_popup_text("close",0);

   image_gallery(curr_file,"init");                                        //  update gallery file list
   image_gallery(0,"paint2");                                              //  refresh gallery window if active

cleanup:

   if (batchrename_filecount) {
      for (ii = 0; flist[ii]; ii++) 
         zfree(flist[ii]);
      zfree(flist);
   }

   return 0;
}


/**************************************************************************/

//  create a collection of image files and save list of files in a file

void m_create_collection(GtkWidget *, cchar *)                             //  v.11.05
{
   char     **flist = 0, *cfile = 0, *file1 = 0;
   FILE     *fid;

   zfuncs::F1_help_topic = "create_collection";

   if (mod_keep()) return;                                                 //  unsaved edits
   if (! menulock(1)) return;                                              //  test menu lock
   menulock(0);

   flist = image_gallery_getfiles(0,mWin);                                 //  get collection file list from user
   if (! flist) return;

   cfile = zgetfile1(ZTX("Save Collection"),"save",collections_dirk);      //  get file to save collection
   if (! cfile) goto cleanup;

   fid = fopen(cfile,"w");                                                 //  open collection file
   if (! fid) goto cleanup;
   
   for (int ii = 0; flist[ii]; ii++)                                       //  write collection file list
      fprintf(fid,"%s\n",flist[ii]);

   fclose(fid);

   image_gallery(cfile,"initF",0,m_gallery2,mWin);                         //  generate gallery from collection
   file1 = image_gallery(0,"find",0);                                      //  top of list
   f_open(file1,1);                                                        //  open first file
   image_gallery(0,"paint1");                                              //  show new image gallery window

cleanup:

   if (cfile) zfree(cfile);                                                //  free memory
   if (file1) zfree(file1);
   
   if (flist) {
      for (int ii = 0; flist[ii]; ii++)
         zfree(flist[ii]);
      zfree(flist);
   }

   return;
}


//  open a previously saved collection of image files,
//  make this the current file list and open thumbnail gallery

void m_open_collection(GtkWidget *, cchar *)                               //  v.11.05
{
   char     *cfile = 0, *file1 = 0;

   zfuncs::F1_help_topic = "open_collection";

   if (mod_keep()) return;                                                 //  unsaved edits
   if (! menulock(1)) return;                                              //  test menu lock
   menulock(0);

   cfile = zgetfile1(ZTX("Open Collection"),"open",collections_dirk);      //  get file for collection
   if (! cfile) return;

   image_gallery(cfile,"initF",0,m_gallery2,mWin);                         //  generate gallery of files in collection
   file1 = image_gallery(0,"find",0);                                      //  top of list
   if (! file1) goto cleanup;

   f_open(file1,1);                                                        //  open first file
   image_gallery(0,"paint1");                                              //  show new image gallery window

cleanup:

   if (cfile) zfree(cfile);                                                //  free memory
   if (file1) zfree(file1);
   return;
}


//  delete an existing collection

void m_delete_collection(GtkWidget *, cchar *)                             //  v.11.05
{
   char     *cfile = 0;

   zfuncs::F1_help_topic = "delete_collection";

   if (mod_keep()) return;                                                 //  unsaved edits
   if (! menulock(1)) return;                                              //  test menu lock
   menulock(0);

   cfile = zgetfile1(ZTX("Delete Collection"),"open",collections_dirk);    //  get collection file to delete
   if (! cfile) return;
   
   if (! zmessageYN(mWin,"delete %s ?",cfile)) return;
   
   remove(cfile);                                                          //  delete it
   zfree(cfile);

   return;
}


//  edit an existing collection - add, remove, and rearrange images

char     **edit_coll_files;
int      edit_coll_Nfiles;
int      edit_coll_mode;
#define  edit_coll_maxfiles 1000

void m_edit_collection(GtkWidget *, cchar *)                               //  v.11.05
{
   void  edit_coll_CBfunc(int Nth);
   int   edit_coll_dialog_event(zdialog *zd, cchar *event);

   char        *file = 0, *collfile = 0, *currfile = 0;
   int         zstat, nfiles, cc;
   FILE        *fid;
   zdialog     *zd;

   zfuncs::F1_help_topic = "edit_collection";

   if (mod_keep()) return;                                                 //  unsaved edits
   if (! menulock(1)) return;                                              //  lock menu
   
   if (curr_file)                                                          //  remember current file
      currfile = strdupz(curr_file,0,"edit_coll");
   
   collfile = zgetfile1(ZTX("Open Collection"),"open",collections_dirk);   //  get file for collection
   if (! collfile) goto cleanup;

   image_gallery(collfile,"initF",0,m_gallery2,mWin);                      //  generate gallery of files in collection
   file = image_gallery(0,"find",0,edit_coll_CBfunc);                      //  set my callback function
   if (! file) goto cleanup;

   f_open(file,0);                                                         //  open first file
   image_gallery(0,"paint1");                                              //  show new image gallery window

   cc = edit_coll_maxfiles * sizeof(char *);
   edit_coll_files = (char **) zmalloc(cc,"edit_coll");
   memset(edit_coll_files,0,cc);
   edit_coll_Nfiles = 0;
   edit_coll_mode = 0;

/***
                Edit Collection
         [Add]      Add new images to collection
         [Delete]   Delete images from collection
         [Remove]   Remove and save images
         [Insert]   Insert new or saved images
               Press F1 for help
                            [cancel]  [done]
***/
   
   zd = zdialog_new(ZTX("Edit Collection"),0,Bdone,Bcancel,null);
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"space=5|homog");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"space=5|homog");
   zdialog_add_widget(zd,"hbox","hb21","vb2");
   zdialog_add_widget(zd,"hbox","hb22","vb2");
   zdialog_add_widget(zd,"hbox","hb23","vb2");
   zdialog_add_widget(zd,"hbox","hb24","vb2");
   zdialog_add_widget(zd,"button","add","vb1",Badd);
   zdialog_add_widget(zd,"button","delete","vb1",Bdelete);
   zdialog_add_widget(zd,"button","remove","vb1",Bremove);
   zdialog_add_widget(zd,"button","insert","vb1",Binsert);
   zdialog_add_widget(zd,"label","labadd","hb21",ZTX("Add new images to collection"));
   zdialog_add_widget(zd,"label","labdel","hb22",ZTX("Delete images from collection"));
   zdialog_add_widget(zd,"label","labrem","hb23",ZTX("Remove and save images"));
   zdialog_add_widget(zd,"label","labins","hb24",ZTX("Insert new or saved images"));
   zdialog_add_widget(zd,"label","space","hb21",0,"expand");
   zdialog_add_widget(zd,"label","space","hb22",0,"expand");
   zdialog_add_widget(zd,"label","space","hb23",0,"expand");
   zdialog_add_widget(zd,"label","space","hb24",0,"expand");
   zdialog_add_widget(zd,"label","labhelp","dialog",ZTX("Press F1 for help"));
   
   zdialog_run(zd,edit_coll_dialog_event);
   zstat = zdialog_wait(zd);
   zdialog_free(zd);
   
   if (zstat == 1) 
   {
      nfiles = image_navi::nfiles;
      if (! nfiles) goto cleanup;
   
      file = zgetfile1(ZTX("Save Collection"),"save",collfile);            //  get file to save collection
      if (! file) goto cleanup;
      zfree(collfile);
      collfile = file;

      fid = fopen(collfile,"w");                                           //  open collection file
      if (! fid) goto cleanup;
      
      for (int ii = 0; ii < nfiles; ii++)                                  //  write image files
      {
         file = image_gallery(0,"find",ii);
         if (! file) continue;
         fprintf(fid,"%s\n",file);
         zfree(file);
      }

      fclose(fid);
   }
   
cleanup:

   if (collfile) zfree(collfile);                                          //  free memory

   if (edit_coll_files) {
      for (int ii = 0; ii < edit_coll_Nfiles; ii++)
         zfree(edit_coll_files[ii]);
      zfree(edit_coll_files);
   }
   
   edit_coll_mode = 0;                                                     //  revert to prior status
   image_gallery(0,"close");
   
   if (currfile) {
      image_gallery(currfile,"init",0,m_gallery2,mWin);
      f_open(currfile,0);
      zfree(currfile);
   }
   
   menulock(0);
   return;
}


//  edit collection dialog event and completion function

int edit_coll_dialog_event(zdialog *zd, cchar *event)                      //  v.11.05
{
   char     *file, **files;
   int      ii, nn;

   if (strEqu(event,"add")) 
   {
      edit_coll_mode = 1;
      zdialog_stuff(zd,"labhelp",ZTX("Select images to add, then \n"
                                     "press [Insert] to insert them."));

      files = zgetfileN(ZTX("select image files"),"openN",curr_file);
      if (! files) return 0;

      for (nn = 0; files[nn]; nn++);                                       //  count added images
      if (nn >= edit_coll_maxfiles) {
         zmessageACK(mWin,"too many files");
         return 0;
      }
      
      for (ii = 0; ii < nn; ii++)                                          //  save for later insert
      {
         file = edit_coll_files[ii];
         if (file) zfree(file);
         edit_coll_files[ii] = files[ii];
      }

      edit_coll_Nfiles = nn;
   }

   if (strEqu(event,"delete")) {
      edit_coll_mode = 2;
      zdialog_stuff(zd,"labhelp",ZTX("Click on images to delete."));
   }

   if (strEqu(event,"remove")) {
      edit_coll_mode = 3;
      zdialog_stuff(zd,"labhelp",ZTX("Click on images to remove and save, \n"
                                     "then press [Insert] to insert them."));
   }

   if (strEqu(event,"insert")) {
      edit_coll_mode = 4;
      zdialog_stuff(zd,"labhelp",ZTX("Click on image where new or saved \n"
                                     "images are to be inserted (after)."));
   }

   return 0;
}


//  callback function for edit_collection
//  clicked image thumbnails call this function
//  save clicked files in list for processing by the dialog function

void edit_coll_CBfunc(int Nth)                                             //  v.11.05
{
   char     *file;
   int      ii;
   
   if (edit_coll_mode < 2) {
      if (Nth < 0) return;
      file = image_gallery(0,"find",Nth);
      if (file) zfree(file);
      return;
   }

   if (edit_coll_mode == 2)                                                //  delete image from collection
   {
      if (Nth < 0) return;
      file = image_gallery(0,"find",Nth);
      if (! file) return;
      zfree(file);
      image_gallery(0,"delete",Nth);
      image_gallery(0,"paint2",-1);                                        //  keep same gallery scroll position
      return;
   }

   if (edit_coll_mode == 3)                                                //  remove image and save for insert
   {
      if (Nth < 0) return;
      file = image_gallery(0,"find",Nth);
      if (! file) return;
      if (edit_coll_Nfiles == edit_coll_maxfiles) {
         zmessageACK(mWin,"too many files");
         return;
      }
      edit_coll_files[edit_coll_Nfiles++] = file;
      image_gallery(0,"delete",Nth);
      image_gallery(0,"paint2",-1);
      return;
   }

   if (edit_coll_mode == 4)                                                //  insert saved images
   {
      if (Nth < 0) return;
      file = image_gallery(0,"find",Nth);
      if (! file) return;
      zfree(file);

      for (ii = 0; ii < edit_coll_Nfiles; ii++)
      {
         file = edit_coll_files[ii];
         image_gallery(file,"insert",Nth+1);                               //  paste at selected file + 1
         image_gallery(0,"paint2",-1);
         zfree(file);                                                      //  free memory
      }

      edit_coll_Nfiles = 0;
      return;
   }

   return;
}


/**************************************************************************/

//  print current image file

void m_print(GtkWidget *, cchar *)                                         //  use GTK print   v.9.4
{
   PXM * print_addgrid(PXM *printimage);                                   //  v.11.01

   int      err;
   char     *printfile;
   PXM      *printimage, *pxmtemp;

   zfuncs::F1_help_topic = "print";
   
   if (! curr_file) return;                                                //  no image file
   
   printfile = strdupz(get_zuserdir(),20,"printfile");                     //  make temp print file:
   strcat(printfile,"/printfile.tif");                                     //    ~/.fotoxx/printfile.tif    v.11.03
   
   if (Fpxm16) printimage = PXM_convbpc(Fpxm16);
   else printimage = PXM_copy(Fpxm8);
   
   pxmtemp = print_addgrid(printimage);                                    //  add grid lines if wanted     v.11.01
   if (pxmtemp) {
      PXM_free(printimage);
      printimage = pxmtemp;
   }
   
   err = PXBwrite(printimage,printfile);
   PXM_free(printimage);

   if (err) {
      zfree(printfile);                                                    //  v.10.3
      return;
   }
   
   print_imagefile(printfile);                                             //  GTK print utility in zfuncs.cc
   zfree(printfile);
   return;
}


//  add grid lines to print image if wanted

PXM * print_addgrid(PXM *printimage)                                       //  v.11.01
{
   PXM      *temp8;
   uint8    *pixel;
   int      px, py, ww, hh, row, stepx, stepy;
   
   if (! Fgrid[0] && ! Fgrid[1]) return 0;

   temp8 = f_load(curr_file,8);
   if (! temp8) return 0;
   
   ww = temp8->ww;
   hh = temp8->hh;
   row = ww * 3;

   stepx = gridspace[0];                                                   //  space between grid lines
   stepy = gridspace[1];
   
   stepx = stepx / Mscale;                                                 //  window scale to image scale
   stepy = stepy / Mscale;
   
   if (gridcount[0]) stepx = ww / (1 + gridcount[0]);                      //  if line counts specified,
   if (gridcount[1]) stepy = hh / ( 1 + gridcount[1]);                     //    set spacing accordingly
   
   if (Fgrid[0]) {
      for (px = stepx; px < ww-1; px += stepx)
      for (py = 0; py < hh; py++)
      {
         pixel = PXMpix8(temp8,px,py);                                     //  adjoining white and black lines
         pixel[0] = pixel[1] = pixel[2] = 255;
         pixel[3] = pixel[4] = pixel[5] = 0;
      }
   }

   if (Fgrid[1]) {
      for (py = stepy; py < hh-1; py += stepy)
      for (px = 0; px < ww; px++)
      {
         pixel = PXMpix8(temp8,px,py);
         pixel[0] = pixel[1] = pixel[2] = 255;
         pixel[row] = pixel[row+1] = pixel[row+2] = 0;
      }
   }

   return temp8;
}


/**************************************************************************/

//  forced quit - can cause running function to crash

void m_quit(GtkWidget *, cchar *)
{
   if (mod_keep()) return;                                                 //  unsaved edits
   printf("quit \n");
   Fshutdown++;                                                            //  bugfix  v.10.11

   for (int ii = 0; ii < 100; ii++)                                        //  wait up to a second if something running
      if (Ffuncbusy) {                                                     //  v.11.01
         zmainloop();
         zsleep(0.01);
      }
   
   if (Ffuncbusy) printf("busy function killed");                          //  v.11.05

   save_params();                                                          //  save state for next session
   free_resources();                                                       //  delete temp files
   if (KBzmalloclog) zmalloc_report();                                     //  report memory v.10.8
   fflush(null);                                                           //  flush stdout, stderr      v.11.05
   gtk_main_quit();                                                        //  gone forever
   return;
}


/**************************************************************************
      tools menu functions
***************************************************************************/

//  set new image zoom level or magnification

void m_zoom(GtkWidget *, cchar *menu)
{
   int      ii, iww, ihh, Dww, Dhh;
   char     zoom;
   double   scalew, scaleh, fitscale;
   double   scales[11] = { 0.125, 0.176, 0.25, 0.354, 0.5, 0.71, 1.0, 1.41, 2.0, 2.83, 4.0 };
   
   if (strnEqu(menu,"Zoom",4)) zoom = menu[4];                             //  get + or -
   else  zoom = *menu;
   
   Dww = drWin->allocation.width;                                          //  drawing window size
   Dhh = drWin->allocation.height;
   
   if (E3pxm16) {                                                          //  bugfix
      iww = E3ww;
      ihh = E3hh;
   }
   else  {
      iww = Fww;
      ihh = Fhh;
   }

   if (iww > Dww || ihh > Dhh) {                                           //  get window fit scale
      scalew = 1.0 * Dww / iww;
      scaleh = 1.0 * Dhh / ihh;
      if (scalew < scaleh) fitscale = scalew;
      else fitscale = scaleh;
   }
   else fitscale = 1.0;                                                    //  if image < window use 100%
   
   if (zoom == '+') {                                                      //  zoom bigger
      if (! Fzoom) Fzoom = fitscale / 1.2;
      Fzoom = Fzoom * sqrt(2.0);                                           //  new scale: 41% bigger
      for (ii = 0; ii < 11; ii++)
         if (Fzoom < 1.01 * scales[ii]) break;                             //  next higher scale in table
      if (ii == 11) ii = 10;
      Fzoom = scales[ii];
      if (Fzoom < fitscale) Fzoom = 0;                                     //  image < window
   }

   if (zoom == '-') Fzoom = 0;                                             //  zoom to fit window

   if (zoom == 'Z') {
      if (Fzoom != 0) Fzoom = 0;                                           //  toggle 100% and fit window
      else  Fzoom = 1;
   }
   
   if (! Fzoom) zoomx = zoomy = 0;                                         //  no req. zoom center
   
   mwpaint2();                                                             //  refresh window
   return;
}


/**************************************************************************/

//  monitor test function

void m_moncheck(GtkWidget *, cchar *)
{
   uint8       *pixel;
   int         red, green, blue;
   int         row, col, row1, row2;
   int         ww = 800, hh = 500;
   zdialog     *zd;
   
   cchar       *message = ZTX("Brightness should show a gradual ramp \n"
                              "extending all the way to the edges.");
   
   zfuncs::F1_help_topic = "check_monitor";

   if (mod_keep()) return;                                                 //  unsaved edits
   if (! menulock(1)) return;
   mutex_lock(&Fpixmap_lock);

   PXM_free(Fpxm8);
   Fpxm8 = PXM_make(ww,hh,8);
   Fww = ww;
   Fhh = hh;
   curr_file_bpc = 8;
   curr_file_size = 0;

   for (red = 0; red <= 1; red++)                                          //  8 RGB combinations
   for (green = 0; green <= 1; green++)
   for (blue = 0; blue <= 1; blue++)
   {
      row1 = 4 * red + 2 * green + blue;                                   //  row 0 to 7
      row1 = row1 * hh / 8;                                                //  stripe, 1/8 of image
      row2 = row1 + hh / 8;
      
      for (row = row1; row < row2; row++)
      for (col = 0; col < ww; col++)
      {
         pixel = PXMpix8(Fpxm8,col,row);
         pixel[0] = red * 256 * col / ww;
         pixel[1] = green * 256 * col / ww;
         pixel[2] = blue * 256 * col / ww;
      }
   }

   Fzoom = 0;                                                              //  scale to window
   gtk_window_set_title(MWIN,"monitor check");
   mutex_unlock(&Fpixmap_lock);
   mwpaint2();                                                             //  repaint window

   zd = zdialog_new(ZTX("Monitor Check"),mWin,Bdone,null);                 //  start user dialog      v.11.04
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","lab1","hb1",message,"space=5");
   
   zdialog_resize(zd,200,0);
   zdialog_run(zd);
   zdialog_wait(zd);                                                       //  wait for dialog complete
   zdialog_free(zd);

   f_open(curr_file,0);                                                    //  back to current file   v.11.04
   menulock(0);
   return;
}


/**************************************************************************/

//  adjust monitor gamma

void m_mongamma(GtkWidget *, cchar *)                                      //  revised v.11.04
{
   int   mongamma_event(zdialog *zd, cchar *event);
   char  gammachart[200];
   
   zdialog     *zd;
   cchar       *permit = "Chart image courtesy of Norman Koren";
   cchar       *website1 = "http://www.normankoren.com";
   cchar       *website2 = "http://www.imatest.org";
   PXM         *pxmtemp;

   zfuncs::F1_help_topic = "monitor_gamma";

   if (mod_keep()) return;                                                 //  unsaved edits
   if (! menulock(1)) return;

   snprintf(gammachart,200,"%s/images/gammachart.png",get_zdocdir());      //  gamma chart .png file
   pxmtemp = PXBread(gammachart);
   if (! pxmtemp) {
      zmessageACK(mWin,"gamma chart is missing");
      menulock(0);
      return;
   }

   mutex_lock(&Fpixmap_lock);                                              //  curr. image = gamma chart
   PXM_free(Fpxm8);
   Fpxm8 = pxmtemp;
   Fww = Fpxm8->ww;
   Fhh = Fpxm8->hh;
   curr_file_bpc = 8;
   curr_file_size = 0;

   Fzoom = 1;                                                              //  scale 100% (required)
   gtk_window_set_title(MWIN,"adjust gamma chart");
   mutex_unlock(&Fpixmap_lock);
   mwpaint2();                                                             //  repaint window

   zd = zdialog_new(ZTX("Monitor Gamma"),mWin,Bdone,null);                 //  start user dialog
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=8");
   zdialog_add_widget(zd,"label","labgamma","hb1","gamma","space=5");
   zdialog_add_widget(zd,"hscale","gamma","hb1","0.6|1.4|0.02|1.0","expand");
   zdialog_add_widget(zd,"hbox","hb2","dialog");
   zdialog_add_widget(zd,"label","permit","hb2",permit,"space=5");
   zdialog_add_widget(zd,"hbox","hb3","dialog");
   zdialog_add_widget(zd,"link","web1","hb3",website1);
   zdialog_add_widget(zd,"hbox","hb4","dialog");
   zdialog_add_widget(zd,"link","web2","hb4",website2);
   
   zdialog_resize(zd,200,0);
   zdialog_run(zd,mongamma_event);
   zdialog_wait(zd);                                                       //  wait for dialog complete
   zdialog_free(zd);

   f_open(curr_file,0);                                                    //  back to current file   v.11.04
   menulock(0);
   return;
}


//  dialog event function

int mongamma_event(zdialog *zd, cchar *event)
{
   int      err;
   double   gamma;

   if (strEqu(event,"gamma")) {
      zdialog_fetch(zd,"gamma",gamma);
      sprintf(command,"xgamma -quiet -gamma %.2f",gamma);
      err = system(command);
      if (err) zmessageACK(mWin,"error: %s",wstrerror(err));
   }
   
   return 0;
}


/**************************************************************************/

//  create or update brightness histogram graph

GtkWidget      *winhisto = 0;                                              //  brightness histogram window
GtkWidget      *winhistoH = 0;                                             //  histogram drawing area
GtkWidget      *winhistoB = 0;                                             //  brightness band underneath


void m_histogram(GtkWidget *, cchar *)                                     //  menu function
{
   zfuncs::F1_help_topic = "brightness_graph";                             //  v.10.8

   if (! Dpxm8) return;

   if (winhisto) {
      histogram_paint();
      return;
   }

   winhisto = gtk_window_new(GTK_WINDOW_TOPLEVEL);
   gtk_window_set_title(GTK_WINDOW(winhisto),ZTX("Brightness Distribution"));
   gtk_window_set_transient_for(GTK_WINDOW(winhisto),MWIN);
   gtk_window_set_default_size(GTK_WINDOW(winhisto),300,200);
   gtk_window_set_position(GTK_WINDOW(winhisto),GTK_WIN_POS_MOUSE);
   GtkWidget * vbox = gtk_vbox_new(0,3);
   gtk_container_add(GTK_CONTAINER(winhisto),vbox);

   winhistoH = gtk_drawing_area_new();                                     //  histogram drawing area
   gtk_box_pack_start(GTK_BOX(vbox),winhistoH,1,1,3);

   winhistoB = gtk_drawing_area_new();                                     //  brightness scale under histogram
   gtk_box_pack_start(GTK_BOX(vbox),winhistoB,0,0,0);                      //  (progresses from black to white)
   gtk_widget_set_size_request(winhistoB,300,12);                          //                   v.9.3

   G_SIGNAL(winhisto,"destroy",histogram_destroy,0)
   G_SIGNAL(winhistoH,"expose-event",histogram_paint,0)
   G_SIGNAL(winhistoB,"expose-event",histogram_paint,0)

   gtk_widget_show_all(winhisto);
   
   return;
}


void histogram_paint()                                                     //  paint graph window
{
   GdkGC          *gdkgc = 0;                                              //  GDK graphics context
   GdkColor       color;
   GdkColormap    *colormap = 0;

   int         brdist[40], bin, nbins = 40;                                //  increased                 v.9.3
   int         dist_maxbin = 0;
   int         winww, winhh;
   int         px, py, ww, hh, orgx, orgy;
   uint8       *pix8;
   uint16      *pix16;
   double      bright;
   
   if (! winhisto) return;
   if (! Dpxm8) return;

   gdkgc = gdk_gc_new(winhistoH->window);                                  //  use separate graphics context
   colormap = gtk_widget_get_colormap(winhistoH);

   for (bin = 0; bin < nbins; bin++)                                       //  clear brightness distribution
      brdist[bin] = 0;

   mutex_lock(&Fpixmap_lock);
   
   if (Factivearea && E3pxm16)                                             //  compute brightness distribution    v.11.02
   {                                                                       //    for selected area being edited
      for (int ii = 0; ii < Fww * Fhh; ii++)
      {
         if (sa_pixisin[ii] == 0) continue;
         py = ii / Fww;
         px = ii - Fww * py;
         pix16 = PXMpix(E3pxm16,px,py);
         bright = 0.003906 * pixbright(pix16);                             //  0 to 255
         brdist[int(bright / 256 * nbins)]++;                              //  0 to nbins
      }
   }
   
   else 
   {
      for (py = 0; py < dhh; py++)                                         //  compute brightness distribution
      for (px = 0; px < dww; px++)                                         //    for image in visible window
      {
         pix8 = (uint8 *) Dpxm8->bmp + (py * dww + px) * 3;
         bright = pixbright(pix8);                                         //  0 to 255
         brdist[int(bright / 256 * nbins)]++;                              //  0 to nbins
      }
   }

   for (bin = 0; bin < nbins; bin++)                                       //  find max. bin
      if (brdist[bin] > dist_maxbin) dist_maxbin = brdist[bin];

   mutex_unlock(&Fpixmap_lock);

   gdk_window_clear(winhistoH->window);

   winww = winhistoH->allocation.width;                                    //  drawing window size
   winhh = winhistoH->allocation.height;
   ww = winww / nbins;                                                     //  bin width
   bin = -1;

   color.red = 10000;                                                      //  a pleasing color
   color.green = 25000;
   color.blue = 40000;
   gdk_rgb_find_color(colormap,&color);
   gdk_gc_set_foreground(gdkgc,&color);

   for (px = 0; px < winww; px++)                                          //  draw each bin
   {
      if (px * nbins / winww > bin) {
         bin++;
         hh = int(0.9 * winhh * brdist[bin] / dist_maxbin);
         orgx = px;
         orgy = winhh - hh;
         gdk_draw_rectangle(winhistoH->window,gdkgc,1,orgx,orgy,ww+1,hh);
      }
   }

   hh = winhistoB->allocation.height;

   for (px = 0; px < winww; px++)                                          //  draw brightness scale underneath
   {                                                                       //                      v.9.3
      color.red = color.green = color.blue = 65536 * px / winww;
      gdk_rgb_find_color(colormap,&color);
      gdk_gc_set_foreground(gdkgc,&color);
      gdk_draw_line(winhistoB->window,gdkgc,px,0,px,hh-1);
   }

   return;
}


void histogram_destroy()                                                   //  delete window
{
   if (winhisto) gtk_widget_destroy(winhisto);
   winhisto = 0;
   return;
}


/**************************************************************************/

//  start a new parallel instance of fotoxx

void m_clone(GtkWidget *, cchar *)
{
   GdkScreen   *screen;
   int         ww, hh, ignore;

   zfuncs::F1_help_topic = "clone";                                        //  v.10.8

   screen = gdk_screen_get_default();
   ww = gdk_screen_get_width(screen);
   hh = gdk_screen_get_height(screen);
   ww = ww / 2 - 20;
   hh = hh - 50;
   gdk_window_move_resize(mWin->window,ww+20,10,ww,hh);
   
   snprintf(command,ccc,"fotoxx -c %d %d -l %s",ww,hh,zfuncs::zlanguage);
   if (curr_file) strncatv(command,ccc," \"",curr_file,"\"",null);
   strcat(command," &");
   ignore = system(command);
   return;
}


/**************************************************************************/

//  enter or leave slideshow mode

int         ss_interval = 3;                                               //  slide show interval
int         ss_timer = 0;                                                  //  slide show timer
char        *ss_oldfile, *ss_newfile;                                      //  image files for transition
PXM         *ss_pxmold, *ss_pxmnew, *ss_pxmmix;                            //  pixmap images: old, new, mixed
int         ss_ww, ss_hh;                                                  //  full screen window size
char        ss_transition[40] = "instant";                                 //  transition type
int         ss_busy = 0;                                                   //  transition underway
int         ss_escape = 0;                                                 //  user pressed Escape key
char        *ss_musicfile = 0;                                             //  music file or playlist

void m_slideshow(GtkWidget *, cchar *)                                     //  overhauled    v.11.01
{
   int   slideshow_dialog_event(zdialog *zd, cchar *event);

   GdkScreen      *screen;
   zdialog        *zd;
   int            zstat, secs, err;
   cchar          *esc_message = ZTX("Press ESC to exit slide show");
   
   zfuncs::F1_help_topic = "slide_show";                                   //  v.10.8

   if (! Fslideshow)                                                       //  start slide show
   {
      if (! curr_file) return;
      if (! menulock(1)) return;

      zd = zdialog_new(ZTX("Time Interval"),mWin,Bproceed,Bcancel,null);   //  user dialog
      zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
      zdialog_add_widget(zd,"label","lab1","hb1",ZTX("seconds"));
      zdialog_add_widget(zd,"entry","secs","hb1","3","scc=5");
      zdialog_add_widget(zd,"label","lab2","hb1",esc_message,"space=10");
      zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=3");
      zdialog_add_widget(zd,"label","lab2","hb2",ZTX("transition"));
      zdialog_add_widget(zd,"combo","trans","hb2");
      zdialog_add_widget(zd,"hbox","hb3","dialog",0,"space=3");
      zdialog_add_widget(zd,"label","lab3","hb3",ZTX("music file"));
      zdialog_add_widget(zd,"entry","musicfile","hb3",0,"scc=30|space=5");
      zdialog_add_widget(zd,"button","browse","hb3",Bbrowse,"space=5");

      if (ss_interval < 9999)
         zdialog_stuff(zd,"secs",ss_interval);                             //  stuff last time interval
         
      zdialog_stuff(zd,"musicfile",ss_musicfile);                          //  stuff last music file
      
      zdialog_cb_app(zd,"trans",ZTX("arrow keys"));                        //  put options into combo box
      zdialog_cb_app(zd,"trans",ZTX("instant"));
      zdialog_cb_app(zd,"trans",ZTX("fade-in"));
      zdialog_cb_app(zd,"trans",ZTX("roll-right"));
      zdialog_cb_app(zd,"trans",ZTX("roll-down"));
      zdialog_cb_app(zd,"trans",ZTX("shift-left"));
      zdialog_cb_app(zd,"trans",ZTX("venetian"));
      zdialog_cb_app(zd,"trans",ZTX("lattice"));
      zdialog_cb_app(zd,"trans",ZTX("rectangle"));
      zdialog_cb_app(zd,"trans",ZTX("ellipse"));
      zdialog_cb_app(zd,"trans",ZTX("random"));

      zdialog_stuff(zd,"trans",ZTX("arrow keys"));                         //  default

      zdialog_resize(zd,200,0);                                            //  run dialog
      zdialog_run(zd,slideshow_dialog_event);
      zstat = zdialog_wait(zd);                                            //  wait for completion

      if (zstat != 1) {                                                    //  cancel
         zdialog_free(zd);
         menulock(0);
         return;
      }

      zdialog_fetch(zd,"secs",secs);                                       //  get interval, seconds
      zdialog_fetch(zd,"trans",ss_transition,40);                          //  get transition type

      char *pp = zmalloc(500);
      zdialog_fetch(zd,"musicfile",pp,499);                                //  get music file
      ss_musicfile = pp;

      zdialog_free(zd);

      ss_interval = secs;                                                  //  interval between slides
      if (strEqu(ss_transition,ZTX("arrow keys")))                         //  if manual transition, huge interval
         ss_interval = 9999;
      
      if (ss_musicfile && *ss_musicfile && strNeq(ss_musicfile,"undefined")) 
      {                                                                    //  if music file present, start it up
         sprintf(command,"xdg-open \"%s\" ",ss_musicfile);                 //  v.11.04
         printf("command: %s \n",command);
         err = system(command);
      }

      gtk_widget_hide_all(GTK_WIDGET(mMbar));                              //  enter slide show mode
      gtk_widget_hide_all(GTK_WIDGET(mTbar));                              //  (full screen, no extras)
      gtk_widget_hide_all(GTK_WIDGET(STbar));
      gtk_window_fullscreen(MWIN);                                         //  (comment out for debug)

      screen = gdk_screen_get_default();                                   //  get full screen dimensions
      ss_ww = gdk_screen_get_width(screen);
      ss_hh = gdk_screen_get_height(screen);

      ss_pxmold = PXM_make(ss_ww,ss_hh,8);                                 //  make 3 screen-size pixmaps
      ss_pxmnew = PXM_make(ss_ww,ss_hh,8);
      ss_pxmmix = PXM_make(ss_ww,ss_hh,8);
      
      ss_newfile = strdupz(curr_file);                                     //  start with current image
      ss_oldfile = 0;

      ss_timer = get_seconds() + ss_interval;                              //  set timer for first call to
      Fslideshow = 1;                                                      //    ss_slideshow_next()
   }

   else                                                                    //  leave slide show mode
   {
      if (ss_busy) {
         ss_escape = 1;                                                    //  wait for transition done
         return;
      }

      ss_escape = 0;
      Fslideshow = 0;

      gtk_window_unfullscreen(MWIN);                                       //  restore old window size
      gtk_widget_show_all(GTK_WIDGET(mMbar));
      gtk_widget_show_all(GTK_WIDGET(mTbar));
      gtk_widget_show_all(GTK_WIDGET(STbar));
      
      if (ss_newfile) zfree(ss_newfile);                                   //  free memory
      if (ss_oldfile) zfree(ss_oldfile);
      ss_newfile = ss_oldfile = 0;
      PXM_free(ss_pxmold);
      PXM_free(ss_pxmnew);
      PXM_free(ss_pxmmix);

      menulock(0);
   }

   Fblowup = Fslideshow;
   Fzoom = 0;                                                              //  fit image to window
   mwpaint2(); 
   return;
}


//  dialog event function - file chooser for music file

int slideshow_dialog_event(zdialog *zd, cchar *event)                      //  v.11.04
{
   char     *pp;
   
   if (! strEqu(event,"browse")) return 0;
   pp = zgetfile1(ZTX("Select music file or playlist"),"open",ss_musicfile);
   if (! pp) return 0;
   zdialog_stuff(zd,"musicfile",pp);
   zfree(pp);
   return 0;
}


//  Show next slide if time is up or user navigates with arrow keys.
//  Called by timer function and by keyboard function if Fslideshow is set.

void slideshow_next(cchar *mode)                                           //  new  v.11.01
{
   void  ss_loadimage(char *file, PXM *pxmout);
   void  ss_instant();
   void  ss_fadein();
   void  ss_rollright();
   void  ss_rolldown();
   void  ss_shiftleft();
   void  ss_venetian();
   void  ss_lattice();
   void  ss_rectangle();
   void  ss_ellipse();

   double         secs;
   char           *pp = 0;
   int            Fkey, rnum, nn;
   
   if (ss_busy) return;                                                    //  come back later

   if (ss_escape) {
      m_slideshow(0,0);                                                    //  user pressed escape key
      return;
   }

   if (strEqu(mode,"timer")) {                                             //  timer trigger
      secs = get_seconds();
      if (secs < ss_timer) return;
      mode = "next";                                                       //  time for next image
      Fkey = 0;
   }
   else Fkey = 1;                                                          //  keyboard trigger

   for (nn = 1; nn < 3; nn++)                                              //  tolerate 1-2 deleted files     v.11.05
   {
      if (strEqu(mode,"prev"))                                             //  get previous or next image file
         pp = image_gallery(0,"find",curr_file_posn-nn);
      if (strEqu(mode,"next"))
         pp = image_gallery(0,"find",curr_file_posn+nn);
      if (! pp) continue;
      if (image_file_type(pp) == 2) break;
      zfree(pp);
      pp = 0;
   }
   
   if (! pp) return;
   
   ss_busy++;

   if (ss_oldfile) zfree(ss_oldfile);
   ss_oldfile = ss_newfile;
   ss_newfile = pp;
   
   ss_loadimage(ss_oldfile,ss_pxmold);   
   ss_loadimage(ss_newfile,ss_pxmnew);

   mutex_lock(&Fpixmap_lock);                                              //  block other window updates

   if (Fkey) ss_instant();                                                 //  keyboard, do instant transition
   else {
      if (strEqu(ss_transition,ZTX("instant"))) ss_instant();              //  timer, do selected transition
      if (strEqu(ss_transition,ZTX("fade-in"))) ss_fadein();
      if (strEqu(ss_transition,ZTX("roll-right"))) ss_rollright();
      if (strEqu(ss_transition,ZTX("roll-down"))) ss_rolldown();
      if (strEqu(ss_transition,ZTX("shift-left"))) ss_shiftleft();
      if (strEqu(ss_transition,ZTX("venetian"))) ss_venetian();
      if (strEqu(ss_transition,ZTX("lattice"))) ss_lattice();
      if (strEqu(ss_transition,ZTX("rectangle"))) ss_rectangle();
      if (strEqu(ss_transition,ZTX("ellipse"))) ss_ellipse();
      if (strEqu(ss_transition,ZTX("random"))) {
         rnum = int(9.0 * drandz());                                       //  0 - 8.99...
         if (rnum == 0) ss_instant();
         if (rnum == 1) ss_fadein();
         if (rnum == 2) ss_rollright();
         if (rnum == 3) ss_rolldown();
         if (rnum == 4) ss_shiftleft();
         if (rnum == 5) ss_venetian();
         if (rnum == 6) ss_lattice();
         if (rnum == 7) ss_rectangle();
         if (rnum == 8) ss_ellipse();
      }
   }

   mutex_unlock(&Fpixmap_lock);

   zmainloop();                                                            //  catch-up anynch window updates?

   f_open(ss_newfile,0);                                                   //  sync all image data

   if (zdeditcomm) m_edit_comments(0,0);                                   //  edit comments dialod
   if (zdeditcapt) m_edit_caption(0,0);                                    //  edit caption dialog

   secs = get_seconds();                                                   //  set time for next image
   ss_timer = secs + ss_interval + 0.5;
   ss_busy = 0;
   return;
}


//  load image and rescale to fit in given pixmap = window size

void ss_loadimage(char *file, PXM *pxmout)                                 //  new v.11.01
{
   int         cc, fww, fhh, px, py, orgx, orgy;
   int         gray = 0.9 * 255;
   PXM         *pxmtemp1, *pxmtemp2;
   double      wscale, hscale, scale;
   uint8       *pix1, *pix2;
   
   cc = ss_ww * ss_hh * 3;                                                 //  clear output pixmap to bright gray
   memset(pxmout->bmp,gray,cc);

   pxmtemp1 = f_load(file,8);                                              //  load image file into 1x pixmap
   if (! pxmtemp1) return;

   fww = pxmtemp1->ww;                                                     //  image size
   fhh = pxmtemp1->hh;
   
   wscale = 1.0 * ss_ww / fww;                                             //  find scale to fit in window
   hscale = 1.0 * ss_hh / fhh;
   if (wscale < hscale) scale = wscale;                                    //  use greatest ww/hh ratio
   else  scale = hscale;
   fww = fww * scale;
   fhh = fhh * scale;
   
   pxmtemp2 = PXM_rescale(pxmtemp1,fww,fhh);                               //  rescale image to fit window
   
   orgx = 0.5 * (ss_ww - fww);                                             //  origin of image in window
   orgy = 0.5 * (ss_hh - fhh);

   for (py = 0; py < fhh; py++)
   for (px = 0; px < fww; px++)
   {
      pix1 = PXMpix8(pxmtemp2,px,py);
      pix2 = PXMpix8(pxmout,px+orgx,py+orgy);
      pix2[0] = pix1[0];
      pix2[1] = pix1[1];
      pix2[2] = pix1[2];
   }

   PXM_free(pxmtemp1);
   PXM_free(pxmtemp2);
   
   return;
}


//  instant transition (for use with keyboard arrow keys)

void ss_instant()                                                          //  new v.11.01
{
   uint8    *pix3;

   pix3 = PXMpix8(ss_pxmnew,0,0);
   gdk_draw_rgb_image(drWin->window, gdkgc, 0, 0, ss_ww, ss_hh, NODITHER, pix3, ss_ww*3);
   return;
}


//  fade-out / fade-in transition

void ss_fadein()                                                           //  new v.11.01
{
   int         ii, jj, kk, px, py;
   double      newpart, oldpart;
   uint8       *pix1, *pix2, *pix3;
   
   PXM_free(ss_pxmmix);
   ss_pxmmix = PXM_copy(ss_pxmold);
   
   for (ii = 0; ii <= 100; ii += 10)
   {
      newpart = 0.01 * ii;
      oldpart = 1.0 - newpart;

      for (jj = 0; jj < 2; jj++)                                           //  four passes, each modifies 25%
      for (kk = 0; kk < 2; kk++)                                           //    of the pixels (visually smoother)
      {
         for (py = jj; py < ss_hh; py += 2)
         for (px = kk; px < ss_ww; px += 2)
         {
            pix1 = PXMpix8(ss_pxmold,px,py);
            pix2 = PXMpix8(ss_pxmnew,px,py);
            pix3 = PXMpix8(ss_pxmmix,px,py);
            pix3[0] = newpart * pix2[0] + oldpart * pix1[0];
            pix3[1] = newpart * pix2[1] + oldpart * pix1[1];
            pix3[2] = newpart * pix2[2] + oldpart * pix1[2];
         }
         
         pix3 = PXMpix8(ss_pxmmix,0,0);
         gdk_draw_rgb_image(drWin->window, gdkgc, 0, 0, ss_ww, ss_hh, NODITHER, pix3, ss_ww*3);
      }
   }

   pix3 = PXMpix8(ss_pxmnew,0,0);
   gdk_draw_rgb_image(drWin->window, gdkgc, 0, 0, ss_ww, ss_hh, NODITHER, pix3, ss_ww*3);
   return;
}


//  new image rolls over prior image from left to right

void ss_rollright()                                                        //  new v.11.01
{
   int         px, py;
   uint8       *pix1, *pix3;
   double      delay = 0.5 / ss_ww;                                        //  v.11.04

   PXM_free(ss_pxmmix);
   ss_pxmmix = PXM_copy(ss_pxmold);
   
   for (px = 0; px < ss_ww; px++)
   {
      pix1 = PXMpix8(ss_pxmnew,px,0);
      pix3 = PXMpix8(ss_pxmmix,px,0);
      
      for (py = 0; py < ss_hh; py++)
      {
         memmove(pix3,pix1,3);
         pix1 += ss_ww * 3;
         pix3 += ss_ww * 3;
      }
      
      pix3 = PXMpix8(ss_pxmmix,px,0);
      gdk_draw_rgb_image(drWin->window, gdkgc, px, 0, 1, ss_hh, NODITHER, pix3, ss_ww*3);
      zsleep(delay);
   }

   pix3 = PXMpix8(ss_pxmnew,0,0);
   gdk_draw_rgb_image(drWin->window, gdkgc, 0, 0, ss_ww, ss_hh, NODITHER, pix3, ss_ww*3);
   return;
}


//  new image rolls over prior image from top down

void ss_rolldown()                                                         //  new v.11.01
{
   int         py;
   uint8       *pix3;
   double      delay = 0.5 / ss_hh;
   
   for (py = 0; py < ss_hh; py++)
   {
      pix3 = PXMpix8(ss_pxmnew,0,py);
      gdk_draw_rgb_image(drWin->window, gdkgc, 0, py, ss_ww, 1, NODITHER, pix3, ss_ww*3);
      zsleep(delay);
   }

   pix3 = PXMpix8(ss_pxmnew,0,0);
   gdk_draw_rgb_image(drWin->window, gdkgc, 0, 0, ss_ww, ss_hh, NODITHER, pix3, ss_ww*3);
   return;
}


//  shift prior image to the left as next image shifts-in from the right
//  (this one flickers annoingly and cannot be fixed)

void ss_shiftleft()                                                        //  new v.11.01
{
   int            px, Nsteps = 50;
   uint8          *pix1, *pix3;

   for (px = 0; px < ss_ww; px += ss_ww / Nsteps)
   {
      pix1 = PXMpix8(ss_pxmold,px,0);
      pix3 = PXMpix8(ss_pxmnew,0,0);
      gdk_draw_rgb_image(drWin->window, gdkgc, 0, 0, ss_ww - px, ss_hh, NODITHER, pix1, ss_ww * 3);
      zmainloop();
      gdk_draw_rgb_image(drWin->window, gdkgc, ss_ww - px, 0, px, ss_hh, NODITHER, pix3, ss_ww * 3);
      zmainloop();
   }

   pix3 = PXMpix8(ss_pxmnew,0,0);
   gdk_draw_rgb_image(drWin->window, gdkgc, 0, 0, ss_ww, ss_hh, NODITHER, pix3, ss_ww * 3);
   return;
}


//  new image opens up in horizontal rows like venetian blinds

void ss_venetian()                                                         //  new v.11.01
{
   int         py1, py2;
   uint8       *pix3;
   int         louver, Nlouvers = 20;
   int         louversize = ss_hh / Nlouvers;
   double      delay = 1.0 / louversize;
   
   for (py1 = 0; py1 < louversize; py1++)                                  //  y-row within each louver
   {
      for (louver = 0; louver < Nlouvers; louver++)                        //  louver, first to last
      {
         py2 = py1 + louver * louversize;
         if (py2 >= ss_hh) break;
         pix3 = PXMpix8(ss_pxmnew,0,py2);
         gdk_draw_rgb_image(drWin->window, gdkgc, 0, py2, ss_ww, 1, NODITHER, pix3, ss_ww*3);
      }

      zsleep(delay);
   }

   pix3 = PXMpix8(ss_pxmnew,0,0);
   gdk_draw_rgb_image(drWin->window, gdkgc, 0, 0, ss_ww, ss_hh, NODITHER, pix3, ss_ww * 3);
   return;
}


//  a lattice opens up to show new image

void ss_lattice()                                                          //  new v.11.01
{
   int         px1, px2, py1, py2;
   uint8       *pix3;
   int         row, col, Nrow, Ncol;                                       //  rows and columns
   int         boxww, boxhh;
   double      delay;
   
   Ncol = 20;                                                              //  20 columns
   boxww = boxhh = ss_ww / Ncol;                                           //  square boxes
   Nrow = ss_hh / boxhh;                                                   //  corresp. rows
   Ncol++;                                                                 //  round up
   Nrow++;
   delay = 1.0 / boxhh;

   for (py1 = 0; py1 < boxhh; py1++)
   {
      for (row = 0; row < Nrow; row++)
      {
         py2 = py1 + row * boxhh;
         if (py2 >= ss_hh) break;
         pix3 = PXMpix8(ss_pxmnew,0,py2);
         gdk_draw_rgb_image(drWin->window, gdkgc, 0, py2, ss_ww, 1, NODITHER, pix3, ss_ww*3);
      }
      
      px1 = py1;

      for (col = 0; col < Ncol; col++)
      {
         px2 = px1 + col * boxww;
         if (px2 >= ss_ww) break;
         pix3 = PXMpix8(ss_pxmnew,px2,0);
         gdk_draw_rgb_image(drWin->window, gdkgc, px2, 0, 1, ss_hh, NODITHER, pix3, ss_ww*3);
      }
      
      zsleep(delay);
   }

   pix3 = PXMpix8(ss_pxmnew,0,0);
   gdk_draw_rgb_image(drWin->window, gdkgc, 0, 0, ss_ww, ss_hh, NODITHER, pix3, ss_ww * 3);
   return;
}


//  A hole opens up from the center and expands outward

void ss_rectangle()                                                        //  new v.11.01
{
   int         px1, py1, px2, py2, px3, py3;
   int         ww1, hh1, ww2, hh2;
   uint8       *pix3;
   int         step, Nsteps = 100;
   double      delay = 1.0 / Nsteps;

   for (step = 1; step < Nsteps; step++)
   {
      ww1 = ss_ww * step / Nsteps;
      hh1 = ww1 * ss_hh / ss_ww;
      ww2 = ss_ww / Nsteps / 2;
      hh2 = ss_hh / Nsteps / 2;

      px1 = (ss_ww - ww1) / 2;
      py1 = (ss_hh - hh1) / 2;
      px2 = px1 + ww1 - ww2;
      py2 = py1;
      px3 = px1;
      py3 = py1 + hh1 - hh2;

      pix3 = PXMpix8(ss_pxmnew,px1,py1);
      gdk_draw_rgb_image(drWin->window, gdkgc, px1, py1, ww1+1, hh2+1, NODITHER, pix3, ss_ww*3);

      pix3 = PXMpix8(ss_pxmnew,px2,py2);
      gdk_draw_rgb_image(drWin->window, gdkgc, px2, py2, ww2+1, hh1+1, NODITHER, pix3, ss_ww*3);

      pix3 = PXMpix8(ss_pxmnew,px3,py3);
      gdk_draw_rgb_image(drWin->window, gdkgc, px3, py3, ww1+1, hh2+1, NODITHER, pix3, ss_ww*3);

      pix3 = PXMpix8(ss_pxmnew,px1,py1);
      gdk_draw_rgb_image(drWin->window, gdkgc, px1, py1, ww2+1, hh1+1, NODITHER, pix3, ss_ww*3);
      
      zmainloop();
      zsleep(delay);
   }
   
   pix3 = PXMpix8(ss_pxmnew,0,0);
   gdk_draw_rgb_image(drWin->window, gdkgc, 0, 0, ss_ww, ss_hh, NODITHER, pix3, ss_ww * 3);
   return;
}


//  A ellipse opens up from the center and expands outward

void ss_ellipse()                                                          //  new v.11.03
{
   uint8       *pix3;
   int         step, Nsteps = 100;
   int         px1, py1, ww;
   double      delay = 1.0 / Nsteps;
   double      a, b, a2, b2, px, py, px2, py2;
   double      ww2 = ss_ww / 2, hh2 = ss_hh / 2;
   
   for (step = 1; step < 1.3 * Nsteps; step++)
   {
      a = ww2 * step / Nsteps;                                             //  ellipse a and b constants
      b = a * ss_hh / ss_ww;                                               //    from tiny to >> image size
      a2 = a * a;
      b2 = b * b;
      
      for (py = -b; py <= +b; py += 2)                                     //  py from top of ellipse to bottom
      {
         if (py < -(hh2-2) || py > (hh2-2)) continue;                      //  bugfix        v.11.04
         py2 = py * py;
         px2 = a2 * (1.0 - py2 / b2);                                      //  corresponding px value,
         px = sqrt(px2);                                                   //  (+/- from center of ellipse)
         if (px > ww2) px = ww2;
         ww = 2 * px;                                                      //  length of line thru ellipse
         px1 = int(ww2 - px);                                              //  relocate origin
         py1 = int(py + hh2);
         pix3 = PXMpix8(ss_pxmnew,px1,py1);
         gdk_draw_rgb_image(drWin->window, gdkgc, px1, py1, ww, 2, NODITHER, pix3, ss_ww*3);
      }

      zmainloop();
      zsleep(delay);
   }
   
   pix3 = PXMpix8(ss_pxmnew,0,0);
   gdk_draw_rgb_image(drWin->window, gdkgc, 0, 0, ss_ww, ss_hh, NODITHER, pix3, ss_ww * 3);
   return;
}


/**************************************************************************/

//  show RGB values for pixel at mouse click

zdialog     *zdRGB = 0;                                                    //  show RGB dialog
void        RGB_mousefunc();

void m_showRGB(GtkWidget *, cchar *)                                       //  menu function
{
   int   RGB_dialog_event(zdialog *zd, cchar *event);
   
   cchar  *rgbmess = ZTX("click on window to show RGB");
   cchar  *format = "Pixel: 0 0   RGB: 0.0  0.0  0.0";

   zfuncs::F1_help_topic = "show_RGB";                                     //  v.10.8

   if (! Fpxm8) return;                                                    //  no image

   zdRGB = zdialog_new(ZTX("Show RGB"),mWin,Bcancel,null);
   zdialog_add_widget(zdRGB,"label","lab1","dialog",rgbmess,"space=5");
   zdialog_add_widget(zdRGB,"label","labrgb","dialog",format,"space=5");
   zdialog_add_widget(zdRGB,"check","mymouse","dialog",BmyMouse);

   takeMouse(zdRGB,RGB_mousefunc,dragcursor);                              //  connect mouse function          v.10.12
   zdialog_run(zdRGB,RGB_dialog_event);                                    //  run dialog

   return;
}


//  dialog event function

int RGB_dialog_event(zdialog *zd, cchar *event)
{
   int      mymouse;

   if (zd->zstat) {
      freeMouse();                                                         //  disconnect mouse function       v.10.12
      zdialog_free(zdRGB);                                                 //  kill dialog
   }

   if (strEqu(event,"mymouse")) {                                          //  toggle mouse capture         v.10.12
      zdialog_fetch(zd,"mymouse",mymouse);
      if (mymouse) takeMouse(zd,RGB_mousefunc,dragcursor);                 //  connect mouse function
      else freeMouse();                                                    //  disconnect mouse
   }

   return 0;
}


//  mouse function

void RGB_mousefunc()                                                       //  mouse function
{
   int         px, py;
   double      red, green, blue;
   double      fbright, fred;
   char        text[60];
   uint8       *ppix8;
   uint16      *ppix16;
   
   if (LMclick)                                                            //  left mouse click
   {
      LMclick = 0;
      px = Mxclick;                                                        //  click position
      py = Myclick;
      
      if (E3pxm16) {                                                       //  use current image being edited
         if (px < 0 || px > E3ww-1 ||                                      //  outside image area
             py < 0 || py > E3hh-1) return;
         ppix16 = PXMpix(E3pxm16,px,py);
         red = ppix16[0] / 256.0;
         green = ppix16[1] / 256.0;
         blue = ppix16[2] / 256.0;
         fbright = pixbright(ppix16) / 256.0;
         fred = pixred(ppix16);
      }

      else if (Fpxm16) {                                                   //  use edited image
         if (px < 0 || px > Fww-1 || 
             py < 0 || py > Fhh-1) return;
         ppix16 = PXMpix(Fpxm16,px,py);
         red = ppix16[0] / 256.0;
         green = ppix16[1] / 256.0;
         blue = ppix16[2] / 256.0;
         fbright = pixbright(ppix16) / 256.0;
         fred = pixred(ppix16);
      }

      else  {                                                              //  use 8 bpc image
         if (px < 0 || px > Fww-1 || 
             py < 0 || py > Fhh-1) return;
         ppix8 = (uint8 *) Fpxm8->bmp + (py * Fww + px) * 3;
         red = ppix8[0];
         green = ppix8[1];
         blue = ppix8[2];
         fbright = pixbright(ppix8);
         fred = pixred(ppix8);
      }
      
      snprintf(text,59,"Pixel: %d %d  RGB: %6.3f %6.3f %6.3f",             //  show pixel and RGB colors
                                  px, py, red, green, blue);
      zdialog_stuff(zdRGB,"labrgb",text);
   }
   
   return;
}


/**************************************************************************/

//  toggle grid lines on / off                                             //  v.10.3

void m_gridlines(GtkWidget *, cchar *)
{
   int gridlines_dialog_event(zdialog *zd, cchar *event);

   zdialog     *zd;
   
   zfuncs::F1_help_topic = "grid_lines";                                   //  v.10.8

   zd = zdialog_new(ZTX("Grid Lines"),mWin,Bdone,Bcancel,null);
   
   zdialog_add_widget(zd,"hbox","hb0","dialog",0,"space=10");
   zdialog_add_widget(zd,"vbox","vb1","hb0");
   zdialog_add_widget(zd,"vbox","vb2","hb0");

   zdialog_add_widget(zd,"hbox","hb11","vb1",0,"space=5");
   zdialog_add_widget(zd,"hbox","hb12","vb1",0,"space=5");
   zdialog_add_widget(zd,"hbox","hb21","vb2",0,"space=5");
   zdialog_add_widget(zd,"hbox","hb22","vb2",0,"space=5");

   zdialog_add_widget(zd,"label","lab1x","hb11","x-spacing","space=5");
   zdialog_add_widget(zd,"spin","spacex","hb11","10|200|1|50");
   zdialog_add_widget(zd,"label","lab1y","hb21","y-spacing","space=5");
   zdialog_add_widget(zd,"spin","spacey","hb21","10|200|1|50");

   zdialog_add_widget(zd,"label","lab2x","hb12","x-count","space=5");
   zdialog_add_widget(zd,"spin","countx","hb12","0|100|1|0");
   zdialog_add_widget(zd,"label","lab2y","hb22","y-count","space=5");
   zdialog_add_widget(zd,"spin","county","hb22","0|100|1|0");

   zdialog_add_widget(zd,"hbox","hb3","dialog");
   zdialog_add_widget(zd,"label","laben","hb3","enable","space=5");
   zdialog_add_widget(zd,"check","xgrid","hb3","x-grid","space=10");
   zdialog_add_widget(zd,"check","ygrid","hb3","y-grid");

   zdialog_stuff(zd,"spacex",gridspace[0]);
   zdialog_stuff(zd,"spacey",gridspace[1]);
   zdialog_stuff(zd,"countx",gridcount[0]);
   zdialog_stuff(zd,"county",gridcount[1]);
   zdialog_stuff(zd,"xgrid",Fgrid[0]);
   zdialog_stuff(zd,"ygrid",Fgrid[1]);

   zdialog_run(zd,gridlines_dialog_event);
   zdialog_wait(zd);
   return;
}


//  dialog event function

int gridlines_dialog_event(zdialog *zd, cchar *event)
{
   int      zstat;

   if (zd->zstat) {
      zstat = zd->zstat;
      if (zstat != 1) Fgrid[0] = Fgrid[1] = 0;
      zdialog_free(zd);
      mwpaint2();
      return 0;
   }

   if (strEqu(event,"xgrid")) {
      zdialog_fetch(zd,"xgrid",Fgrid[0]);
      mwpaint2();
   }
   
   if (strEqu(event,"ygrid")) {
      zdialog_fetch(zd,"ygrid",Fgrid[1]);
      mwpaint2();
   }
   
   if (strEqu(event,"spacex")) {
      zdialog_fetch(zd,"spacex",gridspace[0]);
      if (Fgrid[0]) mwpaint2();
   }

   if (strEqu(event,"spacey")) {
      zdialog_fetch(zd,"spacey",gridspace[1]);
      if (Fgrid[1]) mwpaint2();
   }

   if (strEqu(event,"countx")) {
      zdialog_fetch(zd,"countx",gridcount[0]);
      if (Fgrid[0]) mwpaint2();
   }

   if (strEqu(event,"county")) {
      zdialog_fetch(zd,"county",gridcount[1]);
      if (Fgrid[1]) mwpaint2();
   }

   return 0;
}


/**************************************************************************/

//  choose or set lens parameters for panoramas

void  m_lensparms(GtkWidget *, cchar *)
{
   int         ii, zstat, radb;
   char        text[20];
   zdialog     *zd;
   
   zfuncs::F1_help_topic = "lens_parms";                                   //  v.10.8
   
   if (! lens4_name[0] || strEqu(lens4_name[0],"undefined")) {
      lens4_name[0] = strdupz("lens_1",0,"lensname");                      //  default lens parameters
      lens4_name[1] = strdupz("lens_2",0,"lensname");
      lens4_name[2] = strdupz("lens_3",0,"lensname");
      lens4_name[3] = strdupz("lens_4",0,"lensname");
      lens4_mm[0] = 30;
      lens4_mm[1] = 40;
      lens4_mm[2] = 50;
      lens4_mm[3] = 60;
      lens4_bow[0] = 0;
      lens4_bow[1] = 0;
      lens4_bow[2] = 0;
      lens4_bow[3] = 0;
      curr_lens = 1;                                                       //  default 40 mm lens
   }

   zd = zdialog_new(ZTX("Lens Parameters"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"space=5|homog");            //        Lens    mm    bow
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"space=5|homog");            //   (o)  name1   30    0.33
   zdialog_add_widget(zd,"vbox","vb3","hb1",0,"space=5|homog");            //   (x)  name2   40    0.22
   zdialog_add_widget(zd,"vbox","vb4","hb1",0,"space=5|homog");            //   (o)  name3   45    0.28
   zdialog_add_widget(zd,"label","space","vb1");                           //   (o)  name4   50    0.44
   zdialog_add_widget(zd,"radio","radb0","vb1",0);                         //
   zdialog_add_widget(zd,"radio","radb1","vb1",0);                         //             [done] [cancel]
   zdialog_add_widget(zd,"radio","radb2","vb1",0);
   zdialog_add_widget(zd,"radio","radb3","vb1",0);
   zdialog_add_widget(zd,"label","lname","vb2",ZTX("lens name"));
   zdialog_add_widget(zd,"entry","name0","vb2","scc=10");
   zdialog_add_widget(zd,"entry","name1","vb2","scc=10");
   zdialog_add_widget(zd,"entry","name2","vb2","scc=10");
   zdialog_add_widget(zd,"entry","name3","vb2","scc=10");
   zdialog_add_widget(zd,"label","lmm","vb3",ZTX("lens mm"));
   zdialog_add_widget(zd,"entry","mm0","vb3","0","scc=5");
   zdialog_add_widget(zd,"entry","mm1","vb3","0","scc=5");
   zdialog_add_widget(zd,"entry","mm2","vb3","0","scc=5");
   zdialog_add_widget(zd,"entry","mm3","vb3","0","scc=5");
   zdialog_add_widget(zd,"label","lbow","vb4",ZTX("lens bow"));
   zdialog_add_widget(zd,"entry","bow0","vb4","0.0","scc=6");
   zdialog_add_widget(zd,"entry","bow1","vb4","0.0","scc=6");
   zdialog_add_widget(zd,"entry","bow2","vb4","0.0","scc=6");
   zdialog_add_widget(zd,"entry","bow3","vb4","0.0","scc=6");
   
   for (ii = 0; ii < 4; ii++)                                              //  stuff lens data into dialog
   {
      snprintf(text,20,"name%d",ii);
      zdialog_stuff(zd,text,lens4_name[ii]);
      snprintf(text,20,"mm%d",ii);
      zdialog_stuff(zd,text,lens4_mm[ii]);
      snprintf(text,20,"bow%d",ii);
      zdialog_stuff(zd,text,lens4_bow[ii]);
   }

   snprintf(text,20,"radb%d",curr_lens);                                   //  current lens = selected
   zdialog_stuff(zd,text,1);
   
   zdialog_run(zd,0);                                                      //  run dialog, get inputs
   zstat = zdialog_wait(zd);

   if (zstat != 1) {
      zdialog_free(zd);                                                    //  canceled
      return;
   }
   
   for (ii = 0; ii < 4; ii++)                                              //  fetch lens data (revisions)
   {
      snprintf(text,20,"name%d",ii);
      zdialog_fetch(zd,text,lensname,39);
      repl_1str(lensname,lensname," ","_");                                //  replace blanks with _
      if (lens4_name[ii]) zfree(lens4_name[ii]);
      lens4_name[ii] = strdupz(lensname,0,"lensname");                     //  bugfix  v.11.03.1
      snprintf(text,20,"mm%d",ii);
      zdialog_fetch(zd,text,lens4_mm[ii]);
      snprintf(text,20,"bow%d",ii);
      zdialog_fetch(zd,text,lens4_bow[ii]);
      snprintf(text,20,"radb%d",ii);                                       //  detect which is selected
      zdialog_fetch(zd,text,radb);
      if (radb) curr_lens = ii;
   }
   
   zdialog_free(zd);
   return;
}


/**************************************************************************/

//  set GUI language

void  m_lang(GtkWidget *, cchar *)                                         //  overhauled   v.10.1
{
   zdialog     *zd;
   int         ii, cc, err, val, zstat;
   char        lang1[8], *pp;

   cchar  *langs[12] = { "en English", "de German", "es Spanish",          //  english first
                         "fr French", "gl Galacian", "it Italian", 
                         "nl Dutch", "pt Portuguese", "ru_RU Russian", 
                         "sv Swedish", "zh_CN Chinese", null  };
   
   cchar  *title = ZTX("Available Translations");

   zfuncs::F1_help_topic = "language";                                     //  v.10.8

   zd = zdialog_new(ZTX("Set Language"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zd,"label","title","dialog",title,"space=10");
   zdialog_add_widget(zd,"hbox","hb1","dialog");
   zdialog_add_widget(zd,"vbox","vb1","hb1");

   for (ii = 0; langs[ii]; ii++)                                           //  make radio button per language
      zdialog_add_widget(zd,"radio",langs[ii],"vb1",langs[ii]);

   cc = strlen(zfuncs::zlanguage);                                         //  current language
   for (ii = 0; langs[ii]; ii++)                                           //  match on lc_RC
      if (strnEqu(zfuncs::zlanguage,langs[ii],cc)) break;
   if (! langs[ii]) 
      for (ii = 0; langs[ii]; ii++)                                        //  failed, match on lc alone
         if (strnEqu(zfuncs::zlanguage,langs[ii],2)) break;
   if (! langs[ii]) ii = 0;                                                //  failed, default english
   zdialog_stuff(zd,langs[ii],1);

   zdialog_resize(zd,200,0);
   zdialog_run(zd);                                                        //  run dialog
   zstat = zdialog_wait(zd);

   for (ii = 0; langs[ii]; ii++) {                                         //  get active radio button
      zdialog_fetch(zd,langs[ii],val);
      if (val) break;
   }

   zdialog_free(zd);                                                       //  kill dialog

   if (zstat != 1) return;                                                 //  user cancel
   if (! val) return;                                                      //  no selection
   
   strncpy0(lang1,langs[ii],8);
   pp = strchr(lang1,' ');                                                 //  isolate lc_RC part
   *pp = 0;

   sprintf(command,"fotoxx -l %s &",lang1);                                //  start new fotoxx with language
   err = system(command);

   m_quit(0,0);                                                            //  exit
   return;
}


/**************************************************************************/

//  create desktop icon / launcher

void  m_menu_launcher(GtkWidget *, cchar *)                                //  v.11.05
{
   zfuncs::F1_help_topic = "menu_launcher";
   zmake_menu_launcher("Graphics","Image Editor");
   return;
}


/**************************************************************************/

//  convert multiple RAW files to tiff

char     **raw_files;
int      raw_filecount;
char     *raw_outfile = 0;

void  m_conv_raw(GtkWidget *, cchar *)                                     //  multi-threaded      v.10.1.1
{
   void * conv_raw_thread(void *arg);

   int      ii;

   zfuncs::F1_help_topic = "convert_RAW";                                  //  v.10.8

   if (! Fufraw) {
      zmessageACK(mWin,ZTX("Program ufraw-batch is required"));
      return;
   }

   if (mod_keep()) return;                                                 //  unsaved edits
   if (! menulock(1)) return;                                              //  v.10.5

   raw_files = zgetfileN(ZTX("Select RAW files to convert"),"openN",curr_dirk);
   if (! raw_files) goto rawdone;
   
   for (ii = 0; raw_files[ii]; ii++);                                      //  count selected files
   raw_filecount = ii;

   start_wt(conv_raw_thread,0);                                            //  1 thread    v.11.01

   write_popup_text("open","Converting RAW files",500,200,mWin);           //  status monitor popup window
   write_popup_text("write","converting ...");                             //  v.10.9
   
   while (true)
   {
      zmainloop();                                                         //  v.10.9
      zsleep(0.1);
      if (! raw_outfile && ! wthreads_busy) break;                         //  finished
      if (! raw_outfile) continue;                                         //  wait for more output

      write_popup_text("write",raw_outfile);                               //  write output to log window

      if (*raw_outfile != '*')                                             //  if not an error message, show
         f_open(raw_outfile,0);                                            //    converted file in main window

      zfree(raw_outfile);
      raw_outfile = 0;
      zmainloop();
   }

   wait_wts();                                                             //  v.11.01
   write_popup_text("write","COMPLETED");
   write_popup_text("close",0);                                            //  v.10.8

   image_gallery(curr_file,"init");                                        //  update gallery file list
   image_gallery(0,"paint2");                                              //  refresh gallery window if active

rawdone:

   if (raw_files) {
      for (ii = 0; raw_files[ii]; ii++) 
         zfree(raw_files[ii]);
      zfree(raw_files);
   }

   menulock(0);
   return;
}


//  convert RAW working threads

void * conv_raw_thread(void *arg)
{
   char     *rawfile, *outfile, *pp;
   int      ii, err;

   for (ii = 0; ii < raw_filecount; ii++)                                  //  1 thread   v.11.01
   {
      rawfile = raw_files[ii];
      outfile = strdupz(rawfile,5,"raw_out");
      pp = strrchr(rawfile,'.');
      pp = outfile + (pp - rawfile);
      strcpy(pp,".tiff");
      
      //  try new ufraw command format first, then old command if error    //  --wb  v.10.9

      snprintf(command,ccc,"ufraw-batch --wb=camera --out-type=tiff --out-depth=16"
                  " --overwrite --output=\"%s\" \"%s\" ",outfile, rawfile);
      err = system(command);

      if (err) {
         snprintf(command,ccc,"ufraw-batch  --wb=camera --out-type=tiff16"
                  " --overwrite --output=\"%s\" \"%s\" ",outfile, rawfile);
         err = system(command);
      }
      
      while (raw_outfile) zsleep(0.1);                                     //  wait for prior      bugfix v.10.9

      if (err) {
         zfree(outfile);
         raw_outfile = zmalloc(1000,"raw_errmess");
         snprintf(raw_outfile,999,"*** %s %s",wstrerror(err),rawfile);
      }
      else raw_outfile = outfile;
   }

   exit_wt();                                                              //  exit thread
   return 0;                                                               //  not executed, stop gcc warning
}


/**************************************************************************/

//  burn images to CD/DVD  

void  m_burn(GtkWidget *, cchar *)
{
   int      ii, cc, err;
   char     **filelist, *imagefile, *bcommand;

   if (! menulock(1)) return;                                              //  lock menus
   
   zfuncs::F1_help_topic = "burn";                                         //  v.10.8

   filelist = image_gallery_getfiles(0,mWin);                              //  get list of files to burn   v.10.9
   if (! filelist) {
      menulock(0);
      return;
   }
   
   cc = 0;
   for (ii = 0; filelist[ii]; ii++)                                        //  get memory for brasero command line
      cc += strlen(filelist[ii]) + 4;

   bcommand = zmalloc(cc+20,"brasero-command");
   strcpy(bcommand,"brasero");
   cc = strlen(bcommand);

   for (ii = 0; filelist[ii]; ii++)                                        //  copy files to command line
   {
      imagefile = filelist[ii];
      strcpy(bcommand+cc," \"");
      cc += 2;
      strcpy(bcommand+cc,imagefile);
      cc += strlen(imagefile);
      strcpy(bcommand+cc,"\"");
      cc += 1;
      zfree(imagefile);
   }
   
   zfree(filelist);

   strcat(bcommand," &");                                                  //  brasero command in background
   err = system(bcommand);
   if (err) zmessageACK(mWin,"error: %s",wstrerror(err));
   zfree(bcommand);

   menulock(0);
   return;
}


/**************************************************************************/

//  menu function
//  resize selected images and send to preferred e-mail program

void m_email(GtkWidget *, cchar *)                                         //  new v.10.10
{
   int   email_dialog_event(zdialog *zd, cchar *event);

   zdialog     *zd;                                                        //  email dialog

   zfuncs::F1_help_topic = "e-mail";

   if (mod_keep()) return;                                                 //  unsaved edits
   if (! menulock(1)) return;                                              //  lock menus

   //  [select files]  N files selected
   //  max. width   [____]
   //  max. height  [____]
   //
   //                   [proceed]  [cancel]

   zd = zdialog_new(ZTX("E-mail Images"),mWin,Bproceed,Bcancel,null);
   zdialog_add_widget(zd,"hbox","hbf","dialog",0,"space=5");
   zdialog_add_widget(zd,"button","files","hbf",Bselectfiles,"space=5");
   zdialog_add_widget(zd,"label","fcount","hbf","0 files selected","space=10");
   zdialog_add_widget(zd,"hbox","hbwh","dialog","space=10");
   zdialog_add_widget(zd,"vbox","vbwh1","hbwh",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vbwh2","hbwh",0,"homog|space=5");
   zdialog_add_widget(zd,"label","labw","vbwh1",ZTX("max. width"));
   zdialog_add_widget(zd,"label","labh","vbwh1",ZTX("max. height"));
   zdialog_add_widget(zd,"entry","maxw","vbwh2","1000","scc=5");
   zdialog_add_widget(zd,"entry","maxh","vbwh2","700","scc=5");

   zdialog_stuff(zd,"maxw",emailsize[0]);
   zdialog_stuff(zd,"maxh",emailsize[1]);
   
   zdialog_run(zd,email_dialog_event);                                     //  run dialog
   zdialog_wait(zd);                                                       //  wait for completion

   menulock(0);
   return;
}


//  dialog event and completion callback function

int email_dialog_event(zdialog *zd, cchar *event)
{
   static char    **flist = 0;
   char           countmess[32], tempdir[100], sequence[8];
   int            maxw, maxh, err, fcc, ww, hh;
   int            ii, jj, kk;
   char           *pfile, *pext, *oldfile, *newfile;
   double         scale, wscale, hscale;
   PXM            *pxmin, *pxmout;

   if (strEqu(event,"files"))                                              //  select images to resize
   {
      if (flist) {                                                         //  free prior list
         for (ii = 0; flist[ii]; ii++) 
            zfree(flist[ii]);
         zfree(flist);
      }
      
      flist = image_gallery_getfiles(0,mWin);                              //  get list of files to resize   v.10.9

      if (flist)                                                           //  count files selected
         for (ii = 0; flist[ii]; ii++);
      else ii = 0;

      sprintf(countmess,"%d files selected",ii);                           //  update dialog
      zdialog_stuff(zd,"fcount",countmess);
   }
   
   if (! zd->zstat) return 0;                                              //  dialog still busy

   if (zd->zstat != 1) goto cleanup;                                       //  dialog canceled
   
   if (! flist) {                                                          //  no files selected
      zd->zstat = 0;                                                       //  keep dialog active
      return 0;
   }

   zdialog_fetch(zd,"maxw",maxw);                                          //  new max width
   zdialog_fetch(zd,"maxh",maxh);                                          //  new max height

   if (maxw < 20 || maxh < 20) {
      zmessageACK(mWin,ZTX("max. size %d x %d is not reasonable"),maxw,maxh);
      zd->zstat = 0;
      return 0;
   }

   *tempdir = 0;                                                           //  temp directory: /tmp/<user>/fotoxx/email
   strncatv(tempdir,99,"/tmp/",cuserid(null),"/fotoxx/email",null);

   sprintf(command,"mkdir -p %s",tempdir);                                 //  (re)create directory
   err = system(command);
   if (err) {
      zmessageACK(mWin,"%s",strerror(err));
      goto cleanup;
   }

   sprintf(command,"rm -f %s/*.jpg",tempdir);                              //  purge prior file list
   err = system(command);
   
   write_popup_text("open","preparing files",500,200,mWin);                //  status monitor popup window

   strcpy(command,"xdg-email");                                            //  e-mail command: xdg-email

   for (ii = 0; flist[ii]; ii++)                                           //  loop selected files
   {
      oldfile = flist[ii];
      pfile = strrchr(oldfile,'/');                                        //  /filename.ext
      if (! pfile) continue;

      fcc = strlen(pfile);
      newfile = strdupz(tempdir,fcc+10,"emailresize.new");                 //  add extra space for .nn and .ext
      strcat(newfile,pfile);                                               //  /tmp/<user>/fotoxx/filename.ext
      pfile = strrchr(newfile,'/');

      for (jj = kk = 1; pfile[jj]; jj++, kk++)                             //  remove special characters
      {                                                                    //   (xdg-email substitutes %nn
         pfile[kk] = pfile[jj];                                            //     and thunderbird falls over)
         if (pfile[jj] < 0) continue;                                      //  UTF-8 multibyte characters OK
         if (pfile[jj] == '.') continue;                                   //  periods OK
         if (pfile[jj] >= 'a' && pfile[jj] <= 'z') continue;               //  keep a-z, A-Z, 0-9
         if (pfile[jj] >= 'A' && pfile[jj] <= 'Z') continue;
         if (pfile[jj] >= '0' && pfile[jj] <= '9') continue;
         kk--;                                                             //  omit others
      }
      
      pfile[kk] = 0;

      pext = strrchr(pfile,'.');                                           //  find .ext if there is one
      if (! pext) pext = pfile + strlen(pfile);
      if (strlen(pext) > 5) pext = pext + strlen(pext);
      
      sprintf(sequence,"-%d",ii+1);                                        //  replace with sequence number
      strcpy(pext,sequence);                                               //  filename-nn

      pext = pext + strlen(pext);                                          //  add .jpg extension
      strcpy(pext,".jpg");                                                 //  filename-nn.jpg
      
      write_popup_text("write",newfile);                                   //  log progress
      zmainloop();

      pxmin = f_load(oldfile,8);                                           //  load image as PXM-8 pixmap
      if (! pxmin) {
         printf("f_load error: %s \n",oldfile);
         continue;
      }
      
      ww = pxmin->ww;
      hh = pxmin->hh;

      wscale = hscale = 1.0;
      if (ww > maxw) wscale = 1.0 * maxw / ww;                             //  compute new size
      if (hh > maxh) hscale = 1.0 * maxh / hh;
      if (wscale < hscale) scale = wscale;
      else scale = hscale;
      ww = ww * scale;
      hh = hh * scale;
      pxmout = PXM_rescale(pxmin,ww,hh);                                   //  rescale file

      PXBwrite(pxmout,newfile);                                            //  write to new file
      
      PXM_free(pxmin);
      PXM_free(pxmout);
      
      err = strncatv(command,1990," --attach ","\"",newfile,"\"",null);    //  --attach /tmp/<user>/fotoxx/filename.jpg
      zfree(newfile);
      
      if (err) {
         zmessageACK(mWin,ZTX("too many files"));
         goto cleanup;
      }
   }

   write_popup_text("write","COMPLETED");
   write_popup_text("close",0);

   emailsize[0] = maxw;                                                    //  save preferred size
   emailsize[1] = maxh;
   
   strcat(command," &");                                                   //  wait for email completion
   printf("system: %s \n",command);
   err = system(command);

cleanup:

   if (flist) {                                                            //  free memory
      for (ii = 0; flist[ii]; ii++) 
         zfree(flist[ii]);
      zfree(flist);
      flist = 0;
   }

   zdialog_free(zd);                                                       //  kill dialog
   return 0;
}


/**************************************************************************/

//  Synchronize files when fotoxx is installed for the first time or when
//  the image files have been moved or reorganized outside fotoxx.

struct tag_index_rec {
   char        *file;                                                      //  image filespec
   char        *tags;                                                      //  image tags
   char        *comms;                                                     //  image comments
   char        *capt;                                                      //  image caption
   char        imagedate[12], filedate[16];                                //  image date, file date
   char        stars;                                                      //  image rating, '0' to '5'
   char        update;                                                     //  flag, update is needed
};
tag_index_rec   *xrec_old, *xrec_new;


void m_syncfiles(GtkWidget *, const char *)                                //  v.11.04
{
   //  Rebuild thumbnail files.
   //  Process all image files within given top-level directory.

   int syncfiles_dialog_event(zdialog *zd, cchar *event);

   zdialog     *zd;
   char        *filespec1, *thumbdirk;
   char        *subdirk, *pp;
   cchar       *ppc;
   int         zstat, contx = 0, err, fcount = 0, Ffull = 0, Nth;

   zfuncs::F1_help_topic = "sync_files";

   if (! menulock(1)) return;

   if (! Fexiftool) {                                                      //  exiftool is required
      zmessageACK(mWin,Bexiftoolmissing);
      menulock(0);
      return;
   }
   
   zd = zdialog_new(ZTX("Synchronize Files"),mWin,Bproceed,Bcancel,null);
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","lab1","hb1",ZTX("Top Image Directory:"));
   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=1");
   zdialog_add_widget(zd,"entry","topdirk","hb2","??","scc=40|space=5");
   zdialog_add_widget(zd,"button","browse","hb2",Bbrowse,"space=5");
   zdialog_add_widget(zd,"hbox","hb3","dialog",0,"space=5");
   zdialog_add_widget(zd,"radio","rb1","hb3",ZTX("full rebuild"),"space=5");
   zdialog_add_widget(zd,"radio","rb2","hb3",ZTX("incremental"),"space=10");


   if (topdirk) zdialog_stuff(zd,"topdirk",topdirk);
   zdialog_stuff(zd,"rb1",0);
   zdialog_stuff(zd,"rb2",1);

   zdialog_run(zd,syncfiles_dialog_event);
   zstat = zdialog_wait(zd);

   if (zstat != 1) {
      zdialog_free(zd);
      menulock(0);
      return;
   }

   pp = zmalloc(maxfcc);   
   zdialog_fetch(zd,"topdirk",pp,maxfcc-1);                                //  get top directory
   if (topdirk) zfree(topdirk);
   topdirk = strdupz(pp,0,"topdirk");
   zfree(pp);

   zdialog_fetch(zd,"rb1",zstat);                                          //  full or incremental    v.11.02
   if (zstat) Ffull = 1;

   zdialog_free(zd);

   Ffuncbusy++;                                                            //  v.11.01

   if (Ffull)                                                              //  full rebuild        v.11.02
   {
      printf("deleting thumbnails \n");                                    //  v.11.05

      snprintf(command,ccc,"find \"%s\" -type d",topdirk);                 //  find directories under top directory
      contx = 0;

      while ((subdirk = command_output(contx,command)))
      {
         pp = strrchr(subdirk,'/');
         if (pp && strEqu(pp,"/.thumbnails"))                              //  /.../.thumbnails/ directory
         {
            snprintf(command,ccc,"rm -f -R \"%s\"",subdirk);               //  delete thumbnails   -f v.10.12
            err = system(command);
         }

         zfree(subdirk);
      }
   }

   printf("creating thumbnails \n");                                       //  v.11.05

   snprintf(command,ccc,"find \"%s\" -type d",topdirk);                    //  find directories under top directory
   contx = 0;

   while ((subdirk = command_output(contx,command)))
   {
      pp = strrchr(subdirk,'/');
      if (pp && strEqu(pp,"/.thumbnails")) {                               //  /.../.thumbnails/ directory
         zfree(subdirk);
         continue;                                                         //  skip
      }

      image_gallery(subdirk,"init");                                       //  get all image files in directory
      Nth = 0;
      filespec1 = image_gallery(0,"find",Nth);
      thumbdirk = 0;

      while (filespec1)
      {
         if (image_file_type(filespec1) == 2) 
         {
            fcount++;                                                      //  count images found

            if (! thumbdirk) {                                             //  when first image file found,
               thumbdirk = strdupz(subdirk,20);                            //    create .thumbnails if needed
               strcat(thumbdirk,"/.thumbnails");
               err = mkdir(thumbdirk,0751);
               zfree(thumbdirk);
            }

            pp = image_thumbfile(filespec1);                               //  find/update/create thumbnail
            if (pp) zfree(pp);

            snprintf(SB_text,199,"thumbnails: %d %s",fcount,filespec1);    //  update status bar      v.11.05
            zmainloop();
         }
         
         zfree(filespec1);
         filespec1 = image_gallery(0,"find",++Nth);                        //  next image file
      }

      zfree(subdirk);
   }
   
   printf("rebuild %d thumbnails completed \n",fcount);


   //  Rebuild search index file.
   //  Process all image files within given top-level directory.
   //  Works incrementally and is very fast after the first run.

   printf("rebuild search index \n");                                      //  v.11.05

   int  rebuild_search_index_dialog_event(zdialog *zd, cchar *event);
   int  rebuild_search_index_compare(cchar *rec1, cchar *rec2);

   FILE           *fid;
   char           **ppv, tagsbuff[tagrecl];
   char           *imagedate, *imagetags, *imagestars, *imagecomms, *imagecapt;
   cchar          *exifkeys[5] = { exif_date_key, exif_tags_key, exif_rating_key, 
                                   exif_comment_key, exif_caption_key };
   int            Nold, Nnew, orec, nrec, comp;
   struct tm      bdt;
   struct stat    statb;

   fcount += 100000;                                                       //  possible reduction

   xrec_old = (tag_index_rec *) zmalloc(fcount * sizeof(tag_index_rec),"xrec_old");
   xrec_new = (tag_index_rec *) zmalloc(fcount * sizeof(tag_index_rec),"xrec_new");
   
   //  if incremental mode, read current search index file and build "old list" of tags

   orec = Nold = 0;
   fid = 0;

   if (! Ffull) fid = fopen(search_index_file,"r");                        //  incremental mode

   if (fid) 
   {
      while (true)                                                         //  read current search index records
      {
         pp = fgets_trim(tagsbuff,tagrecl,fid);                            //  next record
         if (! pp) break;

         if (strnEqu(pp,"file: ",6))                                       //  start new file entry
         {
            if (++Nold == fcount) zappcrash("too many image files");
            orec = Nold - 1;
            xrec_old[orec].file = strdupz(pp+6,0,"xrec_old.file");
            xrec_old[orec].imagedate[0] = 0;
            xrec_old[orec].filedate[0] = 0;
            xrec_old[orec].tags = 0;
            xrec_old[orec].stars = '0';
            xrec_old[orec].comms = 0;
            xrec_old[orec].capt = 0;
            xrec_old[orec].update = '0';
         }
         
         if (strnEqu(pp,"date: ",6))
         {
            ppc = strField(pp,' ',2);
            if (ppc) strncpy0(xrec_old[orec].imagedate,ppc,12);
            ppc = strField(pp,' ',3);
            if (ppc) strncpy0(xrec_old[orec].filedate,ppc,16);
         }
         
         if (strnEqu(pp,"tags: ",6))
            xrec_old[orec].tags = strdupz(pp+6,0,"xrec_old.tags");
         
         if (strnEqu(pp,"stars: ",7))
            xrec_old[orec].stars = *(pp+7);

         if (strnEqu(pp,"comms: ",7))
            xrec_old[orec].comms = strdupz(pp+7,0,"xrec_old.comms");

         if (strnEqu(pp,"capt: ",6))
            xrec_old[orec].capt = strdupz(pp+6,0,"xrec_old.capt");         //  v.10.12
      }
      
      fclose(fid);
   }

   printf("%d current index records found \n",Nold);
   
   //  find all image files and create "new list" with no tags
   
   printf("find all image files and build index records \n");              //  v.11.05

   snprintf(command,ccc,"find \"%s\" -type d",topdirk);                    //  find directories under top directory
   contx = 0;
   Nnew = nrec = 0;

   while ((subdirk = command_output(contx,command)))
   {
      pp = strrchr(subdirk,'/');
      if (pp && strEqu(pp,"/.thumbnails"))                                 //  skip ./thumbnails directories
      {
         zfree(subdirk);
         continue;
      }

      image_gallery(subdirk,"init");                                       //  get all image files in directory
      Nth = 0;
      filespec1 = image_gallery(0,"find",Nth);

      while (filespec1)
      {
         if (image_file_type(filespec1) == 2)                              //  construct new tag record
         {
            err = stat(filespec1,&statb);
            if (err) continue;
            if (++Nnew == fcount) zappcrash("too many image files");
            nrec = Nnew - 1;
            xrec_new[nrec].file = strdupz(filespec1,0,"xrec_new.file");    //  filespec
            xrec_new[nrec].imagedate[0] = 0;                               //  image date = empty
            gmtime_r(&statb.st_mtime,&bdt);                                //  file date = yyyymmddhhmmss
            snprintf(xrec_new[nrec].filedate,16,"%04d%02d%02d%02d%02d%02d",
                     bdt.tm_year + 1900, bdt.tm_mon + 1, bdt.tm_mday,
                     bdt.tm_hour, bdt.tm_min, bdt.tm_sec);
            xrec_new[nrec].tags = 0;                                       //  tags = empty
            xrec_new[nrec].stars = '0';                                    //  stars = '0'
            xrec_new[nrec].comms = 0;                                      //  comments = empty
            xrec_new[nrec].capt = 0;                                       //  caption = empty        v.10.12
         }
      
         zfree(filespec1);
         filespec1 = image_gallery(0,"find",++Nth);                        //  next image file
      }

      zfree(subdirk);
   }
   
   printf("found %d image files \n",Nnew);

   //  merge and compare lists
   //  if filespecs match and have the same date, then old tags are OK
   
   printf("merging old and new index records \n");                         //  v.11.05

   HeapSort((char *) xrec_old,sizeof(tag_index_rec),Nold,rebuild_search_index_compare);
   HeapSort((char *) xrec_new,sizeof(tag_index_rec),Nnew,rebuild_search_index_compare);

   for (orec = nrec = 0; nrec < Nnew; )
   {
      xrec_new[nrec].update = '1';                                         //  assume update is needed

      if (orec == Nold) comp = +1;
      else comp = strcmp(xrec_old[orec].file, xrec_new[nrec].file);        //  compare filespecs
      
      if (comp > 0) nrec++;
      else if (comp < 0) orec++;
      else                                                                 //  matching filespecs
      {
         if (strEqu(xrec_new[nrec].filedate, xrec_old[orec].filedate))     //  and matching file dates
         {                                                                 //  copy data from old to new
            strncpy0(xrec_new[nrec].imagedate,xrec_old[orec].imagedate,12);
            xrec_new[nrec].tags = xrec_old[orec].tags;
            xrec_old[orec].tags = 0;
            xrec_new[nrec].stars = xrec_old[orec].stars;
            xrec_new[nrec].comms = xrec_old[orec].comms;
            xrec_old[orec].comms = 0;
            xrec_new[nrec].capt = xrec_old[orec].capt;
            xrec_old[orec].capt = 0;
            xrec_new[nrec].update = '0';                                   //  update is not needed
         }
         nrec++;
         orec++;
      }
   }

   for (orec = 0; orec < Nold; orec++)                                     //  release old list memory
   {
      zfree(xrec_old[orec].file);
      if (xrec_old[orec].tags) zfree(xrec_old[orec].tags);
   }

   zfree(xrec_old);
   xrec_old = 0;

   //  process entries needing update in new list
   //  get updated tags from image file EXIF data
   
   printf("updating new index records \n");                                //  v.11.05

   for (fcount = nrec = 0; nrec < Nnew; nrec++)
   {
      if (xrec_new[nrec].update == '0') continue;                          //  no update needed
      fcount++;                                                            //  running count of updates

      ppv = exif_get(xrec_new[nrec].file,exifkeys,5);                      //  get exif data
      imagedate = ppv[0];
      imagetags = ppv[1];
      imagestars = ppv[2];
      imagecomms = ppv[3];
      imagecapt = ppv[4];
      
      if (imagedate && strlen(imagedate)) {                                //  image date
         if (strlen(imagedate) > 9) imagedate[10] = 0;                     //  truncate to yyyy:mm:dd
         strcpy(xrec_new[nrec].imagedate,imagedate);
      }
      else strcpy(xrec_new[nrec].imagedate,"null ");
      
      if (imagetags && strlen(imagetags))                                  //  image tags
         xrec_new[nrec].tags = strdupz(imagetags,0,"xrec_new.tags");
      else xrec_new[nrec].tags = strdupz("null"tagdelim2,0,"xrec_new.tags");                          //  v.11.02

      if (imagestars && strlen(imagestars))                                //  image rating
         xrec_new[nrec].stars = *imagestars;

      if (imagecomms && strlen(imagecomms))                                //  image comments
         xrec_new[nrec].comms = strdupz(imagecomms,0,"xrec_new.comms");
      else xrec_new[nrec].comms = strdupz("null ",0,"xrec_new.comms");

      if (imagecapt && strlen(imagecapt))                                  //  image caption          v.10.12
         xrec_new[nrec].capt = strdupz(imagecapt,0,"xrec_new.capt");
      else xrec_new[nrec].capt = strdupz("null ",0,"xrec_new.capt");

      if (imagedate) zfree(imagedate);
      if (imagetags) zfree(imagetags);
      if (imagestars) zfree(imagestars);
      if (imagecomms) zfree(imagecomms);
      if (imagecapt) zfree(imagecapt);

      snprintf(SB_text,199,"search index: %d %s",                          //  update status bar      v.11.03
                           fcount,xrec_new[nrec].file);
      zmainloop();
   }

   fid = fopen(search_index_file,"w");                                     //  write new search index file
   if (! fid) zappcrash("cannot write tags file");                         //    with merged data

   for (nrec = 0; nrec < Nnew; nrec++)
   {
      fprintf(fid,"file: %s""\n",xrec_new[nrec].file);
      fprintf(fid,"date: %s  %s""\n", xrec_new[nrec].imagedate, xrec_new[nrec].filedate);
      fprintf(fid,"tags: %s""\n",xrec_new[nrec].tags);
      fprintf(fid,"stars: %c""\n",xrec_new[nrec].stars);
      fprintf(fid,"comms: %s""\n",xrec_new[nrec].comms);
      fprintf(fid,"capt: %s""\n",xrec_new[nrec].capt);
      fprintf(fid,"\n");
   }
   
   fclose(fid);
   
   for (nrec = 0; nrec < Nnew; nrec++)                                     //  release new list memory
   {
      zfree(xrec_new[nrec].file);
      if (xrec_new[nrec].tags) zfree(xrec_new[nrec].tags);
   }
   zfree(xrec_new);
   xrec_new = 0;

   Ffuncbusy--;
   menulock(0);
   Fsyncfiles = 1;                                                         //  v.11.04
   save_params();                                                          //  v.11.04
   *SB_text = 0;                                                           //  v.11.04
   
   printf("rebuild index completed, %d image files processed \n",Nnew);
   zmessageACK(mWin,ZTX("completed"));
   return;
}


//  dialog event function - file chooser for top image directory

int syncfiles_dialog_event(zdialog *zd, cchar *event)
{
   char     *pp;
   
   if (! strEqu(event,"browse")) return 0;
   pp = zgetfile1(ZTX("Select top image directory"),"folder",topdirk);     //  get topmost directory
   if (! pp) return 0;
   zdialog_stuff(zd,"topdirk",pp);
   zfree(pp);
   return 0;
}


//  sort compare function - compare tag record filespecs and return
//  filespec is at the start of the record
//   <0 | 0 | >0   for   file1 < | == | > file2

int rebuild_search_index_compare(cchar *rec1, cchar *rec2)
{
   char * file1 = ((tag_index_rec *) rec1)->file;
   char * file2 = ((tag_index_rec *) rec2)->file;
   return strcmp(file1,file2);
}


/**************************************************************************/

//  dump memory usage by category to STDOUT

void m_memory_usage(GtkWidget *, cchar *)
{
   zmalloc_report();
   return;
}


/**************************************************************************
   plugin menu functions
**************************************************************************/

//  process plugin menu selection
//  execute correspinding command using current image file

void  m_run_plugin(GtkWidget *, cchar *menu)                               //  v.11.03
{
   int         ii, jj, err;
   char        *pp = 0, pluginfile[200];
   PXM         *pxmtemp;
   
   zfuncs::F1_help_topic = "plugins";
   
   for (ii = 0; ii < Nplugins; ii++)                                       //  search plugins for menu name
   {
      pp = strstr(plugins[ii]," = ");
      if (! pp) continue;
      *pp = 0;
      jj = strEqu(plugins[ii],menu);
      *pp = ' ';
      if (jj) break;
   }
   
   if (ii == Nplugins) {
      zmessageACK(mWin,"plugin menu not found %s",menu);
      return;
   }

   pp += 3;   
   if (strlen(pp) < 3) {
      zmessageACK(mWin,"no plugin command");
      return;
   }

   strncpy0(command,pp,ccc);                                               //  corresp. command
   strTrim2(command);

   if (! edit_setup(menu,0,0)) return;                                     //  setup edit, no preview, no select area

   snprintf(pluginfile,199,"%s/plugfile.tif",get_zuserdir());              //  /home/user/.fotoxx/plugfile.tif
   
   TIFFwrite(E1pxm16,pluginfile);                                          //  E1 >> plugin_file

   strcat(command," ");                                                    //  construct: command /.../plugin_file &
   strcat(command,pluginfile);
   printf("plugin: %s \n",command);

   err = system(command);                                                  //  execute plugin command
   if (err) {
      zmessageACK(mWin,"plugin command error");
      edit_cancel();
      return;
   }
   
   pxmtemp = TIFFread(pluginfile);                                         //  read command output file
   if (! pxmtemp) {
      zmessageACK(mWin,"plugin failed");
      edit_cancel();
      return;
   }
   
   PXM_free(E3pxm16);                                                      //  plugin_file >> E3
   if (pxmtemp->bpc == 16) E3pxm16 = pxmtemp;
   else {
      E3pxm16 = PXM_convbpc(pxmtemp);
      PXM_free(pxmtemp);
   }

   Fmodified = 1;
   edit_done();
   return;
}   


//  edit plugins menu

void  m_edit_plugins(GtkWidget *, cchar *)                                 //  v.11.03
{
   int   edit_plugins_event(zdialog *zd, cchar *event);

   int         ii;
   char        *pp;
   zdialog     *zd;
   
   zfuncs::F1_help_topic = "plugins";
   
   zd = zdialog_new("Edit Plugins",mWin,ZTX("Add"),ZTX("Remove"),Bdone,null);
   zdialog_add_widget(zd,"hbox","hbm","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labm","hbm","menu name","space=5");
   zdialog_add_widget(zd,"comboE","menu","hbm",0,"space=5");
   zdialog_add_widget(zd,"hbox","hbc","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labc","hbc","command","space=5");
   zdialog_add_widget(zd,"entry","command","hbc",0,"space=5|expand");

   for (ii = 0; ii < Nplugins; ii++)                                       //  stuff combo box with available
   {                                                                       //     plugin menu names
      pp = strstr(plugins[ii]," = ");
      if (! pp) continue;
      *pp = 0;
      zdialog_cb_app(zd,"menu",plugins[ii]);
      *pp = ' ';
   }
   
   zdialog_run(zd,edit_plugins_event);
   return;
}


//  dialog event function

int  edit_plugins_event(zdialog *zd, cchar *event)
{
   int      ii, jj, cc, zstat;
   char     menu[40], *pp = 0;
   
   zdialog_fetch(zd,"menu",menu,40);                                       //  selected menu
   zdialog_fetch(zd,"command",command,ccc);
   
   if (strEqu(event,"menu")) 
   {
      for (ii = 0; ii < Nplugins; ii++)
      {
         pp = strstr(plugins[ii]," = ");                                   //  find corresp. plugin record
         if (! pp) continue;
         *pp = 0;
         jj = strEqu(plugins[ii],menu);
         *pp = ' ';
         if (jj) break;
      }
      
      if (ii == Nplugins) return 0;

      pp += 3;
      if (strlen(pp) < 3) return 0;
      strncpy0(command,pp,ccc);
      zdialog_stuff(zd,"command",command);                                 //  stuff corresp. command in dialog
      
      return 0;
   }
   
   zstat = zd->zstat;
   if (! zstat) return 0;
   
   if (zstat == 1)                                                         //  add new plugin
   {
      if (strlen(menu) < 3 || strlen(command) < 3) return 0;
      if (Nplugins == maxplugins) {
         zmessageACK(mWin,"too many plugins");
         return 0;
      }
      ii = Nplugins;                                                       //  add plugin record
      cc = strlen(menu) + strlen(command) + 5;
      plugins[ii] = zmalloc(cc,"plugins");                                 //  format: menu = command
      strcpy(plugins[ii],menu);
      strcat(plugins[ii]," = ");
      strcat(plugins[ii],command);
      Nplugins++;

      zmessageACK(mWin,ZTX("Restart Fotoxx to update plugin menu"));
   }
   
   if (zstat == 2)                                                         //  remove current plugin
   {
      for (ii = 0; ii < Nplugins; ii++)
      {
         pp = strstr(plugins[ii]," = ");                                   //  find corresp. plugin record
         if (! pp) continue;
         *pp = 0;
         jj = strEqu(plugins[ii],menu);
         *pp = ' ';
         if (jj) break;
      }
      
      if (ii == Nplugins) return 0;
      
      Nplugins--;                                                          //  remove plugin record
      for (jj = ii; jj < Nplugins; jj++)
         plugins[jj] = plugins[jj+1];
      
      zmessageACK(mWin,ZTX("Restart Fotoxx to update plugin menu"));
   }
   
   if (zstat == 3) {                                                       //  done
      zdialog_free(zd);
      return 0;
   }
   
   return 0;
}


/**************************************************************************

   edit transaction and thread support functions

   edit transaction management
      edit_setup()                     start new edit - copy E3 > E1
      edit_cancel()                    cancel edit - E1 > E3, delete E1
      edit_done()                      commit edit - add to undo stack
      edit_undo()                      undo edit - E1 > E3
      edit_redo()                      redo edit - run thread again
      edit_fullsize()                  convert preview to full-size pixmaps

   main level thread management
      start_thread(func,arg)           start thread running
      signal_thread()                  signal thread that work is pending
      wait_thread_idle()               wait for pending work complete
      wrapup_thread(command)           wait for exit or command thread exit

   thread function
      thread_idle_loop()               wait for pending work, exit if commanded
      thread_exit()                    exit thread unconditionally
      
   thread_status (thread ownership
      0     no thread is running
      1     thread is running and idle (no work)
      2     thread is working
      0     thread has exited

   thread_command (main program ownership)
      0     idle, no work pending
      8     exit when pending work is done
      9     exit now, unconditionally

   thread_pend       work requested counter
   thread_done       work done counter
   thread_hiwater    high water mark
   edit_action       done/cancel/undo/redo in progress

***************************************************************************/

int      thread_command = 0, thread_status = 0;
int      thread_pend = 0, thread_done = 0, thread_hiwater = 0;
int      edit_action = 0;


/**************************************************************************

  Setup for a new edit transaction
  Create E1 (edit input) and E3 (edit output) pixmaps from
  previous edit output (Fpxm16) or image file (new Fpxm16).

  uprev      0     edit full-size image
             1     edit preview image unless select area exists

  uarea      0     select_area is invalid and will be deleted (e.g. rotate)
             1     select_area not used but remains valid (e.g. red-eye)
             2     select_area is used and remains valid (e.g. flatten)

***************************************************************************/

int edit_setup(cchar *func, int uprev, int uarea)
{
   int      yn;

   if (! curr_file) return 0;                                              //  no image file
   if (! menulock(1)) return 0;                                            //  lock menu
   
   if (! Fexiftool && ! Fexifwarned) {
      zmessageACK(mWin,ZTX("exiftool is not installed \n"                  //  warn if starting to edit
                           "edited images will lose EXIF data"));          //    and exiftool is missing  v.9.9
      Fexifwarned = 1;
   }

   if (Pundo > maxedits-2) {                                               //  v.10.2
      zmessageACK(mWin,ZTX("Too many edits, please save image"));
      menulock(0);
      return 0;
   }

   if (uarea == 0 && sa_stat) {                                            //  select area will be lost, warn user
      yn = zmessageYN(mWin,ZTX("Select area cannot be kept.\n"
                               "Continue?"));
      if (! yn) {
         menulock(0);
         return 0;
      }
      sa_delete();
      zdialog_free(zdsela);
   }
   
   if (uprev && uarea == 2 && sa_stat && ! Factivearea) {                  //  select area exists and can be used,
      yn = zmessageYN(mWin,ZTX("Select area not active.\n"                 //    but not active, ask user    v.10.1
                               "Continue?"));
      if (! yn) {
         menulock(0);
         return 0;
      }
   }

   Fpreview = 0;                                                           //  use preview image if supported
   if (uprev && ! (uarea == 2 && Factivearea)) {                           //    and select area will not be used
      sa_show(0);                                                          //  hide area if present
      freeMouse();                                                         //  and disconnect mouse     v.10.12
      Fpreview = 1;
      if (Fzoom) {
         Fzoom = 0;                                                        //  bugfix, mpaint() req.    v.11.01
         mwpaint();
      }
   }

   if (Fpreview && Fshowarea) {                                            //  hide select area during preview mode edit
      sa_show(0);                                                          //    and bring it back when done   v.9.7
      Fshowarea = 1;
   }
   
   mutex_lock(&Fpixmap_lock);                                              //  lock pixmaps

   if (! Fpxm16) Fpxm16 = f_load(curr_file,16);                            //  create Fpxm16 if not already
   if (! Fpxm16) {
      mutex_unlock(&Fpixmap_lock);                                         //  should not happen      v.10.9
      menulock(0);
      printf("f_load (16) failed: %s \n",curr_file);                       //  v.11.05
      return 0;
   }

   if (Fpreview)                                                           //  edit pixmaps are window-size
      E1pxm16 = PXM_rescale(Fpxm16,dww,dhh);                               //  E1pxm16 = Fpxm16 scaled to window
   else E1pxm16 = PXM_copy(Fpxm16);                                        //  edit pixmaps are full-size
   E3pxm16 = PXM_copy(E1pxm16);                                            //  E1 >> E3

   E1ww = E3ww = E1pxm16->ww;
   E1hh = E3hh = E1pxm16->hh;
   
   if (Pundo == 0) {
      edit_function = "initial";                                           //  initial image >> undo stack    v.10.2
      save_undo();
   }

   edit_function = func;                                                   //  set current edit function    v.10.2
   Fmodified = 0;                                                          //  image not modified yet
   thread_command = thread_status = 0;                                     //  no thread running
   thread_pend = thread_done = thread_hiwater = 0;                         //  no work pending or done

   mutex_unlock(&Fpixmap_lock);
   Irefresh++;                                                             //  v.10.12
   mwpaint();                                                              //  mwpaint() not mwpaint2()
   return 1;
}


/**************************************************************************/

//  process edit cancel

void edit_cancel()
{
   if (edit_action) return;
   edit_action++;

   wrapup_thread(9);                                                       //  tell thread to quit, wait

   if (zdedit) {
      zdialog_free(zdedit);                                                //  kill dialog
      if (Espcdat) zfree(Espcdat);                                         //  free spline curves data    v.11.01
      Espcdat = 0;
   }
   
   mutex_lock(&Fpixmap_lock);
   PXM_free(E1pxm16);                                                      //  free edit pixmaps E1, E3
   PXM_free(E3pxm16);
   PXM_free(ERpxm16);                                                      //  free redo copy    v.10.3
   E1ww = E3ww = ERww = 0;

   Fmodified = Fpreview = 0;                                               //  reset flags
   Ntoplines = Nptoplines;                                                 //  no overlay lines
   freeMouse();                                                            //  v.11.04
   mutex_unlock(&Fpixmap_lock);
   menulock(0);                                                            //  unlock menu
   mwpaint2();                                                             //  refresh window
   edit_action = 0;
   return;
}   


/**************************************************************************/

//  process edit dialog [done]  
//  E3pxm16 >> Fpxm16 >> Fpxm8

void edit_done()
{
   if (edit_action) return;
   edit_action++;

   wait_thread_idle();                                                     //  bugfix           v.10.2

   if (Fpreview && Fmodified) {
      Fzoom = 0;
      edit_fullsize();                                                     //  update full image
   }

   wrapup_thread(8);                                                       //  tell thread to exit

   if (zdedit) {
      zdialog_free(zdedit);                                                //  kill dialog
      if (Espcdat) zfree(Espcdat);                                         //  free spline curves data    v.11.01
      Espcdat = 0;
   }

   mutex_lock(&Fpixmap_lock);

   if (Fmodified) {
      PXM_free(Fpxm16);
      Fpxm16 = PXM_copy(E3pxm16);                                          //  E3 >> Fpxm16
      PXM_free(Fpxm8);
      Fpxm8 = PXM_convbpc(Fpxm16);                                         //  Fpxm16 >> Fpxm8
      Fww = Fpxm8->ww;
      Fhh = Fpxm8->hh;
      Pundo++;
      Pumax = Pundo;
      save_undo();                                                         //  save next undo state
   }

   PXM_free(E1pxm16);                                                      //  free edit pixmaps
   PXM_free(E3pxm16);
   PXM_free(ERpxm16);                                                      //  free redo copy    v.10.3
   E1ww = E3ww = ERww = 0;
   
   Fmodified = Fpreview = 0;                                               //  reset flags
   Ntoplines = Nptoplines;                                                 //  no overlay lines
   freeMouse();                                                            //  v.11.04
   mutex_unlock(&Fpixmap_lock);
   menulock(0);                                                            //  unlock menu
   mwpaint2();                                                             //  update window
   edit_action = 0;
   return;
}


/**************************************************************************/

//  edit undo, redo, reset functions

void edit_undo()
{
   if (! Fmodified) return;
   if (thread_status == 2) return;                                         //  bugfix, thread busy      v.10.2
   if (edit_action) return;
   edit_action++;
   
   mutex_lock(&Fpixmap_lock);
   PXM_free(ERpxm16);                                                      //  make redo copy    v.10.3
   ERpxm16 = E3pxm16;
   ERww = E3ww;
   ERhh = E3hh;
   E3pxm16 = PXM_copy(E1pxm16);                                            //  restore initial image status
   E3ww = E1ww;
   E3hh = E1hh;
   Fmodified = 0;                                                          //  reset image modified status
   mutex_unlock(&Fpixmap_lock);

   mwpaint2();                                                             //  refresh window      v.9.8
   edit_action = 0;
   return;
}


void edit_redo()
{
   if (! ERpxm16) return;
   if (edit_action) return;
   edit_action++;
   PXM_free(E3pxm16);                                                      //  edit image = redo copy    v.10.3
   E3pxm16 = ERpxm16;
   E3ww = ERww;
   E3hh = ERhh;
   ERpxm16 = 0;
   Fmodified = 1;
   mwpaint2();
   edit_action = 0;
   return;
}


void edit_reset()                                                          //  new  v.10.3
{
   if (! Fmodified) return;
   if (thread_status == 2) return;                                         //  thread busy
   if (edit_action) return;
   edit_action++;

   mutex_lock(&Fpixmap_lock);
   PXM_free(ERpxm16);                                                      //  no redo copy
   PXM_free(E3pxm16);
   E3pxm16 = PXM_copy(E1pxm16);                                            //  restore initial image
   E3ww = E1ww;
   E3hh = E1hh;
   Fmodified = 0;                                                          //  reset image modified status
   mutex_unlock(&Fpixmap_lock);

   mwpaint2();                                                             //  refresh window      v.9.8
   edit_action = 0;
   return;
}


void edit_zapredo()
{
   PXM_free(ERpxm16);                                                      //  no redo copy
   return;
}


/**************************************************************************/

//  Convert from preview mode (window-size pixmaps) to full-size pixmaps.

void edit_fullsize()
{
   if (! Fpreview) return;
   Fpreview = 0;

   mutex_lock(&Fpixmap_lock);
   PXM_free(E1pxm16);                                                      //  free preview pixmaps
   PXM_free(E3pxm16);
   E1pxm16 = PXM_copy(Fpxm16);                                             //  make full-size pixmaps
   E3pxm16 = PXM_copy(Fpxm16);
   E1ww = E3ww = Fww;
   E1hh = E3hh = Fhh;
   mutex_unlock(&Fpixmap_lock);

   signal_thread();                                                        //  signal thread to do edit
   wait_thread_idle();
   return;
}


/**************************************************************************
      undo / redo toolbar buttons
***************************************************************************/

//  [undo] menu function - reinstate previous edit in undo/redo stack

void m_undo(GtkWidget *, cchar *)
{
   if (zdedit) {                                                           //  v.10.2
      zdialog_send_event(zdedit,"undo");
      return;
   }
   if (Pundo == 0) return;
   if (! menulock(1)) return;
   Pundo--;
   load_undo();
   menulock(0);
   return;
}


//  [redo] menu function - reinstate next edit in undo/redo stack

void m_redo(GtkWidget *, cchar *)
{
   if (zdedit) {                                                           //  v.10.2
      zdialog_send_event(zdedit,"redo");
      return;
   }
   if (Pundo == Pumax) return;
   if (! menulock(1)) return;
   Pundo++;
   load_undo();
   menulock(0);
   return;
}


//  undo all edits of the current image

void undo_all()                                                            //  v.11.04
{
   if (! menulock(1)) return;
   while (Pundo) {
      load_undo();
      Pundo--;
   }
   menulock(0);
   return;
}


//  Save Fpxm16 to undo/redo file stack
//  stack position = Pundo

void save_undo()
{
   char     *pp, buff[24];
   int      fid, cc, cc2;
   
   pp = strstr(undo_files,"_undo_");
   if (! pp) zappcrash("undo/redo stack corrupted 1");
   snprintf(pp+6,3,"%02d",Pundo);
   
   fid = open(undo_files,O_WRONLY|O_CREAT|O_TRUNC,0640);
   if (! fid) zappcrash("undo/redo stack corrupted 2");

   snprintf(buff,24," %05d %05d fotoxx ",Fww,Fhh);
   cc = write(fid,buff,20);
   if (cc != 20) zappcrash("undo/redo stack corrupted 3");
   
   cc = Fww * Fhh * 6;
   cc2 = write(fid,Fpxm16->bmp,cc);
   if (cc2 != cc) zappcrash("undo/redo stack corrupted 4");

   close(fid);

   pvlist_replace(editlog,Pundo,edit_function);                            //  save log of edits done    v.10.2
   return;
}


//  Load Fpxm16 from undo/redo file stack
//  stack position = Pundo

void load_undo()
{
   char     *pp, buff[24], fotoxx[8];
   int      fid, ww, hh, cc, cc2;

   pp = strstr(undo_files,"_undo_");
   if (! pp) zappcrash("undo/redo stack corrupted 1");
   snprintf(pp+6,3,"%02d",Pundo);
   
   fid = open(undo_files,O_RDONLY);
   if (! fid) zappcrash("undo/redo stack corrupted 2");
   
   *fotoxx = 0;
   cc = read(fid,buff,20);
   sscanf(buff," %d %d %8s ",&ww, &hh, fotoxx);
   if (! strEqu(fotoxx,"fotoxx")) zappcrash("undo/redo stack corrupted 4");

   mutex_lock(&Fpixmap_lock);

   PXM_free(Fpxm16);
   Fpxm16 = PXM_make(ww,hh,16);
   cc = ww * hh * 6;
   cc2 = read(fid,Fpxm16->bmp,cc);
   if (cc2 != cc) zappcrash("undo/redo stack corrupted 5");
   close(fid);
   
   PXM_free(Fpxm8);
   Fpxm8 = PXM_convbpc(Fpxm16);
   Fww = ww;
   Fhh = hh;
   
   edit_function = pvlist_get(editlog,Pundo);                              //  last edit func not un-done   v.10.2

   mutex_unlock(&Fpixmap_lock);
   mwpaint2();
   return;
}


/**************************************************************************/

//  ask user if modified image should be kept or discarded
//  returns: 0 = mods discarded,  1 = do not discard mods

int mod_keep()
{
   int      keep = 0;                                                      //  minor bugfix     v.11.05

   if (Fmodified) keep = 1;                                                //  edit in progress
   if (Ftagschanged) keep = 1;                                             //  tags etc. were changed
   if (Fsaved != Pundo) keep = 1;                                          //  curr. edits are not saved
   if (keep == 0) return 0;                                                //  no mods 
   if (! zmessageYN(mWin,ZTX("Discard modifications?"))) return 1;         //  do not discard mods
   undo_all();                                                             //  discard mods
   return 0;
}


/**************************************************************************/

//  menu lock/unlock - some functions must not run concurrently
//  returns 1 if success, else 0

int menulock(int lock)
{
   if (! lock && ! Fmenulock) zappcrash("menu lock error");
   if (lock && Fmenulock) {
      printf("menu locked \n");                                            //  v.10.8
      return 0;
   }
   if (lock) Fmenulock++;
   else Fmenulock--;
   return 1;
}


/**************************************************************************/

//  start thread that does the edit work

void start_thread(threadfunc func, void *arg)
{
   thread_status = 1;                                                      //  thread is running
   thread_command = thread_pend = thread_done = thread_hiwater = 0;        //  nothing pending
   start_detached_thread(func,arg);
   return;
}


//  signal thread that work is pending

void signal_thread()
{
   edit_zapredo();                                                         //  reset redo copy
   if (thread_status > 0) thread_pend++;
   return;
}


//  wait for edit thread to complete pending work and become idle

void wait_thread_idle()
{
   while (thread_status && thread_pend > thread_done)
   {
      zmainloop();
      zsleep(0.01);
   }
   
   return;
}


//  wait for thread exit or command thread exit
//  command = 0    wait for normal completion
//            8    finish pending work and exit
//            9    quit, exit now

void wrapup_thread(int command)
{
   thread_command = command;                                               //  tell thread to quit or finish

   while (thread_status > 0)                                               //  wait for thread to finish
   {                                                                       //    pending work and exit
      zmainloop();
      zsleep(0.01);
   }

   return;
}


//  called only from edit threads
//  idle loop - wait for work request or exit command

void thread_idle_loop()
{
   if (thread_status == 2) Ffuncbusy--;                                    //  v.11.05
   thread_status = 1;                                                      //  status = idle
   thread_done = thread_hiwater;                                           //  work done = high-water mark

   while (true)
   {
      if (thread_command == 9) thread_exit();                              //  quit now command
      if (thread_command == 8)                                             //  finish work and exit
         if (thread_pend <= thread_done) thread_exit();
      if (thread_pend > thread_done) break;                                //  wait for work request
      zsleep(0.01);
   }
   
   thread_hiwater = thread_pend;                                           //  set high-water mark
   thread_status = 2;                                                      //  thread is working
   Ffuncbusy++;                                                            //  v.11.05
   return;                                                                 //  loop to thread
}


//  exit thread unconditionally, called only from edit threads

void thread_exit()
{
   if (thread_status == 2) Ffuncbusy--;                                    //  v.11.05
   thread_pend = thread_done = thread_hiwater = 0;
   thread_status = 0;
   pthread_exit(0);                                                        //  "return" cannot be used here
}


/**************************************************************************/

//  edit support functions for working threads (per processor core)

void start_wt(threadfunc func, void *arg)                                  //  start and increment busy count
{
   zadd_locked(wthreads_busy,+1);
   zadd_locked(Ffuncbusy,+1);                                              //  v.11.01
   start_detached_thread(func,arg);
   return;
}


void exit_wt()                                                             //  decrement busy count and exit
{
   zadd_locked(Ffuncbusy,-1);                                              //  v.11.01
   zadd_locked(wthreads_busy,-1);
   pthread_exit(0);                                                        //  "return" cannot be used here  v.9.4
}


void wait_wts()                                                            //  wait for all working threads done
{
   while (wthreads_busy) 
   {
      zmainloop();
      zsleep(0.01);
   }

   return;
}


/**************************************************************************
      other support functions
***************************************************************************/

//  help menu function

void m_help(GtkWidget *, cchar *menu)
{
   if (strEqu(menu,ZTX("About"))) 
      zmessageACK(mWin,"%s \n%s \n%s \n%s \n\n%s \n\n%s",
        fversion,flicense,fhomepage,fcontact,fcredits,ftranslators);
      
   if (strEqu(menu,ZTX("User Guide"))) 
      showz_userguide();

   if (strEqu(menu,ZTX("Help")))                                           //  toolbar button   v.10.4
      showz_userguide();

   if (strEqu(menu,"README"))
      showz_readme();

   if (strEqu(menu,ZTX("Change Log")))
      showz_changelog();

   if (strEqu(menu,ZTX("Translate")))
      showz_translations();
      
   if (strEqu(menu,ZTX("Home Page")))
      showz_html(fhomepage);

   return;
}


/**************************************************************************/

//  table for loading and saving adjustable parameters between sessions

typedef struct {
   char     name[20];
   char     type[12];
   int      count;
   void     *location;
}  param;

#define Nparms 25

param paramTab[Nparms] = {
//     name                 type        count   location
{     "last file",         "char",        1,    &last_file                 },
{     "top directory",     "char",        1,    &topdirk                   },
{     "sync_files",        "int",         1,    &Fsyncfiles                },
{     "window width",      "int",         1,    &Dww                       },
{     "window height",     "int",         1,    &Dhh                       },
{     "thumbsize",         "int",         1,    &image_navi::thumbsize     },
{     "lens name",         "char",        4,    &lens4_name                },
{     "lens mm",           "double",      4,    &lens4_mm                  },
{     "lens bow",          "double",      4,    &lens4_bow                 },
{     "current lens",      "int",         1,    &curr_lens                 },
{     "grid spacing",      "int",         2,    &gridspace                 },
{     "grid counts",       "int",         2,    &gridcount                 },
{     "trim size",         "int",         2,    &trimsize                  },
{     "trim buttons",      "char",        6,    &trimbuttons               },
{     "trim ratios",       "char",        6,    &trimratios                },
{     "edit resize",       "int",         2,    &editresize                },
{     "batch resize",      "int",         2,    &batchresize               },
{     "e-mail resize",     "int",         2,    &emailsize                 },
{     "annotate font",     "char",        1,    &annotate_font             },
{     "annotate text",     "char",        1,    &annotate_text             },
{     "annotate color",    "char",        3,    &annotate_color            },
{     "annotate trans",    "int",         3,    &annotate_trans            },
{     "annotate towidth",  "int",         1,    &annotate_towidth          },
{     "annotate angle",    "double",      1,    &annotate_angle            },
{     "slideshow music",   "char",        1,    &ss_musicfile              }  };


//  save parameters to file /home/<user>/.fotoxx/parameters

void save_params()                                                         //  new v.10.9
{
   FILE        *fid;
   char        buff[1050], text[1000];                                     //  limit for character data cc
   char        *name, *type;
   int         count;
   void        *location;
   char        **charloc;
   int         *intloc;
   double      *doubleloc;
   
   gtk_window_get_size(MWIN,&Dww,&Dhh);                                    //  prevent shrinkage
   
   if (curr_file) last_file = curr_file;                                   //  save for next session     v.11.04
   else last_file = 0;

   snprintf(buff,199,"%s/parameters",get_zuserdir());                      //  open output file
   fid = fopen(buff,"w");
   if (! fid) return;
   
   fprintf(fid,"%s \n",fversion);                                          //  write fotoxx version
   
   for (int ii = 0; ii < Nparms; ii++)                                     //  write table of state data
   {
      name = paramTab[ii].name;
      type = paramTab[ii].type;
      count = paramTab[ii].count;
      location = paramTab[ii].location;
      charloc = (char **) location;
      intloc = (int *) location;
      doubleloc = (double *) location;

      fprintf(fid,"%-20s  %-8s  %02d  ",name,type,count);                  //  write parm name, type, count

      for (int kk = 0; kk < count; kk++)                                   //  write "value" "value" ...
      {
         if (strEqu(type,"char")) {
            if (! *charloc) printf("bad param data %s %d \n",name,kk);     //  bugfix v.10.11.1
            if (! *charloc) break;
            repl_1str(*charloc++,text,"\n","\\n");                         //  replace newlines with "\n"   v.10.11
            fprintf(fid,"  \"%s\"",text);
         }
         if (strEqu(type,"int"))
            fprintf(fid,"  \"%d\"",*intloc++);

         if (strEqu(type,"double"))
            fprintf(fid,"  \"%.2f\"",*doubleloc++);
      }

      fprintf(fid,"\n");                                                   //  write EOR
   }
   
   fprintf(fid,"\n");
   fclose(fid);                                                            //  close file

   snprintf(buff,199,"%s/recent_files",get_zuserdir());                    //  open file for recent files list
   fid = fopen(buff,"w");
   if (! fid) return;
   
   for (int ii = 0; ii < Nrecentfiles; ii++)                               //  save list of recent files
      if (recentfiles[ii])
         fprintf(fid,"%s \n",recentfiles[ii]);
   
   fclose(fid);

   snprintf(buff,199,"%s/plugins",get_zuserdir());                         //  open file for plugins     v.11.03
   fid = fopen(buff,"w");
   if (! fid) return;
   
   for (int ii = 0; ii < Nplugins; ii++)                                   //  save plugins
      if (plugins[ii])
         fprintf(fid,"%s \n",plugins[ii]);
   
   fclose(fid);

   return;
}


//  load parameters from file /home/<user>/.fotoxx/parameters

void load_params()                                                         //  new v.10.9
{
   FILE        *fid;
   int         ii, err, pcount, Idata;
   double      Rdata;
   char        buff[1050], text[1000], *pp;
   char        name[20], type[12], count[8], data[1000];
   void        *location;
   char        **charloc;
   int         *intloc;
   double      *doubleloc;
   
   for (ii = 0; ii < Nparms; ii++)                                         //  set string parameters to "undefined"
   {                                                                       //  v.10.11
      if (strNeq(paramTab[ii].type,"char")) continue;
      charloc = (char **) paramTab[ii].location;
      pcount = paramTab[ii].count;
      for (int jj = 0; jj < pcount; jj++)
         *charloc++ = strdupz("undefined",20);
   }

   snprintf(buff,499,"%s/parameters",get_zuserdir());                      //  open parameters file
   fid = fopen(buff,"r");
   if (! fid) return;

   pp = fgets_trim(buff,499,fid,1);                                        //  get last fotoxx version
   if (pp && strNeq(pp,fversion))                                          //  bugfix   v.10.11
      printf("version change: %s %s \n",pp,fversion);
   
   while (true)                                                            //  read parameters
   {
      pp = fgets_trim(buff,999,fid,1);
      if (! pp) break;
      if (*pp == '#') continue;                                            //  comment          v.10.10
      if (strlen(pp) < 40) continue;                                       //  rubbish          v.10.10

      err = 0;

      strncpy0(name,pp,20);                                                //  parm name
      strTrim2(name);
      
      strncpy0(type,pp+22,8);                                              //  parm type
      strTrim2(type);
      
      strncpy0(count,pp+32,4);                                             //  parm count
      strTrim2(count);
      err = convSI(count,pcount);
      
      strncpy0(data,pp+38,1000);                                           //  parm value(s)
      strTrim2(data);
      
      for (ii = 0; ii < Nparms; ii++)                                      //  match file record to param table
      {
         if (strNeq(name,paramTab[ii].name)) continue;                     //  parm name
         if (strNeq(type,paramTab[ii].type)) continue;                     //  parm type
         if (paramTab[ii].count != pcount) continue;                       //  parm count

         location = paramTab[ii].location;
         charloc = (char **) location;
         intloc = (int *) location;
         doubleloc = (double *) location;
         
         for (int kk = 1; kk <= pcount; kk++)
         {
            pp = (char *) strField(data,' ',kk);
            if (! pp) break;

            if (strEqu(type,"char")) {
               repl_1str(pp,text,"\\n","\n");                              //  replace "\n" with real newlines  v.10.11
               *charloc++ = strdupz(text,0,"parameters");                  //  v.11.04
            }

            if (strEqu(type,"int")) {
               err = convSI(pp,Idata);
               if (err) continue;
               *intloc++ = Idata;
            }

            if (strEqu(type,"double")) {
               err = convSD(pp,Rdata);
               if (err) continue;
               *doubleloc++ = Rdata;
            }
         }
      }
   }

   fclose(fid);

   for (ii = 0; ii < Nrecentfiles; ii++)                                   //  recent image file list = empty
      recentfiles[ii] = 0;

   snprintf(buff,499,"%s/recent_files",get_zuserdir());                    //  open recent files file
   fid = fopen(buff,"r");

   if (fid)
   {
      for (ii = 0; ii < Nrecentfiles; ii++)                                //  read list of recent files
      {
         pp = fgets_trim(buff,499,fid,1);
         if (! pp) break;
         if (*pp == '/') recentfiles[ii] = strdupz(buff,0,"recentfile");
      }

      fclose(fid);
   }

   for (ii = 0; ii < maxplugins; ii++)                                     //  plugins list = empty      v.11.03
      plugins[ii] = 0;
   
   plugins[0] = strdupz("Gimp = gimp");                                    //  if empty, default Gimp plugin
   Nplugins = 1;

   snprintf(buff,499,"%s/plugins",get_zuserdir());                         //  open plugins file         v.11.03
   fid = fopen(buff,"r");

   if (fid)
   {   
      for (ii = 0; ii < maxplugins; ii++)                                  //  read list of plugins
      {
         pp = fgets_trim(buff,499,fid,1);
         if (! pp) break;
         plugins[ii] = strdupz(buff,0,"plugins");
      }

      fclose(fid);
      Nplugins = ii;
   }

   return;
}


/**************************************************************************/

//  Compute the mean brightness of all pixel neighborhoods,                //  new v.9.6
//  using a Guassian or a flat distribution for the weightings. 
//  If a select area is active, only inside pixels are calculated.
//  The flat method is 10-100x faster.

int         brhood_radius;
double      brhood_kernel[200][200];                                       //  up to radius = 99
float       *brhood_brightness = 0;                                        //  neighborhood brightness per pixel
char        brhood_method;                                                 //  g = gaussian, f = flat distribution


void brhood_calc(int radius, char method)
{
   void * brhood_wthread(void *arg);

   int      rad, rad2, dx, dy, cc, ii;
   double   kern;
   
   brhood_radius = radius;
   brhood_method = method;

   if (brhood_method == 'g')                                               //  compute Gaussian kernel
   {                                                                       //  (not currently used   v.11.02)
      rad = brhood_radius;
      rad2 = rad * rad;

      for (dy = -rad; dy <= rad; dy++)
      for (dx = -rad; dx <= rad; dx++)
      {
         if (dx*dx + dy*dy <= rad2)                                        //  cells within radius
            kern = exp( - (dx*dx + dy*dy) / rad2);
         else kern = 0;                                                    //  outside radius
         brhood_kernel[dy+rad][dx+rad] = kern;
      }
   }

   if (brhood_brightness) zfree(brhood_brightness);                        //  allocate memory for pixel map
   cc = E1ww * E1hh * sizeof(float);
   brhood_brightness = (float *) zmalloc(cc,"brhood");
   memset(brhood_brightness,0,cc);

   if (Factivearea) SB_goal = sa_Npixel;                                   //  set up progress tracking
   else SB_goal = E1ww * E1hh;
   for (ii = 0; ii < 8; ii++) SB_done[ii] = 0;                             //  v.11.03

   for (ii = 0; ii < Nwt; ii++)                                            //  start worker threads
      start_wt(brhood_wthread,&wtnx[ii]);
   wait_wts();                                                             //  wait for completion

   SB_goal = 0;
   return;
}


//  worker thread function

void * brhood_wthread(void *arg)
{
   int      index = *((int *) arg);
   int      rad = brhood_radius;
   int      ii, px, py, qx, qy, Fstart;
   double   kern, bsum, bsamp, bmean;
   uint16   *pixel;

   if (brhood_method == 'g')                                               //  use round gaussian distribution
   {
      for (py = index; py < E1hh; py += Nwt)
      for (px = 0; px < E1ww; px++)
      {
         if (Factivearea && sa_mode != 7) {                                //  select area, not whole image    v.11.02
            ii = py * E1ww + px;                                           //    use only inside pixels
            if (! sa_pixisin[ii]) continue;
         }
         
         bsum = bsamp = 0;

         for (qy = py-rad; qy <= py+rad; qy++)                             //  computed weighted sum of brightness
         for (qx = px-rad; qx <= px+rad; qx++)                             //    for pixels in neighborhood
         {
            if (qy < 0 || qy > E1hh-1) continue;
            if (qx < 0 || qx > E1ww-1) continue;
            kern = brhood_kernel[qy+rad-py][qx+rad-px];
            pixel = PXMpix(E1pxm16,qx,qy);
            bsum += pixbright(pixel) * kern;                               //  sum brightness * weight
            bsamp += kern;                                                 //  sum weights
         }

         bmean = bsum / bsamp;                                             //  mean brightness
         ii = py * E1ww + px;
         brhood_brightness[ii] = bmean;                                    //  pixel value

         SB_done[index]++;
         if (drandz() < 0.0001) zsleep(0.001);                             //  trigger sorry kernel scheduler
      }
   }

   if (brhood_method == 'f')                                               //  use square flat distribution
   {
      Fstart = 1;
      bsum = bsamp = 0;

      for (py = index; py < E1hh; py += Nwt)
      for (px = 0; px < E1ww; px++)
      {
         if (Factivearea && sa_mode != 7) {                                //  select area, not whole image    v.11.02
            ii = py * E1ww + px;                                           //     compute only inside pixels
            if (! sa_pixisin[ii]) {
               Fstart = 1;
               continue;
            }
         }
         
         if (px == 0) Fstart = 1;
         
         if (Fstart) 
         {
            Fstart = 0;
            bsum = bsamp = 0;

            for (qy = py-rad; qy <= py+rad; qy++)                          //  add up all columns
            for (qx = px-rad; qx <= px+rad; qx++)
            {
               if (qy < 0 || qy > E1hh-1) continue;
               if (qx < 0 || qx > E1ww-1) continue;
               pixel = PXMpix(E1pxm16,qx,qy);
               bsum += pixbright(pixel);
               bsamp += 1;
            }
         }
         else
         {
            qx = px-rad-1;                                                 //  subtract first-1 column
            if (qx >= 0) {
               for (qy = py-rad; qy <= py+rad; qy++)
               {
                  if (qy < 0 || qy > E1hh-1) continue;
                  pixel = PXMpix(E1pxm16,qx,qy);
                  bsum -= pixbright(pixel);
                  bsamp -= 1;
               }
            }
            qx = px+rad;                                                   //  add last column
            if (qx < E1ww) {
               for (qy = py-rad; qy <= py+rad; qy++)
               {
                  if (qy < 0 || qy > E1hh-1) continue;
                  pixel = PXMpix(E1pxm16,qx,qy);
                  bsum += pixbright(pixel);
                  bsamp += 1;
               }
            }
         }

         bmean = bsum / bsamp;                                             //  mean brightness
         ii = py * E1ww + px;
         brhood_brightness[ii] = bmean;

         SB_done[index]++;
         if (drandz() < 0.0001) zsleep(0.001);                             //  trigger sorry kernel scheduler
      }
   }
         
   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


//  get the neighborhood brightness for given pixel

float get_brhood(int px, int py)
{
   int ii = py * E1ww + px;
   return brhood_brightness[ii];
}


/**************************************************************************/

//  free all resources associated with the current image file

void free_resources()
{
   char        *pp;
   int         ignore;

   mutex_lock(&Fpixmap_lock);                                              //  lock pixmaps
   
   strcpy(command,"rm -f ");                                               //  delete all undo files
   strcat(command,undo_files);
   pp = strstr(command,"_undo_");                                          //  /home/user/.fotoxx/pppppp_undo_*
   strcpy(pp + 6,"*");
   ignore = system(command);

   Fmodified = Pundo = Pumax = Fsaved = 0;                                 //  reset undo/redo stack
   Ntoplines = Nptoplines;                                                 //  no image overlay lines
   freeMouse();                                                            //  free mouse                v.11.04
   
   if (Fshutdown) {                                                        //  stop here if shutdown mode
      mutex_unlock(&Fpixmap_lock);
      return;
   }
   
   if (curr_file) {
      sa_delete();                                                         //  delete select area
      zdialog_free(zdsela);                                                //  kill dialogs if active
      zdialog_free(zdRGB);
      if (Mspcdat) zfree(Mspcdat);                                         //  v.11.01
      Mspcdat = 0;
      zfree(curr_file);                                                    //  free image file
      curr_file = 0;
      SB_goal = 0;                                                         //  v.9.2
      *SB_text = 0;                                                        //  v.10.7
   }
   
   if (brhood_brightness) zfree(brhood_brightness);                        //  free brightness map       v.9.6
   brhood_brightness = 0;

   PXM_free(Fpxm8);
   PXM_free(Dpxm8);
   PXM_free(Fpxm16);
   PXM_free(E1pxm16);
   PXM_free(E3pxm16);
   PXM_free(ERpxm16);                                                      //  v.10.3

   Fww = E1ww = E3ww = Dww = 0;                                            //  make unusable (crash)

   mutex_unlock(&Fpixmap_lock);
   return;
}


//  Get a virtual pixel at location (px,py) (real) in an PXM-16 pixmap.
//  Get the overlapping real pixels and build a composite.

int vpixel(PXM *pxm, double px, double py, uint16 *vpix)
{
   int            ww, hh, px0, py0;
   uint16         *ppix, *pix0, *pix1, *pix2, *pix3;
   double         f0, f1, f2, f3;
   double         red, green, blue;
   
   ww = pxm->ww;
   hh = pxm->hh;
   ppix = (uint16 *) pxm->bmp;

   px0 = int(px);                                                          //  pixel containing (px,py)
   py0 = int(py);

   if (px0 < 1 || py0 < 1) return 0;                                       //  void edge pixels
   if (px0 > ww-3 || py0 > hh-3) return 0;
   
   pix0 = ppix + (py0 * ww + px0) * 3;                                     //  4 pixels based at (px0,py0)
   pix1 = pix0 + ww * 3;
   pix2 = pix0 + 3;
   pix3 = pix0 + ww * 3 + 3;

   f0 = (px0+1 - px) * (py0+1 - py);                                       //  overlap of (px,py)
   f1 = (px0+1 - px) * (py - py0);                                         //   in each of the 4 pixels
   f2 = (px - px0) * (py0+1 - py);
   f3 = (px - px0) * (py - py0);
   
   red =   f0 * pix0[0] + f1 * pix1[0] + f2 * pix2[0] + f3 * pix3[0];      //  sum the weighted inputs
   green = f0 * pix0[1] + f1 * pix1[1] + f2 * pix2[1] + f3 * pix3[1];
   blue =  f0 * pix0[2] + f1 * pix1[2] + f2 * pix2[2] + f3 * pix3[2];
   
   vpix[0] = red;
   vpix[1] = green;
   vpix[2] = blue;
   
   if (blue < 1) {                                                         //  v.10.7 
      if (blue < 0.9) return 0;                                            //  mostly void
      vpix[2] = 1;                                                         //  avoid 0.999 to integer 0
   }

   return 1;
}


//  compare two doubles for significant difference
//  return:  0  difference not significant
//          +1  d1 > d2
//          -1  d1 < d2

int sigdiff(double d1, double d2, double signf)
{
   double diff = fabs(d1-d2);
   if (diff == 0.0) return 0;
   diff = diff / (fabs(d1) + fabs(d2));
   if (diff < signf) return 0;
   if (d1 > d2) return 1;
   else return -1;
}


/**************************************************************************
   file read and write utilities
   PXM pixmap <--> file on disk
***************************************************************************/

//  Load an image file into an PXM pixmap of 8 or 16 bits/color
//  Also sets the following file scope variables:
//    f_load_type = "jpg" "tif" "png" or "other"
//    f_load_bpc = disk file bits per color = 1/8/16
//    f_load_size = disk file size

PXM * f_load(cchar *filespec, int bpc)                                     //  use pixbuf or tiff library   v.9.8
{
   int            err;
   cchar          *pext;
   PXM            *pxm1, *pxm2;
   struct stat    fstat;

   if (bpc != 8 && bpc != 16)                                              //  requested bpc must be 8 or 16
      zappcrash("image_load bpc: %d",bpc);

   err = stat(filespec,&fstat);
   if (err) return 0;                                                      //  file not found
   if (! S_ISREG(fstat.st_mode)) return 0;                                 //  not a regular file
   if (image_file_type(filespec) != 2) return 0;                           //  not a supported image type
   f_load_size = fstat.st_size;                                            //  disk file bytes
   
   pext = strrchr(filespec,'/');
   if (! pext) pext = filespec;

   if (pext > filespec+12 && strnEqu(pext-12,"/.thumbnails/",13)) {        //  refuse thumbnail files    v.10.0
      zmessageACK(mWin,ZTX("cannot open thumbnail file"));
      return 0;
   }

   pext = strrchr(pext,'.');
   if (! pext) pext = "";
   
   if (strstr(".jpg .jpeg .JPG .JPEG",pext)) strcpy(f_load_type,"jpg");
   else if (strstr(".tif .tiff .TIF .TIFF",pext)) strcpy(f_load_type,"tif");
   else if (strstr(".png .PNG",pext)) strcpy(f_load_type,"png");
   else strcpy(f_load_type,"other");
   
   if (strEqu(f_load_type,"tif"))                                          //  use tiff lib to read tiff file
      pxm1 = TIFFread(filespec);
   else  pxm1 = PXBread(filespec);                                         //  use pixbuf lib for others
   if (! pxm1) return 0;                                                   //  (both set f_load_bpc = file bpc)
   
   if (pxm1->bpc != bpc) {
      pxm2 = PXM_convbpc(pxm1);                                            //  convert to requested bpc
      PXM_free(pxm1);                                                      //    8 <--> 16
      pxm1 = pxm2;
   }
                                                                           //  auto upright image removed    v.10.12
   return pxm1;
}


/**************************************************************************/

//  save current image to specified disk file (same or new).
//  set f_save_type, f_save_bpc, f_save_size
//  update search index file
//  return 0 if OK, else +N

int f_save(char *outfile, cchar *type, int bpc)                            //  use pixbuf or tiff library   v.9.8
{
   PXM            *pxm16;
   cchar          *exifkey[2] = { exif_orientation_key, exif_log_key };
   cchar          *exifdata[2];
   char           **ppv, funcslist[1000];
   char           *pp, *tempfile, *pext;
   int            nkeys, err = 1, cc1, cc2;
   struct stat    fstat;
   
   if ((bpc != 8 && bpc != 16) || ! strstr("jpg tif png",type))            //  check args
      zappcrash("f_save: %s %d",type,bpc);
   
   Ffuncbusy++;                                                            //  v.11.01

   pext = strrchr(outfile,'/');                                            //  force compatible file extension
   if (pext) pext = strrchr(pext,'.');                                     //    if not already
   if (! pext) pext = outfile + strlen(outfile);
   if (strEqu(type,"jpg") && ! strstr(".jpg .JPG .jpeg .JPEG",pext))
      strcpy(pext,".jpg");
   if (strEqu(type,"png") && ! strstr(".png .PNG",pext))
      strcpy(pext,".png");
   if (strEqu(type,"tif") && ! strstr(".tif .TIF .tiff .TIFF",pext))
      strcpy(pext,".tif");

   tempfile = strdupz(get_zuserdir(),24,"tempfile");                       //  use temp output file
   strcat(tempfile,"/temp_");                                              //    ~/.fotoxx/temp_ppppp.ext
   strcat(tempfile,PIDstring);
   strcat(tempfile,".");
   strcat(tempfile,type);

   if (strEqu(type,"png")) {                                               //  save as PNG file (bpc = 8 always)
      if (Fpxm16) err = PXBwrite(Fpxm16,tempfile);
      else  err = PXBwrite(Fpxm8,tempfile);
      bpc = 8;
   }

   else if (strEqu(type,"tif")) {                                          //  save as tiff file
      if (bpc == 8) {
         if (Fpxm16) {
            PXM_free(Fpxm8);                                               //  bugfix     v.10.8
            Fpxm8 = PXM_convbpc(Fpxm16);
         }
         err = TIFFwrite(Fpxm8,tempfile);
      }
      else if (bpc == 16) {
         if (Fpxm16) err = TIFFwrite(Fpxm16,tempfile);                     //  edit file exists, use it
         else {
            if (curr_file_bpc == 16) pxm16 = TIFFread(curr_file);          //  use original 16 bpc file
            else  pxm16 = PXM_convbpc(Fpxm8);                              //  or convert 8 to 16 bpc
            if (! pxm16) err = 1;
            else {
               err = TIFFwrite(pxm16,tempfile);
               PXM_free(pxm16);
            }
         }
      }
   }

   else {                                                                  //  save as JPEG file (bpc = 8 always)
      if (Fpxm16) err = PXBwrite(Fpxm16,tempfile);
      else  err = PXBwrite(Fpxm8,tempfile);
      bpc = 8;
   }

   if (err) {
      remove(tempfile);                                                    //  failure, clean up   v.11.02
      zfree(tempfile);
      Ffuncbusy--;
      return 1;
   }

   exifdata[0] = "";                                                       //  EXIF orientation = upright
   nkeys = 1;                                                              //  remove old width, height        v.10.8

   if (Pundo > 0)                                                          //  if edits were made,
   {                                                                       //    update relevant EXIF key      v.10.2
      *funcslist = 0;
      ppv = exif_get(curr_file,&exifkey[1],1);                             //  get existing list of edits
      if (ppv[0]) {
         strncpy0(funcslist,ppv[0],998);
         zfree(ppv[0]);
      }
      cc1 = strlen(funcslist);                                             //  add blank
      if (cc1 && funcslist[cc1-1] != ' ') {
         strcpy(funcslist+cc1," ");
         cc1++;
      }

      for (int ii = 1; ii <= Pundo; ii++)                                  //  append new list of edits
      {                                                                    //  (omit index 0 = initial)
         pp = pvlist_get(editlog,ii);
         cc2 = strlen(pp);
         if (cc1 + cc2 > 998) break;
         strcpy(funcslist+cc1,pp);
         strcpy(funcslist+cc1+cc2," ");
         cc1 += cc2 + 1;
      }
      
      exifdata[1] = funcslist;                                             //  EXIF log key, fotoxx edits done
      nkeys = 2;
   }

   err = stat(curr_file,&fstat);
   if (! err && S_ISREG(fstat.st_mode)) {                                  //  if current file exists,         v.11.05
      err = exif_copy(curr_file,tempfile,exifkey,exifdata,nkeys);          //    copy all EXIF data to temp file
      if (err) zmessageACK(mWin,"Unable to copy EXIF");                    //      with above revisions
   }
   
   snprintf(command,ccc,"cp -f \"%s\" \"%s\" ",tempfile,outfile);          //  copy temp file to output file
   err = system(command);
   if (err) zmessageACK(mWin,"Unable to save image: %s",wstrerror(err));

   remove(tempfile);                                                       //  delete temp file                v.11.02
   zfree(tempfile);
   
   save_filetags(outfile);                                                 //  save tag changes if any
   if (strNeq(outfile,curr_file))                                          //  if save to new file,            v.10.5
      update_search_index(outfile);                                        //    update search index file

   Fsaved = Pundo;                                                         //  update which mods are saved to disk

   add_recent_file(outfile);                                               //  first in recent files list
   
   strcpy(f_save_type,type);                                               //  update f_save_xxx data
   f_save_bpc = bpc;

   err = stat(outfile,&fstat);
   if (err) {
      Ffuncbusy--;
      return 1;
   }

   f_save_size = fstat.st_size;
   
   Ffuncbusy--;
   mwpaint2();                                                             //  v.11.03
   return 0;
}


/**************************************************************************/

//  Read from TIFF file using TIFF library. 
//  Use native TIFF file bits/pixel.

PXM * TIFFread(cchar *filespec)                                            //  overhauled  v.10.8.1
{
   TIFF        *tiff;
   PXM         *pxm;
   char        *tiffbuff;
   uint8       *tiff8, *pxm8;
   uint16      *tiff16, *pxm16;
   uint16      bpc, nch, fmt; 
   int         ww, hh, rps, stb, nst;                                      //  int not uint        v.11.03
   int         tiffstat, row, col, strip, cc;
   
   tiff = TIFFOpen(filespec,"r");
   if (! tiff) {
      zmessageACK(mWin,ZTX("TIFF open failure"));
      return 0;
   }

   TIFFGetField(tiff, TIFFTAG_IMAGEWIDTH, &ww);                            //  width
   TIFFGetField(tiff, TIFFTAG_IMAGELENGTH, &hh);                           //  height
   TIFFGetField(tiff, TIFFTAG_BITSPERSAMPLE, &bpc);                        //  bits per color, 1/8/16
   TIFFGetField(tiff, TIFFTAG_SAMPLESPERPIXEL, &nch);                      //  no. channels (colors), 1/3/4
   TIFFGetField(tiff, TIFFTAG_ROWSPERSTRIP, &rps);                         //  rows per strip
   TIFFGetField(tiff, TIFFTAG_SAMPLEFORMAT, &fmt);                         //  data format
   stb = TIFFStripSize(tiff);                                              //  strip size
   nst = TIFFNumberOfStrips(tiff);                                         //  number of strips

// printf("ww %d  hh %d  nch %d  bpc %d  rps %d  stb %d  nst %d  fmt %d \n",ww,hh,nch,bpc,rps,stb,nst,fmt);
   
   if (! (bpc <= 8 || bpc == 16)) {                                        //  check for supported bits/color
      zmessageACK(mWin,ZTX("TIFF bits/color=%d not supported"),bpc);
      TIFFClose(tiff);
      return 0;
   }
   
   f_load_bpc = bpc;                                                       //  for f_load(), file bpc 1/8/16   v.10.12

   if (bpc <= 8)                                                           //  use universal TIFF reader
   {                                                                       //    if bits/color <= 8
      tiffbuff = zmalloc(ww*hh*4,"tiffbuff");

      tiffstat = TIFFReadRGBAImage(tiff, ww, hh, (uint *) tiffbuff, 0);
      TIFFClose(tiff);

      if (tiffstat != 1) {
         zmessageACK(mWin,ZTX("TIFF read failure"));
         zfree(tiffbuff);
         return 0;
      }

      pxm = PXM_make(ww,hh,8);

      tiff8 = (uint8 *) tiffbuff;

      for (row = 0; row < hh; row++)                                       //  convert RGBA to RGB
      {
         pxm8 = (uint8 *) pxm->bmp;
         pxm8 += (hh-1 - row) * ww * 3;

         for (col = 0; col < ww; col++)
         {
            pxm8[0] = tiff8[0];
            pxm8[1] = tiff8[1];
            pxm8[2] = tiff8[2];
            pxm8 += 3;
            tiff8 += 4;
         }
      }

      zfree(tiffbuff);
      return pxm;
   }
                                                                           //  16 bits per color

   stb += 1000000;                                                         //  reduce risk of crash   v.10.8.2
   tiffbuff = zmalloc(stb,"tiffbuff");                                     //  read encoded strips 

   pxm = PXM_make(ww,hh,16);
   
   for (strip = 0; strip < nst; strip++)
   {
      cc = TIFFReadEncodedStrip(tiff,strip,tiffbuff,stb);
      if (cc < 0) {
         zmessageACK(mWin,ZTX("TIFF read failure"));
         TIFFClose(tiff);
         zfree(tiffbuff);
         PXM_free(pxm);
         return 0;
      }
      
      if (cc == 0) break;
      
      tiff16 = (uint16 *) tiffbuff;
      pxm16 = (uint16 *) pxm->bmp;
      row = strip * rps;
      pxm16 += row * ww * 3;

      while (cc >= 6)                                                      //  bugfix  v.11.03
      {
         pxm16[0] = tiff16[0];
         pxm16[1] = tiff16[1];
         pxm16[2] = tiff16[2];
         pxm16 += 3;
         tiff16 += nch;
         cc -= nch * 2;
      }
   }

   TIFFClose(tiff);
   zfree(tiffbuff);
   return pxm;
}


//  Write to TIFF file using TIFF library. 
//  File bpc is taken from PXM (8 or 16).
//  returns 0 if OK, +N if error.

int TIFFwrite(PXM *pxm, cchar *filespec)                                   //  new v.9.8
{
   TIFF        *tiff;
   uint8       *tiff8, *pxm8;
   uint16      *tiff16, *pxm16;
   int         tiffstat = 0;
   int         ww, hh, row, col, rowcc;                                    //  int not uint        v.11.03
   int         bpc, nch, pm = 2, pc = 1, comp = 5; 
   char        *tiffbuff;
   
   tiff = TIFFOpen(filespec,"w");
   if (! tiff) {
      zmessageACK(mWin,ZTX("TIFF open failure"));
      return 1;
   }
   
   ww = pxm->ww;
   hh = pxm->hh;
   bpc = pxm->bpc;
   nch = 3;                                                                //  alpha channel removed

   TIFFSetField(tiff, TIFFTAG_IMAGEWIDTH, ww);
   TIFFSetField(tiff, TIFFTAG_BITSPERSAMPLE, bpc);
   TIFFSetField(tiff, TIFFTAG_SAMPLESPERPIXEL, nch);
   TIFFSetField(tiff, TIFFTAG_PHOTOMETRIC, pm);                            //  RGB
   TIFFSetField(tiff, TIFFTAG_PLANARCONFIG, pc);
   TIFFSetField(tiff, TIFFTAG_COMPRESSION, comp);                          //  LZW
   
   rowcc = TIFFScanlineSize(tiff);
   tiffbuff = (char*) zmalloc(rowcc,"tiffbuff");

   for (row = 0; row < hh; row++)
   {
      if (bpc == 8)
      {
         tiff8 = (uint8 *) tiffbuff;
         pxm8 = (uint8 *) pxm->bmp + row * ww * 3;

         for (col = 0; col < ww; col++)
         {
            tiff8[0] = pxm8[0];
            tiff8[1] = pxm8[1];
            tiff8[2] = pxm8[2];
            pxm8 += 3;
            tiff8 += nch;
         }
      }

      if (bpc == 16) 
      {
         tiff16 = (uint16 *) tiffbuff;
         pxm16 = (uint16 *) pxm->bmp + row * ww * 3;

         for (col = 0; col < ww; col++)
         {
            tiff16[0] = pxm16[0];
            tiff16[1] = pxm16[1];
            tiff16[2] = pxm16[2];
            pxm16 += 3;
            tiff16 += nch;
         }
      }

      tiffstat = TIFFWriteScanline(tiff,tiffbuff,row,0);
      if (tiffstat != 1) break;
   }

   TIFFClose(tiff);
   zfree(tiffbuff);
   
   if (tiffstat == 1) return 0;
   zmessageACK(mWin,ZTX("TIFF write failure"));
   return 2;
}


//  intercept TIFF warning messages (many)                                 //  new v.9.8

void tiffwarninghandler(cchar *module, cchar *format, va_list ap)
{
   return;                                                                 //  stop flood of crap
   char  message[200];
   vsnprintf(message,199,format,ap);
   printf("TIFF warning: %s %s \n",module,message);
   return;
}


//  Read from jpeg/png file using pixbuf library. bpc = 8.

PXM * PXBread(cchar *filespec)                                             //  new v.9.8
{
   GError      *gerror = 0;
   PXB         *pxb;
   PXM         *pxm;
   int         ww, hh, px, py, nch, rowst;
   uint8       *bmp1, *bmp2, *pix1, *pix2;
   
   pxb = gdk_pixbuf_new_from_file(filespec,&gerror);
   if (! pxb) {
      printf("%s \n",gerror->message);                                     //  v.10.8
      zmessageACK(mWin,ZTX("file type not supported"));
      return 0;
   }

   ww = gdk_pixbuf_get_width(pxb);
   hh = gdk_pixbuf_get_height(pxb);
   nch = gdk_pixbuf_get_n_channels(pxb);
   rowst = gdk_pixbuf_get_rowstride(pxb);
   bmp1 = gdk_pixbuf_get_pixels(pxb);
   
   pxm = PXM_make(ww,hh,8);
   bmp2 = (uint8 *) pxm->bmp;

   for (py = 0; py < hh; py++)
   {
      pix1 = bmp1 + rowst * py;
      pix2 = bmp2 + py * ww * 3;

      for (px = 0; px < ww; px++)
      {
         pix2[0] = pix1[0];
         pix2[1] = pix1[1];
         pix2[2] = pix1[2];
         pix1 += nch;
         pix2 += 3;
      }
   }
   
   f_load_bpc = 8;                                                         //  file bits per color for f_load()
   g_object_unref(pxb);                                                    //  (any value on disk becomes 8)
   return pxm;
}


//  Write to jpeg/png file using pixbuf library. bpc = 8.
//  returns 0 if OK, +N if error.

int PXBwrite(PXM *pxm, cchar *filespec)                                    //  new v.9.8
{
   int         ww, hh, bpc, px, py, rowst;
   uint8       *bmp8, *pix8, *bmp2, *pix2;
   uint16      *bmp16, *pix16;
   PXB         *pxb;
   cchar       *pext;
   GError      *gerror = 0;
   int         pxbstat;

   ww = pxm->ww;
   hh = pxm->hh;
   bpc = pxm->bpc;
   
   pxb = gdk_pixbuf_new(RGBCOLOR,0,8,ww,hh);
   if (! pxb) zappcrash("pixbuf allocation failure");

   bmp8 = (uint8 *) pxm->bmp;
   bmp16 = (uint16 *) pxm->bmp;
   bmp2 = gdk_pixbuf_get_pixels(pxb);
   rowst = gdk_pixbuf_get_rowstride(pxb);

   if (bpc == 8)
   {
      for (py = 0; py < hh; py++)
      {
         pix8 = bmp8 + py * ww * 3;
         pix2 = bmp2 + rowst * py;

         for (px = 0; px < ww; px++)
         {
            pix2[0] = pix8[0];
            pix2[1] = pix8[1];
            pix2[2] = pix8[2];
            pix2 += 3;
            pix8 += 3;
         }
      }
   }
   
   if (bpc == 16)
   {
      for (py = 0; py < hh; py++)
      {
         pix16 = bmp16 + py * ww * 3;
         pix2 = bmp2 + rowst * py;

         for (px = 0; px < ww; px++)
         {
            pix2[0] = pix16[0] >> 8;
            pix2[1] = pix16[1] >> 8;
            pix2[2] = pix16[2] >> 8;
            pix2 += 3;
            pix16 += 3;
         }
      }
   }
   
   pext = strrchr(filespec,'/');
   if (! pext) pext = filespec;
   pext = strrchr(pext,'.');
   if (! pext) pext = "";
   
   if (strstr(".png .PNG",pext))
      pxbstat = gdk_pixbuf_save(pxb,filespec,"png",&gerror,null);
   else pxbstat = gdk_pixbuf_save(pxb,filespec,"jpeg",&gerror,"quality",jpeg_quality,null);
   g_object_unref(pxb);
   if (pxbstat) return 0;

   printf("%s \n",gerror->message);                                        //  v.10.8
   zmessageACK(mWin,ZTX("pixbuf write failure"));
   return 1;
}


/**************************************************************************
      PXM pixmap conversion and rescale functions
***************************************************************************/

//  initialize PXM pixmap - allocate memory

PXM * PXM_make(int ww, int hh, int bpc)
{
   if (ww < 1 || hh < 1 || (bpc != 8 && bpc != 16))
      zappcrash("PXM_make() %d %d %d",ww,hh,bpc);
   
   PXM *pxm = (PXM *) zmalloc(sizeof(PXM),"PXM_make");
   pxm->ww = ww;
   pxm->hh = hh;
   pxm->bpc = bpc;
   if (bpc == 8) pxm->bmp = zmalloc(ww*hh*3,"PXM.bmp");
   if (bpc == 16) pxm->bmp = zmalloc(ww*hh*6,"PXM.bmp");
   strcpy(pxm->wmi,"rgbrgb");
   return pxm;
}


//  free PXM pixmap

void PXM_free(PXM *&pxm)
{
   if (! pxm) return;
   if (! strEqu(pxm->wmi,"rgbrgb")) 
      zappcrash("PXM_free(), bad PXM");
   strcpy(pxm->wmi,"xxxxxx");
   zfree(pxm->bmp);
   zfree(pxm);
   pxm = 0;                                                                //  v.11.01
   return;
}


//  create a copy of an PXM pixmap

PXM * PXM_copy(PXM *pxm1)
{
   int      cc;
   PXM      *pxm2;

   pxm2 = PXM_make(pxm1->ww, pxm1->hh, pxm1->bpc);
   cc = pxm1->ww * pxm1->hh * (pxm1->bpc / 8 * 3);
   memcpy(pxm2->bmp,pxm1->bmp,cc);
   return pxm2;
}


//  create a copy of an PXM area

PXM * PXM_copy_area(PXM *pxm1, int orgx, int orgy, int ww2, int hh2)
{
   uint8          *bmp1, *pix1, *bmp2, *pix2;
   uint16         *bmp3, *pix3, *bmp4, *pix4;
   PXM            *pxm2 = 0;
   int            ww1, hh1, bpc, px1, py1, px2, py2;

   ww1 = pxm1->ww;
   hh1 = pxm1->hh;
   bpc = pxm1->bpc;

   if (bpc == 8)
   {
      pxm2 = PXM_make(ww2,hh2,8);
      bmp1 = (uint8 *) pxm1->bmp;
      bmp2 = (uint8 *) pxm2->bmp;
     
      for (py1 = orgy, py2 = 0; py2 < hh2; py1++, py2++) 
      {
         for (px1 = orgx, px2 = 0; px2 < ww2; px1++, px2++)
         {
            pix1 = bmp1 + (py1 * ww1 + px1) * 3;
            pix2 = bmp2 + (py2 * ww2 + px2) * 3;

            pix2[0] = pix1[0];
            pix2[1] = pix1[1];
            pix2[2] = pix1[2];
            pix1 += 3;
            pix2 += 3;
         }
      }
   }

   if (bpc == 16)
   {
      pxm2 = PXM_make(ww2,hh2,16);
      bmp3 = (uint16 *) pxm1->bmp;
      bmp4 = (uint16 *) pxm2->bmp;
     
      for (py1 = orgy, py2 = 0; py2 < hh2; py1++, py2++) 
      {
         for (px1 = orgx, px2 = 0; px2 < ww2; px1++, px2++)
         {
            pix3 = bmp3 + (py1 * ww1 + px1) * 3;
            pix4 = bmp4 + (py2 * ww2 + px2) * 3;

            pix4[0] = pix3[0];
            pix4[1] = pix3[1];
            pix4[2] = pix3[2];
            pix3 += 3;
            pix4 += 3;
         }
      }
   }
   
   return pxm2;
}


//   convert PXM pixmap from 8/16 to 16/8 bits per color

PXM * PXM_convbpc(PXM *pxm1)
{
   uint8          *bmp8, *pix8;
   uint16         *bmp16, *pix16;
   PXM            *pxm2 = 0;
   int            ww, hh, bpc, px, py;

   ww = pxm1->ww;
   hh = pxm1->hh;
   bpc = pxm1->bpc;

   if (bpc == 8)                                                           //  8 > 16
   {
      pxm2 = PXM_make(ww,hh,16);
      bmp8 = (uint8 *) pxm1->bmp;
      bmp16 = (uint16 *) pxm2->bmp;
     
      for (py = 0; py < hh; py++) 
      {
         pix8 = bmp8 + py * ww * 3;
         pix16 = bmp16 + py * ww * 3;

         for (px = 0; px < ww; px++)
         {
            pix16[0] = pix8[0] << 8;
            pix16[1] = pix8[1] << 8;
            pix16[2] = pix8[2] << 8;
            pix8 += 3;
            pix16 += 3;
         }
      }
   }

   if (bpc == 16)                                                          //  16 > 8
   {
      pxm2 = PXM_make(ww,hh,8);
      bmp8 = (uint8 *) pxm2->bmp;
      bmp16 = (uint16 *) pxm1->bmp;
     
      for (py = 0; py < hh; py++) 
      {
         pix8 = bmp8 + py * ww * 3;
         pix16 = bmp16 + py * ww * 3;

         for (px = 0; px < ww; px++)
         {
            pix8[0] = pix16[0] >> 8;
            pix8[1] = pix16[1] >> 8;
            pix8[2] = pix16[2] >> 8;
            pix8 += 3;
            pix16 += 3;
         }
      }
   }

   return pxm2;
}


//  rescale PXM pixmap to size ww2 x hh2

PXM * PXM_rescale(PXM *pxm1, int ww2, int hh2)
{
   void bmp8_rescale(uint8*, uint8*, int, int, int, int);
   void bmp16_rescale(uint16*, uint16*, int, int, int, int);

   PXM      *pxm2;
   int      ww1, hh1, bpc;
   uint8    *bmp1, *bmp2;
   uint16   *bmp3, *bmp4;

   ww1 = pxm1->ww;
   hh1 = pxm1->hh;
   bpc = pxm1->bpc;
   
   pxm2 = PXM_make(ww2,hh2,bpc);
   
   if (bpc == 8) {
      bmp1 = (uint8 *) pxm1->bmp;
      bmp2 = (uint8 *) pxm2->bmp;
      bmp8_rescale(bmp1,bmp2,ww1,hh1,ww2,hh2);
   }

   if (bpc == 16) {
      bmp3 = (uint16 *) pxm1->bmp;
      bmp4 = (uint16 *) pxm2->bmp;
      bmp16_rescale(bmp3,bmp4,ww1,hh1,ww2,hh2);
   }
   
   return pxm2;
}


//  Copy and rescale a modified area within a PXM-16 image into the
//  corresponding area of a PXM-8 image previously rescaled from the 
//  PXM-16 image. Keep the same mapping of input to output pixels, so 
//  that the modified area fits seamlessly into the PXM-8 image. Used 
//  when a section of an image is edited and the window image is updated.
//   
//  pxm1           PXM-16 image with changed area to rescale and copy
//  pxm2           existing rescaled PXM-8 copy of pxm1
//  org1x, org1y   pxm1 origin of area to copy (top left corner)
//  ww1a, hh1a     width and height of area to copy                        //  new v.10.11

void PXM_update(PXM *pxm1, PXM *pxm2, int org1x, int org1y, int ww1a, int hh1a)
{
   uint16      *bmp1, *pix1;
   uint8       *bmp2, *pix2;
   int         ww1, hh1, ww2, hh2;
   int         px1, py1, px2, py2;
   int         pxl, pyl, pxm, pym, ii;
   int         *px1L, *py1L;
   int         maxmapx, maxmapy;
   int         org2x, end2x, org2y, end2y;
   float       scalex, scaley;
   float       px1a, py1a, px1b, py1b;
   float       fx, fy, ftot;
   float       red, green, blue;
   float       *pxmap, *pymap;

   bmp1 = (uint16 *) pxm1->bmp;   
   bmp2 = (uint8 *) pxm2->bmp;   
   ww1 = pxm1->ww;
   hh1 = pxm1->hh;
   ww2 = pxm2->ww;
   hh2 = pxm2->hh;

   scalex = 1.0 * ww1 / ww2;                                               //  compute x and y scales
   scaley = 1.0 * hh1 / hh2;
   
   if (scalex <= 1) maxmapx = 2;                                           //  compute max input pixels
   else maxmapx = int(scalex + 2);                                         //    mapping into output pixels
   maxmapx += 1;                                                           //      for both dimensions
   if (scaley <= 1) maxmapy = 2;                                           //  (pixels may not be square)
   else maxmapy = int(scaley + 2);
   maxmapy += 1;
   
   pymap = (float *) zmalloc(hh2 * maxmapy * sizeof(float));               //  maps overlap of < maxmap input
   pxmap = (float *) zmalloc(ww2 * maxmapx * sizeof(float));               //    pixels per output pixel

   py1L = (int *) zmalloc(hh2 * sizeof(int));                              //  maps first (lowest) input pixel
   px1L = (int *) zmalloc(ww2 * sizeof(int));                              //    per output pixel

   for (py2 = 0; py2 < hh2; py2++)                                         //  loop output y-pixels
   {
      py1a = py2 * scaley;                                                 //  corresponding input y-pixels
      py1b = py1a + scaley;
      if (py1b >= hh1) py1b = hh1 - 0.001;                                 //  fix precision limitation
      pyl = int(py1a);
      py1L[py2] = pyl;                                                     //  1st overlapping input pixel

      for (py1 = pyl, pym = 0; py1 < py1b; py1++, pym++)                   //  loop overlapping input pixels
      {
         if (py1 < py1a) {                                                 //  compute amount of overlap
            if (py1+1 < py1b) fy = py1+1 - py1a;                           //    0.0 to 1.0 
            else fy = scaley;
         }
         else if (py1+1 > py1b) fy = py1b - py1;
         else fy = 1;

         ii = py2 * maxmapy + pym;                                         //  save it
         pymap[ii] = 0.9999 * fy / scaley;
      }
      ii = py2 * maxmapy + pym;                                            //  set an end marker after
      pymap[ii] = -1;                                                      //    last overlapping pixel
   }
   
   for (px2 = 0; px2 < ww2; px2++)                                         //  do same for x-pixels
   {
      px1a = px2 * scalex;
      px1b = px1a + scalex;
      if (px1b >= ww1) px1b = ww1 - 0.001;
      pxl = int(px1a);
      px1L[px2] = pxl;

      for (px1 = pxl, pxm = 0; px1 < px1b; px1++, pxm++)
      {
         if (px1 < px1a) {
            if (px1+1 < px1b) fx = px1+1 - px1a;
            else fx = scalex;
         }
         else if (px1+1 > px1b) fx = px1b - px1;
         else fx = 1;

         ii = px2 * maxmapx + pxm;
         pxmap[ii] = 0.9999 * fx / scalex;
      }
      ii = px2 * maxmapx + pxm;
      pxmap[ii] = -1;
   }
   
   org2x = org1x / scalex;                                                 //  compute output image rectangle
   end2x = (org1x + ww1a) / scalex + 1;                                    //    containing any part of input area
   if (org2x < 0) org2x = 0;                                               //       revised  v.10.12
   if (end2x > ww2) end2x = ww2;

   org2y = org1y / scaley;
   end2y = (org1y + hh1a) / scaley + 1;
   if (org2y < 0) org2y = 0;
   if (end2y > hh2) end2y = hh2;

   for (py2 = org2y; py2 < end2y; py2++)                                   //  loop output y-pixels
   {
      pyl = py1L[py2];                                                     //  corresp. 1st input y-pixel

      for (px2 = org2x; px2 < end2x; px2++)                                //  loop output x-pixels
      {
         pxl = px1L[px2];                                                  //  corresp. 1st input x-pixel

         red = green = blue = 0;                                           //  initz. output pixel

         for (py1 = pyl, pym = 0; ; py1++, pym++)                          //  loop overlapping input y-pixels
         {
            ii = py2 * maxmapy + pym;                                      //  get y-overlap
            fy = pymap[ii];
            if (fy < 0) break;                                             //  no more pixels

            for (px1 = pxl, pxm = 0; ; px1++, pxm++)                       //  loop overlapping input x-pixels
            {
               ii = px2 * maxmapx + pxm;                                   //  get x-overlap
               fx = pxmap[ii];
               if (fx < 0) break;                                          //  no more pixels

               ftot = fx * fy;                                             //  area overlap = x * y overlap
               pix1 = bmp1 + (py1 * ww1 + px1) * 3;
               red += pix1[0] * ftot;                                      //  add input pixel * overlap
               green += pix1[1] * ftot;
               blue += pix1[2] * ftot;
            }

            pix2 = bmp2 + (py2 * ww2 + px2) * 3;                           //  save output pixel
            pix2[0] = int(red) >> 8;
            pix2[1] = int(green) >> 8;
            pix2[2] = int(blue) >> 8;
         }
      }
   }

   zfree(px1L);
   zfree(py1L);
   zfree(pxmap);
   zfree(pymap);
   return;
}


//  rotate PXM pixmap through given angle in degrees (+ = clockwise)

PXM * PXM_rotate(PXM *pxm1, double angle)
{
   PXM * PXM_rotate8(PXM *, double);
   PXM * PXM_rotate16(PXM *, double);

   PXM      *pxm2 = 0;
   int      bpc;
   
   bpc = pxm1->bpc;
   if (bpc == 8) pxm2 = PXM_rotate8(pxm1,angle);
   if (bpc == 16) pxm2 = PXM_rotate16(pxm1,angle);
   return pxm2;
}


/**************************************************************************

   Rescale 8 bpc image (3 x 8 bits per color) to new width and height.
   The scale ratios may be different for width and height.

   Method: 
   The input and output images are overlayed, stretching or shrinking the
   output pixels as needed. The contribution of each input pixel overlapping
   an output pixel is proportional to the area of the output pixel covered by
   the input pixel. The contributions of all overlaping input pixels are added.
   The work is spread among Nwt threads to reduce the elapsed time on modern 
   computers having multiple SMP processors.

   Example: if the output image is 40% of the input image, then:
     outpix[0,0] = 0.16 * inpix[0,0] + 0.16 * inpix[1,0] + 0.08 * inpix[2,0]
                 + 0.16 * inpix[0,1] + 0.16 * inpix[1,1] + 0.08 * inpix[2,1]
                 + 0.08 * inpix[0,2] + 0.08 * inpix[1,2] + 0.04 * inpix[2,2]

*********/

namespace bmp8rescale {                                                    //  data for threads
   uint8    *pixmap1;
   uint8    *pixmap2;
   int      ww1;
   int      hh1;
   int      ww2;
   int      hh2;
   int      *px1L;
   int      *py1L;
   float    *pxmap;
   float    *pymap;
   int      maxmapx;
   int      maxmapy;
   int      busy[max_threads];
}


void bmp8_rescale(uint8 *pixmap1x, uint8 *pixmap2x, int ww1x, int hh1x, int ww2x, int hh2x)
{
   using namespace bmp8rescale;

   void * bmp8_rescale_thread(void *arg);

   int         px1, py1, px2, py2;
   int         pxl, pyl, pxm, pym, ii;
   float       scalex, scaley;
   float       px1a, py1a, px1b, py1b;
   float       fx, fy;
   
   pixmap1 = pixmap1x;
   pixmap2 = pixmap2x;
   ww1 = ww1x;
   hh1 = hh1x;
   ww2 = ww2x;
   hh2 = hh2x;

   memset(pixmap2, 0, ww2 * hh2 * 3 * sizeof(uint8));                      //  clear output pixmap

   scalex = 1.0 * ww1 / ww2;                                               //  compute x and y scales
   scaley = 1.0 * hh1 / hh2;
   
   if (scalex <= 1) maxmapx = 2;                                           //  compute max input pixels
   else maxmapx = int(scalex + 2);                                         //    mapping into output pixels
   maxmapx += 1;                                                           //      for both dimensions
   if (scaley <= 1) maxmapy = 2;                                           //  (pixels may not be square)
   else maxmapy = int(scaley + 2);
   maxmapy += 1;
   
   pymap = (float *) zmalloc(hh2 * maxmapy * sizeof(float));               //  maps overlap of < maxmap input
   pxmap = (float *) zmalloc(ww2 * maxmapx * sizeof(float));               //    pixels per output pixel

   py1L = (int *) zmalloc(hh2 * sizeof(int));                              //  maps first (lowest) input pixel
   px1L = (int *) zmalloc(ww2 * sizeof(int));                              //    per output pixel

   for (py2 = 0; py2 < hh2; py2++)                                         //  loop output y-pixels
   {
      py1a = py2 * scaley;                                                 //  corresponding input y-pixels
      py1b = py1a + scaley;
      if (py1b >= hh1) py1b = hh1 - 0.001;                                 //  fix precision limitation
      pyl = int(py1a);
      py1L[py2] = pyl;                                                     //  1st overlapping input pixel

      for (py1 = pyl, pym = 0; py1 < py1b; py1++, pym++)                   //  loop overlapping input pixels
      {
         if (py1 < py1a) {                                                 //  compute amount of overlap
            if (py1+1 < py1b) fy = py1+1 - py1a;                           //    0.0 to 1.0 
            else fy = scaley;
         }
         else if (py1+1 > py1b) fy = py1b - py1;
         else fy = 1;

         ii = py2 * maxmapy + pym;                                         //  save it
         pymap[ii] = 0.9999 * fy / scaley;
      }
      ii = py2 * maxmapy + pym;                                            //  set an end marker after
      pymap[ii] = -1;                                                      //    last overlapping pixel
   }
   
   for (px2 = 0; px2 < ww2; px2++)                                         //  do same for x-pixels
   {
      px1a = px2 * scalex;
      px1b = px1a + scalex;
      if (px1b >= ww1) px1b = ww1 - 0.001;
      pxl = int(px1a);
      px1L[px2] = pxl;

      for (px1 = pxl, pxm = 0; px1 < px1b; px1++, pxm++)
      {
         if (px1 < px1a) {
            if (px1+1 < px1b) fx = px1+1 - px1a;
            else fx = scalex;
         }
         else if (px1+1 > px1b) fx = px1b - px1;
         else fx = 1;

         ii = px2 * maxmapx + pxm;
         pxmap[ii] = 0.9999 * fx / scalex;
      }
      ii = px2 * maxmapx + pxm;
      pxmap[ii] = -1;
   }

   for (ii = 0; ii < Nwt; ii++) {                                          //  start working threads
      busy[ii] = 1;
      start_detached_thread(bmp8_rescale_thread,&wtnx[ii]);
   }
   
   for (ii = 0; ii < Nwt; ii++)                                            //  wait for all done
      while (busy[ii]) zsleep(0.004);

   zfree(px1L);
   zfree(py1L);
   zfree(pxmap);
   zfree(pymap);
   return;
}


void * bmp8_rescale_thread(void *arg)                                      //  worker thread function
{
   using namespace bmp8rescale;

   int         index = *((int *) arg);
   int         px1, py1, px2, py2;
   int         pxl, pyl, pxm, pym, ii;
   uint8       *pixel1, *pixel2;
   float       fx, fy, ftot;
   float       red, green, blue;

   for (py2 = index; py2 < hh2; py2 += Nwt)                                //  loop output y-pixels
   {
      pyl = py1L[py2];                                                     //  corresp. 1st input y-pixel

      for (px2 = 0; px2 < ww2; px2++)                                      //  loop output x-pixels
      {
         pxl = px1L[px2];                                                  //  corresp. 1st input x-pixel

         red = green = blue = 0;                                           //  initz. output pixel

         for (py1 = pyl, pym = 0; ; py1++, pym++)                          //  loop overlapping input y-pixels
         {
            ii = py2 * maxmapy + pym;                                      //  get y-overlap
            fy = pymap[ii];
            if (fy < 0) break;                                             //  no more pixels

            for (px1 = pxl, pxm = 0; ; px1++, pxm++)                       //  loop overlapping input x-pixels
            {
               ii = px2 * maxmapx + pxm;                                   //  get x-overlap
               fx = pxmap[ii];
               if (fx < 0) break;                                          //  no more pixels

               ftot = fx * fy;                                             //  area overlap = x * y overlap
               pixel1 = pixmap1 + (py1 * ww1 + px1) * 3;
               red += pixel1[0] * ftot;                                    //  add input pixel * overlap
               green += pixel1[1] * ftot;
               blue += pixel1[2] * ftot;
            }

            pixel2 = pixmap2 + (py2 * ww2 + px2) * 3;                      //  save output pixel
            pixel2[0] = int(red);
            pixel2[1] = int(green);
            pixel2[2] = int(blue);
         }
      }
   }

   busy[index] = 0;
   return 0;
}


/**************************************************************************

   Rescale 16 bpc image (3 x 16 bits per color) to new width and height.
   Identical to bmp8_rescale except for the following: 
      uint8 >> uint16
      xxx8 >> xxx16

*******/

namespace bmp16rescale {                                                   //  data for threads
   uint16   *pixmap1;
   uint16   *pixmap2;
   int      ww1;
   int      hh1;
   int      ww2;
   int      hh2;
   int      *px1L;
   int      *py1L;
   float    *pxmap;
   float    *pymap;
   int      maxmapx;
   int      maxmapy;
   int      busy[max_threads];
}


void bmp16_rescale(uint16 *pixmap1x, uint16 *pixmap2x, int ww1x, int hh1x, int ww2x, int hh2x)
{
   using namespace bmp16rescale;

   void * bmp16_rescale_thread(void *arg);

   int         px1, py1, px2, py2;
   int         pxl, pyl, pxm, pym, ii;
   float       scalex, scaley;
   float       px1a, py1a, px1b, py1b;
   float       fx, fy;
   
   pixmap1 = pixmap1x;
   pixmap2 = pixmap2x;
   ww1 = ww1x;
   hh1 = hh1x;
   ww2 = ww2x;
   hh2 = hh2x;

   memset(pixmap2, 0, ww2 * hh2 * 3 * sizeof(uint16));                     //  clear output pixmap

   scalex = 1.0 * ww1 / ww2;                                               //  compute x and y scales
   scaley = 1.0 * hh1 / hh2;
   
   if (scalex <= 1) maxmapx = 2;                                           //  compute max input pixels
   else maxmapx = int(scalex + 2);                                         //    mapping into output pixels
   maxmapx += 1;                                                           //      for both dimensions
   if (scaley <= 1) maxmapy = 2;                                           //  (pixels may not be square)
   else maxmapy = int(scaley + 2);
   maxmapy += 1;
   
   pymap = (float *) zmalloc(hh2 * maxmapy * sizeof(float));               //  maps overlap of < maxmap input
   pxmap = (float *) zmalloc(ww2 * maxmapx * sizeof(float));               //    pixels per output pixel

   py1L = (int *) zmalloc(hh2 * sizeof(int));                              //  maps first (lowest) input pixel
   px1L = (int *) zmalloc(ww2 * sizeof(int));                              //    per output pixel

   for (py2 = 0; py2 < hh2; py2++)                                         //  loop output y-pixels
   {
      py1a = py2 * scaley;                                                 //  corresponding input y-pixels
      py1b = py1a + scaley;
      if (py1b >= hh1) py1b = hh1 - 0.001;                                 //  fix precision limitation
      pyl = int(py1a);
      py1L[py2] = pyl;                                                     //  1st overlapping input pixel

      for (py1 = pyl, pym = 0; py1 < py1b; py1++, pym++)                   //  loop overlapping input pixels
      {
         if (py1 < py1a) {                                                 //  compute amount of overlap
            if (py1+1 < py1b) fy = py1+1 - py1a;                           //    0.0 to 1.0 
            else fy = scaley;
         }
         else if (py1+1 > py1b) fy = py1b - py1;
         else fy = 1;

         ii = py2 * maxmapy + pym;                                         //  save it
         pymap[ii] = 0.9999 * fy / scaley;
      }
      ii = py2 * maxmapy + pym;                                            //  set an end marker after
      pymap[ii] = -1;                                                      //    last overlapping pixel
   }
   
   for (px2 = 0; px2 < ww2; px2++)                                         //  do same for x-pixels
   {
      px1a = px2 * scalex;
      px1b = px1a + scalex;
      if (px1b >= ww1) px1b = ww1 - 0.001;
      pxl = int(px1a);
      px1L[px2] = pxl;

      for (px1 = pxl, pxm = 0; px1 < px1b; px1++, pxm++)
      {
         if (px1 < px1a) {
            if (px1+1 < px1b) fx = px1+1 - px1a;
            else fx = scalex;
         }
         else if (px1+1 > px1b) fx = px1b - px1;
         else fx = 1;

         ii = px2 * maxmapx + pxm;
         pxmap[ii] = 0.9999 * fx / scalex;
      }
      ii = px2 * maxmapx + pxm;
      pxmap[ii] = -1;
   }

   for (ii = 0; ii < Nwt; ii++) {                                          //  start working threads
      busy[ii] = 1;
      start_detached_thread(bmp16_rescale_thread,&wtnx[ii]);
   }
   
   for (ii = 0; ii < Nwt; ii++)                                            //  wait for all done
      while (busy[ii]) zsleep(0.004);

   zfree(px1L);
   zfree(py1L);
   zfree(pxmap);
   zfree(pymap);
   return;
}


void * bmp16_rescale_thread(void *arg)                                     //  worker thread function
{
   using namespace bmp16rescale;

   int         index = *((int *) arg);
   int         px1, py1, px2, py2;
   int         pxl, pyl, pxm, pym, ii;
   uint16      *pixel1, *pixel2;
   float       fx, fy, ftot;
   float       red, green, blue;

   for (py2 = index; py2 < hh2; py2 += Nwt)                                //  loop output y-pixels
   {
      pyl = py1L[py2];                                                     //  corresp. 1st input y-pixel

      for (px2 = 0; px2 < ww2; px2++)                                      //  loop output x-pixels
      {
         pxl = px1L[px2];                                                  //  corresp. 1st input x-pixel

         red = green = blue = 0;                                           //  initz. output pixel

         for (py1 = pyl, pym = 0; ; py1++, pym++)                          //  loop overlapping input y-pixels
         {
            ii = py2 * maxmapy + pym;                                      //  get y-overlap
            fy = pymap[ii];
            if (fy < 0) break;                                             //  no more pixels

            for (px1 = pxl, pxm = 0; ; px1++, pxm++)                       //  loop overlapping input x-pixels
            {
               ii = px2 * maxmapx + pxm;                                   //  get x-overlap
               fx = pxmap[ii];
               if (fx < 0) break;                                          //  no more pixels

               ftot = fx * fy;                                             //  area overlap = x * y overlap
               pixel1 = pixmap1 + (py1 * ww1 + px1) * 3;
               red += pixel1[0] * ftot;                                    //  add input pixel * overlap
               green += pixel1[1] * ftot;
               blue += pixel1[2] * ftot;
            }

            pixel2 = pixmap2 + (py2 * ww2 + px2) * 3;                      //  save output pixel
            pixel2[0] = int(red);
            pixel2[1] = int(green);
            pixel2[2] = int(blue);
         }
      }
   }

   busy[index] = 0;
   return 0;
}


/**************************************************************************

      PXM *pxm2 = PXM_rotate8(PXM *pxm1, double angle)

      Rotate PXM-8 pixmap through an arbitrary angle (degrees).

      The returned image has the same size as the original, but the
      pixmap size is increased to accomodate the rotated image.
      (e.g. a 100x100 image rotated 45 deg. needs a 142x142 pixmap).
      The parameters ww and hh are the dimensions of the input
      pixmap, and are updated to the dimensions of the output pixmap.

      The space added around the rotated image is black (RGB 0,0,0).
      Angle is in degrees. Positive direction is clockwise.
      Speed is about 3 million pixels/sec/thread for a 2.4 GHz CPU.
      Loss of resolution is less than 1 pixel.
      
      Work is divided among Nwt threads to gain speed.
      
      v.9.3: affine transform instead of trig functions, for speed

***************************************************************************/

namespace rotpxm8 {
   int      busy = 0;
   uint8    *pixmap1;
   uint8    *pixmap2;
   int      ww1;
   int      hh1;
   int      ww2;
   int      hh2;
   double   angle;
}


PXM * PXM_rotate8(PXM *pxm1, double anglex)
{
   using namespace rotpxm8;

   void     *PXM_rotate8_thread(void *);

   int      cc, ii;
   PXM      *pxm2;
   
   ww1 = pxm1->ww;
   hh1 = pxm1->hh;
   pixmap1 = (uint8 *) pxm1->bmp;
   angle = anglex;

   while (angle < -180) angle += 360;                                      //  normalize, -180 to +180
   while (angle > 180) angle -= 360;
   angle = angle * pi / 180;                                               //  radians, -pi to +pi
   
   if (fabs(angle) < 0.001) {                                              //  angle = 0 within my precision
      pxm2 = PXM_make(ww1,hh1,8);                                          //  return a copy of the input
      pixmap2 = (uint8 *) pxm2->bmp;
      cc = ww1 * hh1 * 3 * sizeof(uint8);
      memcpy(pixmap2,pixmap1,cc);
      return pxm2;
   }

   ww2 = int(ww1*fabs(cos(angle)) + hh1*fabs(sin(angle)));                 //  rectangle containing rotated image
   hh2 = int(ww1*fabs(sin(angle)) + hh1*fabs(cos(angle)));
   
   pxm2 = PXM_make(ww2,hh2,8);
   pixmap2 = (uint8 *) pxm2->bmp;

   for (ii = 0; ii < Nwt; ii++)                                            //  start worker threads
      start_detached_thread(PXM_rotate8_thread,&wtnx[ii]);
   zadd_locked(busy,+Nwt);

   while (busy) zsleep(0.004);                                             //  wait for completion
   return pxm2;
}


void * PXM_rotate8_thread(void *arg)
{
   using namespace rotpxm8;

   int      index = *((int *) (arg));
   int      px2, py2, px0, py0;
   uint8    *pix0, *pix1, *pix2, *pix3;
   double   px1, py1;
   double   f0, f1, f2, f3, red, green, blue;
   double   a, b, d, e, ww15, hh15, ww25, hh25;

   ww15 = 0.5 * ww1;
   hh15 = 0.5 * hh1;
   ww25 = 0.5 * ww2;
   hh25 = 0.5 * hh2;

   a = cos(angle);
   b = sin(angle);
   d = - sin(angle);
   e = cos(angle);
   
   for (py2 = index; py2 < hh2; py2 += Nwt)                                //  loop through output pixels
   for (px2 = 0; px2 < ww2; px2++)                                         //  outer loop y
   {
      px1 = a * (px2 - ww25) + b * (py2 - hh25) + ww15;                    //  (px1,py1) = corresponding    v.9.3
      py1 = d * (px2 - ww25) + e * (py2 - hh25) + hh15;                    //    point within input pixels

      px0 = int(px1);                                                      //  pixel containing (px1,py1)
      py0 = int(py1);
      
      if (px1 < 0 || px0 >= ww1-1 || py1 < 0 || py0 >= hh1-1) {            //  if outside input pixel array
         pix2 = pixmap2 + (py2 * ww2 + px2) * 3;                           //    output is black
         pix2[0] = pix2[1] = pix2[2] = 0;
         continue;
      }

      pix0 = pixmap1 + (py0 * ww1 + px0) * 3;                              //  4 input pixels based at (px0,py0)
      pix1 = pix0 + ww1 * 3;
      pix2 = pix0 + 3;
      pix3 = pix1 + 3;

      f0 = (px0+1 - px1) * (py0+1 - py1);                                  //  overlap of (px1,py1)
      f1 = (px0+1 - px1) * (py1 - py0);                                    //    in each of the 4 pixels
      f2 = (px1 - px0) * (py0+1 - py1);
      f3 = (px1 - px0) * (py1 - py0);
   
      red =   f0 * pix0[0] + f1 * pix1[0] + f2 * pix2[0] + f3 * pix3[0];   //  sum the weighted inputs
      green = f0 * pix0[1] + f1 * pix1[1] + f2 * pix2[1] + f3 * pix3[1];
      blue =  f0 * pix0[2] + f1 * pix1[2] + f2 * pix2[2] + f3 * pix3[2];
      
      pix2 = pixmap2 + (py2 * ww2 + px2) * 3;                              //  output pixel
      pix2[0] = int(red);
      pix2[1] = int(green);
      pix2[2] = int(blue);
   }
   
   zadd_locked(busy,-1);
   return 0;
}


/**************************************************************************

   PXM *pxm2 = PXM_rotate16(PXM *pxm1, double angle)
   Rotate PXM-16 pixmap through an arbitrary angle (degrees).
   Identical to PXM_rotate8() except for:
      uint8 >> uint16
      rotpxm8 >> rotpxm16
      8 >> 16   

**********/

namespace rotpxm16 {
   int      busy = 0;
   uint16   *pixmap1;
   uint16   *pixmap2;
   int      ww1;
   int      hh1;
   int      ww2;
   int      hh2;
   double   angle;
}


PXM * PXM_rotate16(PXM *pxm1, double anglex)
{
   using namespace rotpxm16;

   void     *PXM_rotate16_thread(void *);

   int      cc, ii;
   PXM      *pxm2;
   
   ww1 = pxm1->ww;
   hh1 = pxm1->hh;
   pixmap1 = (uint16 *) pxm1->bmp;
   angle = anglex;

   while (angle < -180) angle += 360;                                      //  normalize, -180 to +180
   while (angle > 180) angle -= 360;
   angle = angle * pi / 180;                                               //  radians, -pi to +pi
   
   if (fabs(angle) < 0.001) {                                              //  angle = 0 within my precision
      pxm2 = PXM_make(ww1,hh1,16);                                         //  return a copy of the input
      pixmap2 = (uint16 *) pxm2->bmp;
      cc = ww1 * hh1 * 3 * sizeof(uint16);
      memcpy(pixmap2,pixmap1,cc);
      return pxm2;
   }

   ww2 = int(ww1*fabs(cos(angle)) + hh1*fabs(sin(angle)));                 //  rectangle containing rotated image
   hh2 = int(ww1*fabs(sin(angle)) + hh1*fabs(cos(angle)));
   
   pxm2 = PXM_make(ww2,hh2,16);
   pixmap2 = (uint16 *) pxm2->bmp;

   for (ii = 0; ii < Nwt; ii++)                                            //  start worker threads
      start_detached_thread(PXM_rotate16_thread,&wtnx[ii]);
   zadd_locked(busy,+Nwt);

   while (busy) zsleep(0.004);                                             //  wait for completion
   return pxm2;
}


void * PXM_rotate16_thread(void *arg)
{
   using namespace rotpxm16;

   int      index = *((int *) (arg));
   int      px2, py2, px0, py0;
   uint16   *pix0, *pix1, *pix2, *pix3;
   double   px1, py1;
   double   f0, f1, f2, f3, red, green, blue;
   double   a, b, d, e, ww15, hh15, ww25, hh25;

   ww15 = 0.5 * ww1;
   hh15 = 0.5 * hh1;
   ww25 = 0.5 * ww2;
   hh25 = 0.5 * hh2;

   a = cos(angle);
   b = sin(angle);
   d = - sin(angle);
   e = cos(angle);
   
   for (py2 = index; py2 < hh2; py2 += Nwt)                                //  loop through output pixels
   for (px2 = 0; px2 < ww2; px2++)                                         //  outer loop y
   {
      px1 = a * (px2 - ww25) + b * (py2 - hh25) + ww15;                    //  (px1,py1) = corresponding    v.9.3
      py1 = d * (px2 - ww25) + e * (py2 - hh25) + hh15;                    //    point within input pixels

      px0 = int(px1);                                                      //  pixel containing (px1,py1)
      py0 = int(py1);
      
      if (px1 < 0 || px0 >= ww1-1 || py1 < 0 || py0 >= hh1-1) {            //  if outside input pixel array
         pix2 = pixmap2 + (py2 * ww2 + px2) * 3;                           //    output is black
         pix2[0] = pix2[1] = pix2[2] = 0;
         continue;
      }

      pix0 = pixmap1 + (py0 * ww1 + px0) * 3;                              //  4 input pixels based at (px0,py0)
      pix1 = pix0 + ww1 * 3;
      pix2 = pix0 + 3;
      pix3 = pix1 + 3;

      f0 = (px0+1 - px1) * (py0+1 - py1);                                  //  overlap of (px1,py1)
      f1 = (px0+1 - px1) * (py1 - py0);                                    //    in each of the 4 pixels
      f2 = (px1 - px0) * (py0+1 - py1);
      f3 = (px1 - px0) * (py1 - py0);
   
      red =   f0 * pix0[0] + f1 * pix1[0] + f2 * pix2[0] + f3 * pix3[0];   //  sum the weighted inputs
      green = f0 * pix0[1] + f1 * pix1[1] + f2 * pix2[1] + f3 * pix3[1];
      blue =  f0 * pix0[2] + f1 * pix1[2] + f2 * pix2[2] + f3 * pix3[2];
      
      pix2 = pixmap2 + (py2 * ww2 + px2) * 3;                              //  output pixel
      pix2[0] = int(red);
      pix2[1] = int(green);
      pix2[2] = int(blue);
   }
   
   zadd_locked(busy,-1);
   return 0;
}



