#!/usr/bin/env perl
# $Id: katarina-status.pl,v 1.20 2014/02/07 14:32:52 updater Exp $
#
# KATARINA
# (C)2011 C.Bellot
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
# 
 
# 
use strict;
use Getopt::Std;
use File::stat;

my $version = q($Revision: 1.20 $);
my $profile = "UNKNOWN";
my $config_dir = "/etc/katarina/";
my $log_dir = "/var/log/katarina";
my $log_debug;
my $main_pid = $$;
my $backup_dir = "/tmp";

my %conf_targets;
my %conf_globals;
my $max_threads = 1;
my $cycle = 0;
my $graphexport = 0;

#######################################################################
# help
#
sub help() {
    warn "Usage: [-h] [-c seconds] [-p profilename]
    use: -h to read this
    use: -c to cycle every X seconds
    use: -g to export graph data
    use: -p profilename to specify $config_dir/profilename.kat
";
}

#######################################################################
# date_tag
#   give a date prefix for log lines
#
sub date_tag() {
    my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) =
        localtime(time());
    $mon += 1; $year += 1900;
    return sprintf("%4d-%.02d-%.02d-%.02d:%.02d:%.02d [%d:%d]",
        $year,$mon,$mday, $hour,$min,$sec,$main_pid,$$);
}
#######################################################################
# wr_debug
#   write to debug log file and regular log file
#
sub wr_debug($) {
    my $msg = shift;
    print DEBUG &date_tag()." $msg\n";
}
#######################################################################
# mydie
#   exit and fail with a message
#
sub mydie($) {
    my $msg = shift;
    &wr_debug($msg);
    die $msg;
}

#######################################################################
# checks
#   basic directories checks
#
sub get_opts() {
    my %opts;
    getopts('hgc:p:', \%opts) or &mydie("Bad options, use -h for help");
    &help && exit(0) if (defined($opts{h}));
    if (defined($opts{p})) {
        $profile = $opts{p};
    }
    $cycle = $opts{c}if (defined($opts{c}));
    $graphexport = (defined($opts{g}));
}
#######################################################################
# checks
#   basic directories checks
#
sub checks() {
    mkdir $log_dir if (!-d $log_dir);
    mkdir "$log_dir/srv" if (!-d "$log_dir/srv");
    mkdir "$log_dir/stats" if (!-d "$log_dir/stats");
    mkdir "$log_dir/status" if (!-d "$log_dir/status");
    mkdir $config_dir if (!-d $config_dir);
    &mydie("No $profile profile in $config_dir") 
	if ((!-f "$config_dir/$profile.kat") && ($profile ne "UNKNOWN"));
}

#######################################################################
# config_load
#   read config.kat file
#
sub config_load() {
    our %targets;
    our %globals;
    &wr_debug("Loading $config_dir/$profile.kat");
    do "$config_dir/$profile.kat";
    %conf_globals = %globals;
    %conf_targets = %targets;
    $max_threads = $conf_globals{"forks"};
    $backup_dir = $conf_globals{"backup_dir"};
}

#######################################################################
# graph_one
#   give one profile graph export
#
sub graph_one() {
    # For each cycle :
    my $nbcycles = 0;
    while (my ($cycle,$v) = each %{$conf_globals{"cycles"}}) {
	if (-f "$log_dir/status/$profile.$cycle.pid") {
	    my $oldpid=`cat $log_dir/status/$profile.$cycle.pid`;
	    if (!system("ps ax|grep -q \"^ *$oldpid\"")) {
		# Running
		my $detail_lines = "";
		my $tot = 0;
		my $done = 0;
		my $curr = 0;
		for (my $i=0; $i<$max_threads; $i++) {
		    if (-f "$log_dir/status/$profile.$cycle.$i") {
			open(F,"<$log_dir/status/$profile.$cycle.$i");
			my $progress = <F>; close(F); chomp($progress);
			if ($progress =~ /(\d+)\/(\d+)/) {
			    $done += $1; $tot +=$2;
			    if ($1 != $2) { $curr++; }
			}
		    }
		}
		my $sb = stat("$log_dir/status/$profile.$cycle.start");
		my $now = time();
		if ($tot != 0) { 
		    print "$profile;$cycle; $curr; ".$sb->mtime."; $now; ";
		    print int(100*$done/$tot).";\n"; }
		else { 
		    print "$profile;$cycle; 1; ".$sb->mtime."; $now; ";
		    print "0;\n"; 
		}
	    } else {
		# Has ended badly
		print "$profile;$cycle; -1; ***; ***; 100;\n";
	    }
	} else {
	    if (-f "$log_dir/status/$profile.$cycle.end") {
		# Has ended
		open(F,"<$log_dir/status/$profile.$cycle.end");
		my $exit = <F>; close(F); chomp($exit);
		my $sstarttime = 0;
		my $sendtime = 0;
		my $s_start = stat("$log_dir/status/$profile.$cycle.start");
		my $s_end = stat("$log_dir/status/$profile.$cycle.end");
                if ($s_start>0) { $sstarttime = $s_start->mtime; }
                if ($s_end>0) { $sendtime = $s_end->mtime; }
		print "$profile;$cycle; 0; ".$sstarttime."; ".
			$sendtime."; 100;\n";
	    } else {
		# Has never run
		print "$profile;$cycle; -2; ***; ***; 100;\n";
	    }
	}
    }
}

#######################################################################
# graph_all
#   loop through profiles to give graph export
#
sub graph_all() {
    opendir S, $config_dir; my @profiles = readdir S; closedir S;
    while(my $file = shift @profiles) {
	# Ignore . and ..
	next if ($file eq ".") || ($file eq "..");
	# Ignore . and ..
	next if ($file =~ /__template__/);
	next if ($file =~ /^\./);
	if ($file =~ /^(.*)\.kat$/) {
	    $profile = $1;
	    &config_load();
	    &graph_one($profile);
	}
    }
}

#######################################################################
# check_status
#   check status for a profile
#
sub check_status() {
    # For each cycle :
    my $nbcycles = 0;
    while (my ($cycle,$v) = each %{$conf_globals{"cycles"}}) {
	printf "%-30s ","$profile ($cycle)";
	printf $conf_globals{"active"} ? "X " : ". ";

	if (-f "$log_dir/status/$profile.$cycle.pid") {
	    my $oldpid=`cat $log_dir/status/$profile.$cycle.pid`;
	    if (!system("ps ax|grep -q \"^ *$oldpid\"")) {
		printf "X ";
		# Progress
		my $detail_lines = "";
		my $tot = 0;
		my $done = 0;
		for (my $i=0; $i<$max_threads; $i++) {
		    if (-f "$log_dir/status/$profile.$cycle.$i") {
			open(F,"<$log_dir/status/$profile.$cycle.$i");
			my $progress = <F>; close(F); chomp($progress);
			if ($progress =~ /(\d+)\/(\d+)/) {
			    $done += $1; $tot +=$2;
			    #$detail_lines .= (" " x 40) . $progress."\n";
			    $detail_lines .= $progress.",";
			}
		    }
		}
		$detail_lines =~ s/, *$//;
		if ($tot != 0) {
		    printf " ".int(100*$done/$tot)."% ($detail_lines)\n";
		} else {
		    printf " (probably a retry)\n";
		} 
	    } else {
		printf ".  A pid file was left over\n";
	    }
	} else {
	    if (-f "$log_dir/status/$profile.$cycle.end") {
		open(F,"<$log_dir/status/$profile.$cycle.end");
		my $exit = <F>; close(F); chomp($exit);
		printf ".  Last exit with status $exit\n";
	    } else {
		printf ".  Has never run\n";
	    }
	}
    }
}

#######################################################################
# check_all_status
#   all profile or one profile ?
#
sub check_all_status() {
    print "                         Running-+           \n".
      "                        Active-+ |           \n".
      "Profile                        | |  Progress \n".
      "-----------------------------  - -  -------------------------------- \n";

    if ($profile ne "UNKNOWN") {
	&config_load();
	&check_status($profile);
    } else {
	opendir S, $config_dir; my @profiles = readdir S; closedir S;
	while(my $file = shift @profiles) {
	    # Ignore . and ..
	    next if ($file eq ".") || ($file eq "..");
	    # Ignore . and ..
	    next if ($file =~ /__template__/);
	    next if ($file =~ /^\./);
	    if ($file =~ /^(.*)\.kat$/) {
		$profile = $1;
		&config_load();
		&check_status($profile);
	    }
	}
	$profile = "UNKNOWN";
    }
}

#######################################################################
# Main
#
&get_opts();
$log_debug = "$log_dir/katarina-status.debug";

# Open log files
open(DEBUG,">$log_debug");

# Checks and config loading
&checks();

# Graph export ?
if ($graphexport) {
    &graph_all();
    exit 0;
}

# Status loop ?
if ($cycle) {
    system("clear");
    while(1) {
	&check_all_status();
        print "\n- ".&date_tag()."\n";
    	sleep($cycle);
	system("clear");
    }
} else {
    &check_all_status();
}
