#!/usr/bin/env perl
# $Id: katarina-scheduler.pl,v 1.9 2014/01/16 08:23:58 updater Exp $
#
# KATARINA
# (C)2011 C.Bellot
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#

#
use strict;
use Getopt::Std;
use Fcntl qw(:DEFAULT :flock);

my $version = q($Revision: 1.9 $);
my $config_dir = "/etc/katarina/";
my $log_dir = "/var/log/katarina";
my $log_file;
my $log_debug;
my $main_pid = $$;
my $backup_dir = "/tmp";
my $started_on = time();
my @conf_profiles;
my %conf_globals;
my %backuptime;
my $current_profiles = 0;
my $default_backuptime = 3600; # Arbitrary value for profiles with no stat
my $max_threads = 2;
my $best_scenario = "";
my $best_scenario_time = -1;
my @sums;
my @threads;
my @lpids;
my $command = "echo";

#######################################################################
# help
#
sub help() {
    warn "Usage: 
     help mode :
       [-h] [-H]
";
}

#######################################################################
# date_tag
#   give a date prefix for log lines
#
sub date_tag() {
    my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) =
        localtime(time());
    $mon += 1; $year += 1900;
    return sprintf("%4d-%.02d-%.02d-%.02d:%.02d:%.02d [%d:%d]",
        $year,$mon,$mday, $hour,$min,$sec,$main_pid,$$);
}
#######################################################################
# wr_log
#   write to regular log file
#
sub wr_log($) {
    my $msg = shift;
    my $autoflush = $|++;
    print LOG &date_tag()." $msg\n";
    print DEBUG &date_tag()." $msg\n";
    $| = $autoflush;
}
#######################################################################
# wr_debug
#   write to debug log file and regular log file
#
sub wr_debug($) {
    my $msg = shift;
    print DEBUG &date_tag()." $msg\n";
}
#######################################################################
# mydie
#   exit and fail with a message
#
sub mydie($) {
    my $msg = shift;
    &wr_log($msg);
    &wr_log("Aborting");
    die $msg;
}

#######################################################################
# get_opts
#   analyse command arguments
#
sub get_opts() {
    my %opts;
    getopts('hHmsct:n:p:', \%opts) or
        &mydie("Bad options, use -h or -H for help");

    # help : -h or -H 
    &help && exit(0) if (defined($opts{h}));
    system("perldoc $0") && exit(0) if (defined($opts{H}));
}

#######################################################################
# directory_checks
#   basic directories checks
#
sub directory_checks() {
    mkdir $log_dir if (!-d $log_dir);
    mkdir "$log_dir/srv" if (!-d "$log_dir/srv");
    mkdir "$log_dir/stats" if (!-d "$log_dir/stats");
    mkdir $config_dir if (!-d $config_dir);
    &mydie("No configuration found in $config_dir")
        if (!-f "$config_dir/katarina-scheduler.conf");
}

#######################################################################
# config_load
#   read config file
#
sub config_load() {
    our @profiles;
    our %globals;
    &wr_debug("Loading $config_dir/katarina-scheduler.conf");
    do "$config_dir/katarina-scheduler.conf";
    %conf_globals = %globals;
    @conf_profiles = @profiles;
    $max_threads = $conf_globals{"forks"};
    $command = $conf_globals{"command"};

    # Read last profiles stats
    my $c = 0;
    for my $profile (@conf_profiles) {
	my $file = $log_dir."/stats/".$profile."-total";
	$backuptime{$c}->{'profile'} = $profile;
	if (-f $file) {
	    open(F, "<$file");
	    my $v = <F>; chomp($v); close(F);
	    $backuptime{$c}->{'time'} = $v;
	} else {
	    $backuptime{$c}->{'time'} = $default_backuptime;
	}
	&wr_debug("Last time for profile $profile : ".$backuptime{$c}->{'time'});
	$c++;
    }
}

#######################################################################
# find_best_schedule
#   sort times and dispatch the longest ones to run in each queue
#
sub find_best_schedule() {
    my $pos = 0;

    undef @threads;
    undef @sums;

    # Sort using time key
    open (F, "> $log_dir/stats/katarina-scheduler.jobs");
    foreach my $key (sort { $backuptime{$b}->{'time'} <=> $backuptime{$a}->{'time'} } keys %backuptime) {
	print F $backuptime{$key}->{'profile'}." ";
    }
    close(F);
}

#######################################################################
# get_next_job
#   get the next job to execute (in status file)
#
sub get_next_job($) {
    my $id = shift;
    open(F, "+< $log_dir/stats/katarina-scheduler.jobs") or die "can't open job file: $!";
    flock F, LOCK_EX;
    my @lines = split " ",<F>;
    seek F,0,0; truncate F,0;
    my $jobtodo = shift @lines;
    print F join " ",@lines;
    close F or die "Cannot close file\n$!";
    return $jobtodo;
}

#######################################################################
# job_loop
#   execute one thread of jobs one after the other
#
sub job_loop($) {
    my $id = shift;
    my $history = "";
    my $thread_before = time();

    &wr_log("Starting thread $id");
    while ((my $p = &get_next_job($id)) ne "") {
	&wr_log("Thread $id : Starting job $p");
	system("$command $p"); 
	&wr_log("Thread $id : End of job $p");
	$history .= $p." ";
    }
    my $thread_after = time();
    &wr_log("Ending thread $id : history = $history: total time = ".
		($thread_after - $thread_before));
}

#######################################################################
# main_schedule
#   start scheduler
#
sub main_schedule() {
    my $ltotal = $#conf_profiles;
    my @table;
    &find_best_schedule();

    &wr_log("Starting threaded jobs' loop");
    for (my $t=0; $t<$max_threads; $t++) {
        my $pid = fork();
        if ($pid) {
            push(@lpids,$pid);
        } elsif ($pid == 0) {
            &job_loop($t);
            exit(0);
        } else {
            &mydie("Can not fork: $!");
        }
    }

    # Wait for children
    foreach (@lpids) {
        waitpid($_, 0);
    }

}

#######################################################################
# the_end
#   terminate
#
sub the_end() {
    &wr_log("Terminating");
    open(F,">$log_dir/status/katarina-scheduler.end");
    printf(F "0");
    close(F);
    exit 0;
}

#######################################################################
# Main
#
&get_opts();
$log_file = "$log_dir/katarina-scheduler.log";
$log_debug = "$log_dir/katarina-scheduler.out";

# Open log files
open(LOG,">>$log_file");
open(DEBUG,">$log_debug");
&wr_log("Starting Katarina-scheduler $version");

# Checks and config loading
&directory_checks();
&config_load();

&main_schedule();

# End everything
&the_end();


#######################################################################
# Online manual

__END__

=head1 NAME

katarina-scheduler - simple scheduler for katarina backup system

=head1 SYNOPSIS

help mode :
[-h] [-H]

=head1 DESCRIPTION

katarina-scheduler is a simple scheduler built to launch numerous daily 
like jobs. If you only have a few jobs to launch, you should stick to the
cron method.

katarina-scheduler reads a configuration file in 
  $config_dir/katarina-scheduler.conf

Here is a katarina-scheduler.conf example file :

our %globals = (
    "mail"       => "amanda\@localhost",
    "forks"      => 2,
    "command"    => "usr/local/bin/katarina -t backup -m -n daily -p ",
);

our @profiles = (
    "test1",
    "test2",
    "test3",
    "test4",
    "test5",
);

=head1 AUTHOR

Cyril Bellot <jcpp@users.sourceforge.net>

=cut

