<?php

/**
  * SquirrelMail Calendar Plugin File Backend
  * Copyright (C) 2004-2005 Paul Lesneiwski <pdontthink@angrynerds.com>
  * This program is licensed under GPL. See COPYING for details
  *
  */


include_once(SM_PATH . 'plugins/calendar/functions.php');
include_once(SM_PATH . 'plugins/calendar_file_backend/constants.php');
include_once(SM_PATH . 'plugins/calendar_file_backend/functions.php');


/**
  * Get global holiday
  *
  * Retrieves the given global holiday from the backend
  *
  * @param string $holidayID The ID of the holiday to be retrieved
  *
  * @return object A Calendar object corresponding to the desired holiday
  *
  */
function cal_file_get_global_holiday_do($holidayID)
{

   global $HOLIDAY_DIR, $color;
          

   if (file_exists($HOLIDAY_DIR . '/' . $holidayID))
      return getHolidayFromFile($HOLIDAY_DIR . '/' . $holidayID);


   plain_error_message('ERROR IN CALENDAR FILE BACKEND (cal_file_get_global_holiday_do): cannot find holiday file for holiday ID ' . $holidayID, $color);
   exit;

}



/**
  * Get holiday
  *
  * Retrieves the given holiday for the given calendar from the backend
  *
  * @param string $calendarID The ID of the calendar whose holiday is to be retrieved
  * @param string $holidayID The ID of the holiday to be retrieved
  *
  * @return object A Holiday object corresponding to the desired holiday
  *
  */
function cal_file_get_holiday_do($calendarID, $holidayID)
{

   global $CAL_HOLIDAY_DIR_NAME, $EVENT_DIR, $color;


   $file = $EVENT_DIR . '/' . $calendarID . '/' . $CAL_HOLIDAY_DIR_NAME . '/' . $holidayID;

   if (file_exists($file))
      return getHolidayFromFile($file);


   plain_error_message('ERROR IN CALENDAR FILE BACKEND (cal_file_get_holiday_do): cannot find holiday file for holiday ID ' . $holidayID, $color);
   exit;

}



/**
  * Retrieves holiday attributes from the given file into
  * a Holiday object, which it then returns.
  *
  * Note: assumes file exists and is readable!
  *
  * @param string $file The full file path to the desired holiday
  *                     attributes file.
  *
  * @return object A Holiday object corresponding to the given file.
  *
  */
function getHolidayFromFile($file)
{

   // read the file all at once
   // 
   $temp_hol_attributes = file($file);
   

   // concatenate multi-line attributes
   // 
   $hol_attributes = array();
   $count = -1;
   foreach ($temp_hol_attributes as $attributeSet)
   {
      if (strpos($attributeSet, '#') === 0)
         $hol_attributes[$count] .= substr($attributeSet, 1);
      else
      {
         $hol_attributes[++$count] = $attributeSet;
      }  
   }  


   // parse attributes 
   //
   $id = basename($file);
   $name = '';
   $comments = '';
   $administrative_comments = '';
   foreach ($hol_attributes as $attribute)
   {

      $colon = strpos($attribute, ':');
      $value = trim(substr($attribute, $colon + 1));
      $attribute = trim(substr($attribute, 0, $colon));
      switch ($attribute)
      {

         case 'name':
            $name = $value;
            break;

         case 'comments':
            $comments = $value;
            break;

         case 'administrative_comments':
            $administrative_comments = $value;
            break;

      }

   }


   return new Holiday($id, $name, $comments, $administrative_comments);

}



/**
  * Returns a listing of all global holidays
  *
  * @return array An array of Holiday objects
  *
  */
function cal_file_get_all_global_holidays_do()
{

   global $HOLIDAY_DIR;


   // we assume that missing directory simply means
   // no holidays are set up yet
   //
   if (!is_dir($HOLIDAY_DIR)) return array();


   // iterate through each holiday and add to return array
   //
   $holidayList = array();
   $DIR = opendir($HOLIDAY_DIR);
   while (($dirfile = readdir($DIR)) !== FALSE)
   {

      // all holiday files are prefaced with the letter h
      // and have no dots in their names
      //
      if (strpos($dirfile, 'h') !== 0 || strpos($dirfile, '.') !== FALSE)
         continue;


      $file = $HOLIDAY_DIR . '/' . $dirfile;


      if (is_file($file))
      {

         $holiday = getHolidayFromFile($file);


         // instantiate and add holiday to return array
         //
         $holidayList[] = $holiday;

      }

   }

   closedir($DIR);
   return $holidayList;

}



?>
