/* Copyright (C) 2005-2008 Damien Stehle.
Copyright (C) 2007 David Cade.

This file is part of the fplll Library.

The fplll Library is free software; you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation; either version 2.1 of the License, or (at your
option) any later version.

The fplll Library is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
License for more details.

You should have received a copy of the GNU Lesser General Public License
along with the fplll Library; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
MA 02111-1307, USA. */

#ifndef NR_CPP
#define NR_CPP

#define REPRODUCIBLE_RANDOMNESS



template<class Z> inline Z& Z_NR<Z>::GetData()
{
  return data;
}
template<class Z> inline const Z& Z_NR<Z>::GetData() const
{
  return data;
}
template<class Z> inline int Z_NR<Z>::cmp(const Z_NR<Z>& m) const
{
  Z_NR<Z> x;
  x.sub(*this,m);
  return x.sgn();
}

/**********************
 * int specialization *
 **********************/

inline int sizeinbase2(long int data)
{
  long int y=abs(data);
  int resul=0;
  if (y==0) resul=1;
  else{
    while (y>1){
      resul++;
      y>>=1;  //y /=2;
    }
  }
  return resul;
}

template<> inline Z_NR<long int>::Z_NR()
{
}
template<> inline Z_NR<long int>::Z_NR(const Z_NR<long int>& z) : data(z.data)
{
}
template<> inline Z_NR<long int>::~Z_NR()
{
}

template<> inline void Z_NR<long int>::print() const
{
  cout << data;
}
template<> inline void Z_NR<long int>::printerr() const
{
  cerr << data;
}
template<> inline void Z_NR<long int>::read()
{
  cin >> data;
}

template<> inline double Z_NR<long int>::get_d() const
{
  return static_cast<double>(data);
}
template<> inline double Z_NR<long int>::get_d_2exp(signed long int* expo) const
{
  if (data==0) *expo = 0;
  else *expo = sizeinbase2 (data);
  return ldexp( (double) data, -*expo);
}
template<> inline signed long int Z_NR<long int>::get_si() const
{
  return data;
}
template<> inline void Z_NR<long int>::set(const Z_NR<long int>& s)
{
  data=s.GetData();
}
template<> inline void Z_NR<long int>::set(/*const*/ long int& s)
{
  data=s;
}
template<> inline void Z_NR<long int>::set(unsigned long int s)
{
  data=static_cast<long int>(s);
}
template<> inline void Z_NR<long int>::set_si(signed long int s)
{
  data = s;
}
template<> inline void Z_NR<long int>::operator=(const Z_NR<long int>& a)
{
  data = a.data;
}
template<> inline void Z_NR<long int>::operator=(long int a)
{
  data = static_cast<long int>(a);
}

template<> inline int Z_NR<long int>::sgn() const
{
  if (data>0) return 1;
  if (data==0) return 0;
  return -1;
}
template<> inline bool Z_NR<long int>::operator<(const Z_NR<long int>& a) const
{
  return data < a.data;
}
template<> inline bool Z_NR<long int>::operator<(long int a) const
{
  return data < a;
}
template<> inline bool Z_NR<long int>::operator>(const Z_NR<long int>& a) const
{
  return data > a.data;
}
template<> inline bool Z_NR<long int>::operator>(long int a) const
{
  return data > a;
}
template<> inline bool Z_NR<long int>::operator<=(const Z_NR<long int>& a) const
{
  return data <= a.data;
}
template<> inline bool Z_NR<long int>::operator<=(long int a) const
{
  return data <= a;
}
template<> inline bool Z_NR<long int>::operator>=(const Z_NR<long int>& a) const
{
  return data >= a.data;
}
template<> inline bool Z_NR<long int>::operator>=(long int a) const
{
  return data >= a;
}
template<> inline bool Z_NR<long int>::operator==(const Z_NR<long int>& a) const
{
  return data == a.data;
}
template<> inline bool Z_NR<long int>::operator==(long int a) const
{
  return data == a;
}
template<> inline bool Z_NR<long int>::operator!=(const Z_NR<long int>& a) const
{
  return data != a.data;
}
template<> inline bool Z_NR<long int>::operator!=(long int a) const
{
  return data != a;
}

template<> inline void Z_NR<long int>::add(const Z_NR<long int>& a, const Z_NR<long int>& b)
{
  data=a.GetData()+b.GetData();
}
template<> inline void Z_NR<long int>::add_ui(const Z_NR<long int>& a,unsigned int b)
{
  data=a.GetData()+b;
}
template<> inline void Z_NR<long int>::sub(const Z_NR<long int>& a, const Z_NR<long int>& b)
{
  data=a.GetData()-b.GetData();
}
template<> template<> inline void Z_NR<long int>::mul(const Z_NR<long int>& a, const Z_NR<long int>& b)
{
  data=a.GetData()*b.GetData();
}
/*template<> template<> inline void Z_NR<long int>::mul(const Z_NR<long int>& a, const Z_NR<double>& b)
{
  data=a.GetData()*((long int) rint(b.GetData()));
}*/
template<> template<> inline void Z_NR<long int>::mul(const Z_NR<long int>& a, const Z_NR<mpz_t>& b)
{
  data=a.GetData()*mpz_get_si(b.GetData());
}
template<> inline void Z_NR<long int>::mul_si(const Z_NR<long int>& a, signed long int b)
{
  data=a.GetData()*b;
}
template<> inline void Z_NR<long int>::mul_ui(const Z_NR<long int>& a, unsigned long int b)
{
  data=a.GetData()*b;
}
template<> inline void Z_NR<long int>::mul_2exp(const Z_NR<long int>& a,long int b)
{
  data=(long int)ldexp((double)a.GetData(),b);
}
template<> inline void Z_NR<long int>::div_2exp(const Z_NR<long int>& a,long int b)
{
  data=(long int)ldexp((double)a.GetData(),-b);
}

template<> inline void Z_NR<long int>::addmul_ui(const Z_NR<long int>& a,unsigned long int b)
{
  data+=a.GetData()*b;
}
template<> inline void Z_NR<long int>::addmul_si(const Z_NR<long int>& a,signed long int b)
{
  data+=a.GetData()*b;
}
template<> template<> inline void Z_NR<long int>::submul(const Z_NR<long int>& a, const Z_NR<long int>& b)
{
  data-=a.GetData()*b.GetData();
}
template<> template<> inline void Z_NR<long int>::submul(const Z_NR<long int>& a, const Z_NR<mpz_t>& b)
{
  data-=a.GetData()*mpz_get_si(b.GetData());
}
/*template<> template<> inline void Z_NR<long int>::submul(const Z_NR<long int>& a, const Z_NR<double>& b)
{
  data-=a.GetData()*((long int) rint(b.GetData()));
}*/
template<> inline void Z_NR<long int>::submul_ui(const Z_NR<long int>& a, unsigned long int b)
{
  data-=a.GetData()*b;
}

template<> inline void Z_NR<long int>::abs(const Z_NR<long int>& a)
{
  data = labs(a.data);
}
template<> inline void Z_NR<long int>::randb(int bits)
{
  static bool init=false;
  if (!init)
    {
#ifndef REPRODUCIBLE_RANDOMNESS
      int seed = time(NULL); // system time
      srand(seed);
#endif
      init=true;
    }
  data=rand()%(1<<bits);
}
template<> inline void Z_NR<long int>::randm(const Z_NR<long int>& max)
{
  static bool init=false;
  if (!init)
    {
#ifndef REPRODUCIBLE_RANDOMNESS
      int seed = time(NULL); // system time
      srand(seed);
#endif
      init=true;
    }
  data=rand()%max.GetData();
}

/***************************
 * Z=double specialization *
 ***************************/

template<> inline Z_NR<double>::Z_NR()
{
}
template<> inline Z_NR<double>::Z_NR(const Z_NR<double>& z) : data(z.data)
{
}
template<> inline Z_NR<double>::~Z_NR()
{
}

template<> inline void Z_NR<double>::print() const
{
  cout << data;
}
template<> inline void Z_NR<double>::printerr() const
{
  cerr << data;
}
template<> inline void Z_NR<double>::read()
{
  cin >> data;
}

template<> inline double Z_NR<double>::get_d() const
{
  return data;
}
template<> inline double Z_NR<double>::get_d_2exp(signed long int* expo) const
{
  if (data==0.0) *expo = 0;
  return frexp (data, (int *) expo);
}
template<> inline signed long int Z_NR<double>::get_si() const
{
  return (long int) rint(data);
}
template<> inline void Z_NR<double>::set(const Z_NR<double>& s)
{
  data = s.data;
}
template<> inline void Z_NR<double>::set(/*const*/ double& s)
{
  data = s;
}
template<> inline void Z_NR<double>::set(unsigned long int s)
{
  data = static_cast<double>(s);
}
template<> inline void Z_NR<double>::set_si(signed long int s)
{
  data = static_cast<double>(s);
}
template<> inline void Z_NR<double>::operator=(const Z_NR<double>& a)
{
  data = a.data;
}
template<> inline void Z_NR<double>::operator=(long int a)
{
  data = static_cast<double>(a);
}

template<> inline int Z_NR<double>::sgn() const
{
  if (data>0.0) return 1;
  if (data==0.0) return 0;
  return -1;
}
template<> inline bool Z_NR<double>::operator<(const Z_NR<double>& a) const
{
  return data < a.data;
}
template<> inline bool Z_NR<double>::operator<(long int a) const
{
  return data < static_cast<double>(a);
}
template<> inline bool Z_NR<double>::operator>(const Z_NR<double>& a) const
{
  return data > a.data;
}
template<> inline bool Z_NR<double>::operator>(long int a) const
{
  return data > static_cast<double>(a);
}
template<> inline bool Z_NR<double>::operator<=(const Z_NR<double>& a) const
{
  return data <= a.data;
}
template<> inline bool Z_NR<double>::operator<=(long int a) const
{
  return data <= static_cast<double>(a);
}
template<> inline bool Z_NR<double>::operator>=(const Z_NR<double>& a) const
{
  return data >= a.data;
}
template<> inline bool Z_NR<double>::operator>=(long int a) const
{
  return data >= static_cast<double>(a);
}
template<> inline bool Z_NR<double>::operator==(const Z_NR<double>& a) const
{
  return data == a.data;
}
template<> inline bool Z_NR<double>::operator==(long int a) const
{
  return data == static_cast<double>(a);
}
template<> inline bool Z_NR<double>::operator!=(const Z_NR<double>& a) const
{
  return data != a.data;
}
template<> inline bool Z_NR<double>::operator!=(long int a) const
{
  return data != static_cast<double>(a);
}

template<> inline void Z_NR<double>::add(const Z_NR<double>& a, const Z_NR<double>& b)
{
  data=a.GetData()+b.GetData();
}
template<> inline void Z_NR<double>::add_ui(const Z_NR<double>& a, unsigned int b)
{
  data=a.GetData()+((double) b);
}
template<> inline void Z_NR<double>::sub(const Z_NR<double>& a, const Z_NR<double>& b)
{
  data=a.GetData()-b.GetData();
}
template<> template<> inline void Z_NR<double>::mul(const Z_NR<double>& a, const Z_NR<double>& b)
{
  data=a.GetData()*b.GetData();
}
template<> template<> inline void Z_NR<double>::mul(const Z_NR<double>& a, const Z_NR<mpz_t>& b)
{
  data=a.GetData()*mpz_get_d(b.GetData());
}
template<> template<> inline void Z_NR<double>::mul(const Z_NR<double>& a, const Z_NR<long int>& b)
{
  data=a.GetData()*((double) b.GetData());
}
template<> inline void Z_NR<double>::mul_si(const Z_NR<double>& a, long signed int b)
{
  data=a.GetData()*((double) b);
}
template<> inline void Z_NR<double>::mul_ui(const Z_NR<double>& a, long unsigned int b)
{
  data=a.GetData()*((double) b);
}
template<> inline void Z_NR<double>::mul_2exp(const Z_NR<double>& a, long int b)
{
  data=ldexp(a.GetData(),b);
}
template<> inline void Z_NR<double>::div_2exp(const Z_NR<double>& a, long int b)
{
  data=ldexp(a.GetData(),-b);
}
template<> inline void Z_NR<double>::addmul_ui(const Z_NR<double>& a, unsigned long int b)
{
  data+=a.GetData()*((double) b);
}
template<> inline void Z_NR<double>::addmul_si(const Z_NR<double>& a, signed long int b)
{
  data+=a.GetData()*((double) b);
}
template<> template<> inline void Z_NR<double>::submul(const Z_NR<double>& a, const Z_NR<double>& b)
{
  data-=a.GetData()*b.GetData();
}
template<> template<> inline void Z_NR<double>::submul(const Z_NR<double>& a, const Z_NR<mpz_t>& b)
{
  data-=a.GetData()*mpz_get_d(b.GetData());
}
template<> template<> inline void Z_NR<double>::submul(const Z_NR<double>& a, const Z_NR<long int>& b)
{
  data-=a.GetData()*((double) b.GetData());
}
template<> inline void Z_NR<double>::submul_ui(const Z_NR<double>& a, unsigned long int b)
{
  data-=a.GetData()*((double) b);
}

template<> inline void Z_NR<double>::abs(const Z_NR<double>& a)
{
  data = fabs(a.data);
}
template<> inline void Z_NR<double>::randb(int bits)
{
  static bool init=false;
  if (!init)
    {
#ifndef REPRODUCIBLE_RANDOMNESS
      int seed = time(NULL); // system time
      srand(seed);
#endif
      init=true;
    }
  data=(double) (rand()%(1<<bits));
}
template<> inline void Z_NR<double>::randm(const Z_NR<double>& max)
{
  static bool init=false;
  if (!init)
    {
#ifndef REPRODUCIBLE_RANDOMNESS
      int seed = time(NULL); // system time
      srand(seed);
#endif
      init=true;
    }
  data= ((double) rand()) / ((double) LONG_MAX) * max.GetData();
}

/**********************
 * mpz specialization *
 **********************/

template<> inline Z_NR<mpz_t>::Z_NR()
{
  mpz_init(data);
}
template<> inline Z_NR<mpz_t>::Z_NR(const Z_NR<mpz_t>& z)
{
  mpz_init_set(data, z.data);
}
template<> inline Z_NR<mpz_t>::~Z_NR()
{
  mpz_clear(data);
}

template<> inline void Z_NR<mpz_t>::print() const
{
  mpz_out_str(stdout,10,data);
  fflush(stdout);
}
template<> inline void Z_NR<mpz_t>::printerr() const
{
  mpz_out_str(stderr,10,data);
  fflush(stderr);
}
template<> inline void Z_NR<mpz_t>::read()
{
  mpz_inp_str(data,stdin,0);
}

template<> inline double Z_NR<mpz_t>::get_d() const
{
  return mpz_get_d(data);
}
template<> inline double Z_NR<mpz_t>::get_d_2exp(signed long int* expo) const
{
  return mpz_get_d_2exp(expo,data);
}
template<> inline signed long int Z_NR<mpz_t>::get_si() const
{
  return mpz_get_si(data);
}
template<> inline void Z_NR<mpz_t>::set(const Z_NR<mpz_t>& d)
{
  mpz_set(data,d.GetData());
}
template<> inline void Z_NR<mpz_t>::set(/*const*/ mpz_t& d)
{
  mpz_set(data,d);
}
template<> inline void Z_NR<mpz_t>::set(unsigned long int d)
{
  mpz_set_ui(data,d);
}
template<> inline void Z_NR<mpz_t>::set_si(signed long int d)
{
  mpz_set_si(data,d);
}
template<> inline void Z_NR<mpz_t>::operator=(const Z_NR<mpz_t>& a)
{
  mpz_set(data, a.data);
}
template<> inline void Z_NR<mpz_t>::operator=(long int a)
{
  mpz_set_si(data, a);
}

template<> inline int Z_NR<mpz_t>::sgn() const
{
  return mpz_sgn(data);
}
template<> inline bool Z_NR<mpz_t>::operator<(const Z_NR<mpz_t>& a) const
{
  return mpz_cmp(data, a.data) < 0;
}
template<> inline bool Z_NR<mpz_t>::operator<(long int a) const
{
  return mpz_cmp_si(data, a) < 0;
}
template<> inline bool Z_NR<mpz_t>::operator>(const Z_NR<mpz_t>& a) const
{
  return mpz_cmp(data, a.data) > 0;
}
template<> inline bool Z_NR<mpz_t>::operator>(long int a) const
{
  return mpz_cmp_si(data, a) > 0;
}
template<> inline bool Z_NR<mpz_t>::operator<=(const Z_NR<mpz_t>& a) const
{
  return mpz_cmp(data, a.data) <= 0;
}
template<> inline bool Z_NR<mpz_t>::operator<=(long int a) const
{
  return mpz_cmp_si(data, a) <= 0;
}
template<> inline bool Z_NR<mpz_t>::operator>=(const Z_NR<mpz_t>& a) const
{
  return mpz_cmp(data, a.data) >= 0;
}
template<> inline bool Z_NR<mpz_t>::operator>=(long int a) const
{
  return mpz_cmp_si(data, a) >= 0;
}
template<> inline bool Z_NR<mpz_t>::operator==(const Z_NR<mpz_t>& a) const
{
  return mpz_cmp(data, a.data) == 0;
}
template<> inline bool Z_NR<mpz_t>::operator==(long int a) const
{
  return mpz_cmp_si(data, a) == 0;
}
template<> inline bool Z_NR<mpz_t>::operator!=(const Z_NR<mpz_t>& a) const
{
  return mpz_cmp(data, a.data) != 0;
}
template<> inline bool Z_NR<mpz_t>::operator!=(long int a) const
{
  return mpz_cmp_si(data, a) != 0;
}

template<> inline void Z_NR<mpz_t>::add(const Z_NR<mpz_t>& a, const Z_NR<mpz_t>& b)
{
  mpz_add(data,a.GetData(),b.GetData());
}
template<> inline void Z_NR<mpz_t>::add_ui(const Z_NR<mpz_t>& a, unsigned int b)
{
  mpz_add_ui(data,a.GetData(),b);
}
template<> inline void Z_NR<mpz_t>::sub(const Z_NR<mpz_t>& a, const Z_NR<mpz_t>& b)
{
  mpz_sub(data,a.GetData(),b.GetData());
}
template<> template<> inline void Z_NR<mpz_t>::mul(const Z_NR<mpz_t>& a, const Z_NR<mpz_t>& b)
{
  mpz_mul(GetData(),a.GetData(),b.GetData());
}
template<> template<> inline void Z_NR<mpz_t>::mul(const Z_NR<mpz_t>& a, const Z_NR<long int>& b)
{
  mpz_mul_si(GetData(),a.GetData(),b.GetData());
}
/*template<> template<> inline void Z_NR<mpz_t>::mul(const Z_NR<mpz_t>& a, const Z_NR<double>& b)
{
  //warning: conversion double->long->mpz_t
  mpz_mul_si(GetData(),a.GetData(),(long int) rint(b.GetData()));
}*/
template<> inline void Z_NR<mpz_t>::mul_si(const Z_NR<mpz_t>& a, signed long int b)
{
  mpz_mul_si(data,a.GetData(),b);
}
template<> inline void Z_NR<mpz_t>::mul_si(const Z_NR<mpz_t>& a, const Z_NR<long int>&b)
{
  mpz_mul_si(data,a.GetData(),b.GetData());
}
template<> inline void Z_NR<mpz_t>::mul_ui(const Z_NR<mpz_t>& a, unsigned long int b)
{
  mpz_mul_ui(data,a.GetData(),b);
}
template<> inline void Z_NR<mpz_t>::mul_2exp(const Z_NR<mpz_t>& a, long int b)
{
  mpz_mul_2exp(data,a.GetData(),b);
}
template<> inline void Z_NR<mpz_t>::div_2exp(const Z_NR<mpz_t>& a, long int b)
{
  mpz_div_2exp(data,a.GetData(),b);
}
template<> template<> inline void Z_NR<mpz_t>::addmul(const Z_NR<mpz_t>& a, const Z_NR<long int>& b)
{
  if (b.sgn()>0)
    mpz_addmul_ui(data,a.GetData(),b.GetData());
  else
    mpz_submul_ui(data,a.GetData(),-(b.GetData()));
}
/*template<> template<> inline void Z_NR<mpz_t>::addmul(const Z_NR<mpz_t>& a, const Z_NR<double>& b)
{
  //warning: conversion double->long->mpz_t
  if (b.sgn()>0)
    mpz_addmul_ui(data,a.GetData(),(long int) rint(b.GetData()));
  else
    mpz_submul_ui(data,a.GetData(),-((long int) rint(b.GetData())));
}*/
template<> template<> inline void Z_NR<mpz_t>::addmul(const Z_NR<mpz_t>& a, const Z_NR<mpz_t>& b)
{
  mpz_addmul(data,a.GetData(),b.GetData());
}
template<> inline void Z_NR<mpz_t>::addmul_ui(const Z_NR<mpz_t>& a, unsigned long int b)
{
  mpz_addmul_ui(data,a.GetData(),b);
}
template<> inline void Z_NR<mpz_t>::addmul_si(const Z_NR<mpz_t>& a, signed long int b)
{
  if (b >= 0)
    mpz_addmul_ui(data, a.data, static_cast<unsigned long int>(b));
  else
    mpz_submul_ui(data, a.data, static_cast<unsigned long int>(-b));
}
template<> template<> inline void Z_NR<mpz_t>::submul(const Z_NR<mpz_t>& a, const Z_NR<long int>& b)
{
  if (b.sgn()>0)
    mpz_submul_ui(data,a.GetData(),b.GetData());
  else
    mpz_addmul_ui(data,a.GetData(),-(b.GetData()));
}
/*template<> template<> inline void Z_NR<mpz_t>::submul(const Z_NR<mpz_t>& a, const Z_NR<double>& b)
{
  //warning: conversion double->long->mpz_t
  if (b.sgn()>0)
    mpz_submul_ui(data,a.GetData(),(long int) rint(b.GetData()));
  else
    mpz_addmul_ui(data,a.GetData(),-((long int) rint(b.GetData())));
}*/
template<> template<> inline void Z_NR<mpz_t>::submul(const Z_NR<mpz_t>& a, const Z_NR<mpz_t>& b)
{
  mpz_submul(data,a.GetData(),b.GetData());
}
template<> inline void Z_NR<mpz_t>::submul_ui(const Z_NR<mpz_t>& a,unsigned long int b)
{
  mpz_submul_ui(data,a.GetData(),b);
}

template<> inline void Z_NR<mpz_t>::abs(const Z_NR<mpz_t>& x)
{
  mpz_abs(data,x.data);
}
template<> inline void Z_NR<mpz_t>::randb(int bits)
{
  static bool init=false;
  static gmp_randstate_t state;
  if (!init)
    {
#ifndef REPRODUCIBLE_RANDOMNESS
      unsigned long seed = time(NULL); // system time
#endif
      gmp_randinit_default(state);
#ifndef REPRODUCIBLE_RANDOMNESS
      gmp_randseed_ui(state, seed);
#endif
      init=true;
    }
  mpz_urandomb(data,state,bits);
}
template<> inline void Z_NR<mpz_t>::randm(const Z_NR<mpz_t>& max)
{
  static bool init=false;
  static gmp_randstate_t state;
  if (!init)
    {
#ifndef REPRODUCIBLE_RANDOMNESS
      unsigned long seed = time(NULL); // system time
#endif
      gmp_randinit_default(state);
#ifndef REPRODUCIBLE_RANDOMNESS
      gmp_randseed_ui(state, seed);
#endif
      init=true;
    }
  mpz_urandomm(data,state,max.GetData());
}


/*********
 * FP_NR *
 *********/

template<class F> inline F& FP_NR<F>::GetData()
{
  return data;
}
template<class F> inline const F& FP_NR<F>::GetData() const
{
  return data;
}

template<class F> inline void FP_NR<F>::addmul(const FP_NR<F>& b, const FP_NR<F>& c, mp_rnd_t rnd)
{
  FP_NR<F> product;
  product.mul(b, c, rnd);
  add(*this, product, rnd);
}
template<class F> inline void FP_NR<F>::submul(const FP_NR<F>& b, const FP_NR<F>& c, mp_rnd_t rnd)
{
  FP_NR<F> product;
  product.mul(b, c, rnd);
  sub(*this, product, rnd);
}

/*************************
 * double specialization *
 *************************/

template<> inline FP_NR<double>::FP_NR()
{
}
template<> inline FP_NR<double>::FP_NR(const FP_NR<double>& f)
{
  data = f.data;
}
template<> inline FP_NR<double>::~FP_NR()
{
}
template<> inline void FP_NR<double>::print() const
{
  cout << data;
}
template<> inline void FP_NR<double>::printerr() const
{
  cerr << data;
}
template<> inline double FP_NR<double>::get() const
{
  return static_cast<double>(data);
}
template<> inline double FP_NR<double>::get_d(mp_rnd_t rnd) const
{
  return static_cast<double>(data);
}
template<> inline signed long int FP_NR<double>::get_si() const
{
  return static_cast<signed long int>(data);
}
template<> inline void FP_NR<double>::set(const FP_NR<double>& s)
{
  data=s.GetData();
}
template<> inline void FP_NR<double>::set(double s)
{
  data=static_cast<double>(s);
}
template<> inline void FP_NR<double>::set(unsigned int s)
{
  data=static_cast<double>(s);
}
template<> inline void FP_NR<double>::operator=(const FP_NR<double>& f)
{
  data = f.data;
}
template<> inline void FP_NR<double>::operator=(double d)
{
  data = d;
}

template<> inline void FP_NR<double>::add(const FP_NR<double>& b, const FP_NR<double>& c, mp_rnd_t rnd)
{
  data= b.GetData()+c.GetData();
}
template<> inline void FP_NR<double>::sub(const FP_NR<double>& b, const FP_NR<double>& c, mp_rnd_t rnd)
{
  data= b.GetData()-c.GetData();
}
template<> inline void FP_NR<double>::neg(const FP_NR<double>& b)
{
  data=-b.GetData();
}
template<> inline void FP_NR<double>::mul(const FP_NR<double>& b, const FP_NR<double>& c, mp_rnd_t rnd)
{
  data= b.GetData()*c.GetData();
}
template<> inline void FP_NR<double>::mul_2ui(const FP_NR<double>& b, unsigned int c)
{
  data=b.GetData()*(1<<c);
}
template<> inline void FP_NR<double>::div(const FP_NR<double>& b, const FP_NR<double>& c, mp_rnd_t rnd)
{
  data= b.GetData()/c.GetData();
}
template<> inline void FP_NR<double>::div_2ui(const FP_NR<double>& b, unsigned int c)
{
  data=b.GetData()/(1<<c);
}

template<> inline int  FP_NR<double>::cmp(const FP_NR<double>& b) const
{
  if (data<=b.GetData())
    return -1;
  else
    return 1;
}
template<> inline int  FP_NR<double>::cmp(double b) const
{
  if (data<=b)
    return -1;
  else
    return 1;
}
template<> inline int FP_NR<double>::sgn() const
{
  if (data>0)
    return 1;
  if (data==0)
    return 0;
  return -1;
}
template<> inline bool FP_NR<double>::operator<=(const FP_NR<double>& a) const
{
  return data <= a.data;
}
template<> inline bool FP_NR<double>::operator<=(double a) const
{
  return data <= a;
}
template<> inline bool FP_NR<double>::operator>=(const FP_NR<double>& a) const
{
  return data >= a.data;
}
template<> inline bool FP_NR<double>::operator>=(double a) const
{
  return data >= a;
}

template<> inline void FP_NR<double>::abs(const FP_NR<double>& b)
{
  data=b.GetData();
  if (data<0) data=-data;
}
template<> inline void FP_NR<double>::rnd(const FP_NR<double>& b)
{
  data=rint(b.GetData());
}
template<> inline int FP_NR<double>::exp() const
{
  return ilogb(data)+1;
}
template<> inline int FP_NR<double>::zero_p() const
{
  return (data==0);
}
template<> inline void FP_NR<double>::set_nan()
{
  data=NAN; // 0.0/0.0;
}
template<> inline int FP_NR<double>::is_nan() const
{
  return (data!=data);
}
template<> inline void FP_NR<double>::sqrt(const FP_NR<double>& s, mp_rnd_t rnd)
{
  data=std::sqrt(s.GetData());
}
template<> inline void FP_NR<double>::exponential(const FP_NR<double>& a, mp_rnd_t rnd)
{
  data = ::exp(a.data);
}
template<> inline void FP_NR<double>::log(const FP_NR<double>& a, mp_rnd_t rnd)
{
  data = ::log(a.data);
}

template<> inline void FP_NR<double>::setprec(unsigned int prec)
{
  // ignored
}


/*************
 * DPE spec. *
 *************/

#define dpe_ncref const_cast<dpe_t&>

template<> inline FP_NR<dpe_t>::FP_NR()
{
  dpe_init(data);
}
template<> inline FP_NR<dpe_t>::FP_NR(const FP_NR<dpe_t>& f)
{
  dpe_init(data);
  dpe_set(data, dpe_ncref(f.data));
}
template<> inline FP_NR<dpe_t>::~FP_NR()
{
  dpe_clear(data);
}

template<> inline void FP_NR<dpe_t>::print() const
{
  dpe_out_str(stdout,10, dpe_ncref(data));
  fflush(stdout);
}
template<> inline void FP_NR<dpe_t>::printerr() const
{
  dpe_out_str(stderr,10, dpe_ncref(data));
  fflush(stderr);
}

template<> inline double FP_NR<dpe_t>::get() const
{
  return dpe_get_d(dpe_ncref(data));
}
template<> inline double FP_NR<dpe_t>::get_d(mp_rnd_t rnd) const
{
  return dpe_get_d(dpe_ncref(data));
}
template<> inline signed long int FP_NR<dpe_t>::get_si() const
{
  return dpe_get_si(dpe_ncref(data));
}
template<> inline void FP_NR<dpe_t>::set(const FP_NR<dpe_t>& f)
{
  dpe_set(data, dpe_ncref(f.data));
}
template<> inline void FP_NR<dpe_t>::set(double d)
{
  dpe_set_d(data, d);
}
template<> inline void FP_NR<dpe_t>::set(unsigned int s)
{
  dpe_set_d(data, static_cast<double>(s));
}
template<> inline void FP_NR<dpe_t>::operator=(const FP_NR<dpe_t>& f)
{
  dpe_set(data, dpe_ncref(f.data));
}
template<> inline void FP_NR<dpe_t>::operator=(double d)
{
  dpe_set_d(data, d);
}

template<> inline void FP_NR<dpe_t>::add(const FP_NR<dpe_t>& a, const FP_NR<dpe_t>& b, mp_rnd_t rnd)
{
  dpe_add(data, dpe_ncref(a.data), dpe_ncref(b.data));
}
template<> inline void FP_NR<dpe_t>::sub(const FP_NR<dpe_t>& a, const FP_NR<dpe_t>& b, mp_rnd_t rnd)
{
  dpe_sub(data, dpe_ncref(a.data), dpe_ncref(b.data));
}
template<> inline void FP_NR<dpe_t>::neg(const FP_NR<dpe_t>& a)
{
  dpe_neg(data, dpe_ncref(a.data));
}
template<> inline void FP_NR<dpe_t>::mul(const FP_NR<dpe_t>& a, const FP_NR<dpe_t>& b, mp_rnd_t rnd)
{
  dpe_mul(data, dpe_ncref(a.data), dpe_ncref(b.data));
}
template<> inline void FP_NR<dpe_t>::mul_2ui(const FP_NR<dpe_t>& a, unsigned int b)
{
  dpe_mul_2exp(data, dpe_ncref(a.data),b);
}
template<> inline void FP_NR<dpe_t>::div(const FP_NR<dpe_t>& a, const FP_NR<dpe_t>& b, mp_rnd_t rnd)
{
  dpe_div(data, dpe_ncref(a.data), dpe_ncref(b.data));
}
template<> inline void FP_NR<dpe_t>::div_2ui(const FP_NR<dpe_t>& a, unsigned int b)
{
  dpe_div_2exp(data, dpe_ncref(a.data),b);
}

template<> inline int FP_NR<dpe_t>::cmp(const FP_NR<dpe_t>& a) const
{
  return dpe_cmp(dpe_ncref(data), dpe_ncref(a.data));
}
template<> inline int FP_NR<dpe_t>::cmp(double a) const
{
  return dpe_cmp_d(dpe_ncref(data), a);
}
template<> inline int FP_NR<dpe_t>::sgn() const
{
  return cmp(0.0);
}
template<> inline bool FP_NR<dpe_t>::operator<=(const FP_NR<dpe_t>& a) const
{
  return dpe_cmp(dpe_ncref(data), dpe_ncref(a.data)) <= 0;
}
template<> inline bool FP_NR<dpe_t>::operator<=(double a) const
{
  return dpe_cmp_d(dpe_ncref(data), a) <= 0;
}
template<> inline bool FP_NR<dpe_t>::operator>=(const FP_NR<dpe_t>& a) const
{
  return dpe_cmp(dpe_ncref(data), dpe_ncref(a.data)) >= 0;
}
template<> inline bool FP_NR<dpe_t>::operator>=(double a) const
{
  return dpe_cmp_d(dpe_ncref(data), a) >= 0;
}

template<> inline void FP_NR<dpe_t>::abs(const FP_NR<dpe_t>& a)
{
  dpe_abs(data, dpe_ncref(a.data));
}
template<> inline void FP_NR<dpe_t>::rnd(const FP_NR<dpe_t>& a)
{
  dpe_round(data, dpe_ncref(a.data));
}
template<> inline int FP_NR<dpe_t>::exp() const
{
  return DPE_EXP(data);
}
template<> inline int FP_NR<dpe_t>::zero_p() const
{
  return dpe_zero_p(dpe_ncref(data));
}
template<> inline void FP_NR<dpe_t>::set_nan()
{
  dpe_set_d(data, NAN/*0.0/0.0*/);
}
template<> inline int FP_NR<dpe_t>::is_nan() const
{
  return (DPE_MANT(data)!=DPE_MANT(data));
}
template<> inline void FP_NR<dpe_t>::sqrt(const FP_NR<dpe_t>& a, mp_rnd_t rnd)
{
  dpe_sqrt(data, dpe_ncref(a.data));
}

template<> inline void FP_NR<dpe_t>::setprec(unsigned int prec)
{
  // ignored
}

#undef dpe_ncref

/*************
 * MPFR spec. *
 *************/

template<> inline FP_NR<mpfr_t>::FP_NR()
{
  mpfr_init(data);
}
template<> inline FP_NR<mpfr_t>::FP_NR(const FP_NR<mpfr_t>& f)
{
  mpfr_init_set(data, f.data, GMP_RNDN);
}
template<> inline FP_NR<mpfr_t>::~FP_NR()
{
  mpfr_clear(data);
}

template<> inline void FP_NR<mpfr_t>::print() const
{
  mpfr_out_str(stdout,10,10,data,GMP_RNDN);
  fflush(stdout);
}
template<> inline void FP_NR<mpfr_t>::printerr() const
{
  mpfr_out_str(stderr,10,5,data,GMP_RNDN);
  fflush(stderr);
}

template<> inline double FP_NR<mpfr_t>::get() const
{
  return mpfr_get_d(data,GMP_RNDN);
}
template<> inline double FP_NR<mpfr_t>::get_d(mp_rnd_t rnd) const
{
  return mpfr_get_d(data, rnd);
}
template<> inline signed long int FP_NR<mpfr_t>::get_si() const
{
  return mpfr_get_si(data,GMP_RNDN);
}
template<> inline void FP_NR<mpfr_t>::set(const FP_NR<mpfr_t>& f)
{
  mpfr_set(data,f.GetData(),GMP_RNDN);
}
template<> inline void FP_NR<mpfr_t>::set(double d)
{
  mpfr_set_d(data,d,GMP_RNDN);
}
template<> inline void FP_NR<mpfr_t>::set(unsigned int s)
{
  mpfr_set_ui(data,s,GMP_RNDN);
}
template<> inline void FP_NR<mpfr_t>::operator=(const FP_NR<mpfr_t>& a)
{
  mpfr_set(data, a.data, GMP_RNDN);
}
template<> inline void FP_NR<mpfr_t>::operator=(double a)
{
  mpfr_set_d(data, a, GMP_RNDN);
}

template<> inline void FP_NR<mpfr_t>::add(const FP_NR<mpfr_t>& a, const FP_NR<mpfr_t>& b, mp_rnd_t rnd)
{
  mpfr_add(data,a.GetData(),b.GetData(),rnd);
}
template<> inline void FP_NR<mpfr_t>::sub(const FP_NR<mpfr_t>& a, const FP_NR<mpfr_t>& b, mp_rnd_t rnd)
{
  mpfr_sub(data,a.GetData(),b.GetData(),rnd);
}
template<> inline void FP_NR<mpfr_t>::neg(const FP_NR<mpfr_t>& a)
{
  mpfr_neg(data,a.GetData(),GMP_RNDN);
}
template<> inline void FP_NR<mpfr_t>::mul(const FP_NR<mpfr_t>& a, const FP_NR<mpfr_t>& b, mp_rnd_t rnd)
{
  mpfr_mul(data,a.GetData(),b.GetData(),rnd);
}
template<> inline void FP_NR<mpfr_t>::mul_2ui(const FP_NR<mpfr_t>& a, unsigned int b)
{
  mpfr_mul_2exp(data,a.GetData(),b,GMP_RNDN);
}
template<> inline void FP_NR<mpfr_t>::div(const FP_NR<mpfr_t>& a, const FP_NR<mpfr_t>& b, mp_rnd_t rnd)
{
  mpfr_div(data,a.GetData(),b.GetData(),rnd);
}
template<> inline void FP_NR<mpfr_t>::div_2ui(const FP_NR<mpfr_t>& a, unsigned int b)
{
  mpfr_div_2ui(data,a.GetData(),b,GMP_RNDN);
}

template<> inline int FP_NR<mpfr_t>::cmp(const FP_NR<mpfr_t>& a) const
{
  return mpfr_cmp(data,a.GetData());
}
template<> inline int FP_NR<mpfr_t>::cmp(double a) const
{
  return mpfr_cmp_d(data,a);
}
template<> inline int FP_NR<mpfr_t>::sgn() const
{
  return mpfr_sgn(data);
}
template<> inline bool FP_NR<mpfr_t>::operator<=(const FP_NR<mpfr_t>& a) const
{
  return mpfr_cmp(data, a.data) <= 0;
}
template<> inline bool FP_NR<mpfr_t>::operator<=(double a) const
{
  return mpfr_cmp_d(data, a) <= 0;
}
template<> inline bool FP_NR<mpfr_t>::operator>=(const FP_NR<mpfr_t>& a) const
{
  return mpfr_cmp(data, a.data) >= 0;
}
template<> inline bool FP_NR<mpfr_t>::operator>=(double a) const
{
  return mpfr_cmp_d(data, a) >= 0;
}

template<> inline void FP_NR<mpfr_t>::abs(const FP_NR<mpfr_t>& a)
{
  mpfr_abs(data,a.GetData(),GMP_RNDN);
}
template<> inline void FP_NR<mpfr_t>::rnd(const FP_NR<mpfr_t>& a)
{
  mpfr_round(data,a.GetData());
}
template<> inline void FP_NR<mpfr_t>::floor(const FP_NR<mpfr_t>& a)
{
  mpfr_floor(data, a.data);
}
template<> inline int FP_NR<mpfr_t>::exp() const
{
  return mpfr_get_exp(data);
}
template<> inline int FP_NR<mpfr_t>::zero_p() const
{
  return mpfr_zero_p(data);
}
template<> inline void FP_NR<mpfr_t>::set_nan()
{
  mpfr_set_nan(data);
}
template<> inline int FP_NR<mpfr_t>::is_nan() const
{
  return mpfr_nan_p(data);
}
template<> inline void FP_NR<mpfr_t>::sqrt(const FP_NR<mpfr_t>& a, mp_rnd_t rnd)
{
  mpfr_sqrt(data,a.GetData(), rnd);
}
template<> inline void FP_NR<mpfr_t>::exponential(const FP_NR<mpfr_t>& a, mp_rnd_t rnd) {
  mpfr_exp(data, a.data, rnd);
}
template<> inline void FP_NR<mpfr_t>::log(const FP_NR<mpfr_t>& a, mp_rnd_t rnd) {
  mpfr_log(data, a.data, rnd);
}

template<> inline void FP_NR<mpfr_t>::addmul(const FP_NR<mpfr_t>& b, const FP_NR<mpfr_t>& c, mp_rnd_t rnd)
{
  mpfr_fma(data, b.data, c.data, data, rnd);
}
template<> inline void FP_NR<mpfr_t>::submul(const FP_NR<mpfr_t>& b, const FP_NR<mpfr_t>& c, mp_rnd_t rnd)
{
  mpfr_fms(data, b.data, c.data, data, rnd);
  mpfr_neg(data, data, GMP_RNDN); // Exact
}

template<> inline void FP_NR<mpfr_t>::setprec(unsigned int prec) {
  mpfr_set_default_prec (prec);
};


/* Conversions
   =========== */

template<> template<>
inline void Z_NR<long int>::set_f(const FP_NR<double>& a) {
  data = a.get_si();
}
template<> template<>
inline void Z_NR<long int>::set_f(const FP_NR<dpe_t>& a) {
  data = a.get_si();
}
template<> template<>
inline void Z_NR<long int>::set_f(const FP_NR<mpfr_t>& a) {
  data = a.get_si();
}

template<> template<>
inline void Z_NR<double>::set_f(const FP_NR<double>& a) {
  data = a.get_d();
}
template<> template<>
inline void Z_NR<double>::set_f(const FP_NR<dpe_t>& a) {
  data = a.get_d();
}
template<> template<>
inline void Z_NR<double>::set_f(const FP_NR<mpfr_t>& a) {
  data = a.get_d();
}

template<> template<>
inline void Z_NR<mpz_t>::set_f(const FP_NR<double>& a) {
  mpz_set_d(data, a.GetData());
}
template<> template<>
inline void Z_NR<mpz_t>::set_f(const FP_NR<dpe_t>& a) {
  dpe_get_z(data, const_cast<dpe_t&>(a.GetData()));
}
template<> template<>
inline void Z_NR<mpz_t>::set_f(const FP_NR<mpfr_t>& a) {
  mpfr_get_z(data, a.GetData(), GMP_RNDN);
}

template<> template<>
inline void FP_NR<double>::set_z(const Z_NR<long int>& a, mp_rnd_t rnd) {
  data = a.get_d();
}
template<> template<>
inline void FP_NR<double>::set_z(const Z_NR<double>& a, mp_rnd_t rnd) {
  data = a.get_d();
}
template<> template<>
inline void FP_NR<double>::set_z(const Z_NR<mpz_t>& a, mp_rnd_t rnd) {
  data = a.get_d();
}

template<> template<>
inline void FP_NR<dpe_t>::set_z(const Z_NR<long int>& a, mp_rnd_t rnd) {
  dpe_set_si(data, a.GetData());
}
template<> template<>
inline void FP_NR<dpe_t>::set_z(const Z_NR<double>& a, mp_rnd_t rnd) {
  dpe_set_d(data, a.GetData());
}
template<> template<>
inline void FP_NR<dpe_t>::set_z(const Z_NR<mpz_t>& a, mp_rnd_t rnd) {
  dpe_set_z(data, const_cast<mpz_t&>(a.GetData()));
}

template<> template<>
inline void FP_NR<mpfr_t>::set_z(const Z_NR<long int>& a, mp_rnd_t rnd) {
  mpfr_set_si(data, a.GetData(), rnd);
}
template<> template<>
inline void FP_NR<mpfr_t>::set_z(const Z_NR<double>& a, mp_rnd_t rnd) {
  mpfr_set_d(data, a.GetData(), rnd);
}
template<> template<>
inline void FP_NR<mpfr_t>::set_z(const Z_NR<mpz_t>& a, mp_rnd_t rnd) {
  mpfr_set_z(data, a.GetData(), rnd);
}

#endif
