/* Copyright (C) 2008 Xavier Pujol.

  This file is part of the fplll Library.

  The fplll Library is free software; you can redistribute it and/or modify
  it under the terms of the GNU Lesser General Public License as published by
  the Free Software Foundation; either version 2.1 of the License, or (at your
  option) any later version.
  
  The fplll Library is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
  License for more details.
  
  You should have received a copy of the GNU Lesser General Public License
  along with the fplll Library; see the file COPYING.  If not, write to
  the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA. */


#ifndef LEXER_H
#define LEXER_H

#include <fstream>
#include <string>
#include "util.h"

using namespace std;

enum TokId {
  tokNumber,
  tokLBracket,
  tokRBracket,
  tokEOF
};

class Lexer {
public:
  Lexer() : fileName(NULL), is(&cin), lineNum(0), pos(0), tokRead(false) {
  }
  Lexer(const char* fileName) : fileName(fileName), lineNum(0), pos(0), tokRead(false) {
    is = new ifstream(fileName);
    STD_CHECK(is, "Cannot open '" << fileName << "'");
  }
  ~Lexer() {
    if (fileName) delete is;
  }
  template<class T> Lexer& operator>>(Z_NR<T>& x);
  template<class T> Lexer& operator>>(Vect<T>& v);
  template<class T> Lexer& operator>>(ZZ_mat<T>& m);

private:
  inline static bool isnumchar(char c) {
    return isdigit(c) || c == '-' || c == '+' || c == '.' || c == 'e' || c == 'E';
  }

  void printPos() {
    if (fileName) cerr << fileName << ':';
    cerr << (lineNum > 0 ? lineNum : 1) << ':' << tokStart + 1 << ':';
  }

  /* Reads the type and the position of the next token */
  void next() {
    while (pos >= line.length() || isspace(line[pos])) {
      if (pos < line.length())
        pos++;
      else if (getline(*is, line)) {
        pos = 0;
        lineNum++;
      }
      else {
        tokStart = pos;
        printPos();
        ABORT_MSG("Unexpected end of file");
      }
    }
    char c = line[pos];
    tokStart = pos;
    if (c == '[') {
      tokId = tokLBracket;
      pos++;
    }
    else if (c == ']') {
      tokId = tokRBracket;
      pos++;
    }
    else if (isnumchar(c)) {
      tokId = tokNumber;
      for (; pos < line.length() && isnumchar(line[pos]); pos++) {}
    }
    else {
      printPos();
      ABORT_MSG("Invalid char '" << c << "'");
    }
    tokRead = true;
  }

  void readToken(TokId id, const char* errMessage) {
    if (!tokRead) next();
    if (tokId != id) {
      printPos();
      ABORT_MSG(errMessage);
    }
    tokRead = false;
  }

  template<class T>
  void numberFromString(const char* str, T& number);

  const char* fileName;
  istream* is;
  int lineNum;        // Line number (1-based)
  string line;
  unsigned int pos;   // Position of the lexer in the line (end of the token)
  TokId tokId;        // Type of the last token
  unsigned int tokStart;  // Pointer to the current token (when tokid = tokNum)
  bool tokRead;
};

template<>
inline void Lexer::numberFromString< Z_NR<long int> >(const char* str, Z_NR<long int>& number) {
  char* endptr;
  number.GetData() = strtol(str, &endptr, 10);
  if (*endptr) {
    printPos();
    ABORT_MSG("Cannot convert '" << str << "' to long int");
  }
}

template<>
inline void Lexer::numberFromString< Z_NR<double> >(const char* str, Z_NR<double>& number) {
  char* endptr;
  number.GetData() = strtod(str, &endptr);
  if (*endptr) {
    printPos();
    ABORT_MSG("Cannot convert '" << str << "' to double");
  }
}

template<>
inline void Lexer::numberFromString< Z_NR<mpz_t> >(const char* str, Z_NR<mpz_t>& number) {
  int result = mpz_init_set_str(number.GetData(), str, 10);
  if (result) {
    printPos();
    ABORT_MSG("Cannot convert '" << str << "' to mpz_t");
  }
}

template<class T>
Lexer& Lexer::operator>>(Z_NR<T>& x) {
  char nextChar = 0;
  readToken(tokNumber, "Integer expected");
  if (pos < line.length()) swap(line[pos], nextChar);
  numberFromString(line.c_str() + tokStart, x);
  if (pos < line.length()) swap(line[pos], nextChar);
  return *this;
}

template<class T>
Lexer& Lexer::operator>>(Vect<T>& v) {
  readToken(tokLBracket, "Invalid vector, '[' expected");
  for (int i = 0;; i++) {
    next();
    if (tokId != tokNumber) break;
    v.resize(i + 1);
    *this >> v[i];
  }
  if (v.empty()) {
    printPos();
    ABORT_MSG("Invalid vector, number expected");
  }
  readToken(tokRBracket, "Invalid vector, number or ']' expected");
  return *this;
}

template<class T>
Lexer& Lexer::operator>>(ZZ_mat<T>& m) {
  Vect< Z_NR<T> > v;
  readToken(tokLBracket, "Invalid matrix, '[' expected");
  *this >> v;
  int cols = v.size();
  if (m.GetNumCols() != cols) {
    m.clear();
    m.SetNumCols(cols);
  }
  m.SetNumRows(1);
  for (int j = 0; j < cols; j++) {
    m(0, j) = v[j];
  }
  for (int i = 1;; i++) {
    next();
    if (tokId != tokLBracket) break;
    m.SetNumRows(i + 1);
    readToken(tokLBracket, "Invalid vector, '[' expected");
    for (int j = 0; j < cols; j++) {
      *this >> m(i, j);
    }
    readToken(tokRBracket, "Invalid vector, ']' expected");
  }
  readToken(tokRBracket, "Invalid matrix, '[' or ']' expected");
  return *this;
}

#endif
