/*
 * Copyright (C) 2001, 2002, 2003 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of
 * the License at http://www.redhat.com/licenses/ccmpl.html
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 */

package com.arsdigita.portalserver.ui;


import com.arsdigita.portalserver.util.GlobalizationUtil; 


import com.arsdigita.portalserver.*;

import com.arsdigita.bebop.*;
import com.arsdigita.bebop.event.*;
import com.arsdigita.bebop.form.*;
import com.arsdigita.kernel.*;
import com.arsdigita.kernel.permissions.*;

import com.arsdigita.bebop.list.AbstractListModelBuilder;
import com.arsdigita.bebop.list.ListModel;
import com.arsdigita.bebop.parameters.StringParameter;
import com.arsdigita.bebop.parameters.NotEmptyValidationListener;

import com.arsdigita.kernel.ui.ACSObjectSelectionModel;
import com.arsdigita.toolbox.ui.ACSObjectCollectionListModel;
import com.arsdigita.toolbox.ui.PrivilegedComponentSelector;


import org.apache.log4j.Category;

/**
 *
 *
 * @author Justin Ross (<a href="mailto:jross@redhat.com">jross@redhat.com</a>)
 */
public class ParticipantSearchPane {
    public static final String versionId =
        "$Id: //portalserver/dev/src/com/arsdigita/portalserver/ui/ParticipantSearchPane.java#4 $" +
        "$Author: dennis $" +
        "$DateTime: 2003/08/15 13:46:34 $";

    private static Category s_log = Category.getInstance
        (ParticipantSearchPane.class.getName());

    private ParticipantSearchPane() {
        // No construction.
    }

    public static Component create(final RequestLocal portalsite,
                                   final StringParameter actionParam,
                                   final ActionListener selectSearch) {
        return buildSearchPane(portalsite, actionParam, selectSearch);
    }

    private static
        Component buildSearchPane(final RequestLocal portalsiteRL,
                                  final StringParameter actionParam,
                                  final ActionListener selectSearch) {
        final ACSObjectSelectionModel selectionModel =
            new ACSObjectSelectionModel("selectedparticipant");


        final StringParameter searchString =
            new StringParameter("searchString");

        final List participantList = new List();
        participantList.setSelectionModel(selectionModel);

        final Label emptyView = new Label("");
        participantList.setEmptyView(emptyView);

        participantList.setModelBuilder(new AbstractListModelBuilder() {
                public ListModel makeModel(List l, PageState ps) {
                    String searchStringValue = (String) ps.getValue(searchString);
                    if (searchStringValue == null || searchStringValue.equals("")) {
                        return new ListModel() {
                                public boolean next() { return false; }
                                public Object getElement() { return null; }
                                public String getKey() { return null; }
                            };
                    } else {
                        PortalSite psite = (PortalSite)portalsiteRL.get(ps);
                        PartyCollection pc = psite.getParticipants();
                        pc.filter(searchStringValue);
                        long count = pc.size();
                        if (count < 100 && count > 0) {
                            return new ACSObjectCollectionListModel(pc);
                        } else {
                            if (count == 0) {
                                emptyView.setLabel( (String) GlobalizationUtil.globalize("cw.workspace.ui.no_matches_found").localize() , ps);
                            } else {
                                emptyView.setLabel(
                                   count + " matches found. Please enter more"
                                   + " specific search criteria.", ps);
                            }
                            return new ListModel() {
                                    public boolean next() { return false; }
                                    public Object getElement() { return null; }
                                    public String getKey() { return null; }
                                };
                        }
                    }
                }
            });

        final DynamicListWizard dlw = new DynamicListWizard
            ("Participants", participantList, selectionModel, "",
             new Label("")) {
                public void register(Page p) {
                    super.register(p);
                    p.addComponentStateParam(this, searchString);
                    p.setVisibleDefault(getListLabel(), false);
                    p.setVisibleDefault(getAddLink(), false);
                }
            };

        final Form searchForm = new Form("participantSearch",
                                         new ColumnPanel(1));

        Label searchLabel = new Label(GlobalizationUtil.globalize("cw.workspace.ui.enter_search_criteria"));
        searchLabel.setFontWeight(Label.BOLD);
        searchForm.add(searchLabel);

        final TextField searchStringEntry = new TextField("searchEntry");
        searchStringEntry.addValidationListener(
                                                new NotEmptyValidationListener("No search criteria entered"));
        searchForm.add(searchStringEntry);
        searchForm.add(new Submit("Search"));

        searchForm.addInitListener(new FormInitListener() {
                public void init(FormSectionEvent ev) {
                    PageState ps = ev.getPageState();
                    String searchStringValue = (String)ps.getValue(searchString);
                    if (searchStringValue == null) {
                        return;
                    }
                    searchStringEntry.setValue(ps, searchStringValue);
                }
            });

        searchForm.addSubmissionListener(new FormSubmissionListener() {
                public void submitted(FormSectionEvent ev) {
                    PageState ps = ev.getPageState();
                    ps.reset(participantList);
                    ps.setValue(searchString, null);
                }
            });

        searchForm.addProcessListener(new FormProcessListener() {
                public void process(FormSectionEvent ev) {
                    PageState ps = ev.getPageState();
                    ps.setValue(searchString, searchStringEntry.getValue(ps));
                }
            });

        dlw.setHeader(searchForm);

        RequestLocal participantRL = new RequestLocal() {
                public Object initialValue(PageState ps) {
                    return selectionModel.getSelectedObject(ps);
                }
            };

        Component participantEdit = new ParticipantEdit
            (portalsiteRL,
             participantRL,
             new ActionListener() {
                 public void actionPerformed(ActionEvent e) {
                     dlw.reset(e.getPageState());
                 }
             });

        Component participantDisplay =
            new ParticipantDisplay(portalsiteRL, participantRL);

        final Component participantEditOrDisplay =
            new PrivilegedComponentSelector(PrivilegeDescriptor.ADMIN,
                                            portalsiteRL,
                                            participantEdit,
                                            participantDisplay);
        dlw.setEditPane(participantEditOrDisplay);

        // This change listener ensures that we don't get an edit pane
        // or add pane in some weird intermediate state
        selectionModel.addChangeListener(new ChangeListener() {
                public void stateChanged(ChangeEvent ev) {
                    ev.getPageState().reset(participantEditOrDisplay);
                }
            });

        return dlw;
    }
}
