/*
 * Copyright (C) 2001, 2002, 2003 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of
 * the License at http://www.redhat.com/licenses/ccmpl.html
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 */

package com.arsdigita.portalserver.pslogin;

import com.arsdigita.bebop.Page;
import com.arsdigita.bebop.event.RequestEvent;
import com.arsdigita.bebop.event.RequestListener;
import com.arsdigita.dispatcher.DispatcherHelper;
import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.kernel.Kernel;
import com.arsdigita.kernel.Kernel;
import com.arsdigita.kernel.KernelExcursion;
import com.arsdigita.kernel.Party;
import com.arsdigita.kernel.User;
import com.arsdigita.kernel.security.Initializer;
import com.arsdigita.util.Assert;
import com.arsdigita.kernel.ResourceTypeConfig;
import com.arsdigita.kernel.permissions.PermissionService;
import com.arsdigita.kernel.permissions.PermissionDescriptor;
import com.arsdigita.kernel.permissions.PrivilegeDescriptor;
import com.arsdigita.portalserver.PortalSite;
import com.arsdigita.portalserver.personal.PersonalPortal;

/**
 * This Page redirects a user to his personal portal.
 * It sets up that portal if it has not been initialized
 * @author <a href="mailto:bryanche@arsdigita.com">Bryan Che</a>
 **/
public class PSPage extends Page {

    //constructor
    public PSPage() {
        addRequestListener(new PSPageRequestListener());
        lock();
    }
}

class PSPageRequestListener implements RequestListener {

    public void pageRequested(RequestEvent e) {
        User user;
        String sUrl = "";
        Party party = Kernel.getContext().getParty();
        if (party == null) {
            sUrl = Initializer.getURL(Initializer.LOGIN_PAGE_KEY);
        } else {
            try {
                user = User.retrieve(party.getOID());

                // Get the user's personal portal
                PortalSite psite = PersonalPortal.
                    retrievePersonalPortal(user);
                if (psite == null) {
                    psite = setupPersonalPortal(user);
                }

            sUrl = psite.getPrimaryURL();

            } catch ( DataObjectNotFoundException donfe ) {
                sUrl = Initializer.getURL(Initializer.LOGIN_PAGE_KEY);
            }

        }

        // Redirect to the user's portal (or login)
        try {
            DispatcherHelper.sendRedirect(e.getPageState().getRequest(),
                                          e.getPageState().getResponse(),
                                          sUrl);
        } catch (java.io.IOException ioe) {
            //this method can't throw an exception in this Interface, so ignore
        }
    }

    //borrow the code for setting up a personal portal from
    //com.arsdigita.portalserver.personal.PersonalPortalCreatorDispatcher
    private PortalSite setupPersonalPortal(final User u) {
        final PortalSite[] m_psite = new PortalSite[] { null };

        KernelExcursion rootExcursion = new KernelExcursion() {
                public void excurse() {
                    setParty(Kernel.getSystemParty());

                    PersonalPortal psite = PersonalPortal.
                        createPersonalPortal(u);
                    Assert.assertNotNull(psite, "workspace");
                    ResourceTypeConfig config = psite.
                        getApplicationType().getConfig();
                    Assert.assertNotNull(config, "config");

                    config.configureResource(psite);
                    psite.save();
                    PermissionService.grantPermission
                        (new PermissionDescriptor
                         (PrivilegeDescriptor.ADMIN, psite, u));

                    m_psite[0] = psite;
                }};

        rootExcursion.run();

        return m_psite[0];

    }
}
