/*
 * Copyright (C) 2001, 2002, 2003 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of
 * the License at http://www.redhat.com/licenses/ccmpl.html
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 */

package com.arsdigita.portalserver;

import com.arsdigita.domain.DomainObjectFactory;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.DataQuery;
import com.arsdigita.persistence.DataCollection;
import com.arsdigita.persistence.SessionManager;
import com.arsdigita.persistence.Session;
import com.arsdigita.persistence.OID;
import com.arsdigita.persistence.DataQueryDataCollectionAdapter;
import com.arsdigita.portal.Portal;
import com.arsdigita.portal.PortletCollection;
import com.arsdigita.kernel.Resource;
import com.arsdigita.kernel.permissions.PermissionService;
import com.arsdigita.util.Assert;
import java.math.BigDecimal;
import java.util.ArrayList;

    /**
     * A <code>PortalTab</code> is a tool organizing content in a Portal.
     * A PortalTab includes logic for sorting and displaying portlets
     * in a specific layout manner.
     *
     * This Class extends PortalTab and is aware of a another PortalTab
     * on the system that it relies on for part of its Portlets. It also
     * overrides the getPortlets() method, and the PortletCollection
     * returned is a union of its native Portlets plus the Portlets
     * that reside in locked tiles in the associated PortalTab.
     * 
     * @author Jim Parsons
     */ 

public class SubPortalTab extends PortalTab {

    private static final int SORT_KEY_JUMP = 10;

    /**
     * The type of the {@link com.arsdigita.persistence.DataObject}
     * that stands behind this {@link
     * com.arsdigita.domain.DomainObject}.
     */
    public static final String BASE_DATA_OBJECT_TYPE =
        "com.arsdigita.workspace.SubWorkspaceTab";

    protected String getBaseDataObjectType() {
        return BASE_DATA_OBJECT_TYPE;
    }

    public SubPortalTab(DataObject dataObject) {
        super(dataObject);
    }


   public static SubPortalTab createSubTab(String title, Resource parent) {
        SubPortalTab ptab =
           (SubPortalTab)Portal.create(BASE_DATA_OBJECT_TYPE, title, parent);
        ptab.setSortKey(0);
        return ptab;
   }

   public static SubPortalTab createSubTab(String title) {
        SubPortalTab ptab =
           (SubPortalTab)Portal.create(BASE_DATA_OBJECT_TYPE, title, null);
        ptab.setSortKey(0);
        return ptab;
   }
  
   public PortletCollection getPortlets() {
       PortalTab ptab = getSuperPortalTab();
       //If the super portal tab that this subportal tab is associated with
       //has been deleted, unlock the cells in this tab,
       //and then just call super.getPortlets();
       if(ptab == null) {
         String thislayout = this.getLayout();
         this.setLayout(thislayout.toUpperCase());
         return super.getPortlets();
       }

       String layout = ptab.getLayout();

       //This section builds up a list of cell numbers that must be
       //rendered from the superportal tab
       ArrayList l = new ArrayList();
       char c;
       boolean found_one = false;
       for(int i=0;i < layout.length();i++) {
           c = layout.charAt(i);
           if((c == 'n') || (c == 'w')) {
             l.add(new Integer(i));
             found_one = true;
           }
       }

       if(found_one) {    //At least one cell region was found to be locked
         Session session = SessionManager.getSession();
         DataQuery query = 
          session.retrieveQuery("com.arsdigita.workspace.getSubPortalPortlets");
         query.setParameter("subPortalTabID", this.getID());
         query.setParameter("cells", l);
         DataCollection dc = 
           new DataQueryDataCollectionAdapter(query,"portlet");
         PortletCollection pc = new PortletCollection(dc);
         return pc; 
       } else { //this case means the super tab still exists, but no longer
                //has locked regions.
         return super.getPortlets();
       }
   }   


    public void setSuperPortalTab(PortalTab ptab) {
        setAssociation("workspaceTab", ptab);
    }

    public PortalTab getSuperPortalTab() {
        DataObject dataObject = (DataObject) get("workspaceTab");

        Assert.assertNotNull(dataObject, "dataObject");

        return new PortalTab(dataObject);
    }

}
