/*
 * Copyright (C) 2001, 2002, 2003 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of
 * the License at http://www.redhat.com/licenses/ccmpl.html
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 */

package com.arsdigita.glossary.ui;


import com.arsdigita.glossary.util.GlobalizationUtil; 

import com.arsdigita.glossary.Glossary;
import com.arsdigita.glossary.GlossaryWord;
import com.arsdigita.glossary.WordCollection;

import com.arsdigita.bebop.ColumnPanel;
import com.arsdigita.bebop.Form;
import com.arsdigita.bebop.FormData;
import com.arsdigita.bebop.FormProcessException;
import com.arsdigita.bebop.Label;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.RequestLocal;
import com.arsdigita.bebop.SingleSelectionModel;
import com.arsdigita.bebop.event.FormInitListener;
import com.arsdigita.bebop.event.FormProcessListener;
import com.arsdigita.bebop.event.FormSectionEvent;
import com.arsdigita.bebop.event.FormValidationListener;
import com.arsdigita.bebop.form.Submit;
import com.arsdigita.bebop.form.TextArea;
import com.arsdigita.bebop.form.TextField;
import com.arsdigita.bebop.parameters.NotEmptyValidationListener;
import com.arsdigita.bebop.parameters.ParameterModel;
import com.arsdigita.bebop.parameters.StringLengthValidationListener;
import com.arsdigita.bebop.parameters.StringParameter;
import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.toolbox.ui.TextTypeWidget;
import com.arsdigita.util.TypedText;

import java.math.BigDecimal;
import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;

/**
 * A Bebop Form to add/edit a word.
 *
 * @author <a href="mailto:gautam.chowdhary@devlogics.com">Gautam Chowdhary</a>
 * @version $Revision: #3 $ $Date: 2003/08/15 $
 * @since ACS 4.6.5
 *
 */


public class GlossaryWordEntryForm extends Form
    implements FormInitListener, FormValidationListener, FormProcessListener {

    private static final Logger s_log =
        Logger.getLogger(GlossaryWordEntryForm.class);

    private RequestLocal m_word;

    public GlossaryWordEntryForm(final SingleSelectionModel selection) {

        super("GlossaryWordEntry");
        setMethod(POST);

        m_word = new RequestLocal() {
                protected Object initialValue(PageState s) {
                    try {
                        if (selection.isSelected(s)) {
                            return new GlossaryWord(
                                                    new BigDecimal(
                                                                   (String)selection.getSelectedKey(s)));
                        } else {
                            return null;
                        }
                    } catch (DataObjectNotFoundException e) {
                        return null;
                    }
                }
            };


        /* Word field */

        add(new Label(GlobalizationUtil.globalize("cw.glossary.ui.word")));
        ParameterModel word = new StringParameter("word");
        word.addParameterListener(new NotEmptyValidationListener());
        TextField txt_word = new TextField(word);
        txt_word.setMaxLength(40);
        add(txt_word);

        /* Description field */
        add(new Label(GlobalizationUtil.globalize("cw.glossary.ui.definition")));
        ParameterModel defn = new StringParameter("definition");
        defn.addParameterListener(new NotEmptyValidationListener());
        defn.addParameterListener(new StringLengthValidationListener(4000));
        add(new TextArea(defn, 4, 40, TextArea.SOFT));

        /* Text type field */
        add(new Label(GlobalizationUtil.globalize("cw.glossary.ui.text_type")));
        ParameterModel textTypeParameter =
            new StringParameter("definitionTextType");
        add(new TextTypeWidget(textTypeParameter));


        add(new Submit("Save"), ColumnPanel.CENTER | ColumnPanel.FULL_WIDTH);

        addInitListener(this);
        addProcessListener(this);
        addValidationListener(this);
    }


    public void init(FormSectionEvent e) throws FormProcessException {
        FormData data = e.getFormData();
        PageState s = e.getPageState();
        HttpServletRequest req = s.getRequest();

        GlossaryWord gw = (GlossaryWord) m_word.get(s);

        if (gw !=null) {
            data.put("word", gw.getWord());
            data.put("definition", gw.getDefinition().getText());
            data.put("definitionTextType", gw.getDefinition().getType());
        }

    }

    public void validate(FormSectionEvent e) throws FormProcessException {

        FormData data = e.getFormData();
        PageState s = e.getPageState();
        HttpServletRequest req = s.getRequest();

        String word = (String)data.get("word");
        word = word.toUpperCase();

        GlossaryWord gw = (GlossaryWord) m_word.get(s);
        Glossary glossary = GlossaryHelper.getGlossary(req);

        WordCollection words = glossary.getWordList();
        words.duplicates(word, gw);

        if (words.next()) {
            data.addError("word","This word is already in the glossary");
        }
    }


    public void process(FormSectionEvent e) throws FormProcessException {

        FormData data = e.getFormData();
        PageState s = e.getPageState();
        HttpServletRequest req = s.getRequest();

        GlossaryWord glossaryWord = (GlossaryWord) m_word.get(s);

        String word = (String) data.get("word");
        String definition = (String) data.get("definition");
        String definitionTextType =
            (String) data.get("definitionTextType");

        if ( glossaryWord != null ) {
            // We are editing.
            glossaryWord.setWord(word);
            glossaryWord.setDefinition(new TypedText(definition,
                                                     definitionTextType));
            glossaryWord.save();
        } else {
            // We are adding.
            Glossary glossary = GlossaryHelper.getGlossary(req);
            glossaryWord = glossary.createWord(
                                               word, new TypedText(definition, definitionTextType));
            glossaryWord.save();
        }
        fireCompletionEvent(e.getPageState());
    }
}
