/*
 * Copyright (C) 2001, 2002, 2003 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of
 * the License at http://www.redhat.com/licenses/ccmpl.html
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 */

package com.arsdigita.glossary;

import com.arsdigita.domain.DomainCollection;
import com.arsdigita.domain.DomainObject;
import com.arsdigita.persistence.DataCollection;
import com.arsdigita.persistence.Filter;
import com.arsdigita.persistence.FilterFactory;

/**
 * @author Kevin Scaldeferri (kevin@arsdigita.com)
 * @version $Revision: #2 $
 */
public class WordCollection extends DomainCollection {

    WordCollection(DataCollection dataCollection) {
        super(dataCollection);
    }

    /**
     * Filters the collection to only return GlossaryWords which
     * start with the specified string.  If the string is null,
     * then all words are included in the collection
     */
    public void startsWith(String start) {
        if (start != null ) {
            Filter f = m_dataCollection.getFilterFactory()
                .compare("upper(word)",
                         FilterFactory.STARTS_WITH,
                         ":word");
            f.set("word", start.toUpperCase());
            m_dataCollection.addFilter(f);
        }
    }

    /**
     * Filters the collection to return only exact matches to the
     * specified word
     */
    public void exactMatch(String word) {
        Filter f = m_dataCollection.addFilter("upper(word) = :word");
        f.set("word", word.toUpperCase());
    }

    /**
     * Filters the collection to return exact matches to the
     * specified String which are not the specified GlossaryWord
     */
    public void duplicates(String word, GlossaryWord gw) {
        exactMatch(word);
        if (gw != null) {
            m_dataCollection.addNotEqualsFilter("id", gw.getID());
        }
    }
    /**
     * Filters the collection to search for the specified string
     * in either the word or the definition.  A null search
     * string will return no results.  (This usually indicates
     * an error condition.)
     */
    public void contains(String search) {
        FilterFactory factory = m_dataCollection.getFilterFactory();
        m_dataCollection.addFilter(
                                   factory.or()
                                   .addFilter(factory.contains("word", search, false))
                                   .addFilter(factory.contains("definition", search, false)));
    }

    public DomainObject getDomainObject() {
        return getWord();
    }

    /**
     * Wrapper to getDomainObject which casts to a GlossaryWord
     */
    public GlossaryWord getWord() {
        return new GlossaryWord(m_dataCollection.getDataObject());
    }
}
