/*
 * Copyright (C) 2001, 2002, 2003 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of
 * the License at http://www.redhat.com/licenses/ccmpl.html
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 */

package com.arsdigita.glossary;

import com.arsdigita.kernel.ACSObject;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.OID;
import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.util.TypedText;

import java.math.BigDecimal;

/**
 * GlossaryWord represent the word / definition pairs in a Glossary
 *
 * @author Kevin Scaldeferri (kevin@arsdigita.com)
 * @version $Revision: #3 $ $Date: 2003/08/15 $
 * @since ACS 4.6.5
 **/

public class GlossaryWord extends ACSObject {

    private static final org.apache.log4j.Logger log =
        org.apache.log4j.Logger.getLogger(GlossaryWord.class);

    private static final String BASE_DATA_OBJECT_TYPE =
        "com.arsdigita.glossary.GlossaryWord";

    private Glossary m_glossary = null;

    /**
     * Default constructor. The contained <code>DataObject</code> is
     * initialized with a new <code>DataObject</code> with an
     * <code>ObjectType</code> of "GlossaryWord".
     *
     * @see com.arsdigita.persistence.DataObject
     * @see com.arsdigita.persistence.metadata.ObjectType
     *
     *  This is package-accessible on purpose.
     * @see com.arsdigita.glossary.createWord.
     *
     */

    GlossaryWord() {
        this(BASE_DATA_OBJECT_TYPE);
    }

    GlossaryWord(String word, TypedText definition) {
        this();
        setWord(word);
        setDefinition(definition);
    }

    /**
     * Constructor. The contained <code>DataObject</code> is
     * initialized with a new <code>DataObject</code> with an
     * <code>ObjectType</code> specified by the string
     * <i>typeName</i>.
     *
     * @param typeName The name of the <code>ObjectType</code> of the
     * contained <code>DataObject</code>.
     *
     * @see com.arsdigita.persistence.DataObject
     * @see com.arsdigita.persistence.metadata.ObjectType
     */
    public GlossaryWord(String typeName) {
        super(typeName);
    }

    /**
     * Constructor. Retrieves a glossary instance, retrieving an existing
     * glossary from the database with OID oid. Throws an exception if an
     * object with OID oid does not exist or the object is not of type Glossary
     *
     * @param oid The <code>OID</code> for the retrieved
     * <code>DataObject</code>.
     *
     * @see com.arsdigita.persistence.DataObject
     * @see com.arsdigita.persistence.OID
     *
     * @exception DataObjectNotFoundException Thrown if we cannot
     * retrieve a data object for the specified OID
     *
     **/
    public GlossaryWord(OID oid) throws DataObjectNotFoundException {
        super(oid);
    }


    /**
     * Constructor. Retrieves a GlossaryWord instance, retrieving an existing
     * glossary from the database with an ID responding to key.
     *
     * @param key The ID for the object to be retrieved.
     * object.
     *
     * @exception DataObjectNotFoundException Thrown if we cannot
     * retrieve a data object for the specified OID
     *
     **/
    public GlossaryWord(BigDecimal key)  throws DataObjectNotFoundException {
        this(new OID(BASE_DATA_OBJECT_TYPE, key));
    }

    public GlossaryWord(DataObject dataObject) {
        super(dataObject);
    }

    /**
     * Returns the appropriate object type for a GlossaryWord so that proper
     * type validation will occur when retrieving Glossary's by  OID
     *
     * @return The fully qualitifed name of of the base data object
     * type for the GlossaryWord object type.
     */
    protected String getBaseDataObjectType() {
        return BASE_DATA_OBJECT_TYPE;
    }

    public Glossary getGlossary() {
        if (m_glossary == null) {
            DataObject glossaryData = (DataObject) get("glossary");
            if (glossaryData != null) {
                m_glossary = new Glossary(glossaryData);
            }
        }
        return m_glossary;
    }

    void setGlossary(Glossary glossary) {
        m_glossary = glossary;
        setAssociation("glossary", glossary);
    }

    public String getWord() {
        return (String)get("word");
    }

    public void setWord(String word) {
        set("word", word);
    }

    public TypedText getDefinition() {
        return new TypedText((String)get("definition"),
                             (String)get("definitionFormat"));
    }

    public void setDefinition(TypedText definition) {
        set("definition", definition.getText());
        set("definitionFormat", definition.getType());
    }

    /**
     * returns the word.  Wrapper to getWord().
     */
    public String toString() {
        return getWord();
    }
}
