/*
 * Copyright (C) 2001, 2002, 2003 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of
 * the License at http://www.redhat.com/licenses/ccmpl.html
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 */

package com.arsdigita.forum.ui;

import com.arsdigita.forum.Forum;
import com.arsdigita.forum.ForumContext;
import com.arsdigita.forum.Post;
import com.arsdigita.forum.ThreadSubscription;
import com.arsdigita.forum.ui.admin.RejectionForm;

import com.arsdigita.bebop.Component;
import com.arsdigita.bebop.Container;
import com.arsdigita.bebop.ModalContainer;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.Form;
import com.arsdigita.bebop.Link;
import com.arsdigita.bebop.Label;
import com.arsdigita.bebop.ActionLink;
import com.arsdigita.bebop.SimpleContainer;

import com.arsdigita.bebop.event.ActionEvent;
import com.arsdigita.bebop.event.ActionListener;
import com.arsdigita.bebop.event.FormCancelListener;
import com.arsdigita.bebop.event.FormSectionEvent;
import com.arsdigita.bebop.event.TableActionEvent;
import com.arsdigita.bebop.event.TableActionListener;
import com.arsdigita.bebop.table.TableColumn;

import com.arsdigita.kernel.Kernel;
import com.arsdigita.kernel.Party;
import com.arsdigita.kernel.ui.ACSObjectSelectionModel;

import com.arsdigita.messaging.MessageThread;
import com.arsdigita.messaging.ThreadedMessage;
import com.arsdigita.util.Assert;
import com.arsdigita.web.RedirectSignal;
import com.arsdigita.web.URL;
import com.arsdigita.xml.Element;

import org.apache.log4j.Logger;

/**
 * Display for looking at a single thread and posting new messages and
 * replying to it. Contains MessagesComponent, EditPostForm, and
 * ReplyToPostForm
 * @see com.arsdigita.forum.ui.MessagesComponent
 * @see com.arsdigita.forum.ui.EditPostForm
 * @see com.arsdigita.forum.ui.ReplyToPostForm
 */

public class ThreadComponent extends ModalContainer implements Constants {

    // References to sub-components for event access.

    private Container m_threadView;
    private Form m_editForm;
    private Form m_replyForm;
    private Form m_rejectForm;

    private static final Logger s_log 
        = Logger.getLogger(ThreadComponent.class);

    private ACSObjectSelectionModel m_postModel;

    /**
     * The forum page contains three components: a MessageList,
     * a ThreadEdit component, and a ThreadReply component.
     * Creates the component.
     */
    public ThreadComponent() {
        // Create a modal container: shows only one containee at a time.
        m_postModel = new ACSObjectSelectionModel("post");

        initComponents();
    }

    private void initComponents() {
        // Add the thread components to the modal container and maintain
        // references for event manipulation purposes.
        m_editForm = new EditPostForm(m_postModel);
        m_replyForm = new ReplyToPostForm(m_postModel);
        m_rejectForm = new RejectionForm(m_postModel);
        addForm(m_editForm);
        addForm(m_replyForm);
        addForm(m_rejectForm);

        m_threadView = new SimpleContainer();        
        Container linksPanel = new SimpleContainer("forum:threadOptions", 
                                                   Constants.FORUM_XML_NS);

        // Offer links to return to index or control alerts.
        Link returnLink = new Link(new Label(Text.gz("forum.ui.threads.viewAll")), 
                                   "index.jsp");
        returnLink.setClassAttr("actionLink");
        linksPanel.add(returnLink);

        SimpleContainer subLinks = new SimpleContainer();
        subLinks.add(createThreadSubscribeLink());
        subLinks.add(createThreadUnsubscribeLink());
        linksPanel.add(subLinks);

        // Add the panel to the view.
        m_threadView.add(linksPanel);
        m_threadView.add(new ThreadDisplay(m_postModel, this));
        add(m_threadView);

        // The thread view is the default component.
        setDefaultComponent(m_threadView);
    }

    public void makeListViewVisible(PageState state) {
        setVisibleComponent(state, m_threadView);
    }

    public void makeEditFormVisible(PageState state) {
        setVisibleComponent(state, m_editForm);
    }

    public void makeReplyFormVisible(PageState state) {
        s_log.debug("making reply form invisible");
        setVisibleComponent(state, m_replyForm);
    }

    public void makeRejectFormVisible(PageState state) {
        setVisibleComponent(state, m_rejectForm);
    }
    
    /**
     * Creates the component for viewing a thread.
     */
    
    private final void addForm(Form form) {
        add(form);
        form.addCompletionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    s_log.debug("FORM ACTION COMPLETED");
                    makeListViewVisible(e.getPageState());
                }
                });

        form.addCancelListener(new FormCancelListener() {
                public void cancel(FormSectionEvent e) {
                    s_log.debug("fire cancel listener");
                    PageState ps = e.getPageState();
                    makeListViewVisible(e.getPageState());
                }
            });
    }



    private Component createThreadSubscribeLink() {
        ActionLink subscribeLink = new ActionLink(
            new Label(Text.gz("forum.ui.thread.subscribe"))) {
                public boolean isVisible(PageState s) {
                    Party party = Kernel.getContext().getParty();

                    if ( party != null ) {
                        ThreadSubscription sub = ThreadSubscription
                            .getThreadSubscription(getCurrentMessage(s));
                        return !sub.isSubscribed(party);
                    } else {
                        return false;
                    }
                }
            };
        subscribeLink.setClassAttr("actionLink");

        subscribeLink.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    PageState s = e.getPageState();
                    Party party = Kernel.getContext().getParty();

                    ThreadSubscription sub = ThreadSubscription
                        .getThreadSubscription(getCurrentMessage(s));
                    if (!sub.isSubscribed(party)) {
                        sub.subscribe(party);
                    }
                    sub.save();
                }
            });
        return subscribeLink;
    }

    private Component createThreadUnsubscribeLink() {
        ActionLink unsubscribeLink = new ActionLink(
            new Label(Text.gz("forum.ui.thread.unsubscribe"))) {
                public boolean isVisible(PageState s) {
                    Party party = Kernel.getContext().getParty();

                    if ( party != null ) {
                        ThreadSubscription sub = ThreadSubscription
                            .getThreadSubscription(getCurrentMessage(s));
                        return sub.isSubscribed(party);
                    } else {
                        return false;
                    }
                }
            };
        unsubscribeLink.setClassAttr("actionLink");

        unsubscribeLink.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    PageState s = e.getPageState();
                    Party party = Kernel.getContext().getParty();


                    ThreadSubscription sub = ThreadSubscription.
                        getThreadSubscription(getCurrentMessage(s));

                    if (sub.isSubscribed(party)) {
                        sub.unsubscribe(party);
                    }
                    sub.save();
                }
            });
        return unsubscribeLink;
    }

    private final ACSObjectSelectionModel getPostModel() {
        return m_postModel;
    }

    private MessageThread getCurrentMessage(PageState state) {
        return ForumContext.getContext(state).getMessageThread();
    }
}
