/*
 * Copyright (C) 2001, 2002, 2003 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of
 * the License at http://www.redhat.com/licenses/ccmpl.html
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 */

package com.arsdigita.forum;

import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.domain.DomainObjectFactory;
import com.arsdigita.kernel.permissions.PermissionService;
import com.arsdigita.kernel.permissions.PrivilegeDescriptor;
import com.arsdigita.messaging.ThreadedMessage;
import com.arsdigita.persistence.DataCollection;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.OID;
import com.arsdigita.persistence.SessionManager;
import com.arsdigita.util.Assert;

import java.math.BigDecimal;
import org.apache.log4j.Logger;

/**
 * The ForumSubscription class provides notification capabilities on a Forum.
 *
 * Note that ForumSubscriptions should not be retrieved directly using the
 * contructors, but rather by calling DomainObjectFactory.newInstance().
 *
 * @author Kevin Scaldeferri (kevin@arsdigita.com)
 *
 * @version $Revision: #7 $ $Author: dan $ $DateTime: 2003/09/15 13:58:12 $
 */

public class ForumSubscription extends Subscription {

    private static final Logger s_log =
        Logger.getLogger(ForumSubscription.class);

    public static final String BASE_DATA_OBJECT_TYPE =
        "com.arsdigita.forum.ForumSubscription";
    
    public static final String IS_MODERATION_ALERT =
        "isModerationAlert";
    public static final String FORUM = "forum";

    private Forum m_forum;

    public ForumSubscription(Forum forum) {
        this(BASE_DATA_OBJECT_TYPE, forum);
        setIsModerationAlert(false);
    }

    public ForumSubscription(String typeName, Forum forum) {
        super(typeName);

        Assert.exists(typeName, String.class);
        Assert.exists(forum, Forum.class);

        setForum(forum);
    }

    protected ForumSubscription(OID oid) throws DataObjectNotFoundException {
        super(oid);
    }

    protected ForumSubscription(BigDecimal id)
        throws DataObjectNotFoundException {

        super(new OID(BASE_DATA_OBJECT_TYPE, id));
    }

    protected ForumSubscription(DataObject data) {
        super(data);
    }

    protected String getBaseDataObjectType() {
        return BASE_DATA_OBJECT_TYPE;
    }

    public Forum getForum() {
        if (m_forum == null) {
            DataObject forumData = (DataObject) get(FORUM);
            if (forumData != null) {
                m_forum = (Forum)DomainObjectFactory.newInstance(forumData);
            }
        }
        return m_forum;
    }

    public void doWriteCheck() {
        getForum().assertPrivilege(PrivilegeDescriptor.READ);
    }

    private void setForum(Forum forum) {
        m_forum = forum;
        setAssociation(FORUM, m_forum);
    }

    protected void afterSave() {
        PermissionService.setContext(this, getForum());
        super.afterSave();
    }

    /**
     * Returns a signature with information about replying to the
     * message.
     */
    public String getSignature(ThreadedMessage post) {
        StringBuffer sb = new StringBuffer();

        // XXX
        /*
        sb.append(SEPARATOR);
        sb.append(ALERT_BLURB);
        sb.append("You are receiving this email because you subscribed to ");
        sb.append("alerts on this forum.\n\n");
        sb.append(REPLY_BLURB);
        */
        sb.append(getReturnURLMessage((Post)post));

        return sb.toString();
    }

    /**
     * gets the ForumSubscription associated with the provided Forum
     */
    public static ForumSubscription getFromForum(Forum forum) {
        DataCollection subs = SessionManager.getSession()
            .retrieve(ForumSubscription.BASE_DATA_OBJECT_TYPE);
        subs.addEqualsFilter("forum.id", forum.getID());
        subs.addEqualsFilter("digest.id", null);

        if (subs.next()) {
            ForumSubscription sub =
                new ForumSubscription(subs.getDataObject());
            subs.close();
            return sub;
        }
        return null;
    }

    public void setIsModerationAlert(boolean isModerationAlert) {
        set(IS_MODERATION_ALERT, new Boolean(isModerationAlert));
    }

    public void setIsModerationAlert(Boolean isModerationAlert) {
        set(IS_MODERATION_ALERT, isModerationAlert);
    }
}
