/*
 * Copyright (C) 2001, 2002, 2003 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of
 * the License at http://www.redhat.com/licenses/ccmpl.html
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 */

package com.arsdigita.forum.ui;


import com.arsdigita.bebop.ColumnPanel;
import com.arsdigita.bebop.Component;
import com.arsdigita.bebop.Form;
import com.arsdigita.bebop.FormData;
import com.arsdigita.bebop.FormProcessException;
import com.arsdigita.bebop.GridPanel;
import com.arsdigita.bebop.Label;
import com.arsdigita.bebop.Link;
import com.arsdigita.bebop.MetaForm;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.SegmentedPanel;
import com.arsdigita.bebop.SimpleContainer;
import com.arsdigita.bebop.event.FormInitListener;
import com.arsdigita.bebop.event.FormProcessListener;
import com.arsdigita.bebop.event.FormSectionEvent;
import com.arsdigita.bebop.form.CheckboxGroup;
import com.arsdigita.bebop.form.Option;
import com.arsdigita.bebop.form.RadioGroup;
import com.arsdigita.bebop.form.Submit;
import com.arsdigita.bebop.parameters.ArrayParameter;
import com.arsdigita.bebop.parameters.BigDecimalParameter;
import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.domain.DomainCollection;
import com.arsdigita.forum.DailySubscription;
import com.arsdigita.forum.Forum;
import com.arsdigita.forum.ForumContext;
import com.arsdigita.forum.ForumSubscription;
import com.arsdigita.forum.ThreadSubscription;
import com.arsdigita.kernel.Kernel;
import com.arsdigita.kernel.Party;
import com.arsdigita.messaging.MessageThread;
import com.arsdigita.util.UncheckedWrapperException;
import com.arsdigita.xml.Element;

import java.io.IOException;
import java.math.BigDecimal;

import org.apache.log4j.Logger;

class ForumAlertsView extends SimpleContainer implements Constants {
    private static final Logger s_log = Logger.getLogger
        (ForumAlertsView.class);

    ForumAlertsView() {
        add(forumAlertsSegment());
        add(threadAlertsSegment());
    }

    private Component forumAlertsSegment() {
        SimpleContainer seg = new SimpleContainer("forum:forumAlerts", 
                                                  FORUM_XML_NS);
        seg.add(forumAlertsForm());
        return seg;
    }

    private Component threadAlertsSegment() {
        SimpleContainer seg = new SimpleContainer("forum:threadAlerts", 
                                                  FORUM_XML_NS);
        seg.add(threadAlertsForm());
        return seg;
    }

    private Component forumAlertsForm() {
        Form alertsForm = new Form("instantAlerts", new ColumnPanel(2));

        final RadioGroup instant = new RadioGroup("instant");
        instant.addOption(new Option("Yes"));
        instant.addOption(new Option("No"));
        alertsForm.add(new Label(Text.gz("forum.ui.receive_instant_alerts")));
        alertsForm.add(instant);

        final RadioGroup daily = new RadioGroup("daily");
        daily.addOption(new Option("Yes"));
        daily.addOption(new Option("No"));
        alertsForm.add(new Label(Text.gz("forum.ui.receive_daily_summary")));
        alertsForm.add(daily);

        alertsForm.add(new Label(""));
        alertsForm.add(new Submit("Save"));

        alertsForm.addInitListener(new FormInitListener() {
                public void init(FormSectionEvent e) {
                    FormData data = e.getFormData();
                    PageState s = e.getPageState();
                    Party party = Kernel.getContext().getParty();
                    Forum forum = ForumContext.getContext(s).getForum();

                    ForumSubscription fSub =
                        ForumSubscription.getFromForum(forum);
                    if (fSub.isSubscribed(party)) {
                        instant.setValue(s,"Yes");
                    } else {
                        instant.setValue(s, "No");
                    }

                    DailySubscription dSub = (DailySubscription)
                        DailySubscription.getFromForum(forum);
                    if (dSub.isSubscribed(party)) {
                        daily.setValue(s,"Yes");
                    } else {
                        daily.setValue(s, "No");
                    }
                }
            });

        alertsForm.addProcessListener(new FormProcessListener() {
                public void process(FormSectionEvent e)
                    throws FormProcessException {

                    FormData data = e.getFormData();
                    PageState s = e.getPageState();
                    Party party = Kernel.getContext().getParty();
                    Forum forum = ForumContext.getContext(s).getForum();

                    ForumSubscription fSub =
                        ForumSubscription.getFromForum(forum);
                    DailySubscription dSub = (DailySubscription)
                        DailySubscription.getFromForum(forum);

                    if (data.get("instant").equals("Yes")) {
                        fSub.subscribe(party);
                    } else if (data.get("instant").equals("No")) {
                        fSub.unsubscribe(party);
                    } else {
                        throw new FormProcessException(
                            "Received bad option for instant: "
                            + data.get("instant"));
                    }
                    fSub.save();

                    if (data.get("daily").equals("Yes")) {
                        dSub.subscribe(party);
                    } else if (data.get("daily").equals("No")) {
                        dSub.unsubscribe(party);
                    } else {
                        throw new FormProcessException(
                            "Received bad option for daily: "
                            + data.get("daily"));
                    }
                    dSub.save();
                }
            });
        return alertsForm;
    }

    private Component threadAlertsForm() {
        Form form = new Form("dailyAlerts");
        form.setRedirecting(true);
        form.add(new ThreadAlertsList() {
                public Element generateAlertXML(ThreadSubscription sub) {
                    Element subEl = super.generateAlertXML(sub);
                    
                    subEl.addAttribute("param", "delete");
                    return subEl;
                }
            });
        
        CheckboxGroup boxes = new CheckboxGroup(
            new ArrayParameter(new BigDecimalParameter("delete")));
        form.add(boxes);
        
        form.add(new Submit(Text.gz("forum.ui.delete")),
                 FULL_WIDTH | RIGHT);
        
        form.addProcessListener(new DeleteProcesser());
        
        return form;
    }

    class DeleteProcesser implements FormProcessListener {
        public void process(FormSectionEvent e) {
            FormData data = e.getFormData();
            PageState s = e.getPageState();
            Party party = Kernel.getContext().getParty();
            
            BigDecimal[] deletes = (BigDecimal[])
                data.get("delete");
            
            if (deletes != null) {
                for (int i = 0; i < deletes.length ; i++) {
                    try {
                        ThreadSubscription tSub =
                            new ThreadSubscription(deletes[i]);
                        tSub.unsubscribe(party);
                        tSub.save();
                    } catch (DataObjectNotFoundException x) {
                        throw new UncheckedWrapperException(x);
                    }
                }
            }
        }
    }
}
