/*
 * Copyright (C) 2001, 2002, 2003 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of
 * the License at http://www.redhat.com/licenses/ccmpl.html
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 */

package com.arsdigita.forum.ui;

import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.Label;
import com.arsdigita.bebop.event.PrintEvent;
import com.arsdigita.bebop.event.PrintListener;
import com.arsdigita.bebop.form.Option;
import com.arsdigita.bebop.form.SingleSelect;
import com.arsdigita.bebop.parameters.ParameterModel;
import com.arsdigita.categorization.Category;
import com.arsdigita.categorization.CategoryCollection;
import com.arsdigita.forum.Forum;
import com.arsdigita.forum.ForumContext;
import com.arsdigita.forum.Post;
import com.arsdigita.util.UncheckedWrapperException;

import java.util.TooManyListenersException;

/**
 * A widget providing a choice of categories for a forum.
 *
 * @author ron@arsdigita.com
 * @author sarah@arsdigita.com
 *
 * @version $Id: //apps/forum/dev/src/com/arsdigita/forum/ui/CategoryWidget.java#11 $
 */

public class CategoryWidget extends SingleSelect implements Constants {

    public CategoryWidget(ParameterModel categoryParameter) {
        super(categoryParameter);
        addOption(new Option(TOPIC_NONE.toString(), 
                             new Label(Text.gz("forum.ui.topic.none"))));

        try {
            addPrintListener(new PrintListener() {
                    public void prepare(PrintEvent e) {
                        PageState s = e.getPageState();
                        final Forum forum = getForum(s);
                        // Get categories for this forum
                        final Category root = forum.getRootCategory();
                        if (root != null) {
                            SingleSelect target = (SingleSelect) e.getTarget();
                            addCategories(root, target);
                        }
                    }
                });
        } catch (TooManyListenersException ex) {
            throw new UncheckedWrapperException(ex);
        }
    }

    /**
     * Adds categories to the SingleSelect widget
     *
     * @param root Root of the category tree
     * @param target SingleSelect element for category names
     */
    private void addCategories(final Category root, final SingleSelect target) {
        CategoryCollection children = root.getChildren();
        try {
            while (children.next()) {
                Category c = children.getCategory();
                target.addOption(new Option
                                 (c.getID().toString(),
                                  c.getName()));

            }
        } finally {
            children.close();
        }
    }

    /**
     * Gets forum from page state
     *
     * @param s The page state
     * @return Forum
     */
    private Forum getForum(final PageState s) {
        Forum forum;
        if (ForumContext.getContext(s).getThreadID() != null ) {
            Post rootPost = (Post)ForumContext.getContext(s).
                getMessageThread().getRootMessage();
            forum = rootPost.getForum();
        } else if (ForumContext.getContext(s).getForum() != null) {
            forum = ForumContext.getContext(s).getForum();
        } else {
            // sanity check
            throw new UncheckedWrapperException
                ("Must be either a forum, or a thread page.");
        }
        return forum;
    }
}
