/*
 * Copyright (C) 2001, 2002, 2003 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of
 * the License at http://www.redhat.com/licenses/ccmpl.html
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 */

package com.arsdigita.forum;

import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.kernel.Party;
import com.arsdigita.persistence.OID;
import com.arsdigita.persistence.DataCollection;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.SessionManager;
import com.arsdigita.notification.Digest;
import com.arsdigita.notification.Notification;
import com.arsdigita.util.Assert;

import java.math.BigDecimal;

/**
 * Extends Subscription to implement digest (periodic) notifications
 * on a forum.
 *
 * @author Kevin Scaldeferri (kevin@arsdigita.com)
 *
 * @version $Revision: #7 $ $Author: dan $ $DateTime: 2003/09/15 13:58:12 $
 */

public class DailySubscription extends ForumSubscription {

    public static final String BASE_DATA_OBJECT_TYPE =
        "com.arsdigita.forum.ForumSubscription";

    private Digest m_digest;

    private static final String DIGEST = "digest";

    /**
     * @param forum the Forum that is subscribed to
     * @param sender the Party which will appear in the From: header
     */
    public DailySubscription(Forum forum, Party sender) {
        this(BASE_DATA_OBJECT_TYPE, forum, sender);
    }

    /**
     * @param forum the Forum that is subscribed to
     * @param sender the Party which will appear in the From: header
     */
    public DailySubscription(String typeName, Forum forum, Party sender) {
        super(typeName, forum);

        Assert.exists(sender, Party.class);

        Digest digest = null;

        String subject = "Daily digest from " + forum.getDisplayName()
            + " forum";
        digest = new Digest(sender, subject, subject, "-- End digest --");
        digest.setFrequency(Digest.DAILY);

        setDigest(digest);
        setIsModerationAlert(false);
    }

    protected DailySubscription(OID oid) throws DataObjectNotFoundException {
        super(oid);
    }

    protected DailySubscription(BigDecimal id)
        throws DataObjectNotFoundException {

        super(new OID(BASE_DATA_OBJECT_TYPE, id));
    }

    protected DailySubscription(DataObject data) {
        super(data);
    }

    protected String getBaseDataObjectType() {
        return BASE_DATA_OBJECT_TYPE;
    }

    private Digest getDigest() {
        if (m_digest == null) {
            DataObject digestData = (DataObject) get(DIGEST);
            if (digestData != null) {
                m_digest = new Digest(digestData);
            }
        }
        return m_digest;
    }

    private void setDigest(Digest digest) {
        m_digest = digest;
        setAssociation(DIGEST, m_digest);
    }

    /**
     * Queues the post for once-daily notification of new posts.
     */
    public void sendNotification(Post post) {
        Notification notification = new Notification(getGroup(), post);
        notification.setDigest(getDigest());
        notification.save();
    }

    /**
     * @return the DailySubscription associated with the provided Forum
     */
    public static ForumSubscription getFromForum(Forum forum) {
        DataCollection subs = SessionManager.getSession()
            .retrieve(DailySubscription.BASE_DATA_OBJECT_TYPE);
        subs.addEqualsFilter("forum.id", forum.getID());
        subs.addNotEqualsFilter("digest.id", null);

        if (subs.next()) {
            DailySubscription sub =
                new DailySubscription(subs.getDataObject());
            subs.close();
            return sub;
        }
        return null;
    }
}
