/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.messaging;

/**
 * Test cases for plain text messages.
 *
 * @version $Id: //core-platform/dev/test/src/com/arsdigita/messaging/TextMessageTest.java#7 $
 */

public class TextMessageTest extends MessageTestCase {

    public static final String versionId = "$Id: //core-platform/dev/test/src/com/arsdigita/messaging/TextMessageTest.java#7 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private static String subject = "This is the subject";
    private static String body    = "This is the body";
    private static String type    = MessageType.TEXT_PLAIN;

    public TextMessageTest(String name) {
        super(name);
    }

    /**
     * Create a simple message using the various set methods.
     */

    public void testMessageCreate001() {
        try {
            Message msg = new Message();

            msg.setFrom(from);
            msg.setSubject(subject);
            msg.setText(body);

            assertEquals(subject,msg.getSubject());
            assertEquals(body,msg.getBody());
            assertTrue(msg.isMimeType(type));

        } catch (Exception e) {
            fail(e.getMessage());
        }
    }

    /**
     * Create a simple message using the default constructor.
     */

    public void testMessageCreate002() {
        try {
            Message msg = new Message(from,subject,body);

            assertEquals(subject,msg.getSubject());
            assertEquals(body,msg.getBody());
            assertTrue(msg.isMimeType(type));

        } catch (Exception e) {
            fail(e.getMessage());
        }
    }

    /**
     * Create a simple message and use the setText() method.
     */

    public void testMessageCreate003() {
        try {
            Message msg = new Message(from,subject);

            msg.setText(body);

            assertEquals(subject,msg.getSubject());
            assertEquals(body,msg.getBody());
            assertTrue(msg.isMimeType(type));

        } catch (Exception e) {
            fail(e.getMessage());
        }
    }

    /**
     * Create a simple message and save it to the database.
     */

    public void testMessageCreate004() {
        try {
            Message msg = new Message(from,subject,body);
            msg.save();

            // Retrieve a copy of the message and verify its contents

            Message msg2 = new Message(msg.getOID());
            assertEquals(msg.getSubject(), msg2.getSubject());
            assertEquals(msg.getBody(), msg2.getBody());

            // Repeat using the BigDecimal constructor

            msg2 = new Message(msg.getID());
            assertEquals(msg.getSubject(), msg2.getSubject());
            assertEquals(msg.getBody(), msg2.getBody());

            // Delete the original message

            msg.delete();

        } catch (Exception e) {
            fail(e.getMessage());
        }
    }


    /**
     * Verify that the default constructor and public set methods can
     * be used to create a Message.  This is essentially a test that
     * the sent date is handled correctly.
     */

    public void testMessage003() {
        Message msg = new Message();
        msg.setFrom(from);
        msg.setText(body);
        msg.setSubject(subject);
        msg.save();
    }

    /**
     * Verify that the sent date can be set explicitly
     */

    public void testMessage004() {
        Message msg = new Message(from,subject,body);
        msg.setSentDate(new java.util.Date());
        msg.save();
    }

    /**
     * Verify that saving a message with a null sender fails.
     */

    public void testMessageCreate005() {
        try {
            Message msg = new Message();
            msg.save();
            fail("Test should have failed because of an invalid party");
        } catch (Exception ex) {
            // nothing
        }
    }

    /**
     * Verify that messages can refer to other ACSObjects.  For
     * convenience this test is written to refer to another Message,
     * but it could that extends ACSObject.
     */

    public void testMessageCreate006() {

        Message msg1 = new Message(from, subject, body);
        msg1.save();

        Message msg2 = new Message(from, subject, body);

        // Test the ACSObject method
        msg2.setRefersTo(msg1);
        msg2.save();

        // Test the BigDecimal method
        msg2.setRefersTo(msg1.getID());
        msg2.save();
    }

    /**
     * Verify that messages can be stored in one of the simple text
     * formats (plain or preformatted).
     */

    public void testMessageCreate007() {

        Message msg = new Message(from, subject, body);
        assertTrue(msg.isMimeType(MessageType.TEXT_PLAIN));

        msg.setBody(body, MessageType.TEXT_PREFORMATTED);
        assertTrue(msg.isMimeType(MessageType.TEXT_PREFORMATTED));
    }

    /**
     * Test the required equivalence of the various MIME types.
     */

    public void testMessageCreate008() {

        String mimeType[] = new String [] {
            MessageType.TEXT_PLAIN,
            MessageType.TEXT_PREFORMATTED,
            MessageType.TEXT_HTML
        };

        Message msg = new Message(from, subject, body);

        msg.setBody(body, MessageType.TEXT_PLAIN);
        checkMimeType(msg, mimeType, new boolean[] { true, true, false });

        msg.setBody(body, MessageType.TEXT_PREFORMATTED);
        checkMimeType(msg, mimeType, new boolean[] { true, true, false });

        msg.setBody(body, MessageType.TEXT_HTML);
        checkMimeType(msg, mimeType, new boolean[] { false, false, true });
    }

    private static void checkMimeType (Message   msg,
                                       String [] type,
                                       boolean[] expected)
    {
        for (int i = 0; i < type.length; i++)
            assertEquals(expected[i], msg.isMimeType(type[i]));
    }

    /**
     * Main method required to make this test runnable.
     */

    public static void main (String args[]) {
        junit.textui.TestRunner.run(TextMessageTest.class);
    }

}
