/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.templating.html;

import com.arsdigita.templating.html.AttributeMap;

/**
 * Receives notification of the logical content of a document.
 *
 * <p>This is similar to the <a
 * href="http://java.sun.com/xml/jaxp/dist/1.1/docs/api/org/xml/sax/ContentHandler.html"><i>ContentHandler</i></a>
 * interface in SAX. The difference is that this class expects the invoking
 * {@link HTMLParser} to only send notifications of those elements that have
 * been registered with the parser. In other words, this handler would most
 * likely be only interested in handling special tags like the
 * <code>&lt;footnote></code> tag (see {@link HTMLParser}). In a typical use
 * case, regular HTML markup like <code>&lt;p></code> or <code>&lt;b></code>
 * would not generate the <code>startElement(String, AttributeMap)</code> and 
 * <code>endElement(String, AttributeMap)</code> events.  </p>
 *
 * @author Vadim Nasardinov (vadimn@redhat.com)
 * @since 2002-08-21
 * @version $Id: //core-platform/dev/src/com/arsdigita/templating/html/ContentHandler.java#4 $
 **/
public interface ContentHandler {

    /**
     * Receives notification of the beginning of a document.
     **/
    void startDocument() throws HTMLParserException;

    /**
     * Receives notification of the end of a document.
     **/
    void endDocument() throws HTMLParserException;

    /**
     * Receives notification of textual data.
     **/
    void text(String text) throws HTMLParserException;

    /**
     * Receives notification of the beginning of an element. The parser will
     * only notify the handler of those elements for which {@link
     * HTMLParser#isRegistered(String)} is true. Any other HTML markup will be
     * treated as text and will be dispatched to {@link #text(String)}.
     **/
    void startElement(String qName, AttributeMap attributes)
        throws HTMLParserException;

    /**
     * Receives notification of the end of an element. The parser will only
     * notify the handler of those elements for which {@link
     * HTMLParser#isRegistered(String)} is true. Any other HTML markup will be
     * treated as text and will be dispatched to {@link #text(String)}.
     **/
    void endElement(String qName) throws HTMLParserException;
}
