/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.templating.html;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * A parent class for a number of classes that define <a
 * href="http://www.w3.org/TR/REC-html40/sgml/entities.html">character entity
 * references</a> for ISO 8859-1 characters</a>. (See also <a
 * href="http://www.htmlhelp.com/reference/html40/entities/">Entities</a>.)
 *
 * @author Vadim Nasardinov (vadimn@redhat.com)
 * @since 2002-09-04
 * @version $Id: //core-platform/dev/src/com/arsdigita/templating/html/CharacterReferenceEntities.java#4 $
 **/
class CharacterReferenceEntities {
    private final static String LINE_END = System.getProperty("line.separator");

    private Map m_names;
    private List m_orderedNames;

    CharacterReferenceEntities() {
        m_names = new HashMap();
        m_orderedNames = new ArrayList();
    }
    /**
     * Returns the decimal reference for the <code>character</code>.  For
     * instance, <code>getDecimalReference(HTMLlat1.nbsp)</code> returns
     * <code>"&amp;#160;"</code>.
     **/
    public String getDecimalReference(String character) {
        CharacterReferenceEntity refEnt =
            (CharacterReferenceEntity) m_names.get(character);
        return refEnt == null ? null : refEnt.getDecimalReference();
    }

    /**
     * Returns the name of the character reference entity for this character.
     * For example, <code>HTMLlat1.getName(HTMLlat1.sect)</code> returns the
     * string <code>"sect"</code>.
     **/
    public String getName(String character) {
        CharacterReferenceEntity refEnt =
            (CharacterReferenceEntity) m_names.get(character);
        return refEnt == null ? null : refEnt.getName();
    }

    /**
     * Returns a string of the form <code>&lt;!ENTITY sect "&amp;#167;"></code>,
     * if called like so:
     *
     * <pre>
     * HTMLlat1.getEntityDeclaration(HTMLlat1.sect);
     * </pre>
     **/
    public String getEntityDeclaration(String character) {
        CharacterReferenceEntity refEnt =
            (CharacterReferenceEntity) m_names.get(character);
        return refEnt == null ? null : refEnt.getEntityDeclaration();
    }

    /**
     * Returns all character entity declarations as a single string of the
     * following form:
     *
     * <pre>
     * &lt;!ENTITY nbsp   "&amp;#160;">
     * &lt;!ENTITY iexcl  "&amp;#161;">
     *   [ ... skipped for brevity ...]
     * &lt;!ENTITY yacute "&amp;#253;">
     * &lt;!ENTITY thorn  "&amp;#254;">
     * &lt;!ENTITY yuml   "&amp;#255;">
     * </pre>
     **/
    public String getAllEntityDeclarations() {
        // FIXME: this should be computed once
        StringBuffer sb = new StringBuffer();
        for (Iterator i=m_orderedNames.iterator(); i.hasNext(); ) {
            sb.append(getEntityDeclaration((String) i.next()));
            sb.append(LINE_END);
        }
        return sb.toString();
    }

    protected String registerEntity(String name,
                                           String decimalReference) {

        CharacterReferenceEntity entity =
            new CharacterReferenceEntity(name, decimalReference);
        m_names.put(entity.toString(), entity);
        m_orderedNames.add(entity.toString());
        return entity.toString();
    }

    private static class CharacterReferenceEntity {
        private String m_character;
        private String m_decimalReference;
        private String m_entityName;
        private String m_entityDeclaration;

        public CharacterReferenceEntity(String name, String decimalReference) {
            m_entityName = name;
            m_decimalReference = decimalReference;
            m_character = "" + (char)
                Integer.parseInt(decimalReference.substring
                                 (2,
                                  decimalReference.length()-1));

            StringBuffer sb = new StringBuffer();
            sb.append("<!ENTITY ").append(getName());
            sb.append(" \"").append(getDecimalReference());
            sb.append("\">");
            m_entityDeclaration = sb.toString();
        }

        public String getEntityDeclaration() {
            return m_entityDeclaration;
        }

        public String toString() {
            return m_character;
        }

        public String getDecimalReference() {
            return m_decimalReference;
        }

        public String getName() {
            return m_entityName;
        }
    }
}
