/*
 * Copyright (C) 2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.search.lucene;

import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.Term;
import org.apache.lucene.index.TermDocs;
import org.apache.lucene.search.Filter;

import java.io.IOException;
import java.util.BitSet;
import java.math.BigDecimal;
import com.arsdigita.kernel.Party;

/**
 *
 * PartyFilter that filters the query based on the passed in party
 *
 * @author Randy Graebner (randyg@alum.mit.edu)
 * @version $Id: //core-platform/dev/src/com/arsdigita/search/lucene/PartyFilter.java#3 $
 *
 **/

public class PartyFilter extends Filter {

    private BigDecimal m_partyID;
    private String m_partyType;

    /**
     * Creates a new PartyFilter
     *
     * @param party The party to use for the filter
     * @param partyType The type of the party to restrict.  Examples
     *  would be Document.CREATION_USER and Document.LAST_MODIFIED_USER
     *
     **/
    public PartyFilter(Party party, String partyType) {
        m_partyID = party.getID();
        m_partyType = partyType;
    }

    /**
     *  This creates a PartyFilter based on the party with the
     *  passed in ID
     */
    public PartyFilter(BigDecimal partyID, String partyType) {
        m_partyID = partyID;
        m_partyType = partyType;
    }

    /**
     * Returns a BitSet with true for documents which
     * should be permitted in search results, and false
     * for those that should not.
     **/
    final public BitSet bits(IndexReader reader) throws IOException {
        BitSet bits = new BitSet(reader.maxDoc());
        TermDocs enum = reader.termDocs(new Term(m_partyType, m_partyID.toString()));

        while (enum.next()) {
            bits.set(enum.doc());
        }

        return bits;
    }
}
